/*
 * Copyright 2012 Hanson Robokind LLC.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.robokind.impl.messaging.services;

import org.apache.avro.util.Utf8;
import org.jflux.api.core.Adapter;
import org.robokind.api.common.utils.TimeUtils;
import org.robokind.api.messaging.services.ServiceCommand;
import org.robokind.api.messaging.services.ServiceCommandFactory;
import org.robokind.avrogen.messaging.ServiceCommandRecord;

/**
 *
 * @author Matthew Stevenson <www.robokind.org>
 */
public class PortableServiceCommand implements ServiceCommand{
    private ServiceCommandRecord myRecord;
    
    public PortableServiceCommand(ServiceCommandRecord record){
        if(record == null){
            throw new NullPointerException();
        }
        myRecord = record;
    }
    
    public PortableServiceCommand(ServiceCommand command){
        if(command == null){
            throw new NullPointerException();
        }else if(command instanceof PortableServiceCommand){
            myRecord = ((PortableServiceCommand)command).getRecord();
            return;
        }
        setRecord(command.getSourceId(), command.getDestinationId(), 
                command.getTimestampMillisecUTC(), command.getCommand());
    }
    
    public PortableServiceCommand(String sourceId, String destId, 
            long timestampMillisecUTC, String command){
        setRecord(sourceId, destId, timestampMillisecUTC, command);
    }
    private void setRecord(String sourceId, String destinationId, 
            long timestampMillisecUTC, String command){
        if(sourceId == null || destinationId == null || command == null){
            throw new NullPointerException();
        }
        myRecord = new ServiceCommandRecord();
        myRecord.sourceId = new Utf8(sourceId);
        myRecord.destinationId = new Utf8(destinationId);
        myRecord.timestampMillisecUTC = timestampMillisecUTC;
        myRecord.command = new Utf8(command);
    }
    
    @Override
    public String getSourceId() {
        return myRecord.sourceId.toString();
    }

    @Override
    public String getDestinationId() {
        return myRecord.destinationId.toString();
    }

    @Override
    public long getTimestampMillisecUTC() {
        return myRecord.timestampMillisecUTC;
    }

    @Override
    public String getCommand() {
        return myRecord.command.toString();
    }
    
    public ServiceCommandRecord getRecord(){
        return myRecord;
    }
    
    public static class Factory implements ServiceCommandFactory{
        
        @Override
        public ServiceCommand create(
                String sourceId, String destId, String command) {
            return new PortableServiceCommand(
                    sourceId, destId, TimeUtils.now(), command);
        }
        
    }
    
    public static class MessageRecordAdapter implements 
            Adapter<ServiceCommand,ServiceCommandRecord>{

        @Override
        public ServiceCommandRecord adapt(ServiceCommand a) {
            if(a == null){
                throw new NullPointerException();
            }
            return new PortableServiceCommand(a).getRecord();
        }
    }
    
    public static class RecordMessageAdapter implements
            Adapter<ServiceCommandRecord, ServiceCommand>{

        @Override
        public ServiceCommand adapt(ServiceCommandRecord a) {
            if(a == null){
                throw new NullPointerException();
            }
            return new PortableServiceCommand(a);
        }
    }
}
