/*
 * Copyright 2012 Hanson Robokind LLC.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.robokind.impl.messaging.services;

import org.apache.avro.util.Utf8;
import org.jflux.api.core.Adapter;
import org.robokind.api.messaging.services.ServiceError;
import org.robokind.avrogen.messaging.ServiceErrorRecord;

/**
 *
 * @author Matthew Stevenson <www.robokind.org>
 */
public class PortableServiceError implements ServiceError {
    private ServiceErrorRecord myRecord;
    
    public PortableServiceError(ServiceErrorRecord record){
        if(record == null){
            throw new NullPointerException();
        }
        myRecord = record;
    }
    
    public PortableServiceError(ServiceError error){
        if(error == null){
            throw new NullPointerException();
        }else if(error instanceof PortableServiceError){
            myRecord = ((PortableServiceError)error).getRecord();
            return;
        }
        setRecord(error.getSourceId(), error.getDestinationId(), 
                error.getTimestampMillisecUTC(), error.getErrorType(), 
                error.getMessage(), error.getDetails());
    }
    
    public PortableServiceError(
            String sourceId, String destinationId, long timestampMillisecUTC, 
            String errorType, String message, String details){
        setRecord(sourceId, destinationId, 
                timestampMillisecUTC, errorType, message, details);
    }
    private void setRecord(
            String sourceId, String destinationId, long timestampMillisecUTC, 
            String errorType, String message, String details){
        if(sourceId == null || destinationId == null || errorType == null){
            throw new NullPointerException();
        }
        myRecord = new ServiceErrorRecord();
        myRecord.sourceId = new Utf8(sourceId);
        myRecord.destinationId = new Utf8(destinationId);
        myRecord.timestampMillisecUTC = timestampMillisecUTC;
        myRecord.errorType = new Utf8(errorType);
        myRecord.message = new Utf8(message == null ? "" : message);
        myRecord.details = new Utf8(details == null ? "" : details);
    }
    
    @Override
    public String getSourceId() {
        return myRecord.sourceId.toString();
    }

    @Override
    public String getDestinationId() {
        return myRecord.destinationId.toString();
    }

    @Override
    public long getTimestampMillisecUTC() {
        return myRecord.timestampMillisecUTC;
    }

    @Override
    public String getErrorType() {
        return myRecord.errorType.toString();
    }

    @Override
    public String getMessage() {
        return myRecord.message.toString();
    }

    @Override
    public String getDetails() {
        return myRecord.details.toString();
    }
    
    public ServiceErrorRecord getRecord(){
        return myRecord;
    }
    
    public static class MessageRecordAdapter implements 
            Adapter<ServiceError,ServiceErrorRecord>{

        @Override
        public ServiceErrorRecord adapt(ServiceError a) {
            if(a == null){
                throw new NullPointerException();
            }
            return new PortableServiceError(a).getRecord();
        }
    }
    
    public static class RecordMessageAdapter implements
            Adapter<ServiceErrorRecord, ServiceError>{

        @Override
        public ServiceError adapt(ServiceErrorRecord a) {
            if(a == null){
                throw new NullPointerException();
            }
            return new PortableServiceError(a);
        }
        
    }
}
