package org.ryoframework.gradle.model

import org.gradle.api.DefaultTask
import org.gradle.api.tasks.Input
import org.gradle.api.tasks.TaskAction
import org.gradle.api.tasks.options.Option

class RyoInitTask extends DefaultTask {

    private Boolean kotlin = false
    private Boolean dev = false
    private String root = "app"
    private String packageName = null

    public static final String DOMAIN = "domain"
    public static final String APPLICATION = "application"
    public static final String EXPOSITION = "exposition"
    public static final String INFRA = "adapter"
    public static final String RYO_CORE_VERSION = "1.0.2"

    @Input
    Boolean getKotlin() {
        return kotlin
    }

    @Option(option = "kotlin", description = "Configure le projet en activant kotlin.")
    void setKotlin(Boolean kotlin) {
        this.kotlin = kotlin
    }

    @Input
    Boolean getDev() {
        return dev
    }

    @Option(option = "dev", description = "Active le mode développement.")
    void setDev(Boolean dev) {
        this.dev = dev
    }

    @Input
    String getPackageName() {
        return packageName
    }

    @Option(option = "package", description = "Package.")
    void setPackageName(String packageName) {
        this.packageName = packageName
    }

    @Input
    String getRoot() {
        return root
    }

    @Option(option = "root", description = "Dossier racine des modules, par défault 'app'")
    void setRoot(String root) {
        this.root = root
    }

    @TaskAction
    void process() {
        if (packageName == null || packageName.isEmpty() || packageName.length() == 0) {
            throw new Error("L'argument --package est obligatoire")
        }
        getLogger().quiet("Création de la structure projet")
        createStructure()
    }

    /// ------------------------------------------------------------------------------------------
    /// ------------------------------------------------------------------------------------------


    void createStructure() {
        def rootFolder = new File(System.getProperty("user.dir"))

        writeGradleProperties(rootFolder)
        writeSettingsFile(rootFolder)
        writeRootBuild(rootFolder)
        createEditorConfigFile(rootFolder)
        createGitIgnoreFile(rootFolder)

        createFolder(rootFolder, [root, "core-$DOMAIN"]).with {
            def dependencies = ["compile \"org.ryoframework.core:ryo-domain:\$ryo_core_version\""]
            createBuildGradle(it, kotlin, dependencies)
            createPackages(it, DOMAIN)
        }
        createFolder(rootFolder, [root, "core-$APPLICATION"]).with {
            def dependencies = ["compile project(':app:core-$DOMAIN')"]
            createBuildGradle(it, kotlin, dependencies)
            createPackages(it, APPLICATION)
        }
        createFolder(rootFolder, [root, "core-$EXPOSITION"]).with {
            def dependencies = ["compile project(':app:core-$APPLICATION')"]
            createBuildGradle(it, kotlin, dependencies)
            createPackages(it, EXPOSITION)
        }
        createFolder(rootFolder, [root, INFRA]).with {
            def dependencies = [
                "implementation project(':app:core-application')",
                "implementation 'org.springframework.boot:spring-boot-starter-webflux'"
                // "implementation 'compile 'org.springframework.boot:spring-boot-starter-webflux''"
            ]
            createBuildGradle(it, kotlin, dependencies, true)
            createPackages(it, INFRA)
        }

    }

    void createEditorConfigFile(File root) {
        def file = createFile(root, '.editorconfig')
        if (file) {
            file << '''
[*]
charset=utf-8
end_of_line=lf
insert_final_newline=true
indent_style=space
indent_size=4

[*.yml]
indent_style=space
indent_size=2

[*.kt]
indent_style=space
indent_size=4

[*.kts]
indent_style=space
indent_size=4

[build.gradle.kts]
indent_style=space
indent_size=4

[*.{js,ts,tsx,less}]
indent_style=space
indent_size=4
                '''
        }
    }

    void createGitIgnoreFile(File root) {
        def file = createFile(root, '.gitignore')
        if (file) {
            file << '''
.gradle
build/
out/
classes/
bower_components
node_modules

# Ignore Gradle GUI config
gradle-app.setting

# Avoid ignoring Gradle wrapper jar file (.jar files are usually ignored)
!gradle-wrapper.jar

# Cache of project
.gradletasknamecache
docker-compose.yml
# # Work around https://youtrack.jetbrains.com/issue/IDEA-116898
# gradle/wrapper/gradle-wrapper.properties
.idea
*.ipr
*.iws
*.iml
*.log
*.js.map
deps.txt
.DS_Store
*.pid
tmp
target
app-cli/dist
.env*
.env.staging
.env.prod
!dist/config.json
*.zip
node/dist
application-local.yml
graphql
            '''
        }
    }

    File createFolder(File root, List<String> path) {
        return createFolder(root, path.join(File.separator))
    }

    File createFolder(File root, String[] path) {
        return createFolder(root, path.join(File.separator))
    }

    File createFolder(File root, String path) {
        def folder = new File(root, path)
        if (folder.exists() && folder.isDirectory()) {
            throw new Error("Le dossier '${path}' existe déjà.")
        } else {
            if (folder.mkdirs()) {
                getLogger().quiet("+ ${path}")
            } else {
                throw new Error("Impossible de créer le dossier '${path}'.")
            }
        }
        return folder
    }

    void createPackages(File folder, String module) {

        getLogger().quiet("[$module]")

        String src = ['src', 'main', kotlin ? 'kotlin' : 'java'].join(File.separator)

        createFolder(folder, src)
        createFolder(folder, ['src', 'test', kotlin ? 'kotlin' : 'java'])

        String pModule = "src.main.${kotlin ? 'kotlin' : 'java'}.${packageName}.${module}"

        createFolder(folder, pModule.split("\\."))

        if (module == DOMAIN) {
            createFolder(folder, "${pModule}.entities".split("\\."))
            createFolder(folder, "${pModule}.repositories".split("\\."))
            createFolder(folder, pModule.replace("src.main.", 'src.test.').split("\\."))
        }else if (module == EXPOSITION) {
            createFolder(folder, "${pModule}.rest".split("\\."))
            createFolder(folder, "${pModule}.graphql".split("\\."))
            createFolder(folder, pModule.replace("src.main.", 'src.test.').split("\\."))
        }else if (module == APPLICATION) {
            createFolder(folder, "${pModule}.service".split("\\."))
            createFolder(folder, "${pModule.replace('src.main', 'src.test.')}.service".split("\\."))
        }else if (module == INFRA) {

            createFolder(folder, pModule.replace('src.main', 'src.test.').split("\\."))
            createFolder(folder, "${pModule}.config".split("\\."))
            createFolder(folder, "${pModule}.service".split("\\."))
            createFolder(folder, "${pModule}.repositories".split("\\."))
            createFolder(folder, "${pModule}.controllers".split("\\."))

            ensureFile(folder, ['src', 'main', 'resources', 'application.yml'])
            ensureFile(folder, ['src', 'main', 'resources', 'bootstrap.yml'])
            ensureFile(folder, ['src', 'test', 'resources', 'application-test.yml'])

            if (kotlin) {
                def file = ensureFile(folder, "${(pModule + '.').replace('.', File.separator)}Bootstrap.kt")
                file << "package ${packageName}.${module};\n"
                file << "\nimport org.springframework.boot.autoconfigure.SpringBootApplication"
                file << "\nimport org.springframework.boot.autoconfigure.domain.EntityScan"
                file << "\nimport org.springframework.boot.runApplication"
                file << "\n"
                file << "\n@SpringBootApplication"
                file << "\n@EntityScan(basePackages = [\"${packageName}.domain.entities\"])"
                file << "\ninternal class Bootstrap"
                file << "\n"
                file << "\nfun main(args: Array<String>) {"
                file << "\n\trunApplication<Bootstrap>(*args)"
                file << "\n}"
                file << "\n"
            }else {
                def file = ensureFile(folder, "${(pModule + '.').replace('.', File.separator)}Bootstrap.java")
                file << "package ${packageName}.${module};\n"
                file << "\nimport org.springframework.boot.SpringApplication;"
                file << "\nimport org.springframework.boot.autoconfigure.SpringBootApplication;"
                file << "\nimport org.springframework.boot.autoconfigure.domain.EntityScan;"
                file << "\n"
                file << "\n@SpringBootApplication"
                file << "\n@EntityScan(basePackages = {\"${packageName}.domain.entities\"})"
                file << "\npublic class Bootstrap {"
                file << "\n"
                file << "\n\tpublic static void main(String[] args) {"
                file << "\n\t\tSpringApplication.run(Bootstrap.class, args);"
                file << "\n\t}"
                file << "\n}"

            }
        }
    }

    @SuppressWarnings("GrMethodMayBeStatic")
    File createBuildGradle(File folder, Boolean kotlin, List<String> dependencies, Boolean infra = false) {
        def gradle = new File(folder, 'build.gradle')
        gradle << "apply plugin: 'ryo${kotlin ? '.kotlin' : ''}'\n"
        if (infra) {
            gradle << "apply plugin: 'org.ryoframework.spring.boot'"
        }
        gradle << "\n\ndependencies {"
        for (String dep : dependencies) {
            gradle << "\n\t${dep}"
        }
        gradle << "\n}"

        if (infra) {
            gradle << "\n\n"
            gradle << "\nbootJar {"
            gradle << "\n\tdestinationDir = file(\"../../build\")"
            gradle << "\n\tarchiveName = \"app.jar\""
            gradle << "\n}"
        }
        return gradle
    }

    void writeRootBuild(File rootFolder) {

        def file = ensureFile(rootFolder, "build.gradle")
        String fileContents = file.getText('UTF-8')

        if (fileContents.contains("subprojects")) {
            getLogger().warn("'subprojects' présent dans le fichier build.gradle. Action ignorée.")
            return
        }

        file << "\n\nsubprojects {"
        file << "\n    if (name == '${this.root}') {"
        file << "\n        return"
        file << "\n    }"
        file << "\n"
        file << "\n    apply plugin: 'java'"
        file << "\n"
        file << "\n    sourceCompatibility = 1.8"
        file << "\n    targetCompatibility = 1.8"
        file << "\n"
        file << "\n    repositories {"
        if (this.dev) {
            file << "\n        mavenLocal()"
        }
        file << "\n        mavenCentral()"
        file << "\n    }"
        file << "\n}"
        getLogger().quiet("+ build.gradle")
    }

    void writeSettingsFile(File rootFolder) {
        def file = ensureFile(rootFolder, "settings.gradle")
        String fileContents = file.getText('UTF-8')
        String rootName = this.root
        def modules = ['core-application', 'core-domain', 'core-exposition', 'adapter']
        for (String module : modules) {
            if (!fileContents.contains("$rootName:$module")) {
                file << "\ninclude ':$rootName:$module'"
            }
        }
        getLogger().quiet("+ settings.gradle")
    }

    void writeGradleProperties(File rootFolder) {
        def file = ensureFile(rootFolder, "gradle.properties")
        String fileContents = file.getText('UTF-8')
        if (fileContents.contains("ryo_core_version")) return

        file << "\nryo_core_version=$RYO_CORE_VERSION"
        getLogger().quiet("+ gradle.properties")
    }

    File ensureFile(File rootFolder, List<String> child) {
        return ensureFile(rootFolder, child.join(File.separator))
    }

    File ensureFile(File rootFolder, String child) {
        def file = new File(rootFolder, child)

        if (!file.getParentFile().exists()) {
            file.getParentFile().mkdirs()
        }

        if (!file.exists()) {
            if (!file.createNewFile()) {
                getLogger().quiet("Impossible de créer le fichier ${file.name}.")
            }
        }
        return file
    }

    @SuppressWarnings("GrMethodMayBeStatic")
    File createFile(File rootFolder, String child) {
        def file = new File(rootFolder, child)
        if (!file.exists()) {

            if (!file.getParentFile().exists()) {
                file.getParentFile().mkdirs()
            }

            file.createNewFile()

            return file
        }
        return null
    }

}
