/**
 * Copyright (c) 2005-2017 The Apereo Foundation
 *
 * Licensed under the Educational Community License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *             http://opensource.org/licenses/ecl2
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.sakaiproject.user.detail;

import java.util.List;
import java.util.Objects;
import java.util.Optional;

import lombok.extern.slf4j.Slf4j;

import org.sakaiproject.component.api.ServerConfigurationService;
import org.sakaiproject.site.api.Site;
import org.sakaiproject.site.api.SiteService;
import org.sakaiproject.tool.api.ToolManager;
import org.sakaiproject.user.api.CandidateDetailProvider;
import org.sakaiproject.user.api.User;

/**
 * This candidate details provider is designed to be a sensible fallback when using a chaining provider.
 */
@Slf4j
public class FallbackDetailProvider implements CandidateDetailProvider {
	
	private final static String PROP_USE_INSTITUTIONAL_ANONYMOUS_ID = "useInstitutionalAnonymousID";
	private final static String PROP_DISPLAY_ADDITIONAL_INFORMATION = "displayAdditionalInformation";
	private final static String PROP_DISPLAY_SPECIAL_NEEDS = "displaySpecialNeeds";
	private final static String PROP_USE_INSTITUTIONAL_NUMERIC_ID = "useInstitutionalNumericID";

	private ServerConfigurationService serverConfigurationService;
	private SiteService siteService;
	private ToolManager toolManager;
	
	public void init() {
		Objects.requireNonNull(siteService, "SiteService must be set");
		Objects.requireNonNull(toolManager, "ToolManager must be set");
		Objects.requireNonNull(serverConfigurationService, "ServerConfigurationService must be set");
	}
	
	public Optional<String> getCandidateID(User user, Site site) {
		log.debug("Getting candidate id from fallback provider");
		try {
			//check if we should use the institutional anonymous id (system-wide or site-based)
			if(user != null && useInstitutionalAnonymousId(site)) {
				String candidateID = "no-candidate-id:"+user.getId();
				return Optional.ofNullable(candidateID);
			}
		} catch(Exception e) {
			log.warn("Error getting fallback candidateID for {}", ((user != null) ? user.getId() : "-null-"), e);
		}
		return Optional.empty();
	}
	
	public boolean useInstitutionalAnonymousId(Site site) {
		log.debug("useInstitutionalAnonymousId from fallback provider");
		try {
			return (serverConfigurationService.getBoolean(PROP_USE_INSTITUTIONAL_ANONYMOUS_ID, false) || (site != null && Boolean.parseBoolean(site.getProperties().getProperty(PROP_USE_INSTITUTIONAL_ANONYMOUS_ID))));
		} catch(Exception ignore) {}
		return false;
	}
	
	public Optional<List<String>> getAdditionalNotes(User user, Site site) {
		return Optional.empty();
	}
	
	public boolean isAdditionalNotesEnabled(Site site) {
		try {
			return (serverConfigurationService.getBoolean(PROP_DISPLAY_ADDITIONAL_INFORMATION, false) || (site != null && Boolean.parseBoolean(site.getProperties().getProperty(PROP_DISPLAY_ADDITIONAL_INFORMATION))));
		} catch(Exception ignore) {}
		return false;
	}
	
	public Optional<List<String>> getSpecialNeeds(User user, Site site) {
		return Optional.empty();
	}
	
	public boolean isSpecialNeedsEnabled(Site site) {
		try {
			return (serverConfigurationService.getBoolean(PROP_DISPLAY_SPECIAL_NEEDS, true) || (site != null && Boolean.parseBoolean(site.getProperties().getProperty(PROP_DISPLAY_SPECIAL_NEEDS))));
		} catch(Exception ignore) {}
		return false;
	}
	
	@Override
	public Optional<String> getInstitutionalNumericId(User user, Site site)
	{
		log.debug("Getting student number from fallback provider");
		if (user != null && isInstitutionalNumericIdEnabled(site))
		{
			return Optional.of(("no-student-number:" + user.getId()));
		}
		
		return Optional.empty();
	}
	
	@Override
	public Optional<String> getInstitutionalNumericIdIgnoringCandidatePermissions(User candidate, Site site)
	{
		return getInstitutionalNumericId(candidate, site);
	}
	
	@Override
	public boolean isInstitutionalNumericIdEnabled(Site site)
	{
		try {
			return (serverConfigurationService.getBoolean(PROP_USE_INSTITUTIONAL_NUMERIC_ID, true)
					|| (site != null && Boolean.parseBoolean(site.getProperties().getProperty(PROP_USE_INSTITUTIONAL_NUMERIC_ID))));
		} catch(Exception ignore) {}
		return false;
	}

	public void setServerConfigurationService(ServerConfigurationService serverConfigurationService) {
		this.serverConfigurationService = serverConfigurationService;
	}
	
	public void setSiteService(SiteService siteService) {
		this.siteService = siteService;
	}

	public void setToolManager(ToolManager toolManager) {
		this.toolManager = toolManager;
	}
}
