package java.nio

// Ported from Scala.js
object DoubleBuffer {
  private final val HashSeed = 2140173175 // "java.nio.DoubleBuffer".##

  def allocate(capacity: Int): DoubleBuffer =
    wrap(new Array[Double](capacity))

  def wrap(array: Array[Double], offset: Int, length: Int): DoubleBuffer =
    HeapDoubleBuffer.wrap(array, 0, array.length, offset, length, false)

  def wrap(array: Array[Double]): DoubleBuffer =
    wrap(array, 0, array.length)
}

abstract class DoubleBuffer private[nio] (
    _capacity: Int,
    private[nio] val _array: Array[Double],
    private[nio] val _mappedData: MappedByteBufferData,
    private[nio] val _arrayOffset: Int
) extends Buffer(_capacity)
    with Comparable[DoubleBuffer] {

  private[nio] type ElementType = Double
  private[nio] type BufferType = DoubleBuffer

  private def genBuffer = GenBuffer[DoubleBuffer](this)

  def this(_capacity: Int) = this(_capacity, null, null, -1)

  def slice(): DoubleBuffer

  def duplicate(): DoubleBuffer

  def asReadOnlyBuffer(): DoubleBuffer

  def get(): Double

  def put(d: Double): DoubleBuffer

  def get(index: Int): Double

  def put(index: Int, d: Double): DoubleBuffer

  @noinline
  def get(dst: Array[Double], offset: Int, length: Int): DoubleBuffer =
    genBuffer.generic_get(dst, offset, length)

  def get(dst: Array[Double]): DoubleBuffer =
    get(dst, 0, dst.length)

  @noinline
  def put(src: DoubleBuffer): DoubleBuffer =
    genBuffer.generic_put(src)

  @noinline
  def put(src: Array[Double], offset: Int, length: Int): DoubleBuffer =
    genBuffer.generic_put(src, offset, length)

  final def put(src: Array[Double]): DoubleBuffer =
    put(src, 0, src.length)

  @inline final def hasArray(): Boolean =
    genBuffer.generic_hasArray()

  @inline final def array(): Array[Double] =
    genBuffer.generic_array()

  @inline final def arrayOffset(): Int =
    genBuffer.generic_arrayOffset()

  @inline override def position(newPosition: Int): DoubleBuffer = {
    super.position(newPosition)
    this
  }

  @inline override def limit(newLimit: Int): DoubleBuffer = {
    super.limit(newLimit)
    this
  }

  @inline override def mark(): DoubleBuffer = {
    super.mark()
    this
  }

  @inline override def reset(): DoubleBuffer = {
    super.reset()
    this
  }

  @inline override def clear(): DoubleBuffer = {
    super.clear()
    this
  }

  @inline override def flip(): DoubleBuffer = {
    super.flip()
    this
  }

  @inline override def rewind(): DoubleBuffer = {
    super.rewind()
    this
  }

  def compact(): DoubleBuffer

  def isDirect(): Boolean

  // toString(): String inherited from Buffer

  @noinline
  override def hashCode(): Int =
    genBuffer.generic_hashCode(DoubleBuffer.HashSeed)

  override def equals(that: Any): Boolean = that match {
    case that: DoubleBuffer => compareTo(that) == 0
    case _                  => false
  }

  @noinline
  def compareTo(that: DoubleBuffer): Int =
    genBuffer.generic_compareTo(that)(_.compareTo(_))

  def order(): ByteOrder

  // Internal API

  private[nio] def load(index: Int): Double

  private[nio] def store(index: Int, elem: Double): Unit

  @inline
  private[nio] def load(
      startIndex: Int,
      dst: Array[Double],
      offset: Int,
      length: Int
  ): Unit =
    genBuffer.generic_load(startIndex, dst, offset, length)

  @inline
  private[nio] def store(
      startIndex: Int,
      src: Array[Double],
      offset: Int,
      length: Int
  ): Unit =
    genBuffer.generic_store(startIndex, src, offset, length)
}
