/*
 * Copyright 2001-2013 Artima, Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.scalatest
package prop


import matchers.ShouldMatchers
import org.scalacheck.Gen
import org.scalacheck.Prop.{Exception => _, _}
import Checkers._

class CheckersImportSuite extends Spec  {

  val famousLastWords = for {
    s <- Gen.oneOf("the", "program", "compiles", "therefore", "it", "should", "work")
  } yield s

  val sevenEleven: Gen[String] =
    Gen.sized { (size: Int) =>
      if (size >= 7 && size <= 11)
        Gen.value("OKAY")
      else
        throw new Exception("expected 7 <= size <= 11 but got " + size)
    }

  val fiveFive: Gen[String] =
    Gen.sized { (size: Int) =>
      if (size == 5)
        Gen.value("OKAY")
      else
        throw new Exception("expected size 5 but got " + size)
    }


  def `ScalaCheck property that takes 1 args, which succeeds` {

    check { (a: String) =>
      a.length == ((a).length)
    }
  }

  def `ScalaCheck property that takes 1 args, which fails` {

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      check { (a: String) =>
        a.length < 0
      }
    }
  }

  def `ScalaCheck property that takes 1 args and generators, which succeeds` {

    val prop = forAll (famousLastWords) { (a: String) =>
      a.length == ((a).length)
    }
    check(prop)
  }

  def `ScalaCheck property that takes 1 args and generators, which fails` {

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      val prop = forAll (famousLastWords) { (a: String) =>
        a.length < 0
      }
      check(prop)
    }
  }

  // Same thing, but with config params
  def `ScalaCheck property that takes 1 args, which succeeds, with config params` {

    check(
      (a: String) => a.length == ((a).length),
      minSize(10),
      maxSize(20)
    )
  }

  def `ScalaCheck property that takes 1 args, which fails, with config params` {

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      check(
        (a: String) => a.length < 0,
        minSize(10),
        maxSize(20)
      )
    }
  }

  def `ScalaCheck property that takes 1 args and generators, which succeeds, with config params` {

    val prop = forAll (famousLastWords) { (a: String) =>
      a.length == ((a).length)
    }
    check(prop, minSize(10), maxSize(20))
  }

  def `ScalaCheck property that takes 1 args and generators, which fails, with config params` {

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      val prop = forAll (famousLastWords) { (a: String) =>
        a.length < 0
      }
      check(prop, minSize(10), maxSize(20))
    }
  }

  // Same thing, but set minSuccessful to 5 with param, prop fails after 5
  def `ScalaCheck property that takes 1 args, which succeeds, with minSuccessful param set to 5` {

    var i = 0
    check(
      (a: String) => {
        val res = i != 5
        i += 1
        res
      },
      minSuccessful(5)
    )
  }

  def `ScalaCheck property that takes 1 args, which fails, with minSuccessful param set to 5` {

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      var i = 0
      check(
        (a: String) => {
          val res = i != 4
          i += 1
        res
        },
        minSuccessful(5)
      ) 
    }
  }

  def `ScalaCheck property that takes 1 args and generators, which succeeds, with minSuccessful param set to 5` {

    var i = 0
    val prop = forAll (famousLastWords) { (a: String) =>
      val res = i != 5
      i += 1
      res
    }
    check(prop, minSuccessful(5))
  }

  def `ScalaCheck property that takes 1 args and generators, which fails, with minSuccessful param set to 5` {

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      var i = 0
      val prop = forAll (famousLastWords) { (a: String) =>
        val res = i != 4
        i += 1
        res
      }
      check(prop, minSuccessful(5))
    }
  }

  // Same thing, but set default minSuccessful to 5, prop fails after 5
  def `ScalaCheck property that takes 1 args, which succeeds, with default minSuccessful param set to 5` {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSuccessful = 5)

    var i = 0
    check { (a: String) =>
      val res = i != 5
      i += 1
      res
    }
  }

  def `ScalaCheck property that takes 1 args, which fails, with default minSuccessful param set to 5` {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSuccessful = 5)

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      var i = 0
      check { (a: String) =>
        val res = i != 4
        i += 1
        res
      }
    }
  }

  def `ScalaCheck property that takes 1 args and generators, which succeeds, with default minSuccessful param set to 5` {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSuccessful = 5)

    var i = 0
    val prop = forAll (famousLastWords) { (a: String) =>
      val res = i != 5
      i += 1
      res
    }
    check(prop)
  }

  def `ScalaCheck property that takes 1 args and generators, which fails, with default minSuccessful param set to 5` {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSuccessful = 5)

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      var i = 0
      val prop = forAll (famousLastWords) { (a: String) =>
        val res = i != 4
        i += 1
        res
      }
      check(prop)
    }
  }

  // Same thing, but set maxDiscarded to 5 with param, prop fails after 5
  def `ScalaCheck property that takes 1 args, which succeeds, with maxDiscarded param set to 5` {

    var i = 0
    check(
      (a: String) => {
        i += 1
        (i > 5) ==> { 1 + 1 == (2) }
      },
      maxDiscarded(5)
    )
  }

  def `ScalaCheck property that takes 1 args, which fails, with maxDiscarded param set to 5` {

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      var i = 0
      check(
        (a: String) => {
          i += 1
          (i > 6) ==> { 1 + 1 == (2) }
        },
        maxDiscarded(5)
      ) 
    }
  }

  def `ScalaCheck property that takes 1 args and generators, which succeeds, with maxDiscarded param set to 5` {

    var i = 0
    val prop = forAll (famousLastWords) { (a: String) =>
      i += 1
      (i > 5) ==> { 1 + 1 == (2) }
    }
    check(prop, maxDiscarded(5))
  }

  def `ScalaCheck property that takes 1 args and generators, which fails, with maxDiscarded param set to 5` {

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      var i = 0
      val prop = forAll (famousLastWords) { (a: String) =>
        i += 1
        (i > 6) ==> { 1 + 1 == (2) }
      }
      check(prop, maxDiscarded(5))
    }
  }

  // Same thing, but set default maxDiscarded to 5, prop fails after 5
  def `ScalaCheck property that takes 1 args, which succeeds, with default maxDiscarded set to 5` {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(maxDiscarded = 5)

    var i = 0
    check { (a: String) =>
      i += 1
      (i > 5) ==> { 1 + 1 == (2) }
    }
  }

  def `ScalaCheck property that takes 1 args, which fails, with default maxDiscarded set to 5` {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(maxDiscarded = 5)

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      var i = 0
      check { (a: String) =>
        i += 1
        (i > 6) ==> { 1 + 1 == (2) }
      }
    }
  }

  def `ScalaCheck property that takes 1 args and generators, which succeeds, with default maxDiscarded set to 5` {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(maxDiscarded = 5)

    var i = 0
    val prop = forAll (famousLastWords) { (a: String) =>
      i += 1
      (i > 5) ==> { 1 + 1 == (2) }
    }
    check(prop)
  }

  def `ScalaCheck property that takes 1 args and generators, which fails, with default maxDiscarded set to 5` {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(maxDiscarded = 5)

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      var i = 0
      val prop = forAll (famousLastWords) { (a: String) =>
        i += 1
        (i > 6) ==> { 1 + 1 == (2) }
      }
      check(prop)
    }
  }

  // set minSize > maxSize with (param, param) (intercept IAE)
  def `ScalaCheck property that takes 1 args, which should throw IAE because maxSize > maxSize, specified as (param, param)` {

    intercept[IllegalArgumentException] {
      check(
        (a: String) => {
          1 + 1 == (2)
        },
        minSize(5),
        maxSize(4)
      ) 
    }
  }

  def `ScalaCheck property that takes 1 args and generators, which should throw IAE because maxSize > maxSize, specified as (param, param)` {

    intercept[IllegalArgumentException] {
      val prop = forAll (famousLastWords) { (a: String) =>
        1 + 1 == (2)
      }
      check(prop, minSize(5), maxSize(4))
    }
  }

  // set minSize > maxSize with (param, default) (intercept IAE)
  def `ScalaCheck property that takes 1 args, which should throw IAE because maxSize > maxSize, specified as (param, default)` {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(maxSize = 4)

    intercept[IllegalArgumentException] {
      check(
        (a: String) => {
          1 + 1 == (2)
        },
        minSize(5)
      )
    }
  }

  def `ScalaCheck property that takes 1 args and generators, which should throw IAE because maxSize > maxSize, specified as (param, default)` {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(maxSize = 4)

    intercept[IllegalArgumentException] {
      val prop = forAll (famousLastWords) { (a: String) =>
        1 + 1 == (2)
      }
      check(prop, minSize(5))
    }
  }

  // set minSize > maxSize with (default, param) (intercept IAE)
  def `ScalaCheck property that takes 1 args, which should throw IAE because maxSize > maxSize, specified as (default, param)` {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSize = 5)

    intercept[IllegalArgumentException] {
      check(
        (a: String) => {
          1 + 1 == (2)
        },
        maxSize(4)
      )
    }
  }

  def `ScalaCheck property that takes 1 args and generators, which should throw IAE because maxSize > maxSize, specified as (default, param)` {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSize = 5)

    intercept[IllegalArgumentException] {
      val prop = forAll (famousLastWords) { (a: String) =>
        1 + 1 == (2)
      }
      check(prop, maxSize(4))
    }
  }

  // set maxSize with param (ensure always passed with a size less than maxSize)
  def `ScalaCheck property that takes 1 args, with maxSize specified as param` {

    check(
      (a: String) => {
      a.length <= 5
      },
      maxSize(5)
    ) 
  }

  // set maxSize with default (ensure always passed with a size less than maxSize)
  def `ScalaCheck property that takes 1 args, with maxSize specified as default` {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(maxSize = 5)

    check { (a: String) =>
      a.length <= 5
    }
  }

  // set minSize == maxSize with (param, param) (ensure always passed with that size)
  def `ScalaCheck property that takes 1 args and generators, with minSize == maxSize, specified as (param, param)` {

    val prop = forAll (fiveFive) { (a: String) =>
        a == ("OKAY")
    }
    check(prop, minSize(5), maxSize(5))
  }

  // set minSize == maxSize with (param, default) (ensure always passed with that size)
  def `ScalaCheck property that takes 1 args and generators, with minSize == maxSize, specified as (param, default)` {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(maxSize = 5)

    val prop = forAll (fiveFive) { (a: String) =>
        a == ("OKAY")
    }
    check(prop, minSize(5))
  }

  // set minSize == maxSize with (default, param) (ensure always passed with that size)
  def `ScalaCheck property that takes 1 args and generators, with minSize == maxSize, specified as (default, param)` {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSize = 5)

    val prop = forAll (fiveFive) { (a: String) =>
        a == ("OKAY")
    }
    check(prop, maxSize(5))
  }

  // set minSize == maxSize with (default, default) (ensure always passed with that size)
  def `ScalaCheck property that takes 1 args and generators, with minSize == maxSize, specified as (default, default)` {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSize = 5, maxSize = 5)

    val prop = forAll (fiveFive) { (a: String) =>
        a == ("OKAY")
    }
    check(prop)
  }

  // set minSize to 7 and maxSize to 11 with (param, param) (ensure always passed with that size)
  def `ScalaCheck property that takes 1 args and generators, with minSize to 7 and maxSize to 11, specified as (param, param)` {

    val prop = forAll (sevenEleven) { (a: String) =>
        a == ("OKAY")
    }
    check(prop, minSize(7), maxSize(11))
  }

  // set minSize to 7 and maxSize to 11 with (param, default) (ensure always passed with that size)
  def `ScalaCheck property that takes 1 args and generators, with minSize to 7 and maxSize to 11, specified as (param, default)` {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(maxSize = 11)

    val prop = forAll (sevenEleven) { (a: String) =>
        a == ("OKAY")
    }
    check(prop, minSize(7))
  }

  // set minSize to 7 and maxSize to 11 with (default, param) (ensure always passed with that size)
  def `ScalaCheck property that takes 1 args and generators, with minSize to 7 and maxSize to 11, specified as (default, param)` {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSize = 7)

    val prop = forAll (sevenEleven) { (a: String) =>
        a == ("OKAY")
    }
    check(prop, maxSize(11))
  }

  // set minSize to 7 and maxSize to 11 with (default, default) (ensure always passed with that size)
  def `ScalaCheck property that takes 1 args and generators, with minSize to 7 and maxSize to 11, specified as (default, default)` {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSize = 7, maxSize = 11)

    val prop = forAll (sevenEleven) { (a: String) =>
        a == ("OKAY")
    }
    check(prop)
  }


  def `ScalaCheck property that takes 2 args, which succeeds` {

    check { (a: String, b: String) =>
      a.length + b.length == ((a + b).length)
    }
  }

  def `ScalaCheck property that takes 2 args, which fails` {

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      check { (a: String, b: String) =>
        a.length + b.length < 0
      }
    }
  }

  def `ScalaCheck property that takes 2 args and generators, which succeeds` {

    val prop = forAll (famousLastWords, famousLastWords) { (a: String, b: String) =>
      a.length + b.length == ((a + b).length)
    }
    check(prop)
  }

  def `ScalaCheck property that takes 2 args and generators, which fails` {

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      val prop = forAll (famousLastWords, famousLastWords) { (a: String, b: String) =>
        a.length + b.length < 0
      }
      check(prop)
    }
  }

  // Same thing, but with config params
  def `ScalaCheck property that takes 2 args, which succeeds, with config params` {

    check(
      (a: String, b: String) => a.length + b.length == ((a + b).length),
      minSize(10),
      maxSize(20)
    )
  }

  def `ScalaCheck property that takes 2 args, which fails, with config params` {

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      check(
        (a: String, b: String) => a.length + b.length < 0,
        minSize(10),
        maxSize(20)
      )
    }
  }

  def `ScalaCheck property that takes 2 args and generators, which succeeds, with config params` {

    val prop = forAll (famousLastWords, famousLastWords) { (a: String, b: String) =>
      a.length + b.length == ((a + b).length)
    }
    check(prop, minSize(10), maxSize(20))
  }

  def `ScalaCheck property that takes 2 args and generators, which fails, with config params` {

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      val prop = forAll (famousLastWords, famousLastWords) { (a: String, b: String) =>
        a.length + b.length < 0
      }
      check(prop, minSize(10), maxSize(20))
    }
  }

  // Same thing, but set minSuccessful to 5 with param, prop fails after 5
  def `ScalaCheck property that takes 2 args, which succeeds, with minSuccessful param set to 5` {

    var i = 0
    check(
      (a: String, b: String) => {
        val res = i != 5
        i += 1
        res
      },
      minSuccessful(5)
    )
  }

  def `ScalaCheck property that takes 2 args, which fails, with minSuccessful param set to 5` {

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      var i = 0
      check(
        (a: String, b: String) => {
          val res = i != 4
          i += 1
        res
        },
        minSuccessful(5)
      ) 
    }
  }

  def `ScalaCheck property that takes 2 args and generators, which succeeds, with minSuccessful param set to 5` {

    var i = 0
    val prop = forAll (famousLastWords, famousLastWords) { (a: String, b: String) =>
      val res = i != 5
      i += 1
      res
    }
    check(prop, minSuccessful(5))
  }

  def `ScalaCheck property that takes 2 args and generators, which fails, with minSuccessful param set to 5` {

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      var i = 0
      val prop = forAll (famousLastWords, famousLastWords) { (a: String, b: String) =>
        val res = i != 4
        i += 1
        res
      }
      check(prop, minSuccessful(5))
    }
  }

  // Same thing, but set default minSuccessful to 5, prop fails after 5
  def `ScalaCheck property that takes 2 args, which succeeds, with default minSuccessful param set to 5` {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSuccessful = 5)

    var i = 0
    check { (a: String, b: String) =>
      val res = i != 5
      i += 1
      res
    }
  }

  def `ScalaCheck property that takes 2 args, which fails, with default minSuccessful param set to 5` {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSuccessful = 5)

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      var i = 0
      check { (a: String, b: String) =>
        val res = i != 4
        i += 1
        res
      }
    }
  }

  def `ScalaCheck property that takes 2 args and generators, which succeeds, with default minSuccessful param set to 5` {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSuccessful = 5)

    var i = 0
    val prop = forAll (famousLastWords, famousLastWords) { (a: String, b: String) =>
      val res = i != 5
      i += 1
      res
    }
    check(prop)
  }

  def `ScalaCheck property that takes 2 args and generators, which fails, with default minSuccessful param set to 5` {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSuccessful = 5)

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      var i = 0
      val prop = forAll (famousLastWords, famousLastWords) { (a: String, b: String) =>
        val res = i != 4
        i += 1
        res
      }
      check(prop)
    }
  }

  // Same thing, but set maxDiscarded to 5 with param, prop fails after 5
  def `ScalaCheck property that takes 2 args, which succeeds, with maxDiscarded param set to 5` {

    var i = 0
    check(
      (a: String, b: String) => {
        i += 1
        (i > 5) ==> { 1 + 1 == (2) }
      },
      maxDiscarded(5)
    )
  }

  def `ScalaCheck property that takes 2 args, which fails, with maxDiscarded param set to 5` {

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      var i = 0
      check(
        (a: String, b: String) => {
          i += 1
          (i > 6) ==> { 1 + 1 == (2) }
        },
        maxDiscarded(5)
      ) 
    }
  }

  def `ScalaCheck property that takes 2 args and generators, which succeeds, with maxDiscarded param set to 5` {

    var i = 0
    val prop = forAll (famousLastWords, famousLastWords) { (a: String, b: String) =>
      i += 1
      (i > 5) ==> { 1 + 1 == (2) }
    }
    check(prop, maxDiscarded(5))
  }

  def `ScalaCheck property that takes 2 args and generators, which fails, with maxDiscarded param set to 5` {

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      var i = 0
      val prop = forAll (famousLastWords, famousLastWords) { (a: String, b: String) =>
        i += 1
        (i > 6) ==> { 1 + 1 == (2) }
      }
      check(prop, maxDiscarded(5))
    }
  }

  // Same thing, but set default maxDiscarded to 5, prop fails after 5
  def `ScalaCheck property that takes 2 args, which succeeds, with default maxDiscarded set to 5` {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(maxDiscarded = 5)

    var i = 0
    check { (a: String, b: String) =>
      i += 1
      (i > 5) ==> { 1 + 1 == (2) }
    }
  }

  def `ScalaCheck property that takes 2 args, which fails, with default maxDiscarded set to 5` {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(maxDiscarded = 5)

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      var i = 0
      check { (a: String, b: String) =>
        i += 1
        (i > 6) ==> { 1 + 1 == (2) }
      }
    }
  }

  def `ScalaCheck property that takes 2 args and generators, which succeeds, with default maxDiscarded set to 5` {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(maxDiscarded = 5)

    var i = 0
    val prop = forAll (famousLastWords, famousLastWords) { (a: String, b: String) =>
      i += 1
      (i > 5) ==> { 1 + 1 == (2) }
    }
    check(prop)
  }

  def `ScalaCheck property that takes 2 args and generators, which fails, with default maxDiscarded set to 5` {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(maxDiscarded = 5)

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      var i = 0
      val prop = forAll (famousLastWords, famousLastWords) { (a: String, b: String) =>
        i += 1
        (i > 6) ==> { 1 + 1 == (2) }
      }
      check(prop)
    }
  }

  // set minSize > maxSize with (param, param) (intercept IAE)
  def `ScalaCheck property that takes 2 args, which should throw IAE because maxSize > maxSize, specified as (param, param)` {

    intercept[IllegalArgumentException] {
      check(
        (a: String, b: String) => {
          1 + 1 == (2)
        },
        minSize(5),
        maxSize(4)
      ) 
    }
  }

  def `ScalaCheck property that takes 2 args and generators, which should throw IAE because maxSize > maxSize, specified as (param, param)` {

    intercept[IllegalArgumentException] {
      val prop = forAll (famousLastWords, famousLastWords) { (a: String, b: String) =>
        1 + 1 == (2)
      }
      check(prop, minSize(5), maxSize(4))
    }
  }

  // set minSize > maxSize with (param, default) (intercept IAE)
  def `ScalaCheck property that takes 2 args, which should throw IAE because maxSize > maxSize, specified as (param, default)` {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(maxSize = 4)

    intercept[IllegalArgumentException] {
      check(
        (a: String, b: String) => {
          1 + 1 == (2)
        },
        minSize(5)
      )
    }
  }

  def `ScalaCheck property that takes 2 args and generators, which should throw IAE because maxSize > maxSize, specified as (param, default)` {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(maxSize = 4)

    intercept[IllegalArgumentException] {
      val prop = forAll (famousLastWords, famousLastWords) { (a: String, b: String) =>
        1 + 1 == (2)
      }
      check(prop, minSize(5))
    }
  }

  // set minSize > maxSize with (default, param) (intercept IAE)
  def `ScalaCheck property that takes 2 args, which should throw IAE because maxSize > maxSize, specified as (default, param)` {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSize = 5)

    intercept[IllegalArgumentException] {
      check(
        (a: String, b: String) => {
          1 + 1 == (2)
        },
        maxSize(4)
      )
    }
  }

  def `ScalaCheck property that takes 2 args and generators, which should throw IAE because maxSize > maxSize, specified as (default, param)` {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSize = 5)

    intercept[IllegalArgumentException] {
      val prop = forAll (famousLastWords, famousLastWords) { (a: String, b: String) =>
        1 + 1 == (2)
      }
      check(prop, maxSize(4))
    }
  }

  // set maxSize with param (ensure always passed with a size less than maxSize)
  def `ScalaCheck property that takes 2 args, with maxSize specified as param` {

    check(
      (a: String, b: String) => {
      a.length <= 5
      b.length <= 5
      },
      maxSize(5)
    ) 
  }

  // set maxSize with default (ensure always passed with a size less than maxSize)
  def `ScalaCheck property that takes 2 args, with maxSize specified as default` {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(maxSize = 5)

    check { (a: String, b: String) =>
      a.length <= 5
      b.length <= 5
    }
  }

  // set minSize == maxSize with (param, param) (ensure always passed with that size)
  def `ScalaCheck property that takes 2 args and generators, with minSize == maxSize, specified as (param, param)` {

    val prop = forAll (fiveFive, fiveFive) { (a: String, b: String) =>
        a == ("OKAY")
        b == ("OKAY")
    }
    check(prop, minSize(5), maxSize(5))
  }

  // set minSize == maxSize with (param, default) (ensure always passed with that size)
  def `ScalaCheck property that takes 2 args and generators, with minSize == maxSize, specified as (param, default)` {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(maxSize = 5)

    val prop = forAll (fiveFive, fiveFive) { (a: String, b: String) =>
        a == ("OKAY")
        b == ("OKAY")
    }
    check(prop, minSize(5))
  }

  // set minSize == maxSize with (default, param) (ensure always passed with that size)
  def `ScalaCheck property that takes 2 args and generators, with minSize == maxSize, specified as (default, param)` {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSize = 5)

    val prop = forAll (fiveFive, fiveFive) { (a: String, b: String) =>
        a == ("OKAY")
        b == ("OKAY")
    }
    check(prop, maxSize(5))
  }

  // set minSize == maxSize with (default, default) (ensure always passed with that size)
  def `ScalaCheck property that takes 2 args and generators, with minSize == maxSize, specified as (default, default)` {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSize = 5, maxSize = 5)

    val prop = forAll (fiveFive, fiveFive) { (a: String, b: String) =>
        a == ("OKAY")
        b == ("OKAY")
    }
    check(prop)
  }

  // set minSize to 7 and maxSize to 11 with (param, param) (ensure always passed with that size)
  def `ScalaCheck property that takes 2 args and generators, with minSize to 7 and maxSize to 11, specified as (param, param)` {

    val prop = forAll (sevenEleven, sevenEleven) { (a: String, b: String) =>
        a == ("OKAY")
        b == ("OKAY")
    }
    check(prop, minSize(7), maxSize(11))
  }

  // set minSize to 7 and maxSize to 11 with (param, default) (ensure always passed with that size)
  def `ScalaCheck property that takes 2 args and generators, with minSize to 7 and maxSize to 11, specified as (param, default)` {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(maxSize = 11)

    val prop = forAll (sevenEleven, sevenEleven) { (a: String, b: String) =>
        a == ("OKAY")
        b == ("OKAY")
    }
    check(prop, minSize(7))
  }

  // set minSize to 7 and maxSize to 11 with (default, param) (ensure always passed with that size)
  def `ScalaCheck property that takes 2 args and generators, with minSize to 7 and maxSize to 11, specified as (default, param)` {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSize = 7)

    val prop = forAll (sevenEleven, sevenEleven) { (a: String, b: String) =>
        a == ("OKAY")
        b == ("OKAY")
    }
    check(prop, maxSize(11))
  }

  // set minSize to 7 and maxSize to 11 with (default, default) (ensure always passed with that size)
  def `ScalaCheck property that takes 2 args and generators, with minSize to 7 and maxSize to 11, specified as (default, default)` {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSize = 7, maxSize = 11)

    val prop = forAll (sevenEleven, sevenEleven) { (a: String, b: String) =>
        a == ("OKAY")
        b == ("OKAY")
    }
    check(prop)
  }


  def `ScalaCheck property that takes 3 args, which succeeds` {

    check { (a: String, b: String, c: String) =>
      a.length + b.length + c.length == ((a + b + c).length)
    }
  }

  def `ScalaCheck property that takes 3 args, which fails` {

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      check { (a: String, b: String, c: String) =>
        a.length + b.length + c.length < 0
      }
    }
  }

  def `ScalaCheck property that takes 3 args and generators, which succeeds` {

    val prop = forAll (famousLastWords, famousLastWords, famousLastWords) { (a: String, b: String, c: String) =>
      a.length + b.length + c.length == ((a + b + c).length)
    }
    check(prop)
  }

  def `ScalaCheck property that takes 3 args and generators, which fails` {

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      val prop = forAll (famousLastWords, famousLastWords, famousLastWords) { (a: String, b: String, c: String) =>
        a.length + b.length + c.length < 0
      }
      check(prop)
    }
  }

  // Same thing, but with config params
  def `ScalaCheck property that takes 3 args, which succeeds, with config params` {

    check(
      (a: String, b: String, c: String) => a.length + b.length + c.length == ((a + b + c).length),
      minSize(10),
      maxSize(20)
    )
  }

  def `ScalaCheck property that takes 3 args, which fails, with config params` {

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      check(
        (a: String, b: String, c: String) => a.length + b.length + c.length < 0,
        minSize(10),
        maxSize(20)
      )
    }
  }

  def `ScalaCheck property that takes 3 args and generators, which succeeds, with config params` {

    val prop = forAll (famousLastWords, famousLastWords, famousLastWords) { (a: String, b: String, c: String) =>
      a.length + b.length + c.length == ((a + b + c).length)
    }
    check(prop, minSize(10), maxSize(20))
  }

  def `ScalaCheck property that takes 3 args and generators, which fails, with config params` {

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      val prop = forAll (famousLastWords, famousLastWords, famousLastWords) { (a: String, b: String, c: String) =>
        a.length + b.length + c.length < 0
      }
      check(prop, minSize(10), maxSize(20))
    }
  }

  // Same thing, but set minSuccessful to 5 with param, prop fails after 5
  def `ScalaCheck property that takes 3 args, which succeeds, with minSuccessful param set to 5` {

    var i = 0
    check(
      (a: String, b: String, c: String) => {
        val res = i != 5
        i += 1
        res
      },
      minSuccessful(5)
    )
  }

  def `ScalaCheck property that takes 3 args, which fails, with minSuccessful param set to 5` {

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      var i = 0
      check(
        (a: String, b: String, c: String) => {
          val res = i != 4
          i += 1
        res
        },
        minSuccessful(5)
      ) 
    }
  }

  def `ScalaCheck property that takes 3 args and generators, which succeeds, with minSuccessful param set to 5` {

    var i = 0
    val prop = forAll (famousLastWords, famousLastWords, famousLastWords) { (a: String, b: String, c: String) =>
      val res = i != 5
      i += 1
      res
    }
    check(prop, minSuccessful(5))
  }

  def `ScalaCheck property that takes 3 args and generators, which fails, with minSuccessful param set to 5` {

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      var i = 0
      val prop = forAll (famousLastWords, famousLastWords, famousLastWords) { (a: String, b: String, c: String) =>
        val res = i != 4
        i += 1
        res
      }
      check(prop, minSuccessful(5))
    }
  }

  // Same thing, but set default minSuccessful to 5, prop fails after 5
  def `ScalaCheck property that takes 3 args, which succeeds, with default minSuccessful param set to 5` {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSuccessful = 5)

    var i = 0
    check { (a: String, b: String, c: String) =>
      val res = i != 5
      i += 1
      res
    }
  }

  def `ScalaCheck property that takes 3 args, which fails, with default minSuccessful param set to 5` {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSuccessful = 5)

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      var i = 0
      check { (a: String, b: String, c: String) =>
        val res = i != 4
        i += 1
        res
      }
    }
  }

  def `ScalaCheck property that takes 3 args and generators, which succeeds, with default minSuccessful param set to 5` {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSuccessful = 5)

    var i = 0
    val prop = forAll (famousLastWords, famousLastWords, famousLastWords) { (a: String, b: String, c: String) =>
      val res = i != 5
      i += 1
      res
    }
    check(prop)
  }

  def `ScalaCheck property that takes 3 args and generators, which fails, with default minSuccessful param set to 5` {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSuccessful = 5)

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      var i = 0
      val prop = forAll (famousLastWords, famousLastWords, famousLastWords) { (a: String, b: String, c: String) =>
        val res = i != 4
        i += 1
        res
      }
      check(prop)
    }
  }

  // Same thing, but set maxDiscarded to 5 with param, prop fails after 5
  def `ScalaCheck property that takes 3 args, which succeeds, with maxDiscarded param set to 5` {

    var i = 0
    check(
      (a: String, b: String, c: String) => {
        i += 1
        (i > 5) ==> { 1 + 1 == (2) }
      },
      maxDiscarded(5)
    )
  }

  def `ScalaCheck property that takes 3 args, which fails, with maxDiscarded param set to 5` {

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      var i = 0
      check(
        (a: String, b: String, c: String) => {
          i += 1
          (i > 6) ==> { 1 + 1 == (2) }
        },
        maxDiscarded(5)
      ) 
    }
  }

  def `ScalaCheck property that takes 3 args and generators, which succeeds, with maxDiscarded param set to 5` {

    var i = 0
    val prop = forAll (famousLastWords, famousLastWords, famousLastWords) { (a: String, b: String, c: String) =>
      i += 1
      (i > 5) ==> { 1 + 1 == (2) }
    }
    check(prop, maxDiscarded(5))
  }

  def `ScalaCheck property that takes 3 args and generators, which fails, with maxDiscarded param set to 5` {

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      var i = 0
      val prop = forAll (famousLastWords, famousLastWords, famousLastWords) { (a: String, b: String, c: String) =>
        i += 1
        (i > 6) ==> { 1 + 1 == (2) }
      }
      check(prop, maxDiscarded(5))
    }
  }

  // Same thing, but set default maxDiscarded to 5, prop fails after 5
  def `ScalaCheck property that takes 3 args, which succeeds, with default maxDiscarded set to 5` {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(maxDiscarded = 5)

    var i = 0
    check { (a: String, b: String, c: String) =>
      i += 1
      (i > 5) ==> { 1 + 1 == (2) }
    }
  }

  def `ScalaCheck property that takes 3 args, which fails, with default maxDiscarded set to 5` {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(maxDiscarded = 5)

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      var i = 0
      check { (a: String, b: String, c: String) =>
        i += 1
        (i > 6) ==> { 1 + 1 == (2) }
      }
    }
  }

  def `ScalaCheck property that takes 3 args and generators, which succeeds, with default maxDiscarded set to 5` {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(maxDiscarded = 5)

    var i = 0
    val prop = forAll (famousLastWords, famousLastWords, famousLastWords) { (a: String, b: String, c: String) =>
      i += 1
      (i > 5) ==> { 1 + 1 == (2) }
    }
    check(prop)
  }

  def `ScalaCheck property that takes 3 args and generators, which fails, with default maxDiscarded set to 5` {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(maxDiscarded = 5)

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      var i = 0
      val prop = forAll (famousLastWords, famousLastWords, famousLastWords) { (a: String, b: String, c: String) =>
        i += 1
        (i > 6) ==> { 1 + 1 == (2) }
      }
      check(prop)
    }
  }

  // set minSize > maxSize with (param, param) (intercept IAE)
  def `ScalaCheck property that takes 3 args, which should throw IAE because maxSize > maxSize, specified as (param, param)` {

    intercept[IllegalArgumentException] {
      check(
        (a: String, b: String, c: String) => {
          1 + 1 == (2)
        },
        minSize(5),
        maxSize(4)
      ) 
    }
  }

  def `ScalaCheck property that takes 3 args and generators, which should throw IAE because maxSize > maxSize, specified as (param, param)` {

    intercept[IllegalArgumentException] {
      val prop = forAll (famousLastWords, famousLastWords, famousLastWords) { (a: String, b: String, c: String) =>
        1 + 1 == (2)
      }
      check(prop, minSize(5), maxSize(4))
    }
  }

  // set minSize > maxSize with (param, default) (intercept IAE)
  def `ScalaCheck property that takes 3 args, which should throw IAE because maxSize > maxSize, specified as (param, default)` {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(maxSize = 4)

    intercept[IllegalArgumentException] {
      check(
        (a: String, b: String, c: String) => {
          1 + 1 == (2)
        },
        minSize(5)
      )
    }
  }

  def `ScalaCheck property that takes 3 args and generators, which should throw IAE because maxSize > maxSize, specified as (param, default)` {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(maxSize = 4)

    intercept[IllegalArgumentException] {
      val prop = forAll (famousLastWords, famousLastWords, famousLastWords) { (a: String, b: String, c: String) =>
        1 + 1 == (2)
      }
      check(prop, minSize(5))
    }
  }

  // set minSize > maxSize with (default, param) (intercept IAE)
  def `ScalaCheck property that takes 3 args, which should throw IAE because maxSize > maxSize, specified as (default, param)` {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSize = 5)

    intercept[IllegalArgumentException] {
      check(
        (a: String, b: String, c: String) => {
          1 + 1 == (2)
        },
        maxSize(4)
      )
    }
  }

  def `ScalaCheck property that takes 3 args and generators, which should throw IAE because maxSize > maxSize, specified as (default, param)` {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSize = 5)

    intercept[IllegalArgumentException] {
      val prop = forAll (famousLastWords, famousLastWords, famousLastWords) { (a: String, b: String, c: String) =>
        1 + 1 == (2)
      }
      check(prop, maxSize(4))
    }
  }

  // set maxSize with param (ensure always passed with a size less than maxSize)
  def `ScalaCheck property that takes 3 args, with maxSize specified as param` {

    check(
      (a: String, b: String, c: String) => {
      a.length <= 5
      b.length <= 5
      c.length <= 5
      },
      maxSize(5)
    ) 
  }

  // set maxSize with default (ensure always passed with a size less than maxSize)
  def `ScalaCheck property that takes 3 args, with maxSize specified as default` {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(maxSize = 5)

    check { (a: String, b: String, c: String) =>
      a.length <= 5
      b.length <= 5
      c.length <= 5
    }
  }

  // set minSize == maxSize with (param, param) (ensure always passed with that size)
  def `ScalaCheck property that takes 3 args and generators, with minSize == maxSize, specified as (param, param)` {

    val prop = forAll (fiveFive, fiveFive, fiveFive) { (a: String, b: String, c: String) =>
        a == ("OKAY")
        b == ("OKAY")
        c == ("OKAY")
    }
    check(prop, minSize(5), maxSize(5))
  }

  // set minSize == maxSize with (param, default) (ensure always passed with that size)
  def `ScalaCheck property that takes 3 args and generators, with minSize == maxSize, specified as (param, default)` {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(maxSize = 5)

    val prop = forAll (fiveFive, fiveFive, fiveFive) { (a: String, b: String, c: String) =>
        a == ("OKAY")
        b == ("OKAY")
        c == ("OKAY")
    }
    check(prop, minSize(5))
  }

  // set minSize == maxSize with (default, param) (ensure always passed with that size)
  def `ScalaCheck property that takes 3 args and generators, with minSize == maxSize, specified as (default, param)` {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSize = 5)

    val prop = forAll (fiveFive, fiveFive, fiveFive) { (a: String, b: String, c: String) =>
        a == ("OKAY")
        b == ("OKAY")
        c == ("OKAY")
    }
    check(prop, maxSize(5))
  }

  // set minSize == maxSize with (default, default) (ensure always passed with that size)
  def `ScalaCheck property that takes 3 args and generators, with minSize == maxSize, specified as (default, default)` {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSize = 5, maxSize = 5)

    val prop = forAll (fiveFive, fiveFive, fiveFive) { (a: String, b: String, c: String) =>
        a == ("OKAY")
        b == ("OKAY")
        c == ("OKAY")
    }
    check(prop)
  }

  // set minSize to 7 and maxSize to 11 with (param, param) (ensure always passed with that size)
  def `ScalaCheck property that takes 3 args and generators, with minSize to 7 and maxSize to 11, specified as (param, param)` {

    val prop = forAll (sevenEleven, sevenEleven, sevenEleven) { (a: String, b: String, c: String) =>
        a == ("OKAY")
        b == ("OKAY")
        c == ("OKAY")
    }
    check(prop, minSize(7), maxSize(11))
  }

  // set minSize to 7 and maxSize to 11 with (param, default) (ensure always passed with that size)
  def `ScalaCheck property that takes 3 args and generators, with minSize to 7 and maxSize to 11, specified as (param, default)` {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(maxSize = 11)

    val prop = forAll (sevenEleven, sevenEleven, sevenEleven) { (a: String, b: String, c: String) =>
        a == ("OKAY")
        b == ("OKAY")
        c == ("OKAY")
    }
    check(prop, minSize(7))
  }

  // set minSize to 7 and maxSize to 11 with (default, param) (ensure always passed with that size)
  def `ScalaCheck property that takes 3 args and generators, with minSize to 7 and maxSize to 11, specified as (default, param)` {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSize = 7)

    val prop = forAll (sevenEleven, sevenEleven, sevenEleven) { (a: String, b: String, c: String) =>
        a == ("OKAY")
        b == ("OKAY")
        c == ("OKAY")
    }
    check(prop, maxSize(11))
  }

  // set minSize to 7 and maxSize to 11 with (default, default) (ensure always passed with that size)
  def `ScalaCheck property that takes 3 args and generators, with minSize to 7 and maxSize to 11, specified as (default, default)` {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSize = 7, maxSize = 11)

    val prop = forAll (sevenEleven, sevenEleven, sevenEleven) { (a: String, b: String, c: String) =>
        a == ("OKAY")
        b == ("OKAY")
        c == ("OKAY")
    }
    check(prop)
  }


  def `ScalaCheck property that takes 4 args, which succeeds` {

    check { (a: String, b: String, c: String, d: String) =>
      a.length + b.length + c.length + d.length == ((a + b + c + d).length)
    }
  }

  def `ScalaCheck property that takes 4 args, which fails` {

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      check { (a: String, b: String, c: String, d: String) =>
        a.length + b.length + c.length + d.length < 0
      }
    }
  }

  def `ScalaCheck property that takes 4 args and generators, which succeeds` {

    val prop = forAll (famousLastWords, famousLastWords, famousLastWords, famousLastWords) { (a: String, b: String, c: String, d: String) =>
      a.length + b.length + c.length + d.length == ((a + b + c + d).length)
    }
    check(prop)
  }

  def `ScalaCheck property that takes 4 args and generators, which fails` {

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      val prop = forAll (famousLastWords, famousLastWords, famousLastWords, famousLastWords) { (a: String, b: String, c: String, d: String) =>
        a.length + b.length + c.length + d.length < 0
      }
      check(prop)
    }
  }

  // Same thing, but with config params
  def `ScalaCheck property that takes 4 args, which succeeds, with config params` {

    check(
      (a: String, b: String, c: String, d: String) => a.length + b.length + c.length + d.length == ((a + b + c + d).length),
      minSize(10),
      maxSize(20)
    )
  }

  def `ScalaCheck property that takes 4 args, which fails, with config params` {

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      check(
        (a: String, b: String, c: String, d: String) => a.length + b.length + c.length + d.length < 0,
        minSize(10),
        maxSize(20)
      )
    }
  }

  def `ScalaCheck property that takes 4 args and generators, which succeeds, with config params` {

    val prop = forAll (famousLastWords, famousLastWords, famousLastWords, famousLastWords) { (a: String, b: String, c: String, d: String) =>
      a.length + b.length + c.length + d.length == ((a + b + c + d).length)
    }
    check(prop, minSize(10), maxSize(20))
  }

  def `ScalaCheck property that takes 4 args and generators, which fails, with config params` {

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      val prop = forAll (famousLastWords, famousLastWords, famousLastWords, famousLastWords) { (a: String, b: String, c: String, d: String) =>
        a.length + b.length + c.length + d.length < 0
      }
      check(prop, minSize(10), maxSize(20))
    }
  }

  // Same thing, but set minSuccessful to 5 with param, prop fails after 5
  def `ScalaCheck property that takes 4 args, which succeeds, with minSuccessful param set to 5` {

    var i = 0
    check(
      (a: String, b: String, c: String, d: String) => {
        val res = i != 5
        i += 1
        res
      },
      minSuccessful(5)
    )
  }

  def `ScalaCheck property that takes 4 args, which fails, with minSuccessful param set to 5` {

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      var i = 0
      check(
        (a: String, b: String, c: String, d: String) => {
          val res = i != 4
          i += 1
        res
        },
        minSuccessful(5)
      ) 
    }
  }

  def `ScalaCheck property that takes 4 args and generators, which succeeds, with minSuccessful param set to 5` {

    var i = 0
    val prop = forAll (famousLastWords, famousLastWords, famousLastWords, famousLastWords) { (a: String, b: String, c: String, d: String) =>
      val res = i != 5
      i += 1
      res
    }
    check(prop, minSuccessful(5))
  }

  def `ScalaCheck property that takes 4 args and generators, which fails, with minSuccessful param set to 5` {

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      var i = 0
      val prop = forAll (famousLastWords, famousLastWords, famousLastWords, famousLastWords) { (a: String, b: String, c: String, d: String) =>
        val res = i != 4
        i += 1
        res
      }
      check(prop, minSuccessful(5))
    }
  }

  // Same thing, but set default minSuccessful to 5, prop fails after 5
  def `ScalaCheck property that takes 4 args, which succeeds, with default minSuccessful param set to 5` {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSuccessful = 5)

    var i = 0
    check { (a: String, b: String, c: String, d: String) =>
      val res = i != 5
      i += 1
      res
    }
  }

  def `ScalaCheck property that takes 4 args, which fails, with default minSuccessful param set to 5` {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSuccessful = 5)

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      var i = 0
      check { (a: String, b: String, c: String, d: String) =>
        val res = i != 4
        i += 1
        res
      }
    }
  }

  def `ScalaCheck property that takes 4 args and generators, which succeeds, with default minSuccessful param set to 5` {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSuccessful = 5)

    var i = 0
    val prop = forAll (famousLastWords, famousLastWords, famousLastWords, famousLastWords) { (a: String, b: String, c: String, d: String) =>
      val res = i != 5
      i += 1
      res
    }
    check(prop)
  }

  def `ScalaCheck property that takes 4 args and generators, which fails, with default minSuccessful param set to 5` {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSuccessful = 5)

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      var i = 0
      val prop = forAll (famousLastWords, famousLastWords, famousLastWords, famousLastWords) { (a: String, b: String, c: String, d: String) =>
        val res = i != 4
        i += 1
        res
      }
      check(prop)
    }
  }

  // Same thing, but set maxDiscarded to 5 with param, prop fails after 5
  def `ScalaCheck property that takes 4 args, which succeeds, with maxDiscarded param set to 5` {

    var i = 0
    check(
      (a: String, b: String, c: String, d: String) => {
        i += 1
        (i > 5) ==> { 1 + 1 == (2) }
      },
      maxDiscarded(5)
    )
  }

  def `ScalaCheck property that takes 4 args, which fails, with maxDiscarded param set to 5` {

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      var i = 0
      check(
        (a: String, b: String, c: String, d: String) => {
          i += 1
          (i > 6) ==> { 1 + 1 == (2) }
        },
        maxDiscarded(5)
      ) 
    }
  }

  def `ScalaCheck property that takes 4 args and generators, which succeeds, with maxDiscarded param set to 5` {

    var i = 0
    val prop = forAll (famousLastWords, famousLastWords, famousLastWords, famousLastWords) { (a: String, b: String, c: String, d: String) =>
      i += 1
      (i > 5) ==> { 1 + 1 == (2) }
    }
    check(prop, maxDiscarded(5))
  }

  def `ScalaCheck property that takes 4 args and generators, which fails, with maxDiscarded param set to 5` {

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      var i = 0
      val prop = forAll (famousLastWords, famousLastWords, famousLastWords, famousLastWords) { (a: String, b: String, c: String, d: String) =>
        i += 1
        (i > 6) ==> { 1 + 1 == (2) }
      }
      check(prop, maxDiscarded(5))
    }
  }

  // Same thing, but set default maxDiscarded to 5, prop fails after 5
  def `ScalaCheck property that takes 4 args, which succeeds, with default maxDiscarded set to 5` {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(maxDiscarded = 5)

    var i = 0
    check { (a: String, b: String, c: String, d: String) =>
      i += 1
      (i > 5) ==> { 1 + 1 == (2) }
    }
  }

  def `ScalaCheck property that takes 4 args, which fails, with default maxDiscarded set to 5` {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(maxDiscarded = 5)

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      var i = 0
      check { (a: String, b: String, c: String, d: String) =>
        i += 1
        (i > 6) ==> { 1 + 1 == (2) }
      }
    }
  }

  def `ScalaCheck property that takes 4 args and generators, which succeeds, with default maxDiscarded set to 5` {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(maxDiscarded = 5)

    var i = 0
    val prop = forAll (famousLastWords, famousLastWords, famousLastWords, famousLastWords) { (a: String, b: String, c: String, d: String) =>
      i += 1
      (i > 5) ==> { 1 + 1 == (2) }
    }
    check(prop)
  }

  def `ScalaCheck property that takes 4 args and generators, which fails, with default maxDiscarded set to 5` {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(maxDiscarded = 5)

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      var i = 0
      val prop = forAll (famousLastWords, famousLastWords, famousLastWords, famousLastWords) { (a: String, b: String, c: String, d: String) =>
        i += 1
        (i > 6) ==> { 1 + 1 == (2) }
      }
      check(prop)
    }
  }

  // set minSize > maxSize with (param, param) (intercept IAE)
  def `ScalaCheck property that takes 4 args, which should throw IAE because maxSize > maxSize, specified as (param, param)` {

    intercept[IllegalArgumentException] {
      check(
        (a: String, b: String, c: String, d: String) => {
          1 + 1 == (2)
        },
        minSize(5),
        maxSize(4)
      ) 
    }
  }

  def `ScalaCheck property that takes 4 args and generators, which should throw IAE because maxSize > maxSize, specified as (param, param)` {

    intercept[IllegalArgumentException] {
      val prop = forAll (famousLastWords, famousLastWords, famousLastWords, famousLastWords) { (a: String, b: String, c: String, d: String) =>
        1 + 1 == (2)
      }
      check(prop, minSize(5), maxSize(4))
    }
  }

  // set minSize > maxSize with (param, default) (intercept IAE)
  def `ScalaCheck property that takes 4 args, which should throw IAE because maxSize > maxSize, specified as (param, default)` {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(maxSize = 4)

    intercept[IllegalArgumentException] {
      check(
        (a: String, b: String, c: String, d: String) => {
          1 + 1 == (2)
        },
        minSize(5)
      )
    }
  }

  def `ScalaCheck property that takes 4 args and generators, which should throw IAE because maxSize > maxSize, specified as (param, default)` {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(maxSize = 4)

    intercept[IllegalArgumentException] {
      val prop = forAll (famousLastWords, famousLastWords, famousLastWords, famousLastWords) { (a: String, b: String, c: String, d: String) =>
        1 + 1 == (2)
      }
      check(prop, minSize(5))
    }
  }

  // set minSize > maxSize with (default, param) (intercept IAE)
  def `ScalaCheck property that takes 4 args, which should throw IAE because maxSize > maxSize, specified as (default, param)` {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSize = 5)

    intercept[IllegalArgumentException] {
      check(
        (a: String, b: String, c: String, d: String) => {
          1 + 1 == (2)
        },
        maxSize(4)
      )
    }
  }

  def `ScalaCheck property that takes 4 args and generators, which should throw IAE because maxSize > maxSize, specified as (default, param)` {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSize = 5)

    intercept[IllegalArgumentException] {
      val prop = forAll (famousLastWords, famousLastWords, famousLastWords, famousLastWords) { (a: String, b: String, c: String, d: String) =>
        1 + 1 == (2)
      }
      check(prop, maxSize(4))
    }
  }

  // set maxSize with param (ensure always passed with a size less than maxSize)
  def `ScalaCheck property that takes 4 args, with maxSize specified as param` {

    check(
      (a: String, b: String, c: String, d: String) => {
      a.length <= 5
      b.length <= 5
      c.length <= 5
      d.length <= 5
      },
      maxSize(5)
    ) 
  }

  // set maxSize with default (ensure always passed with a size less than maxSize)
  def `ScalaCheck property that takes 4 args, with maxSize specified as default` {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(maxSize = 5)

    check { (a: String, b: String, c: String, d: String) =>
      a.length <= 5
      b.length <= 5
      c.length <= 5
      d.length <= 5
    }
  }

  // set minSize == maxSize with (param, param) (ensure always passed with that size)
  def `ScalaCheck property that takes 4 args and generators, with minSize == maxSize, specified as (param, param)` {

    val prop = forAll (fiveFive, fiveFive, fiveFive, fiveFive) { (a: String, b: String, c: String, d: String) =>
        a == ("OKAY")
        b == ("OKAY")
        c == ("OKAY")
        d == ("OKAY")
    }
    check(prop, minSize(5), maxSize(5))
  }

  // set minSize == maxSize with (param, default) (ensure always passed with that size)
  def `ScalaCheck property that takes 4 args and generators, with minSize == maxSize, specified as (param, default)` {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(maxSize = 5)

    val prop = forAll (fiveFive, fiveFive, fiveFive, fiveFive) { (a: String, b: String, c: String, d: String) =>
        a == ("OKAY")
        b == ("OKAY")
        c == ("OKAY")
        d == ("OKAY")
    }
    check(prop, minSize(5))
  }

  // set minSize == maxSize with (default, param) (ensure always passed with that size)
  def `ScalaCheck property that takes 4 args and generators, with minSize == maxSize, specified as (default, param)` {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSize = 5)

    val prop = forAll (fiveFive, fiveFive, fiveFive, fiveFive) { (a: String, b: String, c: String, d: String) =>
        a == ("OKAY")
        b == ("OKAY")
        c == ("OKAY")
        d == ("OKAY")
    }
    check(prop, maxSize(5))
  }

  // set minSize == maxSize with (default, default) (ensure always passed with that size)
  def `ScalaCheck property that takes 4 args and generators, with minSize == maxSize, specified as (default, default)` {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSize = 5, maxSize = 5)

    val prop = forAll (fiveFive, fiveFive, fiveFive, fiveFive) { (a: String, b: String, c: String, d: String) =>
        a == ("OKAY")
        b == ("OKAY")
        c == ("OKAY")
        d == ("OKAY")
    }
    check(prop)
  }

  // set minSize to 7 and maxSize to 11 with (param, param) (ensure always passed with that size)
  def `ScalaCheck property that takes 4 args and generators, with minSize to 7 and maxSize to 11, specified as (param, param)` {

    val prop = forAll (sevenEleven, sevenEleven, sevenEleven, sevenEleven) { (a: String, b: String, c: String, d: String) =>
        a == ("OKAY")
        b == ("OKAY")
        c == ("OKAY")
        d == ("OKAY")
    }
    check(prop, minSize(7), maxSize(11))
  }

  // set minSize to 7 and maxSize to 11 with (param, default) (ensure always passed with that size)
  def `ScalaCheck property that takes 4 args and generators, with minSize to 7 and maxSize to 11, specified as (param, default)` {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(maxSize = 11)

    val prop = forAll (sevenEleven, sevenEleven, sevenEleven, sevenEleven) { (a: String, b: String, c: String, d: String) =>
        a == ("OKAY")
        b == ("OKAY")
        c == ("OKAY")
        d == ("OKAY")
    }
    check(prop, minSize(7))
  }

  // set minSize to 7 and maxSize to 11 with (default, param) (ensure always passed with that size)
  def `ScalaCheck property that takes 4 args and generators, with minSize to 7 and maxSize to 11, specified as (default, param)` {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSize = 7)

    val prop = forAll (sevenEleven, sevenEleven, sevenEleven, sevenEleven) { (a: String, b: String, c: String, d: String) =>
        a == ("OKAY")
        b == ("OKAY")
        c == ("OKAY")
        d == ("OKAY")
    }
    check(prop, maxSize(11))
  }

  // set minSize to 7 and maxSize to 11 with (default, default) (ensure always passed with that size)
  def `ScalaCheck property that takes 4 args and generators, with minSize to 7 and maxSize to 11, specified as (default, default)` {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSize = 7, maxSize = 11)

    val prop = forAll (sevenEleven, sevenEleven, sevenEleven, sevenEleven) { (a: String, b: String, c: String, d: String) =>
        a == ("OKAY")
        b == ("OKAY")
        c == ("OKAY")
        d == ("OKAY")
    }
    check(prop)
  }


  def `ScalaCheck property that takes 5 args, which succeeds` {

    check { (a: String, b: String, c: String, d: String, e: String) =>
      a.length + b.length + c.length + d.length + e.length == ((a + b + c + d + e).length)
    }
  }

  def `ScalaCheck property that takes 5 args, which fails` {

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      check { (a: String, b: String, c: String, d: String, e: String) =>
        a.length + b.length + c.length + d.length + e.length < 0
      }
    }
  }

  def `ScalaCheck property that takes 5 args and generators, which succeeds` {

    val prop = forAll (famousLastWords, famousLastWords, famousLastWords, famousLastWords, famousLastWords) { (a: String, b: String, c: String, d: String, e: String) =>
      a.length + b.length + c.length + d.length + e.length == ((a + b + c + d + e).length)
    }
    check(prop)
  }

  def `ScalaCheck property that takes 5 args and generators, which fails` {

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      val prop = forAll (famousLastWords, famousLastWords, famousLastWords, famousLastWords, famousLastWords) { (a: String, b: String, c: String, d: String, e: String) =>
        a.length + b.length + c.length + d.length + e.length < 0
      }
      check(prop)
    }
  }

  // Same thing, but with config params
  def `ScalaCheck property that takes 5 args, which succeeds, with config params` {

    check(
      (a: String, b: String, c: String, d: String, e: String) => a.length + b.length + c.length + d.length + e.length == ((a + b + c + d + e).length),
      minSize(10),
      maxSize(20)
    )
  }

  def `ScalaCheck property that takes 5 args, which fails, with config params` {

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      check(
        (a: String, b: String, c: String, d: String, e: String) => a.length + b.length + c.length + d.length + e.length < 0,
        minSize(10),
        maxSize(20)
      )
    }
  }

  def `ScalaCheck property that takes 5 args and generators, which succeeds, with config params` {

    val prop = forAll (famousLastWords, famousLastWords, famousLastWords, famousLastWords, famousLastWords) { (a: String, b: String, c: String, d: String, e: String) =>
      a.length + b.length + c.length + d.length + e.length == ((a + b + c + d + e).length)
    }
    check(prop, minSize(10), maxSize(20))
  }

  def `ScalaCheck property that takes 5 args and generators, which fails, with config params` {

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      val prop = forAll (famousLastWords, famousLastWords, famousLastWords, famousLastWords, famousLastWords) { (a: String, b: String, c: String, d: String, e: String) =>
        a.length + b.length + c.length + d.length + e.length < 0
      }
      check(prop, minSize(10), maxSize(20))
    }
  }

  // Same thing, but set minSuccessful to 5 with param, prop fails after 5
  def `ScalaCheck property that takes 5 args, which succeeds, with minSuccessful param set to 5` {

    var i = 0
    check(
      (a: String, b: String, c: String, d: String, e: String) => {
        val res = i != 5
        i += 1
        res
      },
      minSuccessful(5)
    )
  }

  def `ScalaCheck property that takes 5 args, which fails, with minSuccessful param set to 5` {

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      var i = 0
      check(
        (a: String, b: String, c: String, d: String, e: String) => {
          val res = i != 4
          i += 1
        res
        },
        minSuccessful(5)
      ) 
    }
  }

  def `ScalaCheck property that takes 5 args and generators, which succeeds, with minSuccessful param set to 5` {

    var i = 0
    val prop = forAll (famousLastWords, famousLastWords, famousLastWords, famousLastWords, famousLastWords) { (a: String, b: String, c: String, d: String, e: String) =>
      val res = i != 5
      i += 1
      res
    }
    check(prop, minSuccessful(5))
  }

  def `ScalaCheck property that takes 5 args and generators, which fails, with minSuccessful param set to 5` {

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      var i = 0
      val prop = forAll (famousLastWords, famousLastWords, famousLastWords, famousLastWords, famousLastWords) { (a: String, b: String, c: String, d: String, e: String) =>
        val res = i != 4
        i += 1
        res
      }
      check(prop, minSuccessful(5))
    }
  }

  // Same thing, but set default minSuccessful to 5, prop fails after 5
  def `ScalaCheck property that takes 5 args, which succeeds, with default minSuccessful param set to 5` {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSuccessful = 5)

    var i = 0
    check { (a: String, b: String, c: String, d: String, e: String) =>
      val res = i != 5
      i += 1
      res
    }
  }

  def `ScalaCheck property that takes 5 args, which fails, with default minSuccessful param set to 5` {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSuccessful = 5)

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      var i = 0
      check { (a: String, b: String, c: String, d: String, e: String) =>
        val res = i != 4
        i += 1
        res
      }
    }
  }

  def `ScalaCheck property that takes 5 args and generators, which succeeds, with default minSuccessful param set to 5` {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSuccessful = 5)

    var i = 0
    val prop = forAll (famousLastWords, famousLastWords, famousLastWords, famousLastWords, famousLastWords) { (a: String, b: String, c: String, d: String, e: String) =>
      val res = i != 5
      i += 1
      res
    }
    check(prop)
  }

  def `ScalaCheck property that takes 5 args and generators, which fails, with default minSuccessful param set to 5` {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSuccessful = 5)

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      var i = 0
      val prop = forAll (famousLastWords, famousLastWords, famousLastWords, famousLastWords, famousLastWords) { (a: String, b: String, c: String, d: String, e: String) =>
        val res = i != 4
        i += 1
        res
      }
      check(prop)
    }
  }

  // Same thing, but set maxDiscarded to 5 with param, prop fails after 5
  def `ScalaCheck property that takes 5 args, which succeeds, with maxDiscarded param set to 5` {

    var i = 0
    check(
      (a: String, b: String, c: String, d: String, e: String) => {
        i += 1
        (i > 5) ==> { 1 + 1 == (2) }
      },
      maxDiscarded(5)
    )
  }

  def `ScalaCheck property that takes 5 args, which fails, with maxDiscarded param set to 5` {

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      var i = 0
      check(
        (a: String, b: String, c: String, d: String, e: String) => {
          i += 1
          (i > 6) ==> { 1 + 1 == (2) }
        },
        maxDiscarded(5)
      ) 
    }
  }

  def `ScalaCheck property that takes 5 args and generators, which succeeds, with maxDiscarded param set to 5` {

    var i = 0
    val prop = forAll (famousLastWords, famousLastWords, famousLastWords, famousLastWords, famousLastWords) { (a: String, b: String, c: String, d: String, e: String) =>
      i += 1
      (i > 5) ==> { 1 + 1 == (2) }
    }
    check(prop, maxDiscarded(5))
  }

  def `ScalaCheck property that takes 5 args and generators, which fails, with maxDiscarded param set to 5` {

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      var i = 0
      val prop = forAll (famousLastWords, famousLastWords, famousLastWords, famousLastWords, famousLastWords) { (a: String, b: String, c: String, d: String, e: String) =>
        i += 1
        (i > 6) ==> { 1 + 1 == (2) }
      }
      check(prop, maxDiscarded(5))
    }
  }

  // Same thing, but set default maxDiscarded to 5, prop fails after 5
  def `ScalaCheck property that takes 5 args, which succeeds, with default maxDiscarded set to 5` {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(maxDiscarded = 5)

    var i = 0
    check { (a: String, b: String, c: String, d: String, e: String) =>
      i += 1
      (i > 5) ==> { 1 + 1 == (2) }
    }
  }

  def `ScalaCheck property that takes 5 args, which fails, with default maxDiscarded set to 5` {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(maxDiscarded = 5)

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      var i = 0
      check { (a: String, b: String, c: String, d: String, e: String) =>
        i += 1
        (i > 6) ==> { 1 + 1 == (2) }
      }
    }
  }

  def `ScalaCheck property that takes 5 args and generators, which succeeds, with default maxDiscarded set to 5` {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(maxDiscarded = 5)

    var i = 0
    val prop = forAll (famousLastWords, famousLastWords, famousLastWords, famousLastWords, famousLastWords) { (a: String, b: String, c: String, d: String, e: String) =>
      i += 1
      (i > 5) ==> { 1 + 1 == (2) }
    }
    check(prop)
  }

  def `ScalaCheck property that takes 5 args and generators, which fails, with default maxDiscarded set to 5` {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(maxDiscarded = 5)

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      var i = 0
      val prop = forAll (famousLastWords, famousLastWords, famousLastWords, famousLastWords, famousLastWords) { (a: String, b: String, c: String, d: String, e: String) =>
        i += 1
        (i > 6) ==> { 1 + 1 == (2) }
      }
      check(prop)
    }
  }

  // set minSize > maxSize with (param, param) (intercept IAE)
  def `ScalaCheck property that takes 5 args, which should throw IAE because maxSize > maxSize, specified as (param, param)` {

    intercept[IllegalArgumentException] {
      check(
        (a: String, b: String, c: String, d: String, e: String) => {
          1 + 1 == (2)
        },
        minSize(5),
        maxSize(4)
      ) 
    }
  }

  def `ScalaCheck property that takes 5 args and generators, which should throw IAE because maxSize > maxSize, specified as (param, param)` {

    intercept[IllegalArgumentException] {
      val prop = forAll (famousLastWords, famousLastWords, famousLastWords, famousLastWords, famousLastWords) { (a: String, b: String, c: String, d: String, e: String) =>
        1 + 1 == (2)
      }
      check(prop, minSize(5), maxSize(4))
    }
  }

  // set minSize > maxSize with (param, default) (intercept IAE)
  def `ScalaCheck property that takes 5 args, which should throw IAE because maxSize > maxSize, specified as (param, default)` {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(maxSize = 4)

    intercept[IllegalArgumentException] {
      check(
        (a: String, b: String, c: String, d: String, e: String) => {
          1 + 1 == (2)
        },
        minSize(5)
      )
    }
  }

  def `ScalaCheck property that takes 5 args and generators, which should throw IAE because maxSize > maxSize, specified as (param, default)` {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(maxSize = 4)

    intercept[IllegalArgumentException] {
      val prop = forAll (famousLastWords, famousLastWords, famousLastWords, famousLastWords, famousLastWords) { (a: String, b: String, c: String, d: String, e: String) =>
        1 + 1 == (2)
      }
      check(prop, minSize(5))
    }
  }

  // set minSize > maxSize with (default, param) (intercept IAE)
  def `ScalaCheck property that takes 5 args, which should throw IAE because maxSize > maxSize, specified as (default, param)` {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSize = 5)

    intercept[IllegalArgumentException] {
      check(
        (a: String, b: String, c: String, d: String, e: String) => {
          1 + 1 == (2)
        },
        maxSize(4)
      )
    }
  }

  def `ScalaCheck property that takes 5 args and generators, which should throw IAE because maxSize > maxSize, specified as (default, param)` {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSize = 5)

    intercept[IllegalArgumentException] {
      val prop = forAll (famousLastWords, famousLastWords, famousLastWords, famousLastWords, famousLastWords) { (a: String, b: String, c: String, d: String, e: String) =>
        1 + 1 == (2)
      }
      check(prop, maxSize(4))
    }
  }

  // set maxSize with param (ensure always passed with a size less than maxSize)
  def `ScalaCheck property that takes 5 args, with maxSize specified as param` {

    check(
      (a: String, b: String, c: String, d: String, e: String) => {
      a.length <= 5
      b.length <= 5
      c.length <= 5
      d.length <= 5
      e.length <= 5
      },
      maxSize(5)
    ) 
  }

  // set maxSize with default (ensure always passed with a size less than maxSize)
  def `ScalaCheck property that takes 5 args, with maxSize specified as default` {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(maxSize = 5)

    check { (a: String, b: String, c: String, d: String, e: String) =>
      a.length <= 5
      b.length <= 5
      c.length <= 5
      d.length <= 5
      e.length <= 5
    }
  }

  // set minSize == maxSize with (param, param) (ensure always passed with that size)
  def `ScalaCheck property that takes 5 args and generators, with minSize == maxSize, specified as (param, param)` {

    val prop = forAll (fiveFive, fiveFive, fiveFive, fiveFive, fiveFive) { (a: String, b: String, c: String, d: String, e: String) =>
        a == ("OKAY")
        b == ("OKAY")
        c == ("OKAY")
        d == ("OKAY")
        e == ("OKAY")
    }
    check(prop, minSize(5), maxSize(5))
  }

  // set minSize == maxSize with (param, default) (ensure always passed with that size)
  def `ScalaCheck property that takes 5 args and generators, with minSize == maxSize, specified as (param, default)` {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(maxSize = 5)

    val prop = forAll (fiveFive, fiveFive, fiveFive, fiveFive, fiveFive) { (a: String, b: String, c: String, d: String, e: String) =>
        a == ("OKAY")
        b == ("OKAY")
        c == ("OKAY")
        d == ("OKAY")
        e == ("OKAY")
    }
    check(prop, minSize(5))
  }

  // set minSize == maxSize with (default, param) (ensure always passed with that size)
  def `ScalaCheck property that takes 5 args and generators, with minSize == maxSize, specified as (default, param)` {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSize = 5)

    val prop = forAll (fiveFive, fiveFive, fiveFive, fiveFive, fiveFive) { (a: String, b: String, c: String, d: String, e: String) =>
        a == ("OKAY")
        b == ("OKAY")
        c == ("OKAY")
        d == ("OKAY")
        e == ("OKAY")
    }
    check(prop, maxSize(5))
  }

  // set minSize == maxSize with (default, default) (ensure always passed with that size)
  def `ScalaCheck property that takes 5 args and generators, with minSize == maxSize, specified as (default, default)` {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSize = 5, maxSize = 5)

    val prop = forAll (fiveFive, fiveFive, fiveFive, fiveFive, fiveFive) { (a: String, b: String, c: String, d: String, e: String) =>
        a == ("OKAY")
        b == ("OKAY")
        c == ("OKAY")
        d == ("OKAY")
        e == ("OKAY")
    }
    check(prop)
  }

  // set minSize to 7 and maxSize to 11 with (param, param) (ensure always passed with that size)
  def `ScalaCheck property that takes 5 args and generators, with minSize to 7 and maxSize to 11, specified as (param, param)` {

    val prop = forAll (sevenEleven, sevenEleven, sevenEleven, sevenEleven, sevenEleven) { (a: String, b: String, c: String, d: String, e: String) =>
        a == ("OKAY")
        b == ("OKAY")
        c == ("OKAY")
        d == ("OKAY")
        e == ("OKAY")
    }
    check(prop, minSize(7), maxSize(11))
  }

  // set minSize to 7 and maxSize to 11 with (param, default) (ensure always passed with that size)
  def `ScalaCheck property that takes 5 args and generators, with minSize to 7 and maxSize to 11, specified as (param, default)` {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(maxSize = 11)

    val prop = forAll (sevenEleven, sevenEleven, sevenEleven, sevenEleven, sevenEleven) { (a: String, b: String, c: String, d: String, e: String) =>
        a == ("OKAY")
        b == ("OKAY")
        c == ("OKAY")
        d == ("OKAY")
        e == ("OKAY")
    }
    check(prop, minSize(7))
  }

  // set minSize to 7 and maxSize to 11 with (default, param) (ensure always passed with that size)
  def `ScalaCheck property that takes 5 args and generators, with minSize to 7 and maxSize to 11, specified as (default, param)` {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSize = 7)

    val prop = forAll (sevenEleven, sevenEleven, sevenEleven, sevenEleven, sevenEleven) { (a: String, b: String, c: String, d: String, e: String) =>
        a == ("OKAY")
        b == ("OKAY")
        c == ("OKAY")
        d == ("OKAY")
        e == ("OKAY")
    }
    check(prop, maxSize(11))
  }

  // set minSize to 7 and maxSize to 11 with (default, default) (ensure always passed with that size)
  def `ScalaCheck property that takes 5 args and generators, with minSize to 7 and maxSize to 11, specified as (default, default)` {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSize = 7, maxSize = 11)

    val prop = forAll (sevenEleven, sevenEleven, sevenEleven, sevenEleven, sevenEleven) { (a: String, b: String, c: String, d: String, e: String) =>
        a == ("OKAY")
        b == ("OKAY")
        c == ("OKAY")
        d == ("OKAY")
        e == ("OKAY")
    }
    check(prop)
  }


  def `ScalaCheck property that takes 6 args, which succeeds` {

    check { (a: String, b: String, c: String, d: String, e: String, f: String) =>
      a.length + b.length + c.length + d.length + e.length + f.length == ((a + b + c + d + e + f).length)
    }
  }

  def `ScalaCheck property that takes 6 args, which fails` {

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      check { (a: String, b: String, c: String, d: String, e: String, f: String) =>
        a.length + b.length + c.length + d.length + e.length + f.length < 0
      }
    }
  }

  def `ScalaCheck property that takes 6 args and generators, which succeeds` {

    val prop = forAll (famousLastWords, famousLastWords, famousLastWords, famousLastWords, famousLastWords, famousLastWords) { (a: String, b: String, c: String, d: String, e: String, f: String) =>
      a.length + b.length + c.length + d.length + e.length + f.length == ((a + b + c + d + e + f).length)
    }
    check(prop)
  }

  def `ScalaCheck property that takes 6 args and generators, which fails` {

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      val prop = forAll (famousLastWords, famousLastWords, famousLastWords, famousLastWords, famousLastWords, famousLastWords) { (a: String, b: String, c: String, d: String, e: String, f: String) =>
        a.length + b.length + c.length + d.length + e.length + f.length < 0
      }
      check(prop)
    }
  }

  // Same thing, but with config params
  def `ScalaCheck property that takes 6 args, which succeeds, with config params` {

    check(
      (a: String, b: String, c: String, d: String, e: String, f: String) => a.length + b.length + c.length + d.length + e.length + f.length == ((a + b + c + d + e + f).length),
      minSize(10),
      maxSize(20)
    )
  }

  def `ScalaCheck property that takes 6 args, which fails, with config params` {

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      check(
        (a: String, b: String, c: String, d: String, e: String, f: String) => a.length + b.length + c.length + d.length + e.length + f.length < 0,
        minSize(10),
        maxSize(20)
      )
    }
  }

  def `ScalaCheck property that takes 6 args and generators, which succeeds, with config params` {

    val prop = forAll (famousLastWords, famousLastWords, famousLastWords, famousLastWords, famousLastWords, famousLastWords) { (a: String, b: String, c: String, d: String, e: String, f: String) =>
      a.length + b.length + c.length + d.length + e.length + f.length == ((a + b + c + d + e + f).length)
    }
    check(prop, minSize(10), maxSize(20))
  }

  def `ScalaCheck property that takes 6 args and generators, which fails, with config params` {

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      val prop = forAll (famousLastWords, famousLastWords, famousLastWords, famousLastWords, famousLastWords, famousLastWords) { (a: String, b: String, c: String, d: String, e: String, f: String) =>
        a.length + b.length + c.length + d.length + e.length + f.length < 0
      }
      check(prop, minSize(10), maxSize(20))
    }
  }

  // Same thing, but set minSuccessful to 5 with param, prop fails after 5
  def `ScalaCheck property that takes 6 args, which succeeds, with minSuccessful param set to 5` {

    var i = 0
    check(
      (a: String, b: String, c: String, d: String, e: String, f: String) => {
        val res = i != 5
        i += 1
        res
      },
      minSuccessful(5)
    )
  }

  def `ScalaCheck property that takes 6 args, which fails, with minSuccessful param set to 5` {

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      var i = 0
      check(
        (a: String, b: String, c: String, d: String, e: String, f: String) => {
          val res = i != 4
          i += 1
        res
        },
        minSuccessful(5)
      ) 
    }
  }

  def `ScalaCheck property that takes 6 args and generators, which succeeds, with minSuccessful param set to 5` {

    var i = 0
    val prop = forAll (famousLastWords, famousLastWords, famousLastWords, famousLastWords, famousLastWords, famousLastWords) { (a: String, b: String, c: String, d: String, e: String, f: String) =>
      val res = i != 5
      i += 1
      res
    }
    check(prop, minSuccessful(5))
  }

  def `ScalaCheck property that takes 6 args and generators, which fails, with minSuccessful param set to 5` {

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      var i = 0
      val prop = forAll (famousLastWords, famousLastWords, famousLastWords, famousLastWords, famousLastWords, famousLastWords) { (a: String, b: String, c: String, d: String, e: String, f: String) =>
        val res = i != 4
        i += 1
        res
      }
      check(prop, minSuccessful(5))
    }
  }

  // Same thing, but set default minSuccessful to 5, prop fails after 5
  def `ScalaCheck property that takes 6 args, which succeeds, with default minSuccessful param set to 5` {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSuccessful = 5)

    var i = 0
    check { (a: String, b: String, c: String, d: String, e: String, f: String) =>
      val res = i != 5
      i += 1
      res
    }
  }

  def `ScalaCheck property that takes 6 args, which fails, with default minSuccessful param set to 5` {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSuccessful = 5)

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      var i = 0
      check { (a: String, b: String, c: String, d: String, e: String, f: String) =>
        val res = i != 4
        i += 1
        res
      }
    }
  }

  def `ScalaCheck property that takes 6 args and generators, which succeeds, with default minSuccessful param set to 5` {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSuccessful = 5)

    var i = 0
    val prop = forAll (famousLastWords, famousLastWords, famousLastWords, famousLastWords, famousLastWords, famousLastWords) { (a: String, b: String, c: String, d: String, e: String, f: String) =>
      val res = i != 5
      i += 1
      res
    }
    check(prop)
  }

  def `ScalaCheck property that takes 6 args and generators, which fails, with default minSuccessful param set to 5` {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSuccessful = 5)

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      var i = 0
      val prop = forAll (famousLastWords, famousLastWords, famousLastWords, famousLastWords, famousLastWords, famousLastWords) { (a: String, b: String, c: String, d: String, e: String, f: String) =>
        val res = i != 4
        i += 1
        res
      }
      check(prop)
    }
  }

  // Same thing, but set maxDiscarded to 5 with param, prop fails after 5
  def `ScalaCheck property that takes 6 args, which succeeds, with maxDiscarded param set to 5` {

    var i = 0
    check(
      (a: String, b: String, c: String, d: String, e: String, f: String) => {
        i += 1
        (i > 5) ==> { 1 + 1 == (2) }
      },
      maxDiscarded(5)
    )
  }

  def `ScalaCheck property that takes 6 args, which fails, with maxDiscarded param set to 5` {

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      var i = 0
      check(
        (a: String, b: String, c: String, d: String, e: String, f: String) => {
          i += 1
          (i > 6) ==> { 1 + 1 == (2) }
        },
        maxDiscarded(5)
      ) 
    }
  }

  def `ScalaCheck property that takes 6 args and generators, which succeeds, with maxDiscarded param set to 5` {

    var i = 0
    val prop = forAll (famousLastWords, famousLastWords, famousLastWords, famousLastWords, famousLastWords, famousLastWords) { (a: String, b: String, c: String, d: String, e: String, f: String) =>
      i += 1
      (i > 5) ==> { 1 + 1 == (2) }
    }
    check(prop, maxDiscarded(5))
  }

  def `ScalaCheck property that takes 6 args and generators, which fails, with maxDiscarded param set to 5` {

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      var i = 0
      val prop = forAll (famousLastWords, famousLastWords, famousLastWords, famousLastWords, famousLastWords, famousLastWords) { (a: String, b: String, c: String, d: String, e: String, f: String) =>
        i += 1
        (i > 6) ==> { 1 + 1 == (2) }
      }
      check(prop, maxDiscarded(5))
    }
  }

  // Same thing, but set default maxDiscarded to 5, prop fails after 5
  def `ScalaCheck property that takes 6 args, which succeeds, with default maxDiscarded set to 5` {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(maxDiscarded = 5)

    var i = 0
    check { (a: String, b: String, c: String, d: String, e: String, f: String) =>
      i += 1
      (i > 5) ==> { 1 + 1 == (2) }
    }
  }

  def `ScalaCheck property that takes 6 args, which fails, with default maxDiscarded set to 5` {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(maxDiscarded = 5)

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      var i = 0
      check { (a: String, b: String, c: String, d: String, e: String, f: String) =>
        i += 1
        (i > 6) ==> { 1 + 1 == (2) }
      }
    }
  }

  def `ScalaCheck property that takes 6 args and generators, which succeeds, with default maxDiscarded set to 5` {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(maxDiscarded = 5)

    var i = 0
    val prop = forAll (famousLastWords, famousLastWords, famousLastWords, famousLastWords, famousLastWords, famousLastWords) { (a: String, b: String, c: String, d: String, e: String, f: String) =>
      i += 1
      (i > 5) ==> { 1 + 1 == (2) }
    }
    check(prop)
  }

  def `ScalaCheck property that takes 6 args and generators, which fails, with default maxDiscarded set to 5` {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(maxDiscarded = 5)

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      var i = 0
      val prop = forAll (famousLastWords, famousLastWords, famousLastWords, famousLastWords, famousLastWords, famousLastWords) { (a: String, b: String, c: String, d: String, e: String, f: String) =>
        i += 1
        (i > 6) ==> { 1 + 1 == (2) }
      }
      check(prop)
    }
  }

  // set minSize > maxSize with (param, param) (intercept IAE)
  def `ScalaCheck property that takes 6 args, which should throw IAE because maxSize > maxSize, specified as (param, param)` {

    intercept[IllegalArgumentException] {
      check(
        (a: String, b: String, c: String, d: String, e: String, f: String) => {
          1 + 1 == (2)
        },
        minSize(5),
        maxSize(4)
      ) 
    }
  }

  def `ScalaCheck property that takes 6 args and generators, which should throw IAE because maxSize > maxSize, specified as (param, param)` {

    intercept[IllegalArgumentException] {
      val prop = forAll (famousLastWords, famousLastWords, famousLastWords, famousLastWords, famousLastWords, famousLastWords) { (a: String, b: String, c: String, d: String, e: String, f: String) =>
        1 + 1 == (2)
      }
      check(prop, minSize(5), maxSize(4))
    }
  }

  // set minSize > maxSize with (param, default) (intercept IAE)
  def `ScalaCheck property that takes 6 args, which should throw IAE because maxSize > maxSize, specified as (param, default)` {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(maxSize = 4)

    intercept[IllegalArgumentException] {
      check(
        (a: String, b: String, c: String, d: String, e: String, f: String) => {
          1 + 1 == (2)
        },
        minSize(5)
      )
    }
  }

  def `ScalaCheck property that takes 6 args and generators, which should throw IAE because maxSize > maxSize, specified as (param, default)` {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(maxSize = 4)

    intercept[IllegalArgumentException] {
      val prop = forAll (famousLastWords, famousLastWords, famousLastWords, famousLastWords, famousLastWords, famousLastWords) { (a: String, b: String, c: String, d: String, e: String, f: String) =>
        1 + 1 == (2)
      }
      check(prop, minSize(5))
    }
  }

  // set minSize > maxSize with (default, param) (intercept IAE)
  def `ScalaCheck property that takes 6 args, which should throw IAE because maxSize > maxSize, specified as (default, param)` {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSize = 5)

    intercept[IllegalArgumentException] {
      check(
        (a: String, b: String, c: String, d: String, e: String, f: String) => {
          1 + 1 == (2)
        },
        maxSize(4)
      )
    }
  }

  def `ScalaCheck property that takes 6 args and generators, which should throw IAE because maxSize > maxSize, specified as (default, param)` {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSize = 5)

    intercept[IllegalArgumentException] {
      val prop = forAll (famousLastWords, famousLastWords, famousLastWords, famousLastWords, famousLastWords, famousLastWords) { (a: String, b: String, c: String, d: String, e: String, f: String) =>
        1 + 1 == (2)
      }
      check(prop, maxSize(4))
    }
  }

  // set maxSize with param (ensure always passed with a size less than maxSize)
  def `ScalaCheck property that takes 6 args, with maxSize specified as param` {

    check(
      (a: String, b: String, c: String, d: String, e: String, f: String) => {
      a.length <= 5
      b.length <= 5
      c.length <= 5
      d.length <= 5
      e.length <= 5
      f.length <= 5
      },
      maxSize(5)
    ) 
  }

  // set maxSize with default (ensure always passed with a size less than maxSize)
  def `ScalaCheck property that takes 6 args, with maxSize specified as default` {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(maxSize = 5)

    check { (a: String, b: String, c: String, d: String, e: String, f: String) =>
      a.length <= 5
      b.length <= 5
      c.length <= 5
      d.length <= 5
      e.length <= 5
      f.length <= 5
    }
  }

  // set minSize == maxSize with (param, param) (ensure always passed with that size)
  def `ScalaCheck property that takes 6 args and generators, with minSize == maxSize, specified as (param, param)` {

    val prop = forAll (fiveFive, fiveFive, fiveFive, fiveFive, fiveFive, fiveFive) { (a: String, b: String, c: String, d: String, e: String, f: String) =>
        a == ("OKAY")
        b == ("OKAY")
        c == ("OKAY")
        d == ("OKAY")
        e == ("OKAY")
        f == ("OKAY")
    }
    check(prop, minSize(5), maxSize(5))
  }

  // set minSize == maxSize with (param, default) (ensure always passed with that size)
  def `ScalaCheck property that takes 6 args and generators, with minSize == maxSize, specified as (param, default)` {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(maxSize = 5)

    val prop = forAll (fiveFive, fiveFive, fiveFive, fiveFive, fiveFive, fiveFive) { (a: String, b: String, c: String, d: String, e: String, f: String) =>
        a == ("OKAY")
        b == ("OKAY")
        c == ("OKAY")
        d == ("OKAY")
        e == ("OKAY")
        f == ("OKAY")
    }
    check(prop, minSize(5))
  }

  // set minSize == maxSize with (default, param) (ensure always passed with that size)
  def `ScalaCheck property that takes 6 args and generators, with minSize == maxSize, specified as (default, param)` {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSize = 5)

    val prop = forAll (fiveFive, fiveFive, fiveFive, fiveFive, fiveFive, fiveFive) { (a: String, b: String, c: String, d: String, e: String, f: String) =>
        a == ("OKAY")
        b == ("OKAY")
        c == ("OKAY")
        d == ("OKAY")
        e == ("OKAY")
        f == ("OKAY")
    }
    check(prop, maxSize(5))
  }

  // set minSize == maxSize with (default, default) (ensure always passed with that size)
  def `ScalaCheck property that takes 6 args and generators, with minSize == maxSize, specified as (default, default)` {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSize = 5, maxSize = 5)

    val prop = forAll (fiveFive, fiveFive, fiveFive, fiveFive, fiveFive, fiveFive) { (a: String, b: String, c: String, d: String, e: String, f: String) =>
        a == ("OKAY")
        b == ("OKAY")
        c == ("OKAY")
        d == ("OKAY")
        e == ("OKAY")
        f == ("OKAY")
    }
    check(prop)
  }

  // set minSize to 7 and maxSize to 11 with (param, param) (ensure always passed with that size)
  def `ScalaCheck property that takes 6 args and generators, with minSize to 7 and maxSize to 11, specified as (param, param)` {

    val prop = forAll (sevenEleven, sevenEleven, sevenEleven, sevenEleven, sevenEleven, sevenEleven) { (a: String, b: String, c: String, d: String, e: String, f: String) =>
        a == ("OKAY")
        b == ("OKAY")
        c == ("OKAY")
        d == ("OKAY")
        e == ("OKAY")
        f == ("OKAY")
    }
    check(prop, minSize(7), maxSize(11))
  }

  // set minSize to 7 and maxSize to 11 with (param, default) (ensure always passed with that size)
  def `ScalaCheck property that takes 6 args and generators, with minSize to 7 and maxSize to 11, specified as (param, default)` {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(maxSize = 11)

    val prop = forAll (sevenEleven, sevenEleven, sevenEleven, sevenEleven, sevenEleven, sevenEleven) { (a: String, b: String, c: String, d: String, e: String, f: String) =>
        a == ("OKAY")
        b == ("OKAY")
        c == ("OKAY")
        d == ("OKAY")
        e == ("OKAY")
        f == ("OKAY")
    }
    check(prop, minSize(7))
  }

  // set minSize to 7 and maxSize to 11 with (default, param) (ensure always passed with that size)
  def `ScalaCheck property that takes 6 args and generators, with minSize to 7 and maxSize to 11, specified as (default, param)` {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSize = 7)

    val prop = forAll (sevenEleven, sevenEleven, sevenEleven, sevenEleven, sevenEleven, sevenEleven) { (a: String, b: String, c: String, d: String, e: String, f: String) =>
        a == ("OKAY")
        b == ("OKAY")
        c == ("OKAY")
        d == ("OKAY")
        e == ("OKAY")
        f == ("OKAY")
    }
    check(prop, maxSize(11))
  }

  // set minSize to 7 and maxSize to 11 with (default, default) (ensure always passed with that size)
  def `ScalaCheck property that takes 6 args and generators, with minSize to 7 and maxSize to 11, specified as (default, default)` {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSize = 7, maxSize = 11)

    val prop = forAll (sevenEleven, sevenEleven, sevenEleven, sevenEleven, sevenEleven, sevenEleven) { (a: String, b: String, c: String, d: String, e: String, f: String) =>
        a == ("OKAY")
        b == ("OKAY")
        c == ("OKAY")
        d == ("OKAY")
        e == ("OKAY")
        f == ("OKAY")
    }
    check(prop)
  }
}
