/*
 * Copyright 2001-2013 Artima, Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.scalatest
package prop


import matchers.ShouldMatchers

class TableSuite extends Spec with TableDrivenPropertyChecks {

  def `table for 1 that succeeds` {

    val examples =
      Table(
        ("a"),
        (  1)
      )

    forAll (examples) { (a) => assert(a === (1)) }
  }

  def `table for 1, which succeeds even though DiscardedEvaluationException is thrown` {
    val numbers =
      Table(
        ("a"),
        ( -1),
        (  1)
      )

    forAll (numbers) { (a) =>

      whenever (a > 0) {
        assert(a > 0)
      }
    }
  }

  def `table for 1, which fails` {

    val examples =
      Table(
        ("a"),
        (  2)
      )

    intercept[TableDrivenPropertyCheckFailedException] {
      forAll (examples) { (a) => assert(a === (1)) }
    }
  }

  def `table for 1 apply, length, and iterator methods work correctly` {

    val examples =
      Table(
        ("a"),
        (  0),
        (  1),
        (  2),
        (  3),
        (  4),
        (  5),
        (  6),
        (  7),
        (  8),
        (  9)
      )

    for (i <- 0 to 9) {
      assert(examples(i) === (i))
    }

    assert(examples.length === (10))

    var i = 0
    for (example <- examples.iterator) {
      assert(example === (i))
      i += 1
    }

    assert(examples.iterator.length === (10))
  }

  def `table for 2 that succeeds` {

    val examples =
      Table(
        ("a", "b"),
        (  1,   1),
        (  1,   1)
      )

    forAll (examples) { (a, b) => assert(a + b === (2)) }
  }

  def `table for 2, which succeeds even though DiscardedEvaluationException is thrown` {
    val numbers =
      Table(
        ("a", "b"),
        ( -1,  -1),
        (  1,   1),
        (  1,   1)
      )

    forAll (numbers) { (a, b) =>

      whenever (a > 0) {
        assert(a > 0)
      }
    }
  }

  def `table for 2, which fails` {

    val examples =
      Table(
        ("a", "b"),
        (  2,   2),
        (  2,   2)
      )

    intercept[TableDrivenPropertyCheckFailedException] {
      forAll (examples) { (a, b) => assert(a + b === (2)) }
    }
  }

  def `table for 2 apply, length, and iterator methods work correctly` {

    val examples =
      Table(
        ("a", "b"),
        (  0,   0),
        (  1,   1),
        (  2,   2),
        (  3,   3),
        (  4,   4),
        (  5,   5),
        (  6,   6),
        (  7,   7),
        (  8,   8),
        (  9,   9)
      )

    for (i <- 0 to 9) {
      assert(examples(i) === (i, i))
    }

    assert(examples.length === (10))

    var i = 0
    for (example <- examples.iterator) {
      assert(example === (i, i))
      i += 1
    }

    assert(examples.iterator.length === (10))
  }

  def `table for 3 that succeeds` {

    val examples =
      Table(
        ("a", "b", "c"),
        (  1,   1,   1),
        (  1,   1,   1),
        (  1,   1,   1)
      )

    forAll (examples) { (a, b, c) => assert(a + b + c === (3)) }
  }

  def `table for 3, which succeeds even though DiscardedEvaluationException is thrown` {
    val numbers =
      Table(
        ("a", "b", "c"),
        ( -1,  -1,  -1),
        (  1,   1,   1),
        (  1,   1,   1),
        (  1,   1,   1)
      )

    forAll (numbers) { (a, b, c) =>

      whenever (a > 0) {
        assert(a > 0)
      }
    }
  }

  def `table for 3, which fails` {

    val examples =
      Table(
        ("a", "b", "c"),
        (  2,   2,   2),
        (  2,   2,   2),
        (  2,   2,   2)
      )

    intercept[TableDrivenPropertyCheckFailedException] {
      forAll (examples) { (a, b, c) => assert(a + b + c === (3)) }
    }
  }

  def `table for 3 apply, length, and iterator methods work correctly` {

    val examples =
      Table(
        ("a", "b", "c"),
        (  0,   0,   0),
        (  1,   1,   1),
        (  2,   2,   2),
        (  3,   3,   3),
        (  4,   4,   4),
        (  5,   5,   5),
        (  6,   6,   6),
        (  7,   7,   7),
        (  8,   8,   8),
        (  9,   9,   9)
      )

    for (i <- 0 to 9) {
      assert(examples(i) === (i, i, i))
    }

    assert(examples.length === (10))

    var i = 0
    for (example <- examples.iterator) {
      assert(example === (i, i, i))
      i += 1
    }

    assert(examples.iterator.length === (10))
  }

  def `table for 4 that succeeds` {

    val examples =
      Table(
        ("a", "b", "c", "d"),
        (  1,   1,   1,   1),
        (  1,   1,   1,   1),
        (  1,   1,   1,   1),
        (  1,   1,   1,   1)
      )

    forAll (examples) { (a, b, c, d) => assert(a + b + c + d === (4)) }
  }

  def `table for 4, which succeeds even though DiscardedEvaluationException is thrown` {
    val numbers =
      Table(
        ("a", "b", "c", "d"),
        ( -1,  -1,  -1,  -1),
        (  1,   1,   1,   1),
        (  1,   1,   1,   1),
        (  1,   1,   1,   1),
        (  1,   1,   1,   1)
      )

    forAll (numbers) { (a, b, c, d) =>

      whenever (a > 0) {
        assert(a > 0)
      }
    }
  }

  def `table for 4, which fails` {

    val examples =
      Table(
        ("a", "b", "c", "d"),
        (  2,   2,   2,   2),
        (  2,   2,   2,   2),
        (  2,   2,   2,   2),
        (  2,   2,   2,   2)
      )

    intercept[TableDrivenPropertyCheckFailedException] {
      forAll (examples) { (a, b, c, d) => assert(a + b + c + d === (4)) }
    }
  }

  def `table for 4 apply, length, and iterator methods work correctly` {

    val examples =
      Table(
        ("a", "b", "c", "d"),
        (  0,   0,   0,   0),
        (  1,   1,   1,   1),
        (  2,   2,   2,   2),
        (  3,   3,   3,   3),
        (  4,   4,   4,   4),
        (  5,   5,   5,   5),
        (  6,   6,   6,   6),
        (  7,   7,   7,   7),
        (  8,   8,   8,   8),
        (  9,   9,   9,   9)
      )

    for (i <- 0 to 9) {
      assert(examples(i) === (i, i, i, i))
    }

    assert(examples.length === (10))

    var i = 0
    for (example <- examples.iterator) {
      assert(example === (i, i, i, i))
      i += 1
    }

    assert(examples.iterator.length === (10))
  }

  def `table for 5 that succeeds` {

    val examples =
      Table(
        ("a", "b", "c", "d", "e"),
        (  1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1)
      )

    forAll (examples) { (a, b, c, d, e) => assert(a + b + c + d + e === (5)) }
  }

  def `table for 5, which succeeds even though DiscardedEvaluationException is thrown` {
    val numbers =
      Table(
        ("a", "b", "c", "d", "e"),
        ( -1,  -1,  -1,  -1,  -1),
        (  1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1)
      )

    forAll (numbers) { (a, b, c, d, e) =>

      whenever (a > 0) {
        assert(a > 0)
      }
    }
  }

  def `table for 5, which fails` {

    val examples =
      Table(
        ("a", "b", "c", "d", "e"),
        (  2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2)
      )

    intercept[TableDrivenPropertyCheckFailedException] {
      forAll (examples) { (a, b, c, d, e) => assert(a + b + c + d + e === (5)) }
    }
  }

  def `table for 5 apply, length, and iterator methods work correctly` {

    val examples =
      Table(
        ("a", "b", "c", "d", "e"),
        (  0,   0,   0,   0,   0),
        (  1,   1,   1,   1,   1),
        (  2,   2,   2,   2,   2),
        (  3,   3,   3,   3,   3),
        (  4,   4,   4,   4,   4),
        (  5,   5,   5,   5,   5),
        (  6,   6,   6,   6,   6),
        (  7,   7,   7,   7,   7),
        (  8,   8,   8,   8,   8),
        (  9,   9,   9,   9,   9)
      )

    for (i <- 0 to 9) {
      assert(examples(i) === (i, i, i, i, i))
    }

    assert(examples.length === (10))

    var i = 0
    for (example <- examples.iterator) {
      assert(example === (i, i, i, i, i))
      i += 1
    }

    assert(examples.iterator.length === (10))
  }

  def `table for 6 that succeeds` {

    val examples =
      Table(
        ("a", "b", "c", "d", "e", "f"),
        (  1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1)
      )

    forAll (examples) { (a, b, c, d, e, f) => assert(a + b + c + d + e + f === (6)) }
  }

  def `table for 6, which succeeds even though DiscardedEvaluationException is thrown` {
    val numbers =
      Table(
        ("a", "b", "c", "d", "e", "f"),
        ( -1,  -1,  -1,  -1,  -1,  -1),
        (  1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1)
      )

    forAll (numbers) { (a, b, c, d, e, f) =>

      whenever (a > 0) {
        assert(a > 0)
      }
    }
  }

  def `table for 6, which fails` {

    val examples =
      Table(
        ("a", "b", "c", "d", "e", "f"),
        (  2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2)
      )

    intercept[TableDrivenPropertyCheckFailedException] {
      forAll (examples) { (a, b, c, d, e, f) => assert(a + b + c + d + e + f === (6)) }
    }
  }

  def `table for 6 apply, length, and iterator methods work correctly` {

    val examples =
      Table(
        ("a", "b", "c", "d", "e", "f"),
        (  0,   0,   0,   0,   0,   0),
        (  1,   1,   1,   1,   1,   1),
        (  2,   2,   2,   2,   2,   2),
        (  3,   3,   3,   3,   3,   3),
        (  4,   4,   4,   4,   4,   4),
        (  5,   5,   5,   5,   5,   5),
        (  6,   6,   6,   6,   6,   6),
        (  7,   7,   7,   7,   7,   7),
        (  8,   8,   8,   8,   8,   8),
        (  9,   9,   9,   9,   9,   9)
      )

    for (i <- 0 to 9) {
      assert(examples(i) === (i, i, i, i, i, i))
    }

    assert(examples.length === (10))

    var i = 0
    for (example <- examples.iterator) {
      assert(example === (i, i, i, i, i, i))
      i += 1
    }

    assert(examples.iterator.length === (10))
  }

  def `table for 7 that succeeds` {

    val examples =
      Table(
        ("a", "b", "c", "d", "e", "f", "g"),
        (  1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1)
      )

    forAll (examples) { (a, b, c, d, e, f, g) => assert(a + b + c + d + e + f + g === (7)) }
  }

  def `table for 7, which succeeds even though DiscardedEvaluationException is thrown` {
    val numbers =
      Table(
        ("a", "b", "c", "d", "e", "f", "g"),
        ( -1,  -1,  -1,  -1,  -1,  -1,  -1),
        (  1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1)
      )

    forAll (numbers) { (a, b, c, d, e, f, g) =>

      whenever (a > 0) {
        assert(a > 0)
      }
    }
  }

  def `table for 7, which fails` {

    val examples =
      Table(
        ("a", "b", "c", "d", "e", "f", "g"),
        (  2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2)
      )

    intercept[TableDrivenPropertyCheckFailedException] {
      forAll (examples) { (a, b, c, d, e, f, g) => assert(a + b + c + d + e + f + g === (7)) }
    }
  }

  def `table for 7 apply, length, and iterator methods work correctly` {

    val examples =
      Table(
        ("a", "b", "c", "d", "e", "f", "g"),
        (  0,   0,   0,   0,   0,   0,   0),
        (  1,   1,   1,   1,   1,   1,   1),
        (  2,   2,   2,   2,   2,   2,   2),
        (  3,   3,   3,   3,   3,   3,   3),
        (  4,   4,   4,   4,   4,   4,   4),
        (  5,   5,   5,   5,   5,   5,   5),
        (  6,   6,   6,   6,   6,   6,   6),
        (  7,   7,   7,   7,   7,   7,   7),
        (  8,   8,   8,   8,   8,   8,   8),
        (  9,   9,   9,   9,   9,   9,   9)
      )

    for (i <- 0 to 9) {
      assert(examples(i) === (i, i, i, i, i, i, i))
    }

    assert(examples.length === (10))

    var i = 0
    for (example <- examples.iterator) {
      assert(example === (i, i, i, i, i, i, i))
      i += 1
    }

    assert(examples.iterator.length === (10))
  }

  def `table for 8 that succeeds` {

    val examples =
      Table(
        ("a", "b", "c", "d", "e", "f", "g", "h"),
        (  1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1)
      )

    forAll (examples) { (a, b, c, d, e, f, g, h) => assert(a + b + c + d + e + f + g + h === (8)) }
  }

  def `table for 8, which succeeds even though DiscardedEvaluationException is thrown` {
    val numbers =
      Table(
        ("a", "b", "c", "d", "e", "f", "g", "h"),
        ( -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1),
        (  1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1)
      )

    forAll (numbers) { (a, b, c, d, e, f, g, h) =>

      whenever (a > 0) {
        assert(a > 0)
      }
    }
  }

  def `table for 8, which fails` {

    val examples =
      Table(
        ("a", "b", "c", "d", "e", "f", "g", "h"),
        (  2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2)
      )

    intercept[TableDrivenPropertyCheckFailedException] {
      forAll (examples) { (a, b, c, d, e, f, g, h) => assert(a + b + c + d + e + f + g + h === (8)) }
    }
  }

  def `table for 8 apply, length, and iterator methods work correctly` {

    val examples =
      Table(
        ("a", "b", "c", "d", "e", "f", "g", "h"),
        (  0,   0,   0,   0,   0,   0,   0,   0),
        (  1,   1,   1,   1,   1,   1,   1,   1),
        (  2,   2,   2,   2,   2,   2,   2,   2),
        (  3,   3,   3,   3,   3,   3,   3,   3),
        (  4,   4,   4,   4,   4,   4,   4,   4),
        (  5,   5,   5,   5,   5,   5,   5,   5),
        (  6,   6,   6,   6,   6,   6,   6,   6),
        (  7,   7,   7,   7,   7,   7,   7,   7),
        (  8,   8,   8,   8,   8,   8,   8,   8),
        (  9,   9,   9,   9,   9,   9,   9,   9)
      )

    for (i <- 0 to 9) {
      assert(examples(i) === (i, i, i, i, i, i, i, i))
    }

    assert(examples.length === (10))

    var i = 0
    for (example <- examples.iterator) {
      assert(example === (i, i, i, i, i, i, i, i))
      i += 1
    }

    assert(examples.iterator.length === (10))
  }

  def `table for 9 that succeeds` {

    val examples =
      Table(
        ("a", "b", "c", "d", "e", "f", "g", "h", "i"),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1)
      )

    forAll (examples) { (a, b, c, d, e, f, g, h, i) => assert(a + b + c + d + e + f + g + h + i === (9)) }
  }

  def `table for 9, which succeeds even though DiscardedEvaluationException is thrown` {
    val numbers =
      Table(
        ("a", "b", "c", "d", "e", "f", "g", "h", "i"),
        ( -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1)
      )

    forAll (numbers) { (a, b, c, d, e, f, g, h, i) =>

      whenever (a > 0) {
        assert(a > 0)
      }
    }
  }

  def `table for 9, which fails` {

    val examples =
      Table(
        ("a", "b", "c", "d", "e", "f", "g", "h", "i"),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2)
      )

    intercept[TableDrivenPropertyCheckFailedException] {
      forAll (examples) { (a, b, c, d, e, f, g, h, i) => assert(a + b + c + d + e + f + g + h + i === (9)) }
    }
  }

  def `table for 9 apply, length, and iterator methods work correctly` {

    val examples =
      Table(
        ("a", "b", "c", "d", "e", "f", "g", "h", "i"),
        (  0,   0,   0,   0,   0,   0,   0,   0,   0),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  3,   3,   3,   3,   3,   3,   3,   3,   3),
        (  4,   4,   4,   4,   4,   4,   4,   4,   4),
        (  5,   5,   5,   5,   5,   5,   5,   5,   5),
        (  6,   6,   6,   6,   6,   6,   6,   6,   6),
        (  7,   7,   7,   7,   7,   7,   7,   7,   7),
        (  8,   8,   8,   8,   8,   8,   8,   8,   8),
        (  9,   9,   9,   9,   9,   9,   9,   9,   9)
      )

    for (i <- 0 to 9) {
      assert(examples(i) === (i, i, i, i, i, i, i, i, i))
    }

    assert(examples.length === (10))

    var i = 0
    for (example <- examples.iterator) {
      assert(example === (i, i, i, i, i, i, i, i, i))
      i += 1
    }

    assert(examples.iterator.length === (10))
  }

  def `table for 10 that succeeds` {

    val examples =
      Table(
        ("a", "b", "c", "d", "e", "f", "g", "h", "i", "j"),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1)
      )

    forAll (examples) { (a, b, c, d, e, f, g, h, i, j) => assert(a + b + c + d + e + f + g + h + i + j === (10)) }
  }

  def `table for 10, which succeeds even though DiscardedEvaluationException is thrown` {
    val numbers =
      Table(
        ("a", "b", "c", "d", "e", "f", "g", "h", "i", "j"),
        ( -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1)
      )

    forAll (numbers) { (a, b, c, d, e, f, g, h, i, j) =>

      whenever (a > 0) {
        assert(a > 0)
      }
    }
  }

  def `table for 10, which fails` {

    val examples =
      Table(
        ("a", "b", "c", "d", "e", "f", "g", "h", "i", "j"),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2)
      )

    intercept[TableDrivenPropertyCheckFailedException] {
      forAll (examples) { (a, b, c, d, e, f, g, h, i, j) => assert(a + b + c + d + e + f + g + h + i + j === (10)) }
    }
  }

  def `table for 10 apply, length, and iterator methods work correctly` {

    val examples =
      Table(
        ("a", "b", "c", "d", "e", "f", "g", "h", "i", "j"),
        (  0,   0,   0,   0,   0,   0,   0,   0,   0,   0),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  3,   3,   3,   3,   3,   3,   3,   3,   3,   3),
        (  4,   4,   4,   4,   4,   4,   4,   4,   4,   4),
        (  5,   5,   5,   5,   5,   5,   5,   5,   5,   5),
        (  6,   6,   6,   6,   6,   6,   6,   6,   6,   6),
        (  7,   7,   7,   7,   7,   7,   7,   7,   7,   7),
        (  8,   8,   8,   8,   8,   8,   8,   8,   8,   8),
        (  9,   9,   9,   9,   9,   9,   9,   9,   9,   9)
      )

    for (i <- 0 to 9) {
      assert(examples(i) === (i, i, i, i, i, i, i, i, i, i))
    }

    assert(examples.length === (10))

    var i = 0
    for (example <- examples.iterator) {
      assert(example === (i, i, i, i, i, i, i, i, i, i))
      i += 1
    }

    assert(examples.iterator.length === (10))
  }

  def `table for 11 that succeeds` {

    val examples =
      Table(
        ("a", "b", "c", "d", "e", "f", "g", "h", "i", "j", "k"),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1)
      )

    forAll (examples) { (a, b, c, d, e, f, g, h, i, j, k) => assert(a + b + c + d + e + f + g + h + i + j + k === (11)) }
  }

  def `table for 11, which succeeds even though DiscardedEvaluationException is thrown` {
    val numbers =
      Table(
        ("a", "b", "c", "d", "e", "f", "g", "h", "i", "j", "k"),
        ( -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1)
      )

    forAll (numbers) { (a, b, c, d, e, f, g, h, i, j, k) =>

      whenever (a > 0) {
        assert(a > 0)
      }
    }
  }

  def `table for 11, which fails` {

    val examples =
      Table(
        ("a", "b", "c", "d", "e", "f", "g", "h", "i", "j", "k"),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2)
      )

    intercept[TableDrivenPropertyCheckFailedException] {
      forAll (examples) { (a, b, c, d, e, f, g, h, i, j, k) => assert(a + b + c + d + e + f + g + h + i + j + k === (11)) }
    }
  }

  def `table for 11 apply, length, and iterator methods work correctly` {

    val examples =
      Table(
        ("a", "b", "c", "d", "e", "f", "g", "h", "i", "j", "k"),
        (  0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  3,   3,   3,   3,   3,   3,   3,   3,   3,   3,   3),
        (  4,   4,   4,   4,   4,   4,   4,   4,   4,   4,   4),
        (  5,   5,   5,   5,   5,   5,   5,   5,   5,   5,   5),
        (  6,   6,   6,   6,   6,   6,   6,   6,   6,   6,   6),
        (  7,   7,   7,   7,   7,   7,   7,   7,   7,   7,   7),
        (  8,   8,   8,   8,   8,   8,   8,   8,   8,   8,   8),
        (  9,   9,   9,   9,   9,   9,   9,   9,   9,   9,   9)
      )

    for (i <- 0 to 9) {
      assert(examples(i) === (i, i, i, i, i, i, i, i, i, i, i))
    }

    assert(examples.length === (10))

    var i = 0
    for (example <- examples.iterator) {
      assert(example === (i, i, i, i, i, i, i, i, i, i, i))
      i += 1
    }

    assert(examples.iterator.length === (10))
  }

  def `table for 12 that succeeds` {

    val examples =
      Table(
        ("a", "b", "c", "d", "e", "f", "g", "h", "i", "j", "k", "l"),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1)
      )

    forAll (examples) { (a, b, c, d, e, f, g, h, i, j, k, l) => assert(a + b + c + d + e + f + g + h + i + j + k + l === (12)) }
  }

  def `table for 12, which succeeds even though DiscardedEvaluationException is thrown` {
    val numbers =
      Table(
        ("a", "b", "c", "d", "e", "f", "g", "h", "i", "j", "k", "l"),
        ( -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1)
      )

    forAll (numbers) { (a, b, c, d, e, f, g, h, i, j, k, l) =>

      whenever (a > 0) {
        assert(a > 0)
      }
    }
  }

  def `table for 12, which fails` {

    val examples =
      Table(
        ("a", "b", "c", "d", "e", "f", "g", "h", "i", "j", "k", "l"),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2)
      )

    intercept[TableDrivenPropertyCheckFailedException] {
      forAll (examples) { (a, b, c, d, e, f, g, h, i, j, k, l) => assert(a + b + c + d + e + f + g + h + i + j + k + l === (12)) }
    }
  }

  def `table for 12 apply, length, and iterator methods work correctly` {

    val examples =
      Table(
        ("a", "b", "c", "d", "e", "f", "g", "h", "i", "j", "k", "l"),
        (  0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  3,   3,   3,   3,   3,   3,   3,   3,   3,   3,   3,   3),
        (  4,   4,   4,   4,   4,   4,   4,   4,   4,   4,   4,   4),
        (  5,   5,   5,   5,   5,   5,   5,   5,   5,   5,   5,   5),
        (  6,   6,   6,   6,   6,   6,   6,   6,   6,   6,   6,   6),
        (  7,   7,   7,   7,   7,   7,   7,   7,   7,   7,   7,   7),
        (  8,   8,   8,   8,   8,   8,   8,   8,   8,   8,   8,   8),
        (  9,   9,   9,   9,   9,   9,   9,   9,   9,   9,   9,   9)
      )

    for (i <- 0 to 9) {
      assert(examples(i) === (i, i, i, i, i, i, i, i, i, i, i, i))
    }

    assert(examples.length === (10))

    var i = 0
    for (example <- examples.iterator) {
      assert(example === (i, i, i, i, i, i, i, i, i, i, i, i))
      i += 1
    }

    assert(examples.iterator.length === (10))
  }

  def `table for 13 that succeeds` {

    val examples =
      Table(
        ("a", "b", "c", "d", "e", "f", "g", "h", "i", "j", "k", "l", "m"),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1)
      )

    forAll (examples) { (a, b, c, d, e, f, g, h, i, j, k, l, m) => assert(a + b + c + d + e + f + g + h + i + j + k + l + m === (13)) }
  }

  def `table for 13, which succeeds even though DiscardedEvaluationException is thrown` {
    val numbers =
      Table(
        ("a", "b", "c", "d", "e", "f", "g", "h", "i", "j", "k", "l", "m"),
        ( -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1)
      )

    forAll (numbers) { (a, b, c, d, e, f, g, h, i, j, k, l, m) =>

      whenever (a > 0) {
        assert(a > 0)
      }
    }
  }

  def `table for 13, which fails` {

    val examples =
      Table(
        ("a", "b", "c", "d", "e", "f", "g", "h", "i", "j", "k", "l", "m"),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2)
      )

    intercept[TableDrivenPropertyCheckFailedException] {
      forAll (examples) { (a, b, c, d, e, f, g, h, i, j, k, l, m) => assert(a + b + c + d + e + f + g + h + i + j + k + l + m === (13)) }
    }
  }

  def `table for 13 apply, length, and iterator methods work correctly` {

    val examples =
      Table(
        ("a", "b", "c", "d", "e", "f", "g", "h", "i", "j", "k", "l", "m"),
        (  0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  3,   3,   3,   3,   3,   3,   3,   3,   3,   3,   3,   3,   3),
        (  4,   4,   4,   4,   4,   4,   4,   4,   4,   4,   4,   4,   4),
        (  5,   5,   5,   5,   5,   5,   5,   5,   5,   5,   5,   5,   5),
        (  6,   6,   6,   6,   6,   6,   6,   6,   6,   6,   6,   6,   6),
        (  7,   7,   7,   7,   7,   7,   7,   7,   7,   7,   7,   7,   7),
        (  8,   8,   8,   8,   8,   8,   8,   8,   8,   8,   8,   8,   8),
        (  9,   9,   9,   9,   9,   9,   9,   9,   9,   9,   9,   9,   9)
      )

    for (i <- 0 to 9) {
      assert(examples(i) === (i, i, i, i, i, i, i, i, i, i, i, i, i))
    }

    assert(examples.length === (10))

    var i = 0
    for (example <- examples.iterator) {
      assert(example === (i, i, i, i, i, i, i, i, i, i, i, i, i))
      i += 1
    }

    assert(examples.iterator.length === (10))
  }

  def `table for 14 that succeeds` {

    val examples =
      Table(
        ("a", "b", "c", "d", "e", "f", "g", "h", "i", "j", "k", "l", "m", "n"),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1)
      )

    forAll (examples) { (a, b, c, d, e, f, g, h, i, j, k, l, m, n) => assert(a + b + c + d + e + f + g + h + i + j + k + l + m + n === (14)) }
  }

  def `table for 14, which succeeds even though DiscardedEvaluationException is thrown` {
    val numbers =
      Table(
        ("a", "b", "c", "d", "e", "f", "g", "h", "i", "j", "k", "l", "m", "n"),
        ( -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1)
      )

    forAll (numbers) { (a, b, c, d, e, f, g, h, i, j, k, l, m, n) =>

      whenever (a > 0) {
        assert(a > 0)
      }
    }
  }

  def `table for 14, which fails` {

    val examples =
      Table(
        ("a", "b", "c", "d", "e", "f", "g", "h", "i", "j", "k", "l", "m", "n"),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2)
      )

    intercept[TableDrivenPropertyCheckFailedException] {
      forAll (examples) { (a, b, c, d, e, f, g, h, i, j, k, l, m, n) => assert(a + b + c + d + e + f + g + h + i + j + k + l + m + n === (14)) }
    }
  }

  def `table for 14 apply, length, and iterator methods work correctly` {

    val examples =
      Table(
        ("a", "b", "c", "d", "e", "f", "g", "h", "i", "j", "k", "l", "m", "n"),
        (  0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  3,   3,   3,   3,   3,   3,   3,   3,   3,   3,   3,   3,   3,   3),
        (  4,   4,   4,   4,   4,   4,   4,   4,   4,   4,   4,   4,   4,   4),
        (  5,   5,   5,   5,   5,   5,   5,   5,   5,   5,   5,   5,   5,   5),
        (  6,   6,   6,   6,   6,   6,   6,   6,   6,   6,   6,   6,   6,   6),
        (  7,   7,   7,   7,   7,   7,   7,   7,   7,   7,   7,   7,   7,   7),
        (  8,   8,   8,   8,   8,   8,   8,   8,   8,   8,   8,   8,   8,   8),
        (  9,   9,   9,   9,   9,   9,   9,   9,   9,   9,   9,   9,   9,   9)
      )

    for (i <- 0 to 9) {
      assert(examples(i) === (i, i, i, i, i, i, i, i, i, i, i, i, i, i))
    }

    assert(examples.length === (10))

    var i = 0
    for (example <- examples.iterator) {
      assert(example === (i, i, i, i, i, i, i, i, i, i, i, i, i, i))
      i += 1
    }

    assert(examples.iterator.length === (10))
  }

  def `table for 15 that succeeds` {

    val examples =
      Table(
        ("a", "b", "c", "d", "e", "f", "g", "h", "i", "j", "k", "l", "m", "n", "o"),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1)
      )

    forAll (examples) { (a, b, c, d, e, f, g, h, i, j, k, l, m, n, o) => assert(a + b + c + d + e + f + g + h + i + j + k + l + m + n + o === (15)) }
  }

  def `table for 15, which succeeds even though DiscardedEvaluationException is thrown` {
    val numbers =
      Table(
        ("a", "b", "c", "d", "e", "f", "g", "h", "i", "j", "k", "l", "m", "n", "o"),
        ( -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1)
      )

    forAll (numbers) { (a, b, c, d, e, f, g, h, i, j, k, l, m, n, o) =>

      whenever (a > 0) {
        assert(a > 0)
      }
    }
  }

  def `table for 15, which fails` {

    val examples =
      Table(
        ("a", "b", "c", "d", "e", "f", "g", "h", "i", "j", "k", "l", "m", "n", "o"),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2)
      )

    intercept[TableDrivenPropertyCheckFailedException] {
      forAll (examples) { (a, b, c, d, e, f, g, h, i, j, k, l, m, n, o) => assert(a + b + c + d + e + f + g + h + i + j + k + l + m + n + o === (15)) }
    }
  }

  def `table for 15 apply, length, and iterator methods work correctly` {

    val examples =
      Table(
        ("a", "b", "c", "d", "e", "f", "g", "h", "i", "j", "k", "l", "m", "n", "o"),
        (  0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  3,   3,   3,   3,   3,   3,   3,   3,   3,   3,   3,   3,   3,   3,   3),
        (  4,   4,   4,   4,   4,   4,   4,   4,   4,   4,   4,   4,   4,   4,   4),
        (  5,   5,   5,   5,   5,   5,   5,   5,   5,   5,   5,   5,   5,   5,   5),
        (  6,   6,   6,   6,   6,   6,   6,   6,   6,   6,   6,   6,   6,   6,   6),
        (  7,   7,   7,   7,   7,   7,   7,   7,   7,   7,   7,   7,   7,   7,   7),
        (  8,   8,   8,   8,   8,   8,   8,   8,   8,   8,   8,   8,   8,   8,   8),
        (  9,   9,   9,   9,   9,   9,   9,   9,   9,   9,   9,   9,   9,   9,   9)
      )

    for (i <- 0 to 9) {
      assert(examples(i) === (i, i, i, i, i, i, i, i, i, i, i, i, i, i, i))
    }

    assert(examples.length === (10))

    var i = 0
    for (example <- examples.iterator) {
      assert(example === (i, i, i, i, i, i, i, i, i, i, i, i, i, i, i))
      i += 1
    }

    assert(examples.iterator.length === (10))
  }

  def `table for 16 that succeeds` {

    val examples =
      Table(
        ("a", "b", "c", "d", "e", "f", "g", "h", "i", "j", "k", "l", "m", "n", "o", "p"),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1)
      )

    forAll (examples) { (a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p) => assert(a + b + c + d + e + f + g + h + i + j + k + l + m + n + o + p === (16)) }
  }

  def `table for 16, which succeeds even though DiscardedEvaluationException is thrown` {
    val numbers =
      Table(
        ("a", "b", "c", "d", "e", "f", "g", "h", "i", "j", "k", "l", "m", "n", "o", "p"),
        ( -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1)
      )

    forAll (numbers) { (a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p) =>

      whenever (a > 0) {
        assert(a > 0)
      }
    }
  }

  def `table for 16, which fails` {

    val examples =
      Table(
        ("a", "b", "c", "d", "e", "f", "g", "h", "i", "j", "k", "l", "m", "n", "o", "p"),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2)
      )

    intercept[TableDrivenPropertyCheckFailedException] {
      forAll (examples) { (a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p) => assert(a + b + c + d + e + f + g + h + i + j + k + l + m + n + o + p === (16)) }
    }
  }

  def `table for 16 apply, length, and iterator methods work correctly` {

    val examples =
      Table(
        ("a", "b", "c", "d", "e", "f", "g", "h", "i", "j", "k", "l", "m", "n", "o", "p"),
        (  0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  3,   3,   3,   3,   3,   3,   3,   3,   3,   3,   3,   3,   3,   3,   3,   3),
        (  4,   4,   4,   4,   4,   4,   4,   4,   4,   4,   4,   4,   4,   4,   4,   4),
        (  5,   5,   5,   5,   5,   5,   5,   5,   5,   5,   5,   5,   5,   5,   5,   5),
        (  6,   6,   6,   6,   6,   6,   6,   6,   6,   6,   6,   6,   6,   6,   6,   6),
        (  7,   7,   7,   7,   7,   7,   7,   7,   7,   7,   7,   7,   7,   7,   7,   7),
        (  8,   8,   8,   8,   8,   8,   8,   8,   8,   8,   8,   8,   8,   8,   8,   8),
        (  9,   9,   9,   9,   9,   9,   9,   9,   9,   9,   9,   9,   9,   9,   9,   9)
      )

    for (i <- 0 to 9) {
      assert(examples(i) === (i, i, i, i, i, i, i, i, i, i, i, i, i, i, i, i))
    }

    assert(examples.length === (10))

    var i = 0
    for (example <- examples.iterator) {
      assert(example === (i, i, i, i, i, i, i, i, i, i, i, i, i, i, i, i))
      i += 1
    }

    assert(examples.iterator.length === (10))
  }

  def `table for 17 that succeeds` {

    val examples =
      Table(
        ("a", "b", "c", "d", "e", "f", "g", "h", "i", "j", "k", "l", "m", "n", "o", "p", "q"),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1)
      )

    forAll (examples) { (a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q) => assert(a + b + c + d + e + f + g + h + i + j + k + l + m + n + o + p + q === (17)) }
  }

  def `table for 17, which succeeds even though DiscardedEvaluationException is thrown` {
    val numbers =
      Table(
        ("a", "b", "c", "d", "e", "f", "g", "h", "i", "j", "k", "l", "m", "n", "o", "p", "q"),
        ( -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1)
      )

    forAll (numbers) { (a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q) =>

      whenever (a > 0) {
        assert(a > 0)
      }
    }
  }

  def `table for 17, which fails` {

    val examples =
      Table(
        ("a", "b", "c", "d", "e", "f", "g", "h", "i", "j", "k", "l", "m", "n", "o", "p", "q"),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2)
      )

    intercept[TableDrivenPropertyCheckFailedException] {
      forAll (examples) { (a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q) => assert(a + b + c + d + e + f + g + h + i + j + k + l + m + n + o + p + q === (17)) }
    }
  }

  def `table for 17 apply, length, and iterator methods work correctly` {

    val examples =
      Table(
        ("a", "b", "c", "d", "e", "f", "g", "h", "i", "j", "k", "l", "m", "n", "o", "p", "q"),
        (  0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  3,   3,   3,   3,   3,   3,   3,   3,   3,   3,   3,   3,   3,   3,   3,   3,   3),
        (  4,   4,   4,   4,   4,   4,   4,   4,   4,   4,   4,   4,   4,   4,   4,   4,   4),
        (  5,   5,   5,   5,   5,   5,   5,   5,   5,   5,   5,   5,   5,   5,   5,   5,   5),
        (  6,   6,   6,   6,   6,   6,   6,   6,   6,   6,   6,   6,   6,   6,   6,   6,   6),
        (  7,   7,   7,   7,   7,   7,   7,   7,   7,   7,   7,   7,   7,   7,   7,   7,   7),
        (  8,   8,   8,   8,   8,   8,   8,   8,   8,   8,   8,   8,   8,   8,   8,   8,   8),
        (  9,   9,   9,   9,   9,   9,   9,   9,   9,   9,   9,   9,   9,   9,   9,   9,   9)
      )

    for (i <- 0 to 9) {
      assert(examples(i) === (i, i, i, i, i, i, i, i, i, i, i, i, i, i, i, i, i))
    }

    assert(examples.length === (10))

    var i = 0
    for (example <- examples.iterator) {
      assert(example === (i, i, i, i, i, i, i, i, i, i, i, i, i, i, i, i, i))
      i += 1
    }

    assert(examples.iterator.length === (10))
  }

  def `table for 18 that succeeds` {

    val examples =
      Table(
        ("a", "b", "c", "d", "e", "f", "g", "h", "i", "j", "k", "l", "m", "n", "o", "p", "q", "r"),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1)
      )

    forAll (examples) { (a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q, r) => assert(a + b + c + d + e + f + g + h + i + j + k + l + m + n + o + p + q + r === (18)) }
  }

  def `table for 18, which succeeds even though DiscardedEvaluationException is thrown` {
    val numbers =
      Table(
        ("a", "b", "c", "d", "e", "f", "g", "h", "i", "j", "k", "l", "m", "n", "o", "p", "q", "r"),
        ( -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1)
      )

    forAll (numbers) { (a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q, r) =>

      whenever (a > 0) {
        assert(a > 0)
      }
    }
  }

  def `table for 18, which fails` {

    val examples =
      Table(
        ("a", "b", "c", "d", "e", "f", "g", "h", "i", "j", "k", "l", "m", "n", "o", "p", "q", "r"),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2)
      )

    intercept[TableDrivenPropertyCheckFailedException] {
      forAll (examples) { (a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q, r) => assert(a + b + c + d + e + f + g + h + i + j + k + l + m + n + o + p + q + r === (18)) }
    }
  }

  def `table for 18 apply, length, and iterator methods work correctly` {

    val examples =
      Table(
        ("a", "b", "c", "d", "e", "f", "g", "h", "i", "j", "k", "l", "m", "n", "o", "p", "q", "r"),
        (  0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  3,   3,   3,   3,   3,   3,   3,   3,   3,   3,   3,   3,   3,   3,   3,   3,   3,   3),
        (  4,   4,   4,   4,   4,   4,   4,   4,   4,   4,   4,   4,   4,   4,   4,   4,   4,   4),
        (  5,   5,   5,   5,   5,   5,   5,   5,   5,   5,   5,   5,   5,   5,   5,   5,   5,   5),
        (  6,   6,   6,   6,   6,   6,   6,   6,   6,   6,   6,   6,   6,   6,   6,   6,   6,   6),
        (  7,   7,   7,   7,   7,   7,   7,   7,   7,   7,   7,   7,   7,   7,   7,   7,   7,   7),
        (  8,   8,   8,   8,   8,   8,   8,   8,   8,   8,   8,   8,   8,   8,   8,   8,   8,   8),
        (  9,   9,   9,   9,   9,   9,   9,   9,   9,   9,   9,   9,   9,   9,   9,   9,   9,   9)
      )

    for (i <- 0 to 9) {
      assert(examples(i) === (i, i, i, i, i, i, i, i, i, i, i, i, i, i, i, i, i, i))
    }

    assert(examples.length === (10))

    var i = 0
    for (example <- examples.iterator) {
      assert(example === (i, i, i, i, i, i, i, i, i, i, i, i, i, i, i, i, i, i))
      i += 1
    }

    assert(examples.iterator.length === (10))
  }

  def `table for 19 that succeeds` {

    val examples =
      Table(
        ("a", "b", "c", "d", "e", "f", "g", "h", "i", "j", "k", "l", "m", "n", "o", "p", "q", "r", "s"),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1)
      )

    forAll (examples) { (a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q, r, s) => assert(a + b + c + d + e + f + g + h + i + j + k + l + m + n + o + p + q + r + s === (19)) }
  }

  def `table for 19, which succeeds even though DiscardedEvaluationException is thrown` {
    val numbers =
      Table(
        ("a", "b", "c", "d", "e", "f", "g", "h", "i", "j", "k", "l", "m", "n", "o", "p", "q", "r", "s"),
        ( -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1)
      )

    forAll (numbers) { (a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q, r, s) =>

      whenever (a > 0) {
        assert(a > 0)
      }
    }
  }

  def `table for 19, which fails` {

    val examples =
      Table(
        ("a", "b", "c", "d", "e", "f", "g", "h", "i", "j", "k", "l", "m", "n", "o", "p", "q", "r", "s"),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2)
      )

    intercept[TableDrivenPropertyCheckFailedException] {
      forAll (examples) { (a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q, r, s) => assert(a + b + c + d + e + f + g + h + i + j + k + l + m + n + o + p + q + r + s === (19)) }
    }
  }

  def `table for 19 apply, length, and iterator methods work correctly` {

    val examples =
      Table(
        ("a", "b", "c", "d", "e", "f", "g", "h", "i", "j", "k", "l", "m", "n", "o", "p", "q", "r", "s"),
        (  0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  3,   3,   3,   3,   3,   3,   3,   3,   3,   3,   3,   3,   3,   3,   3,   3,   3,   3,   3),
        (  4,   4,   4,   4,   4,   4,   4,   4,   4,   4,   4,   4,   4,   4,   4,   4,   4,   4,   4),
        (  5,   5,   5,   5,   5,   5,   5,   5,   5,   5,   5,   5,   5,   5,   5,   5,   5,   5,   5),
        (  6,   6,   6,   6,   6,   6,   6,   6,   6,   6,   6,   6,   6,   6,   6,   6,   6,   6,   6),
        (  7,   7,   7,   7,   7,   7,   7,   7,   7,   7,   7,   7,   7,   7,   7,   7,   7,   7,   7),
        (  8,   8,   8,   8,   8,   8,   8,   8,   8,   8,   8,   8,   8,   8,   8,   8,   8,   8,   8),
        (  9,   9,   9,   9,   9,   9,   9,   9,   9,   9,   9,   9,   9,   9,   9,   9,   9,   9,   9)
      )

    for (i <- 0 to 9) {
      assert(examples(i) === (i, i, i, i, i, i, i, i, i, i, i, i, i, i, i, i, i, i, i))
    }

    assert(examples.length === (10))

    var i = 0
    for (example <- examples.iterator) {
      assert(example === (i, i, i, i, i, i, i, i, i, i, i, i, i, i, i, i, i, i, i))
      i += 1
    }

    assert(examples.iterator.length === (10))
  }

  def `table for 20 that succeeds` {

    val examples =
      Table(
        ("a", "b", "c", "d", "e", "f", "g", "h", "i", "j", "k", "l", "m", "n", "o", "p", "q", "r", "s", "t"),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1)
      )

    forAll (examples) { (a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q, r, s, t) => assert(a + b + c + d + e + f + g + h + i + j + k + l + m + n + o + p + q + r + s + t === (20)) }
  }

  def `table for 20, which succeeds even though DiscardedEvaluationException is thrown` {
    val numbers =
      Table(
        ("a", "b", "c", "d", "e", "f", "g", "h", "i", "j", "k", "l", "m", "n", "o", "p", "q", "r", "s", "t"),
        ( -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1)
      )

    forAll (numbers) { (a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q, r, s, t) =>

      whenever (a > 0) {
        assert(a > 0)
      }
    }
  }

  def `table for 20, which fails` {

    val examples =
      Table(
        ("a", "b", "c", "d", "e", "f", "g", "h", "i", "j", "k", "l", "m", "n", "o", "p", "q", "r", "s", "t"),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2)
      )

    intercept[TableDrivenPropertyCheckFailedException] {
      forAll (examples) { (a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q, r, s, t) => assert(a + b + c + d + e + f + g + h + i + j + k + l + m + n + o + p + q + r + s + t === (20)) }
    }
  }

  def `table for 20 apply, length, and iterator methods work correctly` {

    val examples =
      Table(
        ("a", "b", "c", "d", "e", "f", "g", "h", "i", "j", "k", "l", "m", "n", "o", "p", "q", "r", "s", "t"),
        (  0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  3,   3,   3,   3,   3,   3,   3,   3,   3,   3,   3,   3,   3,   3,   3,   3,   3,   3,   3,   3),
        (  4,   4,   4,   4,   4,   4,   4,   4,   4,   4,   4,   4,   4,   4,   4,   4,   4,   4,   4,   4),
        (  5,   5,   5,   5,   5,   5,   5,   5,   5,   5,   5,   5,   5,   5,   5,   5,   5,   5,   5,   5),
        (  6,   6,   6,   6,   6,   6,   6,   6,   6,   6,   6,   6,   6,   6,   6,   6,   6,   6,   6,   6),
        (  7,   7,   7,   7,   7,   7,   7,   7,   7,   7,   7,   7,   7,   7,   7,   7,   7,   7,   7,   7),
        (  8,   8,   8,   8,   8,   8,   8,   8,   8,   8,   8,   8,   8,   8,   8,   8,   8,   8,   8,   8),
        (  9,   9,   9,   9,   9,   9,   9,   9,   9,   9,   9,   9,   9,   9,   9,   9,   9,   9,   9,   9)
      )

    for (i <- 0 to 9) {
      assert(examples(i) === (i, i, i, i, i, i, i, i, i, i, i, i, i, i, i, i, i, i, i, i))
    }

    assert(examples.length === (10))

    var i = 0
    for (example <- examples.iterator) {
      assert(example === (i, i, i, i, i, i, i, i, i, i, i, i, i, i, i, i, i, i, i, i))
      i += 1
    }

    assert(examples.iterator.length === (10))
  }
}
