/*
 * Copyright 2001-2012 Artima, Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.scalatest
package prop


import matchers.ShouldMatchers
/* Uncomment this when remove the deprecated type aliases in the org.scalatest.prop package object.
import exceptions.GeneratorDrivenPropertyCheckFailedException
*/
import org.scalacheck.Gen
import GeneratorDrivenPropertyChecks._

class GeneratorDrivenPropertyChecksImportSuite extends FunSuite  with ShouldMatchers {

  val famousLastWords = for {
    s <- Gen.oneOf("the", "program", "compiles", "therefore", "it", "should", "work")
  } yield s

  val sevenEleven: Gen[String] =
    Gen.sized { (size: Int) =>
      if (size >= 7 && size <= 11)
        Gen.value("OKAY")
      else
        throw new Exception("expected 7 <= size <= 11 but got " + size)
    }

  val fiveFive: Gen[String] =
    Gen.sized { (size: Int) =>
      if (size == 5)
        Gen.value("OKAY")
      else
        throw new Exception("expected size 5 but got " + size)
    }


  test("generator-driven property that takes 1 args, which succeeds") {

    forAll { (a: String) =>
      a.length should equal ((a).length)
    }
  }

  test("generator-driven property that takes 1 args, which fails") {

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      forAll { (a: String) =>
        a.length should be < 0
      }
    }
  }

  test("generator-driven property that takes 1 named args, which succeeds") {

    forAll ("a") { (a: String) =>
      a.length should equal ((a).length)
    }
  }

  test("generator-driven property that takes 1 named args, which fails") {

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      forAll ("a") { (a: String) =>
        a.length should be < 0
      }
    }
  }

  test("generator-driven property that takes 1 args and generators, which succeeds") {

    forAll (famousLastWords) { (a: String) =>
      a.length should equal ((a).length)
    }
  }

  test("generator-driven property that takes 1 args and generators, which fails") {

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      forAll (famousLastWords) { (a: String) =>
        a.length should be < 0
      }
    }
  }

  test("generator-driven property that takes 1 named args and generators, which succeeds") {

    forAll ((famousLastWords, "a")) { (a: String) =>
      a.length should equal ((a).length)
    }
  }

  test("generator-driven property that takes 1 named args and generators, which fails") {

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      forAll ((famousLastWords, "a")) { (a: String) =>
        a.length should be < 0
      }
    }
  }

  // Same thing, but with config params
  test("generator-driven property that takes 1 args, which succeeds, with config params") {

    forAll (minSize(10), maxSize(20)) { (a: String) =>
      a.length should equal ((a).length)
    }
  }

  test("generator-driven property that takes 1 args, which fails, with config params") {

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      forAll (minSize(10), maxSize(20)) { (a: String) =>
        a.length should be < 0
      }
    }
  }

  test("generator-driven property that takes 1 named args, which succeeds, with config params") {

    forAll ("a", minSize(10), maxSize(20)) { (a: String) =>
      a.length should equal ((a).length)
    }
  }

  test("generator-driven property that takes 1 named args, which fails, with config params") {

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      forAll ("a", minSize(10), maxSize(20)) { (a: String) =>
        a.length should be < 0
      }
    }
  }

  test("generator-driven property that takes 1 args and generators, which succeeds, with config params") {

    forAll (famousLastWords, minSize(10), maxSize(20)) { (a: String) =>
      a.length should equal ((a).length)
    }
  }

  test("generator-driven property that takes 1 args and generators, which fails, with config params") {

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      forAll (famousLastWords, minSize(10), maxSize(20)) { (a: String) =>
        a.length should be < 0
      }
    }
  }

  test("generator-driven property that takes 1 named args and generators, which succeeds, with config params") {

    forAll ((famousLastWords, "a"), minSize(10), maxSize(20)) { (a: String) =>
      a.length should equal ((a).length)
    }
  }

  test("generator-driven property that takes 1 named args and generators, which fails, with config params") {

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      forAll ((famousLastWords, "a"), minSize(10), maxSize(20)) { (a: String) =>
        a.length should be < 0
      }
    }
  }

  // Same thing, but set minSuccessful to 5 with param, prop fails after 5
  test("generator-driven property that takes 1 args, which succeeds, with minSuccessful param set to 5") {

    var i = 0
    forAll (minSuccessful(5)) { (a: String) =>
      i += 1
      i should not be 6
    }
  }

  test("generator-driven property that takes 1 args, which fails, with minSuccessful param set to 5") {

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      var i = 0
      forAll (minSuccessful(5)) { (a: String) =>
        i += 1
        i should not be 5
      }
    }
  }

  test("generator-driven property that takes 1 named args, which succeeds, with minSuccessful param set to 5") {

    var i = 0
    forAll ("a", minSuccessful(5)) { (a: String) =>
      i += 1
      i should not be 6
    }
  }

  test("generator-driven property that takes 1 named args, which fails, with minSuccessful param set to 5") {

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      var i = 0
      forAll ("a", minSuccessful(5)) { (a: String) =>
        i += 1
        i should not be 5
      }
    }
  }

  test("generator-driven property that takes 1 args and generators, which succeeds, with minSuccessful param set to 5") {

    var i = 0
    forAll (famousLastWords, minSuccessful(5)) { (a: String) =>
      i += 1
      i should not be 6
    }
  }

  test("generator-driven property that takes 1 args and generators, which fails, with minSuccessful param set to 5") {

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      var i = 0
      forAll (famousLastWords, minSuccessful(5)) { (a: String) =>
        i += 1
        i should not be 5
      }
    }
  }

  test("generator-driven property that takes 1 named args and generators, which succeeds, with minSuccessful param set to 5") {

    var i = 0
    forAll ((famousLastWords, "a"), minSuccessful(5)) { (a: String) =>
      i += 1
      i should not be 6
    }
  }

  test("generator-driven property that takes 1 named args and generators, which fails, with minSuccessful param set to 5") {

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      var i = 0
      forAll ((famousLastWords, "a"), minSuccessful(5)) { (a: String) =>
        i += 1
        i should not be 5
      }
    }
  }

  // Same thing, but set default minSuccessful to 5, prop fails after 5
  test("generator-driven property that takes 1 args, which succeeds, with default minSuccessful param set to 5") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSuccessful = 5)

    var i = 0
    forAll { (a: String) =>
      i += 1
      i should not be 6
    }
  }

  test("generator-driven property that takes 1 args, which fails, with default minSuccessful param set to 5") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSuccessful = 5)

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      var i = 0
      forAll { (a: String) =>
        i += 1
        i should not be 5
      }
    }
  }

  test("generator-driven property that takes 1 named args, which succeeds, with default minSuccessful param set to 5") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSuccessful = 5)

    var i = 0
    forAll ("a") { (a: String) =>
      i += 1
      i should not be 6
    }
  }

  test("generator-driven property that takes 1 named args, which fails, with default minSuccessful param set to 5") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSuccessful = 5)

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      var i = 0
      forAll ("a") { (a: String) =>
        i += 1
        i should not be 5
      }
    }
  }

  test("generator-driven property that takes 1 args and generators, which succeeds, with default minSuccessful param set to 5") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSuccessful = 5)

    var i = 0
    forAll (famousLastWords) { (a: String) =>
      i += 1
      i should not be 6
    }
  }

  test("generator-driven property that takes 1 args and generators, which fails, with default minSuccessful param set to 5") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSuccessful = 5)

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      var i = 0
      forAll (famousLastWords) { (a: String) =>
        i += 1
        i should not be 5
      }
    }
  }

  test("generator-driven property that takes 1 named args and generators, which succeeds, with default minSuccessful param set to 5") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSuccessful = 5)

    var i = 0
    forAll ((famousLastWords, "a")) { (a: String) =>
      i += 1
      i should not be 6
    }
  }

  test("generator-driven property that takes 1 named args and generators, which fails, with default minSuccessful param set to 5") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSuccessful = 5)

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      var i = 0
      forAll ((famousLastWords, "a")) { (a: String) =>
        i += 1
        i should not be 5
      }
    }
  }

  // Same thing, but set maxDiscarded to 5 with param, prop fails after 5
  test("generator-driven property that takes 1 args, which succeeds, with maxDiscarded param set to 5") {

    var i = 0
    forAll (maxDiscarded(5)) { (a: String) =>
      i += 1
      whenever (i > 5) { 1 + 1 should equal (2) }
    }
  }

  test("generator-driven property that takes 1 args, which fails, with maxDiscarded param set to 5") {

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      var i = 0
      forAll (maxDiscarded(5)) { (a: String) =>
        i += 1
        whenever (i > 6) { 1 + 1 should equal (2) }
      }
    }
  }

  test("generator-driven property that takes 1 named args, which succeeds, with maxDiscarded param set to 5") {

    var i = 0
    forAll ("a", maxDiscarded(5)) { (a: String) =>
      i += 1
      whenever (i > 5) { 1 + 1 should equal (2) }
    }
  }

  test("generator-driven property that takes 1 named args, which fails, with maxDiscarded param set to 5") {

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      var i = 0
      forAll ("a", maxDiscarded(5)) { (a: String) =>
        i += 1
        whenever (i > 6) { 1 + 1 should equal (2) }
      }
    }
  }

  test("generator-driven property that takes 1 args and generators, which succeeds, with maxDiscarded param set to 5") {

    var i = 0
    forAll (famousLastWords, maxDiscarded(5)) { (a: String) =>
      i += 1
      whenever (i > 5) { 1 + 1 should equal (2) }
    }
  }

  test("generator-driven property that takes 1 args and generators, which fails, with maxDiscarded param set to 5") {

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      var i = 0
      forAll (famousLastWords, maxDiscarded(5)) { (a: String) =>
        i += 1
        whenever (i > 6) { 1 + 1 should equal (2) }
      }
    }
  }

  test("generator-driven property that takes 1 named args and generators, which succeeds, with maxDiscarded param set to 5") {

    var i = 0
    forAll ((famousLastWords, "a"), maxDiscarded(5)) { (a: String) =>
      i += 1
      whenever (i > 5) { 1 + 1 should equal (2) }
    }
  }

  test("generator-driven property that takes 1 named args and generators, which fails, with maxDiscarded param set to 5") {

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      var i = 0
      forAll ((famousLastWords, "a"), maxDiscarded(5)) { (a: String) =>
        i += 1
        whenever (i > 6) { 1 + 1 should equal (2) }
      }
    }
  }

  // Same thing, but set default maxDiscarded to 5, prop fails after 5
  test("generator-driven property that takes 1 args, which succeeds, with default maxDiscarded set to 5") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(maxDiscarded = 5)

    var i = 0
    forAll { (a: String) =>
      i += 1
      whenever (i > 5) { 1 + 1 should equal (2) }
    }
  }

  test("generator-driven property that takes 1 args, which fails, with default maxDiscarded set to 5") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(maxDiscarded = 5)

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      var i = 0
      forAll { (a: String) =>
        i += 1
        whenever (i > 6) { 1 + 1 should equal (2) }
      }
    }
  }

  test("generator-driven property that takes 1 named args, which succeeds, with default maxDiscarded set to 5") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(maxDiscarded = 5)

    var i = 0
    forAll ("a") { (a: String) =>
      i += 1
      whenever (i > 5) { 1 + 1 should equal (2) }
    }
  }

  test("generator-driven property that takes 1 named args, which fails, with default maxDiscarded set to 5") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(maxDiscarded = 5)

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      var i = 0
      forAll ("a") { (a: String) =>
        i += 1
        whenever (i > 6) { 1 + 1 should equal (2) }
      }
    }
  }

  test("generator-driven property that takes 1 args and generators, which succeeds, with default maxDiscarded set to 5") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(maxDiscarded = 5)

    var i = 0
    forAll (famousLastWords) { (a: String) =>
      i += 1
      whenever (i > 5) { 1 + 1 should equal (2) }
    }
  }

  test("generator-driven property that takes 1 args and generators, which fails, with default maxDiscarded set to 5") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(maxDiscarded = 5)

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      var i = 0
      forAll (famousLastWords) { (a: String) =>
        i += 1
        whenever (i > 6) { 1 + 1 should equal (2) }
      }
    }
  }

  test("generator-driven property that takes 1 named args and generators, which succeeds, with default maxDiscarded set to 5") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(maxDiscarded = 5)

    var i = 0
    forAll ((famousLastWords, "a")) { (a: String) =>
      i += 1
      whenever (i > 5) { 1 + 1 should equal (2) }
    }
  }

  test("generator-driven property that takes 1 named args and generators, which fails, with default maxDiscarded set to 5") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(maxDiscarded = 5)

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      var i = 0
      forAll ((famousLastWords, "a")) { (a: String) =>
        i += 1
        whenever (i > 6) { 1 + 1 should equal (2) }
      }
    }
  }

  // set minSize > maxSize with (param, param) (intercept IAE)
  test("generator-driven property that takes 1 args, which should throw IAE because maxSize > maxSize, specified as (param, param)") {

    intercept[IllegalArgumentException] {
      forAll (minSize(5), maxSize(4)) { (a: String) =>
        1 + 1 should equal (2)
      }
    }
  }

  test("generator-driven property that takes 1 named args, which should throw IAE because maxSize > maxSize, specified as (param, param)") {

    intercept[IllegalArgumentException] {
      forAll ("a", minSize(5), maxSize(4)) { (a: String) =>
        1 + 1 should equal (2)
      }
    }
  }

  test("generator-driven property that takes 1 args and generators, which should throw IAE because maxSize > maxSize, specified as (param, param)") {

    intercept[IllegalArgumentException] {
      forAll (famousLastWords, minSize(5), maxSize(4)) { (a: String) =>
        1 + 1 should equal (2)
      }
    }
  }

  test("generator-driven property that takes 1 named args and generators, which should throw IAE because maxSize > maxSize, specified as (param, param)") {

    intercept[IllegalArgumentException] {
      forAll ((famousLastWords, "a"), minSize(5), maxSize(4)) { (a: String) =>
        1 + 1 should equal (2)
      }
    }
  }

  // set minSize > maxSize with (param, default) (intercept IAE)
  test("generator-driven property that takes 1 args, which should throw IAE because maxSize > maxSize, specified as (param, default)") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(maxSize = 4)

    intercept[IllegalArgumentException] {
      forAll (minSize(5)) { (a: String) =>
        1 + 1 should equal (2)
      }
    }
  }

  test("generator-driven property that takes 1 named args, which should throw IAE because maxSize > maxSize, specified as (param, default)") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(maxSize = 4)

    intercept[IllegalArgumentException] {
      forAll ("a", minSize(5)) { (a: String) =>
        1 + 1 should equal (2)
      }
    }
  }

  test("generator-driven property that takes 1 args and generators, which should throw IAE because maxSize > maxSize, specified as (param, default)") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(maxSize = 4)

    intercept[IllegalArgumentException] {
      forAll (famousLastWords, minSize(5)) { (a: String) =>
        1 + 1 should equal (2)
      }
    }
  }

  test("generator-driven property that takes 1 named args and generators, which should throw IAE because maxSize > maxSize, specified as (param, default)") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(maxSize = 4)

    intercept[IllegalArgumentException] {
      forAll ((famousLastWords, "a"), minSize(5)) { (a: String) =>
        1 + 1 should equal (2)
      }
    }
  }

  // set minSize > maxSize with (default, param) (intercept IAE)
  test("generator-driven property that takes 1 args, which should throw IAE because maxSize > maxSize, specified as (default, param)") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSize = 5)

    intercept[IllegalArgumentException] {
      forAll (maxSize(4)) { (a: String) =>
        1 + 1 should equal (2)
      }
    }
  }

  test("generator-driven property that takes 1 named args, which should throw IAE because maxSize > maxSize, specified as (default, param)") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSize = 5)

    intercept[IllegalArgumentException] {
      forAll ("a", maxSize(4)) { (a: String) =>
        1 + 1 should equal (2)
      }
    }
  }

  test("generator-driven property that takes 1 args and generators, which should throw IAE because maxSize > maxSize, specified as (default, param)") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSize = 5)

    intercept[IllegalArgumentException] {
      forAll (famousLastWords, maxSize(4)) { (a: String) =>
        1 + 1 should equal (2)
      }
    }
  }

  test("generator-driven property that takes 1 named args and generators, which should throw IAE because maxSize > maxSize, specified as (default, param)") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSize = 5)

    intercept[IllegalArgumentException] {
      forAll ((famousLastWords, "a"), maxSize(4)) { (a: String) =>
        1 + 1 should equal (2)
      }
    }
  }

  // set maxSize with param (ensure always passed with a size less than maxSize)
  test("generator-driven property that takes 1 args, with maxSize specified as param") {

    forAll (maxSize(5)) { (a: String) =>
      a.length should be <= 5
    }
  }

  test("generator-driven property that takes 1 named args, with maxSize specified as param") {

    forAll ("a", maxSize(5)) { (a: String) =>
      a.length should be <= 5
    }
  }

  // set maxSize with default (ensure always passed with a size less than maxSize)
  test("generator-driven property that takes 1 args, with maxSize specified as default") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(maxSize = 5)

    forAll { (a: String) =>
      a.length should be <= 5
    }
  }

  test("generator-driven property that takes 1 named args, with maxSize specified as default") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(maxSize = 5)

    forAll ("a") { (a: String) =>
      a.length should be <= 5
    }
  }
 
  // set minSize == maxSize with (param, param) (ensure always passed with that size)
  test("generator-driven property that takes 1 args and generators, with minSize == maxSize, specified as (param, param)") {

    pendingUntilFixed {
      forAll (fiveFive, minSize(5), maxSize(5)) { (a: String) =>
        a should be ("OKAY")
      }
    }
  }

  test("generator-driven property that takes 1 named args and generators, with minSize == maxSize, specified as (param, param)") {

    pendingUntilFixed {
      forAll ((fiveFive, "a"), minSize(5), maxSize(5)) { (a: String) =>
        a should be ("OKAY")
      }
    }
  }

  // set minSize == maxSize with (param, default) (ensure always passed with that size)
  test("generator-driven property that takes 1 args and generators, with minSize == maxSize, specified as (param, default)") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(maxSize = 5)

    pendingUntilFixed {
      forAll (fiveFive, minSize(5)) { (a: String) =>
        a should be ("OKAY")
      }
    }
  }

  test("generator-driven property that takes 1 named args and generators, with minSize == maxSize, specified as (param, default)") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(maxSize = 5)

    pendingUntilFixed {
      forAll ((fiveFive, "a"), minSize(5)) { (a: String) =>
        a should be ("OKAY")
      }
    }
  }

  // set minSize == maxSize with (default, param) (ensure always passed with that size)
  test("generator-driven property that takes 1 args and generators, with minSize == maxSize, specified as (default, param)") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSize = 5)

    pendingUntilFixed {
      forAll (fiveFive, maxSize(5)) { (a: String) =>
        a should be ("OKAY")
      }
    }
  }

  test("generator-driven property that takes 1 named args and generators, with minSize == maxSize, specified as (default, param)") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSize = 5)

    pendingUntilFixed {
      forAll ((fiveFive, "a"), maxSize(5)) { (a: String) =>
        a should be ("OKAY")
      }
    }
  }

  // set minSize == maxSize with (default, default) (ensure always passed with that size)
  test("generator-driven property that takes 1 args and generators, with minSize == maxSize, specified as (default, default)") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSize = 5, maxSize = 5)

    pendingUntilFixed {
      forAll (fiveFive) { (a: String) =>
        a should be ("OKAY")
      }
    }
  }

  test("generator-driven property that takes 1 named args and generators, with minSize == maxSize, specified as (default, default)") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSize = 5, maxSize = 5)

    pendingUntilFixed {
      forAll ((fiveFive, "a")) { (a: String) =>
        a should be ("OKAY")
      }
    }
  }

  // set minSize to 7 and maxSize to 11 with (param, param) (ensure always passed with that size)
  test("generator-driven property that takes 1 args and generators, with minSize to 7 and maxSize to 11, specified as (param, param)") {

    pendingUntilFixed {
      forAll (sevenEleven, minSize(7), maxSize(11)) { (a: String) =>
        a should be ("OKAY")
      }
    }
  }

  test("generator-driven property that takes 1 named args and generators, with minSize to 7 and maxSize to 11, specified as (param, param)") {

    pendingUntilFixed {
      forAll ((sevenEleven, "a"), minSize(7), maxSize(11)) { (a: String) =>
        a should be ("OKAY")
      }
    }
  }

  // set minSize to 7 and maxSize to 11 with (param, default) (ensure always passed with that size)
  test("generator-driven property that takes 1 args and generators, with minSize to 7 and maxSize to 11, specified as (param, default)") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(maxSize = 11)

    pendingUntilFixed {
      forAll (sevenEleven, minSize(7)) { (a: String) =>
        a should be ("OKAY")
      }
    }
  }

  test("generator-driven property that takes 1 named args and generators, with minSize to 7 and maxSize to 11, specified as (param, default)") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(maxSize = 11)

    pendingUntilFixed {
      forAll ((sevenEleven, "a"), minSize(7)) { (a: String) =>
        a should be ("OKAY")
      }
    }
  }

  // set minSize to 7 and maxSize to 11 with (default, param) (ensure always passed with that size)
  test("generator-driven property that takes 1 args and generators, with minSize to 7 and maxSize to 11, specified as (default, param)") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSize = 7)

    pendingUntilFixed {
      forAll (sevenEleven, maxSize(11)) { (a: String) =>
        a should be ("OKAY")
      }
    }
  }

  test("generator-driven property that takes 1 named args and generators, with minSize to 7 and maxSize to 11, specified as (default, param)") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSize = 7)

    pendingUntilFixed {
      forAll ((sevenEleven, "a"), maxSize(11)) { (a: String) =>
        a should be ("OKAY")
      }
    }
  }

  // set minSize to 7 and maxSize to 11 with (default, default) (ensure always passed with that size)
  test("generator-driven property that takes 1 args and generators, with minSize to 7 and maxSize to 11, specified as (default, default)") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSize = 7, maxSize = 11)

    pendingUntilFixed {
      forAll (sevenEleven) { (a: String) =>
        a should be ("OKAY")
      }
    }
  }

  test("generator-driven property that takes 1 named args and generators, with minSize to 7 and maxSize to 11, specified as (default, default)") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSize = 7, maxSize = 11)

    pendingUntilFixed {
      forAll ((sevenEleven, "a")) { (a: String) =>
        a should be ("OKAY")
      }
    }
  }


  test("generator-driven property that takes 2 args, which succeeds") {

    forAll { (a: String, b: String) =>
      a.length + b.length should equal ((a + b).length)
    }
  }

  test("generator-driven property that takes 2 args, which fails") {

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      forAll { (a: String, b: String) =>
        a.length + b.length should be < 0
      }
    }
  }

  test("generator-driven property that takes 2 named args, which succeeds") {

    forAll ("a", "b") { (a: String, b: String) =>
      a.length + b.length should equal ((a + b).length)
    }
  }

  test("generator-driven property that takes 2 named args, which fails") {

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      forAll ("a", "b") { (a: String, b: String) =>
        a.length + b.length should be < 0
      }
    }
  }

  test("generator-driven property that takes 2 args and generators, which succeeds") {

    forAll (famousLastWords, famousLastWords) { (a: String, b: String) =>
      a.length + b.length should equal ((a + b).length)
    }
  }

  test("generator-driven property that takes 2 args and generators, which fails") {

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      forAll (famousLastWords, famousLastWords) { (a: String, b: String) =>
        a.length + b.length should be < 0
      }
    }
  }

  test("generator-driven property that takes 2 named args and generators, which succeeds") {

    forAll ((famousLastWords, "a"), (famousLastWords, "b")) { (a: String, b: String) =>
      a.length + b.length should equal ((a + b).length)
    }
  }

  test("generator-driven property that takes 2 named args and generators, which fails") {

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      forAll ((famousLastWords, "a"), (famousLastWords, "b")) { (a: String, b: String) =>
        a.length + b.length should be < 0
      }
    }
  }

  // Same thing, but with config params
  test("generator-driven property that takes 2 args, which succeeds, with config params") {

    forAll (minSize(10), maxSize(20)) { (a: String, b: String) =>
      a.length + b.length should equal ((a + b).length)
    }
  }

  test("generator-driven property that takes 2 args, which fails, with config params") {

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      forAll (minSize(10), maxSize(20)) { (a: String, b: String) =>
        a.length + b.length should be < 0
      }
    }
  }

  test("generator-driven property that takes 2 named args, which succeeds, with config params") {

    forAll ("a", "b", minSize(10), maxSize(20)) { (a: String, b: String) =>
      a.length + b.length should equal ((a + b).length)
    }
  }

  test("generator-driven property that takes 2 named args, which fails, with config params") {

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      forAll ("a", "b", minSize(10), maxSize(20)) { (a: String, b: String) =>
        a.length + b.length should be < 0
      }
    }
  }

  test("generator-driven property that takes 2 args and generators, which succeeds, with config params") {

    forAll (famousLastWords, famousLastWords, minSize(10), maxSize(20)) { (a: String, b: String) =>
      a.length + b.length should equal ((a + b).length)
    }
  }

  test("generator-driven property that takes 2 args and generators, which fails, with config params") {

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      forAll (famousLastWords, famousLastWords, minSize(10), maxSize(20)) { (a: String, b: String) =>
        a.length + b.length should be < 0
      }
    }
  }

  test("generator-driven property that takes 2 named args and generators, which succeeds, with config params") {

    forAll ((famousLastWords, "a"), (famousLastWords, "b"), minSize(10), maxSize(20)) { (a: String, b: String) =>
      a.length + b.length should equal ((a + b).length)
    }
  }

  test("generator-driven property that takes 2 named args and generators, which fails, with config params") {

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      forAll ((famousLastWords, "a"), (famousLastWords, "b"), minSize(10), maxSize(20)) { (a: String, b: String) =>
        a.length + b.length should be < 0
      }
    }
  }

  // Same thing, but set minSuccessful to 5 with param, prop fails after 5
  test("generator-driven property that takes 2 args, which succeeds, with minSuccessful param set to 5") {

    var i = 0
    forAll (minSuccessful(5)) { (a: String, b: String) =>
      i += 1
      i should not be 6
    }
  }

  test("generator-driven property that takes 2 args, which fails, with minSuccessful param set to 5") {

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      var i = 0
      forAll (minSuccessful(5)) { (a: String, b: String) =>
        i += 1
        i should not be 5
      }
    }
  }

  test("generator-driven property that takes 2 named args, which succeeds, with minSuccessful param set to 5") {

    var i = 0
    forAll ("a", "b", minSuccessful(5)) { (a: String, b: String) =>
      i += 1
      i should not be 6
    }
  }

  test("generator-driven property that takes 2 named args, which fails, with minSuccessful param set to 5") {

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      var i = 0
      forAll ("a", "b", minSuccessful(5)) { (a: String, b: String) =>
        i += 1
        i should not be 5
      }
    }
  }

  test("generator-driven property that takes 2 args and generators, which succeeds, with minSuccessful param set to 5") {

    var i = 0
    forAll (famousLastWords, famousLastWords, minSuccessful(5)) { (a: String, b: String) =>
      i += 1
      i should not be 6
    }
  }

  test("generator-driven property that takes 2 args and generators, which fails, with minSuccessful param set to 5") {

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      var i = 0
      forAll (famousLastWords, famousLastWords, minSuccessful(5)) { (a: String, b: String) =>
        i += 1
        i should not be 5
      }
    }
  }

  test("generator-driven property that takes 2 named args and generators, which succeeds, with minSuccessful param set to 5") {

    var i = 0
    forAll ((famousLastWords, "a"), (famousLastWords, "b"), minSuccessful(5)) { (a: String, b: String) =>
      i += 1
      i should not be 6
    }
  }

  test("generator-driven property that takes 2 named args and generators, which fails, with minSuccessful param set to 5") {

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      var i = 0
      forAll ((famousLastWords, "a"), (famousLastWords, "b"), minSuccessful(5)) { (a: String, b: String) =>
        i += 1
        i should not be 5
      }
    }
  }

  // Same thing, but set default minSuccessful to 5, prop fails after 5
  test("generator-driven property that takes 2 args, which succeeds, with default minSuccessful param set to 5") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSuccessful = 5)

    var i = 0
    forAll { (a: String, b: String) =>
      i += 1
      i should not be 6
    }
  }

  test("generator-driven property that takes 2 args, which fails, with default minSuccessful param set to 5") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSuccessful = 5)

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      var i = 0
      forAll { (a: String, b: String) =>
        i += 1
        i should not be 5
      }
    }
  }

  test("generator-driven property that takes 2 named args, which succeeds, with default minSuccessful param set to 5") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSuccessful = 5)

    var i = 0
    forAll ("a", "b") { (a: String, b: String) =>
      i += 1
      i should not be 6
    }
  }

  test("generator-driven property that takes 2 named args, which fails, with default minSuccessful param set to 5") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSuccessful = 5)

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      var i = 0
      forAll ("a", "b") { (a: String, b: String) =>
        i += 1
        i should not be 5
      }
    }
  }

  test("generator-driven property that takes 2 args and generators, which succeeds, with default minSuccessful param set to 5") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSuccessful = 5)

    var i = 0
    forAll (famousLastWords, famousLastWords) { (a: String, b: String) =>
      i += 1
      i should not be 6
    }
  }

  test("generator-driven property that takes 2 args and generators, which fails, with default minSuccessful param set to 5") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSuccessful = 5)

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      var i = 0
      forAll (famousLastWords, famousLastWords) { (a: String, b: String) =>
        i += 1
        i should not be 5
      }
    }
  }

  test("generator-driven property that takes 2 named args and generators, which succeeds, with default minSuccessful param set to 5") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSuccessful = 5)

    var i = 0
    forAll ((famousLastWords, "a"), (famousLastWords, "b")) { (a: String, b: String) =>
      i += 1
      i should not be 6
    }
  }

  test("generator-driven property that takes 2 named args and generators, which fails, with default minSuccessful param set to 5") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSuccessful = 5)

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      var i = 0
      forAll ((famousLastWords, "a"), (famousLastWords, "b")) { (a: String, b: String) =>
        i += 1
        i should not be 5
      }
    }
  }

  // Same thing, but set maxDiscarded to 5 with param, prop fails after 5
  test("generator-driven property that takes 2 args, which succeeds, with maxDiscarded param set to 5") {

    var i = 0
    forAll (maxDiscarded(5)) { (a: String, b: String) =>
      i += 1
      whenever (i > 5) { 1 + 1 should equal (2) }
    }
  }

  test("generator-driven property that takes 2 args, which fails, with maxDiscarded param set to 5") {

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      var i = 0
      forAll (maxDiscarded(5)) { (a: String, b: String) =>
        i += 1
        whenever (i > 6) { 1 + 1 should equal (2) }
      }
    }
  }

  test("generator-driven property that takes 2 named args, which succeeds, with maxDiscarded param set to 5") {

    var i = 0
    forAll ("a", "b", maxDiscarded(5)) { (a: String, b: String) =>
      i += 1
      whenever (i > 5) { 1 + 1 should equal (2) }
    }
  }

  test("generator-driven property that takes 2 named args, which fails, with maxDiscarded param set to 5") {

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      var i = 0
      forAll ("a", "b", maxDiscarded(5)) { (a: String, b: String) =>
        i += 1
        whenever (i > 6) { 1 + 1 should equal (2) }
      }
    }
  }

  test("generator-driven property that takes 2 args and generators, which succeeds, with maxDiscarded param set to 5") {

    var i = 0
    forAll (famousLastWords, famousLastWords, maxDiscarded(5)) { (a: String, b: String) =>
      i += 1
      whenever (i > 5) { 1 + 1 should equal (2) }
    }
  }

  test("generator-driven property that takes 2 args and generators, which fails, with maxDiscarded param set to 5") {

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      var i = 0
      forAll (famousLastWords, famousLastWords, maxDiscarded(5)) { (a: String, b: String) =>
        i += 1
        whenever (i > 6) { 1 + 1 should equal (2) }
      }
    }
  }

  test("generator-driven property that takes 2 named args and generators, which succeeds, with maxDiscarded param set to 5") {

    var i = 0
    forAll ((famousLastWords, "a"), (famousLastWords, "b"), maxDiscarded(5)) { (a: String, b: String) =>
      i += 1
      whenever (i > 5) { 1 + 1 should equal (2) }
    }
  }

  test("generator-driven property that takes 2 named args and generators, which fails, with maxDiscarded param set to 5") {

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      var i = 0
      forAll ((famousLastWords, "a"), (famousLastWords, "b"), maxDiscarded(5)) { (a: String, b: String) =>
        i += 1
        whenever (i > 6) { 1 + 1 should equal (2) }
      }
    }
  }

  // Same thing, but set default maxDiscarded to 5, prop fails after 5
  test("generator-driven property that takes 2 args, which succeeds, with default maxDiscarded set to 5") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(maxDiscarded = 5)

    var i = 0
    forAll { (a: String, b: String) =>
      i += 1
      whenever (i > 5) { 1 + 1 should equal (2) }
    }
  }

  test("generator-driven property that takes 2 args, which fails, with default maxDiscarded set to 5") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(maxDiscarded = 5)

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      var i = 0
      forAll { (a: String, b: String) =>
        i += 1
        whenever (i > 6) { 1 + 1 should equal (2) }
      }
    }
  }

  test("generator-driven property that takes 2 named args, which succeeds, with default maxDiscarded set to 5") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(maxDiscarded = 5)

    var i = 0
    forAll ("a", "b") { (a: String, b: String) =>
      i += 1
      whenever (i > 5) { 1 + 1 should equal (2) }
    }
  }

  test("generator-driven property that takes 2 named args, which fails, with default maxDiscarded set to 5") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(maxDiscarded = 5)

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      var i = 0
      forAll ("a", "b") { (a: String, b: String) =>
        i += 1
        whenever (i > 6) { 1 + 1 should equal (2) }
      }
    }
  }

  test("generator-driven property that takes 2 args and generators, which succeeds, with default maxDiscarded set to 5") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(maxDiscarded = 5)

    var i = 0
    forAll (famousLastWords, famousLastWords) { (a: String, b: String) =>
      i += 1
      whenever (i > 5) { 1 + 1 should equal (2) }
    }
  }

  test("generator-driven property that takes 2 args and generators, which fails, with default maxDiscarded set to 5") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(maxDiscarded = 5)

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      var i = 0
      forAll (famousLastWords, famousLastWords) { (a: String, b: String) =>
        i += 1
        whenever (i > 6) { 1 + 1 should equal (2) }
      }
    }
  }

  test("generator-driven property that takes 2 named args and generators, which succeeds, with default maxDiscarded set to 5") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(maxDiscarded = 5)

    var i = 0
    forAll ((famousLastWords, "a"), (famousLastWords, "b")) { (a: String, b: String) =>
      i += 1
      whenever (i > 5) { 1 + 1 should equal (2) }
    }
  }

  test("generator-driven property that takes 2 named args and generators, which fails, with default maxDiscarded set to 5") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(maxDiscarded = 5)

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      var i = 0
      forAll ((famousLastWords, "a"), (famousLastWords, "b")) { (a: String, b: String) =>
        i += 1
        whenever (i > 6) { 1 + 1 should equal (2) }
      }
    }
  }

  // set minSize > maxSize with (param, param) (intercept IAE)
  test("generator-driven property that takes 2 args, which should throw IAE because maxSize > maxSize, specified as (param, param)") {

    intercept[IllegalArgumentException] {
      forAll (minSize(5), maxSize(4)) { (a: String, b: String) =>
        1 + 1 should equal (2)
      }
    }
  }

  test("generator-driven property that takes 2 named args, which should throw IAE because maxSize > maxSize, specified as (param, param)") {

    intercept[IllegalArgumentException] {
      forAll ("a", "b", minSize(5), maxSize(4)) { (a: String, b: String) =>
        1 + 1 should equal (2)
      }
    }
  }

  test("generator-driven property that takes 2 args and generators, which should throw IAE because maxSize > maxSize, specified as (param, param)") {

    intercept[IllegalArgumentException] {
      forAll (famousLastWords, famousLastWords, minSize(5), maxSize(4)) { (a: String, b: String) =>
        1 + 1 should equal (2)
      }
    }
  }

  test("generator-driven property that takes 2 named args and generators, which should throw IAE because maxSize > maxSize, specified as (param, param)") {

    intercept[IllegalArgumentException] {
      forAll ((famousLastWords, "a"), (famousLastWords, "b"), minSize(5), maxSize(4)) { (a: String, b: String) =>
        1 + 1 should equal (2)
      }
    }
  }

  // set minSize > maxSize with (param, default) (intercept IAE)
  test("generator-driven property that takes 2 args, which should throw IAE because maxSize > maxSize, specified as (param, default)") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(maxSize = 4)

    intercept[IllegalArgumentException] {
      forAll (minSize(5)) { (a: String, b: String) =>
        1 + 1 should equal (2)
      }
    }
  }

  test("generator-driven property that takes 2 named args, which should throw IAE because maxSize > maxSize, specified as (param, default)") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(maxSize = 4)

    intercept[IllegalArgumentException] {
      forAll ("a", "b", minSize(5)) { (a: String, b: String) =>
        1 + 1 should equal (2)
      }
    }
  }

  test("generator-driven property that takes 2 args and generators, which should throw IAE because maxSize > maxSize, specified as (param, default)") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(maxSize = 4)

    intercept[IllegalArgumentException] {
      forAll (famousLastWords, famousLastWords, minSize(5)) { (a: String, b: String) =>
        1 + 1 should equal (2)
      }
    }
  }

  test("generator-driven property that takes 2 named args and generators, which should throw IAE because maxSize > maxSize, specified as (param, default)") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(maxSize = 4)

    intercept[IllegalArgumentException] {
      forAll ((famousLastWords, "a"), (famousLastWords, "b"), minSize(5)) { (a: String, b: String) =>
        1 + 1 should equal (2)
      }
    }
  }

  // set minSize > maxSize with (default, param) (intercept IAE)
  test("generator-driven property that takes 2 args, which should throw IAE because maxSize > maxSize, specified as (default, param)") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSize = 5)

    intercept[IllegalArgumentException] {
      forAll (maxSize(4)) { (a: String, b: String) =>
        1 + 1 should equal (2)
      }
    }
  }

  test("generator-driven property that takes 2 named args, which should throw IAE because maxSize > maxSize, specified as (default, param)") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSize = 5)

    intercept[IllegalArgumentException] {
      forAll ("a", "b", maxSize(4)) { (a: String, b: String) =>
        1 + 1 should equal (2)
      }
    }
  }

  test("generator-driven property that takes 2 args and generators, which should throw IAE because maxSize > maxSize, specified as (default, param)") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSize = 5)

    intercept[IllegalArgumentException] {
      forAll (famousLastWords, famousLastWords, maxSize(4)) { (a: String, b: String) =>
        1 + 1 should equal (2)
      }
    }
  }

  test("generator-driven property that takes 2 named args and generators, which should throw IAE because maxSize > maxSize, specified as (default, param)") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSize = 5)

    intercept[IllegalArgumentException] {
      forAll ((famousLastWords, "a"), (famousLastWords, "b"), maxSize(4)) { (a: String, b: String) =>
        1 + 1 should equal (2)
      }
    }
  }

  // set maxSize with param (ensure always passed with a size less than maxSize)
  test("generator-driven property that takes 2 args, with maxSize specified as param") {

    forAll (maxSize(5)) { (a: String, b: String) =>
      a.length should be <= 5
      b.length should be <= 5
    }
  }

  test("generator-driven property that takes 2 named args, with maxSize specified as param") {

    forAll ("a", "b", maxSize(5)) { (a: String, b: String) =>
      a.length should be <= 5
      b.length should be <= 5
    }
  }

  // set maxSize with default (ensure always passed with a size less than maxSize)
  test("generator-driven property that takes 2 args, with maxSize specified as default") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(maxSize = 5)

    forAll { (a: String, b: String) =>
      a.length should be <= 5
      b.length should be <= 5
    }
  }

  test("generator-driven property that takes 2 named args, with maxSize specified as default") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(maxSize = 5)

    forAll ("a", "b") { (a: String, b: String) =>
      a.length should be <= 5
      b.length should be <= 5
    }
  }
 
  // set minSize == maxSize with (param, param) (ensure always passed with that size)
  test("generator-driven property that takes 2 args and generators, with minSize == maxSize, specified as (param, param)") {

    pendingUntilFixed {
      forAll (fiveFive, fiveFive, minSize(5), maxSize(5)) { (a: String, b: String) =>
        a should be ("OKAY")
        b should be ("OKAY")
      }
    }
  }

  test("generator-driven property that takes 2 named args and generators, with minSize == maxSize, specified as (param, param)") {

    pendingUntilFixed {
      forAll ((fiveFive, "a"), (fiveFive, "b"), minSize(5), maxSize(5)) { (a: String, b: String) =>
        a should be ("OKAY")
        b should be ("OKAY")
      }
    }
  }

  // set minSize == maxSize with (param, default) (ensure always passed with that size)
  test("generator-driven property that takes 2 args and generators, with minSize == maxSize, specified as (param, default)") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(maxSize = 5)

    pendingUntilFixed {
      forAll (fiveFive, fiveFive, minSize(5)) { (a: String, b: String) =>
        a should be ("OKAY")
        b should be ("OKAY")
      }
    }
  }

  test("generator-driven property that takes 2 named args and generators, with minSize == maxSize, specified as (param, default)") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(maxSize = 5)

    pendingUntilFixed {
      forAll ((fiveFive, "a"), (fiveFive, "b"), minSize(5)) { (a: String, b: String) =>
        a should be ("OKAY")
        b should be ("OKAY")
      }
    }
  }

  // set minSize == maxSize with (default, param) (ensure always passed with that size)
  test("generator-driven property that takes 2 args and generators, with minSize == maxSize, specified as (default, param)") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSize = 5)

    pendingUntilFixed {
      forAll (fiveFive, fiveFive, maxSize(5)) { (a: String, b: String) =>
        a should be ("OKAY")
        b should be ("OKAY")
      }
    }
  }

  test("generator-driven property that takes 2 named args and generators, with minSize == maxSize, specified as (default, param)") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSize = 5)

    pendingUntilFixed {
      forAll ((fiveFive, "a"), (fiveFive, "b"), maxSize(5)) { (a: String, b: String) =>
        a should be ("OKAY")
        b should be ("OKAY")
      }
    }
  }

  // set minSize == maxSize with (default, default) (ensure always passed with that size)
  test("generator-driven property that takes 2 args and generators, with minSize == maxSize, specified as (default, default)") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSize = 5, maxSize = 5)

    pendingUntilFixed {
      forAll (fiveFive, fiveFive) { (a: String, b: String) =>
        a should be ("OKAY")
        b should be ("OKAY")
      }
    }
  }

  test("generator-driven property that takes 2 named args and generators, with minSize == maxSize, specified as (default, default)") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSize = 5, maxSize = 5)

    pendingUntilFixed {
      forAll ((fiveFive, "a"), (fiveFive, "b")) { (a: String, b: String) =>
        a should be ("OKAY")
        b should be ("OKAY")
      }
    }
  }

  // set minSize to 7 and maxSize to 11 with (param, param) (ensure always passed with that size)
  test("generator-driven property that takes 2 args and generators, with minSize to 7 and maxSize to 11, specified as (param, param)") {

    pendingUntilFixed {
      forAll (sevenEleven, sevenEleven, minSize(7), maxSize(11)) { (a: String, b: String) =>
        a should be ("OKAY")
        b should be ("OKAY")
      }
    }
  }

  test("generator-driven property that takes 2 named args and generators, with minSize to 7 and maxSize to 11, specified as (param, param)") {

    pendingUntilFixed {
      forAll ((sevenEleven, "a"), (sevenEleven, "b"), minSize(7), maxSize(11)) { (a: String, b: String) =>
        a should be ("OKAY")
        b should be ("OKAY")
      }
    }
  }

  // set minSize to 7 and maxSize to 11 with (param, default) (ensure always passed with that size)
  test("generator-driven property that takes 2 args and generators, with minSize to 7 and maxSize to 11, specified as (param, default)") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(maxSize = 11)

    pendingUntilFixed {
      forAll (sevenEleven, sevenEleven, minSize(7)) { (a: String, b: String) =>
        a should be ("OKAY")
        b should be ("OKAY")
      }
    }
  }

  test("generator-driven property that takes 2 named args and generators, with minSize to 7 and maxSize to 11, specified as (param, default)") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(maxSize = 11)

    pendingUntilFixed {
      forAll ((sevenEleven, "a"), (sevenEleven, "b"), minSize(7)) { (a: String, b: String) =>
        a should be ("OKAY")
        b should be ("OKAY")
      }
    }
  }

  // set minSize to 7 and maxSize to 11 with (default, param) (ensure always passed with that size)
  test("generator-driven property that takes 2 args and generators, with minSize to 7 and maxSize to 11, specified as (default, param)") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSize = 7)

    pendingUntilFixed {
      forAll (sevenEleven, sevenEleven, maxSize(11)) { (a: String, b: String) =>
        a should be ("OKAY")
        b should be ("OKAY")
      }
    }
  }

  test("generator-driven property that takes 2 named args and generators, with minSize to 7 and maxSize to 11, specified as (default, param)") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSize = 7)

    pendingUntilFixed {
      forAll ((sevenEleven, "a"), (sevenEleven, "b"), maxSize(11)) { (a: String, b: String) =>
        a should be ("OKAY")
        b should be ("OKAY")
      }
    }
  }

  // set minSize to 7 and maxSize to 11 with (default, default) (ensure always passed with that size)
  test("generator-driven property that takes 2 args and generators, with minSize to 7 and maxSize to 11, specified as (default, default)") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSize = 7, maxSize = 11)

    pendingUntilFixed {
      forAll (sevenEleven, sevenEleven) { (a: String, b: String) =>
        a should be ("OKAY")
        b should be ("OKAY")
      }
    }
  }

  test("generator-driven property that takes 2 named args and generators, with minSize to 7 and maxSize to 11, specified as (default, default)") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSize = 7, maxSize = 11)

    pendingUntilFixed {
      forAll ((sevenEleven, "a"), (sevenEleven, "b")) { (a: String, b: String) =>
        a should be ("OKAY")
        b should be ("OKAY")
      }
    }
  }


  test("generator-driven property that takes 3 args, which succeeds") {

    forAll { (a: String, b: String, c: String) =>
      a.length + b.length + c.length should equal ((a + b + c).length)
    }
  }

  test("generator-driven property that takes 3 args, which fails") {

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      forAll { (a: String, b: String, c: String) =>
        a.length + b.length + c.length should be < 0
      }
    }
  }

  test("generator-driven property that takes 3 named args, which succeeds") {

    forAll ("a", "b", "c") { (a: String, b: String, c: String) =>
      a.length + b.length + c.length should equal ((a + b + c).length)
    }
  }

  test("generator-driven property that takes 3 named args, which fails") {

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      forAll ("a", "b", "c") { (a: String, b: String, c: String) =>
        a.length + b.length + c.length should be < 0
      }
    }
  }

  test("generator-driven property that takes 3 args and generators, which succeeds") {

    forAll (famousLastWords, famousLastWords, famousLastWords) { (a: String, b: String, c: String) =>
      a.length + b.length + c.length should equal ((a + b + c).length)
    }
  }

  test("generator-driven property that takes 3 args and generators, which fails") {

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      forAll (famousLastWords, famousLastWords, famousLastWords) { (a: String, b: String, c: String) =>
        a.length + b.length + c.length should be < 0
      }
    }
  }

  test("generator-driven property that takes 3 named args and generators, which succeeds") {

    forAll ((famousLastWords, "a"), (famousLastWords, "b"), (famousLastWords, "c")) { (a: String, b: String, c: String) =>
      a.length + b.length + c.length should equal ((a + b + c).length)
    }
  }

  test("generator-driven property that takes 3 named args and generators, which fails") {

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      forAll ((famousLastWords, "a"), (famousLastWords, "b"), (famousLastWords, "c")) { (a: String, b: String, c: String) =>
        a.length + b.length + c.length should be < 0
      }
    }
  }

  // Same thing, but with config params
  test("generator-driven property that takes 3 args, which succeeds, with config params") {

    forAll (minSize(10), maxSize(20)) { (a: String, b: String, c: String) =>
      a.length + b.length + c.length should equal ((a + b + c).length)
    }
  }

  test("generator-driven property that takes 3 args, which fails, with config params") {

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      forAll (minSize(10), maxSize(20)) { (a: String, b: String, c: String) =>
        a.length + b.length + c.length should be < 0
      }
    }
  }

  test("generator-driven property that takes 3 named args, which succeeds, with config params") {

    forAll ("a", "b", "c", minSize(10), maxSize(20)) { (a: String, b: String, c: String) =>
      a.length + b.length + c.length should equal ((a + b + c).length)
    }
  }

  test("generator-driven property that takes 3 named args, which fails, with config params") {

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      forAll ("a", "b", "c", minSize(10), maxSize(20)) { (a: String, b: String, c: String) =>
        a.length + b.length + c.length should be < 0
      }
    }
  }

  test("generator-driven property that takes 3 args and generators, which succeeds, with config params") {

    forAll (famousLastWords, famousLastWords, famousLastWords, minSize(10), maxSize(20)) { (a: String, b: String, c: String) =>
      a.length + b.length + c.length should equal ((a + b + c).length)
    }
  }

  test("generator-driven property that takes 3 args and generators, which fails, with config params") {

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      forAll (famousLastWords, famousLastWords, famousLastWords, minSize(10), maxSize(20)) { (a: String, b: String, c: String) =>
        a.length + b.length + c.length should be < 0
      }
    }
  }

  test("generator-driven property that takes 3 named args and generators, which succeeds, with config params") {

    forAll ((famousLastWords, "a"), (famousLastWords, "b"), (famousLastWords, "c"), minSize(10), maxSize(20)) { (a: String, b: String, c: String) =>
      a.length + b.length + c.length should equal ((a + b + c).length)
    }
  }

  test("generator-driven property that takes 3 named args and generators, which fails, with config params") {

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      forAll ((famousLastWords, "a"), (famousLastWords, "b"), (famousLastWords, "c"), minSize(10), maxSize(20)) { (a: String, b: String, c: String) =>
        a.length + b.length + c.length should be < 0
      }
    }
  }

  // Same thing, but set minSuccessful to 5 with param, prop fails after 5
  test("generator-driven property that takes 3 args, which succeeds, with minSuccessful param set to 5") {

    var i = 0
    forAll (minSuccessful(5)) { (a: String, b: String, c: String) =>
      i += 1
      i should not be 6
    }
  }

  test("generator-driven property that takes 3 args, which fails, with minSuccessful param set to 5") {

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      var i = 0
      forAll (minSuccessful(5)) { (a: String, b: String, c: String) =>
        i += 1
        i should not be 5
      }
    }
  }

  test("generator-driven property that takes 3 named args, which succeeds, with minSuccessful param set to 5") {

    var i = 0
    forAll ("a", "b", "c", minSuccessful(5)) { (a: String, b: String, c: String) =>
      i += 1
      i should not be 6
    }
  }

  test("generator-driven property that takes 3 named args, which fails, with minSuccessful param set to 5") {

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      var i = 0
      forAll ("a", "b", "c", minSuccessful(5)) { (a: String, b: String, c: String) =>
        i += 1
        i should not be 5
      }
    }
  }

  test("generator-driven property that takes 3 args and generators, which succeeds, with minSuccessful param set to 5") {

    var i = 0
    forAll (famousLastWords, famousLastWords, famousLastWords, minSuccessful(5)) { (a: String, b: String, c: String) =>
      i += 1
      i should not be 6
    }
  }

  test("generator-driven property that takes 3 args and generators, which fails, with minSuccessful param set to 5") {

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      var i = 0
      forAll (famousLastWords, famousLastWords, famousLastWords, minSuccessful(5)) { (a: String, b: String, c: String) =>
        i += 1
        i should not be 5
      }
    }
  }

  test("generator-driven property that takes 3 named args and generators, which succeeds, with minSuccessful param set to 5") {

    var i = 0
    forAll ((famousLastWords, "a"), (famousLastWords, "b"), (famousLastWords, "c"), minSuccessful(5)) { (a: String, b: String, c: String) =>
      i += 1
      i should not be 6
    }
  }

  test("generator-driven property that takes 3 named args and generators, which fails, with minSuccessful param set to 5") {

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      var i = 0
      forAll ((famousLastWords, "a"), (famousLastWords, "b"), (famousLastWords, "c"), minSuccessful(5)) { (a: String, b: String, c: String) =>
        i += 1
        i should not be 5
      }
    }
  }

  // Same thing, but set default minSuccessful to 5, prop fails after 5
  test("generator-driven property that takes 3 args, which succeeds, with default minSuccessful param set to 5") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSuccessful = 5)

    var i = 0
    forAll { (a: String, b: String, c: String) =>
      i += 1
      i should not be 6
    }
  }

  test("generator-driven property that takes 3 args, which fails, with default minSuccessful param set to 5") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSuccessful = 5)

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      var i = 0
      forAll { (a: String, b: String, c: String) =>
        i += 1
        i should not be 5
      }
    }
  }

  test("generator-driven property that takes 3 named args, which succeeds, with default minSuccessful param set to 5") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSuccessful = 5)

    var i = 0
    forAll ("a", "b", "c") { (a: String, b: String, c: String) =>
      i += 1
      i should not be 6
    }
  }

  test("generator-driven property that takes 3 named args, which fails, with default minSuccessful param set to 5") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSuccessful = 5)

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      var i = 0
      forAll ("a", "b", "c") { (a: String, b: String, c: String) =>
        i += 1
        i should not be 5
      }
    }
  }

  test("generator-driven property that takes 3 args and generators, which succeeds, with default minSuccessful param set to 5") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSuccessful = 5)

    var i = 0
    forAll (famousLastWords, famousLastWords, famousLastWords) { (a: String, b: String, c: String) =>
      i += 1
      i should not be 6
    }
  }

  test("generator-driven property that takes 3 args and generators, which fails, with default minSuccessful param set to 5") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSuccessful = 5)

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      var i = 0
      forAll (famousLastWords, famousLastWords, famousLastWords) { (a: String, b: String, c: String) =>
        i += 1
        i should not be 5
      }
    }
  }

  test("generator-driven property that takes 3 named args and generators, which succeeds, with default minSuccessful param set to 5") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSuccessful = 5)

    var i = 0
    forAll ((famousLastWords, "a"), (famousLastWords, "b"), (famousLastWords, "c")) { (a: String, b: String, c: String) =>
      i += 1
      i should not be 6
    }
  }

  test("generator-driven property that takes 3 named args and generators, which fails, with default minSuccessful param set to 5") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSuccessful = 5)

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      var i = 0
      forAll ((famousLastWords, "a"), (famousLastWords, "b"), (famousLastWords, "c")) { (a: String, b: String, c: String) =>
        i += 1
        i should not be 5
      }
    }
  }

  // Same thing, but set maxDiscarded to 5 with param, prop fails after 5
  test("generator-driven property that takes 3 args, which succeeds, with maxDiscarded param set to 5") {

    var i = 0
    forAll (maxDiscarded(5)) { (a: String, b: String, c: String) =>
      i += 1
      whenever (i > 5) { 1 + 1 should equal (2) }
    }
  }

  test("generator-driven property that takes 3 args, which fails, with maxDiscarded param set to 5") {

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      var i = 0
      forAll (maxDiscarded(5)) { (a: String, b: String, c: String) =>
        i += 1
        whenever (i > 6) { 1 + 1 should equal (2) }
      }
    }
  }

  test("generator-driven property that takes 3 named args, which succeeds, with maxDiscarded param set to 5") {

    var i = 0
    forAll ("a", "b", "c", maxDiscarded(5)) { (a: String, b: String, c: String) =>
      i += 1
      whenever (i > 5) { 1 + 1 should equal (2) }
    }
  }

  test("generator-driven property that takes 3 named args, which fails, with maxDiscarded param set to 5") {

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      var i = 0
      forAll ("a", "b", "c", maxDiscarded(5)) { (a: String, b: String, c: String) =>
        i += 1
        whenever (i > 6) { 1 + 1 should equal (2) }
      }
    }
  }

  test("generator-driven property that takes 3 args and generators, which succeeds, with maxDiscarded param set to 5") {

    var i = 0
    forAll (famousLastWords, famousLastWords, famousLastWords, maxDiscarded(5)) { (a: String, b: String, c: String) =>
      i += 1
      whenever (i > 5) { 1 + 1 should equal (2) }
    }
  }

  test("generator-driven property that takes 3 args and generators, which fails, with maxDiscarded param set to 5") {

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      var i = 0
      forAll (famousLastWords, famousLastWords, famousLastWords, maxDiscarded(5)) { (a: String, b: String, c: String) =>
        i += 1
        whenever (i > 6) { 1 + 1 should equal (2) }
      }
    }
  }

  test("generator-driven property that takes 3 named args and generators, which succeeds, with maxDiscarded param set to 5") {

    var i = 0
    forAll ((famousLastWords, "a"), (famousLastWords, "b"), (famousLastWords, "c"), maxDiscarded(5)) { (a: String, b: String, c: String) =>
      i += 1
      whenever (i > 5) { 1 + 1 should equal (2) }
    }
  }

  test("generator-driven property that takes 3 named args and generators, which fails, with maxDiscarded param set to 5") {

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      var i = 0
      forAll ((famousLastWords, "a"), (famousLastWords, "b"), (famousLastWords, "c"), maxDiscarded(5)) { (a: String, b: String, c: String) =>
        i += 1
        whenever (i > 6) { 1 + 1 should equal (2) }
      }
    }
  }

  // Same thing, but set default maxDiscarded to 5, prop fails after 5
  test("generator-driven property that takes 3 args, which succeeds, with default maxDiscarded set to 5") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(maxDiscarded = 5)

    var i = 0
    forAll { (a: String, b: String, c: String) =>
      i += 1
      whenever (i > 5) { 1 + 1 should equal (2) }
    }
  }

  test("generator-driven property that takes 3 args, which fails, with default maxDiscarded set to 5") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(maxDiscarded = 5)

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      var i = 0
      forAll { (a: String, b: String, c: String) =>
        i += 1
        whenever (i > 6) { 1 + 1 should equal (2) }
      }
    }
  }

  test("generator-driven property that takes 3 named args, which succeeds, with default maxDiscarded set to 5") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(maxDiscarded = 5)

    var i = 0
    forAll ("a", "b", "c") { (a: String, b: String, c: String) =>
      i += 1
      whenever (i > 5) { 1 + 1 should equal (2) }
    }
  }

  test("generator-driven property that takes 3 named args, which fails, with default maxDiscarded set to 5") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(maxDiscarded = 5)

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      var i = 0
      forAll ("a", "b", "c") { (a: String, b: String, c: String) =>
        i += 1
        whenever (i > 6) { 1 + 1 should equal (2) }
      }
    }
  }

  test("generator-driven property that takes 3 args and generators, which succeeds, with default maxDiscarded set to 5") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(maxDiscarded = 5)

    var i = 0
    forAll (famousLastWords, famousLastWords, famousLastWords) { (a: String, b: String, c: String) =>
      i += 1
      whenever (i > 5) { 1 + 1 should equal (2) }
    }
  }

  test("generator-driven property that takes 3 args and generators, which fails, with default maxDiscarded set to 5") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(maxDiscarded = 5)

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      var i = 0
      forAll (famousLastWords, famousLastWords, famousLastWords) { (a: String, b: String, c: String) =>
        i += 1
        whenever (i > 6) { 1 + 1 should equal (2) }
      }
    }
  }

  test("generator-driven property that takes 3 named args and generators, which succeeds, with default maxDiscarded set to 5") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(maxDiscarded = 5)

    var i = 0
    forAll ((famousLastWords, "a"), (famousLastWords, "b"), (famousLastWords, "c")) { (a: String, b: String, c: String) =>
      i += 1
      whenever (i > 5) { 1 + 1 should equal (2) }
    }
  }

  test("generator-driven property that takes 3 named args and generators, which fails, with default maxDiscarded set to 5") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(maxDiscarded = 5)

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      var i = 0
      forAll ((famousLastWords, "a"), (famousLastWords, "b"), (famousLastWords, "c")) { (a: String, b: String, c: String) =>
        i += 1
        whenever (i > 6) { 1 + 1 should equal (2) }
      }
    }
  }

  // set minSize > maxSize with (param, param) (intercept IAE)
  test("generator-driven property that takes 3 args, which should throw IAE because maxSize > maxSize, specified as (param, param)") {

    intercept[IllegalArgumentException] {
      forAll (minSize(5), maxSize(4)) { (a: String, b: String, c: String) =>
        1 + 1 should equal (2)
      }
    }
  }

  test("generator-driven property that takes 3 named args, which should throw IAE because maxSize > maxSize, specified as (param, param)") {

    intercept[IllegalArgumentException] {
      forAll ("a", "b", "c", minSize(5), maxSize(4)) { (a: String, b: String, c: String) =>
        1 + 1 should equal (2)
      }
    }
  }

  test("generator-driven property that takes 3 args and generators, which should throw IAE because maxSize > maxSize, specified as (param, param)") {

    intercept[IllegalArgumentException] {
      forAll (famousLastWords, famousLastWords, famousLastWords, minSize(5), maxSize(4)) { (a: String, b: String, c: String) =>
        1 + 1 should equal (2)
      }
    }
  }

  test("generator-driven property that takes 3 named args and generators, which should throw IAE because maxSize > maxSize, specified as (param, param)") {

    intercept[IllegalArgumentException] {
      forAll ((famousLastWords, "a"), (famousLastWords, "b"), (famousLastWords, "c"), minSize(5), maxSize(4)) { (a: String, b: String, c: String) =>
        1 + 1 should equal (2)
      }
    }
  }

  // set minSize > maxSize with (param, default) (intercept IAE)
  test("generator-driven property that takes 3 args, which should throw IAE because maxSize > maxSize, specified as (param, default)") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(maxSize = 4)

    intercept[IllegalArgumentException] {
      forAll (minSize(5)) { (a: String, b: String, c: String) =>
        1 + 1 should equal (2)
      }
    }
  }

  test("generator-driven property that takes 3 named args, which should throw IAE because maxSize > maxSize, specified as (param, default)") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(maxSize = 4)

    intercept[IllegalArgumentException] {
      forAll ("a", "b", "c", minSize(5)) { (a: String, b: String, c: String) =>
        1 + 1 should equal (2)
      }
    }
  }

  test("generator-driven property that takes 3 args and generators, which should throw IAE because maxSize > maxSize, specified as (param, default)") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(maxSize = 4)

    intercept[IllegalArgumentException] {
      forAll (famousLastWords, famousLastWords, famousLastWords, minSize(5)) { (a: String, b: String, c: String) =>
        1 + 1 should equal (2)
      }
    }
  }

  test("generator-driven property that takes 3 named args and generators, which should throw IAE because maxSize > maxSize, specified as (param, default)") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(maxSize = 4)

    intercept[IllegalArgumentException] {
      forAll ((famousLastWords, "a"), (famousLastWords, "b"), (famousLastWords, "c"), minSize(5)) { (a: String, b: String, c: String) =>
        1 + 1 should equal (2)
      }
    }
  }

  // set minSize > maxSize with (default, param) (intercept IAE)
  test("generator-driven property that takes 3 args, which should throw IAE because maxSize > maxSize, specified as (default, param)") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSize = 5)

    intercept[IllegalArgumentException] {
      forAll (maxSize(4)) { (a: String, b: String, c: String) =>
        1 + 1 should equal (2)
      }
    }
  }

  test("generator-driven property that takes 3 named args, which should throw IAE because maxSize > maxSize, specified as (default, param)") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSize = 5)

    intercept[IllegalArgumentException] {
      forAll ("a", "b", "c", maxSize(4)) { (a: String, b: String, c: String) =>
        1 + 1 should equal (2)
      }
    }
  }

  test("generator-driven property that takes 3 args and generators, which should throw IAE because maxSize > maxSize, specified as (default, param)") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSize = 5)

    intercept[IllegalArgumentException] {
      forAll (famousLastWords, famousLastWords, famousLastWords, maxSize(4)) { (a: String, b: String, c: String) =>
        1 + 1 should equal (2)
      }
    }
  }

  test("generator-driven property that takes 3 named args and generators, which should throw IAE because maxSize > maxSize, specified as (default, param)") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSize = 5)

    intercept[IllegalArgumentException] {
      forAll ((famousLastWords, "a"), (famousLastWords, "b"), (famousLastWords, "c"), maxSize(4)) { (a: String, b: String, c: String) =>
        1 + 1 should equal (2)
      }
    }
  }

  // set maxSize with param (ensure always passed with a size less than maxSize)
  test("generator-driven property that takes 3 args, with maxSize specified as param") {

    forAll (maxSize(5)) { (a: String, b: String, c: String) =>
      a.length should be <= 5
      b.length should be <= 5
      c.length should be <= 5
    }
  }

  test("generator-driven property that takes 3 named args, with maxSize specified as param") {

    forAll ("a", "b", "c", maxSize(5)) { (a: String, b: String, c: String) =>
      a.length should be <= 5
      b.length should be <= 5
      c.length should be <= 5
    }
  }

  // set maxSize with default (ensure always passed with a size less than maxSize)
  test("generator-driven property that takes 3 args, with maxSize specified as default") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(maxSize = 5)

    forAll { (a: String, b: String, c: String) =>
      a.length should be <= 5
      b.length should be <= 5
      c.length should be <= 5
    }
  }

  test("generator-driven property that takes 3 named args, with maxSize specified as default") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(maxSize = 5)

    forAll ("a", "b", "c") { (a: String, b: String, c: String) =>
      a.length should be <= 5
      b.length should be <= 5
      c.length should be <= 5
    }
  }
 
  // set minSize == maxSize with (param, param) (ensure always passed with that size)
  test("generator-driven property that takes 3 args and generators, with minSize == maxSize, specified as (param, param)") {

    pendingUntilFixed {
      forAll (fiveFive, fiveFive, fiveFive, minSize(5), maxSize(5)) { (a: String, b: String, c: String) =>
        a should be ("OKAY")
        b should be ("OKAY")
        c should be ("OKAY")
      }
    }
  }

  test("generator-driven property that takes 3 named args and generators, with minSize == maxSize, specified as (param, param)") {

    pendingUntilFixed {
      forAll ((fiveFive, "a"), (fiveFive, "b"), (fiveFive, "c"), minSize(5), maxSize(5)) { (a: String, b: String, c: String) =>
        a should be ("OKAY")
        b should be ("OKAY")
        c should be ("OKAY")
      }
    }
  }

  // set minSize == maxSize with (param, default) (ensure always passed with that size)
  test("generator-driven property that takes 3 args and generators, with minSize == maxSize, specified as (param, default)") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(maxSize = 5)

    pendingUntilFixed {
      forAll (fiveFive, fiveFive, fiveFive, minSize(5)) { (a: String, b: String, c: String) =>
        a should be ("OKAY")
        b should be ("OKAY")
        c should be ("OKAY")
      }
    }
  }

  test("generator-driven property that takes 3 named args and generators, with minSize == maxSize, specified as (param, default)") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(maxSize = 5)

    pendingUntilFixed {
      forAll ((fiveFive, "a"), (fiveFive, "b"), (fiveFive, "c"), minSize(5)) { (a: String, b: String, c: String) =>
        a should be ("OKAY")
        b should be ("OKAY")
        c should be ("OKAY")
      }
    }
  }

  // set minSize == maxSize with (default, param) (ensure always passed with that size)
  test("generator-driven property that takes 3 args and generators, with minSize == maxSize, specified as (default, param)") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSize = 5)

    pendingUntilFixed {
      forAll (fiveFive, fiveFive, fiveFive, maxSize(5)) { (a: String, b: String, c: String) =>
        a should be ("OKAY")
        b should be ("OKAY")
        c should be ("OKAY")
      }
    }
  }

  test("generator-driven property that takes 3 named args and generators, with minSize == maxSize, specified as (default, param)") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSize = 5)

    pendingUntilFixed {
      forAll ((fiveFive, "a"), (fiveFive, "b"), (fiveFive, "c"), maxSize(5)) { (a: String, b: String, c: String) =>
        a should be ("OKAY")
        b should be ("OKAY")
        c should be ("OKAY")
      }
    }
  }

  // set minSize == maxSize with (default, default) (ensure always passed with that size)
  test("generator-driven property that takes 3 args and generators, with minSize == maxSize, specified as (default, default)") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSize = 5, maxSize = 5)

    pendingUntilFixed {
      forAll (fiveFive, fiveFive, fiveFive) { (a: String, b: String, c: String) =>
        a should be ("OKAY")
        b should be ("OKAY")
        c should be ("OKAY")
      }
    }
  }

  test("generator-driven property that takes 3 named args and generators, with minSize == maxSize, specified as (default, default)") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSize = 5, maxSize = 5)

    pendingUntilFixed {
      forAll ((fiveFive, "a"), (fiveFive, "b"), (fiveFive, "c")) { (a: String, b: String, c: String) =>
        a should be ("OKAY")
        b should be ("OKAY")
        c should be ("OKAY")
      }
    }
  }

  // set minSize to 7 and maxSize to 11 with (param, param) (ensure always passed with that size)
  test("generator-driven property that takes 3 args and generators, with minSize to 7 and maxSize to 11, specified as (param, param)") {

    pendingUntilFixed {
      forAll (sevenEleven, sevenEleven, sevenEleven, minSize(7), maxSize(11)) { (a: String, b: String, c: String) =>
        a should be ("OKAY")
        b should be ("OKAY")
        c should be ("OKAY")
      }
    }
  }

  test("generator-driven property that takes 3 named args and generators, with minSize to 7 and maxSize to 11, specified as (param, param)") {

    pendingUntilFixed {
      forAll ((sevenEleven, "a"), (sevenEleven, "b"), (sevenEleven, "c"), minSize(7), maxSize(11)) { (a: String, b: String, c: String) =>
        a should be ("OKAY")
        b should be ("OKAY")
        c should be ("OKAY")
      }
    }
  }

  // set minSize to 7 and maxSize to 11 with (param, default) (ensure always passed with that size)
  test("generator-driven property that takes 3 args and generators, with minSize to 7 and maxSize to 11, specified as (param, default)") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(maxSize = 11)

    pendingUntilFixed {
      forAll (sevenEleven, sevenEleven, sevenEleven, minSize(7)) { (a: String, b: String, c: String) =>
        a should be ("OKAY")
        b should be ("OKAY")
        c should be ("OKAY")
      }
    }
  }

  test("generator-driven property that takes 3 named args and generators, with minSize to 7 and maxSize to 11, specified as (param, default)") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(maxSize = 11)

    pendingUntilFixed {
      forAll ((sevenEleven, "a"), (sevenEleven, "b"), (sevenEleven, "c"), minSize(7)) { (a: String, b: String, c: String) =>
        a should be ("OKAY")
        b should be ("OKAY")
        c should be ("OKAY")
      }
    }
  }

  // set minSize to 7 and maxSize to 11 with (default, param) (ensure always passed with that size)
  test("generator-driven property that takes 3 args and generators, with minSize to 7 and maxSize to 11, specified as (default, param)") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSize = 7)

    pendingUntilFixed {
      forAll (sevenEleven, sevenEleven, sevenEleven, maxSize(11)) { (a: String, b: String, c: String) =>
        a should be ("OKAY")
        b should be ("OKAY")
        c should be ("OKAY")
      }
    }
  }

  test("generator-driven property that takes 3 named args and generators, with minSize to 7 and maxSize to 11, specified as (default, param)") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSize = 7)

    pendingUntilFixed {
      forAll ((sevenEleven, "a"), (sevenEleven, "b"), (sevenEleven, "c"), maxSize(11)) { (a: String, b: String, c: String) =>
        a should be ("OKAY")
        b should be ("OKAY")
        c should be ("OKAY")
      }
    }
  }

  // set minSize to 7 and maxSize to 11 with (default, default) (ensure always passed with that size)
  test("generator-driven property that takes 3 args and generators, with minSize to 7 and maxSize to 11, specified as (default, default)") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSize = 7, maxSize = 11)

    pendingUntilFixed {
      forAll (sevenEleven, sevenEleven, sevenEleven) { (a: String, b: String, c: String) =>
        a should be ("OKAY")
        b should be ("OKAY")
        c should be ("OKAY")
      }
    }
  }

  test("generator-driven property that takes 3 named args and generators, with minSize to 7 and maxSize to 11, specified as (default, default)") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSize = 7, maxSize = 11)

    pendingUntilFixed {
      forAll ((sevenEleven, "a"), (sevenEleven, "b"), (sevenEleven, "c")) { (a: String, b: String, c: String) =>
        a should be ("OKAY")
        b should be ("OKAY")
        c should be ("OKAY")
      }
    }
  }


  test("generator-driven property that takes 4 args, which succeeds") {

    forAll { (a: String, b: String, c: String, d: String) =>
      a.length + b.length + c.length + d.length should equal ((a + b + c + d).length)
    }
  }

  test("generator-driven property that takes 4 args, which fails") {

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      forAll { (a: String, b: String, c: String, d: String) =>
        a.length + b.length + c.length + d.length should be < 0
      }
    }
  }

  test("generator-driven property that takes 4 named args, which succeeds") {

    forAll ("a", "b", "c", "d") { (a: String, b: String, c: String, d: String) =>
      a.length + b.length + c.length + d.length should equal ((a + b + c + d).length)
    }
  }

  test("generator-driven property that takes 4 named args, which fails") {

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      forAll ("a", "b", "c", "d") { (a: String, b: String, c: String, d: String) =>
        a.length + b.length + c.length + d.length should be < 0
      }
    }
  }

  test("generator-driven property that takes 4 args and generators, which succeeds") {

    forAll (famousLastWords, famousLastWords, famousLastWords, famousLastWords) { (a: String, b: String, c: String, d: String) =>
      a.length + b.length + c.length + d.length should equal ((a + b + c + d).length)
    }
  }

  test("generator-driven property that takes 4 args and generators, which fails") {

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      forAll (famousLastWords, famousLastWords, famousLastWords, famousLastWords) { (a: String, b: String, c: String, d: String) =>
        a.length + b.length + c.length + d.length should be < 0
      }
    }
  }

  test("generator-driven property that takes 4 named args and generators, which succeeds") {

    forAll ((famousLastWords, "a"), (famousLastWords, "b"), (famousLastWords, "c"), (famousLastWords, "d")) { (a: String, b: String, c: String, d: String) =>
      a.length + b.length + c.length + d.length should equal ((a + b + c + d).length)
    }
  }

  test("generator-driven property that takes 4 named args and generators, which fails") {

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      forAll ((famousLastWords, "a"), (famousLastWords, "b"), (famousLastWords, "c"), (famousLastWords, "d")) { (a: String, b: String, c: String, d: String) =>
        a.length + b.length + c.length + d.length should be < 0
      }
    }
  }

  // Same thing, but with config params
  test("generator-driven property that takes 4 args, which succeeds, with config params") {

    forAll (minSize(10), maxSize(20)) { (a: String, b: String, c: String, d: String) =>
      a.length + b.length + c.length + d.length should equal ((a + b + c + d).length)
    }
  }

  test("generator-driven property that takes 4 args, which fails, with config params") {

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      forAll (minSize(10), maxSize(20)) { (a: String, b: String, c: String, d: String) =>
        a.length + b.length + c.length + d.length should be < 0
      }
    }
  }

  test("generator-driven property that takes 4 named args, which succeeds, with config params") {

    forAll ("a", "b", "c", "d", minSize(10), maxSize(20)) { (a: String, b: String, c: String, d: String) =>
      a.length + b.length + c.length + d.length should equal ((a + b + c + d).length)
    }
  }

  test("generator-driven property that takes 4 named args, which fails, with config params") {

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      forAll ("a", "b", "c", "d", minSize(10), maxSize(20)) { (a: String, b: String, c: String, d: String) =>
        a.length + b.length + c.length + d.length should be < 0
      }
    }
  }

  test("generator-driven property that takes 4 args and generators, which succeeds, with config params") {

    forAll (famousLastWords, famousLastWords, famousLastWords, famousLastWords, minSize(10), maxSize(20)) { (a: String, b: String, c: String, d: String) =>
      a.length + b.length + c.length + d.length should equal ((a + b + c + d).length)
    }
  }

  test("generator-driven property that takes 4 args and generators, which fails, with config params") {

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      forAll (famousLastWords, famousLastWords, famousLastWords, famousLastWords, minSize(10), maxSize(20)) { (a: String, b: String, c: String, d: String) =>
        a.length + b.length + c.length + d.length should be < 0
      }
    }
  }

  test("generator-driven property that takes 4 named args and generators, which succeeds, with config params") {

    forAll ((famousLastWords, "a"), (famousLastWords, "b"), (famousLastWords, "c"), (famousLastWords, "d"), minSize(10), maxSize(20)) { (a: String, b: String, c: String, d: String) =>
      a.length + b.length + c.length + d.length should equal ((a + b + c + d).length)
    }
  }

  test("generator-driven property that takes 4 named args and generators, which fails, with config params") {

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      forAll ((famousLastWords, "a"), (famousLastWords, "b"), (famousLastWords, "c"), (famousLastWords, "d"), minSize(10), maxSize(20)) { (a: String, b: String, c: String, d: String) =>
        a.length + b.length + c.length + d.length should be < 0
      }
    }
  }

  // Same thing, but set minSuccessful to 5 with param, prop fails after 5
  test("generator-driven property that takes 4 args, which succeeds, with minSuccessful param set to 5") {

    var i = 0
    forAll (minSuccessful(5)) { (a: String, b: String, c: String, d: String) =>
      i += 1
      i should not be 6
    }
  }

  test("generator-driven property that takes 4 args, which fails, with minSuccessful param set to 5") {

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      var i = 0
      forAll (minSuccessful(5)) { (a: String, b: String, c: String, d: String) =>
        i += 1
        i should not be 5
      }
    }
  }

  test("generator-driven property that takes 4 named args, which succeeds, with minSuccessful param set to 5") {

    var i = 0
    forAll ("a", "b", "c", "d", minSuccessful(5)) { (a: String, b: String, c: String, d: String) =>
      i += 1
      i should not be 6
    }
  }

  test("generator-driven property that takes 4 named args, which fails, with minSuccessful param set to 5") {

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      var i = 0
      forAll ("a", "b", "c", "d", minSuccessful(5)) { (a: String, b: String, c: String, d: String) =>
        i += 1
        i should not be 5
      }
    }
  }

  test("generator-driven property that takes 4 args and generators, which succeeds, with minSuccessful param set to 5") {

    var i = 0
    forAll (famousLastWords, famousLastWords, famousLastWords, famousLastWords, minSuccessful(5)) { (a: String, b: String, c: String, d: String) =>
      i += 1
      i should not be 6
    }
  }

  test("generator-driven property that takes 4 args and generators, which fails, with minSuccessful param set to 5") {

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      var i = 0
      forAll (famousLastWords, famousLastWords, famousLastWords, famousLastWords, minSuccessful(5)) { (a: String, b: String, c: String, d: String) =>
        i += 1
        i should not be 5
      }
    }
  }

  test("generator-driven property that takes 4 named args and generators, which succeeds, with minSuccessful param set to 5") {

    var i = 0
    forAll ((famousLastWords, "a"), (famousLastWords, "b"), (famousLastWords, "c"), (famousLastWords, "d"), minSuccessful(5)) { (a: String, b: String, c: String, d: String) =>
      i += 1
      i should not be 6
    }
  }

  test("generator-driven property that takes 4 named args and generators, which fails, with minSuccessful param set to 5") {

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      var i = 0
      forAll ((famousLastWords, "a"), (famousLastWords, "b"), (famousLastWords, "c"), (famousLastWords, "d"), minSuccessful(5)) { (a: String, b: String, c: String, d: String) =>
        i += 1
        i should not be 5
      }
    }
  }

  // Same thing, but set default minSuccessful to 5, prop fails after 5
  test("generator-driven property that takes 4 args, which succeeds, with default minSuccessful param set to 5") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSuccessful = 5)

    var i = 0
    forAll { (a: String, b: String, c: String, d: String) =>
      i += 1
      i should not be 6
    }
  }

  test("generator-driven property that takes 4 args, which fails, with default minSuccessful param set to 5") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSuccessful = 5)

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      var i = 0
      forAll { (a: String, b: String, c: String, d: String) =>
        i += 1
        i should not be 5
      }
    }
  }

  test("generator-driven property that takes 4 named args, which succeeds, with default minSuccessful param set to 5") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSuccessful = 5)

    var i = 0
    forAll ("a", "b", "c", "d") { (a: String, b: String, c: String, d: String) =>
      i += 1
      i should not be 6
    }
  }

  test("generator-driven property that takes 4 named args, which fails, with default minSuccessful param set to 5") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSuccessful = 5)

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      var i = 0
      forAll ("a", "b", "c", "d") { (a: String, b: String, c: String, d: String) =>
        i += 1
        i should not be 5
      }
    }
  }

  test("generator-driven property that takes 4 args and generators, which succeeds, with default minSuccessful param set to 5") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSuccessful = 5)

    var i = 0
    forAll (famousLastWords, famousLastWords, famousLastWords, famousLastWords) { (a: String, b: String, c: String, d: String) =>
      i += 1
      i should not be 6
    }
  }

  test("generator-driven property that takes 4 args and generators, which fails, with default minSuccessful param set to 5") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSuccessful = 5)

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      var i = 0
      forAll (famousLastWords, famousLastWords, famousLastWords, famousLastWords) { (a: String, b: String, c: String, d: String) =>
        i += 1
        i should not be 5
      }
    }
  }

  test("generator-driven property that takes 4 named args and generators, which succeeds, with default minSuccessful param set to 5") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSuccessful = 5)

    var i = 0
    forAll ((famousLastWords, "a"), (famousLastWords, "b"), (famousLastWords, "c"), (famousLastWords, "d")) { (a: String, b: String, c: String, d: String) =>
      i += 1
      i should not be 6
    }
  }

  test("generator-driven property that takes 4 named args and generators, which fails, with default minSuccessful param set to 5") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSuccessful = 5)

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      var i = 0
      forAll ((famousLastWords, "a"), (famousLastWords, "b"), (famousLastWords, "c"), (famousLastWords, "d")) { (a: String, b: String, c: String, d: String) =>
        i += 1
        i should not be 5
      }
    }
  }

  // Same thing, but set maxDiscarded to 5 with param, prop fails after 5
  test("generator-driven property that takes 4 args, which succeeds, with maxDiscarded param set to 5") {

    var i = 0
    forAll (maxDiscarded(5)) { (a: String, b: String, c: String, d: String) =>
      i += 1
      whenever (i > 5) { 1 + 1 should equal (2) }
    }
  }

  test("generator-driven property that takes 4 args, which fails, with maxDiscarded param set to 5") {

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      var i = 0
      forAll (maxDiscarded(5)) { (a: String, b: String, c: String, d: String) =>
        i += 1
        whenever (i > 6) { 1 + 1 should equal (2) }
      }
    }
  }

  test("generator-driven property that takes 4 named args, which succeeds, with maxDiscarded param set to 5") {

    var i = 0
    forAll ("a", "b", "c", "d", maxDiscarded(5)) { (a: String, b: String, c: String, d: String) =>
      i += 1
      whenever (i > 5) { 1 + 1 should equal (2) }
    }
  }

  test("generator-driven property that takes 4 named args, which fails, with maxDiscarded param set to 5") {

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      var i = 0
      forAll ("a", "b", "c", "d", maxDiscarded(5)) { (a: String, b: String, c: String, d: String) =>
        i += 1
        whenever (i > 6) { 1 + 1 should equal (2) }
      }
    }
  }

  test("generator-driven property that takes 4 args and generators, which succeeds, with maxDiscarded param set to 5") {

    var i = 0
    forAll (famousLastWords, famousLastWords, famousLastWords, famousLastWords, maxDiscarded(5)) { (a: String, b: String, c: String, d: String) =>
      i += 1
      whenever (i > 5) { 1 + 1 should equal (2) }
    }
  }

  test("generator-driven property that takes 4 args and generators, which fails, with maxDiscarded param set to 5") {

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      var i = 0
      forAll (famousLastWords, famousLastWords, famousLastWords, famousLastWords, maxDiscarded(5)) { (a: String, b: String, c: String, d: String) =>
        i += 1
        whenever (i > 6) { 1 + 1 should equal (2) }
      }
    }
  }

  test("generator-driven property that takes 4 named args and generators, which succeeds, with maxDiscarded param set to 5") {

    var i = 0
    forAll ((famousLastWords, "a"), (famousLastWords, "b"), (famousLastWords, "c"), (famousLastWords, "d"), maxDiscarded(5)) { (a: String, b: String, c: String, d: String) =>
      i += 1
      whenever (i > 5) { 1 + 1 should equal (2) }
    }
  }

  test("generator-driven property that takes 4 named args and generators, which fails, with maxDiscarded param set to 5") {

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      var i = 0
      forAll ((famousLastWords, "a"), (famousLastWords, "b"), (famousLastWords, "c"), (famousLastWords, "d"), maxDiscarded(5)) { (a: String, b: String, c: String, d: String) =>
        i += 1
        whenever (i > 6) { 1 + 1 should equal (2) }
      }
    }
  }

  // Same thing, but set default maxDiscarded to 5, prop fails after 5
  test("generator-driven property that takes 4 args, which succeeds, with default maxDiscarded set to 5") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(maxDiscarded = 5)

    var i = 0
    forAll { (a: String, b: String, c: String, d: String) =>
      i += 1
      whenever (i > 5) { 1 + 1 should equal (2) }
    }
  }

  test("generator-driven property that takes 4 args, which fails, with default maxDiscarded set to 5") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(maxDiscarded = 5)

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      var i = 0
      forAll { (a: String, b: String, c: String, d: String) =>
        i += 1
        whenever (i > 6) { 1 + 1 should equal (2) }
      }
    }
  }

  test("generator-driven property that takes 4 named args, which succeeds, with default maxDiscarded set to 5") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(maxDiscarded = 5)

    var i = 0
    forAll ("a", "b", "c", "d") { (a: String, b: String, c: String, d: String) =>
      i += 1
      whenever (i > 5) { 1 + 1 should equal (2) }
    }
  }

  test("generator-driven property that takes 4 named args, which fails, with default maxDiscarded set to 5") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(maxDiscarded = 5)

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      var i = 0
      forAll ("a", "b", "c", "d") { (a: String, b: String, c: String, d: String) =>
        i += 1
        whenever (i > 6) { 1 + 1 should equal (2) }
      }
    }
  }

  test("generator-driven property that takes 4 args and generators, which succeeds, with default maxDiscarded set to 5") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(maxDiscarded = 5)

    var i = 0
    forAll (famousLastWords, famousLastWords, famousLastWords, famousLastWords) { (a: String, b: String, c: String, d: String) =>
      i += 1
      whenever (i > 5) { 1 + 1 should equal (2) }
    }
  }

  test("generator-driven property that takes 4 args and generators, which fails, with default maxDiscarded set to 5") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(maxDiscarded = 5)

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      var i = 0
      forAll (famousLastWords, famousLastWords, famousLastWords, famousLastWords) { (a: String, b: String, c: String, d: String) =>
        i += 1
        whenever (i > 6) { 1 + 1 should equal (2) }
      }
    }
  }

  test("generator-driven property that takes 4 named args and generators, which succeeds, with default maxDiscarded set to 5") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(maxDiscarded = 5)

    var i = 0
    forAll ((famousLastWords, "a"), (famousLastWords, "b"), (famousLastWords, "c"), (famousLastWords, "d")) { (a: String, b: String, c: String, d: String) =>
      i += 1
      whenever (i > 5) { 1 + 1 should equal (2) }
    }
  }

  test("generator-driven property that takes 4 named args and generators, which fails, with default maxDiscarded set to 5") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(maxDiscarded = 5)

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      var i = 0
      forAll ((famousLastWords, "a"), (famousLastWords, "b"), (famousLastWords, "c"), (famousLastWords, "d")) { (a: String, b: String, c: String, d: String) =>
        i += 1
        whenever (i > 6) { 1 + 1 should equal (2) }
      }
    }
  }

  // set minSize > maxSize with (param, param) (intercept IAE)
  test("generator-driven property that takes 4 args, which should throw IAE because maxSize > maxSize, specified as (param, param)") {

    intercept[IllegalArgumentException] {
      forAll (minSize(5), maxSize(4)) { (a: String, b: String, c: String, d: String) =>
        1 + 1 should equal (2)
      }
    }
  }

  test("generator-driven property that takes 4 named args, which should throw IAE because maxSize > maxSize, specified as (param, param)") {

    intercept[IllegalArgumentException] {
      forAll ("a", "b", "c", "d", minSize(5), maxSize(4)) { (a: String, b: String, c: String, d: String) =>
        1 + 1 should equal (2)
      }
    }
  }

  test("generator-driven property that takes 4 args and generators, which should throw IAE because maxSize > maxSize, specified as (param, param)") {

    intercept[IllegalArgumentException] {
      forAll (famousLastWords, famousLastWords, famousLastWords, famousLastWords, minSize(5), maxSize(4)) { (a: String, b: String, c: String, d: String) =>
        1 + 1 should equal (2)
      }
    }
  }

  test("generator-driven property that takes 4 named args and generators, which should throw IAE because maxSize > maxSize, specified as (param, param)") {

    intercept[IllegalArgumentException] {
      forAll ((famousLastWords, "a"), (famousLastWords, "b"), (famousLastWords, "c"), (famousLastWords, "d"), minSize(5), maxSize(4)) { (a: String, b: String, c: String, d: String) =>
        1 + 1 should equal (2)
      }
    }
  }

  // set minSize > maxSize with (param, default) (intercept IAE)
  test("generator-driven property that takes 4 args, which should throw IAE because maxSize > maxSize, specified as (param, default)") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(maxSize = 4)

    intercept[IllegalArgumentException] {
      forAll (minSize(5)) { (a: String, b: String, c: String, d: String) =>
        1 + 1 should equal (2)
      }
    }
  }

  test("generator-driven property that takes 4 named args, which should throw IAE because maxSize > maxSize, specified as (param, default)") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(maxSize = 4)

    intercept[IllegalArgumentException] {
      forAll ("a", "b", "c", "d", minSize(5)) { (a: String, b: String, c: String, d: String) =>
        1 + 1 should equal (2)
      }
    }
  }

  test("generator-driven property that takes 4 args and generators, which should throw IAE because maxSize > maxSize, specified as (param, default)") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(maxSize = 4)

    intercept[IllegalArgumentException] {
      forAll (famousLastWords, famousLastWords, famousLastWords, famousLastWords, minSize(5)) { (a: String, b: String, c: String, d: String) =>
        1 + 1 should equal (2)
      }
    }
  }

  test("generator-driven property that takes 4 named args and generators, which should throw IAE because maxSize > maxSize, specified as (param, default)") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(maxSize = 4)

    intercept[IllegalArgumentException] {
      forAll ((famousLastWords, "a"), (famousLastWords, "b"), (famousLastWords, "c"), (famousLastWords, "d"), minSize(5)) { (a: String, b: String, c: String, d: String) =>
        1 + 1 should equal (2)
      }
    }
  }

  // set minSize > maxSize with (default, param) (intercept IAE)
  test("generator-driven property that takes 4 args, which should throw IAE because maxSize > maxSize, specified as (default, param)") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSize = 5)

    intercept[IllegalArgumentException] {
      forAll (maxSize(4)) { (a: String, b: String, c: String, d: String) =>
        1 + 1 should equal (2)
      }
    }
  }

  test("generator-driven property that takes 4 named args, which should throw IAE because maxSize > maxSize, specified as (default, param)") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSize = 5)

    intercept[IllegalArgumentException] {
      forAll ("a", "b", "c", "d", maxSize(4)) { (a: String, b: String, c: String, d: String) =>
        1 + 1 should equal (2)
      }
    }
  }

  test("generator-driven property that takes 4 args and generators, which should throw IAE because maxSize > maxSize, specified as (default, param)") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSize = 5)

    intercept[IllegalArgumentException] {
      forAll (famousLastWords, famousLastWords, famousLastWords, famousLastWords, maxSize(4)) { (a: String, b: String, c: String, d: String) =>
        1 + 1 should equal (2)
      }
    }
  }

  test("generator-driven property that takes 4 named args and generators, which should throw IAE because maxSize > maxSize, specified as (default, param)") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSize = 5)

    intercept[IllegalArgumentException] {
      forAll ((famousLastWords, "a"), (famousLastWords, "b"), (famousLastWords, "c"), (famousLastWords, "d"), maxSize(4)) { (a: String, b: String, c: String, d: String) =>
        1 + 1 should equal (2)
      }
    }
  }

  // set maxSize with param (ensure always passed with a size less than maxSize)
  test("generator-driven property that takes 4 args, with maxSize specified as param") {

    forAll (maxSize(5)) { (a: String, b: String, c: String, d: String) =>
      a.length should be <= 5
      b.length should be <= 5
      c.length should be <= 5
      d.length should be <= 5
    }
  }

  test("generator-driven property that takes 4 named args, with maxSize specified as param") {

    forAll ("a", "b", "c", "d", maxSize(5)) { (a: String, b: String, c: String, d: String) =>
      a.length should be <= 5
      b.length should be <= 5
      c.length should be <= 5
      d.length should be <= 5
    }
  }

  // set maxSize with default (ensure always passed with a size less than maxSize)
  test("generator-driven property that takes 4 args, with maxSize specified as default") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(maxSize = 5)

    forAll { (a: String, b: String, c: String, d: String) =>
      a.length should be <= 5
      b.length should be <= 5
      c.length should be <= 5
      d.length should be <= 5
    }
  }

  test("generator-driven property that takes 4 named args, with maxSize specified as default") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(maxSize = 5)

    forAll ("a", "b", "c", "d") { (a: String, b: String, c: String, d: String) =>
      a.length should be <= 5
      b.length should be <= 5
      c.length should be <= 5
      d.length should be <= 5
    }
  }
 
  // set minSize == maxSize with (param, param) (ensure always passed with that size)
  test("generator-driven property that takes 4 args and generators, with minSize == maxSize, specified as (param, param)") {

    pendingUntilFixed {
      forAll (fiveFive, fiveFive, fiveFive, fiveFive, minSize(5), maxSize(5)) { (a: String, b: String, c: String, d: String) =>
        a should be ("OKAY")
        b should be ("OKAY")
        c should be ("OKAY")
        d should be ("OKAY")
      }
    }
  }

  test("generator-driven property that takes 4 named args and generators, with minSize == maxSize, specified as (param, param)") {

    pendingUntilFixed {
      forAll ((fiveFive, "a"), (fiveFive, "b"), (fiveFive, "c"), (fiveFive, "d"), minSize(5), maxSize(5)) { (a: String, b: String, c: String, d: String) =>
        a should be ("OKAY")
        b should be ("OKAY")
        c should be ("OKAY")
        d should be ("OKAY")
      }
    }
  }

  // set minSize == maxSize with (param, default) (ensure always passed with that size)
  test("generator-driven property that takes 4 args and generators, with minSize == maxSize, specified as (param, default)") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(maxSize = 5)

    pendingUntilFixed {
      forAll (fiveFive, fiveFive, fiveFive, fiveFive, minSize(5)) { (a: String, b: String, c: String, d: String) =>
        a should be ("OKAY")
        b should be ("OKAY")
        c should be ("OKAY")
        d should be ("OKAY")
      }
    }
  }

  test("generator-driven property that takes 4 named args and generators, with minSize == maxSize, specified as (param, default)") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(maxSize = 5)

    pendingUntilFixed {
      forAll ((fiveFive, "a"), (fiveFive, "b"), (fiveFive, "c"), (fiveFive, "d"), minSize(5)) { (a: String, b: String, c: String, d: String) =>
        a should be ("OKAY")
        b should be ("OKAY")
        c should be ("OKAY")
        d should be ("OKAY")
      }
    }
  }

  // set minSize == maxSize with (default, param) (ensure always passed with that size)
  test("generator-driven property that takes 4 args and generators, with minSize == maxSize, specified as (default, param)") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSize = 5)

    pendingUntilFixed {
      forAll (fiveFive, fiveFive, fiveFive, fiveFive, maxSize(5)) { (a: String, b: String, c: String, d: String) =>
        a should be ("OKAY")
        b should be ("OKAY")
        c should be ("OKAY")
        d should be ("OKAY")
      }
    }
  }

  test("generator-driven property that takes 4 named args and generators, with minSize == maxSize, specified as (default, param)") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSize = 5)

    pendingUntilFixed {
      forAll ((fiveFive, "a"), (fiveFive, "b"), (fiveFive, "c"), (fiveFive, "d"), maxSize(5)) { (a: String, b: String, c: String, d: String) =>
        a should be ("OKAY")
        b should be ("OKAY")
        c should be ("OKAY")
        d should be ("OKAY")
      }
    }
  }

  // set minSize == maxSize with (default, default) (ensure always passed with that size)
  test("generator-driven property that takes 4 args and generators, with minSize == maxSize, specified as (default, default)") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSize = 5, maxSize = 5)

    pendingUntilFixed {
      forAll (fiveFive, fiveFive, fiveFive, fiveFive) { (a: String, b: String, c: String, d: String) =>
        a should be ("OKAY")
        b should be ("OKAY")
        c should be ("OKAY")
        d should be ("OKAY")
      }
    }
  }

  test("generator-driven property that takes 4 named args and generators, with minSize == maxSize, specified as (default, default)") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSize = 5, maxSize = 5)

    pendingUntilFixed {
      forAll ((fiveFive, "a"), (fiveFive, "b"), (fiveFive, "c"), (fiveFive, "d")) { (a: String, b: String, c: String, d: String) =>
        a should be ("OKAY")
        b should be ("OKAY")
        c should be ("OKAY")
        d should be ("OKAY")
      }
    }
  }

  // set minSize to 7 and maxSize to 11 with (param, param) (ensure always passed with that size)
  test("generator-driven property that takes 4 args and generators, with minSize to 7 and maxSize to 11, specified as (param, param)") {

    pendingUntilFixed {
      forAll (sevenEleven, sevenEleven, sevenEleven, sevenEleven, minSize(7), maxSize(11)) { (a: String, b: String, c: String, d: String) =>
        a should be ("OKAY")
        b should be ("OKAY")
        c should be ("OKAY")
        d should be ("OKAY")
      }
    }
  }

  test("generator-driven property that takes 4 named args and generators, with minSize to 7 and maxSize to 11, specified as (param, param)") {

    pendingUntilFixed {
      forAll ((sevenEleven, "a"), (sevenEleven, "b"), (sevenEleven, "c"), (sevenEleven, "d"), minSize(7), maxSize(11)) { (a: String, b: String, c: String, d: String) =>
        a should be ("OKAY")
        b should be ("OKAY")
        c should be ("OKAY")
        d should be ("OKAY")
      }
    }
  }

  // set minSize to 7 and maxSize to 11 with (param, default) (ensure always passed with that size)
  test("generator-driven property that takes 4 args and generators, with minSize to 7 and maxSize to 11, specified as (param, default)") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(maxSize = 11)

    pendingUntilFixed {
      forAll (sevenEleven, sevenEleven, sevenEleven, sevenEleven, minSize(7)) { (a: String, b: String, c: String, d: String) =>
        a should be ("OKAY")
        b should be ("OKAY")
        c should be ("OKAY")
        d should be ("OKAY")
      }
    }
  }

  test("generator-driven property that takes 4 named args and generators, with minSize to 7 and maxSize to 11, specified as (param, default)") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(maxSize = 11)

    pendingUntilFixed {
      forAll ((sevenEleven, "a"), (sevenEleven, "b"), (sevenEleven, "c"), (sevenEleven, "d"), minSize(7)) { (a: String, b: String, c: String, d: String) =>
        a should be ("OKAY")
        b should be ("OKAY")
        c should be ("OKAY")
        d should be ("OKAY")
      }
    }
  }

  // set minSize to 7 and maxSize to 11 with (default, param) (ensure always passed with that size)
  test("generator-driven property that takes 4 args and generators, with minSize to 7 and maxSize to 11, specified as (default, param)") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSize = 7)

    pendingUntilFixed {
      forAll (sevenEleven, sevenEleven, sevenEleven, sevenEleven, maxSize(11)) { (a: String, b: String, c: String, d: String) =>
        a should be ("OKAY")
        b should be ("OKAY")
        c should be ("OKAY")
        d should be ("OKAY")
      }
    }
  }

  test("generator-driven property that takes 4 named args and generators, with minSize to 7 and maxSize to 11, specified as (default, param)") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSize = 7)

    pendingUntilFixed {
      forAll ((sevenEleven, "a"), (sevenEleven, "b"), (sevenEleven, "c"), (sevenEleven, "d"), maxSize(11)) { (a: String, b: String, c: String, d: String) =>
        a should be ("OKAY")
        b should be ("OKAY")
        c should be ("OKAY")
        d should be ("OKAY")
      }
    }
  }

  // set minSize to 7 and maxSize to 11 with (default, default) (ensure always passed with that size)
  test("generator-driven property that takes 4 args and generators, with minSize to 7 and maxSize to 11, specified as (default, default)") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSize = 7, maxSize = 11)

    pendingUntilFixed {
      forAll (sevenEleven, sevenEleven, sevenEleven, sevenEleven) { (a: String, b: String, c: String, d: String) =>
        a should be ("OKAY")
        b should be ("OKAY")
        c should be ("OKAY")
        d should be ("OKAY")
      }
    }
  }

  test("generator-driven property that takes 4 named args and generators, with minSize to 7 and maxSize to 11, specified as (default, default)") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSize = 7, maxSize = 11)

    pendingUntilFixed {
      forAll ((sevenEleven, "a"), (sevenEleven, "b"), (sevenEleven, "c"), (sevenEleven, "d")) { (a: String, b: String, c: String, d: String) =>
        a should be ("OKAY")
        b should be ("OKAY")
        c should be ("OKAY")
        d should be ("OKAY")
      }
    }
  }


  test("generator-driven property that takes 5 args, which succeeds") {

    forAll { (a: String, b: String, c: String, d: String, e: String) =>
      a.length + b.length + c.length + d.length + e.length should equal ((a + b + c + d + e).length)
    }
  }

  test("generator-driven property that takes 5 args, which fails") {

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      forAll { (a: String, b: String, c: String, d: String, e: String) =>
        a.length + b.length + c.length + d.length + e.length should be < 0
      }
    }
  }

  test("generator-driven property that takes 5 named args, which succeeds") {

    forAll ("a", "b", "c", "d", "e") { (a: String, b: String, c: String, d: String, e: String) =>
      a.length + b.length + c.length + d.length + e.length should equal ((a + b + c + d + e).length)
    }
  }

  test("generator-driven property that takes 5 named args, which fails") {

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      forAll ("a", "b", "c", "d", "e") { (a: String, b: String, c: String, d: String, e: String) =>
        a.length + b.length + c.length + d.length + e.length should be < 0
      }
    }
  }

  test("generator-driven property that takes 5 args and generators, which succeeds") {

    forAll (famousLastWords, famousLastWords, famousLastWords, famousLastWords, famousLastWords) { (a: String, b: String, c: String, d: String, e: String) =>
      a.length + b.length + c.length + d.length + e.length should equal ((a + b + c + d + e).length)
    }
  }

  test("generator-driven property that takes 5 args and generators, which fails") {

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      forAll (famousLastWords, famousLastWords, famousLastWords, famousLastWords, famousLastWords) { (a: String, b: String, c: String, d: String, e: String) =>
        a.length + b.length + c.length + d.length + e.length should be < 0
      }
    }
  }

  test("generator-driven property that takes 5 named args and generators, which succeeds") {

    forAll ((famousLastWords, "a"), (famousLastWords, "b"), (famousLastWords, "c"), (famousLastWords, "d"), (famousLastWords, "e")) { (a: String, b: String, c: String, d: String, e: String) =>
      a.length + b.length + c.length + d.length + e.length should equal ((a + b + c + d + e).length)
    }
  }

  test("generator-driven property that takes 5 named args and generators, which fails") {

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      forAll ((famousLastWords, "a"), (famousLastWords, "b"), (famousLastWords, "c"), (famousLastWords, "d"), (famousLastWords, "e")) { (a: String, b: String, c: String, d: String, e: String) =>
        a.length + b.length + c.length + d.length + e.length should be < 0
      }
    }
  }

  // Same thing, but with config params
  test("generator-driven property that takes 5 args, which succeeds, with config params") {

    forAll (minSize(10), maxSize(20)) { (a: String, b: String, c: String, d: String, e: String) =>
      a.length + b.length + c.length + d.length + e.length should equal ((a + b + c + d + e).length)
    }
  }

  test("generator-driven property that takes 5 args, which fails, with config params") {

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      forAll (minSize(10), maxSize(20)) { (a: String, b: String, c: String, d: String, e: String) =>
        a.length + b.length + c.length + d.length + e.length should be < 0
      }
    }
  }

  test("generator-driven property that takes 5 named args, which succeeds, with config params") {

    forAll ("a", "b", "c", "d", "e", minSize(10), maxSize(20)) { (a: String, b: String, c: String, d: String, e: String) =>
      a.length + b.length + c.length + d.length + e.length should equal ((a + b + c + d + e).length)
    }
  }

  test("generator-driven property that takes 5 named args, which fails, with config params") {

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      forAll ("a", "b", "c", "d", "e", minSize(10), maxSize(20)) { (a: String, b: String, c: String, d: String, e: String) =>
        a.length + b.length + c.length + d.length + e.length should be < 0
      }
    }
  }

  test("generator-driven property that takes 5 args and generators, which succeeds, with config params") {

    forAll (famousLastWords, famousLastWords, famousLastWords, famousLastWords, famousLastWords, minSize(10), maxSize(20)) { (a: String, b: String, c: String, d: String, e: String) =>
      a.length + b.length + c.length + d.length + e.length should equal ((a + b + c + d + e).length)
    }
  }

  test("generator-driven property that takes 5 args and generators, which fails, with config params") {

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      forAll (famousLastWords, famousLastWords, famousLastWords, famousLastWords, famousLastWords, minSize(10), maxSize(20)) { (a: String, b: String, c: String, d: String, e: String) =>
        a.length + b.length + c.length + d.length + e.length should be < 0
      }
    }
  }

  test("generator-driven property that takes 5 named args and generators, which succeeds, with config params") {

    forAll ((famousLastWords, "a"), (famousLastWords, "b"), (famousLastWords, "c"), (famousLastWords, "d"), (famousLastWords, "e"), minSize(10), maxSize(20)) { (a: String, b: String, c: String, d: String, e: String) =>
      a.length + b.length + c.length + d.length + e.length should equal ((a + b + c + d + e).length)
    }
  }

  test("generator-driven property that takes 5 named args and generators, which fails, with config params") {

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      forAll ((famousLastWords, "a"), (famousLastWords, "b"), (famousLastWords, "c"), (famousLastWords, "d"), (famousLastWords, "e"), minSize(10), maxSize(20)) { (a: String, b: String, c: String, d: String, e: String) =>
        a.length + b.length + c.length + d.length + e.length should be < 0
      }
    }
  }

  // Same thing, but set minSuccessful to 5 with param, prop fails after 5
  test("generator-driven property that takes 5 args, which succeeds, with minSuccessful param set to 5") {

    var i = 0
    forAll (minSuccessful(5)) { (a: String, b: String, c: String, d: String, e: String) =>
      i += 1
      i should not be 6
    }
  }

  test("generator-driven property that takes 5 args, which fails, with minSuccessful param set to 5") {

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      var i = 0
      forAll (minSuccessful(5)) { (a: String, b: String, c: String, d: String, e: String) =>
        i += 1
        i should not be 5
      }
    }
  }

  test("generator-driven property that takes 5 named args, which succeeds, with minSuccessful param set to 5") {

    var i = 0
    forAll ("a", "b", "c", "d", "e", minSuccessful(5)) { (a: String, b: String, c: String, d: String, e: String) =>
      i += 1
      i should not be 6
    }
  }

  test("generator-driven property that takes 5 named args, which fails, with minSuccessful param set to 5") {

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      var i = 0
      forAll ("a", "b", "c", "d", "e", minSuccessful(5)) { (a: String, b: String, c: String, d: String, e: String) =>
        i += 1
        i should not be 5
      }
    }
  }

  test("generator-driven property that takes 5 args and generators, which succeeds, with minSuccessful param set to 5") {

    var i = 0
    forAll (famousLastWords, famousLastWords, famousLastWords, famousLastWords, famousLastWords, minSuccessful(5)) { (a: String, b: String, c: String, d: String, e: String) =>
      i += 1
      i should not be 6
    }
  }

  test("generator-driven property that takes 5 args and generators, which fails, with minSuccessful param set to 5") {

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      var i = 0
      forAll (famousLastWords, famousLastWords, famousLastWords, famousLastWords, famousLastWords, minSuccessful(5)) { (a: String, b: String, c: String, d: String, e: String) =>
        i += 1
        i should not be 5
      }
    }
  }

  test("generator-driven property that takes 5 named args and generators, which succeeds, with minSuccessful param set to 5") {

    var i = 0
    forAll ((famousLastWords, "a"), (famousLastWords, "b"), (famousLastWords, "c"), (famousLastWords, "d"), (famousLastWords, "e"), minSuccessful(5)) { (a: String, b: String, c: String, d: String, e: String) =>
      i += 1
      i should not be 6
    }
  }

  test("generator-driven property that takes 5 named args and generators, which fails, with minSuccessful param set to 5") {

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      var i = 0
      forAll ((famousLastWords, "a"), (famousLastWords, "b"), (famousLastWords, "c"), (famousLastWords, "d"), (famousLastWords, "e"), minSuccessful(5)) { (a: String, b: String, c: String, d: String, e: String) =>
        i += 1
        i should not be 5
      }
    }
  }

  // Same thing, but set default minSuccessful to 5, prop fails after 5
  test("generator-driven property that takes 5 args, which succeeds, with default minSuccessful param set to 5") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSuccessful = 5)

    var i = 0
    forAll { (a: String, b: String, c: String, d: String, e: String) =>
      i += 1
      i should not be 6
    }
  }

  test("generator-driven property that takes 5 args, which fails, with default minSuccessful param set to 5") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSuccessful = 5)

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      var i = 0
      forAll { (a: String, b: String, c: String, d: String, e: String) =>
        i += 1
        i should not be 5
      }
    }
  }

  test("generator-driven property that takes 5 named args, which succeeds, with default minSuccessful param set to 5") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSuccessful = 5)

    var i = 0
    forAll ("a", "b", "c", "d", "e") { (a: String, b: String, c: String, d: String, e: String) =>
      i += 1
      i should not be 6
    }
  }

  test("generator-driven property that takes 5 named args, which fails, with default minSuccessful param set to 5") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSuccessful = 5)

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      var i = 0
      forAll ("a", "b", "c", "d", "e") { (a: String, b: String, c: String, d: String, e: String) =>
        i += 1
        i should not be 5
      }
    }
  }

  test("generator-driven property that takes 5 args and generators, which succeeds, with default minSuccessful param set to 5") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSuccessful = 5)

    var i = 0
    forAll (famousLastWords, famousLastWords, famousLastWords, famousLastWords, famousLastWords) { (a: String, b: String, c: String, d: String, e: String) =>
      i += 1
      i should not be 6
    }
  }

  test("generator-driven property that takes 5 args and generators, which fails, with default minSuccessful param set to 5") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSuccessful = 5)

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      var i = 0
      forAll (famousLastWords, famousLastWords, famousLastWords, famousLastWords, famousLastWords) { (a: String, b: String, c: String, d: String, e: String) =>
        i += 1
        i should not be 5
      }
    }
  }

  test("generator-driven property that takes 5 named args and generators, which succeeds, with default minSuccessful param set to 5") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSuccessful = 5)

    var i = 0
    forAll ((famousLastWords, "a"), (famousLastWords, "b"), (famousLastWords, "c"), (famousLastWords, "d"), (famousLastWords, "e")) { (a: String, b: String, c: String, d: String, e: String) =>
      i += 1
      i should not be 6
    }
  }

  test("generator-driven property that takes 5 named args and generators, which fails, with default minSuccessful param set to 5") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSuccessful = 5)

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      var i = 0
      forAll ((famousLastWords, "a"), (famousLastWords, "b"), (famousLastWords, "c"), (famousLastWords, "d"), (famousLastWords, "e")) { (a: String, b: String, c: String, d: String, e: String) =>
        i += 1
        i should not be 5
      }
    }
  }

  // Same thing, but set maxDiscarded to 5 with param, prop fails after 5
  test("generator-driven property that takes 5 args, which succeeds, with maxDiscarded param set to 5") {

    var i = 0
    forAll (maxDiscarded(5)) { (a: String, b: String, c: String, d: String, e: String) =>
      i += 1
      whenever (i > 5) { 1 + 1 should equal (2) }
    }
  }

  test("generator-driven property that takes 5 args, which fails, with maxDiscarded param set to 5") {

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      var i = 0
      forAll (maxDiscarded(5)) { (a: String, b: String, c: String, d: String, e: String) =>
        i += 1
        whenever (i > 6) { 1 + 1 should equal (2) }
      }
    }
  }

  test("generator-driven property that takes 5 named args, which succeeds, with maxDiscarded param set to 5") {

    var i = 0
    forAll ("a", "b", "c", "d", "e", maxDiscarded(5)) { (a: String, b: String, c: String, d: String, e: String) =>
      i += 1
      whenever (i > 5) { 1 + 1 should equal (2) }
    }
  }

  test("generator-driven property that takes 5 named args, which fails, with maxDiscarded param set to 5") {

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      var i = 0
      forAll ("a", "b", "c", "d", "e", maxDiscarded(5)) { (a: String, b: String, c: String, d: String, e: String) =>
        i += 1
        whenever (i > 6) { 1 + 1 should equal (2) }
      }
    }
  }

  test("generator-driven property that takes 5 args and generators, which succeeds, with maxDiscarded param set to 5") {

    var i = 0
    forAll (famousLastWords, famousLastWords, famousLastWords, famousLastWords, famousLastWords, maxDiscarded(5)) { (a: String, b: String, c: String, d: String, e: String) =>
      i += 1
      whenever (i > 5) { 1 + 1 should equal (2) }
    }
  }

  test("generator-driven property that takes 5 args and generators, which fails, with maxDiscarded param set to 5") {

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      var i = 0
      forAll (famousLastWords, famousLastWords, famousLastWords, famousLastWords, famousLastWords, maxDiscarded(5)) { (a: String, b: String, c: String, d: String, e: String) =>
        i += 1
        whenever (i > 6) { 1 + 1 should equal (2) }
      }
    }
  }

  test("generator-driven property that takes 5 named args and generators, which succeeds, with maxDiscarded param set to 5") {

    var i = 0
    forAll ((famousLastWords, "a"), (famousLastWords, "b"), (famousLastWords, "c"), (famousLastWords, "d"), (famousLastWords, "e"), maxDiscarded(5)) { (a: String, b: String, c: String, d: String, e: String) =>
      i += 1
      whenever (i > 5) { 1 + 1 should equal (2) }
    }
  }

  test("generator-driven property that takes 5 named args and generators, which fails, with maxDiscarded param set to 5") {

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      var i = 0
      forAll ((famousLastWords, "a"), (famousLastWords, "b"), (famousLastWords, "c"), (famousLastWords, "d"), (famousLastWords, "e"), maxDiscarded(5)) { (a: String, b: String, c: String, d: String, e: String) =>
        i += 1
        whenever (i > 6) { 1 + 1 should equal (2) }
      }
    }
  }

  // Same thing, but set default maxDiscarded to 5, prop fails after 5
  test("generator-driven property that takes 5 args, which succeeds, with default maxDiscarded set to 5") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(maxDiscarded = 5)

    var i = 0
    forAll { (a: String, b: String, c: String, d: String, e: String) =>
      i += 1
      whenever (i > 5) { 1 + 1 should equal (2) }
    }
  }

  test("generator-driven property that takes 5 args, which fails, with default maxDiscarded set to 5") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(maxDiscarded = 5)

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      var i = 0
      forAll { (a: String, b: String, c: String, d: String, e: String) =>
        i += 1
        whenever (i > 6) { 1 + 1 should equal (2) }
      }
    }
  }

  test("generator-driven property that takes 5 named args, which succeeds, with default maxDiscarded set to 5") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(maxDiscarded = 5)

    var i = 0
    forAll ("a", "b", "c", "d", "e") { (a: String, b: String, c: String, d: String, e: String) =>
      i += 1
      whenever (i > 5) { 1 + 1 should equal (2) }
    }
  }

  test("generator-driven property that takes 5 named args, which fails, with default maxDiscarded set to 5") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(maxDiscarded = 5)

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      var i = 0
      forAll ("a", "b", "c", "d", "e") { (a: String, b: String, c: String, d: String, e: String) =>
        i += 1
        whenever (i > 6) { 1 + 1 should equal (2) }
      }
    }
  }

  test("generator-driven property that takes 5 args and generators, which succeeds, with default maxDiscarded set to 5") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(maxDiscarded = 5)

    var i = 0
    forAll (famousLastWords, famousLastWords, famousLastWords, famousLastWords, famousLastWords) { (a: String, b: String, c: String, d: String, e: String) =>
      i += 1
      whenever (i > 5) { 1 + 1 should equal (2) }
    }
  }

  test("generator-driven property that takes 5 args and generators, which fails, with default maxDiscarded set to 5") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(maxDiscarded = 5)

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      var i = 0
      forAll (famousLastWords, famousLastWords, famousLastWords, famousLastWords, famousLastWords) { (a: String, b: String, c: String, d: String, e: String) =>
        i += 1
        whenever (i > 6) { 1 + 1 should equal (2) }
      }
    }
  }

  test("generator-driven property that takes 5 named args and generators, which succeeds, with default maxDiscarded set to 5") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(maxDiscarded = 5)

    var i = 0
    forAll ((famousLastWords, "a"), (famousLastWords, "b"), (famousLastWords, "c"), (famousLastWords, "d"), (famousLastWords, "e")) { (a: String, b: String, c: String, d: String, e: String) =>
      i += 1
      whenever (i > 5) { 1 + 1 should equal (2) }
    }
  }

  test("generator-driven property that takes 5 named args and generators, which fails, with default maxDiscarded set to 5") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(maxDiscarded = 5)

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      var i = 0
      forAll ((famousLastWords, "a"), (famousLastWords, "b"), (famousLastWords, "c"), (famousLastWords, "d"), (famousLastWords, "e")) { (a: String, b: String, c: String, d: String, e: String) =>
        i += 1
        whenever (i > 6) { 1 + 1 should equal (2) }
      }
    }
  }

  // set minSize > maxSize with (param, param) (intercept IAE)
  test("generator-driven property that takes 5 args, which should throw IAE because maxSize > maxSize, specified as (param, param)") {

    intercept[IllegalArgumentException] {
      forAll (minSize(5), maxSize(4)) { (a: String, b: String, c: String, d: String, e: String) =>
        1 + 1 should equal (2)
      }
    }
  }

  test("generator-driven property that takes 5 named args, which should throw IAE because maxSize > maxSize, specified as (param, param)") {

    intercept[IllegalArgumentException] {
      forAll ("a", "b", "c", "d", "e", minSize(5), maxSize(4)) { (a: String, b: String, c: String, d: String, e: String) =>
        1 + 1 should equal (2)
      }
    }
  }

  test("generator-driven property that takes 5 args and generators, which should throw IAE because maxSize > maxSize, specified as (param, param)") {

    intercept[IllegalArgumentException] {
      forAll (famousLastWords, famousLastWords, famousLastWords, famousLastWords, famousLastWords, minSize(5), maxSize(4)) { (a: String, b: String, c: String, d: String, e: String) =>
        1 + 1 should equal (2)
      }
    }
  }

  test("generator-driven property that takes 5 named args and generators, which should throw IAE because maxSize > maxSize, specified as (param, param)") {

    intercept[IllegalArgumentException] {
      forAll ((famousLastWords, "a"), (famousLastWords, "b"), (famousLastWords, "c"), (famousLastWords, "d"), (famousLastWords, "e"), minSize(5), maxSize(4)) { (a: String, b: String, c: String, d: String, e: String) =>
        1 + 1 should equal (2)
      }
    }
  }

  // set minSize > maxSize with (param, default) (intercept IAE)
  test("generator-driven property that takes 5 args, which should throw IAE because maxSize > maxSize, specified as (param, default)") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(maxSize = 4)

    intercept[IllegalArgumentException] {
      forAll (minSize(5)) { (a: String, b: String, c: String, d: String, e: String) =>
        1 + 1 should equal (2)
      }
    }
  }

  test("generator-driven property that takes 5 named args, which should throw IAE because maxSize > maxSize, specified as (param, default)") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(maxSize = 4)

    intercept[IllegalArgumentException] {
      forAll ("a", "b", "c", "d", "e", minSize(5)) { (a: String, b: String, c: String, d: String, e: String) =>
        1 + 1 should equal (2)
      }
    }
  }

  test("generator-driven property that takes 5 args and generators, which should throw IAE because maxSize > maxSize, specified as (param, default)") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(maxSize = 4)

    intercept[IllegalArgumentException] {
      forAll (famousLastWords, famousLastWords, famousLastWords, famousLastWords, famousLastWords, minSize(5)) { (a: String, b: String, c: String, d: String, e: String) =>
        1 + 1 should equal (2)
      }
    }
  }

  test("generator-driven property that takes 5 named args and generators, which should throw IAE because maxSize > maxSize, specified as (param, default)") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(maxSize = 4)

    intercept[IllegalArgumentException] {
      forAll ((famousLastWords, "a"), (famousLastWords, "b"), (famousLastWords, "c"), (famousLastWords, "d"), (famousLastWords, "e"), minSize(5)) { (a: String, b: String, c: String, d: String, e: String) =>
        1 + 1 should equal (2)
      }
    }
  }

  // set minSize > maxSize with (default, param) (intercept IAE)
  test("generator-driven property that takes 5 args, which should throw IAE because maxSize > maxSize, specified as (default, param)") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSize = 5)

    intercept[IllegalArgumentException] {
      forAll (maxSize(4)) { (a: String, b: String, c: String, d: String, e: String) =>
        1 + 1 should equal (2)
      }
    }
  }

  test("generator-driven property that takes 5 named args, which should throw IAE because maxSize > maxSize, specified as (default, param)") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSize = 5)

    intercept[IllegalArgumentException] {
      forAll ("a", "b", "c", "d", "e", maxSize(4)) { (a: String, b: String, c: String, d: String, e: String) =>
        1 + 1 should equal (2)
      }
    }
  }

  test("generator-driven property that takes 5 args and generators, which should throw IAE because maxSize > maxSize, specified as (default, param)") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSize = 5)

    intercept[IllegalArgumentException] {
      forAll (famousLastWords, famousLastWords, famousLastWords, famousLastWords, famousLastWords, maxSize(4)) { (a: String, b: String, c: String, d: String, e: String) =>
        1 + 1 should equal (2)
      }
    }
  }

  test("generator-driven property that takes 5 named args and generators, which should throw IAE because maxSize > maxSize, specified as (default, param)") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSize = 5)

    intercept[IllegalArgumentException] {
      forAll ((famousLastWords, "a"), (famousLastWords, "b"), (famousLastWords, "c"), (famousLastWords, "d"), (famousLastWords, "e"), maxSize(4)) { (a: String, b: String, c: String, d: String, e: String) =>
        1 + 1 should equal (2)
      }
    }
  }

  // set maxSize with param (ensure always passed with a size less than maxSize)
  test("generator-driven property that takes 5 args, with maxSize specified as param") {

    forAll (maxSize(5)) { (a: String, b: String, c: String, d: String, e: String) =>
      a.length should be <= 5
      b.length should be <= 5
      c.length should be <= 5
      d.length should be <= 5
      e.length should be <= 5
    }
  }

  test("generator-driven property that takes 5 named args, with maxSize specified as param") {

    forAll ("a", "b", "c", "d", "e", maxSize(5)) { (a: String, b: String, c: String, d: String, e: String) =>
      a.length should be <= 5
      b.length should be <= 5
      c.length should be <= 5
      d.length should be <= 5
      e.length should be <= 5
    }
  }

  // set maxSize with default (ensure always passed with a size less than maxSize)
  test("generator-driven property that takes 5 args, with maxSize specified as default") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(maxSize = 5)

    forAll { (a: String, b: String, c: String, d: String, e: String) =>
      a.length should be <= 5
      b.length should be <= 5
      c.length should be <= 5
      d.length should be <= 5
      e.length should be <= 5
    }
  }

  test("generator-driven property that takes 5 named args, with maxSize specified as default") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(maxSize = 5)

    forAll ("a", "b", "c", "d", "e") { (a: String, b: String, c: String, d: String, e: String) =>
      a.length should be <= 5
      b.length should be <= 5
      c.length should be <= 5
      d.length should be <= 5
      e.length should be <= 5
    }
  }
 
  // set minSize == maxSize with (param, param) (ensure always passed with that size)
  test("generator-driven property that takes 5 args and generators, with minSize == maxSize, specified as (param, param)") {

    pendingUntilFixed {
      forAll (fiveFive, fiveFive, fiveFive, fiveFive, fiveFive, minSize(5), maxSize(5)) { (a: String, b: String, c: String, d: String, e: String) =>
        a should be ("OKAY")
        b should be ("OKAY")
        c should be ("OKAY")
        d should be ("OKAY")
        e should be ("OKAY")
      }
    }
  }

  test("generator-driven property that takes 5 named args and generators, with minSize == maxSize, specified as (param, param)") {

    pendingUntilFixed {
      forAll ((fiveFive, "a"), (fiveFive, "b"), (fiveFive, "c"), (fiveFive, "d"), (fiveFive, "e"), minSize(5), maxSize(5)) { (a: String, b: String, c: String, d: String, e: String) =>
        a should be ("OKAY")
        b should be ("OKAY")
        c should be ("OKAY")
        d should be ("OKAY")
        e should be ("OKAY")
      }
    }
  }

  // set minSize == maxSize with (param, default) (ensure always passed with that size)
  test("generator-driven property that takes 5 args and generators, with minSize == maxSize, specified as (param, default)") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(maxSize = 5)

    pendingUntilFixed {
      forAll (fiveFive, fiveFive, fiveFive, fiveFive, fiveFive, minSize(5)) { (a: String, b: String, c: String, d: String, e: String) =>
        a should be ("OKAY")
        b should be ("OKAY")
        c should be ("OKAY")
        d should be ("OKAY")
        e should be ("OKAY")
      }
    }
  }

  test("generator-driven property that takes 5 named args and generators, with minSize == maxSize, specified as (param, default)") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(maxSize = 5)

    pendingUntilFixed {
      forAll ((fiveFive, "a"), (fiveFive, "b"), (fiveFive, "c"), (fiveFive, "d"), (fiveFive, "e"), minSize(5)) { (a: String, b: String, c: String, d: String, e: String) =>
        a should be ("OKAY")
        b should be ("OKAY")
        c should be ("OKAY")
        d should be ("OKAY")
        e should be ("OKAY")
      }
    }
  }

  // set minSize == maxSize with (default, param) (ensure always passed with that size)
  test("generator-driven property that takes 5 args and generators, with minSize == maxSize, specified as (default, param)") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSize = 5)

    pendingUntilFixed {
      forAll (fiveFive, fiveFive, fiveFive, fiveFive, fiveFive, maxSize(5)) { (a: String, b: String, c: String, d: String, e: String) =>
        a should be ("OKAY")
        b should be ("OKAY")
        c should be ("OKAY")
        d should be ("OKAY")
        e should be ("OKAY")
      }
    }
  }

  test("generator-driven property that takes 5 named args and generators, with minSize == maxSize, specified as (default, param)") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSize = 5)

    pendingUntilFixed {
      forAll ((fiveFive, "a"), (fiveFive, "b"), (fiveFive, "c"), (fiveFive, "d"), (fiveFive, "e"), maxSize(5)) { (a: String, b: String, c: String, d: String, e: String) =>
        a should be ("OKAY")
        b should be ("OKAY")
        c should be ("OKAY")
        d should be ("OKAY")
        e should be ("OKAY")
      }
    }
  }

  // set minSize == maxSize with (default, default) (ensure always passed with that size)
  test("generator-driven property that takes 5 args and generators, with minSize == maxSize, specified as (default, default)") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSize = 5, maxSize = 5)

    pendingUntilFixed {
      forAll (fiveFive, fiveFive, fiveFive, fiveFive, fiveFive) { (a: String, b: String, c: String, d: String, e: String) =>
        a should be ("OKAY")
        b should be ("OKAY")
        c should be ("OKAY")
        d should be ("OKAY")
        e should be ("OKAY")
      }
    }
  }

  test("generator-driven property that takes 5 named args and generators, with minSize == maxSize, specified as (default, default)") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSize = 5, maxSize = 5)

    pendingUntilFixed {
      forAll ((fiveFive, "a"), (fiveFive, "b"), (fiveFive, "c"), (fiveFive, "d"), (fiveFive, "e")) { (a: String, b: String, c: String, d: String, e: String) =>
        a should be ("OKAY")
        b should be ("OKAY")
        c should be ("OKAY")
        d should be ("OKAY")
        e should be ("OKAY")
      }
    }
  }

  // set minSize to 7 and maxSize to 11 with (param, param) (ensure always passed with that size)
  test("generator-driven property that takes 5 args and generators, with minSize to 7 and maxSize to 11, specified as (param, param)") {

    pendingUntilFixed {
      forAll (sevenEleven, sevenEleven, sevenEleven, sevenEleven, sevenEleven, minSize(7), maxSize(11)) { (a: String, b: String, c: String, d: String, e: String) =>
        a should be ("OKAY")
        b should be ("OKAY")
        c should be ("OKAY")
        d should be ("OKAY")
        e should be ("OKAY")
      }
    }
  }

  test("generator-driven property that takes 5 named args and generators, with minSize to 7 and maxSize to 11, specified as (param, param)") {

    pendingUntilFixed {
      forAll ((sevenEleven, "a"), (sevenEleven, "b"), (sevenEleven, "c"), (sevenEleven, "d"), (sevenEleven, "e"), minSize(7), maxSize(11)) { (a: String, b: String, c: String, d: String, e: String) =>
        a should be ("OKAY")
        b should be ("OKAY")
        c should be ("OKAY")
        d should be ("OKAY")
        e should be ("OKAY")
      }
    }
  }

  // set minSize to 7 and maxSize to 11 with (param, default) (ensure always passed with that size)
  test("generator-driven property that takes 5 args and generators, with minSize to 7 and maxSize to 11, specified as (param, default)") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(maxSize = 11)

    pendingUntilFixed {
      forAll (sevenEleven, sevenEleven, sevenEleven, sevenEleven, sevenEleven, minSize(7)) { (a: String, b: String, c: String, d: String, e: String) =>
        a should be ("OKAY")
        b should be ("OKAY")
        c should be ("OKAY")
        d should be ("OKAY")
        e should be ("OKAY")
      }
    }
  }

  test("generator-driven property that takes 5 named args and generators, with minSize to 7 and maxSize to 11, specified as (param, default)") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(maxSize = 11)

    pendingUntilFixed {
      forAll ((sevenEleven, "a"), (sevenEleven, "b"), (sevenEleven, "c"), (sevenEleven, "d"), (sevenEleven, "e"), minSize(7)) { (a: String, b: String, c: String, d: String, e: String) =>
        a should be ("OKAY")
        b should be ("OKAY")
        c should be ("OKAY")
        d should be ("OKAY")
        e should be ("OKAY")
      }
    }
  }

  // set minSize to 7 and maxSize to 11 with (default, param) (ensure always passed with that size)
  test("generator-driven property that takes 5 args and generators, with minSize to 7 and maxSize to 11, specified as (default, param)") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSize = 7)

    pendingUntilFixed {
      forAll (sevenEleven, sevenEleven, sevenEleven, sevenEleven, sevenEleven, maxSize(11)) { (a: String, b: String, c: String, d: String, e: String) =>
        a should be ("OKAY")
        b should be ("OKAY")
        c should be ("OKAY")
        d should be ("OKAY")
        e should be ("OKAY")
      }
    }
  }

  test("generator-driven property that takes 5 named args and generators, with minSize to 7 and maxSize to 11, specified as (default, param)") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSize = 7)

    pendingUntilFixed {
      forAll ((sevenEleven, "a"), (sevenEleven, "b"), (sevenEleven, "c"), (sevenEleven, "d"), (sevenEleven, "e"), maxSize(11)) { (a: String, b: String, c: String, d: String, e: String) =>
        a should be ("OKAY")
        b should be ("OKAY")
        c should be ("OKAY")
        d should be ("OKAY")
        e should be ("OKAY")
      }
    }
  }

  // set minSize to 7 and maxSize to 11 with (default, default) (ensure always passed with that size)
  test("generator-driven property that takes 5 args and generators, with minSize to 7 and maxSize to 11, specified as (default, default)") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSize = 7, maxSize = 11)

    pendingUntilFixed {
      forAll (sevenEleven, sevenEleven, sevenEleven, sevenEleven, sevenEleven) { (a: String, b: String, c: String, d: String, e: String) =>
        a should be ("OKAY")
        b should be ("OKAY")
        c should be ("OKAY")
        d should be ("OKAY")
        e should be ("OKAY")
      }
    }
  }

  test("generator-driven property that takes 5 named args and generators, with minSize to 7 and maxSize to 11, specified as (default, default)") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSize = 7, maxSize = 11)

    pendingUntilFixed {
      forAll ((sevenEleven, "a"), (sevenEleven, "b"), (sevenEleven, "c"), (sevenEleven, "d"), (sevenEleven, "e")) { (a: String, b: String, c: String, d: String, e: String) =>
        a should be ("OKAY")
        b should be ("OKAY")
        c should be ("OKAY")
        d should be ("OKAY")
        e should be ("OKAY")
      }
    }
  }


  test("generator-driven property that takes 6 args, which succeeds") {

    forAll { (a: String, b: String, c: String, d: String, e: String, f: String) =>
      a.length + b.length + c.length + d.length + e.length + f.length should equal ((a + b + c + d + e + f).length)
    }
  }

  test("generator-driven property that takes 6 args, which fails") {

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      forAll { (a: String, b: String, c: String, d: String, e: String, f: String) =>
        a.length + b.length + c.length + d.length + e.length + f.length should be < 0
      }
    }
  }

  test("generator-driven property that takes 6 named args, which succeeds") {

    forAll ("a", "b", "c", "d", "e", "f") { (a: String, b: String, c: String, d: String, e: String, f: String) =>
      a.length + b.length + c.length + d.length + e.length + f.length should equal ((a + b + c + d + e + f).length)
    }
  }

  test("generator-driven property that takes 6 named args, which fails") {

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      forAll ("a", "b", "c", "d", "e", "f") { (a: String, b: String, c: String, d: String, e: String, f: String) =>
        a.length + b.length + c.length + d.length + e.length + f.length should be < 0
      }
    }
  }

  test("generator-driven property that takes 6 args and generators, which succeeds") {

    forAll (famousLastWords, famousLastWords, famousLastWords, famousLastWords, famousLastWords, famousLastWords) { (a: String, b: String, c: String, d: String, e: String, f: String) =>
      a.length + b.length + c.length + d.length + e.length + f.length should equal ((a + b + c + d + e + f).length)
    }
  }

  test("generator-driven property that takes 6 args and generators, which fails") {

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      forAll (famousLastWords, famousLastWords, famousLastWords, famousLastWords, famousLastWords, famousLastWords) { (a: String, b: String, c: String, d: String, e: String, f: String) =>
        a.length + b.length + c.length + d.length + e.length + f.length should be < 0
      }
    }
  }

  test("generator-driven property that takes 6 named args and generators, which succeeds") {

    forAll ((famousLastWords, "a"), (famousLastWords, "b"), (famousLastWords, "c"), (famousLastWords, "d"), (famousLastWords, "e"), (famousLastWords, "f")) { (a: String, b: String, c: String, d: String, e: String, f: String) =>
      a.length + b.length + c.length + d.length + e.length + f.length should equal ((a + b + c + d + e + f).length)
    }
  }

  test("generator-driven property that takes 6 named args and generators, which fails") {

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      forAll ((famousLastWords, "a"), (famousLastWords, "b"), (famousLastWords, "c"), (famousLastWords, "d"), (famousLastWords, "e"), (famousLastWords, "f")) { (a: String, b: String, c: String, d: String, e: String, f: String) =>
        a.length + b.length + c.length + d.length + e.length + f.length should be < 0
      }
    }
  }

  // Same thing, but with config params
  test("generator-driven property that takes 6 args, which succeeds, with config params") {

    forAll (minSize(10), maxSize(20)) { (a: String, b: String, c: String, d: String, e: String, f: String) =>
      a.length + b.length + c.length + d.length + e.length + f.length should equal ((a + b + c + d + e + f).length)
    }
  }

  test("generator-driven property that takes 6 args, which fails, with config params") {

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      forAll (minSize(10), maxSize(20)) { (a: String, b: String, c: String, d: String, e: String, f: String) =>
        a.length + b.length + c.length + d.length + e.length + f.length should be < 0
      }
    }
  }

  test("generator-driven property that takes 6 named args, which succeeds, with config params") {

    forAll ("a", "b", "c", "d", "e", "f", minSize(10), maxSize(20)) { (a: String, b: String, c: String, d: String, e: String, f: String) =>
      a.length + b.length + c.length + d.length + e.length + f.length should equal ((a + b + c + d + e + f).length)
    }
  }

  test("generator-driven property that takes 6 named args, which fails, with config params") {

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      forAll ("a", "b", "c", "d", "e", "f", minSize(10), maxSize(20)) { (a: String, b: String, c: String, d: String, e: String, f: String) =>
        a.length + b.length + c.length + d.length + e.length + f.length should be < 0
      }
    }
  }

  test("generator-driven property that takes 6 args and generators, which succeeds, with config params") {

    forAll (famousLastWords, famousLastWords, famousLastWords, famousLastWords, famousLastWords, famousLastWords, minSize(10), maxSize(20)) { (a: String, b: String, c: String, d: String, e: String, f: String) =>
      a.length + b.length + c.length + d.length + e.length + f.length should equal ((a + b + c + d + e + f).length)
    }
  }

  test("generator-driven property that takes 6 args and generators, which fails, with config params") {

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      forAll (famousLastWords, famousLastWords, famousLastWords, famousLastWords, famousLastWords, famousLastWords, minSize(10), maxSize(20)) { (a: String, b: String, c: String, d: String, e: String, f: String) =>
        a.length + b.length + c.length + d.length + e.length + f.length should be < 0
      }
    }
  }

  test("generator-driven property that takes 6 named args and generators, which succeeds, with config params") {

    forAll ((famousLastWords, "a"), (famousLastWords, "b"), (famousLastWords, "c"), (famousLastWords, "d"), (famousLastWords, "e"), (famousLastWords, "f"), minSize(10), maxSize(20)) { (a: String, b: String, c: String, d: String, e: String, f: String) =>
      a.length + b.length + c.length + d.length + e.length + f.length should equal ((a + b + c + d + e + f).length)
    }
  }

  test("generator-driven property that takes 6 named args and generators, which fails, with config params") {

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      forAll ((famousLastWords, "a"), (famousLastWords, "b"), (famousLastWords, "c"), (famousLastWords, "d"), (famousLastWords, "e"), (famousLastWords, "f"), minSize(10), maxSize(20)) { (a: String, b: String, c: String, d: String, e: String, f: String) =>
        a.length + b.length + c.length + d.length + e.length + f.length should be < 0
      }
    }
  }

  // Same thing, but set minSuccessful to 5 with param, prop fails after 5
  test("generator-driven property that takes 6 args, which succeeds, with minSuccessful param set to 5") {

    var i = 0
    forAll (minSuccessful(5)) { (a: String, b: String, c: String, d: String, e: String, f: String) =>
      i += 1
      i should not be 6
    }
  }

  test("generator-driven property that takes 6 args, which fails, with minSuccessful param set to 5") {

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      var i = 0
      forAll (minSuccessful(5)) { (a: String, b: String, c: String, d: String, e: String, f: String) =>
        i += 1
        i should not be 5
      }
    }
  }

  test("generator-driven property that takes 6 named args, which succeeds, with minSuccessful param set to 5") {

    var i = 0
    forAll ("a", "b", "c", "d", "e", "f", minSuccessful(5)) { (a: String, b: String, c: String, d: String, e: String, f: String) =>
      i += 1
      i should not be 6
    }
  }

  test("generator-driven property that takes 6 named args, which fails, with minSuccessful param set to 5") {

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      var i = 0
      forAll ("a", "b", "c", "d", "e", "f", minSuccessful(5)) { (a: String, b: String, c: String, d: String, e: String, f: String) =>
        i += 1
        i should not be 5
      }
    }
  }

  test("generator-driven property that takes 6 args and generators, which succeeds, with minSuccessful param set to 5") {

    var i = 0
    forAll (famousLastWords, famousLastWords, famousLastWords, famousLastWords, famousLastWords, famousLastWords, minSuccessful(5)) { (a: String, b: String, c: String, d: String, e: String, f: String) =>
      i += 1
      i should not be 6
    }
  }

  test("generator-driven property that takes 6 args and generators, which fails, with minSuccessful param set to 5") {

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      var i = 0
      forAll (famousLastWords, famousLastWords, famousLastWords, famousLastWords, famousLastWords, famousLastWords, minSuccessful(5)) { (a: String, b: String, c: String, d: String, e: String, f: String) =>
        i += 1
        i should not be 5
      }
    }
  }

  test("generator-driven property that takes 6 named args and generators, which succeeds, with minSuccessful param set to 5") {

    var i = 0
    forAll ((famousLastWords, "a"), (famousLastWords, "b"), (famousLastWords, "c"), (famousLastWords, "d"), (famousLastWords, "e"), (famousLastWords, "f"), minSuccessful(5)) { (a: String, b: String, c: String, d: String, e: String, f: String) =>
      i += 1
      i should not be 6
    }
  }

  test("generator-driven property that takes 6 named args and generators, which fails, with minSuccessful param set to 5") {

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      var i = 0
      forAll ((famousLastWords, "a"), (famousLastWords, "b"), (famousLastWords, "c"), (famousLastWords, "d"), (famousLastWords, "e"), (famousLastWords, "f"), minSuccessful(5)) { (a: String, b: String, c: String, d: String, e: String, f: String) =>
        i += 1
        i should not be 5
      }
    }
  }

  // Same thing, but set default minSuccessful to 5, prop fails after 5
  test("generator-driven property that takes 6 args, which succeeds, with default minSuccessful param set to 5") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSuccessful = 5)

    var i = 0
    forAll { (a: String, b: String, c: String, d: String, e: String, f: String) =>
      i += 1
      i should not be 6
    }
  }

  test("generator-driven property that takes 6 args, which fails, with default minSuccessful param set to 5") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSuccessful = 5)

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      var i = 0
      forAll { (a: String, b: String, c: String, d: String, e: String, f: String) =>
        i += 1
        i should not be 5
      }
    }
  }

  test("generator-driven property that takes 6 named args, which succeeds, with default minSuccessful param set to 5") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSuccessful = 5)

    var i = 0
    forAll ("a", "b", "c", "d", "e", "f") { (a: String, b: String, c: String, d: String, e: String, f: String) =>
      i += 1
      i should not be 6
    }
  }

  test("generator-driven property that takes 6 named args, which fails, with default minSuccessful param set to 5") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSuccessful = 5)

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      var i = 0
      forAll ("a", "b", "c", "d", "e", "f") { (a: String, b: String, c: String, d: String, e: String, f: String) =>
        i += 1
        i should not be 5
      }
    }
  }

  test("generator-driven property that takes 6 args and generators, which succeeds, with default minSuccessful param set to 5") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSuccessful = 5)

    var i = 0
    forAll (famousLastWords, famousLastWords, famousLastWords, famousLastWords, famousLastWords, famousLastWords) { (a: String, b: String, c: String, d: String, e: String, f: String) =>
      i += 1
      i should not be 6
    }
  }

  test("generator-driven property that takes 6 args and generators, which fails, with default minSuccessful param set to 5") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSuccessful = 5)

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      var i = 0
      forAll (famousLastWords, famousLastWords, famousLastWords, famousLastWords, famousLastWords, famousLastWords) { (a: String, b: String, c: String, d: String, e: String, f: String) =>
        i += 1
        i should not be 5
      }
    }
  }

  test("generator-driven property that takes 6 named args and generators, which succeeds, with default minSuccessful param set to 5") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSuccessful = 5)

    var i = 0
    forAll ((famousLastWords, "a"), (famousLastWords, "b"), (famousLastWords, "c"), (famousLastWords, "d"), (famousLastWords, "e"), (famousLastWords, "f")) { (a: String, b: String, c: String, d: String, e: String, f: String) =>
      i += 1
      i should not be 6
    }
  }

  test("generator-driven property that takes 6 named args and generators, which fails, with default minSuccessful param set to 5") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSuccessful = 5)

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      var i = 0
      forAll ((famousLastWords, "a"), (famousLastWords, "b"), (famousLastWords, "c"), (famousLastWords, "d"), (famousLastWords, "e"), (famousLastWords, "f")) { (a: String, b: String, c: String, d: String, e: String, f: String) =>
        i += 1
        i should not be 5
      }
    }
  }

  // Same thing, but set maxDiscarded to 5 with param, prop fails after 5
  test("generator-driven property that takes 6 args, which succeeds, with maxDiscarded param set to 5") {

    var i = 0
    forAll (maxDiscarded(5)) { (a: String, b: String, c: String, d: String, e: String, f: String) =>
      i += 1
      whenever (i > 5) { 1 + 1 should equal (2) }
    }
  }

  test("generator-driven property that takes 6 args, which fails, with maxDiscarded param set to 5") {

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      var i = 0
      forAll (maxDiscarded(5)) { (a: String, b: String, c: String, d: String, e: String, f: String) =>
        i += 1
        whenever (i > 6) { 1 + 1 should equal (2) }
      }
    }
  }

  test("generator-driven property that takes 6 named args, which succeeds, with maxDiscarded param set to 5") {

    var i = 0
    forAll ("a", "b", "c", "d", "e", "f", maxDiscarded(5)) { (a: String, b: String, c: String, d: String, e: String, f: String) =>
      i += 1
      whenever (i > 5) { 1 + 1 should equal (2) }
    }
  }

  test("generator-driven property that takes 6 named args, which fails, with maxDiscarded param set to 5") {

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      var i = 0
      forAll ("a", "b", "c", "d", "e", "f", maxDiscarded(5)) { (a: String, b: String, c: String, d: String, e: String, f: String) =>
        i += 1
        whenever (i > 6) { 1 + 1 should equal (2) }
      }
    }
  }

  test("generator-driven property that takes 6 args and generators, which succeeds, with maxDiscarded param set to 5") {

    var i = 0
    forAll (famousLastWords, famousLastWords, famousLastWords, famousLastWords, famousLastWords, famousLastWords, maxDiscarded(5)) { (a: String, b: String, c: String, d: String, e: String, f: String) =>
      i += 1
      whenever (i > 5) { 1 + 1 should equal (2) }
    }
  }

  test("generator-driven property that takes 6 args and generators, which fails, with maxDiscarded param set to 5") {

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      var i = 0
      forAll (famousLastWords, famousLastWords, famousLastWords, famousLastWords, famousLastWords, famousLastWords, maxDiscarded(5)) { (a: String, b: String, c: String, d: String, e: String, f: String) =>
        i += 1
        whenever (i > 6) { 1 + 1 should equal (2) }
      }
    }
  }

  test("generator-driven property that takes 6 named args and generators, which succeeds, with maxDiscarded param set to 5") {

    var i = 0
    forAll ((famousLastWords, "a"), (famousLastWords, "b"), (famousLastWords, "c"), (famousLastWords, "d"), (famousLastWords, "e"), (famousLastWords, "f"), maxDiscarded(5)) { (a: String, b: String, c: String, d: String, e: String, f: String) =>
      i += 1
      whenever (i > 5) { 1 + 1 should equal (2) }
    }
  }

  test("generator-driven property that takes 6 named args and generators, which fails, with maxDiscarded param set to 5") {

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      var i = 0
      forAll ((famousLastWords, "a"), (famousLastWords, "b"), (famousLastWords, "c"), (famousLastWords, "d"), (famousLastWords, "e"), (famousLastWords, "f"), maxDiscarded(5)) { (a: String, b: String, c: String, d: String, e: String, f: String) =>
        i += 1
        whenever (i > 6) { 1 + 1 should equal (2) }
      }
    }
  }

  // Same thing, but set default maxDiscarded to 5, prop fails after 5
  test("generator-driven property that takes 6 args, which succeeds, with default maxDiscarded set to 5") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(maxDiscarded = 5)

    var i = 0
    forAll { (a: String, b: String, c: String, d: String, e: String, f: String) =>
      i += 1
      whenever (i > 5) { 1 + 1 should equal (2) }
    }
  }

  test("generator-driven property that takes 6 args, which fails, with default maxDiscarded set to 5") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(maxDiscarded = 5)

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      var i = 0
      forAll { (a: String, b: String, c: String, d: String, e: String, f: String) =>
        i += 1
        whenever (i > 6) { 1 + 1 should equal (2) }
      }
    }
  }

  test("generator-driven property that takes 6 named args, which succeeds, with default maxDiscarded set to 5") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(maxDiscarded = 5)

    var i = 0
    forAll ("a", "b", "c", "d", "e", "f") { (a: String, b: String, c: String, d: String, e: String, f: String) =>
      i += 1
      whenever (i > 5) { 1 + 1 should equal (2) }
    }
  }

  test("generator-driven property that takes 6 named args, which fails, with default maxDiscarded set to 5") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(maxDiscarded = 5)

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      var i = 0
      forAll ("a", "b", "c", "d", "e", "f") { (a: String, b: String, c: String, d: String, e: String, f: String) =>
        i += 1
        whenever (i > 6) { 1 + 1 should equal (2) }
      }
    }
  }

  test("generator-driven property that takes 6 args and generators, which succeeds, with default maxDiscarded set to 5") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(maxDiscarded = 5)

    var i = 0
    forAll (famousLastWords, famousLastWords, famousLastWords, famousLastWords, famousLastWords, famousLastWords) { (a: String, b: String, c: String, d: String, e: String, f: String) =>
      i += 1
      whenever (i > 5) { 1 + 1 should equal (2) }
    }
  }

  test("generator-driven property that takes 6 args and generators, which fails, with default maxDiscarded set to 5") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(maxDiscarded = 5)

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      var i = 0
      forAll (famousLastWords, famousLastWords, famousLastWords, famousLastWords, famousLastWords, famousLastWords) { (a: String, b: String, c: String, d: String, e: String, f: String) =>
        i += 1
        whenever (i > 6) { 1 + 1 should equal (2) }
      }
    }
  }

  test("generator-driven property that takes 6 named args and generators, which succeeds, with default maxDiscarded set to 5") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(maxDiscarded = 5)

    var i = 0
    forAll ((famousLastWords, "a"), (famousLastWords, "b"), (famousLastWords, "c"), (famousLastWords, "d"), (famousLastWords, "e"), (famousLastWords, "f")) { (a: String, b: String, c: String, d: String, e: String, f: String) =>
      i += 1
      whenever (i > 5) { 1 + 1 should equal (2) }
    }
  }

  test("generator-driven property that takes 6 named args and generators, which fails, with default maxDiscarded set to 5") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(maxDiscarded = 5)

    intercept[GeneratorDrivenPropertyCheckFailedException] {
      var i = 0
      forAll ((famousLastWords, "a"), (famousLastWords, "b"), (famousLastWords, "c"), (famousLastWords, "d"), (famousLastWords, "e"), (famousLastWords, "f")) { (a: String, b: String, c: String, d: String, e: String, f: String) =>
        i += 1
        whenever (i > 6) { 1 + 1 should equal (2) }
      }
    }
  }

  // set minSize > maxSize with (param, param) (intercept IAE)
  test("generator-driven property that takes 6 args, which should throw IAE because maxSize > maxSize, specified as (param, param)") {

    intercept[IllegalArgumentException] {
      forAll (minSize(5), maxSize(4)) { (a: String, b: String, c: String, d: String, e: String, f: String) =>
        1 + 1 should equal (2)
      }
    }
  }

  test("generator-driven property that takes 6 named args, which should throw IAE because maxSize > maxSize, specified as (param, param)") {

    intercept[IllegalArgumentException] {
      forAll ("a", "b", "c", "d", "e", "f", minSize(5), maxSize(4)) { (a: String, b: String, c: String, d: String, e: String, f: String) =>
        1 + 1 should equal (2)
      }
    }
  }

  test("generator-driven property that takes 6 args and generators, which should throw IAE because maxSize > maxSize, specified as (param, param)") {

    intercept[IllegalArgumentException] {
      forAll (famousLastWords, famousLastWords, famousLastWords, famousLastWords, famousLastWords, famousLastWords, minSize(5), maxSize(4)) { (a: String, b: String, c: String, d: String, e: String, f: String) =>
        1 + 1 should equal (2)
      }
    }
  }

  test("generator-driven property that takes 6 named args and generators, which should throw IAE because maxSize > maxSize, specified as (param, param)") {

    intercept[IllegalArgumentException] {
      forAll ((famousLastWords, "a"), (famousLastWords, "b"), (famousLastWords, "c"), (famousLastWords, "d"), (famousLastWords, "e"), (famousLastWords, "f"), minSize(5), maxSize(4)) { (a: String, b: String, c: String, d: String, e: String, f: String) =>
        1 + 1 should equal (2)
      }
    }
  }

  // set minSize > maxSize with (param, default) (intercept IAE)
  test("generator-driven property that takes 6 args, which should throw IAE because maxSize > maxSize, specified as (param, default)") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(maxSize = 4)

    intercept[IllegalArgumentException] {
      forAll (minSize(5)) { (a: String, b: String, c: String, d: String, e: String, f: String) =>
        1 + 1 should equal (2)
      }
    }
  }

  test("generator-driven property that takes 6 named args, which should throw IAE because maxSize > maxSize, specified as (param, default)") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(maxSize = 4)

    intercept[IllegalArgumentException] {
      forAll ("a", "b", "c", "d", "e", "f", minSize(5)) { (a: String, b: String, c: String, d: String, e: String, f: String) =>
        1 + 1 should equal (2)
      }
    }
  }

  test("generator-driven property that takes 6 args and generators, which should throw IAE because maxSize > maxSize, specified as (param, default)") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(maxSize = 4)

    intercept[IllegalArgumentException] {
      forAll (famousLastWords, famousLastWords, famousLastWords, famousLastWords, famousLastWords, famousLastWords, minSize(5)) { (a: String, b: String, c: String, d: String, e: String, f: String) =>
        1 + 1 should equal (2)
      }
    }
  }

  test("generator-driven property that takes 6 named args and generators, which should throw IAE because maxSize > maxSize, specified as (param, default)") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(maxSize = 4)

    intercept[IllegalArgumentException] {
      forAll ((famousLastWords, "a"), (famousLastWords, "b"), (famousLastWords, "c"), (famousLastWords, "d"), (famousLastWords, "e"), (famousLastWords, "f"), minSize(5)) { (a: String, b: String, c: String, d: String, e: String, f: String) =>
        1 + 1 should equal (2)
      }
    }
  }

  // set minSize > maxSize with (default, param) (intercept IAE)
  test("generator-driven property that takes 6 args, which should throw IAE because maxSize > maxSize, specified as (default, param)") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSize = 5)

    intercept[IllegalArgumentException] {
      forAll (maxSize(4)) { (a: String, b: String, c: String, d: String, e: String, f: String) =>
        1 + 1 should equal (2)
      }
    }
  }

  test("generator-driven property that takes 6 named args, which should throw IAE because maxSize > maxSize, specified as (default, param)") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSize = 5)

    intercept[IllegalArgumentException] {
      forAll ("a", "b", "c", "d", "e", "f", maxSize(4)) { (a: String, b: String, c: String, d: String, e: String, f: String) =>
        1 + 1 should equal (2)
      }
    }
  }

  test("generator-driven property that takes 6 args and generators, which should throw IAE because maxSize > maxSize, specified as (default, param)") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSize = 5)

    intercept[IllegalArgumentException] {
      forAll (famousLastWords, famousLastWords, famousLastWords, famousLastWords, famousLastWords, famousLastWords, maxSize(4)) { (a: String, b: String, c: String, d: String, e: String, f: String) =>
        1 + 1 should equal (2)
      }
    }
  }

  test("generator-driven property that takes 6 named args and generators, which should throw IAE because maxSize > maxSize, specified as (default, param)") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSize = 5)

    intercept[IllegalArgumentException] {
      forAll ((famousLastWords, "a"), (famousLastWords, "b"), (famousLastWords, "c"), (famousLastWords, "d"), (famousLastWords, "e"), (famousLastWords, "f"), maxSize(4)) { (a: String, b: String, c: String, d: String, e: String, f: String) =>
        1 + 1 should equal (2)
      }
    }
  }

  // set maxSize with param (ensure always passed with a size less than maxSize)
  test("generator-driven property that takes 6 args, with maxSize specified as param") {

    forAll (maxSize(5)) { (a: String, b: String, c: String, d: String, e: String, f: String) =>
      a.length should be <= 5
      b.length should be <= 5
      c.length should be <= 5
      d.length should be <= 5
      e.length should be <= 5
      f.length should be <= 5
    }
  }

  test("generator-driven property that takes 6 named args, with maxSize specified as param") {

    forAll ("a", "b", "c", "d", "e", "f", maxSize(5)) { (a: String, b: String, c: String, d: String, e: String, f: String) =>
      a.length should be <= 5
      b.length should be <= 5
      c.length should be <= 5
      d.length should be <= 5
      e.length should be <= 5
      f.length should be <= 5
    }
  }

  // set maxSize with default (ensure always passed with a size less than maxSize)
  test("generator-driven property that takes 6 args, with maxSize specified as default") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(maxSize = 5)

    forAll { (a: String, b: String, c: String, d: String, e: String, f: String) =>
      a.length should be <= 5
      b.length should be <= 5
      c.length should be <= 5
      d.length should be <= 5
      e.length should be <= 5
      f.length should be <= 5
    }
  }

  test("generator-driven property that takes 6 named args, with maxSize specified as default") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(maxSize = 5)

    forAll ("a", "b", "c", "d", "e", "f") { (a: String, b: String, c: String, d: String, e: String, f: String) =>
      a.length should be <= 5
      b.length should be <= 5
      c.length should be <= 5
      d.length should be <= 5
      e.length should be <= 5
      f.length should be <= 5
    }
  }
 
  // set minSize == maxSize with (param, param) (ensure always passed with that size)
  test("generator-driven property that takes 6 args and generators, with minSize == maxSize, specified as (param, param)") {

    pendingUntilFixed {
      forAll (fiveFive, fiveFive, fiveFive, fiveFive, fiveFive, fiveFive, minSize(5), maxSize(5)) { (a: String, b: String, c: String, d: String, e: String, f: String) =>
        a should be ("OKAY")
        b should be ("OKAY")
        c should be ("OKAY")
        d should be ("OKAY")
        e should be ("OKAY")
        f should be ("OKAY")
      }
    }
  }

  test("generator-driven property that takes 6 named args and generators, with minSize == maxSize, specified as (param, param)") {

    pendingUntilFixed {
      forAll ((fiveFive, "a"), (fiveFive, "b"), (fiveFive, "c"), (fiveFive, "d"), (fiveFive, "e"), (fiveFive, "f"), minSize(5), maxSize(5)) { (a: String, b: String, c: String, d: String, e: String, f: String) =>
        a should be ("OKAY")
        b should be ("OKAY")
        c should be ("OKAY")
        d should be ("OKAY")
        e should be ("OKAY")
        f should be ("OKAY")
      }
    }
  }

  // set minSize == maxSize with (param, default) (ensure always passed with that size)
  test("generator-driven property that takes 6 args and generators, with minSize == maxSize, specified as (param, default)") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(maxSize = 5)

    pendingUntilFixed {
      forAll (fiveFive, fiveFive, fiveFive, fiveFive, fiveFive, fiveFive, minSize(5)) { (a: String, b: String, c: String, d: String, e: String, f: String) =>
        a should be ("OKAY")
        b should be ("OKAY")
        c should be ("OKAY")
        d should be ("OKAY")
        e should be ("OKAY")
        f should be ("OKAY")
      }
    }
  }

  test("generator-driven property that takes 6 named args and generators, with minSize == maxSize, specified as (param, default)") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(maxSize = 5)

    pendingUntilFixed {
      forAll ((fiveFive, "a"), (fiveFive, "b"), (fiveFive, "c"), (fiveFive, "d"), (fiveFive, "e"), (fiveFive, "f"), minSize(5)) { (a: String, b: String, c: String, d: String, e: String, f: String) =>
        a should be ("OKAY")
        b should be ("OKAY")
        c should be ("OKAY")
        d should be ("OKAY")
        e should be ("OKAY")
        f should be ("OKAY")
      }
    }
  }

  // set minSize == maxSize with (default, param) (ensure always passed with that size)
  test("generator-driven property that takes 6 args and generators, with minSize == maxSize, specified as (default, param)") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSize = 5)

    pendingUntilFixed {
      forAll (fiveFive, fiveFive, fiveFive, fiveFive, fiveFive, fiveFive, maxSize(5)) { (a: String, b: String, c: String, d: String, e: String, f: String) =>
        a should be ("OKAY")
        b should be ("OKAY")
        c should be ("OKAY")
        d should be ("OKAY")
        e should be ("OKAY")
        f should be ("OKAY")
      }
    }
  }

  test("generator-driven property that takes 6 named args and generators, with minSize == maxSize, specified as (default, param)") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSize = 5)

    pendingUntilFixed {
      forAll ((fiveFive, "a"), (fiveFive, "b"), (fiveFive, "c"), (fiveFive, "d"), (fiveFive, "e"), (fiveFive, "f"), maxSize(5)) { (a: String, b: String, c: String, d: String, e: String, f: String) =>
        a should be ("OKAY")
        b should be ("OKAY")
        c should be ("OKAY")
        d should be ("OKAY")
        e should be ("OKAY")
        f should be ("OKAY")
      }
    }
  }

  // set minSize == maxSize with (default, default) (ensure always passed with that size)
  test("generator-driven property that takes 6 args and generators, with minSize == maxSize, specified as (default, default)") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSize = 5, maxSize = 5)

    pendingUntilFixed {
      forAll (fiveFive, fiveFive, fiveFive, fiveFive, fiveFive, fiveFive) { (a: String, b: String, c: String, d: String, e: String, f: String) =>
        a should be ("OKAY")
        b should be ("OKAY")
        c should be ("OKAY")
        d should be ("OKAY")
        e should be ("OKAY")
        f should be ("OKAY")
      }
    }
  }

  test("generator-driven property that takes 6 named args and generators, with minSize == maxSize, specified as (default, default)") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSize = 5, maxSize = 5)

    pendingUntilFixed {
      forAll ((fiveFive, "a"), (fiveFive, "b"), (fiveFive, "c"), (fiveFive, "d"), (fiveFive, "e"), (fiveFive, "f")) { (a: String, b: String, c: String, d: String, e: String, f: String) =>
        a should be ("OKAY")
        b should be ("OKAY")
        c should be ("OKAY")
        d should be ("OKAY")
        e should be ("OKAY")
        f should be ("OKAY")
      }
    }
  }

  // set minSize to 7 and maxSize to 11 with (param, param) (ensure always passed with that size)
  test("generator-driven property that takes 6 args and generators, with minSize to 7 and maxSize to 11, specified as (param, param)") {

    pendingUntilFixed {
      forAll (sevenEleven, sevenEleven, sevenEleven, sevenEleven, sevenEleven, sevenEleven, minSize(7), maxSize(11)) { (a: String, b: String, c: String, d: String, e: String, f: String) =>
        a should be ("OKAY")
        b should be ("OKAY")
        c should be ("OKAY")
        d should be ("OKAY")
        e should be ("OKAY")
        f should be ("OKAY")
      }
    }
  }

  test("generator-driven property that takes 6 named args and generators, with minSize to 7 and maxSize to 11, specified as (param, param)") {

    pendingUntilFixed {
      forAll ((sevenEleven, "a"), (sevenEleven, "b"), (sevenEleven, "c"), (sevenEleven, "d"), (sevenEleven, "e"), (sevenEleven, "f"), minSize(7), maxSize(11)) { (a: String, b: String, c: String, d: String, e: String, f: String) =>
        a should be ("OKAY")
        b should be ("OKAY")
        c should be ("OKAY")
        d should be ("OKAY")
        e should be ("OKAY")
        f should be ("OKAY")
      }
    }
  }

  // set minSize to 7 and maxSize to 11 with (param, default) (ensure always passed with that size)
  test("generator-driven property that takes 6 args and generators, with minSize to 7 and maxSize to 11, specified as (param, default)") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(maxSize = 11)

    pendingUntilFixed {
      forAll (sevenEleven, sevenEleven, sevenEleven, sevenEleven, sevenEleven, sevenEleven, minSize(7)) { (a: String, b: String, c: String, d: String, e: String, f: String) =>
        a should be ("OKAY")
        b should be ("OKAY")
        c should be ("OKAY")
        d should be ("OKAY")
        e should be ("OKAY")
        f should be ("OKAY")
      }
    }
  }

  test("generator-driven property that takes 6 named args and generators, with minSize to 7 and maxSize to 11, specified as (param, default)") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(maxSize = 11)

    pendingUntilFixed {
      forAll ((sevenEleven, "a"), (sevenEleven, "b"), (sevenEleven, "c"), (sevenEleven, "d"), (sevenEleven, "e"), (sevenEleven, "f"), minSize(7)) { (a: String, b: String, c: String, d: String, e: String, f: String) =>
        a should be ("OKAY")
        b should be ("OKAY")
        c should be ("OKAY")
        d should be ("OKAY")
        e should be ("OKAY")
        f should be ("OKAY")
      }
    }
  }

  // set minSize to 7 and maxSize to 11 with (default, param) (ensure always passed with that size)
  test("generator-driven property that takes 6 args and generators, with minSize to 7 and maxSize to 11, specified as (default, param)") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSize = 7)

    pendingUntilFixed {
      forAll (sevenEleven, sevenEleven, sevenEleven, sevenEleven, sevenEleven, sevenEleven, maxSize(11)) { (a: String, b: String, c: String, d: String, e: String, f: String) =>
        a should be ("OKAY")
        b should be ("OKAY")
        c should be ("OKAY")
        d should be ("OKAY")
        e should be ("OKAY")
        f should be ("OKAY")
      }
    }
  }

  test("generator-driven property that takes 6 named args and generators, with minSize to 7 and maxSize to 11, specified as (default, param)") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSize = 7)

    pendingUntilFixed {
      forAll ((sevenEleven, "a"), (sevenEleven, "b"), (sevenEleven, "c"), (sevenEleven, "d"), (sevenEleven, "e"), (sevenEleven, "f"), maxSize(11)) { (a: String, b: String, c: String, d: String, e: String, f: String) =>
        a should be ("OKAY")
        b should be ("OKAY")
        c should be ("OKAY")
        d should be ("OKAY")
        e should be ("OKAY")
        f should be ("OKAY")
      }
    }
  }

  // set minSize to 7 and maxSize to 11 with (default, default) (ensure always passed with that size)
  test("generator-driven property that takes 6 args and generators, with minSize to 7 and maxSize to 11, specified as (default, default)") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSize = 7, maxSize = 11)

    pendingUntilFixed {
      forAll (sevenEleven, sevenEleven, sevenEleven, sevenEleven, sevenEleven, sevenEleven) { (a: String, b: String, c: String, d: String, e: String, f: String) =>
        a should be ("OKAY")
        b should be ("OKAY")
        c should be ("OKAY")
        d should be ("OKAY")
        e should be ("OKAY")
        f should be ("OKAY")
      }
    }
  }

  test("generator-driven property that takes 6 named args and generators, with minSize to 7 and maxSize to 11, specified as (default, default)") {

    // Hides the member
    implicit val generatorDrivenConfig = PropertyCheckConfig(minSize = 7, maxSize = 11)

    pendingUntilFixed {
      forAll ((sevenEleven, "a"), (sevenEleven, "b"), (sevenEleven, "c"), (sevenEleven, "d"), (sevenEleven, "e"), (sevenEleven, "f")) { (a: String, b: String, c: String, d: String, e: String, f: String) =>
        a should be ("OKAY")
        b should be ("OKAY")
        c should be ("OKAY")
        d should be ("OKAY")
        e should be ("OKAY")
        f should be ("OKAY")
      }
    }
  }
}
