/*
 * Copyright 2001-2012 Artima, Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.scalatest
package prop


import matchers.ShouldMatchers
/* Uncomment this when remove the deprecated type aliases in the org.scalatest.prop package object.
import exceptions.TableDrivenPropertyCheckFailedException
*/

class TableSuite extends FunSuite with TableDrivenPropertyChecks with ShouldMatchers {

  test("table for 1 that succeeds") {

    val examples =
      Table(
        ("a"),
        (  1)
      )

    forAll (examples) { (a) => a should equal (1) }
  }

  test("table for 1, which succeeds even though DiscardedEvaluationException is thrown") {
    val numbers =
      Table(
        ("a"),
        ( -1),
        (  1)
      )

    forAll (numbers) { (a) =>

      whenever (a > 0) {
        a should be > 0
      }
    }
  }

  test("table for 1, which fails") {

    val examples =
      Table(
        ("a"),
        (  2)
      )

    intercept[TableDrivenPropertyCheckFailedException] {
      forAll (examples) { (a) => a should equal (1) }
    }
  }

  test("table for 1 apply, length, and iterator methods work correctly") {

    val examples =
      Table(
        ("a"),
        (  0),
        (  1),
        (  2),
        (  3),
        (  4),
        (  5),
        (  6),
        (  7),
        (  8),
        (  9)
      )

    for (i <- 0 to 9) {
      examples(i) should equal (i)
    }

    examples.length should equal (10)

    var i = 0
    for (example <- examples.iterator) {
      example should equal (i)
      i += 1
    }

    examples.iterator.length should equal (10)
  }

  test("table for 2 that succeeds") {

    val examples =
      Table(
        ("a", "b"),
        (  1,   1),
        (  1,   1)
      )

    forAll (examples) { (a, b) => a + b should equal (2) }
  }

  test("table for 2, which succeeds even though DiscardedEvaluationException is thrown") {
    val numbers =
      Table(
        ("a", "b"),
        ( -1,  -1),
        (  1,   1),
        (  1,   1)
      )

    forAll (numbers) { (a, b) =>

      whenever (a > 0) {
        a should be > 0
      }
    }
  }

  test("table for 2, which fails") {

    val examples =
      Table(
        ("a", "b"),
        (  2,   2),
        (  2,   2)
      )

    intercept[TableDrivenPropertyCheckFailedException] {
      forAll (examples) { (a, b) => a + b should equal (2) }
    }
  }

  test("table for 2 apply, length, and iterator methods work correctly") {

    val examples =
      Table(
        ("a", "b"),
        (  0,   0),
        (  1,   1),
        (  2,   2),
        (  3,   3),
        (  4,   4),
        (  5,   5),
        (  6,   6),
        (  7,   7),
        (  8,   8),
        (  9,   9)
      )

    for (i <- 0 to 9) {
      examples(i) should equal (i, i)
    }

    examples.length should equal (10)

    var i = 0
    for (example <- examples.iterator) {
      example should equal (i, i)
      i += 1
    }

    examples.iterator.length should equal (10)
  }

  test("table for 3 that succeeds") {

    val examples =
      Table(
        ("a", "b", "c"),
        (  1,   1,   1),
        (  1,   1,   1),
        (  1,   1,   1)
      )

    forAll (examples) { (a, b, c) => a + b + c should equal (3) }
  }

  test("table for 3, which succeeds even though DiscardedEvaluationException is thrown") {
    val numbers =
      Table(
        ("a", "b", "c"),
        ( -1,  -1,  -1),
        (  1,   1,   1),
        (  1,   1,   1),
        (  1,   1,   1)
      )

    forAll (numbers) { (a, b, c) =>

      whenever (a > 0) {
        a should be > 0
      }
    }
  }

  test("table for 3, which fails") {

    val examples =
      Table(
        ("a", "b", "c"),
        (  2,   2,   2),
        (  2,   2,   2),
        (  2,   2,   2)
      )

    intercept[TableDrivenPropertyCheckFailedException] {
      forAll (examples) { (a, b, c) => a + b + c should equal (3) }
    }
  }

  test("table for 3 apply, length, and iterator methods work correctly") {

    val examples =
      Table(
        ("a", "b", "c"),
        (  0,   0,   0),
        (  1,   1,   1),
        (  2,   2,   2),
        (  3,   3,   3),
        (  4,   4,   4),
        (  5,   5,   5),
        (  6,   6,   6),
        (  7,   7,   7),
        (  8,   8,   8),
        (  9,   9,   9)
      )

    for (i <- 0 to 9) {
      examples(i) should equal (i, i, i)
    }

    examples.length should equal (10)

    var i = 0
    for (example <- examples.iterator) {
      example should equal (i, i, i)
      i += 1
    }

    examples.iterator.length should equal (10)
  }

  test("table for 4 that succeeds") {

    val examples =
      Table(
        ("a", "b", "c", "d"),
        (  1,   1,   1,   1),
        (  1,   1,   1,   1),
        (  1,   1,   1,   1),
        (  1,   1,   1,   1)
      )

    forAll (examples) { (a, b, c, d) => a + b + c + d should equal (4) }
  }

  test("table for 4, which succeeds even though DiscardedEvaluationException is thrown") {
    val numbers =
      Table(
        ("a", "b", "c", "d"),
        ( -1,  -1,  -1,  -1),
        (  1,   1,   1,   1),
        (  1,   1,   1,   1),
        (  1,   1,   1,   1),
        (  1,   1,   1,   1)
      )

    forAll (numbers) { (a, b, c, d) =>

      whenever (a > 0) {
        a should be > 0
      }
    }
  }

  test("table for 4, which fails") {

    val examples =
      Table(
        ("a", "b", "c", "d"),
        (  2,   2,   2,   2),
        (  2,   2,   2,   2),
        (  2,   2,   2,   2),
        (  2,   2,   2,   2)
      )

    intercept[TableDrivenPropertyCheckFailedException] {
      forAll (examples) { (a, b, c, d) => a + b + c + d should equal (4) }
    }
  }

  test("table for 4 apply, length, and iterator methods work correctly") {

    val examples =
      Table(
        ("a", "b", "c", "d"),
        (  0,   0,   0,   0),
        (  1,   1,   1,   1),
        (  2,   2,   2,   2),
        (  3,   3,   3,   3),
        (  4,   4,   4,   4),
        (  5,   5,   5,   5),
        (  6,   6,   6,   6),
        (  7,   7,   7,   7),
        (  8,   8,   8,   8),
        (  9,   9,   9,   9)
      )

    for (i <- 0 to 9) {
      examples(i) should equal (i, i, i, i)
    }

    examples.length should equal (10)

    var i = 0
    for (example <- examples.iterator) {
      example should equal (i, i, i, i)
      i += 1
    }

    examples.iterator.length should equal (10)
  }

  test("table for 5 that succeeds") {

    val examples =
      Table(
        ("a", "b", "c", "d", "e"),
        (  1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1)
      )

    forAll (examples) { (a, b, c, d, e) => a + b + c + d + e should equal (5) }
  }

  test("table for 5, which succeeds even though DiscardedEvaluationException is thrown") {
    val numbers =
      Table(
        ("a", "b", "c", "d", "e"),
        ( -1,  -1,  -1,  -1,  -1),
        (  1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1)
      )

    forAll (numbers) { (a, b, c, d, e) =>

      whenever (a > 0) {
        a should be > 0
      }
    }
  }

  test("table for 5, which fails") {

    val examples =
      Table(
        ("a", "b", "c", "d", "e"),
        (  2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2)
      )

    intercept[TableDrivenPropertyCheckFailedException] {
      forAll (examples) { (a, b, c, d, e) => a + b + c + d + e should equal (5) }
    }
  }

  test("table for 5 apply, length, and iterator methods work correctly") {

    val examples =
      Table(
        ("a", "b", "c", "d", "e"),
        (  0,   0,   0,   0,   0),
        (  1,   1,   1,   1,   1),
        (  2,   2,   2,   2,   2),
        (  3,   3,   3,   3,   3),
        (  4,   4,   4,   4,   4),
        (  5,   5,   5,   5,   5),
        (  6,   6,   6,   6,   6),
        (  7,   7,   7,   7,   7),
        (  8,   8,   8,   8,   8),
        (  9,   9,   9,   9,   9)
      )

    for (i <- 0 to 9) {
      examples(i) should equal (i, i, i, i, i)
    }

    examples.length should equal (10)

    var i = 0
    for (example <- examples.iterator) {
      example should equal (i, i, i, i, i)
      i += 1
    }

    examples.iterator.length should equal (10)
  }

  test("table for 6 that succeeds") {

    val examples =
      Table(
        ("a", "b", "c", "d", "e", "f"),
        (  1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1)
      )

    forAll (examples) { (a, b, c, d, e, f) => a + b + c + d + e + f should equal (6) }
  }

  test("table for 6, which succeeds even though DiscardedEvaluationException is thrown") {
    val numbers =
      Table(
        ("a", "b", "c", "d", "e", "f"),
        ( -1,  -1,  -1,  -1,  -1,  -1),
        (  1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1)
      )

    forAll (numbers) { (a, b, c, d, e, f) =>

      whenever (a > 0) {
        a should be > 0
      }
    }
  }

  test("table for 6, which fails") {

    val examples =
      Table(
        ("a", "b", "c", "d", "e", "f"),
        (  2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2)
      )

    intercept[TableDrivenPropertyCheckFailedException] {
      forAll (examples) { (a, b, c, d, e, f) => a + b + c + d + e + f should equal (6) }
    }
  }

  test("table for 6 apply, length, and iterator methods work correctly") {

    val examples =
      Table(
        ("a", "b", "c", "d", "e", "f"),
        (  0,   0,   0,   0,   0,   0),
        (  1,   1,   1,   1,   1,   1),
        (  2,   2,   2,   2,   2,   2),
        (  3,   3,   3,   3,   3,   3),
        (  4,   4,   4,   4,   4,   4),
        (  5,   5,   5,   5,   5,   5),
        (  6,   6,   6,   6,   6,   6),
        (  7,   7,   7,   7,   7,   7),
        (  8,   8,   8,   8,   8,   8),
        (  9,   9,   9,   9,   9,   9)
      )

    for (i <- 0 to 9) {
      examples(i) should equal (i, i, i, i, i, i)
    }

    examples.length should equal (10)

    var i = 0
    for (example <- examples.iterator) {
      example should equal (i, i, i, i, i, i)
      i += 1
    }

    examples.iterator.length should equal (10)
  }

  test("table for 7 that succeeds") {

    val examples =
      Table(
        ("a", "b", "c", "d", "e", "f", "g"),
        (  1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1)
      )

    forAll (examples) { (a, b, c, d, e, f, g) => a + b + c + d + e + f + g should equal (7) }
  }

  test("table for 7, which succeeds even though DiscardedEvaluationException is thrown") {
    val numbers =
      Table(
        ("a", "b", "c", "d", "e", "f", "g"),
        ( -1,  -1,  -1,  -1,  -1,  -1,  -1),
        (  1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1)
      )

    forAll (numbers) { (a, b, c, d, e, f, g) =>

      whenever (a > 0) {
        a should be > 0
      }
    }
  }

  test("table for 7, which fails") {

    val examples =
      Table(
        ("a", "b", "c", "d", "e", "f", "g"),
        (  2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2)
      )

    intercept[TableDrivenPropertyCheckFailedException] {
      forAll (examples) { (a, b, c, d, e, f, g) => a + b + c + d + e + f + g should equal (7) }
    }
  }

  test("table for 7 apply, length, and iterator methods work correctly") {

    val examples =
      Table(
        ("a", "b", "c", "d", "e", "f", "g"),
        (  0,   0,   0,   0,   0,   0,   0),
        (  1,   1,   1,   1,   1,   1,   1),
        (  2,   2,   2,   2,   2,   2,   2),
        (  3,   3,   3,   3,   3,   3,   3),
        (  4,   4,   4,   4,   4,   4,   4),
        (  5,   5,   5,   5,   5,   5,   5),
        (  6,   6,   6,   6,   6,   6,   6),
        (  7,   7,   7,   7,   7,   7,   7),
        (  8,   8,   8,   8,   8,   8,   8),
        (  9,   9,   9,   9,   9,   9,   9)
      )

    for (i <- 0 to 9) {
      examples(i) should equal (i, i, i, i, i, i, i)
    }

    examples.length should equal (10)

    var i = 0
    for (example <- examples.iterator) {
      example should equal (i, i, i, i, i, i, i)
      i += 1
    }

    examples.iterator.length should equal (10)
  }

  test("table for 8 that succeeds") {

    val examples =
      Table(
        ("a", "b", "c", "d", "e", "f", "g", "h"),
        (  1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1)
      )

    forAll (examples) { (a, b, c, d, e, f, g, h) => a + b + c + d + e + f + g + h should equal (8) }
  }

  test("table for 8, which succeeds even though DiscardedEvaluationException is thrown") {
    val numbers =
      Table(
        ("a", "b", "c", "d", "e", "f", "g", "h"),
        ( -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1),
        (  1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1)
      )

    forAll (numbers) { (a, b, c, d, e, f, g, h) =>

      whenever (a > 0) {
        a should be > 0
      }
    }
  }

  test("table for 8, which fails") {

    val examples =
      Table(
        ("a", "b", "c", "d", "e", "f", "g", "h"),
        (  2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2)
      )

    intercept[TableDrivenPropertyCheckFailedException] {
      forAll (examples) { (a, b, c, d, e, f, g, h) => a + b + c + d + e + f + g + h should equal (8) }
    }
  }

  test("table for 8 apply, length, and iterator methods work correctly") {

    val examples =
      Table(
        ("a", "b", "c", "d", "e", "f", "g", "h"),
        (  0,   0,   0,   0,   0,   0,   0,   0),
        (  1,   1,   1,   1,   1,   1,   1,   1),
        (  2,   2,   2,   2,   2,   2,   2,   2),
        (  3,   3,   3,   3,   3,   3,   3,   3),
        (  4,   4,   4,   4,   4,   4,   4,   4),
        (  5,   5,   5,   5,   5,   5,   5,   5),
        (  6,   6,   6,   6,   6,   6,   6,   6),
        (  7,   7,   7,   7,   7,   7,   7,   7),
        (  8,   8,   8,   8,   8,   8,   8,   8),
        (  9,   9,   9,   9,   9,   9,   9,   9)
      )

    for (i <- 0 to 9) {
      examples(i) should equal (i, i, i, i, i, i, i, i)
    }

    examples.length should equal (10)

    var i = 0
    for (example <- examples.iterator) {
      example should equal (i, i, i, i, i, i, i, i)
      i += 1
    }

    examples.iterator.length should equal (10)
  }

  test("table for 9 that succeeds") {

    val examples =
      Table(
        ("a", "b", "c", "d", "e", "f", "g", "h", "i"),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1)
      )

    forAll (examples) { (a, b, c, d, e, f, g, h, i) => a + b + c + d + e + f + g + h + i should equal (9) }
  }

  test("table for 9, which succeeds even though DiscardedEvaluationException is thrown") {
    val numbers =
      Table(
        ("a", "b", "c", "d", "e", "f", "g", "h", "i"),
        ( -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1)
      )

    forAll (numbers) { (a, b, c, d, e, f, g, h, i) =>

      whenever (a > 0) {
        a should be > 0
      }
    }
  }

  test("table for 9, which fails") {

    val examples =
      Table(
        ("a", "b", "c", "d", "e", "f", "g", "h", "i"),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2)
      )

    intercept[TableDrivenPropertyCheckFailedException] {
      forAll (examples) { (a, b, c, d, e, f, g, h, i) => a + b + c + d + e + f + g + h + i should equal (9) }
    }
  }

  test("table for 9 apply, length, and iterator methods work correctly") {

    val examples =
      Table(
        ("a", "b", "c", "d", "e", "f", "g", "h", "i"),
        (  0,   0,   0,   0,   0,   0,   0,   0,   0),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  3,   3,   3,   3,   3,   3,   3,   3,   3),
        (  4,   4,   4,   4,   4,   4,   4,   4,   4),
        (  5,   5,   5,   5,   5,   5,   5,   5,   5),
        (  6,   6,   6,   6,   6,   6,   6,   6,   6),
        (  7,   7,   7,   7,   7,   7,   7,   7,   7),
        (  8,   8,   8,   8,   8,   8,   8,   8,   8),
        (  9,   9,   9,   9,   9,   9,   9,   9,   9)
      )

    for (i <- 0 to 9) {
      examples(i) should equal (i, i, i, i, i, i, i, i, i)
    }

    examples.length should equal (10)

    var i = 0
    for (example <- examples.iterator) {
      example should equal (i, i, i, i, i, i, i, i, i)
      i += 1
    }

    examples.iterator.length should equal (10)
  }

  test("table for 10 that succeeds") {

    val examples =
      Table(
        ("a", "b", "c", "d", "e", "f", "g", "h", "i", "j"),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1)
      )

    forAll (examples) { (a, b, c, d, e, f, g, h, i, j) => a + b + c + d + e + f + g + h + i + j should equal (10) }
  }

  test("table for 10, which succeeds even though DiscardedEvaluationException is thrown") {
    val numbers =
      Table(
        ("a", "b", "c", "d", "e", "f", "g", "h", "i", "j"),
        ( -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1)
      )

    forAll (numbers) { (a, b, c, d, e, f, g, h, i, j) =>

      whenever (a > 0) {
        a should be > 0
      }
    }
  }

  test("table for 10, which fails") {

    val examples =
      Table(
        ("a", "b", "c", "d", "e", "f", "g", "h", "i", "j"),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2)
      )

    intercept[TableDrivenPropertyCheckFailedException] {
      forAll (examples) { (a, b, c, d, e, f, g, h, i, j) => a + b + c + d + e + f + g + h + i + j should equal (10) }
    }
  }

  test("table for 10 apply, length, and iterator methods work correctly") {

    val examples =
      Table(
        ("a", "b", "c", "d", "e", "f", "g", "h", "i", "j"),
        (  0,   0,   0,   0,   0,   0,   0,   0,   0,   0),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  3,   3,   3,   3,   3,   3,   3,   3,   3,   3),
        (  4,   4,   4,   4,   4,   4,   4,   4,   4,   4),
        (  5,   5,   5,   5,   5,   5,   5,   5,   5,   5),
        (  6,   6,   6,   6,   6,   6,   6,   6,   6,   6),
        (  7,   7,   7,   7,   7,   7,   7,   7,   7,   7),
        (  8,   8,   8,   8,   8,   8,   8,   8,   8,   8),
        (  9,   9,   9,   9,   9,   9,   9,   9,   9,   9)
      )

    for (i <- 0 to 9) {
      examples(i) should equal (i, i, i, i, i, i, i, i, i, i)
    }

    examples.length should equal (10)

    var i = 0
    for (example <- examples.iterator) {
      example should equal (i, i, i, i, i, i, i, i, i, i)
      i += 1
    }

    examples.iterator.length should equal (10)
  }

  test("table for 11 that succeeds") {

    val examples =
      Table(
        ("a", "b", "c", "d", "e", "f", "g", "h", "i", "j", "k"),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1)
      )

    forAll (examples) { (a, b, c, d, e, f, g, h, i, j, k) => a + b + c + d + e + f + g + h + i + j + k should equal (11) }
  }

  test("table for 11, which succeeds even though DiscardedEvaluationException is thrown") {
    val numbers =
      Table(
        ("a", "b", "c", "d", "e", "f", "g", "h", "i", "j", "k"),
        ( -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1)
      )

    forAll (numbers) { (a, b, c, d, e, f, g, h, i, j, k) =>

      whenever (a > 0) {
        a should be > 0
      }
    }
  }

  test("table for 11, which fails") {

    val examples =
      Table(
        ("a", "b", "c", "d", "e", "f", "g", "h", "i", "j", "k"),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2)
      )

    intercept[TableDrivenPropertyCheckFailedException] {
      forAll (examples) { (a, b, c, d, e, f, g, h, i, j, k) => a + b + c + d + e + f + g + h + i + j + k should equal (11) }
    }
  }

  test("table for 11 apply, length, and iterator methods work correctly") {

    val examples =
      Table(
        ("a", "b", "c", "d", "e", "f", "g", "h", "i", "j", "k"),
        (  0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  3,   3,   3,   3,   3,   3,   3,   3,   3,   3,   3),
        (  4,   4,   4,   4,   4,   4,   4,   4,   4,   4,   4),
        (  5,   5,   5,   5,   5,   5,   5,   5,   5,   5,   5),
        (  6,   6,   6,   6,   6,   6,   6,   6,   6,   6,   6),
        (  7,   7,   7,   7,   7,   7,   7,   7,   7,   7,   7),
        (  8,   8,   8,   8,   8,   8,   8,   8,   8,   8,   8),
        (  9,   9,   9,   9,   9,   9,   9,   9,   9,   9,   9)
      )

    for (i <- 0 to 9) {
      examples(i) should equal (i, i, i, i, i, i, i, i, i, i, i)
    }

    examples.length should equal (10)

    var i = 0
    for (example <- examples.iterator) {
      example should equal (i, i, i, i, i, i, i, i, i, i, i)
      i += 1
    }

    examples.iterator.length should equal (10)
  }

  test("table for 12 that succeeds") {

    val examples =
      Table(
        ("a", "b", "c", "d", "e", "f", "g", "h", "i", "j", "k", "l"),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1)
      )

    forAll (examples) { (a, b, c, d, e, f, g, h, i, j, k, l) => a + b + c + d + e + f + g + h + i + j + k + l should equal (12) }
  }

  test("table for 12, which succeeds even though DiscardedEvaluationException is thrown") {
    val numbers =
      Table(
        ("a", "b", "c", "d", "e", "f", "g", "h", "i", "j", "k", "l"),
        ( -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1)
      )

    forAll (numbers) { (a, b, c, d, e, f, g, h, i, j, k, l) =>

      whenever (a > 0) {
        a should be > 0
      }
    }
  }

  test("table for 12, which fails") {

    val examples =
      Table(
        ("a", "b", "c", "d", "e", "f", "g", "h", "i", "j", "k", "l"),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2)
      )

    intercept[TableDrivenPropertyCheckFailedException] {
      forAll (examples) { (a, b, c, d, e, f, g, h, i, j, k, l) => a + b + c + d + e + f + g + h + i + j + k + l should equal (12) }
    }
  }

  test("table for 12 apply, length, and iterator methods work correctly") {

    val examples =
      Table(
        ("a", "b", "c", "d", "e", "f", "g", "h", "i", "j", "k", "l"),
        (  0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  3,   3,   3,   3,   3,   3,   3,   3,   3,   3,   3,   3),
        (  4,   4,   4,   4,   4,   4,   4,   4,   4,   4,   4,   4),
        (  5,   5,   5,   5,   5,   5,   5,   5,   5,   5,   5,   5),
        (  6,   6,   6,   6,   6,   6,   6,   6,   6,   6,   6,   6),
        (  7,   7,   7,   7,   7,   7,   7,   7,   7,   7,   7,   7),
        (  8,   8,   8,   8,   8,   8,   8,   8,   8,   8,   8,   8),
        (  9,   9,   9,   9,   9,   9,   9,   9,   9,   9,   9,   9)
      )

    for (i <- 0 to 9) {
      examples(i) should equal (i, i, i, i, i, i, i, i, i, i, i, i)
    }

    examples.length should equal (10)

    var i = 0
    for (example <- examples.iterator) {
      example should equal (i, i, i, i, i, i, i, i, i, i, i, i)
      i += 1
    }

    examples.iterator.length should equal (10)
  }

  test("table for 13 that succeeds") {

    val examples =
      Table(
        ("a", "b", "c", "d", "e", "f", "g", "h", "i", "j", "k", "l", "m"),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1)
      )

    forAll (examples) { (a, b, c, d, e, f, g, h, i, j, k, l, m) => a + b + c + d + e + f + g + h + i + j + k + l + m should equal (13) }
  }

  test("table for 13, which succeeds even though DiscardedEvaluationException is thrown") {
    val numbers =
      Table(
        ("a", "b", "c", "d", "e", "f", "g", "h", "i", "j", "k", "l", "m"),
        ( -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1)
      )

    forAll (numbers) { (a, b, c, d, e, f, g, h, i, j, k, l, m) =>

      whenever (a > 0) {
        a should be > 0
      }
    }
  }

  test("table for 13, which fails") {

    val examples =
      Table(
        ("a", "b", "c", "d", "e", "f", "g", "h", "i", "j", "k", "l", "m"),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2)
      )

    intercept[TableDrivenPropertyCheckFailedException] {
      forAll (examples) { (a, b, c, d, e, f, g, h, i, j, k, l, m) => a + b + c + d + e + f + g + h + i + j + k + l + m should equal (13) }
    }
  }

  test("table for 13 apply, length, and iterator methods work correctly") {

    val examples =
      Table(
        ("a", "b", "c", "d", "e", "f", "g", "h", "i", "j", "k", "l", "m"),
        (  0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  3,   3,   3,   3,   3,   3,   3,   3,   3,   3,   3,   3,   3),
        (  4,   4,   4,   4,   4,   4,   4,   4,   4,   4,   4,   4,   4),
        (  5,   5,   5,   5,   5,   5,   5,   5,   5,   5,   5,   5,   5),
        (  6,   6,   6,   6,   6,   6,   6,   6,   6,   6,   6,   6,   6),
        (  7,   7,   7,   7,   7,   7,   7,   7,   7,   7,   7,   7,   7),
        (  8,   8,   8,   8,   8,   8,   8,   8,   8,   8,   8,   8,   8),
        (  9,   9,   9,   9,   9,   9,   9,   9,   9,   9,   9,   9,   9)
      )

    for (i <- 0 to 9) {
      examples(i) should equal (i, i, i, i, i, i, i, i, i, i, i, i, i)
    }

    examples.length should equal (10)

    var i = 0
    for (example <- examples.iterator) {
      example should equal (i, i, i, i, i, i, i, i, i, i, i, i, i)
      i += 1
    }

    examples.iterator.length should equal (10)
  }

  test("table for 14 that succeeds") {

    val examples =
      Table(
        ("a", "b", "c", "d", "e", "f", "g", "h", "i", "j", "k", "l", "m", "n"),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1)
      )

    forAll (examples) { (a, b, c, d, e, f, g, h, i, j, k, l, m, n) => a + b + c + d + e + f + g + h + i + j + k + l + m + n should equal (14) }
  }

  test("table for 14, which succeeds even though DiscardedEvaluationException is thrown") {
    val numbers =
      Table(
        ("a", "b", "c", "d", "e", "f", "g", "h", "i", "j", "k", "l", "m", "n"),
        ( -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1)
      )

    forAll (numbers) { (a, b, c, d, e, f, g, h, i, j, k, l, m, n) =>

      whenever (a > 0) {
        a should be > 0
      }
    }
  }

  test("table for 14, which fails") {

    val examples =
      Table(
        ("a", "b", "c", "d", "e", "f", "g", "h", "i", "j", "k", "l", "m", "n"),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2)
      )

    intercept[TableDrivenPropertyCheckFailedException] {
      forAll (examples) { (a, b, c, d, e, f, g, h, i, j, k, l, m, n) => a + b + c + d + e + f + g + h + i + j + k + l + m + n should equal (14) }
    }
  }

  test("table for 14 apply, length, and iterator methods work correctly") {

    val examples =
      Table(
        ("a", "b", "c", "d", "e", "f", "g", "h", "i", "j", "k", "l", "m", "n"),
        (  0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  3,   3,   3,   3,   3,   3,   3,   3,   3,   3,   3,   3,   3,   3),
        (  4,   4,   4,   4,   4,   4,   4,   4,   4,   4,   4,   4,   4,   4),
        (  5,   5,   5,   5,   5,   5,   5,   5,   5,   5,   5,   5,   5,   5),
        (  6,   6,   6,   6,   6,   6,   6,   6,   6,   6,   6,   6,   6,   6),
        (  7,   7,   7,   7,   7,   7,   7,   7,   7,   7,   7,   7,   7,   7),
        (  8,   8,   8,   8,   8,   8,   8,   8,   8,   8,   8,   8,   8,   8),
        (  9,   9,   9,   9,   9,   9,   9,   9,   9,   9,   9,   9,   9,   9)
      )

    for (i <- 0 to 9) {
      examples(i) should equal (i, i, i, i, i, i, i, i, i, i, i, i, i, i)
    }

    examples.length should equal (10)

    var i = 0
    for (example <- examples.iterator) {
      example should equal (i, i, i, i, i, i, i, i, i, i, i, i, i, i)
      i += 1
    }

    examples.iterator.length should equal (10)
  }

  test("table for 15 that succeeds") {

    val examples =
      Table(
        ("a", "b", "c", "d", "e", "f", "g", "h", "i", "j", "k", "l", "m", "n", "o"),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1)
      )

    forAll (examples) { (a, b, c, d, e, f, g, h, i, j, k, l, m, n, o) => a + b + c + d + e + f + g + h + i + j + k + l + m + n + o should equal (15) }
  }

  test("table for 15, which succeeds even though DiscardedEvaluationException is thrown") {
    val numbers =
      Table(
        ("a", "b", "c", "d", "e", "f", "g", "h", "i", "j", "k", "l", "m", "n", "o"),
        ( -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1)
      )

    forAll (numbers) { (a, b, c, d, e, f, g, h, i, j, k, l, m, n, o) =>

      whenever (a > 0) {
        a should be > 0
      }
    }
  }

  test("table for 15, which fails") {

    val examples =
      Table(
        ("a", "b", "c", "d", "e", "f", "g", "h", "i", "j", "k", "l", "m", "n", "o"),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2)
      )

    intercept[TableDrivenPropertyCheckFailedException] {
      forAll (examples) { (a, b, c, d, e, f, g, h, i, j, k, l, m, n, o) => a + b + c + d + e + f + g + h + i + j + k + l + m + n + o should equal (15) }
    }
  }

  test("table for 15 apply, length, and iterator methods work correctly") {

    val examples =
      Table(
        ("a", "b", "c", "d", "e", "f", "g", "h", "i", "j", "k", "l", "m", "n", "o"),
        (  0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  3,   3,   3,   3,   3,   3,   3,   3,   3,   3,   3,   3,   3,   3,   3),
        (  4,   4,   4,   4,   4,   4,   4,   4,   4,   4,   4,   4,   4,   4,   4),
        (  5,   5,   5,   5,   5,   5,   5,   5,   5,   5,   5,   5,   5,   5,   5),
        (  6,   6,   6,   6,   6,   6,   6,   6,   6,   6,   6,   6,   6,   6,   6),
        (  7,   7,   7,   7,   7,   7,   7,   7,   7,   7,   7,   7,   7,   7,   7),
        (  8,   8,   8,   8,   8,   8,   8,   8,   8,   8,   8,   8,   8,   8,   8),
        (  9,   9,   9,   9,   9,   9,   9,   9,   9,   9,   9,   9,   9,   9,   9)
      )

    for (i <- 0 to 9) {
      examples(i) should equal (i, i, i, i, i, i, i, i, i, i, i, i, i, i, i)
    }

    examples.length should equal (10)

    var i = 0
    for (example <- examples.iterator) {
      example should equal (i, i, i, i, i, i, i, i, i, i, i, i, i, i, i)
      i += 1
    }

    examples.iterator.length should equal (10)
  }

  test("table for 16 that succeeds") {

    val examples =
      Table(
        ("a", "b", "c", "d", "e", "f", "g", "h", "i", "j", "k", "l", "m", "n", "o", "p"),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1)
      )

    forAll (examples) { (a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p) => a + b + c + d + e + f + g + h + i + j + k + l + m + n + o + p should equal (16) }
  }

  test("table for 16, which succeeds even though DiscardedEvaluationException is thrown") {
    val numbers =
      Table(
        ("a", "b", "c", "d", "e", "f", "g", "h", "i", "j", "k", "l", "m", "n", "o", "p"),
        ( -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1)
      )

    forAll (numbers) { (a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p) =>

      whenever (a > 0) {
        a should be > 0
      }
    }
  }

  test("table for 16, which fails") {

    val examples =
      Table(
        ("a", "b", "c", "d", "e", "f", "g", "h", "i", "j", "k", "l", "m", "n", "o", "p"),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2)
      )

    intercept[TableDrivenPropertyCheckFailedException] {
      forAll (examples) { (a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p) => a + b + c + d + e + f + g + h + i + j + k + l + m + n + o + p should equal (16) }
    }
  }

  test("table for 16 apply, length, and iterator methods work correctly") {

    val examples =
      Table(
        ("a", "b", "c", "d", "e", "f", "g", "h", "i", "j", "k", "l", "m", "n", "o", "p"),
        (  0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  3,   3,   3,   3,   3,   3,   3,   3,   3,   3,   3,   3,   3,   3,   3,   3),
        (  4,   4,   4,   4,   4,   4,   4,   4,   4,   4,   4,   4,   4,   4,   4,   4),
        (  5,   5,   5,   5,   5,   5,   5,   5,   5,   5,   5,   5,   5,   5,   5,   5),
        (  6,   6,   6,   6,   6,   6,   6,   6,   6,   6,   6,   6,   6,   6,   6,   6),
        (  7,   7,   7,   7,   7,   7,   7,   7,   7,   7,   7,   7,   7,   7,   7,   7),
        (  8,   8,   8,   8,   8,   8,   8,   8,   8,   8,   8,   8,   8,   8,   8,   8),
        (  9,   9,   9,   9,   9,   9,   9,   9,   9,   9,   9,   9,   9,   9,   9,   9)
      )

    for (i <- 0 to 9) {
      examples(i) should equal (i, i, i, i, i, i, i, i, i, i, i, i, i, i, i, i)
    }

    examples.length should equal (10)

    var i = 0
    for (example <- examples.iterator) {
      example should equal (i, i, i, i, i, i, i, i, i, i, i, i, i, i, i, i)
      i += 1
    }

    examples.iterator.length should equal (10)
  }

  test("table for 17 that succeeds") {

    val examples =
      Table(
        ("a", "b", "c", "d", "e", "f", "g", "h", "i", "j", "k", "l", "m", "n", "o", "p", "q"),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1)
      )

    forAll (examples) { (a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q) => a + b + c + d + e + f + g + h + i + j + k + l + m + n + o + p + q should equal (17) }
  }

  test("table for 17, which succeeds even though DiscardedEvaluationException is thrown") {
    val numbers =
      Table(
        ("a", "b", "c", "d", "e", "f", "g", "h", "i", "j", "k", "l", "m", "n", "o", "p", "q"),
        ( -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1)
      )

    forAll (numbers) { (a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q) =>

      whenever (a > 0) {
        a should be > 0
      }
    }
  }

  test("table for 17, which fails") {

    val examples =
      Table(
        ("a", "b", "c", "d", "e", "f", "g", "h", "i", "j", "k", "l", "m", "n", "o", "p", "q"),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2)
      )

    intercept[TableDrivenPropertyCheckFailedException] {
      forAll (examples) { (a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q) => a + b + c + d + e + f + g + h + i + j + k + l + m + n + o + p + q should equal (17) }
    }
  }

  test("table for 17 apply, length, and iterator methods work correctly") {

    val examples =
      Table(
        ("a", "b", "c", "d", "e", "f", "g", "h", "i", "j", "k", "l", "m", "n", "o", "p", "q"),
        (  0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  3,   3,   3,   3,   3,   3,   3,   3,   3,   3,   3,   3,   3,   3,   3,   3,   3),
        (  4,   4,   4,   4,   4,   4,   4,   4,   4,   4,   4,   4,   4,   4,   4,   4,   4),
        (  5,   5,   5,   5,   5,   5,   5,   5,   5,   5,   5,   5,   5,   5,   5,   5,   5),
        (  6,   6,   6,   6,   6,   6,   6,   6,   6,   6,   6,   6,   6,   6,   6,   6,   6),
        (  7,   7,   7,   7,   7,   7,   7,   7,   7,   7,   7,   7,   7,   7,   7,   7,   7),
        (  8,   8,   8,   8,   8,   8,   8,   8,   8,   8,   8,   8,   8,   8,   8,   8,   8),
        (  9,   9,   9,   9,   9,   9,   9,   9,   9,   9,   9,   9,   9,   9,   9,   9,   9)
      )

    for (i <- 0 to 9) {
      examples(i) should equal (i, i, i, i, i, i, i, i, i, i, i, i, i, i, i, i, i)
    }

    examples.length should equal (10)

    var i = 0
    for (example <- examples.iterator) {
      example should equal (i, i, i, i, i, i, i, i, i, i, i, i, i, i, i, i, i)
      i += 1
    }

    examples.iterator.length should equal (10)
  }

  test("table for 18 that succeeds") {

    val examples =
      Table(
        ("a", "b", "c", "d", "e", "f", "g", "h", "i", "j", "k", "l", "m", "n", "o", "p", "q", "r"),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1)
      )

    forAll (examples) { (a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q, r) => a + b + c + d + e + f + g + h + i + j + k + l + m + n + o + p + q + r should equal (18) }
  }

  test("table for 18, which succeeds even though DiscardedEvaluationException is thrown") {
    val numbers =
      Table(
        ("a", "b", "c", "d", "e", "f", "g", "h", "i", "j", "k", "l", "m", "n", "o", "p", "q", "r"),
        ( -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1)
      )

    forAll (numbers) { (a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q, r) =>

      whenever (a > 0) {
        a should be > 0
      }
    }
  }

  test("table for 18, which fails") {

    val examples =
      Table(
        ("a", "b", "c", "d", "e", "f", "g", "h", "i", "j", "k", "l", "m", "n", "o", "p", "q", "r"),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2)
      )

    intercept[TableDrivenPropertyCheckFailedException] {
      forAll (examples) { (a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q, r) => a + b + c + d + e + f + g + h + i + j + k + l + m + n + o + p + q + r should equal (18) }
    }
  }

  test("table for 18 apply, length, and iterator methods work correctly") {

    val examples =
      Table(
        ("a", "b", "c", "d", "e", "f", "g", "h", "i", "j", "k", "l", "m", "n", "o", "p", "q", "r"),
        (  0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  3,   3,   3,   3,   3,   3,   3,   3,   3,   3,   3,   3,   3,   3,   3,   3,   3,   3),
        (  4,   4,   4,   4,   4,   4,   4,   4,   4,   4,   4,   4,   4,   4,   4,   4,   4,   4),
        (  5,   5,   5,   5,   5,   5,   5,   5,   5,   5,   5,   5,   5,   5,   5,   5,   5,   5),
        (  6,   6,   6,   6,   6,   6,   6,   6,   6,   6,   6,   6,   6,   6,   6,   6,   6,   6),
        (  7,   7,   7,   7,   7,   7,   7,   7,   7,   7,   7,   7,   7,   7,   7,   7,   7,   7),
        (  8,   8,   8,   8,   8,   8,   8,   8,   8,   8,   8,   8,   8,   8,   8,   8,   8,   8),
        (  9,   9,   9,   9,   9,   9,   9,   9,   9,   9,   9,   9,   9,   9,   9,   9,   9,   9)
      )

    for (i <- 0 to 9) {
      examples(i) should equal (i, i, i, i, i, i, i, i, i, i, i, i, i, i, i, i, i, i)
    }

    examples.length should equal (10)

    var i = 0
    for (example <- examples.iterator) {
      example should equal (i, i, i, i, i, i, i, i, i, i, i, i, i, i, i, i, i, i)
      i += 1
    }

    examples.iterator.length should equal (10)
  }

  test("table for 19 that succeeds") {

    val examples =
      Table(
        ("a", "b", "c", "d", "e", "f", "g", "h", "i", "j", "k", "l", "m", "n", "o", "p", "q", "r", "s"),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1)
      )

    forAll (examples) { (a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q, r, s) => a + b + c + d + e + f + g + h + i + j + k + l + m + n + o + p + q + r + s should equal (19) }
  }

  test("table for 19, which succeeds even though DiscardedEvaluationException is thrown") {
    val numbers =
      Table(
        ("a", "b", "c", "d", "e", "f", "g", "h", "i", "j", "k", "l", "m", "n", "o", "p", "q", "r", "s"),
        ( -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1)
      )

    forAll (numbers) { (a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q, r, s) =>

      whenever (a > 0) {
        a should be > 0
      }
    }
  }

  test("table for 19, which fails") {

    val examples =
      Table(
        ("a", "b", "c", "d", "e", "f", "g", "h", "i", "j", "k", "l", "m", "n", "o", "p", "q", "r", "s"),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2)
      )

    intercept[TableDrivenPropertyCheckFailedException] {
      forAll (examples) { (a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q, r, s) => a + b + c + d + e + f + g + h + i + j + k + l + m + n + o + p + q + r + s should equal (19) }
    }
  }

  test("table for 19 apply, length, and iterator methods work correctly") {

    val examples =
      Table(
        ("a", "b", "c", "d", "e", "f", "g", "h", "i", "j", "k", "l", "m", "n", "o", "p", "q", "r", "s"),
        (  0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  3,   3,   3,   3,   3,   3,   3,   3,   3,   3,   3,   3,   3,   3,   3,   3,   3,   3,   3),
        (  4,   4,   4,   4,   4,   4,   4,   4,   4,   4,   4,   4,   4,   4,   4,   4,   4,   4,   4),
        (  5,   5,   5,   5,   5,   5,   5,   5,   5,   5,   5,   5,   5,   5,   5,   5,   5,   5,   5),
        (  6,   6,   6,   6,   6,   6,   6,   6,   6,   6,   6,   6,   6,   6,   6,   6,   6,   6,   6),
        (  7,   7,   7,   7,   7,   7,   7,   7,   7,   7,   7,   7,   7,   7,   7,   7,   7,   7,   7),
        (  8,   8,   8,   8,   8,   8,   8,   8,   8,   8,   8,   8,   8,   8,   8,   8,   8,   8,   8),
        (  9,   9,   9,   9,   9,   9,   9,   9,   9,   9,   9,   9,   9,   9,   9,   9,   9,   9,   9)
      )

    for (i <- 0 to 9) {
      examples(i) should equal (i, i, i, i, i, i, i, i, i, i, i, i, i, i, i, i, i, i, i)
    }

    examples.length should equal (10)

    var i = 0
    for (example <- examples.iterator) {
      example should equal (i, i, i, i, i, i, i, i, i, i, i, i, i, i, i, i, i, i, i)
      i += 1
    }

    examples.iterator.length should equal (10)
  }

  test("table for 20 that succeeds") {

    val examples =
      Table(
        ("a", "b", "c", "d", "e", "f", "g", "h", "i", "j", "k", "l", "m", "n", "o", "p", "q", "r", "s", "t"),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1)
      )

    forAll (examples) { (a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q, r, s, t) => a + b + c + d + e + f + g + h + i + j + k + l + m + n + o + p + q + r + s + t should equal (20) }
  }

  test("table for 20, which succeeds even though DiscardedEvaluationException is thrown") {
    val numbers =
      Table(
        ("a", "b", "c", "d", "e", "f", "g", "h", "i", "j", "k", "l", "m", "n", "o", "p", "q", "r", "s", "t"),
        ( -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1)
      )

    forAll (numbers) { (a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q, r, s, t) =>

      whenever (a > 0) {
        a should be > 0
      }
    }
  }

  test("table for 20, which fails") {

    val examples =
      Table(
        ("a", "b", "c", "d", "e", "f", "g", "h", "i", "j", "k", "l", "m", "n", "o", "p", "q", "r", "s", "t"),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2)
      )

    intercept[TableDrivenPropertyCheckFailedException] {
      forAll (examples) { (a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q, r, s, t) => a + b + c + d + e + f + g + h + i + j + k + l + m + n + o + p + q + r + s + t should equal (20) }
    }
  }

  test("table for 20 apply, length, and iterator methods work correctly") {

    val examples =
      Table(
        ("a", "b", "c", "d", "e", "f", "g", "h", "i", "j", "k", "l", "m", "n", "o", "p", "q", "r", "s", "t"),
        (  0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0),
        (  1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1),
        (  2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2),
        (  3,   3,   3,   3,   3,   3,   3,   3,   3,   3,   3,   3,   3,   3,   3,   3,   3,   3,   3,   3),
        (  4,   4,   4,   4,   4,   4,   4,   4,   4,   4,   4,   4,   4,   4,   4,   4,   4,   4,   4,   4),
        (  5,   5,   5,   5,   5,   5,   5,   5,   5,   5,   5,   5,   5,   5,   5,   5,   5,   5,   5,   5),
        (  6,   6,   6,   6,   6,   6,   6,   6,   6,   6,   6,   6,   6,   6,   6,   6,   6,   6,   6,   6),
        (  7,   7,   7,   7,   7,   7,   7,   7,   7,   7,   7,   7,   7,   7,   7,   7,   7,   7,   7,   7),
        (  8,   8,   8,   8,   8,   8,   8,   8,   8,   8,   8,   8,   8,   8,   8,   8,   8,   8,   8,   8),
        (  9,   9,   9,   9,   9,   9,   9,   9,   9,   9,   9,   9,   9,   9,   9,   9,   9,   9,   9,   9)
      )

    for (i <- 0 to 9) {
      examples(i) should equal (i, i, i, i, i, i, i, i, i, i, i, i, i, i, i, i, i, i, i, i)
    }

    examples.length should equal (10)

    var i = 0
    for (example <- examples.iterator) {
      example should equal (i, i, i, i, i, i, i, i, i, i, i, i, i, i, i, i, i, i, i, i)
      i += 1
    }

    examples.iterator.length should equal (10)
  }
}
