/*
 * Decompiled with CFR 0.152.
 */
package org.sdase.commons.server.auth.key;

import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import java.math.BigInteger;
import java.security.KeyFactory;
import java.security.NoSuchAlgorithmException;
import java.security.PublicKey;
import java.security.interfaces.RSAPublicKey;
import java.security.spec.InvalidKeySpecException;
import java.security.spec.RSAPublicKeySpec;
import java.util.Base64;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;
import javax.ws.rs.ProcessingException;
import javax.ws.rs.WebApplicationException;
import javax.ws.rs.client.Client;
import org.apache.commons.lang3.StringUtils;
import org.sdase.commons.server.auth.key.KeyLoadFailedException;
import org.sdase.commons.server.auth.key.KeySource;
import org.sdase.commons.server.auth.key.LoadedPublicKey;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class JwksKeySource
implements KeySource {
    private static final Logger LOGGER = LoggerFactory.getLogger(JwksKeySource.class);
    private final String jwksUri;
    private final Client client;
    private final String requiredIssuer;

    public JwksKeySource(String jwksUri, Client client, String requiredIssuer) {
        this.jwksUri = jwksUri;
        this.client = client;
        this.requiredIssuer = requiredIssuer;
    }

    @Override
    public List<LoadedPublicKey> loadKeysFromSource() {
        try {
            Jwks jwks = (Jwks)this.client.target(this.jwksUri).request(new String[]{"application/json"}).get(Jwks.class);
            return jwks.getKeys().stream().filter(Objects::nonNull).filter(this::isForSigning).filter(this::isRsaKeyType).filter(this::isRsa256Key).map(this::toPublicKey).collect(Collectors.toList());
        }
        catch (KeyLoadFailedException e) {
            throw e;
        }
        catch (WebApplicationException e) {
            try {
                e.getResponse().close();
            }
            catch (ProcessingException ex) {
                LOGGER.warn("Error while loading keys from JWKS while closing response", (Throwable)ex);
            }
            throw new KeyLoadFailedException(e);
        }
        catch (Exception e) {
            throw new KeyLoadFailedException(e);
        }
    }

    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || this.getClass() != o.getClass()) {
            return false;
        }
        JwksKeySource keySource = (JwksKeySource)o;
        return Objects.equals(this.jwksUri, keySource.jwksUri) && Objects.equals(this.client, keySource.client);
    }

    public int hashCode() {
        return Objects.hash(this.jwksUri, this.client);
    }

    public String toString() {
        return "JwksKeySource{jwksUri='" + this.jwksUri + '\'' + '}';
    }

    private boolean isForSigning(Key key) {
        return StringUtils.isBlank((CharSequence)key.getUse()) || "sig".equals(key.getUse());
    }

    private boolean isRsaKeyType(Key key) {
        return "RSA".equals(key.getKty());
    }

    private boolean isRsa256Key(Key key) {
        return StringUtils.isBlank((CharSequence)key.getAlg()) || "RS256".equals(key.getAlg());
    }

    private LoadedPublicKey toPublicKey(Key key) throws KeyLoadFailedException {
        try {
            String kid = key.getKid();
            String keyType = key.getKty();
            KeyFactory keyFactory = KeyFactory.getInstance(keyType);
            BigInteger modulus = JwksKeySource.readBase64AsBigInt(key.getN());
            BigInteger exponent = JwksKeySource.readBase64AsBigInt(key.getE());
            PublicKey publicKey = keyFactory.generatePublic(new RSAPublicKeySpec(modulus, exponent));
            if (publicKey instanceof RSAPublicKey) {
                return new LoadedPublicKey(kid, (RSAPublicKey)publicKey, this, this.requiredIssuer);
            }
            throw new KeyLoadFailedException("Only RSA keys are supported but loaded a " + publicKey.getClass() + " from " + this.jwksUri);
        }
        catch (NullPointerException | NoSuchAlgorithmException | InvalidKeySpecException e) {
            throw new KeyLoadFailedException(e);
        }
    }

    private static BigInteger readBase64AsBigInt(String encodedBigInt) {
        return new BigInteger(1, Base64.getUrlDecoder().decode(encodedBigInt));
    }

    @JsonIgnoreProperties(ignoreUnknown=true)
    private static class Key {
        private String kid;
        private String kty;
        private String alg;
        private String use;
        private String n;
        private String e;

        private Key() {
        }

        public String getKid() {
            return this.kid;
        }

        public Key setKid(String kid) {
            this.kid = kid;
            return this;
        }

        public String getKty() {
            return this.kty;
        }

        public Key setKty(String kty) {
            this.kty = kty;
            return this;
        }

        public String getAlg() {
            return this.alg;
        }

        public Key setAlg(String alg) {
            this.alg = alg;
            return this;
        }

        public String getUse() {
            return this.use;
        }

        public Key setUse(String use) {
            this.use = use;
            return this;
        }

        public String getN() {
            return this.n;
        }

        public Key setN(String n) {
            this.n = n;
            return this;
        }

        public String getE() {
            return this.e;
        }

        public Key setE(String e) {
            this.e = e;
            return this;
        }
    }

    @JsonIgnoreProperties(ignoreUnknown=true)
    private static class Jwks {
        private List<Key> keys;

        private Jwks() {
        }

        public List<Key> getKeys() {
            return this.keys;
        }

        public Jwks setKeys(List<Key> keys) {
            this.keys = keys;
            return this;
        }
    }
}

