/*
 * Decompiled with CFR 0.152.
 */
package org.sdase.commons.server.auth.key;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.URI;
import java.nio.charset.StandardCharsets;
import java.security.KeyFactory;
import java.security.NoSuchAlgorithmException;
import java.security.PublicKey;
import java.security.cert.Certificate;
import java.security.cert.CertificateException;
import java.security.cert.CertificateFactory;
import java.security.cert.X509Certificate;
import java.security.interfaces.RSAPublicKey;
import java.security.spec.InvalidKeySpecException;
import java.security.spec.X509EncodedKeySpec;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import org.apache.commons.codec.binary.Base64;
import org.sdase.commons.server.auth.key.KeyLoadFailedException;
import org.sdase.commons.server.auth.key.KeySource;
import org.sdase.commons.server.auth.key.LoadedPublicKey;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class PemKeySource
implements KeySource {
    private static final Logger LOG = LoggerFactory.getLogger(PemKeySource.class);
    private final String kid;
    private final URI pemKeyLocation;
    private final String requiredIssuer;

    public PemKeySource(String kid, URI pemKeyLocation, String requiredIssuer) {
        this.kid = kid;
        this.pemKeyLocation = pemKeyLocation;
        this.requiredIssuer = requiredIssuer;
    }

    @Override
    public List<LoadedPublicKey> loadKeysFromSource() {
        try {
            LOG.info("Loading public key for token signature verification from PEM {}", (Object)this.pemKeyLocation);
            if (this.isPublicKey(this.pemKeyLocation)) {
                RSAPublicKey publicKey = this.loadPublicKey(this.pemKeyLocation);
                return Collections.singletonList(new LoadedPublicKey(this.kid, publicKey, this, this.requiredIssuer));
            }
            X509Certificate cer = this.loadCertificate(this.pemKeyLocation);
            RSAPublicKey publicKey = this.extractRsaPublicKeyFromCertificate(cer);
            LOG.info("Loaded public key for token signature verification from PEM {}", (Object)this.pemKeyLocation);
            return Collections.singletonList(new LoadedPublicKey(this.kid, publicKey, this, this.requiredIssuer));
        }
        catch (IOException | ClassCastException | NullPointerException | CertificateException e) {
            throw new KeyLoadFailedException("Failed to load public key for token signature verification from PEM " + this.pemKeyLocation, e);
        }
    }

    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || this.getClass() != o.getClass()) {
            return false;
        }
        PemKeySource that = (PemKeySource)o;
        return Objects.equals(this.kid, that.kid) && Objects.equals(this.pemKeyLocation, that.pemKeyLocation);
    }

    public int hashCode() {
        return Objects.hash(this.kid, this.pemKeyLocation);
    }

    public String toString() {
        return "PemKeySource{kid='" + this.kid + '\'' + ", pemKeyLocation=" + this.pemKeyLocation + '}';
    }

    private X509Certificate loadCertificate(URI location) throws CertificateException, IOException {
        Throwable throwable = null;
        try (InputStream is = location.toURL().openStream();){
            CertificateFactory fact = CertificateFactory.getInstance("X.509");
            Certificate certificate = fact.generateCertificate(is);
            if (certificate instanceof X509Certificate) {
                X509Certificate x509Certificate = (X509Certificate)certificate;
                return x509Certificate;
            }
            try {
                throw new KeyLoadFailedException("Only X509Certificate certificates are supported but loaded a " + certificate.getClass() + " from " + this.pemKeyLocation);
            }
            catch (Throwable throwable2) {
                throwable = throwable2;
                throw throwable2;
            }
        }
    }

    private RSAPublicKey extractRsaPublicKeyFromCertificate(X509Certificate certificate) throws KeyLoadFailedException {
        PublicKey cerPublicKey = certificate.getPublicKey();
        if (!(cerPublicKey instanceof RSAPublicKey)) {
            throw new KeyLoadFailedException("Only RSA keys are supported but loaded a " + cerPublicKey.getClass() + " from " + this.pemKeyLocation);
        }
        RSAPublicKey publicKey = (RSAPublicKey)cerPublicKey;
        return publicKey;
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private boolean isPublicKey(URI pemKeyLocation) {
        try (InputStream is = pemKeyLocation.toURL().openStream();){
            String keyContent = this.readContent(is).trim();
            boolean bl = keyContent.startsWith("-----BEGIN PUBLIC KEY-----") && keyContent.endsWith("-----END PUBLIC KEY-----");
            return bl;
        }
        catch (IOException e) {
            throw new KeyLoadFailedException("Failed to read key from " + pemKeyLocation, e);
        }
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private RSAPublicKey loadPublicKey(URI pemKeyLocation) {
        LOG.info("Loading public key for token signature verification from PEM {}", (Object)pemKeyLocation);
        try (InputStream is = pemKeyLocation.toURL().openStream();){
            String pemPublicKeyContent = this.readContent(is);
            String publicKeyPem = pemPublicKeyContent.replace("-----BEGIN PUBLIC KEY-----", "").replaceAll("(\\r\\n|\\r|\\n)", "").replace("-----END PUBLIC KEY-----", "");
            byte[] encoded = Base64.decodeBase64((String)publicKeyPem);
            X509EncodedKeySpec keySpec = new X509EncodedKeySpec(encoded);
            KeyFactory keyFactory = KeyFactory.getInstance("RSA");
            RSAPublicKey rSAPublicKey = (RSAPublicKey)keyFactory.generatePublic(keySpec);
            return rSAPublicKey;
        }
        catch (IOException | ClassCastException | NoSuchAlgorithmException | InvalidKeySpecException e) {
            throw new KeyLoadFailedException("Failed to load public key at " + pemKeyLocation, e);
        }
    }

    private String readContent(InputStream content) throws IOException {
        int bytesRead;
        ByteArrayOutputStream buffer = new ByteArrayOutputStream();
        byte[] data = new byte[1024];
        while ((bytesRead = content.read(data, 0, data.length)) != -1) {
            buffer.write(data, 0, bytesRead);
        }
        return new String(buffer.toByteArray(), StandardCharsets.UTF_8);
    }
}

