/*
 * Decompiled with CFR 0.152.
 */
package org.sejda.sambox.pdmodel.graphics.image;

import java.awt.Graphics2D;
import java.awt.Paint;
import java.awt.Point;
import java.awt.image.BufferedImage;
import java.awt.image.DataBufferByte;
import java.awt.image.Raster;
import java.awt.image.WritableRaster;
import java.io.IOException;
import java.io.InputStream;
import java.nio.ByteBuffer;
import java.util.Arrays;
import javax.imageio.stream.MemoryCacheImageInputStream;
import org.sejda.sambox.cos.COSArray;
import org.sejda.sambox.cos.COSNumber;
import org.sejda.sambox.pdmodel.graphics.color.PDColorSpace;
import org.sejda.sambox.pdmodel.graphics.color.PDDeviceGray;
import org.sejda.sambox.pdmodel.graphics.color.PDIndexed;
import org.sejda.sambox.pdmodel.graphics.image.PDImage;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

final class SampledImageReader {
    private static final Logger LOG = LoggerFactory.getLogger(SampledImageReader.class);

    private SampledImageReader() {
    }

    public static BufferedImage getStencilImage(PDImage pdImage, Paint paint) throws IOException {
        int width = pdImage.getWidth();
        int height = pdImage.getHeight();
        BufferedImage masked = new BufferedImage(width, height, 2);
        Graphics2D g = masked.createGraphics();
        g.setPaint(paint);
        g.fillRect(0, 0, width, height);
        g.dispose();
        WritableRaster raster = masked.getRaster();
        int[] transparent = new int[4];
        try (MemoryCacheImageInputStream iis = new MemoryCacheImageInputStream(pdImage.createInputStream());){
            float[] decode = SampledImageReader.getDecodeArray(pdImage);
            int value = decode[0] < decode[1] ? 1 : 0;
            int rowLen = width / 8;
            if (width % 8 > 0) {
                ++rowLen;
            }
            byte[] buff = new byte[rowLen];
            for (int y = 0; y < height; ++y) {
                int x = 0;
                int readLen = iis.read(buff);
                block6: for (int r = 0; r < rowLen && r < readLen; ++r) {
                    byte byteValue = buff[r];
                    int mask = 128;
                    int shift = 7;
                    for (int i = 0; i < 8; ++i) {
                        int bit = (byteValue & mask) >> shift;
                        mask >>= 1;
                        --shift;
                        if (bit == value) {
                            raster.setPixel(x, y, transparent);
                        }
                        if (++x == width) continue block6;
                    }
                }
                if (readLen == rowLen) continue;
                LOG.warn("premature EOF, image will be incomplete");
                break;
            }
        }
        return masked;
    }

    public static BufferedImage getRGBImage(PDImage pdImage, COSArray colorKey) throws IOException {
        if (pdImage.isEmpty()) {
            throw new IOException("Image stream is empty");
        }
        PDColorSpace colorSpace = pdImage.getColorSpace();
        int numComponents = colorSpace.getNumberOfComponents();
        int width = pdImage.getWidth();
        int height = pdImage.getHeight();
        int bitsPerComponent = pdImage.getBitsPerComponent();
        if (width <= 0 || height <= 0) {
            throw new IOException("image width and height must be positive");
        }
        try {
            if (bitsPerComponent == 1 && colorKey == null && numComponents == 1) {
                return SampledImageReader.from1Bit(pdImage);
            }
            WritableRaster raster = Raster.createBandedRaster(0, width, height, numComponents, new Point(0, 0));
            float[] defaultDecode = pdImage.getColorSpace().getDefaultDecode(8);
            float[] decode = SampledImageReader.getDecodeArray(pdImage);
            if (bitsPerComponent == 8 && colorKey == null && Arrays.equals(decode, defaultDecode)) {
                return SampledImageReader.from8bit(pdImage, raster);
            }
            return SampledImageReader.fromAny(pdImage, raster, colorKey);
        }
        catch (NegativeArraySizeException ex) {
            throw new IOException(ex);
        }
    }

    public static WritableRaster getRawRaster(PDImage pdImage) throws IOException {
        if (pdImage.isEmpty()) {
            throw new IOException("Image stream is empty");
        }
        PDColorSpace colorSpace = pdImage.getColorSpace();
        int numComponents = colorSpace.getNumberOfComponents();
        int width = pdImage.getWidth();
        int height = pdImage.getHeight();
        int bitsPerComponent = pdImage.getBitsPerComponent();
        if (width <= 0 || height <= 0) {
            throw new IOException("image width and height must be positive");
        }
        try {
            int dataBufferType = 0;
            if (bitsPerComponent > 8) {
                dataBufferType = 1;
            }
            WritableRaster raster = Raster.createInterleavedRaster(dataBufferType, width, height, numComponents, new Point(0, 0));
            SampledImageReader.readRasterFromAny(pdImage, raster);
            return raster;
        }
        catch (NegativeArraySizeException ex) {
            throw new IOException(ex);
        }
    }

    private static void readRasterFromAny(PDImage pdImage, WritableRaster raster) throws IOException {
        PDColorSpace colorSpace = pdImage.getColorSpace();
        int numComponents = colorSpace.getNumberOfComponents();
        int bitsPerComponent = pdImage.getBitsPerComponent();
        float[] decode = SampledImageReader.getDecodeArray(pdImage);
        try (MemoryCacheImageInputStream iis = new MemoryCacheImageInputStream(pdImage.createInputStream());){
            boolean isShort;
            int inputWidth = pdImage.getWidth();
            int scanWidth = pdImage.getWidth();
            int scanHeight = pdImage.getHeight();
            float sampleMax = (float)Math.pow(2.0, bitsPerComponent) - 1.0f;
            boolean isIndexed = colorSpace instanceof PDIndexed;
            int padding = 0;
            if (inputWidth * numComponents * bitsPerComponent % 8 > 0) {
                padding = 8 - inputWidth * numComponents * bitsPerComponent % 8;
            }
            boolean bl = isShort = raster.getDataBuffer().getDataType() == 1;
            assert (!isIndexed || !isShort);
            byte[] srcColorValuesBytes = isShort ? null : new byte[numComponents];
            short[] srcColorValuesShort = isShort ? new short[numComponents] : null;
            for (int y = 0; y < scanHeight; ++y) {
                for (int x = 0; x < scanWidth; ++x) {
                    for (int c = 0; c < numComponents; ++c) {
                        int value = (int)iis.readBits(bitsPerComponent);
                        float dMin = decode[c * 2];
                        float dMax = decode[c * 2 + 1];
                        float output = dMin + (float)value * ((dMax - dMin) / sampleMax);
                        if (isIndexed) {
                            srcColorValuesBytes[c] = (byte)Math.round(output);
                            continue;
                        }
                        if (isShort) {
                            int outputShort = Math.round((output - Math.min(dMin, dMax)) / Math.abs(dMax - dMin) * 65535.0f);
                            srcColorValuesShort[c] = (short)outputShort;
                            continue;
                        }
                        int outputByte = Math.round((output - Math.min(dMin, dMax)) / Math.abs(dMax - dMin) * 255.0f);
                        srcColorValuesBytes[c] = (byte)outputByte;
                    }
                    if (isShort) {
                        raster.setDataElements(x, y, srcColorValuesShort);
                        continue;
                    }
                    raster.setDataElements(x, y, srcColorValuesBytes);
                }
                iis.readBits(padding);
            }
        }
    }

    private static BufferedImage from1Bit(PDImage pdImage) throws IOException {
        WritableRaster raster;
        PDColorSpace colorSpace = pdImage.getColorSpace();
        int width = pdImage.getWidth();
        int height = pdImage.getHeight();
        float[] decode = SampledImageReader.getDecodeArray(pdImage);
        BufferedImage bim = null;
        if (colorSpace instanceof PDDeviceGray) {
            bim = new BufferedImage(width, height, 10);
            raster = bim.getRaster();
        } else {
            raster = Raster.createBandedRaster(0, width, height, 1, new Point(0, 0));
        }
        byte[] output = ((DataBufferByte)raster.getDataBuffer()).getData();
        try (InputStream iis = pdImage.createInputStream();){
            int value1;
            int value0;
            boolean isIndexed = colorSpace instanceof PDIndexed;
            int rowLen = width / 8;
            if (width % 8 > 0) {
                ++rowLen;
            }
            if (isIndexed || decode[0] < decode[1]) {
                value0 = 0;
                value1 = -1;
            } else {
                value0 = -1;
                value1 = 0;
            }
            byte[] buff = new byte[rowLen];
            int idx = 0;
            for (int y = 0; y < height; ++y) {
                int x = 0;
                int readLen = iis.read(buff);
                block7: for (int r = 0; r < rowLen && r < readLen; ++r) {
                    byte value = buff[r];
                    int mask = 128;
                    for (int i = 0; i < 8; ++i) {
                        int bit = value & mask;
                        mask >>= 1;
                        int n = output[idx++] = bit == 0 ? value0 : value1;
                        if (++x == width) continue block7;
                    }
                }
                if (readLen == rowLen) continue;
                LOG.warn("premature EOF, image will be incomplete");
                break;
            }
            if (bim != null) {
                BufferedImage bufferedImage = bim;
                return bufferedImage;
            }
            BufferedImage bufferedImage = colorSpace.toRGBImage(raster);
            return bufferedImage;
        }
    }

    private static BufferedImage from8bit(PDImage pdImage, WritableRaster raster) throws IOException {
        byte[][] banks = ((DataBufferByte)raster.getDataBuffer()).getBankData();
        ByteBuffer source = pdImage.asByteBuffer();
        int width = pdImage.getWidth();
        int height = pdImage.getHeight();
        int numComponents = pdImage.getColorSpace().getNumberOfComponents();
        int max = width * height;
        boolean warnedAboutIndexOutOfBounds = false;
        for (int c = 0; c < numComponents; ++c) {
            int sourceOffset = c;
            for (int i = 0; i < max; ++i) {
                if (sourceOffset < source.limit()) {
                    banks[c][i] = source.get(sourceOffset);
                    sourceOffset += numComponents;
                    continue;
                }
                if (!warnedAboutIndexOutOfBounds) {
                    LOG.warn("Tried reading: " + sourceOffset + " but only: " + source.limit() + " available (component: " + c + ")");
                    warnedAboutIndexOutOfBounds = true;
                }
                banks[c][i] = -1;
            }
        }
        return pdImage.getColorSpace().toRGBImage(raster);
    }

    private static BufferedImage fromAny(PDImage pdImage, WritableRaster raster, COSArray colorKey) throws IOException {
        PDColorSpace colorSpace = pdImage.getColorSpace();
        int numComponents = colorSpace.getNumberOfComponents();
        int width = pdImage.getWidth();
        int height = pdImage.getHeight();
        int bitsPerComponent = pdImage.getBitsPerComponent();
        float[] decode = SampledImageReader.getDecodeArray(pdImage);
        try (MemoryCacheImageInputStream iis = new MemoryCacheImageInputStream(pdImage.createInputStream());){
            float sampleMax = (float)Math.pow(2.0, bitsPerComponent) - 1.0f;
            boolean isIndexed = colorSpace instanceof PDIndexed;
            float[] colorKeyRanges = null;
            BufferedImage colorKeyMask = null;
            if (colorKey != null) {
                if (colorKey.size() >= numComponents * 2) {
                    colorKeyRanges = colorKey.toFloatArray();
                    colorKeyMask = new BufferedImage(width, height, 10);
                } else {
                    LOG.warn("colorKey mask size is " + colorKey.size() + ", should be " + numComponents * 2 + ", ignored");
                }
            }
            int padding = 0;
            if (width * numComponents * bitsPerComponent % 8 > 0) {
                padding = 8 - width * numComponents * bitsPerComponent % 8;
            }
            byte[] srcColorValues = new byte[numComponents];
            byte[] alpha = new byte[1];
            for (int y = 0; y < height; ++y) {
                for (int x = 0; x < width; ++x) {
                    boolean isMasked = true;
                    for (int c = 0; c < numComponents; ++c) {
                        int value = (int)iis.readBits(bitsPerComponent);
                        if (colorKeyRanges != null) {
                            isMasked &= (float)value >= colorKeyRanges[c * 2] && (float)value <= colorKeyRanges[c * 2 + 1];
                        }
                        float dMin = decode[c * 2];
                        float dMax = decode[c * 2 + 1];
                        float output = dMin + (float)value * ((dMax - dMin) / sampleMax);
                        if (isIndexed) {
                            srcColorValues[c] = (byte)Math.round(output);
                            continue;
                        }
                        int outputByte = Math.round((output - Math.min(dMin, dMax)) / Math.abs(dMax - dMin) * 255.0f);
                        srcColorValues[c] = (byte)outputByte;
                    }
                    raster.setDataElements(x, y, srcColorValues);
                    if (colorKeyMask == null) continue;
                    alpha[0] = (byte)(isMasked ? 255 : 0);
                    colorKeyMask.getRaster().setDataElements(x, y, alpha);
                }
                iis.readBits(padding);
            }
            BufferedImage rgbImage = colorSpace.toRGBImage(raster);
            if (colorKeyMask != null) {
                BufferedImage bufferedImage = SampledImageReader.applyColorKeyMask(rgbImage, colorKeyMask);
                return bufferedImage;
            }
            BufferedImage bufferedImage = rgbImage;
            return bufferedImage;
        }
    }

    private static BufferedImage applyColorKeyMask(BufferedImage image, BufferedImage mask) {
        int width = image.getWidth();
        int height = image.getHeight();
        BufferedImage masked = new BufferedImage(width, height, 2);
        WritableRaster src = image.getRaster();
        WritableRaster dest = masked.getRaster();
        WritableRaster alpha = mask.getRaster();
        float[] rgb = new float[3];
        float[] rgba = new float[4];
        float[] alphaPixel = null;
        for (int y = 0; y < height; ++y) {
            for (int x = 0; x < width; ++x) {
                src.getPixel(x, y, rgb);
                rgba[0] = rgb[0];
                rgba[1] = rgb[1];
                rgba[2] = rgb[2];
                alphaPixel = alpha.getPixel(x, y, alphaPixel);
                rgba[3] = 255.0f - alphaPixel[0];
                dest.setPixel(x, y, rgba);
            }
        }
        return masked;
    }

    private static float[] getDecodeArray(PDImage pdImage) throws IOException {
        COSArray cosDecode = pdImage.getDecode();
        float[] decode = null;
        if (cosDecode != null) {
            int numberOfComponents = pdImage.getColorSpace().getNumberOfComponents();
            if (cosDecode.size() != numberOfComponents * 2) {
                if (pdImage.isStencil() && cosDecode.size() >= 2 && cosDecode.get(0) instanceof COSNumber && cosDecode.get(1) instanceof COSNumber) {
                    float decode0 = ((COSNumber)cosDecode.get(0)).floatValue();
                    float decode1 = ((COSNumber)cosDecode.get(1)).floatValue();
                    if (decode0 >= 0.0f && decode0 <= 1.0f && decode1 >= 0.0f && decode1 <= 1.0f) {
                        LOG.warn("decode array " + cosDecode + " not compatible with color space, using the first two entries");
                        return new float[]{decode0, decode1};
                    }
                }
                LOG.error("decode array " + cosDecode + " not compatible with color space, using default");
            } else {
                decode = cosDecode.toFloatArray();
            }
        }
        if (decode == null) {
            return pdImage.getColorSpace().getDefaultDecode(pdImage.getBitsPerComponent());
        }
        return decode;
    }
}

