/*
 * Created on 11/gen/2014
 * Copyright 2014 by Andrea Vacondio (andrea.vacondio@gmail.com).
 * This file is part of sejda-itext5.
 *
 * sejda-itext5 is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * sejda-itext5 is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with sejda-itext5.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.sejda.impl.itext5.component;

import java.io.Closeable;
import java.util.HashMap;
import java.util.List;

import org.sejda.model.exception.TaskException;
import org.sejda.model.exception.TaskIOException;

import com.itextpdf.text.pdf.PdfPageLabels;
import com.itextpdf.text.pdf.PdfReader;

/**
 * Provides functionalities to copy pages from a given {@link PdfReader}. Users must take care of opening and closing this component.
 * 
 * @author Andrea Vacondio
 * 
 */
public interface PdfCopier extends Closeable {

    /**
     * Opens the underlying document. From this point pages can be added while other operations might be forbidden (setCompression, etc).
     */
    void open();

    /**
     * Adds the given page extracted from the input reader.
     * 
     * @param reader
     * @param pageNumber
     * @throws TaskException
     */
    void addPage(PdfReader reader, int pageNumber) throws TaskException;

    /**
     * Adds all the pages from the input reader
     * 
     * @param reader
     * @throws TaskException
     */
    void addAllPages(PdfReader reader) throws TaskException;

    /**
     * Adds a blank page to the current underlying document.
     * 
     * @param reader
     *            the reader to use to get page size and rotation.
     */
    void addBlankPage(PdfReader reader) throws TaskException;

    /**
     * Adds a blank page to the current underlying document if the total number of written pages is odd.
     * 
     * @param reader
     *            the reader to use to get page size and rotation.
     */
    void addBlankPageIfOdd(PdfReader reader) throws TaskException;

    /**
     * Enables compression of the xref streams if compress is true.
     * 
     * @param compress
     */
    void setCompression(boolean compress) throws TaskException;

    /**
     * sets the input page labels to the underlying pdf copy.
     * 
     * @param labels
     */
    void setPageLabels(PdfPageLabels labels);

    /**
     * Frees the reader on the underlying pdf copy.
     * 
     * @param reader
     * @throws TaskIOException
     */
    void freeReader(PdfReader reader) throws TaskIOException;

    /**
     * Sets the outline on the document generated by this copier.
     * 
     * @param outline
     */
    void setOutline(List<HashMap<String, Object>> outline);

    /**
     * @return the number of pages this copier has copied
     */
    int getNumberOfCopiedPages();

    /**
     * Tells the copier to merge fields
     */
    void mergeFields();
}
