/*
 * Copyright (c) 2015-2017 Petr Zelenka <petr.zelenka@sellcom.org>.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.sellcom.core.math;

import java.math.BigDecimal;
import java.math.BigInteger;

import org.sellcom.core.Contract;

/**
 * Mathematical operations.
 *
 * @since 1.0
 */
public class MoreMath {

	private MoreMath() {
		// Utility class, not to be instantiated
	}


	/**
	 * Returns the absolute difference of the given numbers.
	 *
	 * @throws IllegalArgumentException if {@code x} is {@code null}
	 * @throws IllegalArgumentException if {@code y} is {@code null}
	 *
	 * @since 1.0
	 */
	public static BigDecimal absDifference(BigDecimal x, BigDecimal y) {
		Contract.checkArgument(x != null, "X must not be null");
		Contract.checkArgument(y != null, "Y must not be null");

		return absExact(subtractExact(x, y));
	}

	/**
	 * Returns the absolute difference of the given numbers.
	 *
	 * @throws IllegalArgumentException if {@code x} is {@code null}
	 * @throws IllegalArgumentException if {@code y} is {@code null}
	 *
	 * @since 1.0
	 */
	public static BigInteger absDifference(BigInteger x, BigInteger y) {
		Contract.checkArgument(x != null, "X must not be null");
		Contract.checkArgument(y != null, "Y must not be null");

		return absExact(subtractExact(x, y));
	}

	/**
	 * Returns the absolute difference of the given numbers.
	 *
	 * @throws IllegalArgumentException if {@code x} is not finite
	 * @throws IllegalArgumentException if {@code y} is not finite
	 *
	 * @since 1.0
	 */
	public static double absDifference(double x, double y) {
		Contract.checkArgument(Double.isFinite(x), "X must be finite: {0}", x);
		Contract.checkArgument(Double.isFinite(y), "Y must be finite: {0}", y);

		return Math.abs(x - y);
	}

	/**
	 * Returns the absolute difference of the given numbers.
	 *
	 * @throws IllegalArgumentException if {@code x} is not finite
	 * @throws IllegalArgumentException if {@code y} is not finite
	 *
	 * @since 1.0
	 */
	public static float absDifference(float x, float y) {
		Contract.checkArgument(Float.isFinite(x), "X must be finite: {0}", x);
		Contract.checkArgument(Float.isFinite(y), "Y must be finite: {0}", y);

		return Math.abs(x - y);
	}

	/**
	 * Returns the absolute difference of the given numbers.
	 *
	 * @throws ArithmeticException if the result overflows an {@code int}.
	 *
	 * @since 1.0
	 */
	public static int absDifference(int x, int y) {
		return absExact(subtractExact(x, y));
	}

	/**
	 * Returns the absolute difference of the given numbers.
	 *
	 * @throws ArithmeticException if the result overflows an {@code int}.
	 *
	 * @since 1.0
	 */
	public static long absDifference(long x, long y) {
		return absExact(subtractExact(x, y));
	}

	/**
	 * Returns the absolute value of the given number.
	 *
	 * @throws IllegalArgumentException if {@code x} is {@code null}
	 *
	 * @since 1.0
	 */
	public static BigDecimal absExact(BigDecimal x) {
		Contract.checkArgument(x != null, "X must not be null");

		return x.abs();
	}

	/**
	 * Returns the absolute value of the given number.
	 *
	 * @throws IllegalArgumentException if {@code x} is {@code null}
	 *
	 * @since 1.0
	 */
	public static BigInteger absExact(BigInteger x) {
		Contract.checkArgument(x != null, "X must not be null");

		return x.abs();
	}

	/**
	 * Returns the absolute value of the given number.
	 *
	 * @throws IllegalArgumentException if {@code x} is {@code NaN}
	 *
	 * @since 1.0
	 */
	public static double absExact(double x) {
		Contract.checkArgument(!Double.isNaN(x), "X must not be NaN");

		return Math.abs(x);
	}

	/**
	 * Returns the absolute value of the given number.
	 *
	 * @throws IllegalArgumentException if {@code x} is {@code NaN}
	 *
	 * @since 1.0
	 */
	public static float absExact(float x) {
		Contract.checkArgument(!Float.isNaN(x), "X must not be NaN");

		return Math.abs(x);
	}

	/**
	 * Returns the absolute value of the given number.
	 *
	 * @throws ArithmeticException if {@code x} is equal to {@code Integer.MIN_VALUE}.
	 *
	 * @since 1.0
	 */
	public static int absExact(int x) {
		Contract.check(x != Integer.MIN_VALUE, ArithmeticException.class, "Integer overflow");

		return Math.abs(x);
	}

	/**
	 * Returns the absolute value of the given number.
	 *
	 * @throws ArithmeticException if {@code x} is equal to {@code Long.MIN_VALUE}.
	 *
	 * @since 1.0
	 */
	public static long absExact(long x) {
		Contract.check(x != Long.MIN_VALUE, ArithmeticException.class, "Integer overflow");

		return Math.abs(x);
	}

	/**
	 * Returns the sum of the given numbers.
	 *
	 * @throws IllegalArgumentException if {@code x} is {@code null}
	 * @throws IllegalArgumentException if {@code y} is {@code null}
	 *
	 * @since 1.0
	 */
	public static BigDecimal addExact(BigDecimal x, BigDecimal y) {
		Contract.checkArgument(x != null, "X must not be null");
		Contract.checkArgument(y != null, "Y must not be null");

		return x.add(y);
	}

	/**
	 * Returns the sum of the given numbers.
	 *
	 * @throws IllegalArgumentException if {@code x} is {@code null}
	 * @throws IllegalArgumentException if {@code y} is {@code null}
	 *
	 * @since 1.0
	 */
	public static BigInteger addExact(BigInteger x, BigInteger y) {
		Contract.checkArgument(x != null, "X must not be null");
		Contract.checkArgument(y != null, "Y must not be null");

		return x.add(y);
	}

	/**
	 * Returns the sum of the given numbers.
	 *
	 * @throws ArithmeticException if the result overflows an {@code int}.
	 *
	 * @since 1.0
	 */
	public static int addExact(int x, int y) {
		int result = x + y;
		if (((x ^ result) & (y ^ result)) < 0) {
			throw new ArithmeticException("Integer overflow");
		}

		return result;
	}

	/**
	 * Returns the sum of the given numbers.
	 *
	 * @throws ArithmeticException if the result overflows a {@code long}.
	 *
	 * @since 1.0
	 */
	public static long addExact(long x, long y) {
		long result = x + y;
		if (((x ^ result) & (y ^ result)) < 0) {
			throw new ArithmeticException("Integer overflow");
		}

		return result;
	}

	/**
	 * Returns the given number decremented by one.
	 *
	 * @throws IllegalArgumentException if {@code x} is {@code null}
	 *
	 * @since 1.0
	 */
	public static BigInteger decrementExact(BigInteger x) {
		Contract.checkArgument(x != null, "X must not be null");

		return x.subtract(BigInteger.ONE);
	}

	/**
	 * Returns the given number decremented by one.
	 *
	 * @throws ArithmeticException if the result overflows an {@code int}.
	 *
	 * @since 1.0
	 */
	public static int decrementExact(int x) {
		Contract.check(x != Integer.MIN_VALUE, ArithmeticException.class, "Integer overflow");

		return x - 1;
	}

	/**
	 * Returns the given number decremented by one.
	 *
	 * @throws ArithmeticException if the result overflows a {@code long}.
	 *
	 * @since 1.0
	 */
	public static long decrementExact(long x) {
		Contract.check(x != Long.MIN_VALUE, ArithmeticException.class, "Integer overflow");

		return x - 1;
	}

	/**
	 * Checks whether the given numbers are considered equal in the context of the given tolerance.
	 *
	 * @throws IllegalArgumentException if {@code x} is not finite
	 * @throws IllegalArgumentException if {@code y} is not finite
	 * @throws IllegalArgumentException if {@code tolerance} is not finite
	 * @throws IllegalArgumentException if {@code tolerance} is negative
	 *
	 * @since 1.0
	 */
	public static boolean equals(double x, double y, double tolerance) {
		Contract.checkArgument(Double.isFinite(tolerance), "Tolerance must be finite: {0}", tolerance);
		Contract.checkArgument(tolerance >= 0, "Tolerance must not be negative: {0}", tolerance);

		return absDifference(x, y) <= tolerance;
	}

	/**
	 * Checks whether the given numbers are considered equal in the context of the given tolerance.
	 *
	 * @throws IllegalArgumentException if {@code x} is not finite
	 * @throws IllegalArgumentException if {@code y} is not finite
	 * @throws IllegalArgumentException if {@code tolerance} is not finite
	 * @throws IllegalArgumentException if {@code tolerance} is negative
	 *
	 * @since 1.0
	 */
	public static boolean equals(float x, float y, float tolerance) {
		Contract.checkArgument(Float.isFinite(tolerance), "Tolerance must be finite: {0}", tolerance);
		Contract.checkArgument(tolerance >= 0, "Tolerance must not be negative: {0}", tolerance);

		return absDifference(x, y) <= tolerance;
	}

	/**
	 * Returns the fractional part of the given number.
	 *
	 * @throws IllegalArgumentException if {@code x} is {@code null}
	 *
	 * @since 1.0
	 */
	public static BigDecimal fractionalPart(BigDecimal x) {
		Contract.checkArgument(x != null, "X must not be null");

		return x.remainder(BigDecimal.ONE);
	}

	/**
	 * Returns the fractional part of the given number.
	 *
	 * @throws IllegalArgumentException if {@code x} is not finite
	 *
	 * @since 1.0
	 */
	public static double fractionalPart(double x) {
		Contract.checkArgument(Double.isFinite(x), "X must be finite: {0}", x);

		return x % 1.0;
	}

	/**
	 * Returns the fractional part of the given number.
	 *
	 * @throws IllegalArgumentException if {@code x} is not finite
	 *
	 * @since 1.0
	 */
	public static float fractionalPart(float x) {
		Contract.checkArgument(Float.isFinite(x), "X must be finite: {0}", x);

		return x % 1.0F;
	}

	/**
	 * Returns the greatest common divisor of the given numbers.
	 *
	 * @throws IllegalArgumentException if {@code x} is {@code null}
	 * @throws IllegalArgumentException if {@code y} is {@code null}
	 *
	 * @since 1.0
	 */
	public static BigInteger gcd(BigInteger x, BigInteger y) {
		Contract.checkArgument(x != null, "X must not be null");
		Contract.checkArgument(y != null, "Y must not be null");

		return x.gcd(y);
	}

	/**
	 * Returns the greatest common divisor of the given numbers.
	 *
	 * @throws ArithmeticException if {@code x} is equal to {@code Integer.MIN_VALUE}.
	 * @throws ArithmeticException if {@code y} is equal to {@code Integer.MIN_VALUE}.
	 *
	 * @since 1.0
	 */
	public static int gcd(int x, int y) { // Stein's algorithm
		if (x < 0) {
			x = absExact(x);
		}
		if (y < 0) {
			y = absExact(y);
		}

		if (x == 0) {
			return y;
		}
		if (y == 0) {
			return x;
		}
		if (x == y) {
			return y;
		}

		int shift = 0;
		while (((x | y) & 1) == 0) {
			x >>= 1;
			y >>= 1;
			shift += 1;
		}

		while ((x & 1) == 0) {
			x >>= 1;
		}

		do {
			while ((y & 1) == 0) {
				y >>= 1;
			}

			if (x > y) {
				int tmp = x;
				x = y;
				y = tmp;
			}

			y = subtractExact(y, x);
		} while (y != 0);

		return x << shift;
	}

	/**
	 * Returns the greatest common divisor of the given numbers.
	 *
	 * @throws ArithmeticException if {@code x} is equal to {@code Long.MIN_VALUE}.
	 * @throws ArithmeticException if {@code y} is equal to {@code Long.MIN_VALUE}.
	 *
	 * @since 1.0
	 */
	public static long gcd(long x, long y) { // Stein's algorithm
		if (x < 0) {
			x = absExact(x);
		}
		if (y < 0) {
			y = absExact(y);
		}

		if (x == 0) {
			return y;
		}
		if (y == 0) {
			return x;
		}
		if (x == y) {
			return y;
		}

		int shift = 0;
		while (((x | y) & 1) == 0) {
			x >>= 1;
			y >>= 1;
			shift += 1;
		}

		while ((x & 1) == 0) {
			x >>= 1;
		}

		do {
			while ((y & 1) == 0) {
				y >>= 1;
			}

			if (x > y) {
				long tmp = x;
				x = y;
				y = tmp;
			}

			y = subtractExact(y, x);
		} while (y != 0);

		return x << shift;
	}

	/**
	 * Returns the given number incremented by one.
	 *
	 * @throws IllegalArgumentException if {@code x} is {@code null}
	 *
	 * @since 1.0
	 */
	public static BigInteger incrementExact(BigInteger x) {
		Contract.checkArgument(x != null, "X must not be null");

		return x.add(BigInteger.ONE);
	}

	/**
	 * Returns the given number incremented by one.
	 *
	 * @throws ArithmeticException if the result overflows an {@code int}.
	 *
	 * @since 1.0
	 */
	public static int incrementExact(int x) {
		Contract.check(x != Integer.MAX_VALUE, ArithmeticException.class, "Integer overflow");

		return x + 1;
	}

	/**
	 * Returns the given number incremented by one.
	 *
	 * @throws ArithmeticException if the result overflows a {@code long}.
	 *
	 * @since 1.0
	 */
	public static long incrementExact(long x) {
		Contract.check(x != Long.MAX_VALUE, ArithmeticException.class, "Integer overflow");

		return x + 1;
	}

	/**
	 * Checks whether the given number is even.
	 *
	 * @throws IllegalArgumentException if {@code x} is {@code null}
	 *
	 * @since 1.0
	 */
	public static boolean isEven(BigInteger x) {
		Contract.checkArgument(x != null, "X must not be null");

		return x.getLowestSetBit() != 0;
	}

	/**
	 * Checks whether the given number is even.
	 *
	 * @since 1.0
	 */
	public static boolean isEven(int x) {
		return (x & 1) == 0;
	}

	/**
	 * Checks whether the given number is even.
	 *
	 * @since 1.0
	 */
	public static boolean isEven(long x) {
		return (x & 1) == 0;
	}

	/**
	 * Checks whether the given number is odd.
	 *
	 * @throws IllegalArgumentException if {@code x} is {@code null}
	 *
	 * @since 1.0
	 */
	public static boolean isOdd(BigInteger x) {
		Contract.checkArgument(x != null, "X must not be null");

		return x.getLowestSetBit() == 1;
	}

	/**
	 * Checks whether the given number is odd.
	 *
	 * @since 1.0
	 */
	public static boolean isOdd(int x) {
		return (x & 1) != 0;
	}

	/**
	 * Checks whether the given number is odd.
	 *
	 * @since 1.0
	 */
	public static boolean isOdd(long x) {
		return (x & 1) != 0;
	}

	/**
	 * Returns the least common multiple of the given numbers.
	 *
	 * @throws IllegalArgumentException if {@code x} is {@code null}
	 * @throws IllegalArgumentException if {@code y} is {@code null}
	 *
	 * @since 1.0
	 */
	public static BigInteger lcm(BigInteger x, BigInteger y) {
		Contract.checkArgument(x != null, "X must not be null");
		Contract.checkArgument(y != null, "Y must not be null");

		if (x.equals(BigInteger.ZERO) && (y.equals(BigInteger.ZERO))) {
			return BigInteger.ZERO;
		} else {
			return (absExact(x).divide(gcd(x, y))).multiply(absExact(y));
		}
	}

	/**
	 * Returns the least common multiple of the given numbers.
	 *
	 * @throws ArithmeticException if {@code x} is equal to {@code Integer.MIN_VALUE}.
	 * @throws ArithmeticException if {@code y} is equal to {@code Integer.MIN_VALUE}.
	 *
	 * @since 1.0
	 */
	public static int lcm(int x, int y) {
		if ((x == 0) && (y == 0)) {
			return 0;
		} else {
			return multiplyExact(absExact(x) / gcd(x, y), absExact(y));
		}
	}

	/**
	 * Returns the least common multiple of the given numbers.
	 *
	 * @throws ArithmeticException if {@code x} is equal to {@code Long.MIN_VALUE}.
	 * @throws ArithmeticException if {@code y} is equal to {@code Long.MIN_VALUE}.
	 *
	 * @since 1.0
	 */
	public static long lcm(long x, long y) {
		if ((x == 0L) && (y == 0L)) {
			return 0L;
		} else {
			return multiplyExact(absExact(x) / gcd(x, y), absExact(y));
		}
	}

	/**
	 * Returns the dual (base {@code 2}) logarithm of the given number.
	 *
	 * @throws IllegalArgumentException if {@code x} is not finite
	 *
	 * @since 1.0
	 */
	public static double ld(double x) {
		Contract.checkArgument(Double.isFinite(x), "X must be finite: {0}", x);

		return Math.log(x) / Math.log(2.0);
	}

	/**
	 * Returns the natural (base {@code e}) logarithm of the given number.
	 *
	 * @throws IllegalArgumentException if {@code x} is not finite
	 *
	 * @since 1.0
	 */
	public static double ln(double x) {
		Contract.checkArgument(Double.isFinite(x), "X must be finite: {0}", x);

		return Math.log(x);
	}

	/**
	 * Returns the product of the given numbers.
	 *
	 * @throws IllegalArgumentException if {@code x} is {@code null}
	 * @throws IllegalArgumentException if {@code y} is {@code null}
	 *
	 * @since 1.0
	 */
	public static BigDecimal multiplyExact(BigDecimal x, BigDecimal y) {
		Contract.checkArgument(x != null, "X must not be null");
		Contract.checkArgument(y != null, "Y must not be null");

		return x.multiply(y);
	}

	/**
	 * Returns the product of the given numbers.
	 *
	 * @throws IllegalArgumentException if {@code x} is {@code null}
	 * @throws IllegalArgumentException if {@code y} is {@code null}
	 *
	 * @since 1.0
	 */
	public static BigInteger multiplyExact(BigInteger x, BigInteger y) {
		Contract.checkArgument(x != null, "X must not be null");
		Contract.checkArgument(y != null, "Y must not be null");

		return x.multiply(y);
	}

	/**
	 * Returns the product of the given numbers.
	 *
	 * @throws ArithmeticException if the result overflows an {@code int}.
	 *
	 * @since 1.0
	 */
	public static int multiplyExact(int x, int y) {
		return toIntExact((long) x * (long) y);
	}

	/**
	 * Returns the product of the given numbers.
	 *
	 * @throws ArithmeticException if the result overflows a {@code long}.
	 *
	 * @since 1.0
	 */
	public static long multiplyExact(long x, long y) {
		long result = x * y;
		if ((((Math.abs(x) | Math.abs(y)) >>> 31) != 0)) {
			if (((y != 0) && ((result / y) != x)) || ((x == Long.MIN_VALUE) && (y == -1))) {
				throw new ArithmeticException("Integer overflow");
			}
		}

		return result;
	}

	/**
	 * Returns the negation of the given number.
	 *
	 * @throws IllegalArgumentException if {@code x} is {@code null}
	 *
	 * @since 1.0
	 */
	public static BigDecimal negateExact(BigDecimal x) {
		Contract.checkArgument(x != null, "X must not be null");

		return x.negate();
	}

	/**
	 * Returns the negation of the given number.
	 *
	 * @throws IllegalArgumentException if {@code x} is {@code null}
	 *
	 * @since 1.0
	 */
	public static BigInteger negateExact(BigInteger x) {
		Contract.checkArgument(x != null, "X must not be null");

		return x.negate();
	}

	/**
	 * Returns the negation of the given number.
	 *
	 * @throws IllegalArgumentException if {@code x} is {@code NaN}
	 *
	 * @since 1.0
	 */
	public static double negateExact(double x) {
		Contract.checkArgument(!Double.isNaN(x), "X must not be NaN");

		return -x;
	}

	/**
	 * Returns the negation of the given number.
	 *
	 * @throws IllegalArgumentException if {@code x} is {@code NaN}
	 *
	 * @since 1.0
	 */
	public static float negateExact(float x) {
		Contract.checkArgument(!Float.isNaN(x), "X must not be NaN");

		return -x;
	}

	/**
	 * Returns the negation of the given number.
	 *
	 * @throws ArithmeticException if {@code x} is equal to {@code Integer.MIN_VALUE}.
	 *
	 * @since 1.0
	 */
	public static int negateExact(int x) {
		Contract.check(x != Integer.MIN_VALUE, ArithmeticException.class, "Integer overflow");

		return -x;
	}

	/**
	 * Returns the negation of the given number.
	 *
	 * @throws ArithmeticException if {@code x} is equal to {@code Long.MIN_VALUE}.
	 *
	 * @since 1.0
	 */
	public static long negateExact(long x) {
		Contract.check(x != Long.MIN_VALUE, ArithmeticException.class, "Integer overflow");

		return -x;
	}

	/**
	 * Returns the signum of the given value.
	 *
	 * @throws IllegalArgumentException if {@code x} is {@code null}
	 *
	 * @since 1.0
	 */
	public static int signum(BigDecimal x) {
		Contract.checkArgument(x != null, "X must not be null");

		return x.signum();
	}

	/**
	 * Returns the signum of the given value.
	 *
	 * @throws IllegalArgumentException if {@code x} is {@code null}
	 *
	 * @since 1.0
	 */
	public static int signum(BigInteger x) {
		Contract.checkArgument(x != null, "X must not be null");

		return x.signum();
	}

	/**
	 * Returns the signum of the given value.
	 *
	 * @throws IllegalArgumentException if {@code x} is {@code NaN}
	 *
	 * @since 1.0
	 */
	public static int signum(double x) {
		Contract.checkArgument(!Double.isNaN(x), "X must not be NaN");

		return toIntExact(Math.signum(x));
	}

	/**
	 * Returns the signum of the given value.
	 *
	 * @throws IllegalArgumentException if {@code x} is {@code NaN}
	 *
	 * @since 1.0
	 */
	public static int signum(float x) {
		Contract.checkArgument(!Float.isNaN(x), "X must not be NaN");

		return toIntExact(Math.signum(x));
	}

	/**
	 * Returns the signum of the given value.
	 *
	 * @since 1.0
	 */
	public static int signum(int x) {
		return Integer.signum(x);
	}

	/**
	 * Returns the signum of the given value.
	 *
	 * @since 1.0
	 */
	public static int signum(long x) {
		return Long.signum(x);
	}

	/**
	 * Returns the difference of the given arguments.
	 *
	 * @throws IllegalArgumentException if {@code x} is {@code null}
	 * @throws IllegalArgumentException if {@code y} is {@code null}
	 *
	 * @since 1.0
	 */
	public static BigDecimal subtractExact(BigDecimal x, BigDecimal y) {
		Contract.checkArgument(x != null, "X must not be null");
		Contract.checkArgument(y != null, "Y must not be null");

		return x.subtract(y);
	}

	/**
	 * Returns the difference of the given arguments.
	 *
	 * @throws IllegalArgumentException if {@code x} is {@code null}
	 * @throws IllegalArgumentException if {@code y} is {@code null}
	 *
	 * @since 1.0
	 */
	public static BigInteger subtractExact(BigInteger x, BigInteger y) {
		Contract.checkArgument(x != null, "X must not be null");
		Contract.checkArgument(y != null, "Y must not be null");

		return x.subtract(y);
	}

	/**
	 * Returns the difference of the given arguments.
	 *
	 * @throws ArithmeticException if the result overflows an {@code int}.
	 *
	 * @since 1.0
	 */
	public static int subtractExact(int x, int y) {
		int result = x - y;
		if (((x ^ y) & (x ^ result)) < 0) {
			throw new ArithmeticException("Integer overflow");
		}

		return result;
	}

	/**
	 * Returns the difference of the given arguments.
	 *
	 * @throws ArithmeticException if the result overflows a {@code long}.
	 *
	 * @since 1.0
	 */
	public static long subtractExact(long x, long y) {
		long result = x - y;
		if (((x ^ y) & (x ^ result)) < 0) {
			throw new ArithmeticException("Integer overflow");
		}

		return result;
	}

	/**
	 * Returns the {@code BigInteger} value of the given number.
	 *
	 * @throws ArithmeticException if {@code x} has a non-zero fractional part.
	 * @throws IllegalArgumentException if {@code x} is {@code null}
	 *
	 * @since 1.0
	 */
	public static BigInteger toBigIntegerExact(BigDecimal x) {
		Contract.checkArgument(x != null, "X must not be null");

		return x.toBigIntegerExact();
	}

	/**
	 * Returns the {@code int} value of the given number.
	 *
	 * @throws ArithmeticException if {@code x} has a non-zero fractional part.
	 * @throws ArithmeticException if {@code x} overflows an {@code int}.
	 * @throws IllegalArgumentException if {@code x} is {@code null}
	 *
	 * @since 1.0
	 */
	public static int toIntExact(BigDecimal x) {
		Contract.checkArgument(x != null, "X must not be null");

		return x.intValueExact();
	}

	/**
	 * Returns the {@code int} value of the given number.
	 *
	 * @throws ArithmeticException if {@code x} overflows an {@code int}.
	 * @throws IllegalArgumentException if {@code x} is {@code null}
	 *
	 * @since 1.0
	 */
	public static int toIntExact(BigInteger x) {
		Contract.checkArgument(x != null, "X must not be null");

		return x.intValueExact();
	}

	/**
	 * Returns the {@code int} value of the given number.
	 *
	 * @throws ArithmeticException if {@code x} has a non-zero fractional part.
	 * @throws ArithmeticException if {@code x} overflows an {@code int}.
	 * @throws IllegalArgumentException if {@code x} is not finite
	 *
	 * @since 1.0
	 */
	public static int toIntExact(double x) {
		Contract.check(fractionalPart(x) == 0.0, ArithmeticException.class, "Not an integer");
		Contract.check((int) x == x, ArithmeticException.class, "Integer overflow");

		return (int) x;
	}

	/**
	 * Returns the {@code int} value of the given number.
	 *
	 * @throws ArithmeticException if {@code x} has a non-zero fractional part.
	 * @throws ArithmeticException if {@code x} overflows an {@code int}.
	 * @throws IllegalArgumentException if {@code x} is not finite
	 *
	 * @since 1.0
	 */
	public static int toIntExact(float x) {
		Contract.check(fractionalPart(x) == 0.0F, ArithmeticException.class, "Not an integer");
		Contract.check((int) x == x, ArithmeticException.class, "Integer overflow");

		return (int) x;
	}

	/**
	 * Returns the {@code int} value of the given number.
	 *
	 * @throws ArithmeticException if {@code x} overflows an {@code int}.
	 *
	 * @since 1.0
	 */
	public static int toIntExact(long x) {
		Contract.check((int) x == x, ArithmeticException.class, "Integer overflow");

		return (int) x;
	}

	/**
	 * Returns the {@code long} value of the given number.
	 *
	 * @throws ArithmeticException if {@code x} has a non-zero fractional part.
	 * @throws ArithmeticException if {@code x} overflows a {@code long}.
	 * @throws IllegalArgumentException if {@code x} is {@code null}
	 *
	 * @since 1.0
	 */
	public static long toLongExact(BigDecimal x) {
		Contract.checkArgument(x != null, "X must not be null");

		return x.longValueExact();
	}

	/**
	 * Returns the {@code long} value of the given number.
	 *
	 * @throws ArithmeticException if {@code x} overflows a {@code long}.
	 * @throws IllegalArgumentException if {@code x} is {@code null}
	 *
	 * @since 1.0
	 */
	public static long toLongExact(BigInteger x) {
		Contract.checkArgument(x != null, "X must not be null");

		return x.longValueExact();
	}

	/**
	 * Returns the {@code long} value of the given number.
	 *
	 * @throws ArithmeticException if {@code x} has a non-zero fractional part.
	 * @throws ArithmeticException if {@code x} overflows a {@code long}.
	 * @throws IllegalArgumentException if {@code x} is not finite
	 *
	 * @since 1.0
	 */
	public static long toLongExact(double x) {
		Contract.check(fractionalPart(x) == 0.0, ArithmeticException.class, "Not an integer");
		Contract.check((int) x == x, ArithmeticException.class, "Integer overflow");

		return (long) x;
	}

	/**
	 * Returns the {@code long} value of the given number.
	 *
	 * @throws ArithmeticException if {@code x} has a non-zero fractional part.
	 * @throws ArithmeticException if {@code x} overflows a {@code long}.
	 * @throws IllegalArgumentException if {@code x} is not finite
	 *
	 * @since 1.0
	 */
	public static long toLongExact(float x) {
		Contract.check(fractionalPart(x) == 0.0F, ArithmeticException.class, "Not an integer");
		Contract.check((int) x == x, ArithmeticException.class, "Integer overflow");

		return (long) x;
	}

	/**
	 * Returns the {@code short} value of the given number.
	 *
	 * @throws ArithmeticException if {@code x} has a non-zero fractional part.
	 * @throws ArithmeticException if {@code x} overflows a {@code short}.
	 * @throws IllegalArgumentException if {@code x} is {@code null}
	 *
	 * @since 1.0
	 */
	public static short toShortExact(BigDecimal x) {
		Contract.checkArgument(x != null, "X must not be null");

		return x.shortValueExact();
	}

	/**
	 * Returns the {@code short} value of the given number.
	 *
	 * @throws ArithmeticException if {@code x} overflows a {@code short}.
	 * @throws IllegalArgumentException if {@code x} is {@code null}
	 *
	 * @since 1.0
	 */
	public static short toShortExact(BigInteger x) {
		Contract.checkArgument(x != null, "X must not be null");

		return x.shortValueExact();
	}

	/**
	 * Returns the {@code short} value of the given number.
	 *
	 * @throws ArithmeticException if {@code x} has a non-zero fractional part.
	 * @throws ArithmeticException if {@code x} overflows a {@code short}.
	 * @throws IllegalArgumentException if {@code x} is not finite
	 *
	 * @since 1.0
	 */
	public static short toShortExact(double x) {
		Contract.check(fractionalPart(x) == 0.0, ArithmeticException.class, "Not an integer");
		Contract.check((int) x == x, ArithmeticException.class, "Integer overflow");

		return (short) x;
	}

	/**
	 * Returns the {@code short} value of the given number.
	 *
	 * @throws ArithmeticException if {@code x} has a non-zero fractional part.
	 * @throws ArithmeticException if {@code x} overflows a {@code short}.
	 * @throws IllegalArgumentException if {@code x} is not finite
	 *
	 * @since 1.0
	 */
	public static short toShortExact(float x) {
		Contract.check(fractionalPart(x) == 0.0F, ArithmeticException.class, "Not an integer");
		Contract.check((int) x == x, ArithmeticException.class, "Integer overflow");

		return (short) x;
	}

	/**
	 * Returns the {@code short} value of the given number.
	 *
	 * @throws ArithmeticException if {@code x} overflows a {@code short}.
	 *
	 * @since 1.0
	 */
	public static short toShortExact(int x) {
		Contract.check((short) x == x, ArithmeticException.class, "Integer overflow");

		return (short) x;
	}

	/**
	 * Returns the {@code short} value of the given number.
	 *
	 * @throws ArithmeticException if {@code x} overflows a {@code short}.
	 *
	 * @since 1.0
	 */
	public static short toShortExact(long x) {
		Contract.check((short) x == x, ArithmeticException.class, "Integer overflow");

		return (short) x;
	}

}
