/*
 * Decompiled with CFR 0.152.
 */
package org.sentrysoftware.ssh;

import com.trilead.ssh2.Connection;
import com.trilead.ssh2.InteractiveCallback;
import com.trilead.ssh2.SCPClient;
import com.trilead.ssh2.SFTPv3Client;
import com.trilead.ssh2.SFTPv3DirectoryEntry;
import com.trilead.ssh2.SFTPv3FileAttributes;
import com.trilead.ssh2.SFTPv3FileHandle;
import com.trilead.ssh2.Session;
import java.io.BufferedReader;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.nio.charset.Charset;
import java.util.Arrays;
import java.util.Optional;
import java.util.Vector;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.sentrysoftware.ssh.Utils;

public class SshClient
implements AutoCloseable {
    private static final Pattern DEFAULT_MASK_PATTERN = Pattern.compile(".*");
    private static final int READ_BUFFER_SIZE = 8192;
    private String hostname;
    private Connection sshConnection = null;
    private Charset charset = null;
    private Session sshSession = null;

    public SshClient(String pHostname) {
        this(pHostname, "");
    }

    public SshClient(String pHostname, String pLocale) {
        this(pHostname, Utils.getCharsetFromLocale(pLocale));
    }

    public SshClient(String pHostname, Charset pCharset) {
        this.hostname = pHostname;
        this.charset = pCharset;
    }

    public void connect() throws IOException {
        this.connect(0);
    }

    public void connect(int timeout) throws IOException {
        this.sshConnection = new Connection(this.hostname);
        this.sshConnection.connect(null, timeout, timeout);
    }

    @Deprecated
    public void disconnect() {
        if (this.sshSession != null) {
            this.sshSession.close();
        }
        if (this.sshConnection != null) {
            this.sshConnection.close();
        }
    }

    @Override
    public void close() {
        if (this.sshSession != null) {
            this.sshSession.close();
        }
        if (this.sshConnection != null) {
            this.sshConnection.close();
        }
    }

    @Deprecated
    public boolean authenticate(String username, String privateKeyFile, String password) throws IOException {
        return this.authenticate(username, privateKeyFile != null ? new File(privateKeyFile) : null, password != null ? password.toCharArray() : null);
    }

    public boolean authenticate(String username, File privateKeyFile, char[] password) throws IOException {
        if (this.sshConnection.isAuthMethodAvailable(username, "publickey")) {
            return this.sshConnection.authenticateWithPublicKey(username, privateKeyFile, password != null ? String.valueOf(password) : null);
        }
        return false;
    }

    @Deprecated
    public boolean authenticate(String username, String password) throws IOException {
        return this.authenticate(username, password != null ? password.toCharArray() : null);
    }

    public boolean authenticate(final String username, final char[] password) throws IOException {
        if (this.sshConnection.isAuthMethodAvailable(username, "password") && this.sshConnection.authenticateWithPassword(username, password != null ? String.valueOf(password) : null)) {
            return true;
        }
        if (this.sshConnection.isAuthMethodAvailable(username, "keyboard-interactive")) {
            return this.sshConnection.authenticateWithKeyboardInteractive(username, new InteractiveCallback(){

                @Override
                public String[] replyToChallenge(String name, String instruction, int numPrompts, String[] prompt, boolean[] echo) throws Exception {
                    String[] challengeResponse = new String[numPrompts];
                    for (int i = 0; i < numPrompts; ++i) {
                        challengeResponse[i] = echo[i] ? username : (password != null ? String.valueOf(password) : null);
                    }
                    return challengeResponse;
                }
            });
        }
        return false;
    }

    public boolean authenticate(String username) throws IOException {
        return this.sshConnection.authenticateWithNone(username);
    }

    public String readFileAttributes(String filePath) throws IOException {
        this.checkIfAuthenticated();
        SFTPv3Client sftpClient = new SFTPv3Client(this.sshConnection);
        SFTPv3FileAttributes fileAttributes = sftpClient.stat(filePath);
        String fileType = fileAttributes.isRegularFile() ? "FILE" : (fileAttributes.isDirectory() ? "DIR" : (fileAttributes.isSymlink() ? "LINK" : "UNKNOWN"));
        StringBuilder pslFileResult = new StringBuilder();
        pslFileResult.append(fileAttributes.mtime.toString()).append("\t").append(fileAttributes.atime.toString()).append("\t-\t").append(Integer.toString(fileAttributes.permissions & 0x1FF, 8)).append("\t").append(fileAttributes.size.toString()).append("\t-\t").append(fileType).append("\t").append(fileAttributes.uid.toString()).append("\t").append(fileAttributes.gid.toString()).append("\t").append(sftpClient.canonicalPath(filePath));
        sftpClient.close();
        return pslFileResult.toString();
    }

    private StringBuilder listSubDirectory(SFTPv3Client sftpClient, String remoteDirectoryPath, Pattern fileMaskPattern, boolean includeSubfolders, Integer depth, StringBuilder resultBuilder) throws IOException {
        if (depth <= 15) {
            Vector pathContents = sftpClient.ls(remoteDirectoryPath);
            if (remoteDirectoryPath.endsWith("/")) {
                remoteDirectoryPath = remoteDirectoryPath.substring(0, remoteDirectoryPath.lastIndexOf("/"));
            }
            Integer n = depth;
            depth = depth + 1;
            for (SFTPv3DirectoryEntry file : pathContents) {
                String filename = file.filename.trim();
                if (filename.equals(".") || filename.equals("..")) continue;
                SFTPv3FileAttributes fileAttributes = file.attributes;
                String filePath = remoteDirectoryPath + "/" + filename;
                if ((fileAttributes.permissions & 0xA000) == 40960) continue;
                if ((fileAttributes.permissions & 0x8000) == 32768 || (fileAttributes.permissions & 0x6000) == 24576 || (fileAttributes.permissions & 0x2000) == 8192 || (fileAttributes.permissions & 0xC000) == 49152) {
                    Matcher m = fileMaskPattern.matcher(filename);
                    if (!m.find()) continue;
                    resultBuilder.append(filePath).append(";").append(fileAttributes.mtime.toString()).append(";").append(fileAttributes.size.toString()).append("\n");
                    continue;
                }
                if ((fileAttributes.permissions & 0x4000) != 16384 || !includeSubfolders) continue;
                resultBuilder = this.listSubDirectory(sftpClient, filePath, fileMaskPattern, includeSubfolders, depth, resultBuilder);
            }
        }
        return resultBuilder;
    }

    public String listFiles(String remoteDirectoryPath, String regExpFileMask, boolean includeSubfolders) throws IOException {
        this.checkIfAuthenticated();
        SFTPv3Client sftpClient = new SFTPv3Client(this.sshConnection);
        Pattern fileMaskPattern = regExpFileMask != null && !regExpFileMask.isEmpty() ? Pattern.compile(regExpFileMask, 2) : DEFAULT_MASK_PATTERN;
        StringBuilder resultBuilder = new StringBuilder();
        this.listSubDirectory(sftpClient, remoteDirectoryPath, fileMaskPattern, includeSubfolders, 1, resultBuilder);
        sftpClient.close();
        return resultBuilder.toString();
    }

    public String readFile(String remoteFilePath, Long readOffset, Integer readSize) throws IOException {
        int bufferSize;
        int bytesRead;
        int remainingBytes;
        this.checkIfAuthenticated();
        SFTPv3Client sftpClient = new SFTPv3Client(this.sshConnection);
        long offset = 0L;
        if (readOffset != null) {
            offset = readOffset;
        }
        SFTPv3FileHandle handle = sftpClient.openFileRO(remoteFilePath);
        if (readSize == null) {
            SFTPv3FileAttributes attributes = sftpClient.fstat(handle);
            if (attributes == null) {
                throw new IOException("Couldn't find file " + remoteFilePath + " and get its attributes");
            }
            remainingBytes = (int)(attributes.size - offset);
            if (remainingBytes < 0) {
                remainingBytes = 0;
            }
        } else {
            remainingBytes = readSize;
        }
        ByteArrayOutputStream out = new ByteArrayOutputStream();
        byte[] readBuffer = new byte[8192];
        while (remainingBytes > 0 && (bytesRead = sftpClient.read(handle, offset, readBuffer, 0, bufferSize = remainingBytes < 8192 ? remainingBytes : 8192)) >= 0) {
            ((OutputStream)out).write(readBuffer, 0, bytesRead);
            remainingBytes -= bytesRead;
            offset += (long)bytesRead;
        }
        sftpClient.closeFile(handle);
        sftpClient.close();
        return ((Object)out).toString();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void removeFile(String[] remoteFilePathArray) throws IOException {
        this.checkIfAuthenticated();
        try (SFTPv3Client sftpClient = null;){
            sftpClient = new SFTPv3Client(this.sshConnection);
            for (String remoteFilePath : remoteFilePathArray) {
                sftpClient.rm(remoteFilePath);
            }
        }
    }

    public void removeFile(String remoteFilePath) throws IOException {
        this.removeFile(new String[]{remoteFilePath});
    }

    public CommandResult executeCommand(String command) throws IOException {
        return this.executeCommand(command, 0);
    }

    public CommandResult executeCommand(String command, int timeout) throws IOException {
        this.openSession();
        InputStream stdout = this.sshSession.getStdout();
        InputStream stderr = this.sshSession.getStderr();
        CommandResult commandResult = new CommandResult();
        try (ByteArrayOutputStream output = new ByteArrayOutputStream();){
            long currentTime;
            long startTime = System.currentTimeMillis();
            long timeoutTime = timeout > 0 ? startTime + (long)timeout : Long.MAX_VALUE;
            this.sshSession.execCommand(command);
            int waitForCondition = 0;
            while (!SshClient.hasSessionClosed(waitForCondition) && !SshClient.hasEndOfFileSession(waitForCondition) && (currentTime = System.currentTimeMillis()) < timeoutTime) {
                waitForCondition = this.waitForNewData(Math.min(timeoutTime - currentTime, 5000L));
                if (SshClient.hasStdoutData(waitForCondition)) {
                    SshClient.transferAllBytes(stdout, output);
                }
                if (!SshClient.hasStderrData(waitForCondition)) continue;
                SshClient.transferAllBytes(stderr, output);
            }
            currentTime = System.currentTimeMillis();
            if (currentTime >= timeoutTime) {
                commandResult.success = false;
                commandResult.result = "Timeout (" + timeout / 1000 + " seconds)";
            } else {
                commandResult.executionTime = (currentTime - startTime) / 1000L;
                waitForCondition = this.sshSession.waitForCondition(32, 5000L);
                if ((waitForCondition & 0x20) != 0) {
                    commandResult.exitStatus = this.sshSession.getExitStatus();
                }
                commandResult.result = new String(output.toByteArray(), this.charset);
            }
        }
        return commandResult;
    }

    public void interactiveSession(InputStream in, OutputStream out) throws IOException, InterruptedException {
        this.openSession();
        this.openTerminal();
        final BufferedReader inputReader = new BufferedReader(new InputStreamReader(in));
        final OutputStream outputWriter = this.sshSession.getStdin();
        Thread stdinPipeThread = new Thread(){

            @Override
            public void run() {
                try {
                    String line;
                    while ((line = inputReader.readLine()) != null) {
                        outputWriter.write(line.getBytes());
                        outputWriter.write(10);
                    }
                }
                catch (Exception exception) {
                    // empty catch block
                }
                SshClient.this.sshSession.close();
            }
        };
        stdinPipeThread.setDaemon(true);
        stdinPipeThread.start();
        InputStream stdout = this.sshSession.getStdout();
        InputStream stderr = this.sshSession.getStderr();
        int waitForCondition = 0;
        while (!SshClient.hasSessionClosed(waitForCondition) && !SshClient.hasEndOfFileSession(waitForCondition)) {
            waitForCondition = this.waitForNewData(5000L);
            if (SshClient.hasStdoutData(waitForCondition)) {
                SshClient.transferAllBytes(stdout, out);
            }
            if (!SshClient.hasStderrData(waitForCondition)) continue;
            SshClient.transferAllBytes(stderr, out);
        }
        if (stdinPipeThread.isAlive()) {
            stdinPipeThread.interrupt();
        }
    }

    public void scp(String localFilePath, String remoteFilename, String remoteDirectory, String fileMode) throws IOException {
        this.checkIfAuthenticated();
        SCPClient scpClient = new SCPClient(this.sshConnection);
        scpClient.put(localFilePath, remoteFilename, remoteDirectory, fileMode);
    }

    public void openSession() throws IOException {
        this.checkIfConnected();
        this.checkIfAuthenticated();
        this.sshSession = this.getSshConnection().openSession();
    }

    public void openTerminal() throws IOException {
        this.checkIfConnected();
        this.checkIfAuthenticated();
        this.checkIfSessionOpened();
        this.getSshSession().requestPTY("dumb", 10000, 24, 640, 480, new byte[]{53, 0, 0, 0, 0, 0});
        this.getSshSession().startShell();
    }

    public void write(String text) throws IOException {
        if (text == null || text.isEmpty()) {
            return;
        }
        this.checkIfConnected();
        this.checkIfAuthenticated();
        this.checkIfSessionOpened();
        Utils.checkNonNullField(this.charset, "charset");
        OutputStream outputStream = this.getSshSession().getStdin();
        Utils.checkNonNullField(outputStream, "Stdin");
        String[] split = text.split("\\R", -1);
        if (split.length == 1) {
            outputStream.write(text.getBytes(this.charset));
        } else {
            for (int i = 0; i < split.length; ++i) {
                if (split[i].length() != 0) {
                    outputStream.write(split[i].getBytes(this.charset));
                }
                if (i >= split.length - 1) continue;
                outputStream.write(10);
            }
        }
        outputStream.flush();
    }

    public Optional<String> read(int size, int timeout) throws IOException {
        Utils.checkArgumentNotZeroOrNegative(timeout, "timeout");
        this.checkIfConnected();
        this.checkIfAuthenticated();
        this.checkIfSessionOpened();
        Utils.checkNonNullField(this.charset, "charset");
        InputStream stdout = this.getSshSession().getStdout();
        InputStream stderr = this.getSshSession().getStderr();
        Utils.checkNonNullField(stdout, "stdout");
        Utils.checkNonNullField(stderr, "stderr");
        try (ByteArrayOutputStream byteArrayOutputStream = new ByteArrayOutputStream();){
            int waitForCondition = this.waitForNewData((long)timeout * 1000L);
            boolean stdoutData = SshClient.hasStdoutData(waitForCondition);
            boolean stderrData = SshClient.hasStderrData(waitForCondition);
            int stdoutRead = 0;
            if (stdoutData) {
                stdoutRead = SshClient.transferBytes(stdout, byteArrayOutputStream, size);
                if (size > 0 && stdoutRead >= size) {
                    Optional<String> optional = Optional.of(new String(byteArrayOutputStream.toByteArray(), this.charset));
                    return optional;
                }
            }
            if (stderrData) {
                SshClient.transferBytes(stderr, byteArrayOutputStream, size - stdoutRead);
            }
            Optional<String> optional = stdoutData || stderrData ? Optional.of(new String(byteArrayOutputStream.toByteArray(), this.charset)) : Optional.empty();
            return optional;
        }
    }

    static boolean hasTimeoutSession(int waitForCondition) {
        return (waitForCondition & 1) != 0;
    }

    static boolean hasEndOfFileSession(int waitForCondition) {
        return (waitForCondition & 0x10) != 0;
    }

    static boolean hasSessionClosed(int waitForCondition) {
        return (waitForCondition & 2) != 0;
    }

    static boolean hasStdoutData(int waitForCondition) {
        return (waitForCondition & 4) != 0;
    }

    static boolean hasStderrData(int waitForCondition) {
        return (waitForCondition & 8) != 0;
    }

    int waitForNewData(long timeout) {
        return this.sshSession.waitForCondition(30, timeout);
    }

    void checkIfConnected() {
        if (this.getSshConnection() == null) {
            throw new IllegalStateException("Connection is required first");
        }
    }

    void checkIfAuthenticated() {
        if (!this.getSshConnection().isAuthenticationComplete()) {
            throw new IllegalStateException("Authentication is required first");
        }
    }

    public void checkIfSessionOpened() {
        if (this.getSshSession() == null) {
            throw new IllegalStateException("SSH session should be opened first");
        }
    }

    static int transferAllBytes(InputStream inputStream, OutputStream outputStream) throws IOException {
        return SshClient.transferBytes(inputStream, outputStream, -1);
    }

    static int transferBytes(InputStream inputStream, OutputStream outputStream, int size) throws IOException {
        int bufferSize = size > 0 && size < 8192 ? size : 8192;
        byte[] buffer = new byte[bufferSize];
        int total = 0;
        int bytesRead = 0;
        while (inputStream.available() > 0 && (bytesRead = inputStream.read(buffer)) > 0) {
            int bytesCopy = Math.min(bytesRead, 8192);
            outputStream.write(Arrays.copyOf(buffer, bytesCopy));
            outputStream.flush();
            if (size <= 0 || (total += bytesRead) < size) continue;
            return total;
        }
        return total;
    }

    Connection getSshConnection() {
        return this.sshConnection;
    }

    Session getSshSession() {
        return this.sshSession;
    }

    public static class CommandResult {
        public boolean success = true;
        public float executionTime = 0.0f;
        public Integer exitStatus = null;
        public String result = "";
    }
}

