/*************************************************************************
 *                                                                       *
 *  SignServer: The OpenSource Automated Signing Server                  *
 *                                                                       *
 *  This software is free software; you can redistribute it and/or       *
 *  modify it under the terms of the GNU Lesser General Public           *
 *  License as published by the Free Software Foundation; either         *
 *  version 2.1 of the License, or any later version.                    *
 *                                                                       *
 *  See terms of license at gnu.org.                                     *
 *                                                                       *
 *************************************************************************/
package org.signserver.deploytools.common;

import java.io.*;
import java.util.*;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import static java.util.regex.Pattern.compile;
import java.util.zip.ZipEntry;
import java.util.zip.ZipFile;
import java.util.zip.ZipOutputStream;
import org.apache.commons.lang.text.StrSubstitutor;

/**
 * Task postprocessing the modules.
 *
 * @author Markus Kilås
 * @author $Id$
 */
public abstract class AbstractModulesPostProcessor {
    public static final int MSG_ERR = 0;
    public static final int MSG_WARN = 1;
    public static final int MSG_INFO = 2;
    public static final int MSG_VERBOSE = 3;
    public static final int MSG_DEBUG = 4;

    private String modules;
    private String destDir;

    public void process() {
        if (modules == null) {
            throw new RuntimeException("Attribute 'modules' not specified");
        }

        if (destDir == null) {
            throw new RuntimeException("Attribute 'destDir' not specified");
        }

        // Get the list of all modules to process
        List<String> modulesList = asList(modules);
        log("Modules: " + modulesList, MSG_DEBUG);
        
        // Process each module
        for (String module : modulesList) {
            
            // Get the list of all files to postprocess
            String postProcessFiles = getProperty(module + ".postprocess.files");
            if (postProcessFiles != null) {
                List<String> postProcessFilesList = asList(postProcessFiles);
                if (!postProcessFilesList.isEmpty()) {
                    log("Post processing module: " + module, MSG_WARN);
                    String moduleType = getProperty(module + ".module.type");
                    log("    module.type: " + moduleType, MSG_VERBOSE);
                    log("    Files to process: " + postProcessFilesList, MSG_VERBOSE);
                    
                    // Process each jar-file
                    for (String postProcessFile : postProcessFilesList) {
                        try {
                            String dest = getProperty(module + "." + postProcessFile +".dest");
                            if (dest == null) {
                                dest = "";
                            }
                            log("    "+postProcessFile+".dest: " + dest, MSG_DEBUG);
                            String src = getProperty(module  + "." + postProcessFile +".src");
                            log("    "+postProcessFile+".src: " + src, MSG_DEBUG);
                            String includes = getProperty(module  + "." + postProcessFile +".includes");
                            log("    "+postProcessFile+".includes: " + includes, MSG_DEBUG);
                            String destfile = getDestDir() + "/" + dest + src;
                            
                            // Postprocess the files
                            replaceInJar(includes, getBaseDir() + "/lib/" + src, destfile, getProperties());
                        } catch (IOException ex) {
                            throw new RuntimeException(ex);
                        }
                    }
                }
            }
        }
    }
    
    private static List<String> asList(String items) {
        final LinkedList<String> result = new LinkedList<>();
        for (String item : items.split(",")) {
            item = item.trim();
            if (!item.isEmpty()) {
                result.add(item);
            }
        }
        return result;
    }

    public String getModules() {
        return modules;
    }

    public void setModules(String modules) {
        this.modules = modules;
    }

    public String getDestDir() {
        return destDir;
    }

    public void setDestDir(String destDir) {
        this.destDir = destDir;
    }

    /**
     * Replacer for the postprocess-jar Ant macro.
     * 
     * @param replaceincludes Ant list of all files in the jar to replace in
     * @param src Source jar file
     * @param destfile Destination jar file
     * @param properties Properties to replace from
     * @throws IOException in case of error
     */
    protected void replaceInJar(String replaceincludes, String src, String destfile, Map properties) throws IOException {
        try {
            log("Replace " + replaceincludes + " in " + src + " to " + destfile, MSG_VERBOSE);
            
            File srcFile = new File(src);
            if (!srcFile.exists()) {
                throw new FileNotFoundException(srcFile.getAbsolutePath());
            }
            
            // Expand properties of all files in replaceIncludes
            HashSet<String> replaceFiles = new HashSet<>();
            String[] rfiles = replaceincludes.split(",");
            for (int i = 0; i < rfiles.length; i++) {
                rfiles[i] = rfiles[i].trim();
            }
            replaceFiles.addAll(Arrays.asList(rfiles));
            log("Files to replace: " + replaceFiles, MSG_INFO);
            
            // Open source zip file
            ZipFile zipSrc = new ZipFile(srcFile);
            ZipOutputStream zipDest = new ZipOutputStream(new FileOutputStream(destfile));

            // For each entry in the source file copy them to dest file and postprocess if necessary
            Enumeration<? extends ZipEntry> entries = zipSrc.entries();
            while (entries.hasMoreElements()) {
                ZipEntry entry = entries.nextElement();
                String name = entry.getName();
                
                if (entry.isDirectory()) {
                    // Just put the directory
                    zipDest.putNextEntry(entry);
                } else {
                    // If we should postprocess the entry
                    if (replaceFiles.contains(name)) {
                        name += (" [REPLACE]");
                        log(name, MSG_VERBOSE);
                        
                        // Create a new zip entry for the file
                        ZipEntry newEntry = new ZipEntry(entry.getName());
                        newEntry.setComment(entry.getComment());
                        newEntry.setExtra(entry.getExtra());
                        zipDest.putNextEntry(newEntry);
                        
                        // Read the old document
                        StringBuffer oldDocument = stringBufferFromFile(zipSrc.getInputStream(entry));
                        log("Before replace ********\n" + oldDocument.toString() + "\n", MSG_DEBUG);
                        
                        // Do properties substitution
                        StrSubstitutor sub = new StrSubstitutor(properties);
                        StringBuffer newerDocument = commentReplacement(oldDocument, properties);
                        String newDocument = sub.replace(newerDocument);
                        log("After replace ********\n" + newDocument + "\n", MSG_DEBUG);
                        
                        // Write the new document
                        byte[] newBytes = newDocument.getBytes("UTF-8");
                        entry.setSize(newBytes.length);
                        copy(new ByteArrayInputStream(newBytes), zipDest);
                    } else {
                        // Just copy the entry to dest zip file
                        name += (" []");
                        log(name, MSG_VERBOSE);
                        zipDest.putNextEntry(entry);
                        copy(zipSrc.getInputStream(entry), zipDest);
                    }
                    zipDest.closeEntry();
                }
            }
            zipSrc.close();
            zipDest.close();
        } catch (IOException ex) {
            throw new RuntimeException(ex);
        }
    }
    
    /**
     * Reads from one stream and writes on the other, with a buffer.
     */
    private static void copy(InputStream in, OutputStream out) throws IOException {
        int read;
        byte[] buff = new byte[10 * 1024];
        while ((read = in.read(buff))!= -1) {
            out.write(buff, 0, read);
        }
    }

    /**
     * Reads a text file into a StringBuffer. The StringBuffer can then be used 
     * by the StrSubstitutor.
     */
    private static StringBuffer stringBufferFromFile(InputStream in) throws IOException {
        StringBuffer result = new StringBuffer();
        BufferedReader reader = null;
        try {
            reader = new BufferedReader(new InputStreamReader(in));
            String line;
            while ((line = reader.readLine()) != null) {
                result.append(line);
                result.append("\n");
            }
        } finally {
            if (reader != null) {
                try {
                    reader.close();
                } catch (IOException ignored) {} // NOPMD
            }
        }
        return result;
    }
    
    /**
     * Replaces &lt;!--COMMENT-REPLACEMENT(variable.name)--&gt; with the value
     * of the property variable.name.
     * @param oldDocument to replace in
     * @param properties to replace
     * @return the new text
     * @throws java.io.IOException in case of errors
     */
    protected StringBuffer commentReplacement(final StringBuffer oldDocument, final Map properties) throws IOException {
        final StringBuffer result = new StringBuffer();
        final Pattern pattern = compile(".*<!--COMMENT-REPLACEMENT\\(([a-zA-Z\\._]+)\\)-->.*");
        final BufferedReader reader = new BufferedReader(new StringReader(oldDocument.toString()));
        String line;
        
        while ((line = reader.readLine()) != null) {
            final Matcher m = pattern.matcher(line);
            if (m.matches()) {
                final String propertyName = m.group(1);
                if (propertyName != null) {
                    final Object value = properties.get(propertyName);
                    if (value instanceof String) {
                        log("Comment replacement for " + propertyName, MSG_VERBOSE);
                        line = line.replace("<!--COMMENT-REPLACEMENT(" + propertyName + ")-->", (String) value);
                    } else {
                        log("No comment replacement value for " + propertyName, MSG_VERBOSE);
                    }
                }
            }
            result.append(line).append("\n");
        }
        return result;
    }
    
    protected abstract void log(String msg, int msgLevel);
    protected abstract String getProperty(String string);
    protected abstract void setProperty(String newName, String property);

    private String getBaseDir() {
        return getProperty("basedir");
    }

    protected abstract Map getProperties();
}
