/*************************************************************************
 *                                                                       *
 *  SignServer: The OpenSource Automated Signing Server                  *
 *                                                                       *
 *  This software is free software; you can redistribute it and/or       *
 *  modify it under the terms of the GNU Lesser General Public           *
 *  License as published by the Free Software Foundation; either         *
 *  version 2.1 of the License, or any later version.                    *
 *                                                                       *
 *  See terms of license at gnu.org.                                     *
 *                                                                       *
 *************************************************************************/
package org.signserver.deploytools.common;

import java.io.IOException;
import java.net.URI;
import java.net.URISyntaxException;
import java.nio.charset.Charset;
import java.nio.file.FileSystem;
import java.nio.file.FileSystems;
import java.nio.file.FileVisitResult;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.StandardOpenOption;
import java.nio.file.attribute.BasicFileAttributes;
import java.nio.file.attribute.FileTime;
import java.text.SimpleDateFormat;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Locale;
import java.util.TimeZone;

/**
 * Helper methods for ZIP files.
 *
 * @author Markus Kilås
 * @version $Id: ZipFileUtils.java 7424 2016-09-02 12:25:50Z netmackan $
 */
public class ZipFileUtils {
    
    private final LoggerFacade log;
    
    public ZipFileUtils(LoggerFacade log) {
        this.log = log;
    }

    /**
     * Opens the specified JAR (ZIP) file and sets the last modified time on
     * each entry to the specified time and also sets the time-stamp in 
     * pom.properties if found.
     *
     * @param jar file to open
     * @param time stamp in milliseconds to set
     * @throws IOException in case a path in the file can not be walked
     * @throws URISyntaxException in case the input jar file name can not be
     * used to construct a jar URI
     */
    public void rewriteZipEntriesLastModifiedTimeAndPomProperties(final String jar, final long time) throws IOException, URISyntaxException {
        final URI zipURI = new URI("jar:file:" + jar);
        
        // Format the time-stamp
        final SimpleDateFormat sdf = new SimpleDateFormat("EEE MMM dd HH:mm:ss zzz yyyy", Locale.ENGLISH);
        sdf.setTimeZone(TimeZone.getTimeZone("GMT"));
        final String timeStamp = sdf.format(new Date(time));
        
        try (FileSystem fs = FileSystems.newFileSystem(zipURI, new HashMap<String, Object>())) {
            for (Path path : fs.getRootDirectories()) {
                final LinkedList<Path> directories = new LinkedList<>();
                
                Files.walkFileTree(path, new SimpleFileVisitor<Path>() {
                    
                    @Override
                    public FileVisitResult preVisitDirectory(Path dir, BasicFileAttributes attrs)
                            throws IOException {
                        log.logDebug("Found directory: " + dir);
                        
                        if (!dir.equals(dir.getRoot())) {
                            // We can not modify the directory now as it seems to messup the file walking
                            directories.add(dir);
                        }
                        return super.preVisitDirectory(dir, attrs);
                    }
                    
                    @Override
                    public FileVisitResult visitFile(Path file,
                            BasicFileAttributes attr) {
                        log.logDebug("Setting last modified time on file: " + file);
                
                        // Rewrite the time-stamp in pom.properties
                        if (file.startsWith("/META-INF") && file.endsWith("pom.properties")) {
                            try {
                                Charset latin1 = Charset.forName("latin1");
                                List<String> propFileLines = Files.readAllLines(file, latin1);
                                propFileLines.set(1, "#" + timeStamp);
                                Files.write(file, propFileLines, latin1, StandardOpenOption.TRUNCATE_EXISTING);
                                log.logDebug("Wrote pom.properties");
                            } catch (IOException ex) {
                                log.logError("Failed to set deterministic time in " + zipURI + "!" + file, ex);
                            }
                        }
                        
                        // Rewrite the modified time
                        try {
                            Files.setLastModifiedTime(file, FileTime.fromMillis(time));
                        } catch (IOException ex) {
                            log.logError("Failed to set deterministic time on " + zipURI + "!" + file, ex);
                        }
                        
                        return FileVisitResult.CONTINUE;
                    }
                });
                
                // Now take care of the directories in reversed order
                Collections.reverse(directories);
                for (Path dir : directories) {
                    log.logDebug("Setting last modified time on directory: " + dir);
                    try {
                        Files.setLastModifiedTime(dir, FileTime.fromMillis(time));
                    } catch (IOException ex) {
                        log.logError("Failed to set deterministic time on " + zipURI + "!" + dir, ex);
                    }
                }
            }
        }
    }
}
