package org.sfm.datastax.impl;

import com.datastax.driver.core.DataType;
import com.datastax.driver.core.GettableData;
import org.sfm.reflect.Getter;
import org.sfm.reflect.primitive.IntGetter;

public class DatastaxGenericIntegerGetter implements IntGetter<GettableData>, Getter<GettableData, Integer> {

    private final int index;
    private final DataType.Name dataTypeName;

    public DatastaxGenericIntegerGetter(int index, DataType dataType) {
        this.index = index;
        this.dataTypeName = validateName(dataType);
    }

    private DataType.Name validateName(DataType dataType) {

        final DataType.Name name = dataType.getName();
        switch (name) {
            case BIGINT:
            case VARINT:
            case INT:
            case DECIMAL:
            case FLOAT:
            case DOUBLE:
            case COUNTER:
            return name;
        }
        throw new IllegalArgumentException("Datatype " + dataType + " not a number");
    }

    @Override
    public Integer get(GettableData target) throws Exception {
        if (target.isNull(index)) {
            return null;
        }
          return getInt(target);
    }

    @Override
    public int getInt(GettableData target) throws Exception {
        switch (dataTypeName) {
            case BIGINT:
            case COUNTER:
                return (int)target.getLong(index);
            case VARINT:
                return target.getVarint(index).intValue();
            case INT:
                return target.getInt(index);
            case DECIMAL:
                return target.getDecimal(index).intValue();
            case FLOAT:
                return (int)target.getFloat(index);
            case DOUBLE:
                return (int)target.getDouble(index);
        }
        return 0;
    }
}
