package org.sfm.reflect.impl;

import org.sfm.reflect.ConstructorDefinition;
import org.sfm.reflect.ConstructorParameter;
import org.sfm.reflect.Getter;
import org.sfm.reflect.TypeHelper;

import java.util.HashMap;
import java.util.Map;

public final class ArgumentBuilder<S, T> {

	@SuppressWarnings("rawtypes")
	private static final Getter NULL_GETTER = new Getter() {
		@Override
		public Object get(Object target) throws Exception {
			return null;
		}
	};
	@SuppressWarnings({  "rawtypes" })
	private static final Map<Class<?>, Getter> DEFAULT_GETTERS = new HashMap<Class<?>, Getter>();
	static {
		DEFAULT_GETTERS.put(boolean.class, new Getter() {
			@Override
			public Object get(Object target) throws Exception {
				return Boolean.TRUE;
			}
		});
		DEFAULT_GETTERS.put(byte.class, new Getter() {
			@Override
			public Object get(Object target) throws Exception {
				return (byte) 0;
			}
		});
		DEFAULT_GETTERS.put(char.class, new Getter() {
			@Override
			public Object get(Object target) throws Exception {
				return (char) 0;
			}
		});
		DEFAULT_GETTERS.put(short.class, new Getter() {
			@Override
			public Object get(Object target) throws Exception {
				return (short) 0;
			}
		});
		DEFAULT_GETTERS.put(int.class, new Getter() {
			@Override
			public Object get(Object target) throws Exception {
				return 0;
			}
		});
		DEFAULT_GETTERS.put(long.class, new Getter() {
			@Override
			public Object get(Object target) throws Exception {
				return (long) 0;
			}
		});
		DEFAULT_GETTERS.put(float.class, new Getter() {
			@Override
			public Object get(Object target) throws Exception {
				return 0.0f;
			}
		});
		DEFAULT_GETTERS.put(double.class, new Getter() {
				@Override
				public Object get(Object target) throws Exception {
					return 0.0d;
				}
			});
	}
	
	
	private final Getter<S, ?>[] getters;
	@SuppressWarnings("unchecked")
	public ArgumentBuilder(ConstructorDefinition<T> constructorDefinition,
			Map<ConstructorParameter, Getter<S, ?>> injections) {
		ConstructorParameter[] parameters = constructorDefinition.getParameters();
		getters = new Getter[parameters.length];
		for (int i = 0; i < getters.length; i++) {
			ConstructorParameter param = parameters[i];
			Getter<S, ?> getter = injections.get(param);
			if (getter == null) {
				if (TypeHelper.isPrimitive(param.getType())) {
					getter = DEFAULT_GETTERS.get(param.getType());
				} else {
					getter = NULL_GETTER;
				}
			}
			getters[i] = getter;
		}
		
	}
	
	public Object[] build(S source) throws Exception {
		Object[] args = new Object[getters.length];
		
		for(int i = 0; i < args.length; i++) {
			args[i] = getters[i].get(source);
		}
		
		return args;
	}

}
