/*
 * Copyright (c) 2016 simplity.org
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */
package org.simplity.kernel.data;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;

import org.simplity.kernel.ApplicationError;
import org.simplity.kernel.Tracer;
import org.simplity.kernel.value.Value;
import org.simplity.kernel.value.ValueType;

/**
 * represents a fields collection internally, but implements all methods of a
 * data sheet. This is to be used when you are likely to add/remove fields, and
 * if used as a row of data, the order of the columns is not important.
 *
 * @author simplity.org
 * @version $Id: $Id
 */
public class DynamicSheet implements DataSheet {
	private final Map<String, Value> fieldValues = new HashMap<String, Value>();

	/** {@inheritDoc} */
	@Override
	public String[][] getRawData() {
		int n = this.fieldValues.size();
		String[] header = new String[n];
		String[] values = new String[n];
		n = 0;
		for (Entry<String, Value> entry : this.fieldValues.entrySet()) {
			header[n] = entry.getKey();
			values[n] = entry.getValue().toString();
			n++;
		}
		String[][] rawData = { header, values };
		return rawData;
	}

	/** {@inheritDoc} */
	@Override
	public int length() {
		return 1;
	}

	/** {@inheritDoc} */
	@Override
	public int width() {
		return this.fieldValues.size();
	}

	/** {@inheritDoc} */
	@Override
	public String[] getColumnNames() {
		return this.fieldValues.keySet().toArray(new String[0]);
	}

	/** {@inheritDoc} */
	@Override
	public ValueType[] getValueTypes() {
		String[] names = this.getColumnNames();
		ValueType[] types = new ValueType[names.length];
		int i = 0;
		for (String colName : names) {
			types[i] = this.fieldValues.get(colName).getValueType();
			i++;
		}
		return types;
	}

	/** {@inheritDoc} */
	@Override
	public Value[] getRow(int zeroBasedRowNumber) {
		if (zeroBasedRowNumber != 0) {
			return null;
		}
		String[] names = this.getColumnNames();
		Value[] values = new Value[names.length];
		int i = 0;
		for (String colName : names) {
			values[i] = this.fieldValues.get(colName);
			i++;
		}
		return values;
	}

	/** {@inheritDoc} */
	@Override
	public List<Value[]> getAllRows() {
		List<Value[]> rows = new ArrayList<Value[]>(1);
		rows.add(this.getRow(0));
		return rows;
	}

	/** {@inheritDoc} */
	@Override
	public Value getColumnValue(String columnName, int zeroBasedRowNumber) {
		if (zeroBasedRowNumber != 0) {
			return null;
		}
		return this.fieldValues.get(columnName);
	}

	/** {@inheritDoc} */
	@Override
	public void setColumnValue(String columnName, int zeroBasedRowNumber,
			Value value) {
		if (zeroBasedRowNumber != 0) {
			return;
		}
		if (value == null) {
			this.fieldValues.remove(columnName);
		} else {
			this.fieldValues.put(columnName, value);
		}
	}

	/** {@inheritDoc} */
	@Override
	public Iterator<FieldsInterface> iterator() {
		return new DataRows(this);
	}

	/** {@inheritDoc} */
	@Override
	public Value getValue(String fieldName) {
		return this.fieldValues.get(fieldName);
	}

	/** {@inheritDoc} */
	@Override
	public void setValue(String fieldName, Value value) {
		if (value == null) {
			this.fieldValues.remove(fieldName);
		} else {
			this.fieldValues.put(fieldName, value);
		}
	}

	/** {@inheritDoc} */
	@Override
	public boolean hasValue(String fieldName) {
		return this.fieldValues.containsKey(fieldName);
	}

	/** {@inheritDoc} */
	@Override
	public Value removeValue(String fieldName) {
		return this.fieldValues.remove(fieldName);
	}

	/** {@inheritDoc} */
	@Override
	public void addRow(Value[] row) {
		throw new ApplicationError(
				"addRow() should not be called for SingleRowSheet");

	}

	/** {@inheritDoc} */
	@Override
	public Value[] getColumnValues(String columnName) {
		Value value = this.getValue(columnName);
		Value[] values = { value };
		return values;
	}

	/** {@inheritDoc} */
	@Override
	public void addColumn(String columnName, ValueType valueType,
			Value[] columnValues) {
		if (columnValues != null) {
			this.fieldValues.put(columnName, columnValues[0]);
		}
	}

	/** {@inheritDoc} */
	@Override
	public Set<Entry<String, Value>> getAllFields() {
		return this.fieldValues.entrySet();
	}

	/** {@inheritDoc} */
	@Override
	public Set<Entry<String, Value>> getAllFields(int rowIdx) {
		return this.fieldValues.entrySet();
	}

	/*
	 * (non-Javadoc)
	 *
	 * @see org.simplity.kernel.data.DataSheet#trace()
	 */
	/** {@inheritDoc} */
	@Override
	public void trace() {
		Tracer.trace("(Dynamic Sheet)");
		for (Map.Entry<String, Value> field : this.fieldValues.entrySet()) {
			Tracer.trace(field.getKey() + '=' + field.getValue());
		}
	}

	/*
	 * (non-Javadoc)
	 *
	 * @see
	 * org.simplity.kernel.data.DataSheet#appendRows(org.simplity.kernel.data.
	 * DataSheet)
	 */
	/** {@inheritDoc} */
	@Override
	public int appendRows(DataSheet sheet) {
		throw new ApplicationError(
				"Dynamic sheet can not have more than one rows, and hence appendRows operation is invalid");
	}

	/*
	 * (non-Javadoc)
	 *
	 * @see org.simplity.kernel.data.DataSheet#addColumn(java.lang.String,
	 * org.simplity.kernel.value.Value)
	 */
	/** {@inheritDoc} */
	@Override
	public void addColumn(String columnName, Value value) {
		this.setValue(columnName, value);
	}

	/** {@inheritDoc} */
	@Override
	public int getColIdx(String columnName) {
		int i = 0;
		for (String colName : this.getColumnNames()) {
			if (colName.equals(columnName)) {
				return i;
			}
			i++;
		}
		Tracer.trace("We did not find column " + columnName
				+ " in this dynamic sheet");
		return -1;
	}

	/* (non-Javadoc)
	 * @see org.simplity.kernel.data.DataSheet#toSerializedText(org.simplity.kernel.data.DataSerializationType)
	 */
	/** {@inheritDoc} */
	@Override
	public String toSerializedText(DataSerializationType serializationType) {
		throw new ApplicationError("Sorry, serialization is not yet implemented for Dynamic sheet");
		//TODO to be built
	}

	/* (non-Javadoc)
	 * @see org.simplity.kernel.data.DataSheet#fromSerializedText(java.lang.String, org.simplity.kernel.data.DataSerializationType, boolean)
	 */
	/** {@inheritDoc} */
	@Override
	public void fromSerializedText(String text,
			DataSerializationType serializationType,
			boolean replaceExistingRows) {
		throw new ApplicationError("Sorry, de-serialization is not yet implemented for Dynamic sheet");
		//TODO to be built
	}
}
