/*
 * Decompiled with CFR 0.152.
 */
package org.springframework.cloud.config.server;

import java.io.File;
import java.lang.reflect.Method;
import java.net.URISyntaxException;
import org.assertj.core.api.Assertions;
import org.eclipse.jgit.api.TransportConfigCallback;
import org.eclipse.jgit.transport.FetchConnection;
import org.eclipse.jgit.transport.PushConnection;
import org.eclipse.jgit.transport.SshConfigStore;
import org.eclipse.jgit.transport.SshSessionFactory;
import org.eclipse.jgit.transport.SshTransport;
import org.eclipse.jgit.transport.Transport;
import org.eclipse.jgit.transport.URIish;
import org.eclipse.jgit.transport.sshd.SshdSessionFactory;
import org.junit.jupiter.api.Test;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.test.context.SpringBootTest;
import org.springframework.cloud.config.server.environment.MultipleJGitEnvironmentProperties;
import org.springframework.cloud.config.server.environment.MultipleJGitEnvironmentRepository;
import org.springframework.cloud.config.server.ssh.FileBasedSshSessionFactory;
import org.springframework.cloud.config.server.ssh.FileBasedSshTransportConfigCallback;
import org.springframework.cloud.config.server.ssh.PropertiesBasedSshTransportConfigCallback;
import org.springframework.cloud.config.server.ssh.PropertyBasedSshSessionFactory;
import org.springframework.cloud.config.server.ssh.SshPropertyValidator;
import org.springframework.cloud.config.server.test.TestConfigServerApplication;
import org.springframework.test.context.ActiveProfiles;

public class TransportConfigurationIntegrationTests {

    private static class TestProperties {
        private static final String TEST_PRIVATE_KEY_1 = "-----BEGIN RSA PRIVATE KEY-----\nMIIEpAIBAAKCAQEAoqyz6YaYMTr7L8GLPSQpAQXaM04gRx4CCsGK2kfLQdw4BlqI\nyyxp38YcuZG9cUDBAxby+K2TKmwHaC1R61QTwbPuCRdIPrDwRz+FLoegm3iDLCmn\nuP6rjZDneYsqfU1KSdrOwIbCnONfDdvYL/vnZC/o8DDMlk5Orw2SfHkT3pq0o8km\nayBwN4Sf3bpyWTY0oZcmNeSCCoIdE59k8Pa7/t9bwY9caLj05C3DEsjucc7Ei/Eq\nTOyGyobtXwaya5CqKLUHes74Poz1aEP/yVFdUud91uezd8ZK1P1t5/ZKA3R6aHir\n+diDJ2/GQ2tD511FW46yw+EtBUJTO6ADVv4UnQIDAQABAoIBAF+5qwEfX82QfKFk\njfADqFFexUDtl1biFKeJrpC2MKhn01wByH9uejrhFKQqW8UaKroLthyZ34DWIyGt\nlDnHGv0gSVF2LuAdNLdobJGt49e4+c9yD61vxzm97Eh8mRs08SM2q/VlF35E2fmI\nxdWusUImYzd8L9e+6tRd8zZl9UhG5vR5XIstKqxC6S0g79aAt0hasE4Gw1FKOf2V\n4mlL15atjQSKCPdOicuyc4zpjAtU1A9AfF51iG8oOUuJebPW8tCftfOQxaeGFgMG\n7M9aai1KzXR6M5IBAKEv31yBvz/SHTneP7oZXNLeC1GIR420PKybmeZdNK8BbEAu\n3reKgm0CgYEA03Sx8JoF5UBsIvFPpP1fjSlTgKryM5EJR6KQtj5e4YfyxccJepN8\nq4MrqDfNKleG/a1acEtDMhBNovU7Usp2QIP7zpAeioHBOhmE5WSieZGc3icOGWWq\nmRkdulSONruqWKv76ZoluxftekE03bDhZDNlcCgmrslEKB/ufHd2oc8CgYEAxPFa\nlKOdSeiYFV5CtvO8Ro8em6rGpSsVz4qkPxbeBqUDCb9KXHhq6YrhRxOIfQJKfT7M\nZFCn8ArJXKgOGu+KsvwIErFHF9g2jJMG4DOUTpkQgi2yveihFxcmz/AltyVXgrnv\nZWQbAerH77pdKKhNivLGgEv72GYawdYjYNjemdMCgYA2kEMmMahZyrDcp2YEzfit\nBT/t0K6kzcUWPgWXcSqsiZcEn+J7RbmCzFskkhmX1nQX23adyV3yejB+X0dKisHO\nzf/ZAmlPFkJVCqa3RquCMSfIT02dEhXeYZPBM/Zqeyxuqxpa4hLgX0FBLbhFiFHw\nuC5xrXql2XuD2xF//peXEwKBgQC+pa28Cg7vRxxCQzduB9CQtWc55j3aEjVQ7bNF\n54sS/5ZLT0Ra8677WZfuyDfuW9NkHvCZg4Ku2qJG8eCFrrGjxlrCTZ62tHVJ6+JS\nE1xUIdRbUIWhVZrr0VufG6hG/P0T7Y6Tpi6G0pKtvMkF3LcD9TS3adboix8H2ZXx\n4L7MRQKBgQC0OO3qqNXOjIVYWOoqXLybOY/Wqu9lxCAgGyCYaMcstnBI7W0MZTBr\n/syluvGsaFc1sE7MMGOOzKi1tF4YvDmSnzA/R1nmaPguuD9fOA+w7Pwkv5vLvuJq\n2U7EeNwxq1I1L3Ag6E7wH4BHLHd4TKaZR6agFkn8oomz71yZPGjuZQ==\n-----END RSA PRIVATE KEY-----";
        private static final String TEST_PRIVATE_KEY_2 = "-----BEGIN RSA PRIVATE KEY-----\nMIIEpgIBAAKCAQEAx4UbaDzY5xjW6hc9jwN0mX33XpTDVW9WqHp5AKaRbtAC3DqX\nIXFMPgw3K45jxRb93f8tv9vL3rD9CUG1Gv4FM+o7ds7FRES5RTjv2RT/JVNJCoqF\nol8+ngLqRZCyBtQN7zYByWMRirPGoDUqdPYrj2yq+ObBBNhg5N+hOwKjjpzdj2Ud\n1l7R+wxIqmJo1IYyy16xS8WsjyQuyC0lL456qkd5BDZ0Ag8j2X9H9D5220Ln7s9i\noezTipXipS7p7Jekf3Ywx6abJwOmB0rX79dV4qiNcGgzATnG1PkXxqt76VhcGa0W\nDDVHEEYGbSQ6hIGSh0I7BQun0aLRZojfE3gqHQIDAQABAoIBAQCZmGrk8BK6tXCd\nfY6yTiKxFzwb38IQP0ojIUWNrq0+9Xt+NsypviLHkXfXXCKKU4zUHeIGVRq5MN9b\nBO56/RrcQHHOoJdUWuOV2qMqJvPUtC0CpGkD+valhfD75MxoXU7s3FK7yjxy3rsG\nEmfA6tHV8/4a5umo5TqSd2YTm5B19AhRqiuUVI1wTB41DjULUGiMYrnYrhzQlVvj\n5MjnKTlYu3V8PoYDfv1GmxPPh6vlpafXEeEYN8VB97e5x3DGHjZ5UrurAmTLTdO8\n+AahyoKsIY612TkkQthJlt7FJAwnCGMgY6podzzvzICLFmmTXYiZ/28I4BX/mOSe\npZVnfRixAoGBAO6Uiwt40/PKs53mCEWngslSCsh9oGAaLTf/XdvMns5VmuyyAyKG\nti8Ol5wqBMi4GIUzjbgUvSUt+IowIrG3f5tN85wpjQ1UGVcpTnl5Qo9xaS1PFScQ\nxrtWZ9eNj2TsIAMp/svJsyGG3OibxfnuAIpSXNQiJPwRlW3irzpGgVx/AoGBANYW\ndnhshUcEHMJi3aXwR12OTDnaLoanVGLwLnkqLSYUZA7ZegpKq90UAuBdcEfgdpyi\nPhKpeaeIiAaNnFo8m9aoTKr+7I6/uMTlwrVnfrsVTZv3orxjwQV20YIBCVRKD1uX\nVhE0ozPZxwwKSPAFocpyWpGHGreGF1AIYBE9UBtjAoGBAI8bfPgJpyFyMiGBjO6z\nFwlJc/xlFqDusrcHL7abW5qq0L4v3R+FrJw3ZYufzLTVcKfdj6GelwJJO+8wBm+R\ngTKYJItEhT48duLIfTDyIpHGVm9+I1MGhh5zKuCqIhxIYr9jHloBB7kRm0rPvYY4\nVAykcNgyDvtAVODP+4m6JvhjAoGBALbtTqErKN47V0+JJpapLnF0KxGrqeGIjIRV\ncYA6V4WYGr7NeIfesecfOC356PyhgPfpcVyEztwlvwTKb3RzIT1TZN8fH4YBr6Ee\nKTbTjefRFhVUjQqnucAvfGi29f+9oE3Ei9f7wA+H35ocF6JvTYUsHNMIO/3gZ38N\nCPjyCMa9AoGBAMhsITNe3QcbsXAbdUR00dDsIFVROzyFJ2m40i4KCRM35bC/BIBs\nq0TY3we+ERB40U8Z2BvU61QuwaunJ2+uGadHo58VSVdggqAo0BSkH58innKKt96J\n69pcVH/4rmLbXdcmNYGm6iu+MlPQk4BUZknHSmVHIFdJ0EPupVaQ8RHT\n-----END RSA PRIVATE KEY-----\n";

        private TestProperties() {
        }
    }

    private static class DummySshTransport
    extends SshTransport {
        DummySshTransport(String uri) throws URISyntaxException {
            super(new URIish(uri));
        }

        public FetchConnection openFetch() {
            return null;
        }

        public PushConnection openPush() {
            return null;
        }

        public static SshTransport newInstance() {
            try {
                return new DummySshTransport("git+ssh://git@uri");
            }
            catch (URISyntaxException e) {
                throw new IllegalStateException("Not expected", e);
            }
        }
    }

    public static class CallbackWithHttpUrlsOnly {

        @SpringBootTest(classes={TestConfigServerApplication.class, SshPropertyValidator.class}, webEnvironment=SpringBootTest.WebEnvironment.RANDOM_PORT, properties={"spring.cloud.config.server.composite[0].type=git", "spring.cloud.config.server.composite[0].uri=https://gitserver.com/team/repo.git"})
        @ActiveProfiles(value={"test", "composite"})
        public static class ListTest {
            @Autowired
            private MultipleJGitEnvironmentRepository jGitEnvironmentRepository;

            @Test
            public void sshTransportCallbackIsConfigured() {
                TransportConfigCallback transportConfigCallback = this.jGitEnvironmentRepository.getTransportConfigCallback();
                Assertions.assertThat((Object)transportConfigCallback).isNotNull();
            }

            @Test
            public void noSessionFactoryIsConfiguredForSshTransports() throws Exception {
                this.jGitEnvironmentRepository.afterPropertiesSet();
                SshTransport sshTransport = DummySshTransport.newInstance();
                SshSessionFactory defaultSessionFactory = sshTransport.getSshSessionFactory();
                this.jGitEnvironmentRepository.getTransportConfigCallback().configure((Transport)sshTransport);
                Assertions.assertThat((Object)sshTransport.getSshSessionFactory()).isSameAs((Object)defaultSessionFactory);
            }
        }

        @SpringBootTest(classes={TestConfigServerApplication.class, SshPropertyValidator.class}, webEnvironment=SpringBootTest.WebEnvironment.RANDOM_PORT, properties={"spring.cloud.config.server.git.uri=https://gitserver.com/team/repo.git"})
        @ActiveProfiles(value={"test", "git"})
        public static class StaticTest {
            @Autowired
            private MultipleJGitEnvironmentRepository jGitEnvironmentRepository;

            @Test
            public void sshTransportCallbackIsConfigured() {
                TransportConfigCallback transportConfigCallback = this.jGitEnvironmentRepository.getTransportConfigCallback();
                Assertions.assertThat((Object)transportConfigCallback).isNotNull();
            }

            @Test
            public void noSessionFactoryIsConfiguredForSshTransports() throws Exception {
                this.jGitEnvironmentRepository.afterPropertiesSet();
                SshTransport sshTransport = DummySshTransport.newInstance();
                SshSessionFactory defaultSessionFactory = sshTransport.getSshSessionFactory();
                this.jGitEnvironmentRepository.getTransportConfigCallback().configure((Transport)sshTransport);
                Assertions.assertThat((Object)sshTransport.getSshSessionFactory()).isSameAs((Object)defaultSessionFactory);
            }
        }
    }

    public static class FileBasedCallbackTest {

        @SpringBootTest(classes={TestConfigServerApplication.class, SshPropertyValidator.class}, webEnvironment=SpringBootTest.WebEnvironment.RANDOM_PORT, properties={"spring.cloud.config.server.composite[0].type=git", "spring.cloud.config.server.composite[0].uri=git@gitserver.com:team/repo.git", "spring.cloud.config.server.composite[0].ignoreLocalSshSettings=false"})
        @ActiveProfiles(value={"test", "composite"})
        public static class ListTest {
            @Autowired
            private MultipleJGitEnvironmentRepository jGitEnvironmentRepository;

            @Test
            public void fileBasedTransportCallbackIsConfigured() {
                TransportConfigCallback transportConfigCallback = this.jGitEnvironmentRepository.getTransportConfigCallback();
                Assertions.assertThat((Object)transportConfigCallback).isInstanceOf(FileBasedSshTransportConfigCallback.class);
            }

            @Test
            public void fileBasedSessionFactoryIsUsedForSshTransports() throws Exception {
                this.jGitEnvironmentRepository.afterPropertiesSet();
                SshTransport sshTransport = DummySshTransport.newInstance();
                this.jGitEnvironmentRepository.getTransportConfigCallback().configure((Transport)sshTransport);
                Assertions.assertThat((Object)sshTransport.getSshSessionFactory()).isInstanceOf(FileBasedSshSessionFactory.class);
            }

            @Test
            public void strictHostKeyCheckShouldCheck() throws Exception {
                Assertions.assertThat((boolean)this.jGitEnvironmentRepository.isStrictHostKeyChecking()).isTrue();
                this.jGitEnvironmentRepository.afterPropertiesSet();
                SshTransport sshTransport = DummySshTransport.newInstance();
                this.jGitEnvironmentRepository.getTransportConfigCallback().configure((Transport)sshTransport);
                SshdSessionFactory factory = (SshdSessionFactory)sshTransport.getSshSessionFactory();
                Method createSshConfigStore = factory.getClass().getDeclaredMethod("createSshConfigStore", File.class, File.class, String.class);
                createSshConfigStore.setAccessible(true);
                SshConfigStore configStore = (SshConfigStore)createSshConfigStore.invoke((Object)factory, new File("."), new File("."), "local-username");
                createSshConfigStore.setAccessible(false);
                Assertions.assertThat((boolean)"yes".equals(configStore.lookup("gitserver.com", 22, "username").getValue("StrictHostKeyChecking"))).isTrue();
            }
        }

        @SpringBootTest(classes={TestConfigServerApplication.class, SshPropertyValidator.class}, webEnvironment=SpringBootTest.WebEnvironment.RANDOM_PORT, properties={"spring.cloud.config.server.git.uri=git@gitserver.com:team/repo.git", "spring.cloud.config.server.git.ignoreLocalSshSettings=false"})
        @ActiveProfiles(value={"test", "git"})
        public static class StaticTest {
            @Autowired
            private MultipleJGitEnvironmentRepository jGitEnvironmentRepository;

            @Test
            public void fileBasedTransportCallbackIsConfigured() {
                TransportConfigCallback transportConfigCallback = this.jGitEnvironmentRepository.getTransportConfigCallback();
                Assertions.assertThat((Object)transportConfigCallback).isInstanceOf(FileBasedSshTransportConfigCallback.class);
            }

            @Test
            public void fileBasedSessionFactoryIsUsedForSshTransports() throws Exception {
                this.jGitEnvironmentRepository.afterPropertiesSet();
                SshTransport sshTransport = DummySshTransport.newInstance();
                this.jGitEnvironmentRepository.getTransportConfigCallback().configure((Transport)sshTransport);
                Assertions.assertThat((Object)sshTransport.getSshSessionFactory()).isInstanceOf(FileBasedSshSessionFactory.class);
            }

            @Test
            public void strictHostKeyCheckShouldCheck() throws Exception {
                Assertions.assertThat((boolean)this.jGitEnvironmentRepository.isStrictHostKeyChecking()).isTrue();
                this.jGitEnvironmentRepository.afterPropertiesSet();
                SshTransport sshTransport = DummySshTransport.newInstance();
                this.jGitEnvironmentRepository.getTransportConfigCallback().configure((Transport)sshTransport);
                SshdSessionFactory factory = (SshdSessionFactory)sshTransport.getSshSessionFactory();
                Method createSshConfigStore = factory.getClass().getDeclaredMethod("createSshConfigStore", File.class, File.class, String.class);
                createSshConfigStore.setAccessible(true);
                SshConfigStore configStore = (SshConfigStore)createSshConfigStore.invoke((Object)factory, new File("."), new File("."), "local-username");
                createSshConfigStore.setAccessible(false);
                Assertions.assertThat((boolean)"yes".equals(configStore.lookup("gitserver.com", 22, "username").getValue("StrictHostKeyChecking"))).isTrue();
            }
        }
    }

    public static class SshPropertiesWithinNestedRepo {

        @SpringBootTest(classes={TestConfigServerApplication.class, SshPropertyValidator.class}, webEnvironment=SpringBootTest.WebEnvironment.RANDOM_PORT, properties={"spring.config.additional-location=optional:file:/ssh/,optional:classpath:/ssh/", "spring.config.name:ssh-nested-settings-list"})
        @ActiveProfiles(value={"test", "composite"})
        public static class ListTest {
            @Autowired
            private MultipleJGitEnvironmentRepository jGitEnvironmentRepository;

            @Test
            public void sshPropertiesWithinNestedRepo() {
                TransportConfigCallback callback = this.jGitEnvironmentRepository.getTransportConfigCallback();
                Assertions.assertThat((Object)callback).isInstanceOf(PropertiesBasedSshTransportConfigCallback.class);
                PropertiesBasedSshTransportConfigCallback configCallback = (PropertiesBasedSshTransportConfigCallback)callback;
                MultipleJGitEnvironmentProperties sshUriProperties = configCallback.getSshUriProperties();
                Assertions.assertThat((String)configCallback.getSshUriProperties().getPrivateKey()).isEqualTo("-----BEGIN RSA PRIVATE KEY-----\nMIIEpAIBAAKCAQEAoqyz6YaYMTr7L8GLPSQpAQXaM04gRx4CCsGK2kfLQdw4BlqI\nyyxp38YcuZG9cUDBAxby+K2TKmwHaC1R61QTwbPuCRdIPrDwRz+FLoegm3iDLCmn\nuP6rjZDneYsqfU1KSdrOwIbCnONfDdvYL/vnZC/o8DDMlk5Orw2SfHkT3pq0o8km\nayBwN4Sf3bpyWTY0oZcmNeSCCoIdE59k8Pa7/t9bwY9caLj05C3DEsjucc7Ei/Eq\nTOyGyobtXwaya5CqKLUHes74Poz1aEP/yVFdUud91uezd8ZK1P1t5/ZKA3R6aHir\n+diDJ2/GQ2tD511FW46yw+EtBUJTO6ADVv4UnQIDAQABAoIBAF+5qwEfX82QfKFk\njfADqFFexUDtl1biFKeJrpC2MKhn01wByH9uejrhFKQqW8UaKroLthyZ34DWIyGt\nlDnHGv0gSVF2LuAdNLdobJGt49e4+c9yD61vxzm97Eh8mRs08SM2q/VlF35E2fmI\nxdWusUImYzd8L9e+6tRd8zZl9UhG5vR5XIstKqxC6S0g79aAt0hasE4Gw1FKOf2V\n4mlL15atjQSKCPdOicuyc4zpjAtU1A9AfF51iG8oOUuJebPW8tCftfOQxaeGFgMG\n7M9aai1KzXR6M5IBAKEv31yBvz/SHTneP7oZXNLeC1GIR420PKybmeZdNK8BbEAu\n3reKgm0CgYEA03Sx8JoF5UBsIvFPpP1fjSlTgKryM5EJR6KQtj5e4YfyxccJepN8\nq4MrqDfNKleG/a1acEtDMhBNovU7Usp2QIP7zpAeioHBOhmE5WSieZGc3icOGWWq\nmRkdulSONruqWKv76ZoluxftekE03bDhZDNlcCgmrslEKB/ufHd2oc8CgYEAxPFa\nlKOdSeiYFV5CtvO8Ro8em6rGpSsVz4qkPxbeBqUDCb9KXHhq6YrhRxOIfQJKfT7M\nZFCn8ArJXKgOGu+KsvwIErFHF9g2jJMG4DOUTpkQgi2yveihFxcmz/AltyVXgrnv\nZWQbAerH77pdKKhNivLGgEv72GYawdYjYNjemdMCgYA2kEMmMahZyrDcp2YEzfit\nBT/t0K6kzcUWPgWXcSqsiZcEn+J7RbmCzFskkhmX1nQX23adyV3yejB+X0dKisHO\nzf/ZAmlPFkJVCqa3RquCMSfIT02dEhXeYZPBM/Zqeyxuqxpa4hLgX0FBLbhFiFHw\nuC5xrXql2XuD2xF//peXEwKBgQC+pa28Cg7vRxxCQzduB9CQtWc55j3aEjVQ7bNF\n54sS/5ZLT0Ra8677WZfuyDfuW9NkHvCZg4Ku2qJG8eCFrrGjxlrCTZ62tHVJ6+JS\nE1xUIdRbUIWhVZrr0VufG6hG/P0T7Y6Tpi6G0pKtvMkF3LcD9TS3adboix8H2ZXx\n4L7MRQKBgQC0OO3qqNXOjIVYWOoqXLybOY/Wqu9lxCAgGyCYaMcstnBI7W0MZTBr\n/syluvGsaFc1sE7MMGOOzKi1tF4YvDmSnzA/R1nmaPguuD9fOA+w7Pwkv5vLvuJq\n2U7EeNwxq1I1L3Ag6E7wH4BHLHd4TKaZR6agFkn8oomz71yZPGjuZQ==\n-----END RSA PRIVATE KEY-----");
                Assertions.assertThat((Object)((MultipleJGitEnvironmentProperties.PatternMatchingJGitEnvironmentProperties)sshUriProperties.getRepos().get("repo1"))).isNotNull();
                Assertions.assertThat((String)((MultipleJGitEnvironmentProperties.PatternMatchingJGitEnvironmentProperties)sshUriProperties.getRepos().get("repo1")).getPrivateKey()).isEqualTo("-----BEGIN RSA PRIVATE KEY-----\nMIIEpgIBAAKCAQEAx4UbaDzY5xjW6hc9jwN0mX33XpTDVW9WqHp5AKaRbtAC3DqX\nIXFMPgw3K45jxRb93f8tv9vL3rD9CUG1Gv4FM+o7ds7FRES5RTjv2RT/JVNJCoqF\nol8+ngLqRZCyBtQN7zYByWMRirPGoDUqdPYrj2yq+ObBBNhg5N+hOwKjjpzdj2Ud\n1l7R+wxIqmJo1IYyy16xS8WsjyQuyC0lL456qkd5BDZ0Ag8j2X9H9D5220Ln7s9i\noezTipXipS7p7Jekf3Ywx6abJwOmB0rX79dV4qiNcGgzATnG1PkXxqt76VhcGa0W\nDDVHEEYGbSQ6hIGSh0I7BQun0aLRZojfE3gqHQIDAQABAoIBAQCZmGrk8BK6tXCd\nfY6yTiKxFzwb38IQP0ojIUWNrq0+9Xt+NsypviLHkXfXXCKKU4zUHeIGVRq5MN9b\nBO56/RrcQHHOoJdUWuOV2qMqJvPUtC0CpGkD+valhfD75MxoXU7s3FK7yjxy3rsG\nEmfA6tHV8/4a5umo5TqSd2YTm5B19AhRqiuUVI1wTB41DjULUGiMYrnYrhzQlVvj\n5MjnKTlYu3V8PoYDfv1GmxPPh6vlpafXEeEYN8VB97e5x3DGHjZ5UrurAmTLTdO8\n+AahyoKsIY612TkkQthJlt7FJAwnCGMgY6podzzvzICLFmmTXYiZ/28I4BX/mOSe\npZVnfRixAoGBAO6Uiwt40/PKs53mCEWngslSCsh9oGAaLTf/XdvMns5VmuyyAyKG\nti8Ol5wqBMi4GIUzjbgUvSUt+IowIrG3f5tN85wpjQ1UGVcpTnl5Qo9xaS1PFScQ\nxrtWZ9eNj2TsIAMp/svJsyGG3OibxfnuAIpSXNQiJPwRlW3irzpGgVx/AoGBANYW\ndnhshUcEHMJi3aXwR12OTDnaLoanVGLwLnkqLSYUZA7ZegpKq90UAuBdcEfgdpyi\nPhKpeaeIiAaNnFo8m9aoTKr+7I6/uMTlwrVnfrsVTZv3orxjwQV20YIBCVRKD1uX\nVhE0ozPZxwwKSPAFocpyWpGHGreGF1AIYBE9UBtjAoGBAI8bfPgJpyFyMiGBjO6z\nFwlJc/xlFqDusrcHL7abW5qq0L4v3R+FrJw3ZYufzLTVcKfdj6GelwJJO+8wBm+R\ngTKYJItEhT48duLIfTDyIpHGVm9+I1MGhh5zKuCqIhxIYr9jHloBB7kRm0rPvYY4\nVAykcNgyDvtAVODP+4m6JvhjAoGBALbtTqErKN47V0+JJpapLnF0KxGrqeGIjIRV\ncYA6V4WYGr7NeIfesecfOC356PyhgPfpcVyEztwlvwTKb3RzIT1TZN8fH4YBr6Ee\nKTbTjefRFhVUjQqnucAvfGi29f+9oE3Ei9f7wA+H35ocF6JvTYUsHNMIO/3gZ38N\nCPjyCMa9AoGBAMhsITNe3QcbsXAbdUR00dDsIFVROzyFJ2m40i4KCRM35bC/BIBs\nq0TY3we+ERB40U8Z2BvU61QuwaunJ2+uGadHo58VSVdggqAo0BSkH58innKKt96J\n69pcVH/4rmLbXdcmNYGm6iu+MlPQk4BUZknHSmVHIFdJ0EPupVaQ8RHT\n-----END RSA PRIVATE KEY-----\n");
            }
        }

        @SpringBootTest(classes={TestConfigServerApplication.class, SshPropertyValidator.class}, webEnvironment=SpringBootTest.WebEnvironment.RANDOM_PORT, properties={"spring.config.additional-location=optional:file:/ssh/,optional:classpath:/ssh/", "spring.config.name:ssh-nested-settings"})
        @ActiveProfiles(value={"test", "git"})
        public static class StaticTest {
            @Autowired
            private MultipleJGitEnvironmentRepository jGitEnvironmentRepository;

            @Test
            public void sshPropertiesWithinNestedRepo() {
                TransportConfigCallback callback = this.jGitEnvironmentRepository.getTransportConfigCallback();
                Assertions.assertThat((Object)callback).isInstanceOf(PropertiesBasedSshTransportConfigCallback.class);
                PropertiesBasedSshTransportConfigCallback configCallback = (PropertiesBasedSshTransportConfigCallback)callback;
                MultipleJGitEnvironmentProperties sshUriProperties = configCallback.getSshUriProperties();
                Assertions.assertThat((String)configCallback.getSshUriProperties().getPrivateKey()).isEqualTo("-----BEGIN RSA PRIVATE KEY-----\nMIIEpAIBAAKCAQEAoqyz6YaYMTr7L8GLPSQpAQXaM04gRx4CCsGK2kfLQdw4BlqI\nyyxp38YcuZG9cUDBAxby+K2TKmwHaC1R61QTwbPuCRdIPrDwRz+FLoegm3iDLCmn\nuP6rjZDneYsqfU1KSdrOwIbCnONfDdvYL/vnZC/o8DDMlk5Orw2SfHkT3pq0o8km\nayBwN4Sf3bpyWTY0oZcmNeSCCoIdE59k8Pa7/t9bwY9caLj05C3DEsjucc7Ei/Eq\nTOyGyobtXwaya5CqKLUHes74Poz1aEP/yVFdUud91uezd8ZK1P1t5/ZKA3R6aHir\n+diDJ2/GQ2tD511FW46yw+EtBUJTO6ADVv4UnQIDAQABAoIBAF+5qwEfX82QfKFk\njfADqFFexUDtl1biFKeJrpC2MKhn01wByH9uejrhFKQqW8UaKroLthyZ34DWIyGt\nlDnHGv0gSVF2LuAdNLdobJGt49e4+c9yD61vxzm97Eh8mRs08SM2q/VlF35E2fmI\nxdWusUImYzd8L9e+6tRd8zZl9UhG5vR5XIstKqxC6S0g79aAt0hasE4Gw1FKOf2V\n4mlL15atjQSKCPdOicuyc4zpjAtU1A9AfF51iG8oOUuJebPW8tCftfOQxaeGFgMG\n7M9aai1KzXR6M5IBAKEv31yBvz/SHTneP7oZXNLeC1GIR420PKybmeZdNK8BbEAu\n3reKgm0CgYEA03Sx8JoF5UBsIvFPpP1fjSlTgKryM5EJR6KQtj5e4YfyxccJepN8\nq4MrqDfNKleG/a1acEtDMhBNovU7Usp2QIP7zpAeioHBOhmE5WSieZGc3icOGWWq\nmRkdulSONruqWKv76ZoluxftekE03bDhZDNlcCgmrslEKB/ufHd2oc8CgYEAxPFa\nlKOdSeiYFV5CtvO8Ro8em6rGpSsVz4qkPxbeBqUDCb9KXHhq6YrhRxOIfQJKfT7M\nZFCn8ArJXKgOGu+KsvwIErFHF9g2jJMG4DOUTpkQgi2yveihFxcmz/AltyVXgrnv\nZWQbAerH77pdKKhNivLGgEv72GYawdYjYNjemdMCgYA2kEMmMahZyrDcp2YEzfit\nBT/t0K6kzcUWPgWXcSqsiZcEn+J7RbmCzFskkhmX1nQX23adyV3yejB+X0dKisHO\nzf/ZAmlPFkJVCqa3RquCMSfIT02dEhXeYZPBM/Zqeyxuqxpa4hLgX0FBLbhFiFHw\nuC5xrXql2XuD2xF//peXEwKBgQC+pa28Cg7vRxxCQzduB9CQtWc55j3aEjVQ7bNF\n54sS/5ZLT0Ra8677WZfuyDfuW9NkHvCZg4Ku2qJG8eCFrrGjxlrCTZ62tHVJ6+JS\nE1xUIdRbUIWhVZrr0VufG6hG/P0T7Y6Tpi6G0pKtvMkF3LcD9TS3adboix8H2ZXx\n4L7MRQKBgQC0OO3qqNXOjIVYWOoqXLybOY/Wqu9lxCAgGyCYaMcstnBI7W0MZTBr\n/syluvGsaFc1sE7MMGOOzKi1tF4YvDmSnzA/R1nmaPguuD9fOA+w7Pwkv5vLvuJq\n2U7EeNwxq1I1L3Ag6E7wH4BHLHd4TKaZR6agFkn8oomz71yZPGjuZQ==\n-----END RSA PRIVATE KEY-----");
                Assertions.assertThat((Object)((MultipleJGitEnvironmentProperties.PatternMatchingJGitEnvironmentProperties)sshUriProperties.getRepos().get("repo1"))).isNotNull();
                Assertions.assertThat((String)((MultipleJGitEnvironmentProperties.PatternMatchingJGitEnvironmentProperties)sshUriProperties.getRepos().get("repo1")).getPrivateKey()).isEqualTo("-----BEGIN RSA PRIVATE KEY-----\nMIIEpgIBAAKCAQEAx4UbaDzY5xjW6hc9jwN0mX33XpTDVW9WqHp5AKaRbtAC3DqX\nIXFMPgw3K45jxRb93f8tv9vL3rD9CUG1Gv4FM+o7ds7FRES5RTjv2RT/JVNJCoqF\nol8+ngLqRZCyBtQN7zYByWMRirPGoDUqdPYrj2yq+ObBBNhg5N+hOwKjjpzdj2Ud\n1l7R+wxIqmJo1IYyy16xS8WsjyQuyC0lL456qkd5BDZ0Ag8j2X9H9D5220Ln7s9i\noezTipXipS7p7Jekf3Ywx6abJwOmB0rX79dV4qiNcGgzATnG1PkXxqt76VhcGa0W\nDDVHEEYGbSQ6hIGSh0I7BQun0aLRZojfE3gqHQIDAQABAoIBAQCZmGrk8BK6tXCd\nfY6yTiKxFzwb38IQP0ojIUWNrq0+9Xt+NsypviLHkXfXXCKKU4zUHeIGVRq5MN9b\nBO56/RrcQHHOoJdUWuOV2qMqJvPUtC0CpGkD+valhfD75MxoXU7s3FK7yjxy3rsG\nEmfA6tHV8/4a5umo5TqSd2YTm5B19AhRqiuUVI1wTB41DjULUGiMYrnYrhzQlVvj\n5MjnKTlYu3V8PoYDfv1GmxPPh6vlpafXEeEYN8VB97e5x3DGHjZ5UrurAmTLTdO8\n+AahyoKsIY612TkkQthJlt7FJAwnCGMgY6podzzvzICLFmmTXYiZ/28I4BX/mOSe\npZVnfRixAoGBAO6Uiwt40/PKs53mCEWngslSCsh9oGAaLTf/XdvMns5VmuyyAyKG\nti8Ol5wqBMi4GIUzjbgUvSUt+IowIrG3f5tN85wpjQ1UGVcpTnl5Qo9xaS1PFScQ\nxrtWZ9eNj2TsIAMp/svJsyGG3OibxfnuAIpSXNQiJPwRlW3irzpGgVx/AoGBANYW\ndnhshUcEHMJi3aXwR12OTDnaLoanVGLwLnkqLSYUZA7ZegpKq90UAuBdcEfgdpyi\nPhKpeaeIiAaNnFo8m9aoTKr+7I6/uMTlwrVnfrsVTZv3orxjwQV20YIBCVRKD1uX\nVhE0ozPZxwwKSPAFocpyWpGHGreGF1AIYBE9UBtjAoGBAI8bfPgJpyFyMiGBjO6z\nFwlJc/xlFqDusrcHL7abW5qq0L4v3R+FrJw3ZYufzLTVcKfdj6GelwJJO+8wBm+R\ngTKYJItEhT48duLIfTDyIpHGVm9+I1MGhh5zKuCqIhxIYr9jHloBB7kRm0rPvYY4\nVAykcNgyDvtAVODP+4m6JvhjAoGBALbtTqErKN47V0+JJpapLnF0KxGrqeGIjIRV\ncYA6V4WYGr7NeIfesecfOC356PyhgPfpcVyEztwlvwTKb3RzIT1TZN8fH4YBr6Ee\nKTbTjefRFhVUjQqnucAvfGi29f+9oE3Ei9f7wA+H35ocF6JvTYUsHNMIO/3gZ38N\nCPjyCMa9AoGBAMhsITNe3QcbsXAbdUR00dDsIFVROzyFJ2m40i4KCRM35bC/BIBs\nq0TY3we+ERB40U8Z2BvU61QuwaunJ2+uGadHo58VSVdggqAo0BSkH58innKKt96J\n69pcVH/4rmLbXdcmNYGm6iu+MlPQk4BUZknHSmVHIFdJ0EPupVaQ8RHT\n-----END RSA PRIVATE KEY-----\n");
            }
        }
    }

    public static class PrivateKeyPropertyWithLineBreaks {

        @SpringBootTest(classes={TestConfigServerApplication.class, SshPropertyValidator.class}, webEnvironment=SpringBootTest.WebEnvironment.RANDOM_PORT, properties={"spring.config.additional-location=optional:file:/ssh/,optional:classpath:/ssh/", "spring.config.name:ssh-private-key-newline-list"})
        @ActiveProfiles(value={"test", "composite"})
        public static class ListTest {
            @Autowired
            private MultipleJGitEnvironmentRepository jGitEnvironmentRepository;

            @Test
            public void privateKeyPropertyWithLineBreaks() {
                TransportConfigCallback callback = this.jGitEnvironmentRepository.getTransportConfigCallback();
                Assertions.assertThat((Object)callback).isInstanceOf(PropertiesBasedSshTransportConfigCallback.class);
                PropertiesBasedSshTransportConfigCallback configCallback = (PropertiesBasedSshTransportConfigCallback)callback;
                Assertions.assertThat((String)configCallback.getSshUriProperties().getPrivateKey()).isEqualTo("-----BEGIN RSA PRIVATE KEY-----\nMIIEpAIBAAKCAQEAoqyz6YaYMTr7L8GLPSQpAQXaM04gRx4CCsGK2kfLQdw4BlqI\nyyxp38YcuZG9cUDBAxby+K2TKmwHaC1R61QTwbPuCRdIPrDwRz+FLoegm3iDLCmn\nuP6rjZDneYsqfU1KSdrOwIbCnONfDdvYL/vnZC/o8DDMlk5Orw2SfHkT3pq0o8km\nayBwN4Sf3bpyWTY0oZcmNeSCCoIdE59k8Pa7/t9bwY9caLj05C3DEsjucc7Ei/Eq\nTOyGyobtXwaya5CqKLUHes74Poz1aEP/yVFdUud91uezd8ZK1P1t5/ZKA3R6aHir\n+diDJ2/GQ2tD511FW46yw+EtBUJTO6ADVv4UnQIDAQABAoIBAF+5qwEfX82QfKFk\njfADqFFexUDtl1biFKeJrpC2MKhn01wByH9uejrhFKQqW8UaKroLthyZ34DWIyGt\nlDnHGv0gSVF2LuAdNLdobJGt49e4+c9yD61vxzm97Eh8mRs08SM2q/VlF35E2fmI\nxdWusUImYzd8L9e+6tRd8zZl9UhG5vR5XIstKqxC6S0g79aAt0hasE4Gw1FKOf2V\n4mlL15atjQSKCPdOicuyc4zpjAtU1A9AfF51iG8oOUuJebPW8tCftfOQxaeGFgMG\n7M9aai1KzXR6M5IBAKEv31yBvz/SHTneP7oZXNLeC1GIR420PKybmeZdNK8BbEAu\n3reKgm0CgYEA03Sx8JoF5UBsIvFPpP1fjSlTgKryM5EJR6KQtj5e4YfyxccJepN8\nq4MrqDfNKleG/a1acEtDMhBNovU7Usp2QIP7zpAeioHBOhmE5WSieZGc3icOGWWq\nmRkdulSONruqWKv76ZoluxftekE03bDhZDNlcCgmrslEKB/ufHd2oc8CgYEAxPFa\nlKOdSeiYFV5CtvO8Ro8em6rGpSsVz4qkPxbeBqUDCb9KXHhq6YrhRxOIfQJKfT7M\nZFCn8ArJXKgOGu+KsvwIErFHF9g2jJMG4DOUTpkQgi2yveihFxcmz/AltyVXgrnv\nZWQbAerH77pdKKhNivLGgEv72GYawdYjYNjemdMCgYA2kEMmMahZyrDcp2YEzfit\nBT/t0K6kzcUWPgWXcSqsiZcEn+J7RbmCzFskkhmX1nQX23adyV3yejB+X0dKisHO\nzf/ZAmlPFkJVCqa3RquCMSfIT02dEhXeYZPBM/Zqeyxuqxpa4hLgX0FBLbhFiFHw\nuC5xrXql2XuD2xF//peXEwKBgQC+pa28Cg7vRxxCQzduB9CQtWc55j3aEjVQ7bNF\n54sS/5ZLT0Ra8677WZfuyDfuW9NkHvCZg4Ku2qJG8eCFrrGjxlrCTZ62tHVJ6+JS\nE1xUIdRbUIWhVZrr0VufG6hG/P0T7Y6Tpi6G0pKtvMkF3LcD9TS3adboix8H2ZXx\n4L7MRQKBgQC0OO3qqNXOjIVYWOoqXLybOY/Wqu9lxCAgGyCYaMcstnBI7W0MZTBr\n/syluvGsaFc1sE7MMGOOzKi1tF4YvDmSnzA/R1nmaPguuD9fOA+w7Pwkv5vLvuJq\n2U7EeNwxq1I1L3Ag6E7wH4BHLHd4TKaZR6agFkn8oomz71yZPGjuZQ==\n-----END RSA PRIVATE KEY-----");
            }
        }

        @SpringBootTest(classes={TestConfigServerApplication.class, SshPropertyValidator.class}, webEnvironment=SpringBootTest.WebEnvironment.RANDOM_PORT, properties={"spring.config.additional-location=optional:file:/ssh/,optional:classpath:/ssh/", "spring.config.name:ssh-private-key-newline"})
        @ActiveProfiles(value={"test", "git"})
        public static class StaticTest {
            @Autowired
            private MultipleJGitEnvironmentRepository jGitEnvironmentRepository;

            @Test
            public void privateKeyPropertyWithLineBreaks() {
                TransportConfigCallback callback = this.jGitEnvironmentRepository.getTransportConfigCallback();
                Assertions.assertThat((Object)callback).isInstanceOf(PropertiesBasedSshTransportConfigCallback.class);
                PropertiesBasedSshTransportConfigCallback configCallback = (PropertiesBasedSshTransportConfigCallback)callback;
                Assertions.assertThat((String)configCallback.getSshUriProperties().getPrivateKey()).isEqualTo("-----BEGIN RSA PRIVATE KEY-----\nMIIEpAIBAAKCAQEAoqyz6YaYMTr7L8GLPSQpAQXaM04gRx4CCsGK2kfLQdw4BlqI\nyyxp38YcuZG9cUDBAxby+K2TKmwHaC1R61QTwbPuCRdIPrDwRz+FLoegm3iDLCmn\nuP6rjZDneYsqfU1KSdrOwIbCnONfDdvYL/vnZC/o8DDMlk5Orw2SfHkT3pq0o8km\nayBwN4Sf3bpyWTY0oZcmNeSCCoIdE59k8Pa7/t9bwY9caLj05C3DEsjucc7Ei/Eq\nTOyGyobtXwaya5CqKLUHes74Poz1aEP/yVFdUud91uezd8ZK1P1t5/ZKA3R6aHir\n+diDJ2/GQ2tD511FW46yw+EtBUJTO6ADVv4UnQIDAQABAoIBAF+5qwEfX82QfKFk\njfADqFFexUDtl1biFKeJrpC2MKhn01wByH9uejrhFKQqW8UaKroLthyZ34DWIyGt\nlDnHGv0gSVF2LuAdNLdobJGt49e4+c9yD61vxzm97Eh8mRs08SM2q/VlF35E2fmI\nxdWusUImYzd8L9e+6tRd8zZl9UhG5vR5XIstKqxC6S0g79aAt0hasE4Gw1FKOf2V\n4mlL15atjQSKCPdOicuyc4zpjAtU1A9AfF51iG8oOUuJebPW8tCftfOQxaeGFgMG\n7M9aai1KzXR6M5IBAKEv31yBvz/SHTneP7oZXNLeC1GIR420PKybmeZdNK8BbEAu\n3reKgm0CgYEA03Sx8JoF5UBsIvFPpP1fjSlTgKryM5EJR6KQtj5e4YfyxccJepN8\nq4MrqDfNKleG/a1acEtDMhBNovU7Usp2QIP7zpAeioHBOhmE5WSieZGc3icOGWWq\nmRkdulSONruqWKv76ZoluxftekE03bDhZDNlcCgmrslEKB/ufHd2oc8CgYEAxPFa\nlKOdSeiYFV5CtvO8Ro8em6rGpSsVz4qkPxbeBqUDCb9KXHhq6YrhRxOIfQJKfT7M\nZFCn8ArJXKgOGu+KsvwIErFHF9g2jJMG4DOUTpkQgi2yveihFxcmz/AltyVXgrnv\nZWQbAerH77pdKKhNivLGgEv72GYawdYjYNjemdMCgYA2kEMmMahZyrDcp2YEzfit\nBT/t0K6kzcUWPgWXcSqsiZcEn+J7RbmCzFskkhmX1nQX23adyV3yejB+X0dKisHO\nzf/ZAmlPFkJVCqa3RquCMSfIT02dEhXeYZPBM/Zqeyxuqxpa4hLgX0FBLbhFiFHw\nuC5xrXql2XuD2xF//peXEwKBgQC+pa28Cg7vRxxCQzduB9CQtWc55j3aEjVQ7bNF\n54sS/5ZLT0Ra8677WZfuyDfuW9NkHvCZg4Ku2qJG8eCFrrGjxlrCTZ62tHVJ6+JS\nE1xUIdRbUIWhVZrr0VufG6hG/P0T7Y6Tpi6G0pKtvMkF3LcD9TS3adboix8H2ZXx\n4L7MRQKBgQC0OO3qqNXOjIVYWOoqXLybOY/Wqu9lxCAgGyCYaMcstnBI7W0MZTBr\n/syluvGsaFc1sE7MMGOOzKi1tF4YvDmSnzA/R1nmaPguuD9fOA+w7Pwkv5vLvuJq\n2U7EeNwxq1I1L3Ag6E7wH4BHLHd4TKaZR6agFkn8oomz71yZPGjuZQ==\n-----END RSA PRIVATE KEY-----");
            }
        }
    }

    public static class PropertyBasedCallbackTest {

        @SpringBootTest(classes={TestConfigServerApplication.class, SshPropertyValidator.class}, webEnvironment=SpringBootTest.WebEnvironment.RANDOM_PORT, properties={"spring.config.additional-location=optional:file:/ssh/,optional:classpath:/ssh/", "spring.config.name:ssh-private-key-block-list"})
        @ActiveProfiles(value={"test", "composite"})
        public static class ListTest {
            @Autowired
            private MultipleJGitEnvironmentRepository jGitEnvironmentRepository;

            @Test
            public void propertyBasedTransportCallbackIsConfigured() {
                TransportConfigCallback transportConfigCallback = this.jGitEnvironmentRepository.getTransportConfigCallback();
                Assertions.assertThat((Object)transportConfigCallback).isInstanceOf(PropertiesBasedSshTransportConfigCallback.class);
            }

            @Test
            public void propertyBasedSessionFactoryIsUsedForSshTransports() throws Exception {
                this.jGitEnvironmentRepository.afterPropertiesSet();
                SshTransport sshTransport = DummySshTransport.newInstance();
                this.jGitEnvironmentRepository.getTransportConfigCallback().configure((Transport)sshTransport);
                Assertions.assertThat((Object)sshTransport.getSshSessionFactory()).isInstanceOf(PropertyBasedSshSessionFactory.class);
            }
        }

        @SpringBootTest(classes={TestConfigServerApplication.class, SshPropertyValidator.class}, webEnvironment=SpringBootTest.WebEnvironment.RANDOM_PORT, properties={"spring.config.additional-location=optional:file:/ssh/,optional:classpath:/ssh/", "spring.config.name:ssh-private-key-block"})
        @ActiveProfiles(value={"test", "git"})
        public static class StaticTest {
            @Autowired
            private MultipleJGitEnvironmentRepository jGitEnvironmentRepository;

            @Test
            public void propertyBasedTransportCallbackIsConfigured() {
                TransportConfigCallback transportConfigCallback = this.jGitEnvironmentRepository.getTransportConfigCallback();
                Assertions.assertThat((Object)transportConfigCallback).isInstanceOf(PropertiesBasedSshTransportConfigCallback.class);
            }

            @Test
            public void propertyBasedSessionFactoryIsUsedForSshTransports() throws Exception {
                this.jGitEnvironmentRepository.afterPropertiesSet();
                SshTransport sshTransport = DummySshTransport.newInstance();
                this.jGitEnvironmentRepository.getTransportConfigCallback().configure((Transport)sshTransport);
                Assertions.assertThat((Object)sshTransport.getSshSessionFactory()).isInstanceOf(PropertyBasedSshSessionFactory.class);
            }
        }
    }
}

