/*
 * Decompiled with CFR 0.152.
 */
package org.springframework.cloud.gateway.filter.ratelimit;

import java.util.List;
import java.util.Map;
import java.util.UUID;
import org.assertj.core.api.AbstractBooleanAssert;
import org.assertj.core.api.Assertions;
import org.hamcrest.Matcher;
import org.hamcrest.Matchers;
import org.junit.Assume;
import org.junit.jupiter.api.AfterEach;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Tag;
import org.junit.jupiter.api.Test;
import org.junitpioneer.jupiter.RetryingTest;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.SpringBootConfiguration;
import org.springframework.boot.autoconfigure.EnableAutoConfiguration;
import org.springframework.boot.test.context.SpringBootTest;
import org.springframework.cloud.gateway.filter.ratelimit.RateLimiter;
import org.springframework.cloud.gateway.filter.ratelimit.RedisRateLimiter;
import org.springframework.cloud.gateway.test.BaseWebClientTests;
import org.springframework.context.annotation.Import;
import org.springframework.test.annotation.DirtiesContext;
import org.springframework.test.context.DynamicPropertyRegistry;
import org.springframework.test.context.DynamicPropertySource;
import org.testcontainers.containers.GenericContainer;
import org.testcontainers.junit.jupiter.Container;
import org.testcontainers.junit.jupiter.Testcontainers;

@SpringBootTest(webEnvironment=SpringBootTest.WebEnvironment.RANDOM_PORT)
@DirtiesContext
@Testcontainers
@Tag(value="DockerRequired")
public class RedisRateLimiterTests
extends BaseWebClientTests {
    @Container
    public static GenericContainer redis = new GenericContainer("redis:5.0.14-alpine").withExposedPorts(new Integer[]{6379});
    @Autowired
    private RedisRateLimiter rateLimiter;

    @DynamicPropertySource
    static void containerProperties(DynamicPropertyRegistry registry) {
        registry.add("spring.data.redis.host", () -> ((GenericContainer)redis).getHost());
        registry.add("spring.data.redis.port", () -> ((GenericContainer)redis).getFirstMappedPort());
    }

    @BeforeEach
    public void setUp() {
        Assume.assumeThat((String)"Ignore on Circle", (Object)System.getenv("CIRCLECI"), (Matcher)Matchers.is((Matcher)Matchers.nullValue()));
    }

    @AfterEach
    public void tearDown() {
        this.rateLimiter.setIncludeHeaders(true);
    }

    @RetryingTest(value=3)
    public void redisRateLimiterWorks() throws Exception {
        String id = UUID.randomUUID().toString();
        int replenishRate = 10;
        int burstCapacity = 2 * replenishRate;
        int requestedTokens = 1;
        String routeId = "myroute";
        this.rateLimiter.getConfig().put(routeId, new RedisRateLimiter.Config().setBurstCapacity(burstCapacity).setReplenishRate(replenishRate).setRequestedTokens(requestedTokens));
        this.checkLimitEnforced(id, replenishRate, burstCapacity, requestedTokens, routeId);
    }

    @Test
    public void redisRateLimiterWorksForLowRates() throws Exception {
        String id = UUID.randomUUID().toString();
        int replenishRate = 1;
        int burstCapacity = 3;
        int requestedTokens = 3;
        String routeId = "low_rate_route";
        this.rateLimiter.getConfig().put(routeId, new RedisRateLimiter.Config().setBurstCapacity(burstCapacity).setReplenishRate(replenishRate).setRequestedTokens(requestedTokens));
        this.checkLimitEnforced(id, replenishRate, burstCapacity, requestedTokens, routeId);
    }

    @Test
    public void redisRateLimiterWorksForZeroBurstCapacity() throws Exception {
        String id = UUID.randomUUID().toString();
        int replenishRate = 1;
        int burstCapacity = 0;
        int requestedTokens = 1;
        String routeId = "zero_burst_capacity_route";
        this.rateLimiter.getConfig().put(routeId, new RedisRateLimiter.Config().setBurstCapacity(burstCapacity).setReplenishRate(replenishRate).setRequestedTokens(requestedTokens));
        RateLimiter.Response response = (RateLimiter.Response)this.rateLimiter.isAllowed(routeId, id).block();
        Assertions.assertThat((boolean)response.isAllowed()).isFalse();
    }

    @Test
    public void keysUseRedisKeyHashTags() {
        Assertions.assertThat((List)RedisRateLimiter.getKeys((String)"1")).containsExactly((Object[])new String[]{"request_rate_limiter.{1}.tokens", "request_rate_limiter.{1}.timestamp"});
    }

    @Test
    public void redisRateLimiterDoesNotSendHeadersIfDeactivated() throws Exception {
        String id = UUID.randomUUID().toString();
        String routeId = "myroute";
        this.rateLimiter.setIncludeHeaders(false);
        RateLimiter.Response response = (RateLimiter.Response)this.rateLimiter.isAllowed(routeId, id).block();
        Assertions.assertThat((boolean)response.isAllowed()).isTrue();
        Assertions.assertThat((Map)response.getHeaders()).doesNotContainKey((Object)"X-RateLimit-Remaining");
        Assertions.assertThat((Map)response.getHeaders()).doesNotContainKey((Object)"X-RateLimit-Replenish-Rate");
        Assertions.assertThat((Map)response.getHeaders()).doesNotContainKey((Object)"X-RateLimit-Burst-Capacity");
        Assertions.assertThat((Map)response.getHeaders()).doesNotContainKey((Object)"X-RateLimit-Requested-Tokens");
    }

    private void checkLimitEnforced(String id, int replenishRate, int burstCapacity, int requestedTokens, String routeId) throws InterruptedException {
        this.simulateBurst(id, replenishRate, burstCapacity, requestedTokens, routeId);
        this.checkLimitReached(id, burstCapacity, routeId);
        Thread.sleep(Math.max(1, requestedTokens / replenishRate) * 1000);
        this.checkSteadyState(id, replenishRate, routeId);
    }

    private void simulateBurst(String id, int replenishRate, int burstCapacity, int requestedTokens, String routeId) {
        for (int i = 0; i < burstCapacity / requestedTokens; ++i) {
            RateLimiter.Response response = (RateLimiter.Response)this.rateLimiter.isAllowed(routeId, id).block();
            ((AbstractBooleanAssert)Assertions.assertThat((boolean)response.isAllowed()).as("Burst # %s is allowed", new Object[]{i})).isTrue();
            Assertions.assertThat((Map)response.getHeaders()).containsKey((Object)"X-RateLimit-Remaining");
            Assertions.assertThat((Map)response.getHeaders()).containsEntry((Object)"X-RateLimit-Replenish-Rate", (Object)String.valueOf(replenishRate));
            Assertions.assertThat((Map)response.getHeaders()).containsEntry((Object)"X-RateLimit-Burst-Capacity", (Object)String.valueOf(burstCapacity));
            Assertions.assertThat((Map)response.getHeaders()).containsEntry((Object)"X-RateLimit-Requested-Tokens", (Object)String.valueOf(requestedTokens));
        }
    }

    private void checkLimitReached(String id, int burstCapacity, String routeId) {
        RateLimiter.Response response = (RateLimiter.Response)this.rateLimiter.isAllowed(routeId, id).block();
        if (response.isAllowed()) {
            response = (RateLimiter.Response)this.rateLimiter.isAllowed(routeId, id).block();
        }
        ((AbstractBooleanAssert)Assertions.assertThat((boolean)response.isAllowed()).as("Burst # %s is not allowed", new Object[]{burstCapacity})).isFalse();
    }

    private void checkSteadyState(String id, int replenishRate, String routeId) {
        RateLimiter.Response response;
        for (int i = 0; i < replenishRate; ++i) {
            response = (RateLimiter.Response)this.rateLimiter.isAllowed(routeId, id).block();
            ((AbstractBooleanAssert)Assertions.assertThat((boolean)response.isAllowed()).as("steady state # %s is allowed", new Object[]{i})).isTrue();
        }
        response = (RateLimiter.Response)this.rateLimiter.isAllowed(routeId, id).block();
        ((AbstractBooleanAssert)Assertions.assertThat((boolean)response.isAllowed()).as("steady state # %s is allowed", new Object[]{replenishRate})).isFalse();
    }

    @EnableAutoConfiguration
    @SpringBootConfiguration
    @Import(value={BaseWebClientTests.DefaultTestConfig.class})
    public static class TestConfig {
    }
}

