/*
 * Decompiled with CFR 0.152.
 */
package org.terrakube.terraform;

import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.JavaType;
import com.fasterxml.jackson.databind.ObjectMapper;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.net.URL;
import java.util.List;
import java.util.stream.Collectors;
import java.util.zip.ZipEntry;
import java.util.zip.ZipInputStream;
import org.apache.commons.io.FileUtils;
import org.apache.commons.io.FilenameUtils;
import org.apache.commons.lang3.SystemUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.terrakube.terraform.TerraformBuild;
import org.terrakube.terraform.TerraformResponse;
import org.terrakube.terraform.TerraformVersion;
import org.terrakube.terraform.TofuAsset;
import org.terrakube.terraform.TofuRelease;

public class TerraformDownloader {
    private static final Logger log = LoggerFactory.getLogger(TerraformDownloader.class);
    private static final String TERRAFORM_DOWNLOAD_DIRECTORY = "/.terraform-spring-boot/download/";
    private static final String TOFU_DOWNLOAD_DIRECTORY = "/.terraform-spring-boot/download/tofu/";
    private static final String TERRAFORM_DIRECTORY = "/.terraform-spring-boot/terraform/";
    private static final String TOFU_DIRECTORY = "/.terraform-spring-boot/tofu/";
    private static final String TEMP_DIRECTORY = "/.terraform-spring-boot/";
    public static final String TERRAFORM_RELEASES_URL = "https://releases.hashicorp.com/terraform/index.json";
    public static final String TOFU_RELEASES_URL = "https://api.github.com/repos/opentofu/opentofu/releases";
    private TerraformResponse terraformReleases;
    private List<TofuRelease> tofuReleases;
    private File terraformDownloadDirectory;
    private File tofuDownloadDirectory;
    private File terraformDirectory;
    private String userHomeDirectory;

    public TerraformDownloader() {
        try {
            log.info("Initialize Terraform and Tofu Downloader using default URL");
            this.createDownloadTempDirectory();
            this.createDownloadTofuTempDirectory();
            this.getTerraformReleases(TERRAFORM_RELEASES_URL);
            this.getTofuReleases(TOFU_RELEASES_URL);
        }
        catch (IOException ex) {
            log.error(ex.getMessage());
        }
    }

    public TerraformDownloader(String terraformReleasesUrl, String tofuReleasesUrl) {
        log.info("Initialize TerraformDownloader using custom URL");
        try {
            this.createDownloadTempDirectory();
            this.createDownloadTofuTempDirectory();
            this.getTerraformReleases(terraformReleasesUrl);
            this.getTofuReleases(tofuReleasesUrl);
        }
        catch (IOException ex) {
            log.error(ex.getMessage());
        }
    }

    private void createDownloadTempDirectory() throws IOException {
        this.userHomeDirectory = FileUtils.getUserDirectoryPath();
        log.info("User Home Directory: {}", (Object)this.userHomeDirectory);
        String downloadPath = this.userHomeDirectory.concat(FilenameUtils.separatorsToSystem((String)TERRAFORM_DOWNLOAD_DIRECTORY));
        this.terraformDownloadDirectory = new File(downloadPath);
        FileUtils.forceMkdir((File)this.terraformDownloadDirectory);
        log.info("Validate/Create download temp directory: {}", (Object)downloadPath);
        String terrafomVersionPath = this.userHomeDirectory.concat(FilenameUtils.separatorsToSystem((String)TERRAFORM_DIRECTORY));
        this.terraformDirectory = new File(terrafomVersionPath);
        FileUtils.forceMkdir((File)this.terraformDirectory);
        log.info("Validate/Create terraform directory: {}", (Object)terrafomVersionPath);
    }

    private void createDownloadTofuTempDirectory() throws IOException {
        this.userHomeDirectory = FileUtils.getUserDirectoryPath();
        log.info("User Home Directory for tofu download: {}", (Object)this.userHomeDirectory);
        String tofuDownloadPath = this.userHomeDirectory.concat(FilenameUtils.separatorsToSystem((String)TOFU_DOWNLOAD_DIRECTORY));
        this.tofuDownloadDirectory = new File(tofuDownloadPath);
        FileUtils.forceMkdir((File)this.tofuDownloadDirectory);
        log.info("Validate/Create tofu download temp directory: {}", (Object)tofuDownloadPath);
        String tofuVersionPath = this.userHomeDirectory.concat(FilenameUtils.separatorsToSystem((String)TOFU_DIRECTORY));
        this.terraformDirectory = new File(tofuVersionPath);
        FileUtils.forceMkdir((File)this.terraformDirectory);
        log.info("Validate/Create tofu directory: {}", (Object)tofuVersionPath);
    }

    private void getTerraformReleases(String terraformReleasesUrl) throws IOException {
        File tempFile;
        log.info("Downloading terraform releases list");
        ObjectMapper objectMapper = new ObjectMapper();
        String tempPath = this.userHomeDirectory.concat(FilenameUtils.separatorsToSystem((String)TEMP_DIRECTORY));
        if (SystemUtils.IS_OS_UNIX) {
            tempFile = File.createTempFile("terraform-", "-release", new File(tempPath));
        } else {
            tempFile = File.createTempFile("terraform-", "-release", new File(tempPath));
            if (tempFile.setReadable(true, true)) {
                log.info("File permission Readable applied");
            }
            if (tempFile.setWritable(true, true)) {
                log.info("File permission Writable applied");
            }
            if (tempFile.setExecutable(true, true)) {
                log.info("File permission Executable applied");
            }
        }
        FileUtils.copyURLToFile((URL)new URL(terraformReleasesUrl), (File)tempFile);
        this.terraformReleases = (TerraformResponse)objectMapper.readValue(tempFile, TerraformResponse.class);
        log.info("Deleting Temp {}", (Object)tempFile.getAbsolutePath());
        log.info("Found {} terraform releases", (Object)this.terraformReleases.getVersions().size());
    }

    private void getTofuReleases(String tofuReleasesUrl) throws IOException {
        File tempFile;
        log.info("Downloading tofu releases list");
        ObjectMapper objectMapper = new ObjectMapper();
        objectMapper.configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false);
        String tempPath = this.userHomeDirectory.concat(FilenameUtils.separatorsToSystem((String)TEMP_DIRECTORY));
        if (SystemUtils.IS_OS_UNIX) {
            tempFile = File.createTempFile("tofu-", "-release", new File(tempPath));
        } else {
            tempFile = File.createTempFile("tofu-", "-release", new File(tempPath));
            if (tempFile.setReadable(true, true)) {
                log.info("File permission Readable applied");
            }
            if (tempFile.setWritable(true, true)) {
                log.info("File permission Writable applied");
            }
            if (tempFile.setExecutable(true, true)) {
                log.info("File permission Executable applied");
            }
        }
        FileUtils.copyURLToFile((URL)new URL(tofuReleasesUrl), (File)tempFile);
        this.tofuReleases = (List)objectMapper.readValue(tempFile, (JavaType)objectMapper.getTypeFactory().constructCollectionType(List.class, TofuRelease.class));
        log.info("Deleting Temp {}", (Object)tempFile.getAbsolutePath());
        log.info("Found {} tofu releases", (Object)this.tofuReleases.size());
    }

    private String downloadFileOrReturnPathIfAlreadyExists(String fileName, String zipReleaseUrl, String version, boolean tofu) throws IOException {
        File downloadDirectory;
        String downloadPath = tofu ? TOFU_DOWNLOAD_DIRECTORY : TERRAFORM_DOWNLOAD_DIRECTORY;
        String path = tofu ? TOFU_DIRECTORY : TERRAFORM_DIRECTORY;
        String product = tofu ? "tofu" : "terraform";
        File file = downloadDirectory = tofu ? this.tofuDownloadDirectory : this.terraformDownloadDirectory;
        if (!FileUtils.directoryContains((File)downloadDirectory, (File)new File(this.userHomeDirectory.concat(FilenameUtils.separatorsToSystem((String)downloadPath.concat("/").concat(fileName)))))) {
            log.info("Downloading {} from: {}", (Object)product, (Object)zipReleaseUrl);
            try {
                File zipFile = new File(this.userHomeDirectory.concat(FilenameUtils.separatorsToSystem((String)downloadPath.concat(fileName))));
                FileUtils.copyURLToFile((URL)new URL(zipReleaseUrl), (File)zipFile);
                if (tofu) {
                    return this.unzipTofuVersion(version, zipFile);
                }
                return this.unzipTerraformVersion(version, zipFile);
            }
            catch (IOException exception) {
                throw new IOException("Unable to download ".concat(zipReleaseUrl));
            }
        }
        log.info("{} {} already exists", (Object)fileName, (Object)product);
        return this.userHomeDirectory.concat(FilenameUtils.separatorsToSystem((String)path.concat(version.concat("/").concat(product))));
    }

    private boolean doSystemAndReleaseMatch(String arch, String os) {
        return arch.equals(SystemUtils.OS_ARCH) && (SystemUtils.IS_OS_WINDOWS && os.equals("windows") || SystemUtils.IS_OS_LINUX && os.equals("linux")) || SystemUtils.IS_OS_MAC && os.equals("darwin");
    }

    public String downloadTerraformVersion(String terraformVersion) throws IOException {
        log.info("Downloading terraform version {} architecture {} Type {}", new Object[]{terraformVersion, SystemUtils.OS_ARCH, SystemUtils.OS_NAME});
        TerraformVersion version = this.terraformReleases.getVersions().get(terraformVersion);
        boolean notFound = true;
        String terraformFilePath = "";
        if (version == null) {
            throw new IllegalArgumentException("Invalid Terraform Version");
        }
        for (TerraformBuild terraformBuild : version.getBuilds()) {
            if (!this.doSystemAndReleaseMatch(terraformBuild.getArch(), terraformBuild.getOs())) continue;
            String terraformZipReleaseURL = terraformBuild.getUrl();
            String fileName = terraformBuild.getFilename();
            terraformFilePath = this.downloadFileOrReturnPathIfAlreadyExists(fileName, terraformZipReleaseURL, terraformVersion, false);
            notFound = false;
            break;
        }
        if (notFound) {
            throw new IllegalArgumentException("Invalid Terraform Version");
        }
        return terraformFilePath;
    }

    public String downloadTofuVersion(String tofuVersion) throws IOException {
        log.info("Downloading tofu version {} architecture {} Type {}", new Object[]{tofuVersion, SystemUtils.OS_ARCH, SystemUtils.OS_NAME});
        String defaultFileName = "tofu_%s_%s_%s.zip";
        List releases = this.tofuReleases.stream().filter(release -> release.getName().equals("v" + tofuVersion)).collect(Collectors.toList());
        if (releases.size() != 1) {
            throw new IllegalArgumentException("Invalid Tofu Version");
        }
        List assets = ((TofuRelease)releases.get(0)).getAssets().stream().filter(asset -> asset.getName().endsWith(".zip")).collect(Collectors.toList());
        boolean notFound = true;
        String tofuFilePath = "";
        for (TofuAsset asset2 : assets) {
            String[] parts = asset2.getName().split("_");
            String os = parts[2];
            String arch = parts[3].replace(".zip", "");
            if (!this.doSystemAndReleaseMatch(arch, os)) continue;
            String zipReleaseURL = asset2.getBrowser_download_url();
            String fileName = String.format(defaultFileName, tofuVersion, this.getOs(), arch);
            tofuFilePath = this.downloadFileOrReturnPathIfAlreadyExists(fileName, zipReleaseURL, tofuVersion, true);
            notFound = false;
            break;
        }
        if (notFound) {
            throw new IllegalArgumentException("Invalid Tofu Version");
        }
        return tofuFilePath;
    }

    public String getOs() {
        if (SystemUtils.IS_OS_LINUX) {
            return "linux";
        }
        if (SystemUtils.IS_OS_MAC) {
            return "darwin";
        }
        if (SystemUtils.IS_OS_WINDOWS) {
            return "windows";
        }
        return "linux";
    }

    private String unzipTerraformVersion(String terraformVersion, File terraformZipFile) throws IOException {
        this.createVersionDirectory(terraformVersion, TERRAFORM_DIRECTORY);
        String newFilePath = null;
        try (ZipInputStream zis = new ZipInputStream(new FileInputStream(terraformZipFile));){
            ZipEntry zipEntry = zis.getNextEntry();
            byte[] buffer = new byte[1024];
            while (zipEntry != null) {
                newFilePath = this.userHomeDirectory.concat(FilenameUtils.separatorsToSystem((String)TERRAFORM_DIRECTORY.concat(terraformVersion.concat("/").concat(zipEntry.getName()))));
                log.info("Unzip: {}", (Object)newFilePath);
                File newFile = new File(newFilePath);
                if (zipEntry.isDirectory()) {
                    if (!newFile.isDirectory() && !newFile.mkdirs()) {
                        throw new IOException("Failed to create directory " + newFile);
                    }
                } else {
                    File parent = newFile.getParentFile();
                    if (!parent.isDirectory() && !parent.mkdirs()) {
                        throw new IOException("Failed to create directory " + parent);
                    }
                    try (FileOutputStream fos = new FileOutputStream(newFile);){
                        int len;
                        while ((len = zis.read(buffer)) > 0) {
                            fos.write(buffer, 0, len);
                        }
                    }
                    if (SystemUtils.IS_OS_LINUX || SystemUtils.IS_OS_MAC) {
                        File updateAccess = new File(newFilePath);
                        if (updateAccess.setExecutable(true, true)) {
                            log.info("Terraform setExecutable successful");
                        } else {
                            log.error("Terraform setExecutable successful");
                        }
                    }
                }
                zipEntry = zis.getNextEntry();
            }
            zis.closeEntry();
        }
        return newFilePath;
    }

    private String unzipTofuVersion(String tofuVersion, File tofuZipFile) throws IOException {
        this.createVersionDirectory(tofuVersion, TOFU_DIRECTORY);
        String newFilePath = null;
        try (ZipInputStream zis = new ZipInputStream(new FileInputStream(tofuZipFile));){
            ZipEntry zipEntry = zis.getNextEntry();
            byte[] buffer = new byte[1024];
            while (zipEntry != null) {
                newFilePath = this.userHomeDirectory.concat(FilenameUtils.separatorsToSystem((String)TOFU_DIRECTORY.concat(tofuVersion.concat("/").concat(zipEntry.getName()))));
                log.info("Unzip Tofu files: {}", (Object)newFilePath);
                File newTofuFile = new File(newFilePath);
                if (zipEntry.isDirectory()) {
                    if (!newTofuFile.isDirectory() && !newTofuFile.mkdirs()) {
                        throw new IOException("Failed to create directory for" + newTofuFile);
                    }
                } else {
                    File parent = newTofuFile.getParentFile();
                    if (!parent.isDirectory() && !parent.mkdirs()) {
                        throw new IOException("Failed to create directory for" + parent);
                    }
                    try (FileOutputStream file = new FileOutputStream(newTofuFile);){
                        int len;
                        while ((len = zis.read(buffer)) > 0) {
                            file.write(buffer, 0, len);
                        }
                    }
                    if (SystemUtils.IS_OS_LINUX || SystemUtils.IS_OS_MAC) {
                        File updateAccess = new File(newFilePath);
                        if (updateAccess.setExecutable(true, true)) {
                            log.info("Tofu setExecutable successful");
                        } else {
                            log.error("Tofu setExecutable successful");
                        }
                    }
                }
                zipEntry = zis.getNextEntry();
            }
            zis.closeEntry();
        }
        return this.userHomeDirectory.concat(FilenameUtils.separatorsToSystem((String)TOFU_DIRECTORY.concat(tofuVersion.concat("/").concat("tofu"))));
    }

    private void createVersionDirectory(String version, String directoryPath) throws IOException {
        File terraformVersionDirectory = new File(this.userHomeDirectory.concat(FilenameUtils.separatorsToSystem((String)directoryPath.concat(version))));
        FileUtils.forceMkdir((File)terraformVersionDirectory);
    }
}

