package org.unlaxer.jaddress;

import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.Optional;
import java.util.function.Predicate;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import io.vavr.Tuple2;

public class DynamicProjectContext {
	
	static Logger logger = LoggerFactory.getLogger(DynamicProjectContext.class);
	
	
	Path projectHome;
	
	public DynamicProjectContext(Path projectHome) {
		super();
		this.projectHome = projectHome;
	}
	
	public static DynamicProjectContext fromCurrentFolder() {
		return new DynamicProjectContext(resolve());
	}
	

	public Path getProjectHome(){
		
		return projectHome;
	}
	
	
	public Path getJavaFilePath(SubProjects subProjects , GradleStructure gradleStructure , Class<?> clazz){
		
		return getJavaFilePath(subProjects, gradleStructure, clazz.getPackage() , clazz.getSimpleName());
	}
	
	
	public  Path getJavaFilePath(SubProjects subProjects , GradleStructure gradleStructure , 
			Package targetPackage , String className){
		
		Path path = getPath(subProjects, gradleStructure, targetPackage);
		return path.resolve(className+".java");
	}
	
	public  Path getPath(SubProjects subProjects , GradleStructure gradleStructure , 
			Package targetPackage ){
		
		Path path = getProjectHome().resolve(subProjects.relativePath).resolve(gradleStructure.relativePath);

		for (String successor: targetPackage.toString().split(" ")[1].split("\\.")) {
			path = path.resolve(successor);
		}
		return path;
	}
	
	public  Path getPath(SubProjects subProjects , GradleStructure gradleStructure , 
			Package targetPackage ,String...successorPaths){
		
		Path path = getPath(subProjects , gradleStructure , targetPackage);

		return withSuccessor(path, successorPaths);
	}


	
	public  Path getPath(SubProjects subProjects , GradleStructure gradleStructure , String...successorPaths) {
		
		Path path = getProjectHome().resolve(subProjects.relativePath).resolve(gradleStructure.relativePath);
		
		for (String successor: successorPaths) {
			path = path.resolve(successor);
		}
		
		return path;
	}

	
	public  Path getPath(SubProjects subProjects , String...successorPaths){
		
		Path path = getProjectHome().resolve(subProjects.relativePath);

		return withSuccessor(path, successorPaths);
	}
	
	public Path withSuccessor(Path path , String...successorPaths) {
		
		for (String successor: successorPaths) {
			path = path.resolve(successor);
		}
		
		return path;
	}
	
	public Path switchProject(Path pathThatCreateWithThisContext , DynamicProjectContext switchToProjectContext) {
		if(false == pathThatCreateWithThisContext.startsWith(projectHome)) {

			throw new IllegalArgumentException();
		}
		
		
		int nameCount = projectHome.getNameCount();
		
		Path subpath = pathThatCreateWithThisContext.subpath(nameCount, pathThatCreateWithThisContext.getNameCount());
	
		return switchToProjectContext.getProjectHome().resolve(subpath);
	}
	
	public static Path getCurrentFolder(){
		
		return Paths.get(".").toAbsolutePath();
	}
	
	private static Path resolve(Tuple2<Predicate<Path>, Integer> predicateForContainsAndMatchCount) {
		
		Optional<Path> folder = getFolder(predicateForContainsAndMatchCount);
		
		return folder.map(Path::getParent).orElseThrow();
	}

	
	
	private static Path resolve() {
		
		return resolve(gradleOrMaven);
	}
	
	static final Predicate<Path> gradleOrMavenPredicate = path->{
		
		String targetPath = path.getFileName().toString();
		return 
				targetPath.equals("src") || 
				targetPath.equals("build.gradle") || 
				targetPath.equals("pom.xml");
	} ;
	
	public static Tuple2<Predicate<Path>, Integer>  gradleOrMaven = new Tuple2<>(gradleOrMavenPredicate, 2);
	
	
	public static Optional<Path> getFolder(Tuple2<Predicate<Path>, Integer> predicateForContainsAndMatchCount) {
		
		Predicate<Path> predicateForContains = predicateForContainsAndMatchCount._1;
		
		Integer matchCount = predicateForContainsAndMatchCount._2;
		
		Path currentFolder = getCurrentFolder();
		
		while (true) {
			long count;
			try {
				count = Files.list(currentFolder)
						.filter(predicateForContains)
						.count();
				if(count >= matchCount) {
					return Optional.of(currentFolder);
				}
				
				if(currentFolder.getParent() == null) {
					return Optional.empty();
				}
				
				currentFolder = currentFolder.getParent();
				
			} catch (IOException e) {
				
				logger.error( "io error" , e);
				
				return Optional.empty();
			}
		}
	}
	
}
