package org.unlaxer.jaddress.parser;

import java.util.Optional;

import org.unlaxer.jaddress.util.normalize.WordReplacer;

public class AddressTokenImpl implements AddressToken {
	
	final StringAndCharacterKinds stringAndCharacterKinds;
	final SeparatorKind separatorKindOfLeading;
	final SeparatorKind separatorKindOfTailing;
	private String suffix;
	private String prefix;

	
	public AddressTokenImpl(StringAndCharacterKinds stringAndCharacterKinds, SeparatorKind separatorKindOfLeading,
			SeparatorKind separatorKindOfTailing) {
		super();
		this.stringAndCharacterKinds = normalize(stringAndCharacterKinds);
		this.separatorKindOfLeading = separatorKindOfLeading;
		this.separatorKindOfTailing = separatorKindOfTailing;
	}
	
	public AddressTokenImpl(AddressToken addressToken) {
		this(
			normalize(addressToken.stringAndCharacterKinds()) ,
			addressToken.separatorKindOfLeading(),
			addressToken.separatorKindOfTailing()
		);
	}
	
	static StringAndCharacterKinds normalize(StringAndCharacterKinds stringAndCharacterKinds) {
		return stringAndCharacterKinds.isEmpty() ?
			stringAndCharacterKinds:
			StringAndCharacterKinds.of(
					WordReplacer.replace(stringAndCharacterKinds.joined()), false);
	}
	
	@Override
	public StringAndCharacterKinds stringAndCharacterKinds() {
		return stringAndCharacterKinds;
	}
	@Override
	public SeparatorKind separatorKindOfLeading() {
		return separatorKindOfLeading;
	}
	@Override
	public SeparatorKind separatorKindOfTailing() {
		return separatorKindOfTailing;
	}
	
	static final AddressToken EMPTY = new AddressTokenImpl(
			StringAndCharacterKinds.empty(), SeparatorKind.terminator , SeparatorKind.terminator);
	
	public static AddressToken empty() {
		return EMPTY;
	}
	
	@Override
	public AddressToken substring(
		StringIndex beginIndexInclusive, 
		StringIndex endIndexExclusive , 
		SeparatorKind leadingSeparator ,
		SeparatorKind tailingSeparator) {
	
		StringAndCharacterKinds substring = stringAndCharacterKinds.substring(
			beginIndexInclusive, 
			endIndexExclusive, 
			leadingSeparator, 
			tailingSeparator);
		
		return new AddressTokenImpl(substring, leadingSeparator, tailingSeparator);
	}
	
	@Override
	public AddressToken strip() {
		
		return new AddressTokenImpl(
			stringAndCharacterKinds.strip(), 
			separatorKindOfLeading, 
			separatorKindOfTailing
		);
	}
	
	@Override
	public String asString() {
		return stringAndCharacterKinds.joined();
	}
	
	@Override
	public int length() {
		return stringAndCharacterKinds.joined().length();
	}
	
	@Override
	public String toString() {
		return 
				prefix().map(x->"[" + x  +"]").orElse("")+
				stringAndCharacterKinds.joined() + 
				suffix().map(x->"[" + x  +"]").orElse("");
	}
	@Override
	public boolean isEmpty() {
		return stringAndCharacterKinds.isEmpty();
	}
	
	@Override
	public boolean isPresent() {
		return stringAndCharacterKinds.isPresent();
	}
	
	@Override
	public StringIndex indexOf(String word) {
		return StringIndex.of(stringAndCharacterKinds.joined.indexOf(word));
	}
	@Override
	public AddressToken subList(
		ListIndex beginIndexInclusive, 
		ListIndex endIndexExclusive,
		SeparatorKind leadingSeparator, 
		SeparatorKind tailingSeparator) {
	
		StringAndCharacterKinds subListAsStringAndCharacterKind = 
			stringAndCharacterKinds.subListAsStringAndCharacterKinds(beginIndexInclusive , endIndexExclusive);
		
		AddressToken addressToken = new AddressTokenImpl(subListAsStringAndCharacterKind , leadingSeparator ,tailingSeparator);
		
		return addressToken;
	}
	@Override
	public Optional<String> prefix() {
		return Optional.ofNullable(prefix);
	}
	@Override
	public Optional<String> suffix() {
		return Optional.ofNullable(suffix);
	}
	@Override
	public void setPrefix(String prefix) {
		this.prefix = prefix;
		
	}
	@Override
	public void setSuffix(String suffix) {
		this.suffix = suffix;
	}
}
