@file:Suppress("NAME_SHADOWING", "DEPRECATION")

package com.pulumi.aws.apigateway.kotlin

import com.pulumi.aws.apigateway.MethodArgs.builder
import com.pulumi.core.Output
import com.pulumi.core.Output.of
import com.pulumi.kotlin.ConvertibleToJava
import com.pulumi.kotlin.PulumiTagMarker
import kotlin.Boolean
import kotlin.Pair
import kotlin.String
import kotlin.Suppress
import kotlin.collections.List
import kotlin.collections.Map
import kotlin.jvm.JvmName

/**
 * Provides a HTTP Method for an API Gateway Resource.
 * ## Example Usage
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.apigateway.RestApi;
 * import com.pulumi.aws.apigateway.RestApiArgs;
 * import com.pulumi.aws.apigateway.Resource;
 * import com.pulumi.aws.apigateway.ResourceArgs;
 * import com.pulumi.aws.apigateway.Method;
 * import com.pulumi.aws.apigateway.MethodArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var myDemoAPI = new RestApi("myDemoAPI", RestApiArgs.builder()
 *             .description("This is my API for demonstration purposes")
 *             .build());
 *         var myDemoResource = new Resource("myDemoResource", ResourceArgs.builder()
 *             .restApi(myDemoAPI.id())
 *             .parentId(myDemoAPI.rootResourceId())
 *             .pathPart("mydemoresource")
 *             .build());
 *         var myDemoMethod = new Method("myDemoMethod", MethodArgs.builder()
 *             .restApi(myDemoAPI.id())
 *             .resourceId(myDemoResource.id())
 *             .httpMethod("GET")
 *             .authorization("NONE")
 *             .build());
 *     }
 * }
 * ```
 * ## Usage with Cognito User Pool Authorizer
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const config = new pulumi.Config();
 * const cognitoUserPoolName = config.requireObject("cognitoUserPoolName");
 * const thisUserPools = aws.cognito.getUserPools({
 *     name: cognitoUserPoolName,
 * });
 * const thisRestApi = new aws.apigateway.RestApi("thisRestApi", {});
 * const thisResource = new aws.apigateway.Resource("thisResource", {
 *     restApi: thisRestApi.id,
 *     parentId: thisRestApi.rootResourceId,
 *     pathPart: "{proxy+}",
 * });
 * const thisAuthorizer = new aws.apigateway.Authorizer("thisAuthorizer", {
 *     type: "COGNITO_USER_POOLS",
 *     restApi: thisRestApi.id,
 *     providerArns: thisUserPools.then(thisUserPools => thisUserPools.arns),
 * });
 * const any = new aws.apigateway.Method("any", {
 *     restApi: thisRestApi.id,
 *     resourceId: thisResource.id,
 *     httpMethod: "ANY",
 *     authorization: "COGNITO_USER_POOLS",
 *     authorizerId: thisAuthorizer.id,
 *     requestParameters: {
 *         "method.request.path.proxy": true,
 *     },
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * config = pulumi.Config()
 * cognito_user_pool_name = config.require_object("cognitoUserPoolName")
 * this_user_pools = aws.cognito.get_user_pools(name=cognito_user_pool_name)
 * this_rest_api = aws.apigateway.RestApi("thisRestApi")
 * this_resource = aws.apigateway.Resource("thisResource",
 *     rest_api=this_rest_api.id,
 *     parent_id=this_rest_api.root_resource_id,
 *     path_part="{proxy+}")
 * this_authorizer = aws.apigateway.Authorizer("thisAuthorizer",
 *     type="COGNITO_USER_POOLS",
 *     rest_api=this_rest_api.id,
 *     provider_arns=this_user_pools.arns)
 * any = aws.apigateway.Method("any",
 *     rest_api=this_rest_api.id,
 *     resource_id=this_resource.id,
 *     http_method="ANY",
 *     authorization="COGNITO_USER_POOLS",
 *     authorizer_id=this_authorizer.id,
 *     request_parameters={
 *         "method.request.path.proxy": True,
 *     })
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var config = new Config();
 *     var cognitoUserPoolName = config.RequireObject<dynamic>("cognitoUserPoolName");
 *     var thisUserPools = Aws.Cognito.GetUserPools.Invoke(new()
 *     {
 *         Name = cognitoUserPoolName,
 *     });
 *     var thisRestApi = new Aws.ApiGateway.RestApi("thisRestApi");
 *     var thisResource = new Aws.ApiGateway.Resource("thisResource", new()
 *     {
 *         RestApi = thisRestApi.Id,
 *         ParentId = thisRestApi.RootResourceId,
 *         PathPart = "{proxy+}",
 *     });
 *     var thisAuthorizer = new Aws.ApiGateway.Authorizer("thisAuthorizer", new()
 *     {
 *         Type = "COGNITO_USER_POOLS",
 *         RestApi = thisRestApi.Id,
 *         ProviderArns = thisUserPools.Apply(getUserPoolsResult => getUserPoolsResult.Arns),
 *     });
 *     var any = new Aws.ApiGateway.Method("any", new()
 *     {
 *         RestApi = thisRestApi.Id,
 *         ResourceId = thisResource.Id,
 *         HttpMethod = "ANY",
 *         Authorization = "COGNITO_USER_POOLS",
 *         AuthorizerId = thisAuthorizer.Id,
 *         RequestParameters =
 *         {
 *             { "method.request.path.proxy", true },
 *         },
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/apigateway"
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/cognito"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi/config"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		cfg := config.New(ctx, "")
 * 		cognitoUserPoolName := cfg.RequireObject("cognitoUserPoolName")
 * 		thisUserPools, err := cognito.GetUserPools(ctx, &cognito.GetUserPoolsArgs{
 * 			Name: cognitoUserPoolName,
 * 		}, nil)
 * 		if err != nil {
 * 			return err
 * 		}
 * 		thisRestApi, err := apigateway.NewRestApi(ctx, "thisRestApi", nil)
 * 		if err != nil {
 * 			return err
 * 		}
 * 		thisResource, err := apigateway.NewResource(ctx, "thisResource", &apigateway.ResourceArgs{
 * 			RestApi:  thisRestApi.ID(),
 * 			ParentId: thisRestApi.RootResourceId,
 * 			PathPart: pulumi.String("{proxy+}"),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		thisAuthorizer, err := apigateway.NewAuthorizer(ctx, "thisAuthorizer", &apigateway.AuthorizerArgs{
 * 			Type:         pulumi.String("COGNITO_USER_POOLS"),
 * 			RestApi:      thisRestApi.ID(),
 * 			ProviderArns: interface{}(thisUserPools.Arns),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		_, err = apigateway.NewMethod(ctx, "any", &apigateway.MethodArgs{
 * 			RestApi:       thisRestApi.ID(),
 * 			ResourceId:    thisResource.ID(),
 * 			HttpMethod:    pulumi.String("ANY"),
 * 			Authorization: pulumi.String("COGNITO_USER_POOLS"),
 * 			AuthorizerId:  thisAuthorizer.ID(),
 * 			RequestParameters: pulumi.BoolMap{
 * 				"method.request.path.proxy": pulumi.Bool(true),
 * 			},
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.cognito.CognitoFunctions;
 * import com.pulumi.aws.cognito.inputs.GetUserPoolsArgs;
 * import com.pulumi.aws.apigateway.RestApi;
 * import com.pulumi.aws.apigateway.Resource;
 * import com.pulumi.aws.apigateway.ResourceArgs;
 * import com.pulumi.aws.apigateway.Authorizer;
 * import com.pulumi.aws.apigateway.AuthorizerArgs;
 * import com.pulumi.aws.apigateway.Method;
 * import com.pulumi.aws.apigateway.MethodArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         final var config = ctx.config();
 *         final var cognitoUserPoolName = config.get("cognitoUserPoolName");
 *         final var thisUserPools = CognitoFunctions.getUserPools(GetUserPoolsArgs.builder()
 *             .name(cognitoUserPoolName)
 *             .build());
 *         var thisRestApi = new RestApi("thisRestApi");
 *         var thisResource = new Resource("thisResource", ResourceArgs.builder()
 *             .restApi(thisRestApi.id())
 *             .parentId(thisRestApi.rootResourceId())
 *             .pathPart("{proxy+}")
 *             .build());
 *         var thisAuthorizer = new Authorizer("thisAuthorizer", AuthorizerArgs.builder()
 *             .type("COGNITO_USER_POOLS")
 *             .restApi(thisRestApi.id())
 *             .providerArns(thisUserPools.applyValue(getUserPoolsResult -> getUserPoolsResult.arns()))
 *             .build());
 *         var any = new Method("any", MethodArgs.builder()
 *             .restApi(thisRestApi.id())
 *             .resourceId(thisResource.id())
 *             .httpMethod("ANY")
 *             .authorization("COGNITO_USER_POOLS")
 *             .authorizerId(thisAuthorizer.id())
 *             .requestParameters(Map.of("method.request.path.proxy", true))
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * configuration:
 *   cognitoUserPoolName:
 *     type: dynamic
 * resources:
 *   thisRestApi:
 *     type: aws:apigateway:RestApi
 *   thisResource:
 *     type: aws:apigateway:Resource
 *     properties:
 *       restApi: ${thisRestApi.id}
 *       parentId: ${thisRestApi.rootResourceId}
 *       pathPart: '{proxy+}'
 *   thisAuthorizer:
 *     type: aws:apigateway:Authorizer
 *     properties:
 *       type: COGNITO_USER_POOLS
 *       restApi: ${thisRestApi.id}
 *       providerArns: ${thisUserPools.arns}
 *   any:
 *     type: aws:apigateway:Method
 *     properties:
 *       restApi: ${thisRestApi.id}
 *       resourceId: ${thisResource.id}
 *       httpMethod: ANY
 *       authorization: COGNITO_USER_POOLS
 *       authorizerId: ${thisAuthorizer.id}
 *       requestParameters:
 *         method.request.path.proxy: true
 * variables:
 *   thisUserPools:
 *     fn::invoke:
 *       Function: aws:cognito:getUserPools
 *       Arguments:
 *         name: ${cognitoUserPoolName}
 * ```
 * ## Import
 * Using `pulumi import`, import `aws_api_gateway_method` using `REST-API-ID/RESOURCE-ID/HTTP-METHOD`. For example:
 * ```sh
 *  $ pulumi import aws:apigateway/method:Method example 12345abcde/67890fghij/GET
 * ```
 * @property apiKeyRequired Specify if the method requires an API key
 * @property authorization Type of authorization used for the method (`NONE`, `CUSTOM`, `AWS_IAM`, `COGNITO_USER_POOLS`)
 * @property authorizationScopes Authorization scopes used when the authorization is `COGNITO_USER_POOLS`
 * @property authorizerId Authorizer id to be used when the authorization is `CUSTOM` or `COGNITO_USER_POOLS`
 * @property httpMethod HTTP Method (`GET`, `POST`, `PUT`, `DELETE`, `HEAD`, `OPTIONS`, `ANY`)
 * @property operationName Function name that will be given to the method when generating an SDK through API Gateway. If omitted, API Gateway will generate a function name based on the resource path and HTTP verb.
 * @property requestModels Map of the API models used for the request's content type
 * where key is the content type (e.g., `application/json`)
 * and value is either `Error`, `Empty` (built-in models) or `aws.apigateway.Model`'s `name`.
 * @property requestParameters Map of request parameters (from the path, query string and headers) that should be passed to the integration. The boolean value indicates whether the parameter is required (`true`) or optional (`false`).
 * For example: `request_parameters = {"method.request.header.X-Some-Header" = true "method.request.querystring.some-query-param" = true}` would define that the header `X-Some-Header` and the query string `some-query-param` must be provided in the request.
 * @property requestValidatorId ID of a `aws.apigateway.RequestValidator`
 * @property resourceId API resource ID
 * @property restApi ID of the associated REST API
 */
public data class MethodArgs(
    public val apiKeyRequired: Output<Boolean>? = null,
    public val authorization: Output<String>? = null,
    public val authorizationScopes: Output<List<String>>? = null,
    public val authorizerId: Output<String>? = null,
    public val httpMethod: Output<String>? = null,
    public val operationName: Output<String>? = null,
    public val requestModels: Output<Map<String, String>>? = null,
    public val requestParameters: Output<Map<String, Boolean>>? = null,
    public val requestValidatorId: Output<String>? = null,
    public val resourceId: Output<String>? = null,
    public val restApi: Output<String>? = null,
) : ConvertibleToJava<com.pulumi.aws.apigateway.MethodArgs> {
    override fun toJava(): com.pulumi.aws.apigateway.MethodArgs =
        com.pulumi.aws.apigateway.MethodArgs.builder()
            .apiKeyRequired(apiKeyRequired?.applyValue({ args0 -> args0 }))
            .authorization(authorization?.applyValue({ args0 -> args0 }))
            .authorizationScopes(authorizationScopes?.applyValue({ args0 -> args0.map({ args0 -> args0 }) }))
            .authorizerId(authorizerId?.applyValue({ args0 -> args0 }))
            .httpMethod(httpMethod?.applyValue({ args0 -> args0 }))
            .operationName(operationName?.applyValue({ args0 -> args0 }))
            .requestModels(
                requestModels?.applyValue({ args0 ->
                    args0.map({ args0 ->
                        args0.key.to(args0.value)
                    }).toMap()
                }),
            )
            .requestParameters(
                requestParameters?.applyValue({ args0 ->
                    args0.map({ args0 ->
                        args0.key.to(args0.value)
                    }).toMap()
                }),
            )
            .requestValidatorId(requestValidatorId?.applyValue({ args0 -> args0 }))
            .resourceId(resourceId?.applyValue({ args0 -> args0 }))
            .restApi(restApi?.applyValue({ args0 -> args0 })).build()
}

/**
 * Builder for [MethodArgs].
 */
@PulumiTagMarker
public class MethodArgsBuilder internal constructor() {
    private var apiKeyRequired: Output<Boolean>? = null

    private var authorization: Output<String>? = null

    private var authorizationScopes: Output<List<String>>? = null

    private var authorizerId: Output<String>? = null

    private var httpMethod: Output<String>? = null

    private var operationName: Output<String>? = null

    private var requestModels: Output<Map<String, String>>? = null

    private var requestParameters: Output<Map<String, Boolean>>? = null

    private var requestValidatorId: Output<String>? = null

    private var resourceId: Output<String>? = null

    private var restApi: Output<String>? = null

    /**
     * @param value Specify if the method requires an API key
     */
    @JvmName("pnxbsrblklxdefcr")
    public suspend fun apiKeyRequired(`value`: Output<Boolean>) {
        this.apiKeyRequired = value
    }

    /**
     * @param value Type of authorization used for the method (`NONE`, `CUSTOM`, `AWS_IAM`, `COGNITO_USER_POOLS`)
     */
    @JvmName("pfyxoerwsaemgnfo")
    public suspend fun authorization(`value`: Output<String>) {
        this.authorization = value
    }

    /**
     * @param value Authorization scopes used when the authorization is `COGNITO_USER_POOLS`
     */
    @JvmName("tigljcaabajcxwkc")
    public suspend fun authorizationScopes(`value`: Output<List<String>>) {
        this.authorizationScopes = value
    }

    @JvmName("xjfutkrrrvorytdg")
    public suspend fun authorizationScopes(vararg values: Output<String>) {
        this.authorizationScopes = Output.all(values.asList())
    }

    /**
     * @param values Authorization scopes used when the authorization is `COGNITO_USER_POOLS`
     */
    @JvmName("ihbmbwqvvpioueja")
    public suspend fun authorizationScopes(values: List<Output<String>>) {
        this.authorizationScopes = Output.all(values)
    }

    /**
     * @param value Authorizer id to be used when the authorization is `CUSTOM` or `COGNITO_USER_POOLS`
     */
    @JvmName("vajpyctvyykcqrty")
    public suspend fun authorizerId(`value`: Output<String>) {
        this.authorizerId = value
    }

    /**
     * @param value HTTP Method (`GET`, `POST`, `PUT`, `DELETE`, `HEAD`, `OPTIONS`, `ANY`)
     */
    @JvmName("hkwqcnaakkwrusvh")
    public suspend fun httpMethod(`value`: Output<String>) {
        this.httpMethod = value
    }

    /**
     * @param value Function name that will be given to the method when generating an SDK through API Gateway. If omitted, API Gateway will generate a function name based on the resource path and HTTP verb.
     */
    @JvmName("sgauliagssjncycu")
    public suspend fun operationName(`value`: Output<String>) {
        this.operationName = value
    }

    /**
     * @param value Map of the API models used for the request's content type
     * where key is the content type (e.g., `application/json`)
     * and value is either `Error`, `Empty` (built-in models) or `aws.apigateway.Model`'s `name`.
     */
    @JvmName("ijcomrbsokuhyyvq")
    public suspend fun requestModels(`value`: Output<Map<String, String>>) {
        this.requestModels = value
    }

    /**
     * @param value Map of request parameters (from the path, query string and headers) that should be passed to the integration. The boolean value indicates whether the parameter is required (`true`) or optional (`false`).
     * For example: `request_parameters = {"method.request.header.X-Some-Header" = true "method.request.querystring.some-query-param" = true}` would define that the header `X-Some-Header` and the query string `some-query-param` must be provided in the request.
     */
    @JvmName("gtntolktqsttecvn")
    public suspend fun requestParameters(`value`: Output<Map<String, Boolean>>) {
        this.requestParameters = value
    }

    /**
     * @param value ID of a `aws.apigateway.RequestValidator`
     */
    @JvmName("tetbkyhujcrwfanl")
    public suspend fun requestValidatorId(`value`: Output<String>) {
        this.requestValidatorId = value
    }

    /**
     * @param value API resource ID
     */
    @JvmName("pnrbbccykdkbcmgi")
    public suspend fun resourceId(`value`: Output<String>) {
        this.resourceId = value
    }

    /**
     * @param value ID of the associated REST API
     */
    @JvmName("anonqsfoajctytlw")
    public suspend fun restApi(`value`: Output<String>) {
        this.restApi = value
    }

    /**
     * @param value Specify if the method requires an API key
     */
    @JvmName("xtuowgewxmdlyotl")
    public suspend fun apiKeyRequired(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.apiKeyRequired = mapped
    }

    /**
     * @param value Type of authorization used for the method (`NONE`, `CUSTOM`, `AWS_IAM`, `COGNITO_USER_POOLS`)
     */
    @JvmName("fjoepaxknhbgvstn")
    public suspend fun authorization(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.authorization = mapped
    }

    /**
     * @param value Authorization scopes used when the authorization is `COGNITO_USER_POOLS`
     */
    @JvmName("clhscatjwukvksdi")
    public suspend fun authorizationScopes(`value`: List<String>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.authorizationScopes = mapped
    }

    /**
     * @param values Authorization scopes used when the authorization is `COGNITO_USER_POOLS`
     */
    @JvmName("phsvxndjoltkbbwr")
    public suspend fun authorizationScopes(vararg values: String) {
        val toBeMapped = values.toList()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.authorizationScopes = mapped
    }

    /**
     * @param value Authorizer id to be used when the authorization is `CUSTOM` or `COGNITO_USER_POOLS`
     */
    @JvmName("jqnvommonjuvjgye")
    public suspend fun authorizerId(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.authorizerId = mapped
    }

    /**
     * @param value HTTP Method (`GET`, `POST`, `PUT`, `DELETE`, `HEAD`, `OPTIONS`, `ANY`)
     */
    @JvmName("ujufyrqoepujlmal")
    public suspend fun httpMethod(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.httpMethod = mapped
    }

    /**
     * @param value Function name that will be given to the method when generating an SDK through API Gateway. If omitted, API Gateway will generate a function name based on the resource path and HTTP verb.
     */
    @JvmName("fvdepvowswbvnghs")
    public suspend fun operationName(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.operationName = mapped
    }

    /**
     * @param value Map of the API models used for the request's content type
     * where key is the content type (e.g., `application/json`)
     * and value is either `Error`, `Empty` (built-in models) or `aws.apigateway.Model`'s `name`.
     */
    @JvmName("hitbbnwetixoukyb")
    public suspend fun requestModels(`value`: Map<String, String>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.requestModels = mapped
    }

    /**
     * @param values Map of the API models used for the request's content type
     * where key is the content type (e.g., `application/json`)
     * and value is either `Error`, `Empty` (built-in models) or `aws.apigateway.Model`'s `name`.
     */
    @JvmName("hlobastcgdwoacls")
    public fun requestModels(vararg values: Pair<String, String>) {
        val toBeMapped = values.toMap()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.requestModels = mapped
    }

    /**
     * @param value Map of request parameters (from the path, query string and headers) that should be passed to the integration. The boolean value indicates whether the parameter is required (`true`) or optional (`false`).
     * For example: `request_parameters = {"method.request.header.X-Some-Header" = true "method.request.querystring.some-query-param" = true}` would define that the header `X-Some-Header` and the query string `some-query-param` must be provided in the request.
     */
    @JvmName("cgifcebsmkfaaehs")
    public suspend fun requestParameters(`value`: Map<String, Boolean>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.requestParameters = mapped
    }

    /**
     * @param values Map of request parameters (from the path, query string and headers) that should be passed to the integration. The boolean value indicates whether the parameter is required (`true`) or optional (`false`).
     * For example: `request_parameters = {"method.request.header.X-Some-Header" = true "method.request.querystring.some-query-param" = true}` would define that the header `X-Some-Header` and the query string `some-query-param` must be provided in the request.
     */
    @JvmName("vdsocybjnvinrsll")
    public fun requestParameters(vararg values: Pair<String, Boolean>) {
        val toBeMapped = values.toMap()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.requestParameters = mapped
    }

    /**
     * @param value ID of a `aws.apigateway.RequestValidator`
     */
    @JvmName("ceiytqdmasetaveh")
    public suspend fun requestValidatorId(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.requestValidatorId = mapped
    }

    /**
     * @param value API resource ID
     */
    @JvmName("eclhawsxtowjtkum")
    public suspend fun resourceId(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.resourceId = mapped
    }

    /**
     * @param value ID of the associated REST API
     */
    @JvmName("rsaiarpftujtseii")
    public suspend fun restApi(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.restApi = mapped
    }

    internal fun build(): MethodArgs = MethodArgs(
        apiKeyRequired = apiKeyRequired,
        authorization = authorization,
        authorizationScopes = authorizationScopes,
        authorizerId = authorizerId,
        httpMethod = httpMethod,
        operationName = operationName,
        requestModels = requestModels,
        requestParameters = requestParameters,
        requestValidatorId = requestValidatorId,
        resourceId = resourceId,
        restApi = restApi,
    )
}
