@file:Suppress("NAME_SHADOWING", "DEPRECATION")

package com.pulumi.aws.autoscaling.kotlin

import com.pulumi.aws.autoscaling.PolicyArgs.builder
import com.pulumi.aws.autoscaling.kotlin.inputs.PolicyPredictiveScalingConfigurationArgs
import com.pulumi.aws.autoscaling.kotlin.inputs.PolicyPredictiveScalingConfigurationArgsBuilder
import com.pulumi.aws.autoscaling.kotlin.inputs.PolicyStepAdjustmentArgs
import com.pulumi.aws.autoscaling.kotlin.inputs.PolicyStepAdjustmentArgsBuilder
import com.pulumi.aws.autoscaling.kotlin.inputs.PolicyTargetTrackingConfigurationArgs
import com.pulumi.aws.autoscaling.kotlin.inputs.PolicyTargetTrackingConfigurationArgsBuilder
import com.pulumi.core.Output
import com.pulumi.core.Output.of
import com.pulumi.kotlin.ConvertibleToJava
import com.pulumi.kotlin.PulumiTagMarker
import com.pulumi.kotlin.applySuspend
import kotlin.Boolean
import kotlin.Int
import kotlin.String
import kotlin.Suppress
import kotlin.Unit
import kotlin.collections.List
import kotlin.jvm.JvmName

/**
 * Provides an AutoScaling Scaling Policy resource.
 * > **NOTE:** You may want to omit `desired_capacity` attribute from attached `aws.autoscaling.Group`
 * when using autoscaling policies. It's good practice to pick either
 * [manual](https://docs.aws.amazon.com/AutoScaling/latest/DeveloperGuide/as-manual-scaling.html)
 * or [dynamic](https://docs.aws.amazon.com/AutoScaling/latest/DeveloperGuide/as-scale-based-on-demand.html)
 * (policy-based) scaling.
 * ## Example Usage
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.autoscaling.Group;
 * import com.pulumi.aws.autoscaling.GroupArgs;
 * import com.pulumi.aws.autoscaling.Policy;
 * import com.pulumi.aws.autoscaling.PolicyArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var bar = new Group("bar", GroupArgs.builder()
 *             .availabilityZones("us-east-1a")
 *             .maxSize(5)
 *             .minSize(2)
 *             .healthCheckGracePeriod(300)
 *             .healthCheckType("ELB")
 *             .forceDelete(true)
 *             .launchConfiguration(aws_launch_configuration.foo().name())
 *             .build());
 *         var bat = new Policy("bat", PolicyArgs.builder()
 *             .scalingAdjustment(4)
 *             .adjustmentType("ChangeInCapacity")
 *             .cooldown(300)
 *             .autoscalingGroupName(bar.name())
 *             .build());
 *     }
 * }
 * ```
 * ### Create target tracking scaling policy using metric math
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.autoscaling.Policy;
 * import com.pulumi.aws.autoscaling.PolicyArgs;
 * import com.pulumi.aws.autoscaling.inputs.PolicyTargetTrackingConfigurationArgs;
 * import com.pulumi.aws.autoscaling.inputs.PolicyTargetTrackingConfigurationCustomizedMetricSpecificationArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var example = new Policy("example", PolicyArgs.builder()
 *             .autoscalingGroupName("my-test-asg")
 *             .policyType("TargetTrackingScaling")
 *             .targetTrackingConfiguration(PolicyTargetTrackingConfigurationArgs.builder()
 *                 .customizedMetricSpecification(PolicyTargetTrackingConfigurationCustomizedMetricSpecificationArgs.builder()
 *                     .metrics(
 *                         PolicyTargetTrackingConfigurationCustomizedMetricSpecificationMetricArgs.builder()
 *                             .id("m1")
 *                             .label("Get the queue size (the number of messages waiting to be processed)")
 *                             .metricStat(PolicyTargetTrackingConfigurationCustomizedMetricSpecificationMetricMetricStatArgs.builder()
 *                                 .metric(PolicyTargetTrackingConfigurationCustomizedMetricSpecificationMetricMetricStatMetricArgs.builder()
 *                                     .dimensions(PolicyTargetTrackingConfigurationCustomizedMetricSpecificationMetricMetricStatMetricDimensionArgs.builder()
 *                                         .name("QueueName")
 *                                         .value("my-queue")
 *                                         .build())
 *                                     .metricName("ApproximateNumberOfMessagesVisible")
 *                                     .namespace("AWS/SQS")
 *                                     .build())
 *                                 .stat("Sum")
 *                                 .build())
 *                             .returnData(false)
 *                             .build(),
 *                         PolicyTargetTrackingConfigurationCustomizedMetricSpecificationMetricArgs.builder()
 *                             .id("m2")
 *                             .label("Get the group size (the number of InService instances)")
 *                             .metricStat(PolicyTargetTrackingConfigurationCustomizedMetricSpecificationMetricMetricStatArgs.builder()
 *                                 .metric(PolicyTargetTrackingConfigurationCustomizedMetricSpecificationMetricMetricStatMetricArgs.builder()
 *                                     .dimensions(PolicyTargetTrackingConfigurationCustomizedMetricSpecificationMetricMetricStatMetricDimensionArgs.builder()
 *                                         .name("AutoScalingGroupName")
 *                                         .value("my-asg")
 *                                         .build())
 *                                     .metricName("GroupInServiceInstances")
 *                                     .namespace("AWS/AutoScaling")
 *                                     .build())
 *                                 .stat("Average")
 *                                 .build())
 *                             .returnData(false)
 *                             .build(),
 *                         PolicyTargetTrackingConfigurationCustomizedMetricSpecificationMetricArgs.builder()
 *                             .expression("m1 / m2")
 *                             .id("e1")
 *                             .label("Calculate the backlog per instance")
 *                             .returnData(true)
 *                             .build())
 *                     .build())
 *                 .targetValue(100)
 *                 .build())
 *             .build());
 *     }
 * }
 * ```
 * ### Create predictive scaling policy using customized metrics
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.autoscaling.Policy;
 * import com.pulumi.aws.autoscaling.PolicyArgs;
 * import com.pulumi.aws.autoscaling.inputs.PolicyPredictiveScalingConfigurationArgs;
 * import com.pulumi.aws.autoscaling.inputs.PolicyPredictiveScalingConfigurationMetricSpecificationArgs;
 * import com.pulumi.aws.autoscaling.inputs.PolicyPredictiveScalingConfigurationMetricSpecificationCustomizedCapacityMetricSpecificationArgs;
 * import com.pulumi.aws.autoscaling.inputs.PolicyPredictiveScalingConfigurationMetricSpecificationCustomizedLoadMetricSpecificationArgs;
 * import com.pulumi.aws.autoscaling.inputs.PolicyPredictiveScalingConfigurationMetricSpecificationCustomizedScalingMetricSpecificationArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var example = new Policy("example", PolicyArgs.builder()
 *             .autoscalingGroupName("my-test-asg")
 *             .policyType("PredictiveScaling")
 *             .predictiveScalingConfiguration(PolicyPredictiveScalingConfigurationArgs.builder()
 *                 .metricSpecification(PolicyPredictiveScalingConfigurationMetricSpecificationArgs.builder()
 *                     .customizedCapacityMetricSpecification(PolicyPredictiveScalingConfigurationMetricSpecificationCustomizedCapacityMetricSpecificationArgs.builder()
 *                         .metricDataQueries(PolicyPredictiveScalingConfigurationMetricSpecificationCustomizedCapacityMetricSpecificationMetricDataQueryArgs.builder()
 *                             .expression("SUM(SEARCH('{AWS/AutoScaling,AutoScalingGroupName} MetricName=\"GroupInServiceIntances\" my-test-asg', 'Average', 300))")
 *                             .id("capacity_sum")
 *                             .build())
 *                         .build())
 *                     .customizedLoadMetricSpecification(PolicyPredictiveScalingConfigurationMetricSpecificationCustomizedLoadMetricSpecificationArgs.builder()
 *                         .metricDataQueries(PolicyPredictiveScalingConfigurationMetricSpecificationCustomizedLoadMetricSpecificationMetricDataQueryArgs.builder()
 *                             .expression("SUM(SEARCH('{AWS/EC2,AutoScalingGroupName} MetricName=\"CPUUtilization\" my-test-asg', 'Sum', 3600))")
 *                             .id("load_sum")
 *                             .build())
 *                         .build())
 *                     .customizedScalingMetricSpecification(PolicyPredictiveScalingConfigurationMetricSpecificationCustomizedScalingMetricSpecificationArgs.builder()
 *                         .metricDataQueries(
 *                             PolicyPredictiveScalingConfigurationMetricSpecificationCustomizedScalingMetricSpecificationMetricDataQueryArgs.builder()
 *                                 .expression("SUM(SEARCH('{AWS/AutoScaling,AutoScalingGroupName} MetricName=\"GroupInServiceIntances\" my-test-asg', 'Average', 300))")
 *                                 .id("capacity_sum")
 *                                 .returnData(false)
 *                                 .build(),
 *                             PolicyPredictiveScalingConfigurationMetricSpecificationCustomizedScalingMetricSpecificationMetricDataQueryArgs.builder()
 *                                 .expression("SUM(SEARCH('{AWS/EC2,AutoScalingGroupName} MetricName=\"CPUUtilization\" my-test-asg', 'Sum', 300))")
 *                                 .id("load_sum")
 *                                 .returnData(false)
 *                                 .build(),
 *                             PolicyPredictiveScalingConfigurationMetricSpecificationCustomizedScalingMetricSpecificationMetricDataQueryArgs.builder()
 *                                 .expression("load_sum / (capacity_sum * PERIOD(capacity_sum) / 60)")
 *                                 .id("weighted_average")
 *                                 .build())
 *                         .build())
 *                     .targetValue(10)
 *                     .build())
 *                 .build())
 *             .build());
 *     }
 * }
 * ```
 * ### Create predictive scaling policy using customized scaling and predefined load metric
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.autoscaling.Policy;
 * import com.pulumi.aws.autoscaling.PolicyArgs;
 * import com.pulumi.aws.autoscaling.inputs.PolicyPredictiveScalingConfigurationArgs;
 * import com.pulumi.aws.autoscaling.inputs.PolicyPredictiveScalingConfigurationMetricSpecificationArgs;
 * import com.pulumi.aws.autoscaling.inputs.PolicyPredictiveScalingConfigurationMetricSpecificationCustomizedScalingMetricSpecificationArgs;
 * import com.pulumi.aws.autoscaling.inputs.PolicyPredictiveScalingConfigurationMetricSpecificationPredefinedLoadMetricSpecificationArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var example = new Policy("example", PolicyArgs.builder()
 *             .autoscalingGroupName("my-test-asg")
 *             .policyType("PredictiveScaling")
 *             .predictiveScalingConfiguration(PolicyPredictiveScalingConfigurationArgs.builder()
 *                 .metricSpecification(PolicyPredictiveScalingConfigurationMetricSpecificationArgs.builder()
 *                     .customizedScalingMetricSpecification(PolicyPredictiveScalingConfigurationMetricSpecificationCustomizedScalingMetricSpecificationArgs.builder()
 *                         .metricDataQueries(PolicyPredictiveScalingConfigurationMetricSpecificationCustomizedScalingMetricSpecificationMetricDataQueryArgs.builder()
 *                             .id("scaling")
 *                             .metricStat(PolicyPredictiveScalingConfigurationMetricSpecificationCustomizedScalingMetricSpecificationMetricDataQueryMetricStatArgs.builder()
 *                                 .metric(PolicyPredictiveScalingConfigurationMetricSpecificationCustomizedScalingMetricSpecificationMetricDataQueryMetricStatMetricArgs.builder()
 *                                     .dimensions(PolicyPredictiveScalingConfigurationMetricSpecificationCustomizedScalingMetricSpecificationMetricDataQueryMetricStatMetricDimensionArgs.builder()
 *                                         .name("AutoScalingGroupName")
 *                                         .value("my-test-asg")
 *                                         .build())
 *                                     .metricName("CPUUtilization")
 *                                     .namespace("AWS/EC2")
 *                                     .build())
 *                                 .stat("Average")
 *                                 .build())
 *                             .build())
 *                         .build())
 *                     .predefinedLoadMetricSpecification(PolicyPredictiveScalingConfigurationMetricSpecificationPredefinedLoadMetricSpecificationArgs.builder()
 *                         .predefinedMetricType("ASGTotalCPUUtilization")
 *                         .resourceLabel("app/my-alb/778d41231b141a0f/targetgroup/my-alb-target-group/943f017f100becff")
 *                         .build())
 *                     .targetValue(10)
 *                     .build())
 *                 .build())
 *             .build());
 *     }
 * }
 * ```
 * ## Import
 * Using `pulumi import`, import AutoScaling scaling policy using the role autoscaling_group_name and name separated by `/`. For example:
 * ```sh
 *  $ pulumi import aws:autoscaling/policy:Policy test-policy asg-name/policy-name
 * ```
 * @property adjustmentType Whether the adjustment is an absolute number or a percentage of the current capacity. Valid values are `ChangeInCapacity`, `ExactCapacity`, and `PercentChangeInCapacity`.
 * @property autoscalingGroupName Name of the autoscaling group.
 * @property cooldown Amount of time, in seconds, after a scaling activity completes and before the next scaling activity can start.
 * @property enabled Whether the scaling policy is enabled or disabled. Default: `true`.
 * The following argument is only available to "SimpleScaling" and "StepScaling" type policies:
 * @property estimatedInstanceWarmup Estimated time, in seconds, until a newly launched instance will contribute CloudWatch metrics. Without a value, AWS will default to the group's specified cooldown period.
 * @property metricAggregationType Aggregation type for the policy's metrics. Valid values are "Minimum", "Maximum", and "Average". Without a value, AWS will treat the aggregation type as "Average".
 * @property minAdjustmentMagnitude Minimum value to scale by when `adjustment_type` is set to `PercentChangeInCapacity`.
 * The following arguments are only available to "SimpleScaling" type policies:
 * @property name Name of the policy.
 * @property policyType Policy type, either "SimpleScaling", "StepScaling", "TargetTrackingScaling", or "PredictiveScaling". If this value isn't provided, AWS will default to "SimpleScaling."
 * @property predictiveScalingConfiguration Predictive scaling policy configuration to use with Amazon EC2 Auto Scaling.
 * @property scalingAdjustment Number of members by which to
 * scale, when the adjustment bounds are breached. A positive value scales
 * up. A negative value scales down.
 * @property stepAdjustments Set of adjustments that manage
 * group scaling. These have the following structure:
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const example = new aws.autoscaling.Policy("example", {stepAdjustments: [
 *     {
 *         metricIntervalLowerBound: "1",
 *         metricIntervalUpperBound: "2",
 *         scalingAdjustment: -1,
 *     },
 *     {
 *         metricIntervalLowerBound: "2",
 *         metricIntervalUpperBound: "3",
 *         scalingAdjustment: 1,
 *     },
 * ]});
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * example = aws.autoscaling.Policy("example", step_adjustments=[
 *     aws.autoscaling.PolicyStepAdjustmentArgs(
 *         metric_interval_lower_bound="1",
 *         metric_interval_upper_bound="2",
 *         scaling_adjustment=-1,
 *     ),
 *     aws.autoscaling.PolicyStepAdjustmentArgs(
 *         metric_interval_lower_bound="2",
 *         metric_interval_upper_bound="3",
 *         scaling_adjustment=1,
 *     ),
 * ])
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var example = new Aws.AutoScaling.Policy("example", new()
 *     {
 *         StepAdjustments = new[]
 *         {
 *             new Aws.AutoScaling.Inputs.PolicyStepAdjustmentArgs
 *             {
 *                 MetricIntervalLowerBound = "1",
 *                 MetricIntervalUpperBound = "2",
 *                 ScalingAdjustment = -1,
 *             },
 *             new Aws.AutoScaling.Inputs.PolicyStepAdjustmentArgs
 *             {
 *                 MetricIntervalLowerBound = "2",
 *                 MetricIntervalUpperBound = "3",
 *                 ScalingAdjustment = 1,
 *             },
 *         },
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/autoscaling"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		_, err := autoscaling.NewPolicy(ctx, "example", &autoscaling.PolicyArgs{
 * 			StepAdjustments: autoscaling.PolicyStepAdjustmentArray{
 * 				&autoscaling.PolicyStepAdjustmentArgs{
 * 					MetricIntervalLowerBound: pulumi.String("1"),
 * 					MetricIntervalUpperBound: pulumi.String("2"),
 * 					ScalingAdjustment:        -1,
 * 				},
 * 				&autoscaling.PolicyStepAdjustmentArgs{
 * 					MetricIntervalLowerBound: pulumi.String("2"),
 * 					MetricIntervalUpperBound: pulumi.String("3"),
 * 					ScalingAdjustment:        pulumi.Int(1),
 * 				},
 * 			},
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * The following fields are available in step adjustments:
 * @property targetTrackingConfiguration Target tracking policy. These have the following structure:
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const example = new aws.autoscaling.Policy("example", {targetTrackingConfiguration: {
 *     predefinedMetricSpecification: {
 *         predefinedMetricType: "ASGAverageCPUUtilization",
 *     },
 *     targetValue: 40,
 * }});
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * example = aws.autoscaling.Policy("example", target_tracking_configuration=aws.autoscaling.PolicyTargetTrackingConfigurationArgs(
 *     predefined_metric_specification=aws.autoscaling.PolicyTargetTrackingConfigurationPredefinedMetricSpecificationArgs(
 *         predefined_metric_type="ASGAverageCPUUtilization",
 *     ),
 *     target_value=40,
 * ))
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var example = new Aws.AutoScaling.Policy("example", new()
 *     {
 *         TargetTrackingConfiguration = new Aws.AutoScaling.Inputs.PolicyTargetTrackingConfigurationArgs
 *         {
 *             PredefinedMetricSpecification = new Aws.AutoScaling.Inputs.PolicyTargetTrackingConfigurationPredefinedMetricSpecificationArgs
 *             {
 *                 PredefinedMetricType = "ASGAverageCPUUtilization",
 *             },
 *             TargetValue = 40,
 *         },
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/autoscaling"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		_, err := autoscaling.NewPolicy(ctx, "example", &autoscaling.PolicyArgs{
 * 			TargetTrackingConfiguration: &autoscaling.PolicyTargetTrackingConfigurationArgs{
 * 				PredefinedMetricSpecification: &autoscaling.PolicyTargetTrackingConfigurationPredefinedMetricSpecificationArgs{
 * 					PredefinedMetricType: pulumi.String("ASGAverageCPUUtilization"),
 * 				},
 * 				TargetValue: pulumi.Float64(40),
 * 			},
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.autoscaling.Policy;
 * import com.pulumi.aws.autoscaling.PolicyArgs;
 * import com.pulumi.aws.autoscaling.inputs.PolicyTargetTrackingConfigurationArgs;
 * import com.pulumi.aws.autoscaling.inputs.PolicyTargetTrackingConfigurationPredefinedMetricSpecificationArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var example = new Policy("example", PolicyArgs.builder()
 *             .targetTrackingConfiguration(PolicyTargetTrackingConfigurationArgs.builder()
 *                 .predefinedMetricSpecification(PolicyTargetTrackingConfigurationPredefinedMetricSpecificationArgs.builder()
 *                     .predefinedMetricType("ASGAverageCPUUtilization")
 *                     .build())
 *                 .targetValue(40)
 *                 .build())
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   example:
 *     type: aws:autoscaling:Policy
 *     properties:
 *       targetTrackingConfiguration:
 *         predefinedMetricSpecification:
 *           predefinedMetricType: ASGAverageCPUUtilization
 *         targetValue: 40
 * ```
 * The following fields are available in target tracking configuration:
 */
public data class PolicyArgs(
    public val adjustmentType: Output<String>? = null,
    public val autoscalingGroupName: Output<String>? = null,
    public val cooldown: Output<Int>? = null,
    public val enabled: Output<Boolean>? = null,
    public val estimatedInstanceWarmup: Output<Int>? = null,
    public val metricAggregationType: Output<String>? = null,
    public val minAdjustmentMagnitude: Output<Int>? = null,
    public val name: Output<String>? = null,
    public val policyType: Output<String>? = null,
    public val predictiveScalingConfiguration: Output<PolicyPredictiveScalingConfigurationArgs>? =
        null,
    public val scalingAdjustment: Output<Int>? = null,
    public val stepAdjustments: Output<List<PolicyStepAdjustmentArgs>>? = null,
    public val targetTrackingConfiguration: Output<PolicyTargetTrackingConfigurationArgs>? = null,
) : ConvertibleToJava<com.pulumi.aws.autoscaling.PolicyArgs> {
    override fun toJava(): com.pulumi.aws.autoscaling.PolicyArgs =
        com.pulumi.aws.autoscaling.PolicyArgs.builder()
            .adjustmentType(adjustmentType?.applyValue({ args0 -> args0 }))
            .autoscalingGroupName(autoscalingGroupName?.applyValue({ args0 -> args0 }))
            .cooldown(cooldown?.applyValue({ args0 -> args0 }))
            .enabled(enabled?.applyValue({ args0 -> args0 }))
            .estimatedInstanceWarmup(estimatedInstanceWarmup?.applyValue({ args0 -> args0 }))
            .metricAggregationType(metricAggregationType?.applyValue({ args0 -> args0 }))
            .minAdjustmentMagnitude(minAdjustmentMagnitude?.applyValue({ args0 -> args0 }))
            .name(name?.applyValue({ args0 -> args0 }))
            .policyType(policyType?.applyValue({ args0 -> args0 }))
            .predictiveScalingConfiguration(
                predictiveScalingConfiguration?.applyValue({ args0 ->
                    args0.let({ args0 -> args0.toJava() })
                }),
            )
            .scalingAdjustment(scalingAdjustment?.applyValue({ args0 -> args0 }))
            .stepAdjustments(
                stepAdjustments?.applyValue({ args0 ->
                    args0.map({ args0 ->
                        args0.let({ args0 ->
                            args0.toJava()
                        })
                    })
                }),
            )
            .targetTrackingConfiguration(
                targetTrackingConfiguration?.applyValue({ args0 ->
                    args0.let({ args0 ->
                        args0.toJava()
                    })
                }),
            ).build()
}

/**
 * Builder for [PolicyArgs].
 */
@PulumiTagMarker
public class PolicyArgsBuilder internal constructor() {
    private var adjustmentType: Output<String>? = null

    private var autoscalingGroupName: Output<String>? = null

    private var cooldown: Output<Int>? = null

    private var enabled: Output<Boolean>? = null

    private var estimatedInstanceWarmup: Output<Int>? = null

    private var metricAggregationType: Output<String>? = null

    private var minAdjustmentMagnitude: Output<Int>? = null

    private var name: Output<String>? = null

    private var policyType: Output<String>? = null

    private var predictiveScalingConfiguration: Output<PolicyPredictiveScalingConfigurationArgs>? =
        null

    private var scalingAdjustment: Output<Int>? = null

    private var stepAdjustments: Output<List<PolicyStepAdjustmentArgs>>? = null

    private var targetTrackingConfiguration: Output<PolicyTargetTrackingConfigurationArgs>? = null

    /**
     * @param value Whether the adjustment is an absolute number or a percentage of the current capacity. Valid values are `ChangeInCapacity`, `ExactCapacity`, and `PercentChangeInCapacity`.
     */
    @JvmName("obpffwrixgnctpnd")
    public suspend fun adjustmentType(`value`: Output<String>) {
        this.adjustmentType = value
    }

    /**
     * @param value Name of the autoscaling group.
     */
    @JvmName("jmnxculnhplfrpst")
    public suspend fun autoscalingGroupName(`value`: Output<String>) {
        this.autoscalingGroupName = value
    }

    /**
     * @param value Amount of time, in seconds, after a scaling activity completes and before the next scaling activity can start.
     */
    @JvmName("ylibscaeyavllgco")
    public suspend fun cooldown(`value`: Output<Int>) {
        this.cooldown = value
    }

    /**
     * @param value Whether the scaling policy is enabled or disabled. Default: `true`.
     * The following argument is only available to "SimpleScaling" and "StepScaling" type policies:
     */
    @JvmName("mywtdsfnbrfkuwtx")
    public suspend fun enabled(`value`: Output<Boolean>) {
        this.enabled = value
    }

    /**
     * @param value Estimated time, in seconds, until a newly launched instance will contribute CloudWatch metrics. Without a value, AWS will default to the group's specified cooldown period.
     */
    @JvmName("frlwqhjtevdderhi")
    public suspend fun estimatedInstanceWarmup(`value`: Output<Int>) {
        this.estimatedInstanceWarmup = value
    }

    /**
     * @param value Aggregation type for the policy's metrics. Valid values are "Minimum", "Maximum", and "Average". Without a value, AWS will treat the aggregation type as "Average".
     */
    @JvmName("xujcbmfnylaitgpj")
    public suspend fun metricAggregationType(`value`: Output<String>) {
        this.metricAggregationType = value
    }

    /**
     * @param value Minimum value to scale by when `adjustment_type` is set to `PercentChangeInCapacity`.
     * The following arguments are only available to "SimpleScaling" type policies:
     */
    @JvmName("rltiysbwfqgalhap")
    public suspend fun minAdjustmentMagnitude(`value`: Output<Int>) {
        this.minAdjustmentMagnitude = value
    }

    /**
     * @param value Name of the policy.
     */
    @JvmName("taurjwjvggfghikf")
    public suspend fun name(`value`: Output<String>) {
        this.name = value
    }

    /**
     * @param value Policy type, either "SimpleScaling", "StepScaling", "TargetTrackingScaling", or "PredictiveScaling". If this value isn't provided, AWS will default to "SimpleScaling."
     */
    @JvmName("nmgnfgtogenewyri")
    public suspend fun policyType(`value`: Output<String>) {
        this.policyType = value
    }

    /**
     * @param value Predictive scaling policy configuration to use with Amazon EC2 Auto Scaling.
     */
    @JvmName("iodknhfsqshrlqeu")
    public suspend
    fun predictiveScalingConfiguration(`value`: Output<PolicyPredictiveScalingConfigurationArgs>) {
        this.predictiveScalingConfiguration = value
    }

    /**
     * @param value Number of members by which to
     * scale, when the adjustment bounds are breached. A positive value scales
     * up. A negative value scales down.
     */
    @JvmName("ghgvwwixguiibosm")
    public suspend fun scalingAdjustment(`value`: Output<Int>) {
        this.scalingAdjustment = value
    }

    /**
     * @param value Set of adjustments that manage
     * group scaling. These have the following structure:
     * ```typescript
     * import * as pulumi from "@pulumi/pulumi";
     * import * as aws from "@pulumi/aws";
     * const example = new aws.autoscaling.Policy("example", {stepAdjustments: [
     *     {
     *         metricIntervalLowerBound: "1",
     *         metricIntervalUpperBound: "2",
     *         scalingAdjustment: -1,
     *     },
     *     {
     *         metricIntervalLowerBound: "2",
     *         metricIntervalUpperBound: "3",
     *         scalingAdjustment: 1,
     *     },
     * ]});
     * ```
     * ```python
     * import pulumi
     * import pulumi_aws as aws
     * example = aws.autoscaling.Policy("example", step_adjustments=[
     *     aws.autoscaling.PolicyStepAdjustmentArgs(
     *         metric_interval_lower_bound="1",
     *         metric_interval_upper_bound="2",
     *         scaling_adjustment=-1,
     *     ),
     *     aws.autoscaling.PolicyStepAdjustmentArgs(
     *         metric_interval_lower_bound="2",
     *         metric_interval_upper_bound="3",
     *         scaling_adjustment=1,
     *     ),
     * ])
     * ```
     * ```csharp
     * using System.Collections.Generic;
     * using System.Linq;
     * using Pulumi;
     * using Aws = Pulumi.Aws;
     * return await Deployment.RunAsync(() =>
     * {
     *     var example = new Aws.AutoScaling.Policy("example", new()
     *     {
     *         StepAdjustments = new[]
     *         {
     *             new Aws.AutoScaling.Inputs.PolicyStepAdjustmentArgs
     *             {
     *                 MetricIntervalLowerBound = "1",
     *                 MetricIntervalUpperBound = "2",
     *                 ScalingAdjustment = -1,
     *             },
     *             new Aws.AutoScaling.Inputs.PolicyStepAdjustmentArgs
     *             {
     *                 MetricIntervalLowerBound = "2",
     *                 MetricIntervalUpperBound = "3",
     *                 ScalingAdjustment = 1,
     *             },
     *         },
     *     });
     * });
     * ```
     * ```go
     * package main
     * import (
     * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/autoscaling"
     * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
     * )
     * func main() {
     * 	pulumi.Run(func(ctx *pulumi.Context) error {
     * 		_, err := autoscaling.NewPolicy(ctx, "example", &autoscaling.PolicyArgs{
     * 			StepAdjustments: autoscaling.PolicyStepAdjustmentArray{
     * 				&autoscaling.PolicyStepAdjustmentArgs{
     * 					MetricIntervalLowerBound: pulumi.String("1"),
     * 					MetricIntervalUpperBound: pulumi.String("2"),
     * 					ScalingAdjustment:        -1,
     * 				},
     * 				&autoscaling.PolicyStepAdjustmentArgs{
     * 					MetricIntervalLowerBound: pulumi.String("2"),
     * 					MetricIntervalUpperBound: pulumi.String("3"),
     * 					ScalingAdjustment:        pulumi.Int(1),
     * 				},
     * 			},
     * 		})
     * 		if err != nil {
     * 			return err
     * 		}
     * 		return nil
     * 	})
     * }
     * ```
     * The following fields are available in step adjustments:
     */
    @JvmName("pmsbleigkwfqgnqs")
    public suspend fun stepAdjustments(`value`: Output<List<PolicyStepAdjustmentArgs>>) {
        this.stepAdjustments = value
    }

    @JvmName("lfwdgfgvaowimebr")
    public suspend fun stepAdjustments(vararg values: Output<PolicyStepAdjustmentArgs>) {
        this.stepAdjustments = Output.all(values.asList())
    }

    /**
     * @param values Set of adjustments that manage
     * group scaling. These have the following structure:
     * ```typescript
     * import * as pulumi from "@pulumi/pulumi";
     * import * as aws from "@pulumi/aws";
     * const example = new aws.autoscaling.Policy("example", {stepAdjustments: [
     *     {
     *         metricIntervalLowerBound: "1",
     *         metricIntervalUpperBound: "2",
     *         scalingAdjustment: -1,
     *     },
     *     {
     *         metricIntervalLowerBound: "2",
     *         metricIntervalUpperBound: "3",
     *         scalingAdjustment: 1,
     *     },
     * ]});
     * ```
     * ```python
     * import pulumi
     * import pulumi_aws as aws
     * example = aws.autoscaling.Policy("example", step_adjustments=[
     *     aws.autoscaling.PolicyStepAdjustmentArgs(
     *         metric_interval_lower_bound="1",
     *         metric_interval_upper_bound="2",
     *         scaling_adjustment=-1,
     *     ),
     *     aws.autoscaling.PolicyStepAdjustmentArgs(
     *         metric_interval_lower_bound="2",
     *         metric_interval_upper_bound="3",
     *         scaling_adjustment=1,
     *     ),
     * ])
     * ```
     * ```csharp
     * using System.Collections.Generic;
     * using System.Linq;
     * using Pulumi;
     * using Aws = Pulumi.Aws;
     * return await Deployment.RunAsync(() =>
     * {
     *     var example = new Aws.AutoScaling.Policy("example", new()
     *     {
     *         StepAdjustments = new[]
     *         {
     *             new Aws.AutoScaling.Inputs.PolicyStepAdjustmentArgs
     *             {
     *                 MetricIntervalLowerBound = "1",
     *                 MetricIntervalUpperBound = "2",
     *                 ScalingAdjustment = -1,
     *             },
     *             new Aws.AutoScaling.Inputs.PolicyStepAdjustmentArgs
     *             {
     *                 MetricIntervalLowerBound = "2",
     *                 MetricIntervalUpperBound = "3",
     *                 ScalingAdjustment = 1,
     *             },
     *         },
     *     });
     * });
     * ```
     * ```go
     * package main
     * import (
     * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/autoscaling"
     * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
     * )
     * func main() {
     * 	pulumi.Run(func(ctx *pulumi.Context) error {
     * 		_, err := autoscaling.NewPolicy(ctx, "example", &autoscaling.PolicyArgs{
     * 			StepAdjustments: autoscaling.PolicyStepAdjustmentArray{
     * 				&autoscaling.PolicyStepAdjustmentArgs{
     * 					MetricIntervalLowerBound: pulumi.String("1"),
     * 					MetricIntervalUpperBound: pulumi.String("2"),
     * 					ScalingAdjustment:        -1,
     * 				},
     * 				&autoscaling.PolicyStepAdjustmentArgs{
     * 					MetricIntervalLowerBound: pulumi.String("2"),
     * 					MetricIntervalUpperBound: pulumi.String("3"),
     * 					ScalingAdjustment:        pulumi.Int(1),
     * 				},
     * 			},
     * 		})
     * 		if err != nil {
     * 			return err
     * 		}
     * 		return nil
     * 	})
     * }
     * ```
     * The following fields are available in step adjustments:
     */
    @JvmName("oimdqlwrnwxflbxo")
    public suspend fun stepAdjustments(values: List<Output<PolicyStepAdjustmentArgs>>) {
        this.stepAdjustments = Output.all(values)
    }

    /**
     * @param value Target tracking policy. These have the following structure:
     * ```typescript
     * import * as pulumi from "@pulumi/pulumi";
     * import * as aws from "@pulumi/aws";
     * const example = new aws.autoscaling.Policy("example", {targetTrackingConfiguration: {
     *     predefinedMetricSpecification: {
     *         predefinedMetricType: "ASGAverageCPUUtilization",
     *     },
     *     targetValue: 40,
     * }});
     * ```
     * ```python
     * import pulumi
     * import pulumi_aws as aws
     * example = aws.autoscaling.Policy("example", target_tracking_configuration=aws.autoscaling.PolicyTargetTrackingConfigurationArgs(
     *     predefined_metric_specification=aws.autoscaling.PolicyTargetTrackingConfigurationPredefinedMetricSpecificationArgs(
     *         predefined_metric_type="ASGAverageCPUUtilization",
     *     ),
     *     target_value=40,
     * ))
     * ```
     * ```csharp
     * using System.Collections.Generic;
     * using System.Linq;
     * using Pulumi;
     * using Aws = Pulumi.Aws;
     * return await Deployment.RunAsync(() =>
     * {
     *     var example = new Aws.AutoScaling.Policy("example", new()
     *     {
     *         TargetTrackingConfiguration = new Aws.AutoScaling.Inputs.PolicyTargetTrackingConfigurationArgs
     *         {
     *             PredefinedMetricSpecification = new Aws.AutoScaling.Inputs.PolicyTargetTrackingConfigurationPredefinedMetricSpecificationArgs
     *             {
     *                 PredefinedMetricType = "ASGAverageCPUUtilization",
     *             },
     *             TargetValue = 40,
     *         },
     *     });
     * });
     * ```
     * ```go
     * package main
     * import (
     * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/autoscaling"
     * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
     * )
     * func main() {
     * 	pulumi.Run(func(ctx *pulumi.Context) error {
     * 		_, err := autoscaling.NewPolicy(ctx, "example", &autoscaling.PolicyArgs{
     * 			TargetTrackingConfiguration: &autoscaling.PolicyTargetTrackingConfigurationArgs{
     * 				PredefinedMetricSpecification: &autoscaling.PolicyTargetTrackingConfigurationPredefinedMetricSpecificationArgs{
     * 					PredefinedMetricType: pulumi.String("ASGAverageCPUUtilization"),
     * 				},
     * 				TargetValue: pulumi.Float64(40),
     * 			},
     * 		})
     * 		if err != nil {
     * 			return err
     * 		}
     * 		return nil
     * 	})
     * }
     * ```
     * ```java
     * package generated_program;
     * import com.pulumi.Context;
     * import com.pulumi.Pulumi;
     * import com.pulumi.core.Output;
     * import com.pulumi.aws.autoscaling.Policy;
     * import com.pulumi.aws.autoscaling.PolicyArgs;
     * import com.pulumi.aws.autoscaling.inputs.PolicyTargetTrackingConfigurationArgs;
     * import com.pulumi.aws.autoscaling.inputs.PolicyTargetTrackingConfigurationPredefinedMetricSpecificationArgs;
     * import java.util.List;
     * import java.util.ArrayList;
     * import java.util.Map;
     * import java.io.File;
     * import java.nio.file.Files;
     * import java.nio.file.Paths;
     * public class App {
     *     public static void main(String[] args) {
     *         Pulumi.run(App::stack);
     *     }
     *     public static void stack(Context ctx) {
     *         var example = new Policy("example", PolicyArgs.builder()
     *             .targetTrackingConfiguration(PolicyTargetTrackingConfigurationArgs.builder()
     *                 .predefinedMetricSpecification(PolicyTargetTrackingConfigurationPredefinedMetricSpecificationArgs.builder()
     *                     .predefinedMetricType("ASGAverageCPUUtilization")
     *                     .build())
     *                 .targetValue(40)
     *                 .build())
     *             .build());
     *     }
     * }
     * ```
     * ```yaml
     * resources:
     *   example:
     *     type: aws:autoscaling:Policy
     *     properties:
     *       targetTrackingConfiguration:
     *         predefinedMetricSpecification:
     *           predefinedMetricType: ASGAverageCPUUtilization
     *         targetValue: 40
     * ```
     * The following fields are available in target tracking configuration:
     */
    @JvmName("cwxtymqvogdjgjgu")
    public suspend
    fun targetTrackingConfiguration(`value`: Output<PolicyTargetTrackingConfigurationArgs>) {
        this.targetTrackingConfiguration = value
    }

    /**
     * @param value Whether the adjustment is an absolute number or a percentage of the current capacity. Valid values are `ChangeInCapacity`, `ExactCapacity`, and `PercentChangeInCapacity`.
     */
    @JvmName("syfatcuqelqjoixc")
    public suspend fun adjustmentType(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.adjustmentType = mapped
    }

    /**
     * @param value Name of the autoscaling group.
     */
    @JvmName("hkujcjubuksbafgg")
    public suspend fun autoscalingGroupName(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.autoscalingGroupName = mapped
    }

    /**
     * @param value Amount of time, in seconds, after a scaling activity completes and before the next scaling activity can start.
     */
    @JvmName("uryblknwcwkmsqvu")
    public suspend fun cooldown(`value`: Int?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.cooldown = mapped
    }

    /**
     * @param value Whether the scaling policy is enabled or disabled. Default: `true`.
     * The following argument is only available to "SimpleScaling" and "StepScaling" type policies:
     */
    @JvmName("ifxfuacnhmyfbkst")
    public suspend fun enabled(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.enabled = mapped
    }

    /**
     * @param value Estimated time, in seconds, until a newly launched instance will contribute CloudWatch metrics. Without a value, AWS will default to the group's specified cooldown period.
     */
    @JvmName("sfvxuuvjvfndyrqp")
    public suspend fun estimatedInstanceWarmup(`value`: Int?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.estimatedInstanceWarmup = mapped
    }

    /**
     * @param value Aggregation type for the policy's metrics. Valid values are "Minimum", "Maximum", and "Average". Without a value, AWS will treat the aggregation type as "Average".
     */
    @JvmName("niwsqygvrtwwfcxk")
    public suspend fun metricAggregationType(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.metricAggregationType = mapped
    }

    /**
     * @param value Minimum value to scale by when `adjustment_type` is set to `PercentChangeInCapacity`.
     * The following arguments are only available to "SimpleScaling" type policies:
     */
    @JvmName("iqglpirfjxbmkoiu")
    public suspend fun minAdjustmentMagnitude(`value`: Int?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.minAdjustmentMagnitude = mapped
    }

    /**
     * @param value Name of the policy.
     */
    @JvmName("ysbvlppeoncqicdt")
    public suspend fun name(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.name = mapped
    }

    /**
     * @param value Policy type, either "SimpleScaling", "StepScaling", "TargetTrackingScaling", or "PredictiveScaling". If this value isn't provided, AWS will default to "SimpleScaling."
     */
    @JvmName("doxrdudrqwougawq")
    public suspend fun policyType(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.policyType = mapped
    }

    /**
     * @param value Predictive scaling policy configuration to use with Amazon EC2 Auto Scaling.
     */
    @JvmName("outgsmigsxsjyyil")
    public suspend
    fun predictiveScalingConfiguration(`value`: PolicyPredictiveScalingConfigurationArgs?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.predictiveScalingConfiguration = mapped
    }

    /**
     * @param argument Predictive scaling policy configuration to use with Amazon EC2 Auto Scaling.
     */
    @JvmName("vqfloieigrytkcwi")
    public suspend
    fun predictiveScalingConfiguration(argument: suspend PolicyPredictiveScalingConfigurationArgsBuilder.() -> Unit) {
        val toBeMapped = PolicyPredictiveScalingConfigurationArgsBuilder().applySuspend {
            argument()
        }.build()
        val mapped = of(toBeMapped)
        this.predictiveScalingConfiguration = mapped
    }

    /**
     * @param value Number of members by which to
     * scale, when the adjustment bounds are breached. A positive value scales
     * up. A negative value scales down.
     */
    @JvmName("bwurnpubayudwocy")
    public suspend fun scalingAdjustment(`value`: Int?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.scalingAdjustment = mapped
    }

    /**
     * @param value Set of adjustments that manage
     * group scaling. These have the following structure:
     * ```typescript
     * import * as pulumi from "@pulumi/pulumi";
     * import * as aws from "@pulumi/aws";
     * const example = new aws.autoscaling.Policy("example", {stepAdjustments: [
     *     {
     *         metricIntervalLowerBound: "1",
     *         metricIntervalUpperBound: "2",
     *         scalingAdjustment: -1,
     *     },
     *     {
     *         metricIntervalLowerBound: "2",
     *         metricIntervalUpperBound: "3",
     *         scalingAdjustment: 1,
     *     },
     * ]});
     * ```
     * ```python
     * import pulumi
     * import pulumi_aws as aws
     * example = aws.autoscaling.Policy("example", step_adjustments=[
     *     aws.autoscaling.PolicyStepAdjustmentArgs(
     *         metric_interval_lower_bound="1",
     *         metric_interval_upper_bound="2",
     *         scaling_adjustment=-1,
     *     ),
     *     aws.autoscaling.PolicyStepAdjustmentArgs(
     *         metric_interval_lower_bound="2",
     *         metric_interval_upper_bound="3",
     *         scaling_adjustment=1,
     *     ),
     * ])
     * ```
     * ```csharp
     * using System.Collections.Generic;
     * using System.Linq;
     * using Pulumi;
     * using Aws = Pulumi.Aws;
     * return await Deployment.RunAsync(() =>
     * {
     *     var example = new Aws.AutoScaling.Policy("example", new()
     *     {
     *         StepAdjustments = new[]
     *         {
     *             new Aws.AutoScaling.Inputs.PolicyStepAdjustmentArgs
     *             {
     *                 MetricIntervalLowerBound = "1",
     *                 MetricIntervalUpperBound = "2",
     *                 ScalingAdjustment = -1,
     *             },
     *             new Aws.AutoScaling.Inputs.PolicyStepAdjustmentArgs
     *             {
     *                 MetricIntervalLowerBound = "2",
     *                 MetricIntervalUpperBound = "3",
     *                 ScalingAdjustment = 1,
     *             },
     *         },
     *     });
     * });
     * ```
     * ```go
     * package main
     * import (
     * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/autoscaling"
     * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
     * )
     * func main() {
     * 	pulumi.Run(func(ctx *pulumi.Context) error {
     * 		_, err := autoscaling.NewPolicy(ctx, "example", &autoscaling.PolicyArgs{
     * 			StepAdjustments: autoscaling.PolicyStepAdjustmentArray{
     * 				&autoscaling.PolicyStepAdjustmentArgs{
     * 					MetricIntervalLowerBound: pulumi.String("1"),
     * 					MetricIntervalUpperBound: pulumi.String("2"),
     * 					ScalingAdjustment:        -1,
     * 				},
     * 				&autoscaling.PolicyStepAdjustmentArgs{
     * 					MetricIntervalLowerBound: pulumi.String("2"),
     * 					MetricIntervalUpperBound: pulumi.String("3"),
     * 					ScalingAdjustment:        pulumi.Int(1),
     * 				},
     * 			},
     * 		})
     * 		if err != nil {
     * 			return err
     * 		}
     * 		return nil
     * 	})
     * }
     * ```
     * The following fields are available in step adjustments:
     */
    @JvmName("egjxyfetivutceqr")
    public suspend fun stepAdjustments(`value`: List<PolicyStepAdjustmentArgs>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.stepAdjustments = mapped
    }

    /**
     * @param argument Set of adjustments that manage
     * group scaling. These have the following structure:
     * ```typescript
     * import * as pulumi from "@pulumi/pulumi";
     * import * as aws from "@pulumi/aws";
     * const example = new aws.autoscaling.Policy("example", {stepAdjustments: [
     *     {
     *         metricIntervalLowerBound: "1",
     *         metricIntervalUpperBound: "2",
     *         scalingAdjustment: -1,
     *     },
     *     {
     *         metricIntervalLowerBound: "2",
     *         metricIntervalUpperBound: "3",
     *         scalingAdjustment: 1,
     *     },
     * ]});
     * ```
     * ```python
     * import pulumi
     * import pulumi_aws as aws
     * example = aws.autoscaling.Policy("example", step_adjustments=[
     *     aws.autoscaling.PolicyStepAdjustmentArgs(
     *         metric_interval_lower_bound="1",
     *         metric_interval_upper_bound="2",
     *         scaling_adjustment=-1,
     *     ),
     *     aws.autoscaling.PolicyStepAdjustmentArgs(
     *         metric_interval_lower_bound="2",
     *         metric_interval_upper_bound="3",
     *         scaling_adjustment=1,
     *     ),
     * ])
     * ```
     * ```csharp
     * using System.Collections.Generic;
     * using System.Linq;
     * using Pulumi;
     * using Aws = Pulumi.Aws;
     * return await Deployment.RunAsync(() =>
     * {
     *     var example = new Aws.AutoScaling.Policy("example", new()
     *     {
     *         StepAdjustments = new[]
     *         {
     *             new Aws.AutoScaling.Inputs.PolicyStepAdjustmentArgs
     *             {
     *                 MetricIntervalLowerBound = "1",
     *                 MetricIntervalUpperBound = "2",
     *                 ScalingAdjustment = -1,
     *             },
     *             new Aws.AutoScaling.Inputs.PolicyStepAdjustmentArgs
     *             {
     *                 MetricIntervalLowerBound = "2",
     *                 MetricIntervalUpperBound = "3",
     *                 ScalingAdjustment = 1,
     *             },
     *         },
     *     });
     * });
     * ```
     * ```go
     * package main
     * import (
     * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/autoscaling"
     * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
     * )
     * func main() {
     * 	pulumi.Run(func(ctx *pulumi.Context) error {
     * 		_, err := autoscaling.NewPolicy(ctx, "example", &autoscaling.PolicyArgs{
     * 			StepAdjustments: autoscaling.PolicyStepAdjustmentArray{
     * 				&autoscaling.PolicyStepAdjustmentArgs{
     * 					MetricIntervalLowerBound: pulumi.String("1"),
     * 					MetricIntervalUpperBound: pulumi.String("2"),
     * 					ScalingAdjustment:        -1,
     * 				},
     * 				&autoscaling.PolicyStepAdjustmentArgs{
     * 					MetricIntervalLowerBound: pulumi.String("2"),
     * 					MetricIntervalUpperBound: pulumi.String("3"),
     * 					ScalingAdjustment:        pulumi.Int(1),
     * 				},
     * 			},
     * 		})
     * 		if err != nil {
     * 			return err
     * 		}
     * 		return nil
     * 	})
     * }
     * ```
     * The following fields are available in step adjustments:
     */
    @JvmName("ayomieqegaqmqmbo")
    public suspend
    fun stepAdjustments(argument: List<suspend PolicyStepAdjustmentArgsBuilder.() -> Unit>) {
        val toBeMapped = argument.toList().map {
            PolicyStepAdjustmentArgsBuilder().applySuspend {
                it()
            }.build()
        }
        val mapped = of(toBeMapped)
        this.stepAdjustments = mapped
    }

    /**
     * @param argument Set of adjustments that manage
     * group scaling. These have the following structure:
     * ```typescript
     * import * as pulumi from "@pulumi/pulumi";
     * import * as aws from "@pulumi/aws";
     * const example = new aws.autoscaling.Policy("example", {stepAdjustments: [
     *     {
     *         metricIntervalLowerBound: "1",
     *         metricIntervalUpperBound: "2",
     *         scalingAdjustment: -1,
     *     },
     *     {
     *         metricIntervalLowerBound: "2",
     *         metricIntervalUpperBound: "3",
     *         scalingAdjustment: 1,
     *     },
     * ]});
     * ```
     * ```python
     * import pulumi
     * import pulumi_aws as aws
     * example = aws.autoscaling.Policy("example", step_adjustments=[
     *     aws.autoscaling.PolicyStepAdjustmentArgs(
     *         metric_interval_lower_bound="1",
     *         metric_interval_upper_bound="2",
     *         scaling_adjustment=-1,
     *     ),
     *     aws.autoscaling.PolicyStepAdjustmentArgs(
     *         metric_interval_lower_bound="2",
     *         metric_interval_upper_bound="3",
     *         scaling_adjustment=1,
     *     ),
     * ])
     * ```
     * ```csharp
     * using System.Collections.Generic;
     * using System.Linq;
     * using Pulumi;
     * using Aws = Pulumi.Aws;
     * return await Deployment.RunAsync(() =>
     * {
     *     var example = new Aws.AutoScaling.Policy("example", new()
     *     {
     *         StepAdjustments = new[]
     *         {
     *             new Aws.AutoScaling.Inputs.PolicyStepAdjustmentArgs
     *             {
     *                 MetricIntervalLowerBound = "1",
     *                 MetricIntervalUpperBound = "2",
     *                 ScalingAdjustment = -1,
     *             },
     *             new Aws.AutoScaling.Inputs.PolicyStepAdjustmentArgs
     *             {
     *                 MetricIntervalLowerBound = "2",
     *                 MetricIntervalUpperBound = "3",
     *                 ScalingAdjustment = 1,
     *             },
     *         },
     *     });
     * });
     * ```
     * ```go
     * package main
     * import (
     * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/autoscaling"
     * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
     * )
     * func main() {
     * 	pulumi.Run(func(ctx *pulumi.Context) error {
     * 		_, err := autoscaling.NewPolicy(ctx, "example", &autoscaling.PolicyArgs{
     * 			StepAdjustments: autoscaling.PolicyStepAdjustmentArray{
     * 				&autoscaling.PolicyStepAdjustmentArgs{
     * 					MetricIntervalLowerBound: pulumi.String("1"),
     * 					MetricIntervalUpperBound: pulumi.String("2"),
     * 					ScalingAdjustment:        -1,
     * 				},
     * 				&autoscaling.PolicyStepAdjustmentArgs{
     * 					MetricIntervalLowerBound: pulumi.String("2"),
     * 					MetricIntervalUpperBound: pulumi.String("3"),
     * 					ScalingAdjustment:        pulumi.Int(1),
     * 				},
     * 			},
     * 		})
     * 		if err != nil {
     * 			return err
     * 		}
     * 		return nil
     * 	})
     * }
     * ```
     * The following fields are available in step adjustments:
     */
    @JvmName("rjjhlvthrftwvsrx")
    public suspend fun stepAdjustments(
        vararg
        argument: suspend PolicyStepAdjustmentArgsBuilder.() -> Unit,
    ) {
        val toBeMapped = argument.toList().map {
            PolicyStepAdjustmentArgsBuilder().applySuspend {
                it()
            }.build()
        }
        val mapped = of(toBeMapped)
        this.stepAdjustments = mapped
    }

    /**
     * @param argument Set of adjustments that manage
     * group scaling. These have the following structure:
     * ```typescript
     * import * as pulumi from "@pulumi/pulumi";
     * import * as aws from "@pulumi/aws";
     * const example = new aws.autoscaling.Policy("example", {stepAdjustments: [
     *     {
     *         metricIntervalLowerBound: "1",
     *         metricIntervalUpperBound: "2",
     *         scalingAdjustment: -1,
     *     },
     *     {
     *         metricIntervalLowerBound: "2",
     *         metricIntervalUpperBound: "3",
     *         scalingAdjustment: 1,
     *     },
     * ]});
     * ```
     * ```python
     * import pulumi
     * import pulumi_aws as aws
     * example = aws.autoscaling.Policy("example", step_adjustments=[
     *     aws.autoscaling.PolicyStepAdjustmentArgs(
     *         metric_interval_lower_bound="1",
     *         metric_interval_upper_bound="2",
     *         scaling_adjustment=-1,
     *     ),
     *     aws.autoscaling.PolicyStepAdjustmentArgs(
     *         metric_interval_lower_bound="2",
     *         metric_interval_upper_bound="3",
     *         scaling_adjustment=1,
     *     ),
     * ])
     * ```
     * ```csharp
     * using System.Collections.Generic;
     * using System.Linq;
     * using Pulumi;
     * using Aws = Pulumi.Aws;
     * return await Deployment.RunAsync(() =>
     * {
     *     var example = new Aws.AutoScaling.Policy("example", new()
     *     {
     *         StepAdjustments = new[]
     *         {
     *             new Aws.AutoScaling.Inputs.PolicyStepAdjustmentArgs
     *             {
     *                 MetricIntervalLowerBound = "1",
     *                 MetricIntervalUpperBound = "2",
     *                 ScalingAdjustment = -1,
     *             },
     *             new Aws.AutoScaling.Inputs.PolicyStepAdjustmentArgs
     *             {
     *                 MetricIntervalLowerBound = "2",
     *                 MetricIntervalUpperBound = "3",
     *                 ScalingAdjustment = 1,
     *             },
     *         },
     *     });
     * });
     * ```
     * ```go
     * package main
     * import (
     * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/autoscaling"
     * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
     * )
     * func main() {
     * 	pulumi.Run(func(ctx *pulumi.Context) error {
     * 		_, err := autoscaling.NewPolicy(ctx, "example", &autoscaling.PolicyArgs{
     * 			StepAdjustments: autoscaling.PolicyStepAdjustmentArray{
     * 				&autoscaling.PolicyStepAdjustmentArgs{
     * 					MetricIntervalLowerBound: pulumi.String("1"),
     * 					MetricIntervalUpperBound: pulumi.String("2"),
     * 					ScalingAdjustment:        -1,
     * 				},
     * 				&autoscaling.PolicyStepAdjustmentArgs{
     * 					MetricIntervalLowerBound: pulumi.String("2"),
     * 					MetricIntervalUpperBound: pulumi.String("3"),
     * 					ScalingAdjustment:        pulumi.Int(1),
     * 				},
     * 			},
     * 		})
     * 		if err != nil {
     * 			return err
     * 		}
     * 		return nil
     * 	})
     * }
     * ```
     * The following fields are available in step adjustments:
     */
    @JvmName("xmexxmqubklqxxcx")
    public suspend fun stepAdjustments(argument: suspend PolicyStepAdjustmentArgsBuilder.() -> Unit) {
        val toBeMapped = listOf(PolicyStepAdjustmentArgsBuilder().applySuspend { argument() }.build())
        val mapped = of(toBeMapped)
        this.stepAdjustments = mapped
    }

    /**
     * @param values Set of adjustments that manage
     * group scaling. These have the following structure:
     * ```typescript
     * import * as pulumi from "@pulumi/pulumi";
     * import * as aws from "@pulumi/aws";
     * const example = new aws.autoscaling.Policy("example", {stepAdjustments: [
     *     {
     *         metricIntervalLowerBound: "1",
     *         metricIntervalUpperBound: "2",
     *         scalingAdjustment: -1,
     *     },
     *     {
     *         metricIntervalLowerBound: "2",
     *         metricIntervalUpperBound: "3",
     *         scalingAdjustment: 1,
     *     },
     * ]});
     * ```
     * ```python
     * import pulumi
     * import pulumi_aws as aws
     * example = aws.autoscaling.Policy("example", step_adjustments=[
     *     aws.autoscaling.PolicyStepAdjustmentArgs(
     *         metric_interval_lower_bound="1",
     *         metric_interval_upper_bound="2",
     *         scaling_adjustment=-1,
     *     ),
     *     aws.autoscaling.PolicyStepAdjustmentArgs(
     *         metric_interval_lower_bound="2",
     *         metric_interval_upper_bound="3",
     *         scaling_adjustment=1,
     *     ),
     * ])
     * ```
     * ```csharp
     * using System.Collections.Generic;
     * using System.Linq;
     * using Pulumi;
     * using Aws = Pulumi.Aws;
     * return await Deployment.RunAsync(() =>
     * {
     *     var example = new Aws.AutoScaling.Policy("example", new()
     *     {
     *         StepAdjustments = new[]
     *         {
     *             new Aws.AutoScaling.Inputs.PolicyStepAdjustmentArgs
     *             {
     *                 MetricIntervalLowerBound = "1",
     *                 MetricIntervalUpperBound = "2",
     *                 ScalingAdjustment = -1,
     *             },
     *             new Aws.AutoScaling.Inputs.PolicyStepAdjustmentArgs
     *             {
     *                 MetricIntervalLowerBound = "2",
     *                 MetricIntervalUpperBound = "3",
     *                 ScalingAdjustment = 1,
     *             },
     *         },
     *     });
     * });
     * ```
     * ```go
     * package main
     * import (
     * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/autoscaling"
     * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
     * )
     * func main() {
     * 	pulumi.Run(func(ctx *pulumi.Context) error {
     * 		_, err := autoscaling.NewPolicy(ctx, "example", &autoscaling.PolicyArgs{
     * 			StepAdjustments: autoscaling.PolicyStepAdjustmentArray{
     * 				&autoscaling.PolicyStepAdjustmentArgs{
     * 					MetricIntervalLowerBound: pulumi.String("1"),
     * 					MetricIntervalUpperBound: pulumi.String("2"),
     * 					ScalingAdjustment:        -1,
     * 				},
     * 				&autoscaling.PolicyStepAdjustmentArgs{
     * 					MetricIntervalLowerBound: pulumi.String("2"),
     * 					MetricIntervalUpperBound: pulumi.String("3"),
     * 					ScalingAdjustment:        pulumi.Int(1),
     * 				},
     * 			},
     * 		})
     * 		if err != nil {
     * 			return err
     * 		}
     * 		return nil
     * 	})
     * }
     * ```
     * The following fields are available in step adjustments:
     */
    @JvmName("qjpweuebpebcrxdp")
    public suspend fun stepAdjustments(vararg values: PolicyStepAdjustmentArgs) {
        val toBeMapped = values.toList()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.stepAdjustments = mapped
    }

    /**
     * @param value Target tracking policy. These have the following structure:
     * ```typescript
     * import * as pulumi from "@pulumi/pulumi";
     * import * as aws from "@pulumi/aws";
     * const example = new aws.autoscaling.Policy("example", {targetTrackingConfiguration: {
     *     predefinedMetricSpecification: {
     *         predefinedMetricType: "ASGAverageCPUUtilization",
     *     },
     *     targetValue: 40,
     * }});
     * ```
     * ```python
     * import pulumi
     * import pulumi_aws as aws
     * example = aws.autoscaling.Policy("example", target_tracking_configuration=aws.autoscaling.PolicyTargetTrackingConfigurationArgs(
     *     predefined_metric_specification=aws.autoscaling.PolicyTargetTrackingConfigurationPredefinedMetricSpecificationArgs(
     *         predefined_metric_type="ASGAverageCPUUtilization",
     *     ),
     *     target_value=40,
     * ))
     * ```
     * ```csharp
     * using System.Collections.Generic;
     * using System.Linq;
     * using Pulumi;
     * using Aws = Pulumi.Aws;
     * return await Deployment.RunAsync(() =>
     * {
     *     var example = new Aws.AutoScaling.Policy("example", new()
     *     {
     *         TargetTrackingConfiguration = new Aws.AutoScaling.Inputs.PolicyTargetTrackingConfigurationArgs
     *         {
     *             PredefinedMetricSpecification = new Aws.AutoScaling.Inputs.PolicyTargetTrackingConfigurationPredefinedMetricSpecificationArgs
     *             {
     *                 PredefinedMetricType = "ASGAverageCPUUtilization",
     *             },
     *             TargetValue = 40,
     *         },
     *     });
     * });
     * ```
     * ```go
     * package main
     * import (
     * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/autoscaling"
     * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
     * )
     * func main() {
     * 	pulumi.Run(func(ctx *pulumi.Context) error {
     * 		_, err := autoscaling.NewPolicy(ctx, "example", &autoscaling.PolicyArgs{
     * 			TargetTrackingConfiguration: &autoscaling.PolicyTargetTrackingConfigurationArgs{
     * 				PredefinedMetricSpecification: &autoscaling.PolicyTargetTrackingConfigurationPredefinedMetricSpecificationArgs{
     * 					PredefinedMetricType: pulumi.String("ASGAverageCPUUtilization"),
     * 				},
     * 				TargetValue: pulumi.Float64(40),
     * 			},
     * 		})
     * 		if err != nil {
     * 			return err
     * 		}
     * 		return nil
     * 	})
     * }
     * ```
     * ```java
     * package generated_program;
     * import com.pulumi.Context;
     * import com.pulumi.Pulumi;
     * import com.pulumi.core.Output;
     * import com.pulumi.aws.autoscaling.Policy;
     * import com.pulumi.aws.autoscaling.PolicyArgs;
     * import com.pulumi.aws.autoscaling.inputs.PolicyTargetTrackingConfigurationArgs;
     * import com.pulumi.aws.autoscaling.inputs.PolicyTargetTrackingConfigurationPredefinedMetricSpecificationArgs;
     * import java.util.List;
     * import java.util.ArrayList;
     * import java.util.Map;
     * import java.io.File;
     * import java.nio.file.Files;
     * import java.nio.file.Paths;
     * public class App {
     *     public static void main(String[] args) {
     *         Pulumi.run(App::stack);
     *     }
     *     public static void stack(Context ctx) {
     *         var example = new Policy("example", PolicyArgs.builder()
     *             .targetTrackingConfiguration(PolicyTargetTrackingConfigurationArgs.builder()
     *                 .predefinedMetricSpecification(PolicyTargetTrackingConfigurationPredefinedMetricSpecificationArgs.builder()
     *                     .predefinedMetricType("ASGAverageCPUUtilization")
     *                     .build())
     *                 .targetValue(40)
     *                 .build())
     *             .build());
     *     }
     * }
     * ```
     * ```yaml
     * resources:
     *   example:
     *     type: aws:autoscaling:Policy
     *     properties:
     *       targetTrackingConfiguration:
     *         predefinedMetricSpecification:
     *           predefinedMetricType: ASGAverageCPUUtilization
     *         targetValue: 40
     * ```
     * The following fields are available in target tracking configuration:
     */
    @JvmName("tirexiprgegcjpjh")
    public suspend fun targetTrackingConfiguration(`value`: PolicyTargetTrackingConfigurationArgs?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.targetTrackingConfiguration = mapped
    }

    /**
     * @param argument Target tracking policy. These have the following structure:
     * ```typescript
     * import * as pulumi from "@pulumi/pulumi";
     * import * as aws from "@pulumi/aws";
     * const example = new aws.autoscaling.Policy("example", {targetTrackingConfiguration: {
     *     predefinedMetricSpecification: {
     *         predefinedMetricType: "ASGAverageCPUUtilization",
     *     },
     *     targetValue: 40,
     * }});
     * ```
     * ```python
     * import pulumi
     * import pulumi_aws as aws
     * example = aws.autoscaling.Policy("example", target_tracking_configuration=aws.autoscaling.PolicyTargetTrackingConfigurationArgs(
     *     predefined_metric_specification=aws.autoscaling.PolicyTargetTrackingConfigurationPredefinedMetricSpecificationArgs(
     *         predefined_metric_type="ASGAverageCPUUtilization",
     *     ),
     *     target_value=40,
     * ))
     * ```
     * ```csharp
     * using System.Collections.Generic;
     * using System.Linq;
     * using Pulumi;
     * using Aws = Pulumi.Aws;
     * return await Deployment.RunAsync(() =>
     * {
     *     var example = new Aws.AutoScaling.Policy("example", new()
     *     {
     *         TargetTrackingConfiguration = new Aws.AutoScaling.Inputs.PolicyTargetTrackingConfigurationArgs
     *         {
     *             PredefinedMetricSpecification = new Aws.AutoScaling.Inputs.PolicyTargetTrackingConfigurationPredefinedMetricSpecificationArgs
     *             {
     *                 PredefinedMetricType = "ASGAverageCPUUtilization",
     *             },
     *             TargetValue = 40,
     *         },
     *     });
     * });
     * ```
     * ```go
     * package main
     * import (
     * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/autoscaling"
     * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
     * )
     * func main() {
     * 	pulumi.Run(func(ctx *pulumi.Context) error {
     * 		_, err := autoscaling.NewPolicy(ctx, "example", &autoscaling.PolicyArgs{
     * 			TargetTrackingConfiguration: &autoscaling.PolicyTargetTrackingConfigurationArgs{
     * 				PredefinedMetricSpecification: &autoscaling.PolicyTargetTrackingConfigurationPredefinedMetricSpecificationArgs{
     * 					PredefinedMetricType: pulumi.String("ASGAverageCPUUtilization"),
     * 				},
     * 				TargetValue: pulumi.Float64(40),
     * 			},
     * 		})
     * 		if err != nil {
     * 			return err
     * 		}
     * 		return nil
     * 	})
     * }
     * ```
     * ```java
     * package generated_program;
     * import com.pulumi.Context;
     * import com.pulumi.Pulumi;
     * import com.pulumi.core.Output;
     * import com.pulumi.aws.autoscaling.Policy;
     * import com.pulumi.aws.autoscaling.PolicyArgs;
     * import com.pulumi.aws.autoscaling.inputs.PolicyTargetTrackingConfigurationArgs;
     * import com.pulumi.aws.autoscaling.inputs.PolicyTargetTrackingConfigurationPredefinedMetricSpecificationArgs;
     * import java.util.List;
     * import java.util.ArrayList;
     * import java.util.Map;
     * import java.io.File;
     * import java.nio.file.Files;
     * import java.nio.file.Paths;
     * public class App {
     *     public static void main(String[] args) {
     *         Pulumi.run(App::stack);
     *     }
     *     public static void stack(Context ctx) {
     *         var example = new Policy("example", PolicyArgs.builder()
     *             .targetTrackingConfiguration(PolicyTargetTrackingConfigurationArgs.builder()
     *                 .predefinedMetricSpecification(PolicyTargetTrackingConfigurationPredefinedMetricSpecificationArgs.builder()
     *                     .predefinedMetricType("ASGAverageCPUUtilization")
     *                     .build())
     *                 .targetValue(40)
     *                 .build())
     *             .build());
     *     }
     * }
     * ```
     * ```yaml
     * resources:
     *   example:
     *     type: aws:autoscaling:Policy
     *     properties:
     *       targetTrackingConfiguration:
     *         predefinedMetricSpecification:
     *           predefinedMetricType: ASGAverageCPUUtilization
     *         targetValue: 40
     * ```
     * The following fields are available in target tracking configuration:
     */
    @JvmName("gdhylsvqarsbxovq")
    public suspend
    fun targetTrackingConfiguration(argument: suspend PolicyTargetTrackingConfigurationArgsBuilder.() -> Unit) {
        val toBeMapped = PolicyTargetTrackingConfigurationArgsBuilder().applySuspend {
            argument()
        }.build()
        val mapped = of(toBeMapped)
        this.targetTrackingConfiguration = mapped
    }

    internal fun build(): PolicyArgs = PolicyArgs(
        adjustmentType = adjustmentType,
        autoscalingGroupName = autoscalingGroupName,
        cooldown = cooldown,
        enabled = enabled,
        estimatedInstanceWarmup = estimatedInstanceWarmup,
        metricAggregationType = metricAggregationType,
        minAdjustmentMagnitude = minAdjustmentMagnitude,
        name = name,
        policyType = policyType,
        predictiveScalingConfiguration = predictiveScalingConfiguration,
        scalingAdjustment = scalingAdjustment,
        stepAdjustments = stepAdjustments,
        targetTrackingConfiguration = targetTrackingConfiguration,
    )
}
