@file:Suppress("NAME_SHADOWING", "DEPRECATION")

package com.pulumi.aws.cognito.kotlin

import com.pulumi.aws.cognito.UserPoolClientArgs.builder
import com.pulumi.aws.cognito.kotlin.inputs.UserPoolClientAnalyticsConfigurationArgs
import com.pulumi.aws.cognito.kotlin.inputs.UserPoolClientAnalyticsConfigurationArgsBuilder
import com.pulumi.aws.cognito.kotlin.inputs.UserPoolClientTokenValidityUnitsArgs
import com.pulumi.aws.cognito.kotlin.inputs.UserPoolClientTokenValidityUnitsArgsBuilder
import com.pulumi.core.Output
import com.pulumi.core.Output.of
import com.pulumi.kotlin.ConvertibleToJava
import com.pulumi.kotlin.PulumiTagMarker
import com.pulumi.kotlin.applySuspend
import kotlin.Boolean
import kotlin.Int
import kotlin.String
import kotlin.Suppress
import kotlin.Unit
import kotlin.collections.List
import kotlin.jvm.JvmName

/**
 * Provides a Cognito User Pool Client resource.
 * To manage a User Pool Client created by another service, such as when [configuring an OpenSearch Domain to use Cognito authentication](https://docs.aws.amazon.com/opensearch-service/latest/developerguide/cognito-auth.html),
 * use the `aws.cognito.ManagedUserPoolClient` resource instead.
 * ## Example Usage
 * ### Create a basic user pool client
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.cognito.UserPool;
 * import com.pulumi.aws.cognito.UserPoolClient;
 * import com.pulumi.aws.cognito.UserPoolClientArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var pool = new UserPool("pool");
 *         var client = new UserPoolClient("client", UserPoolClientArgs.builder()
 *             .userPoolId(pool.id())
 *             .build());
 *     }
 * }
 * ```
 * ### Create a user pool client with no SRP authentication
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.cognito.UserPool;
 * import com.pulumi.aws.cognito.UserPoolClient;
 * import com.pulumi.aws.cognito.UserPoolClientArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var pool = new UserPool("pool");
 *         var client = new UserPoolClient("client", UserPoolClientArgs.builder()
 *             .userPoolId(pool.id())
 *             .generateSecret(true)
 *             .explicitAuthFlows("ADMIN_NO_SRP_AUTH")
 *             .build());
 *     }
 * }
 * ```
 * ### Create a user pool client with pinpoint analytics
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.cognito.UserPool;
 * import com.pulumi.aws.pinpoint.App;
 * import com.pulumi.aws.iam.IamFunctions;
 * import com.pulumi.aws.iam.inputs.GetPolicyDocumentArgs;
 * import com.pulumi.aws.iam.Role;
 * import com.pulumi.aws.iam.RoleArgs;
 * import com.pulumi.aws.cognito.UserPoolClient;
 * import com.pulumi.aws.cognito.UserPoolClientArgs;
 * import com.pulumi.aws.cognito.inputs.UserPoolClientAnalyticsConfigurationArgs;
 * import com.pulumi.aws.AwsFunctions;
 * import com.pulumi.aws.inputs.GetCallerIdentityArgs;
 * import com.pulumi.aws.iam.RolePolicy;
 * import com.pulumi.aws.iam.RolePolicyArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var testUserPool = new UserPool("testUserPool");
 *         var testApp = new App("testApp");
 *         final var assumeRole = IamFunctions.getPolicyDocument(GetPolicyDocumentArgs.builder()
 *             .statements(GetPolicyDocumentStatementArgs.builder()
 *                 .effect("Allow")
 *                 .principals(GetPolicyDocumentStatementPrincipalArgs.builder()
 *                     .type("Service")
 *                     .identifiers("cognito-idp.amazonaws.com")
 *                     .build())
 *                 .actions("sts:AssumeRole")
 *                 .build())
 *             .build());
 *         var testRole = new Role("testRole", RoleArgs.builder()
 *             .assumeRolePolicy(assumeRole.applyValue(getPolicyDocumentResult -> getPolicyDocumentResult.json()))
 *             .build());
 *         var testUserPoolClient = new UserPoolClient("testUserPoolClient", UserPoolClientArgs.builder()
 *             .userPoolId(testUserPool.id())
 *             .analyticsConfiguration(UserPoolClientAnalyticsConfigurationArgs.builder()
 *                 .applicationId(testApp.applicationId())
 *                 .externalId("some_id")
 *                 .roleArn(testRole.arn())
 *                 .userDataShared(true)
 *                 .build())
 *             .build());
 *         final var current = AwsFunctions.getCallerIdentity();
 *         final var testPolicyDocument = IamFunctions.getPolicyDocument(GetPolicyDocumentArgs.builder()
 *             .statements(GetPolicyDocumentStatementArgs.builder()
 *                 .effect("Allow")
 *                 .actions(
 *                     "mobiletargeting:UpdateEndpoint",
 *                     "mobiletargeting:PutEvents")
 *                 .resources(testApp.applicationId().applyValue(applicationId -> String.format("arn:aws:mobiletargeting:*:%s:apps/%s*", current.applyValue(getCallerIdentityResult -> getCallerIdentityResult.accountId()),applicationId)))
 *                 .build())
 *             .build());
 *         var testRolePolicy = new RolePolicy("testRolePolicy", RolePolicyArgs.builder()
 *             .role(testRole.id())
 *             .policy(testPolicyDocument.applyValue(getPolicyDocumentResult -> getPolicyDocumentResult).applyValue(testPolicyDocument -> testPolicyDocument.applyValue(getPolicyDocumentResult -> getPolicyDocumentResult.json())))
 *             .build());
 *     }
 * }
 * ```
 * ### Create a user pool client with Cognito as the identity provider
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.cognito.UserPool;
 * import com.pulumi.aws.cognito.UserPoolClient;
 * import com.pulumi.aws.cognito.UserPoolClientArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var pool = new UserPool("pool");
 *         var userpoolClient = new UserPoolClient("userpoolClient", UserPoolClientArgs.builder()
 *             .userPoolId(pool.id())
 *             .callbackUrls("https://example.com")
 *             .allowedOauthFlowsUserPoolClient(true)
 *             .allowedOauthFlows(
 *                 "code",
 *                 "implicit")
 *             .allowedOauthScopes(
 *                 "email",
 *                 "openid")
 *             .supportedIdentityProviders("COGNITO")
 *             .build());
 *     }
 * }
 * ```
 * ## Import
 * Using `pulumi import`, import Cognito User Pool Clients using the `id` of the Cognito User Pool, and the `id` of the Cognito User Pool Client. For example:
 * ```sh
 *  $ pulumi import aws:cognito/userPoolClient:UserPoolClient client us-west-2_abc123/3ho4ek12345678909nh3fmhpko
 * ```
 * @property accessTokenValidity Time limit, between 5 minutes and 1 day, after which the access token is no longer valid and cannot be used.
 * By default, the unit is hours.
 * The unit can be overridden by a value in `token_validity_units.access_token`.
 * @property allowedOauthFlows List of allowed OAuth flows (code, implicit, client_credentials).
 * @property allowedOauthFlowsUserPoolClient Whether the client is allowed to follow the OAuth protocol when interacting with Cognito user pools.
 * @property allowedOauthScopes List of allowed OAuth scopes (phone, email, openid, profile, and aws.cognito.signin.user.admin).
 * @property analyticsConfiguration Configuration block for Amazon Pinpoint analytics for collecting metrics for this user pool. Detailed below.
 * @property authSessionValidity Amazon Cognito creates a session token for each API request in an authentication flow. AuthSessionValidity is the duration, in minutes, of that session token. Your user pool native user must respond to each authentication challenge before the session expires. Valid values between `3` and `15`. Default value is `3`.
 * @property callbackUrls List of allowed callback URLs for the identity providers.
 * @property defaultRedirectUri Default redirect URI. Must be in the list of callback URLs.
 * @property enablePropagateAdditionalUserContextData Activates the propagation of additional user context data.
 * @property enableTokenRevocation Enables or disables token revocation.
 * @property explicitAuthFlows List of authentication flows (ADMIN_NO_SRP_AUTH, CUSTOM_AUTH_FLOW_ONLY, USER_PASSWORD_AUTH, ALLOW_ADMIN_USER_PASSWORD_AUTH, ALLOW_CUSTOM_AUTH, ALLOW_USER_PASSWORD_AUTH, ALLOW_USER_SRP_AUTH, ALLOW_REFRESH_TOKEN_AUTH).
 * @property generateSecret Should an application secret be generated.
 * @property idTokenValidity Time limit, between 5 minutes and 1 day, after which the ID token is no longer valid and cannot be used.
 * By default, the unit is hours.
 * The unit can be overridden by a value in `token_validity_units.id_token`.
 * @property logoutUrls List of allowed logout URLs for the identity providers.
 * @property name Name of the application client.
 * @property preventUserExistenceErrors Choose which errors and responses are returned by Cognito APIs during authentication, account confirmation, and password recovery when the user does not exist in the user pool. When set to `ENABLED` and the user does not exist, authentication returns an error indicating either the username or password was incorrect, and account confirmation and password recovery return a response indicating a code was sent to a simulated destination. When set to `LEGACY`, those APIs will return a `UserNotFoundException` exception if the user does not exist in the user pool.
 * @property readAttributes List of user pool attributes the application client can read from.
 * @property refreshTokenValidity Time limit, between 60 minutes and 10 years, after which the refresh token is no longer valid and cannot be used.
 * By default, the unit is days.
 * The unit can be overridden by a value in `token_validity_units.refresh_token`.
 * @property supportedIdentityProviders List of provider names for the identity providers that are supported on this client. Uses the `provider_name` attribute of `aws.cognito.IdentityProvider` resource(s), or the equivalent string(s).
 * @property tokenValidityUnits Configuration block for units in which the validity times are represented in. Detailed below.
 * @property userPoolId User pool the client belongs to.
 * The following arguments are optional:
 * @property writeAttributes List of user pool attributes the application client can write to.
 */
public data class UserPoolClientArgs(
    public val accessTokenValidity: Output<Int>? = null,
    public val allowedOauthFlows: Output<List<String>>? = null,
    public val allowedOauthFlowsUserPoolClient: Output<Boolean>? = null,
    public val allowedOauthScopes: Output<List<String>>? = null,
    public val analyticsConfiguration: Output<UserPoolClientAnalyticsConfigurationArgs>? = null,
    public val authSessionValidity: Output<Int>? = null,
    public val callbackUrls: Output<List<String>>? = null,
    public val defaultRedirectUri: Output<String>? = null,
    public val enablePropagateAdditionalUserContextData: Output<Boolean>? = null,
    public val enableTokenRevocation: Output<Boolean>? = null,
    public val explicitAuthFlows: Output<List<String>>? = null,
    public val generateSecret: Output<Boolean>? = null,
    public val idTokenValidity: Output<Int>? = null,
    public val logoutUrls: Output<List<String>>? = null,
    public val name: Output<String>? = null,
    public val preventUserExistenceErrors: Output<String>? = null,
    public val readAttributes: Output<List<String>>? = null,
    public val refreshTokenValidity: Output<Int>? = null,
    public val supportedIdentityProviders: Output<List<String>>? = null,
    public val tokenValidityUnits: Output<UserPoolClientTokenValidityUnitsArgs>? = null,
    public val userPoolId: Output<String>? = null,
    public val writeAttributes: Output<List<String>>? = null,
) : ConvertibleToJava<com.pulumi.aws.cognito.UserPoolClientArgs> {
    override fun toJava(): com.pulumi.aws.cognito.UserPoolClientArgs =
        com.pulumi.aws.cognito.UserPoolClientArgs.builder()
            .accessTokenValidity(accessTokenValidity?.applyValue({ args0 -> args0 }))
            .allowedOauthFlows(allowedOauthFlows?.applyValue({ args0 -> args0.map({ args0 -> args0 }) }))
            .allowedOauthFlowsUserPoolClient(allowedOauthFlowsUserPoolClient?.applyValue({ args0 -> args0 }))
            .allowedOauthScopes(allowedOauthScopes?.applyValue({ args0 -> args0.map({ args0 -> args0 }) }))
            .analyticsConfiguration(
                analyticsConfiguration?.applyValue({ args0 ->
                    args0.let({ args0 ->
                        args0.toJava()
                    })
                }),
            )
            .authSessionValidity(authSessionValidity?.applyValue({ args0 -> args0 }))
            .callbackUrls(callbackUrls?.applyValue({ args0 -> args0.map({ args0 -> args0 }) }))
            .defaultRedirectUri(defaultRedirectUri?.applyValue({ args0 -> args0 }))
            .enablePropagateAdditionalUserContextData(
                enablePropagateAdditionalUserContextData?.applyValue({ args0 ->
                    args0
                }),
            )
            .enableTokenRevocation(enableTokenRevocation?.applyValue({ args0 -> args0 }))
            .explicitAuthFlows(explicitAuthFlows?.applyValue({ args0 -> args0.map({ args0 -> args0 }) }))
            .generateSecret(generateSecret?.applyValue({ args0 -> args0 }))
            .idTokenValidity(idTokenValidity?.applyValue({ args0 -> args0 }))
            .logoutUrls(logoutUrls?.applyValue({ args0 -> args0.map({ args0 -> args0 }) }))
            .name(name?.applyValue({ args0 -> args0 }))
            .preventUserExistenceErrors(preventUserExistenceErrors?.applyValue({ args0 -> args0 }))
            .readAttributes(readAttributes?.applyValue({ args0 -> args0.map({ args0 -> args0 }) }))
            .refreshTokenValidity(refreshTokenValidity?.applyValue({ args0 -> args0 }))
            .supportedIdentityProviders(
                supportedIdentityProviders?.applyValue({ args0 ->
                    args0.map({ args0 ->
                        args0
                    })
                }),
            )
            .tokenValidityUnits(
                tokenValidityUnits?.applyValue({ args0 ->
                    args0.let({ args0 ->
                        args0.toJava()
                    })
                }),
            )
            .userPoolId(userPoolId?.applyValue({ args0 -> args0 }))
            .writeAttributes(writeAttributes?.applyValue({ args0 -> args0.map({ args0 -> args0 }) })).build()
}

/**
 * Builder for [UserPoolClientArgs].
 */
@PulumiTagMarker
public class UserPoolClientArgsBuilder internal constructor() {
    private var accessTokenValidity: Output<Int>? = null

    private var allowedOauthFlows: Output<List<String>>? = null

    private var allowedOauthFlowsUserPoolClient: Output<Boolean>? = null

    private var allowedOauthScopes: Output<List<String>>? = null

    private var analyticsConfiguration: Output<UserPoolClientAnalyticsConfigurationArgs>? = null

    private var authSessionValidity: Output<Int>? = null

    private var callbackUrls: Output<List<String>>? = null

    private var defaultRedirectUri: Output<String>? = null

    private var enablePropagateAdditionalUserContextData: Output<Boolean>? = null

    private var enableTokenRevocation: Output<Boolean>? = null

    private var explicitAuthFlows: Output<List<String>>? = null

    private var generateSecret: Output<Boolean>? = null

    private var idTokenValidity: Output<Int>? = null

    private var logoutUrls: Output<List<String>>? = null

    private var name: Output<String>? = null

    private var preventUserExistenceErrors: Output<String>? = null

    private var readAttributes: Output<List<String>>? = null

    private var refreshTokenValidity: Output<Int>? = null

    private var supportedIdentityProviders: Output<List<String>>? = null

    private var tokenValidityUnits: Output<UserPoolClientTokenValidityUnitsArgs>? = null

    private var userPoolId: Output<String>? = null

    private var writeAttributes: Output<List<String>>? = null

    /**
     * @param value Time limit, between 5 minutes and 1 day, after which the access token is no longer valid and cannot be used.
     * By default, the unit is hours.
     * The unit can be overridden by a value in `token_validity_units.access_token`.
     */
    @JvmName("egindcgcpjyincyr")
    public suspend fun accessTokenValidity(`value`: Output<Int>) {
        this.accessTokenValidity = value
    }

    /**
     * @param value List of allowed OAuth flows (code, implicit, client_credentials).
     */
    @JvmName("twjwrochshlsqwpm")
    public suspend fun allowedOauthFlows(`value`: Output<List<String>>) {
        this.allowedOauthFlows = value
    }

    @JvmName("vqxxfbdgswpttvbk")
    public suspend fun allowedOauthFlows(vararg values: Output<String>) {
        this.allowedOauthFlows = Output.all(values.asList())
    }

    /**
     * @param values List of allowed OAuth flows (code, implicit, client_credentials).
     */
    @JvmName("dlyphotkcplkevqw")
    public suspend fun allowedOauthFlows(values: List<Output<String>>) {
        this.allowedOauthFlows = Output.all(values)
    }

    /**
     * @param value Whether the client is allowed to follow the OAuth protocol when interacting with Cognito user pools.
     */
    @JvmName("vxsninyahcenyjgd")
    public suspend fun allowedOauthFlowsUserPoolClient(`value`: Output<Boolean>) {
        this.allowedOauthFlowsUserPoolClient = value
    }

    /**
     * @param value List of allowed OAuth scopes (phone, email, openid, profile, and aws.cognito.signin.user.admin).
     */
    @JvmName("eawldptaeiwgqcnm")
    public suspend fun allowedOauthScopes(`value`: Output<List<String>>) {
        this.allowedOauthScopes = value
    }

    @JvmName("fdxkpdjbeotehnpu")
    public suspend fun allowedOauthScopes(vararg values: Output<String>) {
        this.allowedOauthScopes = Output.all(values.asList())
    }

    /**
     * @param values List of allowed OAuth scopes (phone, email, openid, profile, and aws.cognito.signin.user.admin).
     */
    @JvmName("athwnrnxqqbwsjct")
    public suspend fun allowedOauthScopes(values: List<Output<String>>) {
        this.allowedOauthScopes = Output.all(values)
    }

    /**
     * @param value Configuration block for Amazon Pinpoint analytics for collecting metrics for this user pool. Detailed below.
     */
    @JvmName("uayjkturobuyawcm")
    public suspend
    fun analyticsConfiguration(`value`: Output<UserPoolClientAnalyticsConfigurationArgs>) {
        this.analyticsConfiguration = value
    }

    /**
     * @param value Amazon Cognito creates a session token for each API request in an authentication flow. AuthSessionValidity is the duration, in minutes, of that session token. Your user pool native user must respond to each authentication challenge before the session expires. Valid values between `3` and `15`. Default value is `3`.
     */
    @JvmName("wadwnxypxfxutksr")
    public suspend fun authSessionValidity(`value`: Output<Int>) {
        this.authSessionValidity = value
    }

    /**
     * @param value List of allowed callback URLs for the identity providers.
     */
    @JvmName("usnrbqcsxrgyegnw")
    public suspend fun callbackUrls(`value`: Output<List<String>>) {
        this.callbackUrls = value
    }

    @JvmName("yktfktmrjqubelvb")
    public suspend fun callbackUrls(vararg values: Output<String>) {
        this.callbackUrls = Output.all(values.asList())
    }

    /**
     * @param values List of allowed callback URLs for the identity providers.
     */
    @JvmName("mfsxjuiebfdwdaxy")
    public suspend fun callbackUrls(values: List<Output<String>>) {
        this.callbackUrls = Output.all(values)
    }

    /**
     * @param value Default redirect URI. Must be in the list of callback URLs.
     */
    @JvmName("tvxyjuowbdafsguu")
    public suspend fun defaultRedirectUri(`value`: Output<String>) {
        this.defaultRedirectUri = value
    }

    /**
     * @param value Activates the propagation of additional user context data.
     */
    @JvmName("lsbynwtlwbdvfuaa")
    public suspend fun enablePropagateAdditionalUserContextData(`value`: Output<Boolean>) {
        this.enablePropagateAdditionalUserContextData = value
    }

    /**
     * @param value Enables or disables token revocation.
     */
    @JvmName("gnghbhkfitidtuhc")
    public suspend fun enableTokenRevocation(`value`: Output<Boolean>) {
        this.enableTokenRevocation = value
    }

    /**
     * @param value List of authentication flows (ADMIN_NO_SRP_AUTH, CUSTOM_AUTH_FLOW_ONLY, USER_PASSWORD_AUTH, ALLOW_ADMIN_USER_PASSWORD_AUTH, ALLOW_CUSTOM_AUTH, ALLOW_USER_PASSWORD_AUTH, ALLOW_USER_SRP_AUTH, ALLOW_REFRESH_TOKEN_AUTH).
     */
    @JvmName("pmgaquafdhpkvujs")
    public suspend fun explicitAuthFlows(`value`: Output<List<String>>) {
        this.explicitAuthFlows = value
    }

    @JvmName("vtmxskeijhyswqjf")
    public suspend fun explicitAuthFlows(vararg values: Output<String>) {
        this.explicitAuthFlows = Output.all(values.asList())
    }

    /**
     * @param values List of authentication flows (ADMIN_NO_SRP_AUTH, CUSTOM_AUTH_FLOW_ONLY, USER_PASSWORD_AUTH, ALLOW_ADMIN_USER_PASSWORD_AUTH, ALLOW_CUSTOM_AUTH, ALLOW_USER_PASSWORD_AUTH, ALLOW_USER_SRP_AUTH, ALLOW_REFRESH_TOKEN_AUTH).
     */
    @JvmName("rfrvshertjvgpacg")
    public suspend fun explicitAuthFlows(values: List<Output<String>>) {
        this.explicitAuthFlows = Output.all(values)
    }

    /**
     * @param value Should an application secret be generated.
     */
    @JvmName("rtcrerirxglypnkk")
    public suspend fun generateSecret(`value`: Output<Boolean>) {
        this.generateSecret = value
    }

    /**
     * @param value Time limit, between 5 minutes and 1 day, after which the ID token is no longer valid and cannot be used.
     * By default, the unit is hours.
     * The unit can be overridden by a value in `token_validity_units.id_token`.
     */
    @JvmName("msqdityrryfhgjty")
    public suspend fun idTokenValidity(`value`: Output<Int>) {
        this.idTokenValidity = value
    }

    /**
     * @param value List of allowed logout URLs for the identity providers.
     */
    @JvmName("iybgcvyjbmxcuuif")
    public suspend fun logoutUrls(`value`: Output<List<String>>) {
        this.logoutUrls = value
    }

    @JvmName("ujcamknqauqfjocq")
    public suspend fun logoutUrls(vararg values: Output<String>) {
        this.logoutUrls = Output.all(values.asList())
    }

    /**
     * @param values List of allowed logout URLs for the identity providers.
     */
    @JvmName("ipmgnaksrloyegmo")
    public suspend fun logoutUrls(values: List<Output<String>>) {
        this.logoutUrls = Output.all(values)
    }

    /**
     * @param value Name of the application client.
     */
    @JvmName("xyphwxxqcusqrclg")
    public suspend fun name(`value`: Output<String>) {
        this.name = value
    }

    /**
     * @param value Choose which errors and responses are returned by Cognito APIs during authentication, account confirmation, and password recovery when the user does not exist in the user pool. When set to `ENABLED` and the user does not exist, authentication returns an error indicating either the username or password was incorrect, and account confirmation and password recovery return a response indicating a code was sent to a simulated destination. When set to `LEGACY`, those APIs will return a `UserNotFoundException` exception if the user does not exist in the user pool.
     */
    @JvmName("cxpirowiqabfqwnn")
    public suspend fun preventUserExistenceErrors(`value`: Output<String>) {
        this.preventUserExistenceErrors = value
    }

    /**
     * @param value List of user pool attributes the application client can read from.
     */
    @JvmName("paewexylpnvvtxxy")
    public suspend fun readAttributes(`value`: Output<List<String>>) {
        this.readAttributes = value
    }

    @JvmName("wmfjpfxmintcnclm")
    public suspend fun readAttributes(vararg values: Output<String>) {
        this.readAttributes = Output.all(values.asList())
    }

    /**
     * @param values List of user pool attributes the application client can read from.
     */
    @JvmName("nhbwgpygxiaunspn")
    public suspend fun readAttributes(values: List<Output<String>>) {
        this.readAttributes = Output.all(values)
    }

    /**
     * @param value Time limit, between 60 minutes and 10 years, after which the refresh token is no longer valid and cannot be used.
     * By default, the unit is days.
     * The unit can be overridden by a value in `token_validity_units.refresh_token`.
     */
    @JvmName("cvraatbmtlbtxsyj")
    public suspend fun refreshTokenValidity(`value`: Output<Int>) {
        this.refreshTokenValidity = value
    }

    /**
     * @param value List of provider names for the identity providers that are supported on this client. Uses the `provider_name` attribute of `aws.cognito.IdentityProvider` resource(s), or the equivalent string(s).
     */
    @JvmName("jhnebsflpixhknxl")
    public suspend fun supportedIdentityProviders(`value`: Output<List<String>>) {
        this.supportedIdentityProviders = value
    }

    @JvmName("jdforvpkmpnnrhkv")
    public suspend fun supportedIdentityProviders(vararg values: Output<String>) {
        this.supportedIdentityProviders = Output.all(values.asList())
    }

    /**
     * @param values List of provider names for the identity providers that are supported on this client. Uses the `provider_name` attribute of `aws.cognito.IdentityProvider` resource(s), or the equivalent string(s).
     */
    @JvmName("vcldcnjhiwipgarv")
    public suspend fun supportedIdentityProviders(values: List<Output<String>>) {
        this.supportedIdentityProviders = Output.all(values)
    }

    /**
     * @param value Configuration block for units in which the validity times are represented in. Detailed below.
     */
    @JvmName("fftqgvhsqbdwujkl")
    public suspend fun tokenValidityUnits(`value`: Output<UserPoolClientTokenValidityUnitsArgs>) {
        this.tokenValidityUnits = value
    }

    /**
     * @param value User pool the client belongs to.
     * The following arguments are optional:
     */
    @JvmName("rnnkuwatakndinxi")
    public suspend fun userPoolId(`value`: Output<String>) {
        this.userPoolId = value
    }

    /**
     * @param value List of user pool attributes the application client can write to.
     */
    @JvmName("frrnaspvsvwutlbk")
    public suspend fun writeAttributes(`value`: Output<List<String>>) {
        this.writeAttributes = value
    }

    @JvmName("tjnrsppronkgqnga")
    public suspend fun writeAttributes(vararg values: Output<String>) {
        this.writeAttributes = Output.all(values.asList())
    }

    /**
     * @param values List of user pool attributes the application client can write to.
     */
    @JvmName("ysxqtwfwelgkrkhi")
    public suspend fun writeAttributes(values: List<Output<String>>) {
        this.writeAttributes = Output.all(values)
    }

    /**
     * @param value Time limit, between 5 minutes and 1 day, after which the access token is no longer valid and cannot be used.
     * By default, the unit is hours.
     * The unit can be overridden by a value in `token_validity_units.access_token`.
     */
    @JvmName("brtskdlbusgbpjnl")
    public suspend fun accessTokenValidity(`value`: Int?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.accessTokenValidity = mapped
    }

    /**
     * @param value List of allowed OAuth flows (code, implicit, client_credentials).
     */
    @JvmName("cxqaelcdpfbefpsy")
    public suspend fun allowedOauthFlows(`value`: List<String>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.allowedOauthFlows = mapped
    }

    /**
     * @param values List of allowed OAuth flows (code, implicit, client_credentials).
     */
    @JvmName("qyrhfgopxwcaudwd")
    public suspend fun allowedOauthFlows(vararg values: String) {
        val toBeMapped = values.toList()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.allowedOauthFlows = mapped
    }

    /**
     * @param value Whether the client is allowed to follow the OAuth protocol when interacting with Cognito user pools.
     */
    @JvmName("noajmnwqhotdjuhg")
    public suspend fun allowedOauthFlowsUserPoolClient(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.allowedOauthFlowsUserPoolClient = mapped
    }

    /**
     * @param value List of allowed OAuth scopes (phone, email, openid, profile, and aws.cognito.signin.user.admin).
     */
    @JvmName("xembqdsqdmkdmffh")
    public suspend fun allowedOauthScopes(`value`: List<String>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.allowedOauthScopes = mapped
    }

    /**
     * @param values List of allowed OAuth scopes (phone, email, openid, profile, and aws.cognito.signin.user.admin).
     */
    @JvmName("wofulyrftorxmqrl")
    public suspend fun allowedOauthScopes(vararg values: String) {
        val toBeMapped = values.toList()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.allowedOauthScopes = mapped
    }

    /**
     * @param value Configuration block for Amazon Pinpoint analytics for collecting metrics for this user pool. Detailed below.
     */
    @JvmName("anogecdeuwrjlxph")
    public suspend fun analyticsConfiguration(`value`: UserPoolClientAnalyticsConfigurationArgs?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.analyticsConfiguration = mapped
    }

    /**
     * @param argument Configuration block for Amazon Pinpoint analytics for collecting metrics for this user pool. Detailed below.
     */
    @JvmName("pnbdnsqtoiryuoos")
    public suspend
    fun analyticsConfiguration(argument: suspend UserPoolClientAnalyticsConfigurationArgsBuilder.() -> Unit) {
        val toBeMapped = UserPoolClientAnalyticsConfigurationArgsBuilder().applySuspend {
            argument()
        }.build()
        val mapped = of(toBeMapped)
        this.analyticsConfiguration = mapped
    }

    /**
     * @param value Amazon Cognito creates a session token for each API request in an authentication flow. AuthSessionValidity is the duration, in minutes, of that session token. Your user pool native user must respond to each authentication challenge before the session expires. Valid values between `3` and `15`. Default value is `3`.
     */
    @JvmName("qqjdkaixppryeoxr")
    public suspend fun authSessionValidity(`value`: Int?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.authSessionValidity = mapped
    }

    /**
     * @param value List of allowed callback URLs for the identity providers.
     */
    @JvmName("xnnwcmnviaayobvg")
    public suspend fun callbackUrls(`value`: List<String>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.callbackUrls = mapped
    }

    /**
     * @param values List of allowed callback URLs for the identity providers.
     */
    @JvmName("gtupifaruhlgqskh")
    public suspend fun callbackUrls(vararg values: String) {
        val toBeMapped = values.toList()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.callbackUrls = mapped
    }

    /**
     * @param value Default redirect URI. Must be in the list of callback URLs.
     */
    @JvmName("gbducfmtqdjijigu")
    public suspend fun defaultRedirectUri(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.defaultRedirectUri = mapped
    }

    /**
     * @param value Activates the propagation of additional user context data.
     */
    @JvmName("qoumuywwconppxnf")
    public suspend fun enablePropagateAdditionalUserContextData(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.enablePropagateAdditionalUserContextData = mapped
    }

    /**
     * @param value Enables or disables token revocation.
     */
    @JvmName("ottqfslumnanhyax")
    public suspend fun enableTokenRevocation(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.enableTokenRevocation = mapped
    }

    /**
     * @param value List of authentication flows (ADMIN_NO_SRP_AUTH, CUSTOM_AUTH_FLOW_ONLY, USER_PASSWORD_AUTH, ALLOW_ADMIN_USER_PASSWORD_AUTH, ALLOW_CUSTOM_AUTH, ALLOW_USER_PASSWORD_AUTH, ALLOW_USER_SRP_AUTH, ALLOW_REFRESH_TOKEN_AUTH).
     */
    @JvmName("xqrsebajooyfuxmq")
    public suspend fun explicitAuthFlows(`value`: List<String>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.explicitAuthFlows = mapped
    }

    /**
     * @param values List of authentication flows (ADMIN_NO_SRP_AUTH, CUSTOM_AUTH_FLOW_ONLY, USER_PASSWORD_AUTH, ALLOW_ADMIN_USER_PASSWORD_AUTH, ALLOW_CUSTOM_AUTH, ALLOW_USER_PASSWORD_AUTH, ALLOW_USER_SRP_AUTH, ALLOW_REFRESH_TOKEN_AUTH).
     */
    @JvmName("irwhlxkufyhrmplo")
    public suspend fun explicitAuthFlows(vararg values: String) {
        val toBeMapped = values.toList()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.explicitAuthFlows = mapped
    }

    /**
     * @param value Should an application secret be generated.
     */
    @JvmName("rwawcigobrcdrbhg")
    public suspend fun generateSecret(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.generateSecret = mapped
    }

    /**
     * @param value Time limit, between 5 minutes and 1 day, after which the ID token is no longer valid and cannot be used.
     * By default, the unit is hours.
     * The unit can be overridden by a value in `token_validity_units.id_token`.
     */
    @JvmName("qgipxgdikyvpkddr")
    public suspend fun idTokenValidity(`value`: Int?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.idTokenValidity = mapped
    }

    /**
     * @param value List of allowed logout URLs for the identity providers.
     */
    @JvmName("vrojekqqrbydivsd")
    public suspend fun logoutUrls(`value`: List<String>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.logoutUrls = mapped
    }

    /**
     * @param values List of allowed logout URLs for the identity providers.
     */
    @JvmName("rtykiwoptbsqnusl")
    public suspend fun logoutUrls(vararg values: String) {
        val toBeMapped = values.toList()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.logoutUrls = mapped
    }

    /**
     * @param value Name of the application client.
     */
    @JvmName("tgtmsnoutgxolpgd")
    public suspend fun name(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.name = mapped
    }

    /**
     * @param value Choose which errors and responses are returned by Cognito APIs during authentication, account confirmation, and password recovery when the user does not exist in the user pool. When set to `ENABLED` and the user does not exist, authentication returns an error indicating either the username or password was incorrect, and account confirmation and password recovery return a response indicating a code was sent to a simulated destination. When set to `LEGACY`, those APIs will return a `UserNotFoundException` exception if the user does not exist in the user pool.
     */
    @JvmName("ycvtuaqelykmuaru")
    public suspend fun preventUserExistenceErrors(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.preventUserExistenceErrors = mapped
    }

    /**
     * @param value List of user pool attributes the application client can read from.
     */
    @JvmName("vvjkqraxexvgrhog")
    public suspend fun readAttributes(`value`: List<String>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.readAttributes = mapped
    }

    /**
     * @param values List of user pool attributes the application client can read from.
     */
    @JvmName("sburneumbfhiatym")
    public suspend fun readAttributes(vararg values: String) {
        val toBeMapped = values.toList()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.readAttributes = mapped
    }

    /**
     * @param value Time limit, between 60 minutes and 10 years, after which the refresh token is no longer valid and cannot be used.
     * By default, the unit is days.
     * The unit can be overridden by a value in `token_validity_units.refresh_token`.
     */
    @JvmName("ckahiepfptxpdqlq")
    public suspend fun refreshTokenValidity(`value`: Int?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.refreshTokenValidity = mapped
    }

    /**
     * @param value List of provider names for the identity providers that are supported on this client. Uses the `provider_name` attribute of `aws.cognito.IdentityProvider` resource(s), or the equivalent string(s).
     */
    @JvmName("jgxqaamumxcsolqc")
    public suspend fun supportedIdentityProviders(`value`: List<String>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.supportedIdentityProviders = mapped
    }

    /**
     * @param values List of provider names for the identity providers that are supported on this client. Uses the `provider_name` attribute of `aws.cognito.IdentityProvider` resource(s), or the equivalent string(s).
     */
    @JvmName("nwyoobhsyrbmumol")
    public suspend fun supportedIdentityProviders(vararg values: String) {
        val toBeMapped = values.toList()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.supportedIdentityProviders = mapped
    }

    /**
     * @param value Configuration block for units in which the validity times are represented in. Detailed below.
     */
    @JvmName("mgpqpcwuticjbqve")
    public suspend fun tokenValidityUnits(`value`: UserPoolClientTokenValidityUnitsArgs?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.tokenValidityUnits = mapped
    }

    /**
     * @param argument Configuration block for units in which the validity times are represented in. Detailed below.
     */
    @JvmName("hftahprsqhnhpigi")
    public suspend
    fun tokenValidityUnits(argument: suspend UserPoolClientTokenValidityUnitsArgsBuilder.() -> Unit) {
        val toBeMapped = UserPoolClientTokenValidityUnitsArgsBuilder().applySuspend {
            argument()
        }.build()
        val mapped = of(toBeMapped)
        this.tokenValidityUnits = mapped
    }

    /**
     * @param value User pool the client belongs to.
     * The following arguments are optional:
     */
    @JvmName("emwlbhlganrlqpdc")
    public suspend fun userPoolId(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.userPoolId = mapped
    }

    /**
     * @param value List of user pool attributes the application client can write to.
     */
    @JvmName("aemppokmwysfxafn")
    public suspend fun writeAttributes(`value`: List<String>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.writeAttributes = mapped
    }

    /**
     * @param values List of user pool attributes the application client can write to.
     */
    @JvmName("ehguoliehxkvjdbm")
    public suspend fun writeAttributes(vararg values: String) {
        val toBeMapped = values.toList()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.writeAttributes = mapped
    }

    internal fun build(): UserPoolClientArgs = UserPoolClientArgs(
        accessTokenValidity = accessTokenValidity,
        allowedOauthFlows = allowedOauthFlows,
        allowedOauthFlowsUserPoolClient = allowedOauthFlowsUserPoolClient,
        allowedOauthScopes = allowedOauthScopes,
        analyticsConfiguration = analyticsConfiguration,
        authSessionValidity = authSessionValidity,
        callbackUrls = callbackUrls,
        defaultRedirectUri = defaultRedirectUri,
        enablePropagateAdditionalUserContextData = enablePropagateAdditionalUserContextData,
        enableTokenRevocation = enableTokenRevocation,
        explicitAuthFlows = explicitAuthFlows,
        generateSecret = generateSecret,
        idTokenValidity = idTokenValidity,
        logoutUrls = logoutUrls,
        name = name,
        preventUserExistenceErrors = preventUserExistenceErrors,
        readAttributes = readAttributes,
        refreshTokenValidity = refreshTokenValidity,
        supportedIdentityProviders = supportedIdentityProviders,
        tokenValidityUnits = tokenValidityUnits,
        userPoolId = userPoolId,
        writeAttributes = writeAttributes,
    )
}
