@file:Suppress("NAME_SHADOWING", "DEPRECATION")

package com.pulumi.aws.dlm.kotlin

import com.pulumi.aws.dlm.LifecyclePolicyArgs.builder
import com.pulumi.aws.dlm.kotlin.inputs.LifecyclePolicyPolicyDetailsArgs
import com.pulumi.aws.dlm.kotlin.inputs.LifecyclePolicyPolicyDetailsArgsBuilder
import com.pulumi.core.Output
import com.pulumi.core.Output.of
import com.pulumi.kotlin.ConvertibleToJava
import com.pulumi.kotlin.PulumiTagMarker
import com.pulumi.kotlin.applySuspend
import kotlin.Pair
import kotlin.String
import kotlin.Suppress
import kotlin.Unit
import kotlin.collections.Map
import kotlin.jvm.JvmName

/**
 * Provides a [Data Lifecycle Manager (DLM) lifecycle policy](https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/snapshot-lifecycle.html) for managing snapshots.
 * ## Example Usage
 * ### Basic
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.iam.IamFunctions;
 * import com.pulumi.aws.iam.inputs.GetPolicyDocumentArgs;
 * import com.pulumi.aws.iam.Role;
 * import com.pulumi.aws.iam.RoleArgs;
 * import com.pulumi.aws.iam.RolePolicy;
 * import com.pulumi.aws.iam.RolePolicyArgs;
 * import com.pulumi.aws.dlm.LifecyclePolicy;
 * import com.pulumi.aws.dlm.LifecyclePolicyArgs;
 * import com.pulumi.aws.dlm.inputs.LifecyclePolicyPolicyDetailsArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         final var assumeRole = IamFunctions.getPolicyDocument(GetPolicyDocumentArgs.builder()
 *             .statements(GetPolicyDocumentStatementArgs.builder()
 *                 .effect("Allow")
 *                 .principals(GetPolicyDocumentStatementPrincipalArgs.builder()
 *                     .type("Service")
 *                     .identifiers("dlm.amazonaws.com")
 *                     .build())
 *                 .actions("sts:AssumeRole")
 *                 .build())
 *             .build());
 *         var dlmLifecycleRole = new Role("dlmLifecycleRole", RoleArgs.builder()
 *             .assumeRolePolicy(assumeRole.applyValue(getPolicyDocumentResult -> getPolicyDocumentResult.json()))
 *             .build());
 *         final var dlmLifecyclePolicyDocument = IamFunctions.getPolicyDocument(GetPolicyDocumentArgs.builder()
 *             .statements(
 *                 GetPolicyDocumentStatementArgs.builder()
 *                     .effect("Allow")
 *                     .actions(
 *                         "ec2:CreateSnapshot",
 *                         "ec2:CreateSnapshots",
 *                         "ec2:DeleteSnapshot",
 *                         "ec2:DescribeInstances",
 *                         "ec2:DescribeVolumes",
 *                         "ec2:DescribeSnapshots")
 *                     .resources("*")
 *                     .build(),
 *                 GetPolicyDocumentStatementArgs.builder()
 *                     .effect("Allow")
 *                     .actions("ec2:CreateTags")
 *                     .resources("arn:aws:ec2:*::snapshot/*")
 *                     .build())
 *             .build());
 *         var dlmLifecycleRolePolicy = new RolePolicy("dlmLifecycleRolePolicy", RolePolicyArgs.builder()
 *             .role(dlmLifecycleRole.id())
 *             .policy(dlmLifecyclePolicyDocument.applyValue(getPolicyDocumentResult -> getPolicyDocumentResult.json()))
 *             .build());
 *         var example = new LifecyclePolicy("example", LifecyclePolicyArgs.builder()
 *             .description("example DLM lifecycle policy")
 *             .executionRoleArn(dlmLifecycleRole.arn())
 *             .state("ENABLED")
 *             .policyDetails(LifecyclePolicyPolicyDetailsArgs.builder()
 *                 .resourceTypes("VOLUME")
 *                 .schedules(LifecyclePolicyPolicyDetailsScheduleArgs.builder()
 *                     .name("2 weeks of daily snapshots")
 *                     .createRule(LifecyclePolicyPolicyDetailsScheduleCreateRuleArgs.builder()
 *                         .interval(24)
 *                         .intervalUnit("HOURS")
 *                         .times("23:45")
 *                         .build())
 *                     .retainRule(LifecyclePolicyPolicyDetailsScheduleRetainRuleArgs.builder()
 *                         .count(14)
 *                         .build())
 *                     .tagsToAdd(Map.of("SnapshotCreator", "DLM"))
 *                     .copyTags(false)
 *                     .build())
 *                 .targetTags(Map.of("Snapshot", "true"))
 *                 .build())
 *             .build());
 *     }
 * }
 * ```
 * ### Example Cross-Region Snapshot Copy Usage
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.AwsFunctions;
 * import com.pulumi.aws.inputs.GetCallerIdentityArgs;
 * import com.pulumi.aws.iam.IamFunctions;
 * import com.pulumi.aws.iam.inputs.GetPolicyDocumentArgs;
 * import com.pulumi.aws.kms.Key;
 * import com.pulumi.aws.kms.KeyArgs;
 * import com.pulumi.aws.dlm.LifecyclePolicy;
 * import com.pulumi.aws.dlm.LifecyclePolicyArgs;
 * import com.pulumi.aws.dlm.inputs.LifecyclePolicyPolicyDetailsArgs;
 * import com.pulumi.resources.CustomResourceOptions;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         final var current = AwsFunctions.getCallerIdentity();
 *         final var key = IamFunctions.getPolicyDocument(GetPolicyDocumentArgs.builder()
 *             .statements(GetPolicyDocumentStatementArgs.builder()
 *                 .sid("Enable IAM User Permissions")
 *                 .effect("Allow")
 *                 .principals(GetPolicyDocumentStatementPrincipalArgs.builder()
 *                     .type("AWS")
 *                     .identifiers(String.format("arn:aws:iam::%s:root", current.applyValue(getCallerIdentityResult -> getCallerIdentityResult.accountId())))
 *                     .build())
 *                 .actions("kms:*")
 *                 .resources("*")
 *                 .build())
 *             .build());
 *         var dlmCrossRegionCopyCmk = new Key("dlmCrossRegionCopyCmk", KeyArgs.builder()
 *             .description("Example Alternate Region KMS Key")
 *             .policy(key.applyValue(getPolicyDocumentResult -> getPolicyDocumentResult.json()))
 *             .build(), CustomResourceOptions.builder()
 *                 .provider(aws.alternate())
 *                 .build());
 *         var example = new LifecyclePolicy("example", LifecyclePolicyArgs.builder()
 *             .description("example DLM lifecycle policy")
 *             .executionRoleArn(aws_iam_role.dlm_lifecycle_role().arn())
 *             .state("ENABLED")
 *             .policyDetails(LifecyclePolicyPolicyDetailsArgs.builder()
 *                 .resourceTypes("VOLUME")
 *                 .schedules(LifecyclePolicyPolicyDetailsScheduleArgs.builder()
 *                     .name("2 weeks of daily snapshots")
 *                     .createRule(LifecyclePolicyPolicyDetailsScheduleCreateRuleArgs.builder()
 *                         .interval(24)
 *                         .intervalUnit("HOURS")
 *                         .times("23:45")
 *                         .build())
 *                     .retainRule(LifecyclePolicyPolicyDetailsScheduleRetainRuleArgs.builder()
 *                         .count(14)
 *                         .build())
 *                     .tagsToAdd(Map.of("SnapshotCreator", "DLM"))
 *                     .copyTags(false)
 *                     .crossRegionCopyRules(LifecyclePolicyPolicyDetailsScheduleCrossRegionCopyRuleArgs.builder()
 *                         .target("us-west-2")
 *                         .encrypted(true)
 *                         .cmkArn(dlmCrossRegionCopyCmk.arn())
 *                         .copyTags(true)
 *                         .retainRule(LifecyclePolicyPolicyDetailsScheduleCrossRegionCopyRuleRetainRuleArgs.builder()
 *                             .interval(30)
 *                             .intervalUnit("DAYS")
 *                             .build())
 *                         .build())
 *                     .build())
 *                 .targetTags(Map.of("Snapshot", "true"))
 *                 .build())
 *             .build());
 *     }
 * }
 * ```
 * ### Example Event Based Policy Usage
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.AwsFunctions;
 * import com.pulumi.aws.inputs.GetCallerIdentityArgs;
 * import com.pulumi.aws.dlm.LifecyclePolicy;
 * import com.pulumi.aws.dlm.LifecyclePolicyArgs;
 * import com.pulumi.aws.dlm.inputs.LifecyclePolicyPolicyDetailsArgs;
 * import com.pulumi.aws.dlm.inputs.LifecyclePolicyPolicyDetailsActionArgs;
 * import com.pulumi.aws.dlm.inputs.LifecyclePolicyPolicyDetailsEventSourceArgs;
 * import com.pulumi.aws.dlm.inputs.LifecyclePolicyPolicyDetailsEventSourceParametersArgs;
 * import com.pulumi.aws.iam.IamFunctions;
 * import com.pulumi.aws.iam.inputs.GetPolicyArgs;
 * import com.pulumi.aws.iam.RolePolicyAttachment;
 * import com.pulumi.aws.iam.RolePolicyAttachmentArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         final var current = AwsFunctions.getCallerIdentity();
 *         var exampleLifecyclePolicy = new LifecyclePolicy("exampleLifecyclePolicy", LifecyclePolicyArgs.builder()
 *             .description("tf-acc-basic")
 *             .executionRoleArn(aws_iam_role.example().arn())
 *             .policyDetails(LifecyclePolicyPolicyDetailsArgs.builder()
 *                 .policyType("EVENT_BASED_POLICY")
 *                 .action(LifecyclePolicyPolicyDetailsActionArgs.builder()
 *                     .name("tf-acc-basic")
 *                     .crossRegionCopies(LifecyclePolicyPolicyDetailsActionCrossRegionCopyArgs.builder()
 *                         .encryptionConfiguration()
 *                         .retainRule(LifecyclePolicyPolicyDetailsActionCrossRegionCopyRetainRuleArgs.builder()
 *                             .interval(15)
 *                             .intervalUnit("MONTHS")
 *                             .build())
 *                         .target("us-east-1")
 *                         .build())
 *                     .build())
 *                 .eventSource(LifecyclePolicyPolicyDetailsEventSourceArgs.builder()
 *                     .type("MANAGED_CWE")
 *                     .parameters(LifecyclePolicyPolicyDetailsEventSourceParametersArgs.builder()
 *                         .descriptionRegex("^.*Created for policy: policy-1234567890abcdef0.*$")
 *                         .eventType("shareSnapshot")
 *                         .snapshotOwners(current.applyValue(getCallerIdentityResult -> getCallerIdentityResult.accountId()))
 *                         .build())
 *                     .build())
 *                 .build())
 *             .build());
 *         final var examplePolicy = IamFunctions.getPolicy(GetPolicyArgs.builder()
 *             .name("AWSDataLifecycleManagerServiceRole")
 *             .build());
 *         var exampleRolePolicyAttachment = new RolePolicyAttachment("exampleRolePolicyAttachment", RolePolicyAttachmentArgs.builder()
 *             .role(aws_iam_role.example().id())
 *             .policyArn(examplePolicy.applyValue(getPolicyResult -> getPolicyResult.arn()))
 *             .build());
 *     }
 * }
 * ```
 * ## Import
 * Using `pulumi import`, import DLM lifecycle policies using their policy ID. For example:
 * ```sh
 *  $ pulumi import aws:dlm/lifecyclePolicy:LifecyclePolicy example policy-abcdef12345678901
 * ```
 * @property description A description for the DLM lifecycle policy.
 * @property executionRoleArn The ARN of an IAM role that is able to be assumed by the DLM service.
 * @property policyDetails See the `policy_details` configuration block. Max of 1.
 * @property state Whether the lifecycle policy should be enabled or disabled. `ENABLED` or `DISABLED` are valid values. Defaults to `ENABLED`.
 * @property tags Key-value map of resource tags. If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
 * */
 */
public data class LifecyclePolicyArgs(
    public val description: Output<String>? = null,
    public val executionRoleArn: Output<String>? = null,
    public val policyDetails: Output<LifecyclePolicyPolicyDetailsArgs>? = null,
    public val state: Output<String>? = null,
    public val tags: Output<Map<String, String>>? = null,
) : ConvertibleToJava<com.pulumi.aws.dlm.LifecyclePolicyArgs> {
    override fun toJava(): com.pulumi.aws.dlm.LifecyclePolicyArgs =
        com.pulumi.aws.dlm.LifecyclePolicyArgs.builder()
            .description(description?.applyValue({ args0 -> args0 }))
            .executionRoleArn(executionRoleArn?.applyValue({ args0 -> args0 }))
            .policyDetails(policyDetails?.applyValue({ args0 -> args0.let({ args0 -> args0.toJava() }) }))
            .state(state?.applyValue({ args0 -> args0 }))
            .tags(
                tags?.applyValue({ args0 ->
                    args0.map({ args0 ->
                        args0.key.to(args0.value)
                    }).toMap()
                }),
            ).build()
}

/**
 * Builder for [LifecyclePolicyArgs].
 */
@PulumiTagMarker
public class LifecyclePolicyArgsBuilder internal constructor() {
    private var description: Output<String>? = null

    private var executionRoleArn: Output<String>? = null

    private var policyDetails: Output<LifecyclePolicyPolicyDetailsArgs>? = null

    private var state: Output<String>? = null

    private var tags: Output<Map<String, String>>? = null

    /**
     * @param value A description for the DLM lifecycle policy.
     */
    @JvmName("eodoxuggjqxdwyon")
    public suspend fun description(`value`: Output<String>) {
        this.description = value
    }

    /**
     * @param value The ARN of an IAM role that is able to be assumed by the DLM service.
     */
    @JvmName("oredfldmemgeutxt")
    public suspend fun executionRoleArn(`value`: Output<String>) {
        this.executionRoleArn = value
    }

    /**
     * @param value See the `policy_details` configuration block. Max of 1.
     */
    @JvmName("xsgpsapxmxtkbwjr")
    public suspend fun policyDetails(`value`: Output<LifecyclePolicyPolicyDetailsArgs>) {
        this.policyDetails = value
    }

    /**
     * @param value Whether the lifecycle policy should be enabled or disabled. `ENABLED` or `DISABLED` are valid values. Defaults to `ENABLED`.
     */
    @JvmName("xbutktvfgggnerbh")
    public suspend fun state(`value`: Output<String>) {
        this.state = value
    }

    /**
     * @param value Key-value map of resource tags. If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
     */
    @JvmName("anjfhghdsdcusuyl")
    public suspend fun tags(`value`: Output<Map<String, String>>) {
        this.tags = value
    }

    /**
     * @param value A description for the DLM lifecycle policy.
     */
    @JvmName("xkjcgcureewprrrb")
    public suspend fun description(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.description = mapped
    }

    /**
     * @param value The ARN of an IAM role that is able to be assumed by the DLM service.
     */
    @JvmName("kxjgaqoyrmgrrspl")
    public suspend fun executionRoleArn(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.executionRoleArn = mapped
    }

    /**
     * @param value See the `policy_details` configuration block. Max of 1.
     */
    @JvmName("jrhfinndulsadeed")
    public suspend fun policyDetails(`value`: LifecyclePolicyPolicyDetailsArgs?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.policyDetails = mapped
    }

    /**
     * @param argument See the `policy_details` configuration block. Max of 1.
     */
    @JvmName("gembkjoamhqvnakd")
    public suspend
    fun policyDetails(argument: suspend LifecyclePolicyPolicyDetailsArgsBuilder.() -> Unit) {
        val toBeMapped = LifecyclePolicyPolicyDetailsArgsBuilder().applySuspend { argument() }.build()
        val mapped = of(toBeMapped)
        this.policyDetails = mapped
    }

    /**
     * @param value Whether the lifecycle policy should be enabled or disabled. `ENABLED` or `DISABLED` are valid values. Defaults to `ENABLED`.
     */
    @JvmName("invatxlyplcgfhje")
    public suspend fun state(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.state = mapped
    }

    /**
     * @param value Key-value map of resource tags. If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
     */
    @JvmName("oaqoyetsqfyxohrq")
    public suspend fun tags(`value`: Map<String, String>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.tags = mapped
    }

    /**
     * @param values Key-value map of resource tags. If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
     */
    @JvmName("xysdvqooldsjaapi")
    public fun tags(vararg values: Pair<String, String>) {
        val toBeMapped = values.toMap()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.tags = mapped
    }

    internal fun build(): LifecyclePolicyArgs = LifecyclePolicyArgs(
        description = description,
        executionRoleArn = executionRoleArn,
        policyDetails = policyDetails,
        state = state,
        tags = tags,
    )
}
