@file:Suppress("NAME_SHADOWING", "DEPRECATION")

package com.pulumi.aws.dynamodb.kotlin

import com.pulumi.aws.dynamodb.TableArgs.builder
import com.pulumi.aws.dynamodb.kotlin.inputs.TableAttributeArgs
import com.pulumi.aws.dynamodb.kotlin.inputs.TableAttributeArgsBuilder
import com.pulumi.aws.dynamodb.kotlin.inputs.TableGlobalSecondaryIndexArgs
import com.pulumi.aws.dynamodb.kotlin.inputs.TableGlobalSecondaryIndexArgsBuilder
import com.pulumi.aws.dynamodb.kotlin.inputs.TableImportTableArgs
import com.pulumi.aws.dynamodb.kotlin.inputs.TableImportTableArgsBuilder
import com.pulumi.aws.dynamodb.kotlin.inputs.TableLocalSecondaryIndexArgs
import com.pulumi.aws.dynamodb.kotlin.inputs.TableLocalSecondaryIndexArgsBuilder
import com.pulumi.aws.dynamodb.kotlin.inputs.TablePointInTimeRecoveryArgs
import com.pulumi.aws.dynamodb.kotlin.inputs.TablePointInTimeRecoveryArgsBuilder
import com.pulumi.aws.dynamodb.kotlin.inputs.TableReplicaArgs
import com.pulumi.aws.dynamodb.kotlin.inputs.TableReplicaArgsBuilder
import com.pulumi.aws.dynamodb.kotlin.inputs.TableServerSideEncryptionArgs
import com.pulumi.aws.dynamodb.kotlin.inputs.TableServerSideEncryptionArgsBuilder
import com.pulumi.aws.dynamodb.kotlin.inputs.TableTtlArgs
import com.pulumi.aws.dynamodb.kotlin.inputs.TableTtlArgsBuilder
import com.pulumi.core.Output
import com.pulumi.core.Output.of
import com.pulumi.kotlin.ConvertibleToJava
import com.pulumi.kotlin.PulumiTagMarker
import com.pulumi.kotlin.applySuspend
import kotlin.Boolean
import kotlin.Int
import kotlin.Pair
import kotlin.String
import kotlin.Suppress
import kotlin.Unit
import kotlin.collections.List
import kotlin.collections.Map
import kotlin.jvm.JvmName

/**
 * Provides a DynamoDB table resource.
 * > **Note:** It is recommended to use [`ignoreChanges`](https://www.pulumi.com/docs/intro/concepts/programming-model/#ignorechanges) for `read_capacity` and/or `write_capacity` if there's `autoscaling policy` attached to the table.
 * > **Note:** When using aws.dynamodb.TableReplica with this resource, use `lifecycle` `ignore_changes` for `replica`, _e.g._, `lifecycle { ignore_changes = [replica] }`.
 * ## DynamoDB Table attributes
 * Only define attributes on the table object that are going to be used as:
 * * Table hash key or range key
 * * LSI or GSI hash key or range key
 * The DynamoDB API expects attribute structure (name and type) to be passed along when creating or updating GSI/LSIs or creating the initial table. In these cases it expects the Hash / Range keys to be provided. Because these get re-used in numerous places (i.e the table's range key could be a part of one or more GSIs), they are stored on the table object to prevent duplication and increase consistency. If you add attributes here that are not used in these scenarios it can cause an infinite loop in planning.
 * ## Example Usage
 * ### Basic Example
 * The following dynamodb table description models the table and GSI shown in the [AWS SDK example documentation](https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/GSI.html)
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.dynamodb.Table;
 * import com.pulumi.aws.dynamodb.TableArgs;
 * import com.pulumi.aws.dynamodb.inputs.TableAttributeArgs;
 * import com.pulumi.aws.dynamodb.inputs.TableGlobalSecondaryIndexArgs;
 * import com.pulumi.aws.dynamodb.inputs.TableTtlArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var basic_dynamodb_table = new Table("basic-dynamodb-table", TableArgs.builder()
 *             .attributes(
 *                 TableAttributeArgs.builder()
 *                     .name("UserId")
 *                     .type("S")
 *                     .build(),
 *                 TableAttributeArgs.builder()
 *                     .name("GameTitle")
 *                     .type("S")
 *                     .build(),
 *                 TableAttributeArgs.builder()
 *                     .name("TopScore")
 *                     .type("N")
 *                     .build())
 *             .billingMode("PROVISIONED")
 *             .globalSecondaryIndexes(TableGlobalSecondaryIndexArgs.builder()
 *                 .hashKey("GameTitle")
 *                 .name("GameTitleIndex")
 *                 .nonKeyAttributes("UserId")
 *                 .projectionType("INCLUDE")
 *                 .rangeKey("TopScore")
 *                 .readCapacity(10)
 *                 .writeCapacity(10)
 *                 .build())
 *             .hashKey("UserId")
 *             .rangeKey("GameTitle")
 *             .readCapacity(20)
 *             .tags(Map.ofEntries(
 *                 Map.entry("Environment", "production"),
 *                 Map.entry("Name", "dynamodb-table-1")
 *             ))
 *             .ttl(TableTtlArgs.builder()
 *                 .attributeName("TimeToExist")
 *                 .enabled(false)
 *                 .build())
 *             .writeCapacity(20)
 *             .build());
 *     }
 * }
 * ```
 * ### Global Tables
 * This resource implements support for [DynamoDB Global Tables V2 (version 2019.11.21)](https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/globaltables.V2.html) via `replica` configuration blocks. For working with [DynamoDB Global Tables V1 (version 2017.11.29)](https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/globaltables.V1.html), see the `aws.dynamodb.GlobalTable` resource.
 * > **Note:** aws.dynamodb.TableReplica is an alternate way of configuring Global Tables. Do not use `replica` configuration blocks of `aws.dynamodb.Table` together with aws_dynamodb_table_replica.
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.dynamodb.Table;
 * import com.pulumi.aws.dynamodb.TableArgs;
 * import com.pulumi.aws.dynamodb.inputs.TableAttributeArgs;
 * import com.pulumi.aws.dynamodb.inputs.TableReplicaArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var example = new Table("example", TableArgs.builder()
 *             .attributes(TableAttributeArgs.builder()
 *                 .name("TestTableHashKey")
 *                 .type("S")
 *                 .build())
 *             .billingMode("PAY_PER_REQUEST")
 *             .hashKey("TestTableHashKey")
 *             .replicas(
 *                 TableReplicaArgs.builder()
 *                     .regionName("us-east-2")
 *                     .build(),
 *                 TableReplicaArgs.builder()
 *                     .regionName("us-west-2")
 *                     .build())
 *             .streamEnabled(true)
 *             .streamViewType("NEW_AND_OLD_IMAGES")
 *             .build());
 *     }
 * }
 * ```
 * ## Import
 * Using `pulumi import`, import DynamoDB tables using the `name`. For example:
 * ```sh
 *  $ pulumi import aws:dynamodb/table:Table basic-dynamodb-table GameScores
 * ```
 * @property attributes Set of nested attribute definitions. Only required for `hash_key` and `range_key` attributes. See below.
 * @property billingMode Controls how you are charged for read and write throughput and how you manage capacity. The valid values are `PROVISIONED` and `PAY_PER_REQUEST`. Defaults to `PROVISIONED`.
 * @property deletionProtectionEnabled Enables deletion protection for table. Defaults to `false`.
 * @property globalSecondaryIndexes Describe a GSI for the table; subject to the normal limits on the number of GSIs, projected attributes, etc. See below.
 * @property hashKey Attribute to use as the hash (partition) key. Must also be defined as an `attribute`. See below.
 * @property importTable Import Amazon S3 data into a new table. See below.
 * @property localSecondaryIndexes Describe an LSI on the table; these can only be allocated _at creation_ so you cannot change this definition after you have created the resource. See below.
 * @property name Unique within a region name of the table.
 * Optional arguments:
 * @property pointInTimeRecovery Enable point-in-time recovery options. See below.
 * @property rangeKey Attribute to use as the range (sort) key. Must also be defined as an `attribute`, see below.
 * @property readCapacity Number of read units for this table. If the `billing_mode` is `PROVISIONED`, this field is required.
 * @property replicas Configuration block(s) with [DynamoDB Global Tables V2 (version 2019&#46;11&#46;21)](https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/globaltables.V2.html) replication configurations. See below.
 * @property restoreDateTime Time of the point-in-time recovery point to restore.
 * @property restoreSourceName Name of the table to restore. Must match the name of an existing table.
 * @property restoreToLatestTime If set, restores table to the most recent point-in-time recovery point.
 * @property serverSideEncryption Encryption at rest options. AWS DynamoDB tables are automatically encrypted at rest with an AWS-owned Customer Master Key if this argument isn't specified. See below.
 * @property streamEnabled Whether Streams are enabled.
 * @property streamViewType When an item in the table is modified, StreamViewType determines what information is written to the table's stream. Valid values are `KEYS_ONLY`, `NEW_IMAGE`, `OLD_IMAGE`, `NEW_AND_OLD_IMAGES`.
 * @property tableClass Storage class of the table.
 * Valid values are `STANDARD` and `STANDARD_INFREQUENT_ACCESS`.
 * Default value is `STANDARD`.
 * @property tags A map of tags to populate on the created table. If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
 * @property ttl Configuration block for TTL. See below.
 * @property writeCapacity Number of write units for this table. If the `billing_mode` is `PROVISIONED`, this field is required.
 */
public data class TableArgs(
    public val attributes: Output<List<TableAttributeArgs>>? = null,
    public val billingMode: Output<String>? = null,
    public val deletionProtectionEnabled: Output<Boolean>? = null,
    public val globalSecondaryIndexes: Output<List<TableGlobalSecondaryIndexArgs>>? = null,
    public val hashKey: Output<String>? = null,
    public val importTable: Output<TableImportTableArgs>? = null,
    public val localSecondaryIndexes: Output<List<TableLocalSecondaryIndexArgs>>? = null,
    public val name: Output<String>? = null,
    public val pointInTimeRecovery: Output<TablePointInTimeRecoveryArgs>? = null,
    public val rangeKey: Output<String>? = null,
    public val readCapacity: Output<Int>? = null,
    public val replicas: Output<List<TableReplicaArgs>>? = null,
    public val restoreDateTime: Output<String>? = null,
    public val restoreSourceName: Output<String>? = null,
    public val restoreToLatestTime: Output<Boolean>? = null,
    public val serverSideEncryption: Output<TableServerSideEncryptionArgs>? = null,
    public val streamEnabled: Output<Boolean>? = null,
    public val streamViewType: Output<String>? = null,
    public val tableClass: Output<String>? = null,
    public val tags: Output<Map<String, String>>? = null,
    public val ttl: Output<TableTtlArgs>? = null,
    public val writeCapacity: Output<Int>? = null,
) : ConvertibleToJava<com.pulumi.aws.dynamodb.TableArgs> {
    override fun toJava(): com.pulumi.aws.dynamodb.TableArgs =
        com.pulumi.aws.dynamodb.TableArgs.builder()
            .attributes(
                attributes?.applyValue({ args0 ->
                    args0.map({ args0 ->
                        args0.let({ args0 ->
                            args0.toJava()
                        })
                    })
                }),
            )
            .billingMode(billingMode?.applyValue({ args0 -> args0 }))
            .deletionProtectionEnabled(deletionProtectionEnabled?.applyValue({ args0 -> args0 }))
            .globalSecondaryIndexes(
                globalSecondaryIndexes?.applyValue({ args0 ->
                    args0.map({ args0 ->
                        args0.let({ args0 -> args0.toJava() })
                    })
                }),
            )
            .hashKey(hashKey?.applyValue({ args0 -> args0 }))
            .importTable(importTable?.applyValue({ args0 -> args0.let({ args0 -> args0.toJava() }) }))
            .localSecondaryIndexes(
                localSecondaryIndexes?.applyValue({ args0 ->
                    args0.map({ args0 ->
                        args0.let({ args0 -> args0.toJava() })
                    })
                }),
            )
            .name(name?.applyValue({ args0 -> args0 }))
            .pointInTimeRecovery(
                pointInTimeRecovery?.applyValue({ args0 ->
                    args0.let({ args0 ->
                        args0.toJava()
                    })
                }),
            )
            .rangeKey(rangeKey?.applyValue({ args0 -> args0 }))
            .readCapacity(readCapacity?.applyValue({ args0 -> args0 }))
            .replicas(
                replicas?.applyValue({ args0 ->
                    args0.map({ args0 ->
                        args0.let({ args0 ->
                            args0.toJava()
                        })
                    })
                }),
            )
            .restoreDateTime(restoreDateTime?.applyValue({ args0 -> args0 }))
            .restoreSourceName(restoreSourceName?.applyValue({ args0 -> args0 }))
            .restoreToLatestTime(restoreToLatestTime?.applyValue({ args0 -> args0 }))
            .serverSideEncryption(
                serverSideEncryption?.applyValue({ args0 ->
                    args0.let({ args0 ->
                        args0.toJava()
                    })
                }),
            )
            .streamEnabled(streamEnabled?.applyValue({ args0 -> args0 }))
            .streamViewType(streamViewType?.applyValue({ args0 -> args0 }))
            .tableClass(tableClass?.applyValue({ args0 -> args0 }))
            .tags(tags?.applyValue({ args0 -> args0.map({ args0 -> args0.key.to(args0.value) }).toMap() }))
            .ttl(ttl?.applyValue({ args0 -> args0.let({ args0 -> args0.toJava() }) }))
            .writeCapacity(writeCapacity?.applyValue({ args0 -> args0 })).build()
}

/**
 * Builder for [TableArgs].
 */
@PulumiTagMarker
public class TableArgsBuilder internal constructor() {
    private var attributes: Output<List<TableAttributeArgs>>? = null

    private var billingMode: Output<String>? = null

    private var deletionProtectionEnabled: Output<Boolean>? = null

    private var globalSecondaryIndexes: Output<List<TableGlobalSecondaryIndexArgs>>? = null

    private var hashKey: Output<String>? = null

    private var importTable: Output<TableImportTableArgs>? = null

    private var localSecondaryIndexes: Output<List<TableLocalSecondaryIndexArgs>>? = null

    private var name: Output<String>? = null

    private var pointInTimeRecovery: Output<TablePointInTimeRecoveryArgs>? = null

    private var rangeKey: Output<String>? = null

    private var readCapacity: Output<Int>? = null

    private var replicas: Output<List<TableReplicaArgs>>? = null

    private var restoreDateTime: Output<String>? = null

    private var restoreSourceName: Output<String>? = null

    private var restoreToLatestTime: Output<Boolean>? = null

    private var serverSideEncryption: Output<TableServerSideEncryptionArgs>? = null

    private var streamEnabled: Output<Boolean>? = null

    private var streamViewType: Output<String>? = null

    private var tableClass: Output<String>? = null

    private var tags: Output<Map<String, String>>? = null

    private var ttl: Output<TableTtlArgs>? = null

    private var writeCapacity: Output<Int>? = null

    /**
     * @param value Set of nested attribute definitions. Only required for `hash_key` and `range_key` attributes. See below.
     */
    @JvmName("xjvkawhvtdafsyvv")
    public suspend fun attributes(`value`: Output<List<TableAttributeArgs>>) {
        this.attributes = value
    }

    @JvmName("bkmiweybbotohuyl")
    public suspend fun attributes(vararg values: Output<TableAttributeArgs>) {
        this.attributes = Output.all(values.asList())
    }

    /**
     * @param values Set of nested attribute definitions. Only required for `hash_key` and `range_key` attributes. See below.
     */
    @JvmName("lbpfjprrkjyeyrnc")
    public suspend fun attributes(values: List<Output<TableAttributeArgs>>) {
        this.attributes = Output.all(values)
    }

    /**
     * @param value Controls how you are charged for read and write throughput and how you manage capacity. The valid values are `PROVISIONED` and `PAY_PER_REQUEST`. Defaults to `PROVISIONED`.
     */
    @JvmName("thhbiprsfunjxrru")
    public suspend fun billingMode(`value`: Output<String>) {
        this.billingMode = value
    }

    /**
     * @param value Enables deletion protection for table. Defaults to `false`.
     */
    @JvmName("ocbpnfctabktplyt")
    public suspend fun deletionProtectionEnabled(`value`: Output<Boolean>) {
        this.deletionProtectionEnabled = value
    }

    /**
     * @param value Describe a GSI for the table; subject to the normal limits on the number of GSIs, projected attributes, etc. See below.
     */
    @JvmName("xliosscnwfyvnyls")
    public suspend fun globalSecondaryIndexes(`value`: Output<List<TableGlobalSecondaryIndexArgs>>) {
        this.globalSecondaryIndexes = value
    }

    @JvmName("mepjjxrejdrwuwbe")
    public suspend fun globalSecondaryIndexes(vararg values: Output<TableGlobalSecondaryIndexArgs>) {
        this.globalSecondaryIndexes = Output.all(values.asList())
    }

    /**
     * @param values Describe a GSI for the table; subject to the normal limits on the number of GSIs, projected attributes, etc. See below.
     */
    @JvmName("irtrdroiigreohkr")
    public suspend fun globalSecondaryIndexes(values: List<Output<TableGlobalSecondaryIndexArgs>>) {
        this.globalSecondaryIndexes = Output.all(values)
    }

    /**
     * @param value Attribute to use as the hash (partition) key. Must also be defined as an `attribute`. See below.
     */
    @JvmName("ukfbgambmmmpmthe")
    public suspend fun hashKey(`value`: Output<String>) {
        this.hashKey = value
    }

    /**
     * @param value Import Amazon S3 data into a new table. See below.
     */
    @JvmName("oqgdowfarlfpmpis")
    public suspend fun importTable(`value`: Output<TableImportTableArgs>) {
        this.importTable = value
    }

    /**
     * @param value Describe an LSI on the table; these can only be allocated _at creation_ so you cannot change this definition after you have created the resource. See below.
     */
    @JvmName("jqvimboanmlbnrap")
    public suspend fun localSecondaryIndexes(`value`: Output<List<TableLocalSecondaryIndexArgs>>) {
        this.localSecondaryIndexes = value
    }

    @JvmName("qrndsivsqkkkoufe")
    public suspend fun localSecondaryIndexes(vararg values: Output<TableLocalSecondaryIndexArgs>) {
        this.localSecondaryIndexes = Output.all(values.asList())
    }

    /**
     * @param values Describe an LSI on the table; these can only be allocated _at creation_ so you cannot change this definition after you have created the resource. See below.
     */
    @JvmName("pxtbcifovjkwpsfb")
    public suspend fun localSecondaryIndexes(values: List<Output<TableLocalSecondaryIndexArgs>>) {
        this.localSecondaryIndexes = Output.all(values)
    }

    /**
     * @param value Unique within a region name of the table.
     * Optional arguments:
     */
    @JvmName("mattllodxtmunsio")
    public suspend fun name(`value`: Output<String>) {
        this.name = value
    }

    /**
     * @param value Enable point-in-time recovery options. See below.
     */
    @JvmName("spyifsepxxvhhspt")
    public suspend fun pointInTimeRecovery(`value`: Output<TablePointInTimeRecoveryArgs>) {
        this.pointInTimeRecovery = value
    }

    /**
     * @param value Attribute to use as the range (sort) key. Must also be defined as an `attribute`, see below.
     */
    @JvmName("kumqskqktixmiiop")
    public suspend fun rangeKey(`value`: Output<String>) {
        this.rangeKey = value
    }

    /**
     * @param value Number of read units for this table. If the `billing_mode` is `PROVISIONED`, this field is required.
     */
    @JvmName("xghumltuvdyaucnu")
    public suspend fun readCapacity(`value`: Output<Int>) {
        this.readCapacity = value
    }

    /**
     * @param value Configuration block(s) with [DynamoDB Global Tables V2 (version 2019&#46;11&#46;21)](https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/globaltables.V2.html) replication configurations. See below.
     */
    @JvmName("vvcbnptaffmtgdwm")
    public suspend fun replicas(`value`: Output<List<TableReplicaArgs>>) {
        this.replicas = value
    }

    @JvmName("eoikfdgfwmixssna")
    public suspend fun replicas(vararg values: Output<TableReplicaArgs>) {
        this.replicas = Output.all(values.asList())
    }

    /**
     * @param values Configuration block(s) with [DynamoDB Global Tables V2 (version 2019&#46;11&#46;21)](https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/globaltables.V2.html) replication configurations. See below.
     */
    @JvmName("estesvhfbanpqhyc")
    public suspend fun replicas(values: List<Output<TableReplicaArgs>>) {
        this.replicas = Output.all(values)
    }

    /**
     * @param value Time of the point-in-time recovery point to restore.
     */
    @JvmName("vnoriqgptbnokoyx")
    public suspend fun restoreDateTime(`value`: Output<String>) {
        this.restoreDateTime = value
    }

    /**
     * @param value Name of the table to restore. Must match the name of an existing table.
     */
    @JvmName("dguqbkeknghywjln")
    public suspend fun restoreSourceName(`value`: Output<String>) {
        this.restoreSourceName = value
    }

    /**
     * @param value If set, restores table to the most recent point-in-time recovery point.
     */
    @JvmName("gomewrsdbxsrrqjh")
    public suspend fun restoreToLatestTime(`value`: Output<Boolean>) {
        this.restoreToLatestTime = value
    }

    /**
     * @param value Encryption at rest options. AWS DynamoDB tables are automatically encrypted at rest with an AWS-owned Customer Master Key if this argument isn't specified. See below.
     */
    @JvmName("donxwxjddwlgjris")
    public suspend fun serverSideEncryption(`value`: Output<TableServerSideEncryptionArgs>) {
        this.serverSideEncryption = value
    }

    /**
     * @param value Whether Streams are enabled.
     */
    @JvmName("ojvyimecmhynmvrh")
    public suspend fun streamEnabled(`value`: Output<Boolean>) {
        this.streamEnabled = value
    }

    /**
     * @param value When an item in the table is modified, StreamViewType determines what information is written to the table's stream. Valid values are `KEYS_ONLY`, `NEW_IMAGE`, `OLD_IMAGE`, `NEW_AND_OLD_IMAGES`.
     */
    @JvmName("toqlbtqbhplorsky")
    public suspend fun streamViewType(`value`: Output<String>) {
        this.streamViewType = value
    }

    /**
     * @param value Storage class of the table.
     * Valid values are `STANDARD` and `STANDARD_INFREQUENT_ACCESS`.
     * Default value is `STANDARD`.
     */
    @JvmName("jyrqiljlovviyipe")
    public suspend fun tableClass(`value`: Output<String>) {
        this.tableClass = value
    }

    /**
     * @param value A map of tags to populate on the created table. If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
     */
    @JvmName("vofrcimksvkfoxtx")
    public suspend fun tags(`value`: Output<Map<String, String>>) {
        this.tags = value
    }

    /**
     * @param value Configuration block for TTL. See below.
     */
    @JvmName("gaqrgwnluxdlnejb")
    public suspend fun ttl(`value`: Output<TableTtlArgs>) {
        this.ttl = value
    }

    /**
     * @param value Number of write units for this table. If the `billing_mode` is `PROVISIONED`, this field is required.
     */
    @JvmName("ydwfkdxtmpljejlp")
    public suspend fun writeCapacity(`value`: Output<Int>) {
        this.writeCapacity = value
    }

    /**
     * @param value Set of nested attribute definitions. Only required for `hash_key` and `range_key` attributes. See below.
     */
    @JvmName("bbeikvxvpavkmkhw")
    public suspend fun attributes(`value`: List<TableAttributeArgs>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.attributes = mapped
    }

    /**
     * @param argument Set of nested attribute definitions. Only required for `hash_key` and `range_key` attributes. See below.
     */
    @JvmName("sbxmdygnkxklnbhk")
    public suspend fun attributes(argument: List<suspend TableAttributeArgsBuilder.() -> Unit>) {
        val toBeMapped = argument.toList().map {
            TableAttributeArgsBuilder().applySuspend {
                it()
            }.build()
        }
        val mapped = of(toBeMapped)
        this.attributes = mapped
    }

    /**
     * @param argument Set of nested attribute definitions. Only required for `hash_key` and `range_key` attributes. See below.
     */
    @JvmName("gtrjialnytfvnhmm")
    public suspend fun attributes(vararg argument: suspend TableAttributeArgsBuilder.() -> Unit) {
        val toBeMapped = argument.toList().map {
            TableAttributeArgsBuilder().applySuspend {
                it()
            }.build()
        }
        val mapped = of(toBeMapped)
        this.attributes = mapped
    }

    /**
     * @param argument Set of nested attribute definitions. Only required for `hash_key` and `range_key` attributes. See below.
     */
    @JvmName("pjfnjnxgbkbaqrjf")
    public suspend fun attributes(argument: suspend TableAttributeArgsBuilder.() -> Unit) {
        val toBeMapped = listOf(TableAttributeArgsBuilder().applySuspend { argument() }.build())
        val mapped = of(toBeMapped)
        this.attributes = mapped
    }

    /**
     * @param values Set of nested attribute definitions. Only required for `hash_key` and `range_key` attributes. See below.
     */
    @JvmName("riqjovhrfmukbggx")
    public suspend fun attributes(vararg values: TableAttributeArgs) {
        val toBeMapped = values.toList()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.attributes = mapped
    }

    /**
     * @param value Controls how you are charged for read and write throughput and how you manage capacity. The valid values are `PROVISIONED` and `PAY_PER_REQUEST`. Defaults to `PROVISIONED`.
     */
    @JvmName("qjqplvbnkcaqgiwi")
    public suspend fun billingMode(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.billingMode = mapped
    }

    /**
     * @param value Enables deletion protection for table. Defaults to `false`.
     */
    @JvmName("ojitodmljwwiaoda")
    public suspend fun deletionProtectionEnabled(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.deletionProtectionEnabled = mapped
    }

    /**
     * @param value Describe a GSI for the table; subject to the normal limits on the number of GSIs, projected attributes, etc. See below.
     */
    @JvmName("ntfkflsrxglhnobk")
    public suspend fun globalSecondaryIndexes(`value`: List<TableGlobalSecondaryIndexArgs>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.globalSecondaryIndexes = mapped
    }

    /**
     * @param argument Describe a GSI for the table; subject to the normal limits on the number of GSIs, projected attributes, etc. See below.
     */
    @JvmName("dyhxrrebcowgikoy")
    public suspend
    fun globalSecondaryIndexes(argument: List<suspend TableGlobalSecondaryIndexArgsBuilder.() -> Unit>) {
        val toBeMapped = argument.toList().map {
            TableGlobalSecondaryIndexArgsBuilder().applySuspend {
                it()
            }.build()
        }
        val mapped = of(toBeMapped)
        this.globalSecondaryIndexes = mapped
    }

    /**
     * @param argument Describe a GSI for the table; subject to the normal limits on the number of GSIs, projected attributes, etc. See below.
     */
    @JvmName("wqolkojkwhdchscc")
    public suspend fun globalSecondaryIndexes(
        vararg
        argument: suspend TableGlobalSecondaryIndexArgsBuilder.() -> Unit,
    ) {
        val toBeMapped = argument.toList().map {
            TableGlobalSecondaryIndexArgsBuilder().applySuspend {
                it()
            }.build()
        }
        val mapped = of(toBeMapped)
        this.globalSecondaryIndexes = mapped
    }

    /**
     * @param argument Describe a GSI for the table; subject to the normal limits on the number of GSIs, projected attributes, etc. See below.
     */
    @JvmName("ucabsarifwarsffc")
    public suspend
    fun globalSecondaryIndexes(argument: suspend TableGlobalSecondaryIndexArgsBuilder.() -> Unit) {
        val toBeMapped = listOf(
            TableGlobalSecondaryIndexArgsBuilder().applySuspend {
                argument()
            }.build(),
        )
        val mapped = of(toBeMapped)
        this.globalSecondaryIndexes = mapped
    }

    /**
     * @param values Describe a GSI for the table; subject to the normal limits on the number of GSIs, projected attributes, etc. See below.
     */
    @JvmName("myjkfdbimhsduqgs")
    public suspend fun globalSecondaryIndexes(vararg values: TableGlobalSecondaryIndexArgs) {
        val toBeMapped = values.toList()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.globalSecondaryIndexes = mapped
    }

    /**
     * @param value Attribute to use as the hash (partition) key. Must also be defined as an `attribute`. See below.
     */
    @JvmName("lbynaudwevwkfglg")
    public suspend fun hashKey(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.hashKey = mapped
    }

    /**
     * @param value Import Amazon S3 data into a new table. See below.
     */
    @JvmName("qtbwefxfgvqimrno")
    public suspend fun importTable(`value`: TableImportTableArgs?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.importTable = mapped
    }

    /**
     * @param argument Import Amazon S3 data into a new table. See below.
     */
    @JvmName("hwmumqxtwawvojbe")
    public suspend fun importTable(argument: suspend TableImportTableArgsBuilder.() -> Unit) {
        val toBeMapped = TableImportTableArgsBuilder().applySuspend { argument() }.build()
        val mapped = of(toBeMapped)
        this.importTable = mapped
    }

    /**
     * @param value Describe an LSI on the table; these can only be allocated _at creation_ so you cannot change this definition after you have created the resource. See below.
     */
    @JvmName("werxlpyvnorrlbxt")
    public suspend fun localSecondaryIndexes(`value`: List<TableLocalSecondaryIndexArgs>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.localSecondaryIndexes = mapped
    }

    /**
     * @param argument Describe an LSI on the table; these can only be allocated _at creation_ so you cannot change this definition after you have created the resource. See below.
     */
    @JvmName("aokfxvrllsyqrrck")
    public suspend
    fun localSecondaryIndexes(argument: List<suspend TableLocalSecondaryIndexArgsBuilder.() -> Unit>) {
        val toBeMapped = argument.toList().map {
            TableLocalSecondaryIndexArgsBuilder().applySuspend {
                it()
            }.build()
        }
        val mapped = of(toBeMapped)
        this.localSecondaryIndexes = mapped
    }

    /**
     * @param argument Describe an LSI on the table; these can only be allocated _at creation_ so you cannot change this definition after you have created the resource. See below.
     */
    @JvmName("lugscisberkfsvpd")
    public suspend fun localSecondaryIndexes(
        vararg
        argument: suspend TableLocalSecondaryIndexArgsBuilder.() -> Unit,
    ) {
        val toBeMapped = argument.toList().map {
            TableLocalSecondaryIndexArgsBuilder().applySuspend {
                it()
            }.build()
        }
        val mapped = of(toBeMapped)
        this.localSecondaryIndexes = mapped
    }

    /**
     * @param argument Describe an LSI on the table; these can only be allocated _at creation_ so you cannot change this definition after you have created the resource. See below.
     */
    @JvmName("bvqksydhmupcfjfc")
    public suspend
    fun localSecondaryIndexes(argument: suspend TableLocalSecondaryIndexArgsBuilder.() -> Unit) {
        val toBeMapped = listOf(
            TableLocalSecondaryIndexArgsBuilder().applySuspend {
                argument()
            }.build(),
        )
        val mapped = of(toBeMapped)
        this.localSecondaryIndexes = mapped
    }

    /**
     * @param values Describe an LSI on the table; these can only be allocated _at creation_ so you cannot change this definition after you have created the resource. See below.
     */
    @JvmName("kdntyqeskersvrpc")
    public suspend fun localSecondaryIndexes(vararg values: TableLocalSecondaryIndexArgs) {
        val toBeMapped = values.toList()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.localSecondaryIndexes = mapped
    }

    /**
     * @param value Unique within a region name of the table.
     * Optional arguments:
     */
    @JvmName("ldudgedacqoccmju")
    public suspend fun name(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.name = mapped
    }

    /**
     * @param value Enable point-in-time recovery options. See below.
     */
    @JvmName("vbhjixbbrnxtyonk")
    public suspend fun pointInTimeRecovery(`value`: TablePointInTimeRecoveryArgs?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.pointInTimeRecovery = mapped
    }

    /**
     * @param argument Enable point-in-time recovery options. See below.
     */
    @JvmName("caleujvwsdwouwyt")
    public suspend
    fun pointInTimeRecovery(argument: suspend TablePointInTimeRecoveryArgsBuilder.() -> Unit) {
        val toBeMapped = TablePointInTimeRecoveryArgsBuilder().applySuspend { argument() }.build()
        val mapped = of(toBeMapped)
        this.pointInTimeRecovery = mapped
    }

    /**
     * @param value Attribute to use as the range (sort) key. Must also be defined as an `attribute`, see below.
     */
    @JvmName("hpnhblhqnqdljgvq")
    public suspend fun rangeKey(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.rangeKey = mapped
    }

    /**
     * @param value Number of read units for this table. If the `billing_mode` is `PROVISIONED`, this field is required.
     */
    @JvmName("gtsbsqifpuwrniip")
    public suspend fun readCapacity(`value`: Int?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.readCapacity = mapped
    }

    /**
     * @param value Configuration block(s) with [DynamoDB Global Tables V2 (version 2019&#46;11&#46;21)](https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/globaltables.V2.html) replication configurations. See below.
     */
    @JvmName("jraofxgglhatlnuv")
    public suspend fun replicas(`value`: List<TableReplicaArgs>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.replicas = mapped
    }

    /**
     * @param argument Configuration block(s) with [DynamoDB Global Tables V2 (version 2019&#46;11&#46;21)](https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/globaltables.V2.html) replication configurations. See below.
     */
    @JvmName("ctaeurfwmbnuehyp")
    public suspend fun replicas(argument: List<suspend TableReplicaArgsBuilder.() -> Unit>) {
        val toBeMapped = argument.toList().map {
            TableReplicaArgsBuilder().applySuspend { it() }.build()
        }
        val mapped = of(toBeMapped)
        this.replicas = mapped
    }

    /**
     * @param argument Configuration block(s) with [DynamoDB Global Tables V2 (version 2019&#46;11&#46;21)](https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/globaltables.V2.html) replication configurations. See below.
     */
    @JvmName("bgjwhfslaqmuqroe")
    public suspend fun replicas(vararg argument: suspend TableReplicaArgsBuilder.() -> Unit) {
        val toBeMapped = argument.toList().map {
            TableReplicaArgsBuilder().applySuspend { it() }.build()
        }
        val mapped = of(toBeMapped)
        this.replicas = mapped
    }

    /**
     * @param argument Configuration block(s) with [DynamoDB Global Tables V2 (version 2019&#46;11&#46;21)](https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/globaltables.V2.html) replication configurations. See below.
     */
    @JvmName("pcosoyslosgtihtt")
    public suspend fun replicas(argument: suspend TableReplicaArgsBuilder.() -> Unit) {
        val toBeMapped = listOf(TableReplicaArgsBuilder().applySuspend { argument() }.build())
        val mapped = of(toBeMapped)
        this.replicas = mapped
    }

    /**
     * @param values Configuration block(s) with [DynamoDB Global Tables V2 (version 2019&#46;11&#46;21)](https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/globaltables.V2.html) replication configurations. See below.
     */
    @JvmName("bhlsjfadspnhtcip")
    public suspend fun replicas(vararg values: TableReplicaArgs) {
        val toBeMapped = values.toList()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.replicas = mapped
    }

    /**
     * @param value Time of the point-in-time recovery point to restore.
     */
    @JvmName("dpdjqdoiwdhwblkh")
    public suspend fun restoreDateTime(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.restoreDateTime = mapped
    }

    /**
     * @param value Name of the table to restore. Must match the name of an existing table.
     */
    @JvmName("lanoujwpotoafujg")
    public suspend fun restoreSourceName(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.restoreSourceName = mapped
    }

    /**
     * @param value If set, restores table to the most recent point-in-time recovery point.
     */
    @JvmName("rwgsdlhfjwvwlibj")
    public suspend fun restoreToLatestTime(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.restoreToLatestTime = mapped
    }

    /**
     * @param value Encryption at rest options. AWS DynamoDB tables are automatically encrypted at rest with an AWS-owned Customer Master Key if this argument isn't specified. See below.
     */
    @JvmName("enemautmkhhovqic")
    public suspend fun serverSideEncryption(`value`: TableServerSideEncryptionArgs?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.serverSideEncryption = mapped
    }

    /**
     * @param argument Encryption at rest options. AWS DynamoDB tables are automatically encrypted at rest with an AWS-owned Customer Master Key if this argument isn't specified. See below.
     */
    @JvmName("ulcijowcqgfgkboj")
    public suspend
    fun serverSideEncryption(argument: suspend TableServerSideEncryptionArgsBuilder.() -> Unit) {
        val toBeMapped = TableServerSideEncryptionArgsBuilder().applySuspend { argument() }.build()
        val mapped = of(toBeMapped)
        this.serverSideEncryption = mapped
    }

    /**
     * @param value Whether Streams are enabled.
     */
    @JvmName("lkgnhcldtxjhkgvj")
    public suspend fun streamEnabled(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.streamEnabled = mapped
    }

    /**
     * @param value When an item in the table is modified, StreamViewType determines what information is written to the table's stream. Valid values are `KEYS_ONLY`, `NEW_IMAGE`, `OLD_IMAGE`, `NEW_AND_OLD_IMAGES`.
     */
    @JvmName("nrjdxvqdaqjmcngn")
    public suspend fun streamViewType(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.streamViewType = mapped
    }

    /**
     * @param value Storage class of the table.
     * Valid values are `STANDARD` and `STANDARD_INFREQUENT_ACCESS`.
     * Default value is `STANDARD`.
     */
    @JvmName("vspkrmamspuxfjxw")
    public suspend fun tableClass(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.tableClass = mapped
    }

    /**
     * @param value A map of tags to populate on the created table. If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
     */
    @JvmName("qiqmmkwcistjxxsn")
    public suspend fun tags(`value`: Map<String, String>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.tags = mapped
    }

    /**
     * @param values A map of tags to populate on the created table. If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
     */
    @JvmName("yasqfvjhqdpiudyg")
    public fun tags(vararg values: Pair<String, String>) {
        val toBeMapped = values.toMap()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.tags = mapped
    }

    /**
     * @param value Configuration block for TTL. See below.
     */
    @JvmName("npgnddvifesogbjw")
    public suspend fun ttl(`value`: TableTtlArgs?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.ttl = mapped
    }

    /**
     * @param argument Configuration block for TTL. See below.
     */
    @JvmName("dihfmsmkfihogovi")
    public suspend fun ttl(argument: suspend TableTtlArgsBuilder.() -> Unit) {
        val toBeMapped = TableTtlArgsBuilder().applySuspend { argument() }.build()
        val mapped = of(toBeMapped)
        this.ttl = mapped
    }

    /**
     * @param value Number of write units for this table. If the `billing_mode` is `PROVISIONED`, this field is required.
     */
    @JvmName("idwnxmktnwkpsxug")
    public suspend fun writeCapacity(`value`: Int?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.writeCapacity = mapped
    }

    internal fun build(): TableArgs = TableArgs(
        attributes = attributes,
        billingMode = billingMode,
        deletionProtectionEnabled = deletionProtectionEnabled,
        globalSecondaryIndexes = globalSecondaryIndexes,
        hashKey = hashKey,
        importTable = importTable,
        localSecondaryIndexes = localSecondaryIndexes,
        name = name,
        pointInTimeRecovery = pointInTimeRecovery,
        rangeKey = rangeKey,
        readCapacity = readCapacity,
        replicas = replicas,
        restoreDateTime = restoreDateTime,
        restoreSourceName = restoreSourceName,
        restoreToLatestTime = restoreToLatestTime,
        serverSideEncryption = serverSideEncryption,
        streamEnabled = streamEnabled,
        streamViewType = streamViewType,
        tableClass = tableClass,
        tags = tags,
        ttl = ttl,
        writeCapacity = writeCapacity,
    )
}
