@file:Suppress("NAME_SHADOWING", "DEPRECATION")

package com.pulumi.aws.ec2.kotlin

import com.pulumi.aws.ec2.LaunchConfigurationArgs.builder
import com.pulumi.aws.ec2.kotlin.inputs.LaunchConfigurationEbsBlockDeviceArgs
import com.pulumi.aws.ec2.kotlin.inputs.LaunchConfigurationEbsBlockDeviceArgsBuilder
import com.pulumi.aws.ec2.kotlin.inputs.LaunchConfigurationEphemeralBlockDeviceArgs
import com.pulumi.aws.ec2.kotlin.inputs.LaunchConfigurationEphemeralBlockDeviceArgsBuilder
import com.pulumi.aws.ec2.kotlin.inputs.LaunchConfigurationMetadataOptionsArgs
import com.pulumi.aws.ec2.kotlin.inputs.LaunchConfigurationMetadataOptionsArgsBuilder
import com.pulumi.aws.ec2.kotlin.inputs.LaunchConfigurationRootBlockDeviceArgs
import com.pulumi.aws.ec2.kotlin.inputs.LaunchConfigurationRootBlockDeviceArgsBuilder
import com.pulumi.core.Output
import com.pulumi.core.Output.of
import com.pulumi.kotlin.ConvertibleToJava
import com.pulumi.kotlin.PulumiTagMarker
import com.pulumi.kotlin.applySuspend
import kotlin.Boolean
import kotlin.String
import kotlin.Suppress
import kotlin.Unit
import kotlin.collections.List
import kotlin.jvm.JvmName

/**
 * Provides a resource to create a new launch configuration, used for autoscaling groups.
 * !> **WARNING:** The use of launch configurations is discouraged in favour of launch templates. Read more in the [AWS EC2 Documentation](https://docs.aws.amazon.com/autoscaling/ec2/userguide/launch-configurations.html).
 * > **Note** When using `aws.ec2.LaunchConfiguration` with `aws.autoscaling.Group`, it is recommended to use the `name_prefix` (Optional) instead of the `name` (Optional) attribute.
 * ## Example Usage
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.ec2.Ec2Functions;
 * import com.pulumi.aws.ec2.inputs.GetAmiArgs;
 * import com.pulumi.aws.ec2.LaunchConfiguration;
 * import com.pulumi.aws.ec2.LaunchConfigurationArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         final var ubuntu = Ec2Functions.getAmi(GetAmiArgs.builder()
 *             .mostRecent(true)
 *             .filters(
 *                 GetAmiFilterArgs.builder()
 *                     .name("name")
 *                     .values("ubuntu/images/hvm-ssd/ubuntu-trusty-14.04-amd64-server-*")
 *                     .build(),
 *                 GetAmiFilterArgs.builder()
 *                     .name("virtualization-type")
 *                     .values("hvm")
 *                     .build())
 *             .owners("099720109477")
 *             .build());
 *         var asConf = new LaunchConfiguration("asConf", LaunchConfigurationArgs.builder()
 *             .imageId(ubuntu.applyValue(getAmiResult -> getAmiResult.id()))
 *             .instanceType("t2.micro")
 *             .build());
 *     }
 * }
 * ```
 * ## Using with AutoScaling Groups
 * Launch Configurations cannot be updated after creation with the Amazon
 * Web Service API. In order to update a Launch Configuration, this provider will
 * destroy the existing resource and create a replacement. In order to effectively
 * use a Launch Configuration resource with an AutoScaling Group resource,
 * it's recommended to specify `create_before_destroy` in a lifecycle block.
 * Either omit the Launch Configuration `name` attribute, or specify a partial name
 * with `name_prefix`.  Example:
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const ubuntu = aws.ec2.getAmi({
 *     mostRecent: true,
 *     filters: [
 *         {
 *             name: "name",
 *             values: ["ubuntu/images/hvm-ssd/ubuntu-trusty-14&#46;04-amd64-server-*"],
 *         },
 *         {
 *             name: "virtualization-type",
 *             values: ["hvm"],
 *         },
 *     ],
 *     owners: ["099720109477"],
 * });
 * const asConf = new aws.ec2.LaunchConfiguration("asConf", {
 *     namePrefix: "lc-example-",
 *     imageId: ubuntu.then(ubuntu => ubuntu.id),
 *     instanceType: "t2.micro",
 * });
 * const bar = new aws.autoscaling.Group("bar", {
 *     launchConfiguration: asConf.name,
 *     minSize: 1,
 *     maxSize: 2,
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * ubuntu = aws.ec2.get_ami(most_recent=True,
 *     filters=[
 *         aws.ec2.GetAmiFilterArgs(
 *             name="name",
 *             values=["ubuntu/images/hvm-ssd/ubuntu-trusty-14&#46;04-amd64-server-*"],
 *         ),
 *         aws.ec2.GetAmiFilterArgs(
 *             name="virtualization-type",
 *             values=["hvm"],
 *         ),
 *     ],
 *     owners=["099720109477"])
 * as_conf = aws.ec2.LaunchConfiguration("asConf",
 *     name_prefix="lc-example-",
 *     image_id=ubuntu.id,
 *     instance_type="t2.micro")
 * bar = aws.autoscaling.Group("bar",
 *     launch_configuration=as_conf.name,
 *     min_size=1,
 *     max_size=2)
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var ubuntu = Aws.Ec2.GetAmi.Invoke(new()
 *     {
 *         MostRecent = true,
 *         Filters = new[]
 *         {
 *             new Aws.Ec2.Inputs.GetAmiFilterInputArgs
 *             {
 *                 Name = "name",
 *                 Values = new[]
 *                 {
 *                     "ubuntu/images/hvm-ssd/ubuntu-trusty-14.04-amd64-server-*",
 *                 },
 *             },
 *             new Aws.Ec2.Inputs.GetAmiFilterInputArgs
 *             {
 *                 Name = "virtualization-type",
 *                 Values = new[]
 *                 {
 *                     "hvm",
 *                 },
 *             },
 *         },
 *         Owners = new[]
 *         {
 *             "099720109477",
 *         },
 *     });
 *     var asConf = new Aws.Ec2.LaunchConfiguration("asConf", new()
 *     {
 *         NamePrefix = "lc-example-",
 *         ImageId = ubuntu.Apply(getAmiResult => getAmiResult.Id),
 *         InstanceType = "t2.micro",
 *     });
 *     var bar = new Aws.AutoScaling.Group("bar", new()
 *     {
 *         LaunchConfiguration = asConf.Name,
 *         MinSize = 1,
 *         MaxSize = 2,
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/autoscaling"
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/ec2"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		ubuntu, err := ec2.LookupAmi(ctx, &ec2.LookupAmiArgs{
 * 			MostRecent: pulumi.BoolRef(true),
 * 			Filters: []ec2.GetAmiFilter{
 * 				{
 * 					Name: "name",
 * 					Values: []string{
 * 						"ubuntu/images/hvm-ssd/ubuntu-trusty-14.04-amd64-server-*",
 * 					},
 * 				},
 * 				{
 * 					Name: "virtualization-type",
 * 					Values: []string{
 * 						"hvm",
 * 					},
 * 				},
 * 			},
 * 			Owners: []string{
 * 				"099720109477",
 * 			},
 * 		}, nil)
 * 		if err != nil {
 * 			return err
 * 		}
 * 		asConf, err := ec2.NewLaunchConfiguration(ctx, "asConf", &ec2.LaunchConfigurationArgs{
 * 			NamePrefix:   pulumi.String("lc-example-"),
 * 			ImageId:      *pulumi.String(ubuntu.Id),
 * 			InstanceType: pulumi.String("t2.micro"),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		_, err = autoscaling.NewGroup(ctx, "bar", &autoscaling.GroupArgs{
 * 			LaunchConfiguration: asConf.Name,
 * 			MinSize:             pulumi.Int(1),
 * 			MaxSize:             pulumi.Int(2),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.ec2.Ec2Functions;
 * import com.pulumi.aws.ec2.inputs.GetAmiArgs;
 * import com.pulumi.aws.ec2.LaunchConfiguration;
 * import com.pulumi.aws.ec2.LaunchConfigurationArgs;
 * import com.pulumi.aws.autoscaling.Group;
 * import com.pulumi.aws.autoscaling.GroupArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         final var ubuntu = Ec2Functions.getAmi(GetAmiArgs.builder()
 *             .mostRecent(true)
 *             .filters(
 *                 GetAmiFilterArgs.builder()
 *                     .name("name")
 *                     .values("ubuntu/images/hvm-ssd/ubuntu-trusty-14.04-amd64-server-*")
 *                     .build(),
 *                 GetAmiFilterArgs.builder()
 *                     .name("virtualization-type")
 *                     .values("hvm")
 *                     .build())
 *             .owners("099720109477")
 *             .build());
 *         var asConf = new LaunchConfiguration("asConf", LaunchConfigurationArgs.builder()
 *             .namePrefix("lc-example-")
 *             .imageId(ubuntu.applyValue(getAmiResult -> getAmiResult.id()))
 *             .instanceType("t2.micro")
 *             .build());
 *         var bar = new Group("bar", GroupArgs.builder()
 *             .launchConfiguration(asConf.name())
 *             .minSize(1)
 *             .maxSize(2)
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   asConf:
 *     type: aws:ec2:LaunchConfiguration
 *     properties:
 *       namePrefix: lc-example-
 *       imageId: ${ubuntu.id}
 *       instanceType: t2.micro
 *   bar:
 *     type: aws:autoscaling:Group
 *     properties:
 *       launchConfiguration: ${asConf.name}
 *       minSize: 1
 *       maxSize: 2
 * variables:
 *   ubuntu:
 *     fn::invoke:
 *       Function: aws:ec2:getAmi
 *       Arguments:
 *         mostRecent: true
 *         filters:
 *           - name: name
 *             values:
 *               - ubuntu/images/hvm-ssd/ubuntu-trusty-14.04-amd64-server-*
 *           - name: virtualization-type
 *             values:
 *               - hvm
 *         owners:
 *           - '099720109477'
 * ```
 * With this setup this provider generates a unique name for your Launch
 * Configuration and can then update the AutoScaling Group without conflict before
 * destroying the previous Launch Configuration.
 * ## Using with Spot Instances
 * Launch configurations can set the spot instance pricing to be used for the
 * Auto Scaling Group to reserve instances. Simply specifying the `spot_price`
 * parameter will set the price on the Launch Configuration which will attempt to
 * reserve your instances at this price.  See the [AWS Spot Instance
 * documentation](https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/using-spot-instances.html)
 * for more information or how to launch [Spot Instances][3] with this provider.
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const ubuntu = aws.ec2.getAmi({
 *     mostRecent: true,
 *     filters: [
 *         {
 *             name: "name",
 *             values: ["ubuntu/images/hvm-ssd/ubuntu-trusty-14&#46;04-amd64-server-*"],
 *         },
 *         {
 *             name: "virtualization-type",
 *             values: ["hvm"],
 *         },
 *     ],
 *     owners: ["099720109477"],
 * });
 * const asConf = new aws.ec2.LaunchConfiguration("asConf", {
 *     imageId: ubuntu.then(ubuntu => ubuntu.id),
 *     instanceType: "m4.large",
 *     spotPrice: "0.001",
 * });
 * const bar = new aws.autoscaling.Group("bar", {launchConfiguration: asConf.name});
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * ubuntu = aws.ec2.get_ami(most_recent=True,
 *     filters=[
 *         aws.ec2.GetAmiFilterArgs(
 *             name="name",
 *             values=["ubuntu/images/hvm-ssd/ubuntu-trusty-14&#46;04-amd64-server-*"],
 *         ),
 *         aws.ec2.GetAmiFilterArgs(
 *             name="virtualization-type",
 *             values=["hvm"],
 *         ),
 *     ],
 *     owners=["099720109477"])
 * as_conf = aws.ec2.LaunchConfiguration("asConf",
 *     image_id=ubuntu.id,
 *     instance_type="m4.large",
 *     spot_price="0.001")
 * bar = aws.autoscaling.Group("bar", launch_configuration=as_conf.name)
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var ubuntu = Aws.Ec2.GetAmi.Invoke(new()
 *     {
 *         MostRecent = true,
 *         Filters = new[]
 *         {
 *             new Aws.Ec2.Inputs.GetAmiFilterInputArgs
 *             {
 *                 Name = "name",
 *                 Values = new[]
 *                 {
 *                     "ubuntu/images/hvm-ssd/ubuntu-trusty-14.04-amd64-server-*",
 *                 },
 *             },
 *             new Aws.Ec2.Inputs.GetAmiFilterInputArgs
 *             {
 *                 Name = "virtualization-type",
 *                 Values = new[]
 *                 {
 *                     "hvm",
 *                 },
 *             },
 *         },
 *         Owners = new[]
 *         {
 *             "099720109477",
 *         },
 *     });
 *     var asConf = new Aws.Ec2.LaunchConfiguration("asConf", new()
 *     {
 *         ImageId = ubuntu.Apply(getAmiResult => getAmiResult.Id),
 *         InstanceType = "m4.large",
 *         SpotPrice = "0.001",
 *     });
 *     var bar = new Aws.AutoScaling.Group("bar", new()
 *     {
 *         LaunchConfiguration = asConf.Name,
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/autoscaling"
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/ec2"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		ubuntu, err := ec2.LookupAmi(ctx, &ec2.LookupAmiArgs{
 * 			MostRecent: pulumi.BoolRef(true),
 * 			Filters: []ec2.GetAmiFilter{
 * 				{
 * 					Name: "name",
 * 					Values: []string{
 * 						"ubuntu/images/hvm-ssd/ubuntu-trusty-14.04-amd64-server-*",
 * 					},
 * 				},
 * 				{
 * 					Name: "virtualization-type",
 * 					Values: []string{
 * 						"hvm",
 * 					},
 * 				},
 * 			},
 * 			Owners: []string{
 * 				"099720109477",
 * 			},
 * 		}, nil)
 * 		if err != nil {
 * 			return err
 * 		}
 * 		asConf, err := ec2.NewLaunchConfiguration(ctx, "asConf", &ec2.LaunchConfigurationArgs{
 * 			ImageId:      *pulumi.String(ubuntu.Id),
 * 			InstanceType: pulumi.String("m4.large"),
 * 			SpotPrice:    pulumi.String("0.001"),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		_, err = autoscaling.NewGroup(ctx, "bar", &autoscaling.GroupArgs{
 * 			LaunchConfiguration: asConf.Name,
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.ec2.Ec2Functions;
 * import com.pulumi.aws.ec2.inputs.GetAmiArgs;
 * import com.pulumi.aws.ec2.LaunchConfiguration;
 * import com.pulumi.aws.ec2.LaunchConfigurationArgs;
 * import com.pulumi.aws.autoscaling.Group;
 * import com.pulumi.aws.autoscaling.GroupArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         final var ubuntu = Ec2Functions.getAmi(GetAmiArgs.builder()
 *             .mostRecent(true)
 *             .filters(
 *                 GetAmiFilterArgs.builder()
 *                     .name("name")
 *                     .values("ubuntu/images/hvm-ssd/ubuntu-trusty-14.04-amd64-server-*")
 *                     .build(),
 *                 GetAmiFilterArgs.builder()
 *                     .name("virtualization-type")
 *                     .values("hvm")
 *                     .build())
 *             .owners("099720109477")
 *             .build());
 *         var asConf = new LaunchConfiguration("asConf", LaunchConfigurationArgs.builder()
 *             .imageId(ubuntu.applyValue(getAmiResult -> getAmiResult.id()))
 *             .instanceType("m4.large")
 *             .spotPrice("0.001")
 *             .build());
 *         var bar = new Group("bar", GroupArgs.builder()
 *             .launchConfiguration(asConf.name())
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   asConf:
 *     type: aws:ec2:LaunchConfiguration
 *     properties:
 *       imageId: ${ubuntu.id}
 *       instanceType: m4.large
 *       spotPrice: '0.001'
 *   bar:
 *     type: aws:autoscaling:Group
 *     properties:
 *       launchConfiguration: ${asConf.name}
 * variables:
 *   ubuntu:
 *     fn::invoke:
 *       Function: aws:ec2:getAmi
 *       Arguments:
 *         mostRecent: true
 *         filters:
 *           - name: name
 *             values:
 *               - ubuntu/images/hvm-ssd/ubuntu-trusty-14.04-amd64-server-*
 *           - name: virtualization-type
 *             values:
 *               - hvm
 *         owners:
 *           - '099720109477'
 * ```
 * ## Block devices
 * Each of the `*_block_device` attributes controls a portion of the AWS
 * Launch Configuration's "Block Device Mapping". It's a good idea to familiarize yourself with [AWS's Block Device
 * Mapping docs](https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/block-device-mapping-concepts.html)
 * to understand the implications of using these attributes.
 * Each AWS Instance type has a different set of Instance Store block devices
 * available for attachment. AWS [publishes a
 * list](https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/InstanceStorage.html#StorageOnInstanceTypes)
 * of which ephemeral devices are available on each type. The devices are always
 * identified by the `virtual_name` in the format `ephemeral{0..N}`.
 * > **NOTE:** Changes to `*_block_device` configuration of _existing_ resources
 * cannot currently be detected by this provider. After updating to block device
 * configuration, resource recreation can be manually triggered by using the
 * [`up` command with the --replace argument](https://www.pulumi.com/docs/reference/cli/pulumi_up/).
 * ### ebs_block_device
 * Modifying any of the `ebs_block_device` settings requires resource replacement.
 * * `device_name` - (Required) The name of the device to mount.
 * * `snapshot_id` - (Optional) The Snapshot ID to mount.
 * * `volume_type` - (Optional) The type of volume. Can be `standard`, `gp2`, `gp3`, `st1`, `sc1` or `io1`.
 * * `volume_size` - (Optional) The size of the volume in gigabytes.
 * * `iops` - (Optional) The amount of provisioned
 *   [IOPS](https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ebs-io-characteristics.html).
 *   This must be set with a `volume_type` of `"io1"`.
 * * `throughput` - (Optional) The throughput (MiBps) to provision for a `gp3` volume.
 * * `delete_on_termination` - (Optional) Whether the volume should be destroyed
 *   on instance termination (Default: `true`).
 * * `encrypted` - (Optional) Whether the volume should be encrypted or not. Defaults to `false`.
 * * `no_device` - (Optional) Whether the device in the block device mapping of the AMI is suppressed.
 * ### ephemeral_block_device
 * * `device_name` - (Required) The name of the block device to mount on the instance.
 * * `no_device` - (Optional) Whether the device in the block device mapping of the AMI is suppressed.
 * * `virtual_name` - (Optional) The [Instance Store Device Name](https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/InstanceStorage.html#InstanceStoreDeviceNames).
 * ### root_block_device
 * > Modifying any of the `root_block_device` settings requires resource replacement.
 * * `delete_on_termination` - (Optional) Whether the volume should be destroyed on instance termination. Defaults to `true`.
 * * `encrypted` - (Optional) Whether the volume should be encrypted or not. Defaults to `false`.
 * * `iops` - (Optional) The amount of provisioned [IOPS](https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ebs-io-characteristics.html). This must be set with a `volume_type` of `io1`.
 * * `throughput` - (Optional) The throughput (MiBps) to provision for a `gp3` volume.
 * * `volume_size` - (Optional) The size of the volume in gigabytes.
 * * `volume_type` - (Optional) The type of volume. Can be `standard`, `gp2`, `gp3`, `st1`, `sc1` or `io1`.
 * ## Import
 * Using `pulumi import`, import launch configurations using the `name`. For example:
 * ```sh
 *  $ pulumi import aws:ec2/launchConfiguration:LaunchConfiguration as_conf pulumi-lg-123456
 * ```
 * @property associatePublicIpAddress Associate a public ip address with an instance in a VPC.
 * @property ebsBlockDevices Additional EBS block devices to attach to the instance. See Block Devices below for details.
 * @property ebsOptimized If true, the launched EC2 instance will be EBS-optimized.
 * @property enableMonitoring Enables/disables detailed monitoring. This is enabled by default.
 * @property ephemeralBlockDevices Customize Ephemeral (also known as "Instance Store") volumes on the instance. See Block Devices below for details.
 * @property iamInstanceProfile The name attribute of the IAM instance profile to associate with launched instances.
 * @property imageId The EC2 image ID to launch.
 * @property instanceType The size of instance to launch.
 * The following arguments are optional:
 * @property keyName The key name that should be used for the instance.
 * @property metadataOptions The metadata options for the instance.
 * @property name The name of the launch configuration. If you leave this blank, this provider will auto-generate a unique name. Conflicts with `name_prefix`.
 * @property namePrefix Creates a unique name beginning with the specified prefix. Conflicts with `name`.
 * @property placementTenancy The tenancy of the instance. Valid values are `default` or `dedicated`, see [AWS's Create Launch Configuration](http://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_CreateLaunchConfiguration.html) for more details.
 * @property rootBlockDevice Customize details about the root block device of the instance. See Block Devices below for details.
 * @property securityGroups A list of associated security group IDS.
 * @property spotPrice The maximum price to use for reserving spot instances.
 * @property userData The user data to provide when launching the instance. Do not pass gzip-compressed data via this argument; see `user_data_base64` instead.
 * @property userDataBase64 Can be used instead of `user_data` to pass base64-encoded binary data directly. Use this instead of `user_data` whenever the value is not a valid UTF-8 string. For example, gzip-encoded user data must be base64-encoded and passed via this argument to avoid corruption.
 */
public data class LaunchConfigurationArgs(
    public val associatePublicIpAddress: Output<Boolean>? = null,
    public val ebsBlockDevices: Output<List<LaunchConfigurationEbsBlockDeviceArgs>>? = null,
    public val ebsOptimized: Output<Boolean>? = null,
    public val enableMonitoring: Output<Boolean>? = null,
    public val ephemeralBlockDevices: Output<List<LaunchConfigurationEphemeralBlockDeviceArgs>>? =
        null,
    public val iamInstanceProfile: Output<String>? = null,
    public val imageId: Output<String>? = null,
    public val instanceType: Output<String>? = null,
    public val keyName: Output<String>? = null,
    public val metadataOptions: Output<LaunchConfigurationMetadataOptionsArgs>? = null,
    public val name: Output<String>? = null,
    public val namePrefix: Output<String>? = null,
    public val placementTenancy: Output<String>? = null,
    public val rootBlockDevice: Output<LaunchConfigurationRootBlockDeviceArgs>? = null,
    public val securityGroups: Output<List<String>>? = null,
    public val spotPrice: Output<String>? = null,
    public val userData: Output<String>? = null,
    public val userDataBase64: Output<String>? = null,
) : ConvertibleToJava<com.pulumi.aws.ec2.LaunchConfigurationArgs> {
    override fun toJava(): com.pulumi.aws.ec2.LaunchConfigurationArgs =
        com.pulumi.aws.ec2.LaunchConfigurationArgs.builder()
            .associatePublicIpAddress(associatePublicIpAddress?.applyValue({ args0 -> args0 }))
            .ebsBlockDevices(
                ebsBlockDevices?.applyValue({ args0 ->
                    args0.map({ args0 ->
                        args0.let({ args0 ->
                            args0.toJava()
                        })
                    })
                }),
            )
            .ebsOptimized(ebsOptimized?.applyValue({ args0 -> args0 }))
            .enableMonitoring(enableMonitoring?.applyValue({ args0 -> args0 }))
            .ephemeralBlockDevices(
                ephemeralBlockDevices?.applyValue({ args0 ->
                    args0.map({ args0 ->
                        args0.let({ args0 -> args0.toJava() })
                    })
                }),
            )
            .iamInstanceProfile(iamInstanceProfile?.applyValue({ args0 -> args0 }))
            .imageId(imageId?.applyValue({ args0 -> args0 }))
            .instanceType(instanceType?.applyValue({ args0 -> args0 }))
            .keyName(keyName?.applyValue({ args0 -> args0 }))
            .metadataOptions(metadataOptions?.applyValue({ args0 -> args0.let({ args0 -> args0.toJava() }) }))
            .name(name?.applyValue({ args0 -> args0 }))
            .namePrefix(namePrefix?.applyValue({ args0 -> args0 }))
            .placementTenancy(placementTenancy?.applyValue({ args0 -> args0 }))
            .rootBlockDevice(rootBlockDevice?.applyValue({ args0 -> args0.let({ args0 -> args0.toJava() }) }))
            .securityGroups(securityGroups?.applyValue({ args0 -> args0.map({ args0 -> args0 }) }))
            .spotPrice(spotPrice?.applyValue({ args0 -> args0 }))
            .userData(userData?.applyValue({ args0 -> args0 }))
            .userDataBase64(userDataBase64?.applyValue({ args0 -> args0 })).build()
}

/**
 * Builder for [LaunchConfigurationArgs].
 */
@PulumiTagMarker
public class LaunchConfigurationArgsBuilder internal constructor() {
    private var associatePublicIpAddress: Output<Boolean>? = null

    private var ebsBlockDevices: Output<List<LaunchConfigurationEbsBlockDeviceArgs>>? = null

    private var ebsOptimized: Output<Boolean>? = null

    private var enableMonitoring: Output<Boolean>? = null

    private var ephemeralBlockDevices: Output<List<LaunchConfigurationEphemeralBlockDeviceArgs>>? =
        null

    private var iamInstanceProfile: Output<String>? = null

    private var imageId: Output<String>? = null

    private var instanceType: Output<String>? = null

    private var keyName: Output<String>? = null

    private var metadataOptions: Output<LaunchConfigurationMetadataOptionsArgs>? = null

    private var name: Output<String>? = null

    private var namePrefix: Output<String>? = null

    private var placementTenancy: Output<String>? = null

    private var rootBlockDevice: Output<LaunchConfigurationRootBlockDeviceArgs>? = null

    private var securityGroups: Output<List<String>>? = null

    private var spotPrice: Output<String>? = null

    private var userData: Output<String>? = null

    private var userDataBase64: Output<String>? = null

    /**
     * @param value Associate a public ip address with an instance in a VPC.
     */
    @JvmName("pitnvkmbqfrfesas")
    public suspend fun associatePublicIpAddress(`value`: Output<Boolean>) {
        this.associatePublicIpAddress = value
    }

    /**
     * @param value Additional EBS block devices to attach to the instance. See Block Devices below for details.
     */
    @JvmName("rcoogqgqqmqbgcar")
    public suspend fun ebsBlockDevices(`value`: Output<List<LaunchConfigurationEbsBlockDeviceArgs>>) {
        this.ebsBlockDevices = value
    }

    @JvmName("ieafxqfclymalyim")
    public suspend fun ebsBlockDevices(vararg values: Output<LaunchConfigurationEbsBlockDeviceArgs>) {
        this.ebsBlockDevices = Output.all(values.asList())
    }

    /**
     * @param values Additional EBS block devices to attach to the instance. See Block Devices below for details.
     */
    @JvmName("byrbtsomnsrknctg")
    public suspend fun ebsBlockDevices(values: List<Output<LaunchConfigurationEbsBlockDeviceArgs>>) {
        this.ebsBlockDevices = Output.all(values)
    }

    /**
     * @param value If true, the launched EC2 instance will be EBS-optimized.
     */
    @JvmName("bwxpkycdrwvsusad")
    public suspend fun ebsOptimized(`value`: Output<Boolean>) {
        this.ebsOptimized = value
    }

    /**
     * @param value Enables/disables detailed monitoring. This is enabled by default.
     */
    @JvmName("kjevwcdnmgqeffwy")
    public suspend fun enableMonitoring(`value`: Output<Boolean>) {
        this.enableMonitoring = value
    }

    /**
     * @param value Customize Ephemeral (also known as "Instance Store") volumes on the instance. See Block Devices below for details.
     */
    @JvmName("jadpgpmoifvcrdoe")
    public suspend
    fun ephemeralBlockDevices(`value`: Output<List<LaunchConfigurationEphemeralBlockDeviceArgs>>) {
        this.ephemeralBlockDevices = value
    }

    @JvmName("cbmswkfroeuyrglq")
    public suspend fun ephemeralBlockDevices(
        vararg
        values: Output<LaunchConfigurationEphemeralBlockDeviceArgs>,
    ) {
        this.ephemeralBlockDevices = Output.all(values.asList())
    }

    /**
     * @param values Customize Ephemeral (also known as "Instance Store") volumes on the instance. See Block Devices below for details.
     */
    @JvmName("ojebsmghivsytglw")
    public suspend
    fun ephemeralBlockDevices(values: List<Output<LaunchConfigurationEphemeralBlockDeviceArgs>>) {
        this.ephemeralBlockDevices = Output.all(values)
    }

    /**
     * @param value The name attribute of the IAM instance profile to associate with launched instances.
     */
    @JvmName("hlftxbvmlbjypaln")
    public suspend fun iamInstanceProfile(`value`: Output<String>) {
        this.iamInstanceProfile = value
    }

    /**
     * @param value The EC2 image ID to launch.
     */
    @JvmName("ltkerfvsjiipoycg")
    public suspend fun imageId(`value`: Output<String>) {
        this.imageId = value
    }

    /**
     * @param value The size of instance to launch.
     * The following arguments are optional:
     */
    @JvmName("yvgjxfxjcrstuyyj")
    public suspend fun instanceType(`value`: Output<String>) {
        this.instanceType = value
    }

    /**
     * @param value The key name that should be used for the instance.
     */
    @JvmName("uqgevjyqvwfutunj")
    public suspend fun keyName(`value`: Output<String>) {
        this.keyName = value
    }

    /**
     * @param value The metadata options for the instance.
     */
    @JvmName("yyadejjmfprialun")
    public suspend fun metadataOptions(`value`: Output<LaunchConfigurationMetadataOptionsArgs>) {
        this.metadataOptions = value
    }

    /**
     * @param value The name of the launch configuration. If you leave this blank, this provider will auto-generate a unique name. Conflicts with `name_prefix`.
     */
    @JvmName("tjtkdqlcjxniemnl")
    public suspend fun name(`value`: Output<String>) {
        this.name = value
    }

    /**
     * @param value Creates a unique name beginning with the specified prefix. Conflicts with `name`.
     */
    @JvmName("frtqskdnoxgckwed")
    public suspend fun namePrefix(`value`: Output<String>) {
        this.namePrefix = value
    }

    /**
     * @param value The tenancy of the instance. Valid values are `default` or `dedicated`, see [AWS's Create Launch Configuration](http://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_CreateLaunchConfiguration.html) for more details.
     */
    @JvmName("xkmejaukpontsbxb")
    public suspend fun placementTenancy(`value`: Output<String>) {
        this.placementTenancy = value
    }

    /**
     * @param value Customize details about the root block device of the instance. See Block Devices below for details.
     */
    @JvmName("rjjpyuxrmoyftgbb")
    public suspend fun rootBlockDevice(`value`: Output<LaunchConfigurationRootBlockDeviceArgs>) {
        this.rootBlockDevice = value
    }

    /**
     * @param value A list of associated security group IDS.
     */
    @JvmName("dlortynuaeyggurm")
    public suspend fun securityGroups(`value`: Output<List<String>>) {
        this.securityGroups = value
    }

    @JvmName("raedfugtnqwbooki")
    public suspend fun securityGroups(vararg values: Output<String>) {
        this.securityGroups = Output.all(values.asList())
    }

    /**
     * @param values A list of associated security group IDS.
     */
    @JvmName("xeyslooemxodyihx")
    public suspend fun securityGroups(values: List<Output<String>>) {
        this.securityGroups = Output.all(values)
    }

    /**
     * @param value The maximum price to use for reserving spot instances.
     */
    @JvmName("wrwkygqsobtykjqf")
    public suspend fun spotPrice(`value`: Output<String>) {
        this.spotPrice = value
    }

    /**
     * @param value The user data to provide when launching the instance. Do not pass gzip-compressed data via this argument; see `user_data_base64` instead.
     */
    @JvmName("whxbrycrlobnqobw")
    public suspend fun userData(`value`: Output<String>) {
        this.userData = value
    }

    /**
     * @param value Can be used instead of `user_data` to pass base64-encoded binary data directly. Use this instead of `user_data` whenever the value is not a valid UTF-8 string. For example, gzip-encoded user data must be base64-encoded and passed via this argument to avoid corruption.
     */
    @JvmName("bjmopplfwypndxev")
    public suspend fun userDataBase64(`value`: Output<String>) {
        this.userDataBase64 = value
    }

    /**
     * @param value Associate a public ip address with an instance in a VPC.
     */
    @JvmName("sgjxdygrsqwjjiaq")
    public suspend fun associatePublicIpAddress(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.associatePublicIpAddress = mapped
    }

    /**
     * @param value Additional EBS block devices to attach to the instance. See Block Devices below for details.
     */
    @JvmName("bpfgfwtellwvgiud")
    public suspend fun ebsBlockDevices(`value`: List<LaunchConfigurationEbsBlockDeviceArgs>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.ebsBlockDevices = mapped
    }

    /**
     * @param argument Additional EBS block devices to attach to the instance. See Block Devices below for details.
     */
    @JvmName("hcbgcciugigegfnw")
    public suspend
    fun ebsBlockDevices(argument: List<suspend LaunchConfigurationEbsBlockDeviceArgsBuilder.() -> Unit>) {
        val toBeMapped = argument.toList().map {
            LaunchConfigurationEbsBlockDeviceArgsBuilder().applySuspend { it() }.build()
        }
        val mapped = of(toBeMapped)
        this.ebsBlockDevices = mapped
    }

    /**
     * @param argument Additional EBS block devices to attach to the instance. See Block Devices below for details.
     */
    @JvmName("jkfynvewckrcgqhv")
    public suspend fun ebsBlockDevices(
        vararg
        argument: suspend LaunchConfigurationEbsBlockDeviceArgsBuilder.() -> Unit,
    ) {
        val toBeMapped = argument.toList().map {
            LaunchConfigurationEbsBlockDeviceArgsBuilder().applySuspend { it() }.build()
        }
        val mapped = of(toBeMapped)
        this.ebsBlockDevices = mapped
    }

    /**
     * @param argument Additional EBS block devices to attach to the instance. See Block Devices below for details.
     */
    @JvmName("rfrahhseftnkphcs")
    public suspend
    fun ebsBlockDevices(argument: suspend LaunchConfigurationEbsBlockDeviceArgsBuilder.() -> Unit) {
        val toBeMapped = listOf(
            LaunchConfigurationEbsBlockDeviceArgsBuilder().applySuspend {
                argument()
            }.build(),
        )
        val mapped = of(toBeMapped)
        this.ebsBlockDevices = mapped
    }

    /**
     * @param values Additional EBS block devices to attach to the instance. See Block Devices below for details.
     */
    @JvmName("odpiqwhdjrrjbeet")
    public suspend fun ebsBlockDevices(vararg values: LaunchConfigurationEbsBlockDeviceArgs) {
        val toBeMapped = values.toList()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.ebsBlockDevices = mapped
    }

    /**
     * @param value If true, the launched EC2 instance will be EBS-optimized.
     */
    @JvmName("wotliiflstpeetwe")
    public suspend fun ebsOptimized(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.ebsOptimized = mapped
    }

    /**
     * @param value Enables/disables detailed monitoring. This is enabled by default.
     */
    @JvmName("klxsepjaykrnfbwe")
    public suspend fun enableMonitoring(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.enableMonitoring = mapped
    }

    /**
     * @param value Customize Ephemeral (also known as "Instance Store") volumes on the instance. See Block Devices below for details.
     */
    @JvmName("algiopjbxkwyncba")
    public suspend
    fun ephemeralBlockDevices(`value`: List<LaunchConfigurationEphemeralBlockDeviceArgs>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.ephemeralBlockDevices = mapped
    }

    /**
     * @param argument Customize Ephemeral (also known as "Instance Store") volumes on the instance. See Block Devices below for details.
     */
    @JvmName("jwagynwnseaxmkxb")
    public suspend
    fun ephemeralBlockDevices(argument: List<suspend LaunchConfigurationEphemeralBlockDeviceArgsBuilder.() -> Unit>) {
        val toBeMapped = argument.toList().map {
            LaunchConfigurationEphemeralBlockDeviceArgsBuilder().applySuspend { it() }.build()
        }
        val mapped = of(toBeMapped)
        this.ephemeralBlockDevices = mapped
    }

    /**
     * @param argument Customize Ephemeral (also known as "Instance Store") volumes on the instance. See Block Devices below for details.
     */
    @JvmName("pmbwcimxaigkslju")
    public suspend fun ephemeralBlockDevices(
        vararg
        argument: suspend LaunchConfigurationEphemeralBlockDeviceArgsBuilder.() -> Unit,
    ) {
        val toBeMapped = argument.toList().map {
            LaunchConfigurationEphemeralBlockDeviceArgsBuilder().applySuspend { it() }.build()
        }
        val mapped = of(toBeMapped)
        this.ephemeralBlockDevices = mapped
    }

    /**
     * @param argument Customize Ephemeral (also known as "Instance Store") volumes on the instance. See Block Devices below for details.
     */
    @JvmName("osawhksudptboydp")
    public suspend
    fun ephemeralBlockDevices(argument: suspend LaunchConfigurationEphemeralBlockDeviceArgsBuilder.() -> Unit) {
        val toBeMapped = listOf(
            LaunchConfigurationEphemeralBlockDeviceArgsBuilder().applySuspend {
                argument()
            }.build(),
        )
        val mapped = of(toBeMapped)
        this.ephemeralBlockDevices = mapped
    }

    /**
     * @param values Customize Ephemeral (also known as "Instance Store") volumes on the instance. See Block Devices below for details.
     */
    @JvmName("skqnqdpwibffqfhm")
    public suspend fun ephemeralBlockDevices(
        vararg
        values: LaunchConfigurationEphemeralBlockDeviceArgs,
    ) {
        val toBeMapped = values.toList()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.ephemeralBlockDevices = mapped
    }

    /**
     * @param value The name attribute of the IAM instance profile to associate with launched instances.
     */
    @JvmName("cgahlsnpbwtqxpud")
    public suspend fun iamInstanceProfile(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.iamInstanceProfile = mapped
    }

    /**
     * @param value The EC2 image ID to launch.
     */
    @JvmName("ukdodpkflsmeocqk")
    public suspend fun imageId(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.imageId = mapped
    }

    /**
     * @param value The size of instance to launch.
     * The following arguments are optional:
     */
    @JvmName("nihfvrhuwppwekhl")
    public suspend fun instanceType(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.instanceType = mapped
    }

    /**
     * @param value The key name that should be used for the instance.
     */
    @JvmName("gfnmtfvccodfyerg")
    public suspend fun keyName(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.keyName = mapped
    }

    /**
     * @param value The metadata options for the instance.
     */
    @JvmName("nyhrqpfsnpdljlgh")
    public suspend fun metadataOptions(`value`: LaunchConfigurationMetadataOptionsArgs?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.metadataOptions = mapped
    }

    /**
     * @param argument The metadata options for the instance.
     */
    @JvmName("kbfaevtvdymcqvdi")
    public suspend
    fun metadataOptions(argument: suspend LaunchConfigurationMetadataOptionsArgsBuilder.() -> Unit) {
        val toBeMapped = LaunchConfigurationMetadataOptionsArgsBuilder().applySuspend {
            argument()
        }.build()
        val mapped = of(toBeMapped)
        this.metadataOptions = mapped
    }

    /**
     * @param value The name of the launch configuration. If you leave this blank, this provider will auto-generate a unique name. Conflicts with `name_prefix`.
     */
    @JvmName("sfxqbathucrkwqki")
    public suspend fun name(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.name = mapped
    }

    /**
     * @param value Creates a unique name beginning with the specified prefix. Conflicts with `name`.
     */
    @JvmName("jswaqejdhqrdvlvl")
    public suspend fun namePrefix(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.namePrefix = mapped
    }

    /**
     * @param value The tenancy of the instance. Valid values are `default` or `dedicated`, see [AWS's Create Launch Configuration](http://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_CreateLaunchConfiguration.html) for more details.
     */
    @JvmName("etvhkkdbpiuypvlh")
    public suspend fun placementTenancy(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.placementTenancy = mapped
    }

    /**
     * @param value Customize details about the root block device of the instance. See Block Devices below for details.
     */
    @JvmName("uhfphmmbssxcbdky")
    public suspend fun rootBlockDevice(`value`: LaunchConfigurationRootBlockDeviceArgs?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.rootBlockDevice = mapped
    }

    /**
     * @param argument Customize details about the root block device of the instance. See Block Devices below for details.
     */
    @JvmName("ibqxuqqerpxhytud")
    public suspend
    fun rootBlockDevice(argument: suspend LaunchConfigurationRootBlockDeviceArgsBuilder.() -> Unit) {
        val toBeMapped = LaunchConfigurationRootBlockDeviceArgsBuilder().applySuspend {
            argument()
        }.build()
        val mapped = of(toBeMapped)
        this.rootBlockDevice = mapped
    }

    /**
     * @param value A list of associated security group IDS.
     */
    @JvmName("axmutxrivmhcgguk")
    public suspend fun securityGroups(`value`: List<String>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.securityGroups = mapped
    }

    /**
     * @param values A list of associated security group IDS.
     */
    @JvmName("gjqvoknabdylguls")
    public suspend fun securityGroups(vararg values: String) {
        val toBeMapped = values.toList()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.securityGroups = mapped
    }

    /**
     * @param value The maximum price to use for reserving spot instances.
     */
    @JvmName("buqucqndukffiiyh")
    public suspend fun spotPrice(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.spotPrice = mapped
    }

    /**
     * @param value The user data to provide when launching the instance. Do not pass gzip-compressed data via this argument; see `user_data_base64` instead.
     */
    @JvmName("ofhqtacwctyucuen")
    public suspend fun userData(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.userData = mapped
    }

    /**
     * @param value Can be used instead of `user_data` to pass base64-encoded binary data directly. Use this instead of `user_data` whenever the value is not a valid UTF-8 string. For example, gzip-encoded user data must be base64-encoded and passed via this argument to avoid corruption.
     */
    @JvmName("oxsxdieyomickmds")
    public suspend fun userDataBase64(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.userDataBase64 = mapped
    }

    internal fun build(): LaunchConfigurationArgs = LaunchConfigurationArgs(
        associatePublicIpAddress = associatePublicIpAddress,
        ebsBlockDevices = ebsBlockDevices,
        ebsOptimized = ebsOptimized,
        enableMonitoring = enableMonitoring,
        ephemeralBlockDevices = ephemeralBlockDevices,
        iamInstanceProfile = iamInstanceProfile,
        imageId = imageId,
        instanceType = instanceType,
        keyName = keyName,
        metadataOptions = metadataOptions,
        name = name,
        namePrefix = namePrefix,
        placementTenancy = placementTenancy,
        rootBlockDevice = rootBlockDevice,
        securityGroups = securityGroups,
        spotPrice = spotPrice,
        userData = userData,
        userDataBase64 = userDataBase64,
    )
}
