@file:Suppress("NAME_SHADOWING", "DEPRECATION")

package com.pulumi.aws.ec2.kotlin

import com.pulumi.aws.ec2.VpnConnectionArgs.builder
import com.pulumi.aws.ec2.kotlin.inputs.VpnConnectionTunnel1LogOptionsArgs
import com.pulumi.aws.ec2.kotlin.inputs.VpnConnectionTunnel1LogOptionsArgsBuilder
import com.pulumi.aws.ec2.kotlin.inputs.VpnConnectionTunnel2LogOptionsArgs
import com.pulumi.aws.ec2.kotlin.inputs.VpnConnectionTunnel2LogOptionsArgsBuilder
import com.pulumi.core.Output
import com.pulumi.core.Output.of
import com.pulumi.kotlin.ConvertibleToJava
import com.pulumi.kotlin.PulumiTagMarker
import com.pulumi.kotlin.applySuspend
import kotlin.Boolean
import kotlin.Int
import kotlin.Pair
import kotlin.String
import kotlin.Suppress
import kotlin.Unit
import kotlin.collections.List
import kotlin.collections.Map
import kotlin.jvm.JvmName

/**
 * Manages a Site-to-Site VPN connection. A Site-to-Site VPN connection is an Internet Protocol security (IPsec) VPN connection between a VPC and an on-premises network.
 * Any new Site-to-Site VPN connection that you create is an [AWS VPN connection](https://docs.aws.amazon.com/vpn/latest/s2svpn/vpn-categories.html).
 * > **Note:** The CIDR blocks in the arguments `tunnel1_inside_cidr` and `tunnel2_inside_cidr` must have a prefix of /30 and be a part of a specific range.
 * [Read more about this in the AWS documentation](https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_VpnTunnelOptionsSpecification.html).
 * ## Example Usage
 * ### EC2 Transit Gateway
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.ec2transitgateway.TransitGateway;
 * import com.pulumi.aws.ec2.CustomerGateway;
 * import com.pulumi.aws.ec2.CustomerGatewayArgs;
 * import com.pulumi.aws.ec2.VpnConnection;
 * import com.pulumi.aws.ec2.VpnConnectionArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var exampleTransitGateway = new TransitGateway("exampleTransitGateway");
 *         var exampleCustomerGateway = new CustomerGateway("exampleCustomerGateway", CustomerGatewayArgs.builder()
 *             .bgpAsn(65000)
 *             .ipAddress("172.0.0.1")
 *             .type("ipsec.1")
 *             .build());
 *         var exampleVpnConnection = new VpnConnection("exampleVpnConnection", VpnConnectionArgs.builder()
 *             .customerGatewayId(exampleCustomerGateway.id())
 *             .transitGatewayId(exampleTransitGateway.id())
 *             .type(exampleCustomerGateway.type())
 *             .build());
 *     }
 * }
 * ```
 * ### Virtual Private Gateway
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.ec2.Vpc;
 * import com.pulumi.aws.ec2.VpcArgs;
 * import com.pulumi.aws.ec2.VpnGateway;
 * import com.pulumi.aws.ec2.VpnGatewayArgs;
 * import com.pulumi.aws.ec2.CustomerGateway;
 * import com.pulumi.aws.ec2.CustomerGatewayArgs;
 * import com.pulumi.aws.ec2.VpnConnection;
 * import com.pulumi.aws.ec2.VpnConnectionArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var vpc = new Vpc("vpc", VpcArgs.builder()
 *             .cidrBlock("10.0.0.0/16")
 *             .build());
 *         var vpnGateway = new VpnGateway("vpnGateway", VpnGatewayArgs.builder()
 *             .vpcId(vpc.id())
 *             .build());
 *         var customerGateway = new CustomerGateway("customerGateway", CustomerGatewayArgs.builder()
 *             .bgpAsn(65000)
 *             .ipAddress("172.0.0.1")
 *             .type("ipsec.1")
 *             .build());
 *         var main = new VpnConnection("main", VpnConnectionArgs.builder()
 *             .vpnGatewayId(vpnGateway.id())
 *             .customerGatewayId(customerGateway.id())
 *             .type("ipsec.1")
 *             .staticRoutesOnly(true)
 *             .build());
 *     }
 * }
 * ```
 * ### AWS Site to Site Private VPN
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.directconnect.Gateway;
 * import com.pulumi.aws.directconnect.GatewayArgs;
 * import com.pulumi.aws.ec2transitgateway.TransitGateway;
 * import com.pulumi.aws.ec2transitgateway.TransitGatewayArgs;
 * import com.pulumi.aws.ec2.CustomerGateway;
 * import com.pulumi.aws.ec2.CustomerGatewayArgs;
 * import com.pulumi.aws.directconnect.GatewayAssociation;
 * import com.pulumi.aws.directconnect.GatewayAssociationArgs;
 * import com.pulumi.aws.ec2transitgateway.Ec2transitgatewayFunctions;
 * import com.pulumi.aws.ec2transitgateway.inputs.GetDirectConnectGatewayAttachmentArgs;
 * import com.pulumi.aws.ec2.VpnConnection;
 * import com.pulumi.aws.ec2.VpnConnectionArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var exampleGateway = new Gateway("exampleGateway", GatewayArgs.builder()
 *             .amazonSideAsn("64512")
 *             .build());
 *         var exampleTransitGateway = new TransitGateway("exampleTransitGateway", TransitGatewayArgs.builder()
 *             .amazonSideAsn("64513")
 *             .description("example_ipsec_vpn_example")
 *             .transitGatewayCidrBlocks("10.0.0.0/24")
 *             .build());
 *         var exampleCustomerGateway = new CustomerGateway("exampleCustomerGateway", CustomerGatewayArgs.builder()
 *             .bgpAsn(64514)
 *             .ipAddress("10.0.0.1")
 *             .type("ipsec.1")
 *             .tags(Map.of("Name", "example_ipsec_vpn_example"))
 *             .build());
 *         var exampleGatewayAssociation = new GatewayAssociation("exampleGatewayAssociation", GatewayAssociationArgs.builder()
 *             .dxGatewayId(exampleGateway.id())
 *             .associatedGatewayId(exampleTransitGateway.id())
 *             .allowedPrefixes("10.0.0.0/8")
 *             .build());
 *         final var exampleDirectConnectGatewayAttachment = Ec2transitgatewayFunctions.getDirectConnectGatewayAttachment(GetDirectConnectGatewayAttachmentArgs.builder()
 *             .transitGatewayId(exampleTransitGateway.id())
 *             .dxGatewayId(exampleGateway.id())
 *             .build());
 *         var exampleVpnConnection = new VpnConnection("exampleVpnConnection", VpnConnectionArgs.builder()
 *             .customerGatewayId(exampleCustomerGateway.id())
 *             .outsideIpAddressType("PrivateIpv4")
 *             .transitGatewayId(exampleTransitGateway.id())
 *             .transportTransitGatewayAttachmentId(exampleDirectConnectGatewayAttachment.applyValue(getDirectConnectGatewayAttachmentResult -> getDirectConnectGatewayAttachmentResult).applyValue(exampleDirectConnectGatewayAttachment -> exampleDirectConnectGatewayAttachment.applyValue(getDirectConnectGatewayAttachmentResult -> getDirectConnectGatewayAttachmentResult.id())))
 *             .type("ipsec.1")
 *             .tags(Map.of("Name", "example_ipsec_vpn_example"))
 *             .build());
 *     }
 * }
 * ```
 * ## Import
 * Using `pulumi import`, import VPN Connections using the VPN connection `id`. For example:
 * ```sh
 *  $ pulumi import aws:ec2/vpnConnection:VpnConnection testvpnconnection vpn-40f41529
 * ```
 * @property customerGatewayId The ID of the customer gateway.
 * @property enableAcceleration Indicate whether to enable acceleration for the VPN connection. Supports only EC2 Transit Gateway.
 * @property localIpv4NetworkCidr The IPv4 CIDR on the customer gateway (on-premises) side of the VPN connection.
 * @property localIpv6NetworkCidr The IPv6 CIDR on the customer gateway (on-premises) side of the VPN connection.
 * @property outsideIpAddressType Indicates if a Public S2S VPN or Private S2S VPN over AWS Direct Connect. Valid values are `PublicIpv4 | PrivateIpv4`
 * @property remoteIpv4NetworkCidr The IPv4 CIDR on the AWS side of the VPN connection.
 * @property remoteIpv6NetworkCidr The IPv6 CIDR on the customer gateway (on-premises) side of the VPN connection.
 * @property staticRoutesOnly Whether the VPN connection uses static routes exclusively. Static routes must be used for devices that don't support BGP.
 * @property tags Tags to apply to the connection. If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
 * @property transitGatewayId The ID of the EC2 Transit Gateway.
 * @property transportTransitGatewayAttachmentId . The attachment ID of the Transit Gateway attachment to Direct Connect Gateway. The ID is obtained through a data source only.
 * @property tunnel1DpdTimeoutAction The action to take after DPD timeout occurs for the first VPN tunnel. Specify restart to restart the IKE initiation. Specify clear to end the IKE session. Valid values are `clear | none | restart`.
 * @property tunnel1DpdTimeoutSeconds The number of seconds after which a DPD timeout occurs for the first VPN tunnel. Valid value is equal or higher than `30`.
 * @property tunnel1EnableTunnelLifecycleControl Turn on or off tunnel endpoint lifecycle control feature for the first VPN tunnel. Valid values are `true | false`.
 * @property tunnel1IkeVersions The IKE versions that are permitted for the first VPN tunnel. Valid values are `ikev1 | ikev2`.
 * @property tunnel1InsideCidr The CIDR block of the inside IP addresses for the first VPN tunnel. Valid value is a size /30 CIDR block from the 169.254.0.0/16 range.
 * @property tunnel1InsideIpv6Cidr The range of inside IPv6 addresses for the first VPN tunnel. Supports only EC2 Transit Gateway. Valid value is a size /126 CIDR block from the local fd00::/8 range.
 * @property tunnel1LogOptions Options for logging VPN tunnel activity. See Log Options below for more details.
 * @property tunnel1Phase1DhGroupNumbers List of one or more Diffie-Hellman group numbers that are permitted for the first VPN tunnel for phase 1 IKE negotiations. Valid values are ` 2 | 14 | 15 | 16 | 17 | 18 | 19 | 20 | 21 | 22 | 23 | 24`.
 * @property tunnel1Phase1EncryptionAlgorithms List of one or more encryption algorithms that are permitted for the first VPN tunnel for phase 1 IKE negotiations. Valid values are `AES128 | AES256 | AES128-GCM-16 | AES256-GCM-16`.
 * @property tunnel1Phase1IntegrityAlgorithms One or more integrity algorithms that are permitted for the first VPN tunnel for phase 1 IKE negotiations. Valid values are `SHA1 | SHA2-256 | SHA2-384 | SHA2-512`.
 * @property tunnel1Phase1LifetimeSeconds The lifetime for phase 1 of the IKE negotiation for the first VPN tunnel, in seconds. Valid value is between `900` and `28800`.
 * @property tunnel1Phase2DhGroupNumbers List of one or more Diffie-Hellman group numbers that are permitted for the first VPN tunnel for phase 2 IKE negotiations. Valid values are `2 | 5 | 14 | 15 | 16 | 17 | 18 | 19 | 20 | 21 | 22 | 23 | 24`.
 * @property tunnel1Phase2EncryptionAlgorithms List of one or more encryption algorithms that are permitted for the first VPN tunnel for phase 2 IKE negotiations. Valid values are `AES128 | AES256 | AES128-GCM-16 | AES256-GCM-16`.
 * @property tunnel1Phase2IntegrityAlgorithms List of one or more integrity algorithms that are permitted for the first VPN tunnel for phase 2 IKE negotiations. Valid values are `SHA1 | SHA2-256 | SHA2-384 | SHA2-512`.
 * @property tunnel1Phase2LifetimeSeconds The lifetime for phase 2 of the IKE negotiation for the first VPN tunnel, in seconds. Valid value is between `900` and `3600`.
 * @property tunnel1PresharedKey The preshared key of the first VPN tunnel. The preshared key must be between 8 and 64 characters in length and cannot start with zero(0). Allowed characters are alphanumeric characters, periods(.) and underscores(_).
 * @property tunnel1RekeyFuzzPercentage The percentage of the rekey window for the first VPN tunnel (determined by `tunnel1_rekey_margin_time_seconds`) during which the rekey time is randomly selected. Valid value is between `0` and `100`.
 * @property tunnel1RekeyMarginTimeSeconds The margin time, in seconds, before the phase 2 lifetime expires, during which the AWS side of the first VPN connection performs an IKE rekey. The exact time of the rekey is randomly selected based on the value for `tunnel1_rekey_fuzz_percentage`. Valid value is between `60` and half of `tunnel1_phase2_lifetime_seconds`.
 * @property tunnel1ReplayWindowSize The number of packets in an IKE replay window for the first VPN tunnel. Valid value is between `64` and `2048`.
 * @property tunnel1StartupAction The action to take when the establishing the tunnel for the first VPN connection. By default, your customer gateway device must initiate the IKE negotiation and bring up the tunnel. Specify start for AWS to initiate the IKE negotiation. Valid values are `add | start`.
 * @property tunnel2DpdTimeoutAction The action to take after DPD timeout occurs for the second VPN tunnel. Specify restart to restart the IKE initiation. Specify clear to end the IKE session. Valid values are `clear | none | restart`.
 * @property tunnel2DpdTimeoutSeconds The number of seconds after which a DPD timeout occurs for the second VPN tunnel. Valid value is equal or higher than `30`.
 * @property tunnel2EnableTunnelLifecycleControl Turn on or off tunnel endpoint lifecycle control feature for the second VPN tunnel. Valid values are `true | false`.
 * @property tunnel2IkeVersions The IKE versions that are permitted for the second VPN tunnel. Valid values are `ikev1 | ikev2`.
 * @property tunnel2InsideCidr The CIDR block of the inside IP addresses for the second VPN tunnel. Valid value is a size /30 CIDR block from the 169.254.0.0/16 range.
 * @property tunnel2InsideIpv6Cidr The range of inside IPv6 addresses for the second VPN tunnel. Supports only EC2 Transit Gateway. Valid value is a size /126 CIDR block from the local fd00::/8 range.
 * @property tunnel2LogOptions Options for logging VPN tunnel activity. See Log Options below for more details.
 * @property tunnel2Phase1DhGroupNumbers List of one or more Diffie-Hellman group numbers that are permitted for the second VPN tunnel for phase 1 IKE negotiations. Valid values are ` 2 | 14 | 15 | 16 | 17 | 18 | 19 | 20 | 21 | 22 | 23 | 24`.
 * @property tunnel2Phase1EncryptionAlgorithms List of one or more encryption algorithms that are permitted for the second VPN tunnel for phase 1 IKE negotiations. Valid values are `AES128 | AES256 | AES128-GCM-16 | AES256-GCM-16`.
 * @property tunnel2Phase1IntegrityAlgorithms One or more integrity algorithms that are permitted for the second VPN tunnel for phase 1 IKE negotiations. Valid values are `SHA1 | SHA2-256 | SHA2-384 | SHA2-512`.
 * @property tunnel2Phase1LifetimeSeconds The lifetime for phase 1 of the IKE negotiation for the second VPN tunnel, in seconds. Valid value is between `900` and `28800`.
 * @property tunnel2Phase2DhGroupNumbers List of one or more Diffie-Hellman group numbers that are permitted for the second VPN tunnel for phase 2 IKE negotiations. Valid values are `2 | 5 | 14 | 15 | 16 | 17 | 18 | 19 | 20 | 21 | 22 | 23 | 24`.
 * @property tunnel2Phase2EncryptionAlgorithms List of one or more encryption algorithms that are permitted for the second VPN tunnel for phase 2 IKE negotiations. Valid values are `AES128 | AES256 | AES128-GCM-16 | AES256-GCM-16`.
 * @property tunnel2Phase2IntegrityAlgorithms List of one or more integrity algorithms that are permitted for the second VPN tunnel for phase 2 IKE negotiations. Valid values are `SHA1 | SHA2-256 | SHA2-384 | SHA2-512`.
 * @property tunnel2Phase2LifetimeSeconds The lifetime for phase 2 of the IKE negotiation for the second VPN tunnel, in seconds. Valid value is between `900` and `3600`.
 * @property tunnel2PresharedKey The preshared key of the second VPN tunnel. The preshared key must be between 8 and 64 characters in length and cannot start with zero(0). Allowed characters are alphanumeric characters, periods(.) and underscores(_).
 * @property tunnel2RekeyFuzzPercentage The percentage of the rekey window for the second VPN tunnel (determined by `tunnel2_rekey_margin_time_seconds`) during which the rekey time is randomly selected. Valid value is between `0` and `100`.
 * @property tunnel2RekeyMarginTimeSeconds The margin time, in seconds, before the phase 2 lifetime expires, during which the AWS side of the second VPN connection performs an IKE rekey. The exact time of the rekey is randomly selected based on the value for `tunnel2_rekey_fuzz_percentage`. Valid value is between `60` and half of `tunnel2_phase2_lifetime_seconds`.
 * @property tunnel2ReplayWindowSize The number of packets in an IKE replay window for the second VPN tunnel. Valid value is between `64` and `2048`.
 * @property tunnel2StartupAction The action to take when the establishing the tunnel for the second VPN connection. By default, your customer gateway device must initiate the IKE negotiation and bring up the tunnel. Specify start for AWS to initiate the IKE negotiation. Valid values are `add | start`.
 * @property tunnelInsideIpVersion Indicate whether the VPN tunnels process IPv4 or IPv6 traffic. Valid values are `ipv4 | ipv6`. `ipv6` Supports only EC2 Transit Gateway.
 * @property type The type of VPN connection. The only type AWS supports at this time is "ipsec.1".
 * @property vpnGatewayId The ID of the Virtual Private Gateway.
 */
public data class VpnConnectionArgs(
    public val customerGatewayId: Output<String>? = null,
    public val enableAcceleration: Output<Boolean>? = null,
    public val localIpv4NetworkCidr: Output<String>? = null,
    public val localIpv6NetworkCidr: Output<String>? = null,
    public val outsideIpAddressType: Output<String>? = null,
    public val remoteIpv4NetworkCidr: Output<String>? = null,
    public val remoteIpv6NetworkCidr: Output<String>? = null,
    public val staticRoutesOnly: Output<Boolean>? = null,
    public val tags: Output<Map<String, String>>? = null,
    public val transitGatewayId: Output<String>? = null,
    public val transportTransitGatewayAttachmentId: Output<String>? = null,
    public val tunnel1DpdTimeoutAction: Output<String>? = null,
    public val tunnel1DpdTimeoutSeconds: Output<Int>? = null,
    public val tunnel1EnableTunnelLifecycleControl: Output<Boolean>? = null,
    public val tunnel1IkeVersions: Output<List<String>>? = null,
    public val tunnel1InsideCidr: Output<String>? = null,
    public val tunnel1InsideIpv6Cidr: Output<String>? = null,
    public val tunnel1LogOptions: Output<VpnConnectionTunnel1LogOptionsArgs>? = null,
    public val tunnel1Phase1DhGroupNumbers: Output<List<Int>>? = null,
    public val tunnel1Phase1EncryptionAlgorithms: Output<List<String>>? = null,
    public val tunnel1Phase1IntegrityAlgorithms: Output<List<String>>? = null,
    public val tunnel1Phase1LifetimeSeconds: Output<Int>? = null,
    public val tunnel1Phase2DhGroupNumbers: Output<List<Int>>? = null,
    public val tunnel1Phase2EncryptionAlgorithms: Output<List<String>>? = null,
    public val tunnel1Phase2IntegrityAlgorithms: Output<List<String>>? = null,
    public val tunnel1Phase2LifetimeSeconds: Output<Int>? = null,
    public val tunnel1PresharedKey: Output<String>? = null,
    public val tunnel1RekeyFuzzPercentage: Output<Int>? = null,
    public val tunnel1RekeyMarginTimeSeconds: Output<Int>? = null,
    public val tunnel1ReplayWindowSize: Output<Int>? = null,
    public val tunnel1StartupAction: Output<String>? = null,
    public val tunnel2DpdTimeoutAction: Output<String>? = null,
    public val tunnel2DpdTimeoutSeconds: Output<Int>? = null,
    public val tunnel2EnableTunnelLifecycleControl: Output<Boolean>? = null,
    public val tunnel2IkeVersions: Output<List<String>>? = null,
    public val tunnel2InsideCidr: Output<String>? = null,
    public val tunnel2InsideIpv6Cidr: Output<String>? = null,
    public val tunnel2LogOptions: Output<VpnConnectionTunnel2LogOptionsArgs>? = null,
    public val tunnel2Phase1DhGroupNumbers: Output<List<Int>>? = null,
    public val tunnel2Phase1EncryptionAlgorithms: Output<List<String>>? = null,
    public val tunnel2Phase1IntegrityAlgorithms: Output<List<String>>? = null,
    public val tunnel2Phase1LifetimeSeconds: Output<Int>? = null,
    public val tunnel2Phase2DhGroupNumbers: Output<List<Int>>? = null,
    public val tunnel2Phase2EncryptionAlgorithms: Output<List<String>>? = null,
    public val tunnel2Phase2IntegrityAlgorithms: Output<List<String>>? = null,
    public val tunnel2Phase2LifetimeSeconds: Output<Int>? = null,
    public val tunnel2PresharedKey: Output<String>? = null,
    public val tunnel2RekeyFuzzPercentage: Output<Int>? = null,
    public val tunnel2RekeyMarginTimeSeconds: Output<Int>? = null,
    public val tunnel2ReplayWindowSize: Output<Int>? = null,
    public val tunnel2StartupAction: Output<String>? = null,
    public val tunnelInsideIpVersion: Output<String>? = null,
    public val type: Output<String>? = null,
    public val vpnGatewayId: Output<String>? = null,
) : ConvertibleToJava<com.pulumi.aws.ec2.VpnConnectionArgs> {
    override fun toJava(): com.pulumi.aws.ec2.VpnConnectionArgs =
        com.pulumi.aws.ec2.VpnConnectionArgs.builder()
            .customerGatewayId(customerGatewayId?.applyValue({ args0 -> args0 }))
            .enableAcceleration(enableAcceleration?.applyValue({ args0 -> args0 }))
            .localIpv4NetworkCidr(localIpv4NetworkCidr?.applyValue({ args0 -> args0 }))
            .localIpv6NetworkCidr(localIpv6NetworkCidr?.applyValue({ args0 -> args0 }))
            .outsideIpAddressType(outsideIpAddressType?.applyValue({ args0 -> args0 }))
            .remoteIpv4NetworkCidr(remoteIpv4NetworkCidr?.applyValue({ args0 -> args0 }))
            .remoteIpv6NetworkCidr(remoteIpv6NetworkCidr?.applyValue({ args0 -> args0 }))
            .staticRoutesOnly(staticRoutesOnly?.applyValue({ args0 -> args0 }))
            .tags(tags?.applyValue({ args0 -> args0.map({ args0 -> args0.key.to(args0.value) }).toMap() }))
            .transitGatewayId(transitGatewayId?.applyValue({ args0 -> args0 }))
            .transportTransitGatewayAttachmentId(
                transportTransitGatewayAttachmentId?.applyValue({ args0 ->
                    args0
                }),
            )
            .tunnel1DpdTimeoutAction(tunnel1DpdTimeoutAction?.applyValue({ args0 -> args0 }))
            .tunnel1DpdTimeoutSeconds(tunnel1DpdTimeoutSeconds?.applyValue({ args0 -> args0 }))
            .tunnel1EnableTunnelLifecycleControl(
                tunnel1EnableTunnelLifecycleControl?.applyValue({ args0 ->
                    args0
                }),
            )
            .tunnel1IkeVersions(tunnel1IkeVersions?.applyValue({ args0 -> args0.map({ args0 -> args0 }) }))
            .tunnel1InsideCidr(tunnel1InsideCidr?.applyValue({ args0 -> args0 }))
            .tunnel1InsideIpv6Cidr(tunnel1InsideIpv6Cidr?.applyValue({ args0 -> args0 }))
            .tunnel1LogOptions(tunnel1LogOptions?.applyValue({ args0 -> args0.let({ args0 -> args0.toJava() }) }))
            .tunnel1Phase1DhGroupNumbers(
                tunnel1Phase1DhGroupNumbers?.applyValue({ args0 ->
                    args0.map({ args0 ->
                        args0
                    })
                }),
            )
            .tunnel1Phase1EncryptionAlgorithms(
                tunnel1Phase1EncryptionAlgorithms?.applyValue({ args0 ->
                    args0.map({ args0 -> args0 })
                }),
            )
            .tunnel1Phase1IntegrityAlgorithms(
                tunnel1Phase1IntegrityAlgorithms?.applyValue({ args0 ->
                    args0.map({ args0 -> args0 })
                }),
            )
            .tunnel1Phase1LifetimeSeconds(tunnel1Phase1LifetimeSeconds?.applyValue({ args0 -> args0 }))
            .tunnel1Phase2DhGroupNumbers(
                tunnel1Phase2DhGroupNumbers?.applyValue({ args0 ->
                    args0.map({ args0 ->
                        args0
                    })
                }),
            )
            .tunnel1Phase2EncryptionAlgorithms(
                tunnel1Phase2EncryptionAlgorithms?.applyValue({ args0 ->
                    args0.map({ args0 -> args0 })
                }),
            )
            .tunnel1Phase2IntegrityAlgorithms(
                tunnel1Phase2IntegrityAlgorithms?.applyValue({ args0 ->
                    args0.map({ args0 -> args0 })
                }),
            )
            .tunnel1Phase2LifetimeSeconds(tunnel1Phase2LifetimeSeconds?.applyValue({ args0 -> args0 }))
            .tunnel1PresharedKey(tunnel1PresharedKey?.applyValue({ args0 -> args0 }))
            .tunnel1RekeyFuzzPercentage(tunnel1RekeyFuzzPercentage?.applyValue({ args0 -> args0 }))
            .tunnel1RekeyMarginTimeSeconds(tunnel1RekeyMarginTimeSeconds?.applyValue({ args0 -> args0 }))
            .tunnel1ReplayWindowSize(tunnel1ReplayWindowSize?.applyValue({ args0 -> args0 }))
            .tunnel1StartupAction(tunnel1StartupAction?.applyValue({ args0 -> args0 }))
            .tunnel2DpdTimeoutAction(tunnel2DpdTimeoutAction?.applyValue({ args0 -> args0 }))
            .tunnel2DpdTimeoutSeconds(tunnel2DpdTimeoutSeconds?.applyValue({ args0 -> args0 }))
            .tunnel2EnableTunnelLifecycleControl(
                tunnel2EnableTunnelLifecycleControl?.applyValue({ args0 ->
                    args0
                }),
            )
            .tunnel2IkeVersions(tunnel2IkeVersions?.applyValue({ args0 -> args0.map({ args0 -> args0 }) }))
            .tunnel2InsideCidr(tunnel2InsideCidr?.applyValue({ args0 -> args0 }))
            .tunnel2InsideIpv6Cidr(tunnel2InsideIpv6Cidr?.applyValue({ args0 -> args0 }))
            .tunnel2LogOptions(tunnel2LogOptions?.applyValue({ args0 -> args0.let({ args0 -> args0.toJava() }) }))
            .tunnel2Phase1DhGroupNumbers(
                tunnel2Phase1DhGroupNumbers?.applyValue({ args0 ->
                    args0.map({ args0 ->
                        args0
                    })
                }),
            )
            .tunnel2Phase1EncryptionAlgorithms(
                tunnel2Phase1EncryptionAlgorithms?.applyValue({ args0 ->
                    args0.map({ args0 -> args0 })
                }),
            )
            .tunnel2Phase1IntegrityAlgorithms(
                tunnel2Phase1IntegrityAlgorithms?.applyValue({ args0 ->
                    args0.map({ args0 -> args0 })
                }),
            )
            .tunnel2Phase1LifetimeSeconds(tunnel2Phase1LifetimeSeconds?.applyValue({ args0 -> args0 }))
            .tunnel2Phase2DhGroupNumbers(
                tunnel2Phase2DhGroupNumbers?.applyValue({ args0 ->
                    args0.map({ args0 ->
                        args0
                    })
                }),
            )
            .tunnel2Phase2EncryptionAlgorithms(
                tunnel2Phase2EncryptionAlgorithms?.applyValue({ args0 ->
                    args0.map({ args0 -> args0 })
                }),
            )
            .tunnel2Phase2IntegrityAlgorithms(
                tunnel2Phase2IntegrityAlgorithms?.applyValue({ args0 ->
                    args0.map({ args0 -> args0 })
                }),
            )
            .tunnel2Phase2LifetimeSeconds(tunnel2Phase2LifetimeSeconds?.applyValue({ args0 -> args0 }))
            .tunnel2PresharedKey(tunnel2PresharedKey?.applyValue({ args0 -> args0 }))
            .tunnel2RekeyFuzzPercentage(tunnel2RekeyFuzzPercentage?.applyValue({ args0 -> args0 }))
            .tunnel2RekeyMarginTimeSeconds(tunnel2RekeyMarginTimeSeconds?.applyValue({ args0 -> args0 }))
            .tunnel2ReplayWindowSize(tunnel2ReplayWindowSize?.applyValue({ args0 -> args0 }))
            .tunnel2StartupAction(tunnel2StartupAction?.applyValue({ args0 -> args0 }))
            .tunnelInsideIpVersion(tunnelInsideIpVersion?.applyValue({ args0 -> args0 }))
            .type(type?.applyValue({ args0 -> args0 }))
            .vpnGatewayId(vpnGatewayId?.applyValue({ args0 -> args0 })).build()
}

/**
 * Builder for [VpnConnectionArgs].
 */
@PulumiTagMarker
public class VpnConnectionArgsBuilder internal constructor() {
    private var customerGatewayId: Output<String>? = null

    private var enableAcceleration: Output<Boolean>? = null

    private var localIpv4NetworkCidr: Output<String>? = null

    private var localIpv6NetworkCidr: Output<String>? = null

    private var outsideIpAddressType: Output<String>? = null

    private var remoteIpv4NetworkCidr: Output<String>? = null

    private var remoteIpv6NetworkCidr: Output<String>? = null

    private var staticRoutesOnly: Output<Boolean>? = null

    private var tags: Output<Map<String, String>>? = null

    private var transitGatewayId: Output<String>? = null

    private var transportTransitGatewayAttachmentId: Output<String>? = null

    private var tunnel1DpdTimeoutAction: Output<String>? = null

    private var tunnel1DpdTimeoutSeconds: Output<Int>? = null

    private var tunnel1EnableTunnelLifecycleControl: Output<Boolean>? = null

    private var tunnel1IkeVersions: Output<List<String>>? = null

    private var tunnel1InsideCidr: Output<String>? = null

    private var tunnel1InsideIpv6Cidr: Output<String>? = null

    private var tunnel1LogOptions: Output<VpnConnectionTunnel1LogOptionsArgs>? = null

    private var tunnel1Phase1DhGroupNumbers: Output<List<Int>>? = null

    private var tunnel1Phase1EncryptionAlgorithms: Output<List<String>>? = null

    private var tunnel1Phase1IntegrityAlgorithms: Output<List<String>>? = null

    private var tunnel1Phase1LifetimeSeconds: Output<Int>? = null

    private var tunnel1Phase2DhGroupNumbers: Output<List<Int>>? = null

    private var tunnel1Phase2EncryptionAlgorithms: Output<List<String>>? = null

    private var tunnel1Phase2IntegrityAlgorithms: Output<List<String>>? = null

    private var tunnel1Phase2LifetimeSeconds: Output<Int>? = null

    private var tunnel1PresharedKey: Output<String>? = null

    private var tunnel1RekeyFuzzPercentage: Output<Int>? = null

    private var tunnel1RekeyMarginTimeSeconds: Output<Int>? = null

    private var tunnel1ReplayWindowSize: Output<Int>? = null

    private var tunnel1StartupAction: Output<String>? = null

    private var tunnel2DpdTimeoutAction: Output<String>? = null

    private var tunnel2DpdTimeoutSeconds: Output<Int>? = null

    private var tunnel2EnableTunnelLifecycleControl: Output<Boolean>? = null

    private var tunnel2IkeVersions: Output<List<String>>? = null

    private var tunnel2InsideCidr: Output<String>? = null

    private var tunnel2InsideIpv6Cidr: Output<String>? = null

    private var tunnel2LogOptions: Output<VpnConnectionTunnel2LogOptionsArgs>? = null

    private var tunnel2Phase1DhGroupNumbers: Output<List<Int>>? = null

    private var tunnel2Phase1EncryptionAlgorithms: Output<List<String>>? = null

    private var tunnel2Phase1IntegrityAlgorithms: Output<List<String>>? = null

    private var tunnel2Phase1LifetimeSeconds: Output<Int>? = null

    private var tunnel2Phase2DhGroupNumbers: Output<List<Int>>? = null

    private var tunnel2Phase2EncryptionAlgorithms: Output<List<String>>? = null

    private var tunnel2Phase2IntegrityAlgorithms: Output<List<String>>? = null

    private var tunnel2Phase2LifetimeSeconds: Output<Int>? = null

    private var tunnel2PresharedKey: Output<String>? = null

    private var tunnel2RekeyFuzzPercentage: Output<Int>? = null

    private var tunnel2RekeyMarginTimeSeconds: Output<Int>? = null

    private var tunnel2ReplayWindowSize: Output<Int>? = null

    private var tunnel2StartupAction: Output<String>? = null

    private var tunnelInsideIpVersion: Output<String>? = null

    private var type: Output<String>? = null

    private var vpnGatewayId: Output<String>? = null

    /**
     * @param value The ID of the customer gateway.
     */
    @JvmName("uudsepgypqyjuvej")
    public suspend fun customerGatewayId(`value`: Output<String>) {
        this.customerGatewayId = value
    }

    /**
     * @param value Indicate whether to enable acceleration for the VPN connection. Supports only EC2 Transit Gateway.
     */
    @JvmName("lnbfjnmxhvnoeoku")
    public suspend fun enableAcceleration(`value`: Output<Boolean>) {
        this.enableAcceleration = value
    }

    /**
     * @param value The IPv4 CIDR on the customer gateway (on-premises) side of the VPN connection.
     */
    @JvmName("jsigqsmswpowqmvd")
    public suspend fun localIpv4NetworkCidr(`value`: Output<String>) {
        this.localIpv4NetworkCidr = value
    }

    /**
     * @param value The IPv6 CIDR on the customer gateway (on-premises) side of the VPN connection.
     */
    @JvmName("qnnttcsrnkdoldce")
    public suspend fun localIpv6NetworkCidr(`value`: Output<String>) {
        this.localIpv6NetworkCidr = value
    }

    /**
     * @param value Indicates if a Public S2S VPN or Private S2S VPN over AWS Direct Connect. Valid values are `PublicIpv4 | PrivateIpv4`
     */
    @JvmName("whpnfdmpdgbnccah")
    public suspend fun outsideIpAddressType(`value`: Output<String>) {
        this.outsideIpAddressType = value
    }

    /**
     * @param value The IPv4 CIDR on the AWS side of the VPN connection.
     */
    @JvmName("ddcokengwaeaiabg")
    public suspend fun remoteIpv4NetworkCidr(`value`: Output<String>) {
        this.remoteIpv4NetworkCidr = value
    }

    /**
     * @param value The IPv6 CIDR on the customer gateway (on-premises) side of the VPN connection.
     */
    @JvmName("xifgihsnsynsmlbj")
    public suspend fun remoteIpv6NetworkCidr(`value`: Output<String>) {
        this.remoteIpv6NetworkCidr = value
    }

    /**
     * @param value Whether the VPN connection uses static routes exclusively. Static routes must be used for devices that don't support BGP.
     */
    @JvmName("jbejhupttptgmexl")
    public suspend fun staticRoutesOnly(`value`: Output<Boolean>) {
        this.staticRoutesOnly = value
    }

    /**
     * @param value Tags to apply to the connection. If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
     */
    @JvmName("vjqxbyadjmdlwfwb")
    public suspend fun tags(`value`: Output<Map<String, String>>) {
        this.tags = value
    }

    /**
     * @param value The ID of the EC2 Transit Gateway.
     */
    @JvmName("gpjrteoiafjwvyth")
    public suspend fun transitGatewayId(`value`: Output<String>) {
        this.transitGatewayId = value
    }

    /**
     * @param value . The attachment ID of the Transit Gateway attachment to Direct Connect Gateway. The ID is obtained through a data source only.
     */
    @JvmName("yarojaeodqfdgmrs")
    public suspend fun transportTransitGatewayAttachmentId(`value`: Output<String>) {
        this.transportTransitGatewayAttachmentId = value
    }

    /**
     * @param value The action to take after DPD timeout occurs for the first VPN tunnel. Specify restart to restart the IKE initiation. Specify clear to end the IKE session. Valid values are `clear | none | restart`.
     */
    @JvmName("quvkbspuydsmqqry")
    public suspend fun tunnel1DpdTimeoutAction(`value`: Output<String>) {
        this.tunnel1DpdTimeoutAction = value
    }

    /**
     * @param value The number of seconds after which a DPD timeout occurs for the first VPN tunnel. Valid value is equal or higher than `30`.
     */
    @JvmName("yxluvywwryoaxqpp")
    public suspend fun tunnel1DpdTimeoutSeconds(`value`: Output<Int>) {
        this.tunnel1DpdTimeoutSeconds = value
    }

    /**
     * @param value Turn on or off tunnel endpoint lifecycle control feature for the first VPN tunnel. Valid values are `true | false`.
     */
    @JvmName("hniqloomftgkjkks")
    public suspend fun tunnel1EnableTunnelLifecycleControl(`value`: Output<Boolean>) {
        this.tunnel1EnableTunnelLifecycleControl = value
    }

    /**
     * @param value The IKE versions that are permitted for the first VPN tunnel. Valid values are `ikev1 | ikev2`.
     */
    @JvmName("pydejlkmgkueseau")
    public suspend fun tunnel1IkeVersions(`value`: Output<List<String>>) {
        this.tunnel1IkeVersions = value
    }

    @JvmName("rahjabldciqnceqn")
    public suspend fun tunnel1IkeVersions(vararg values: Output<String>) {
        this.tunnel1IkeVersions = Output.all(values.asList())
    }

    /**
     * @param values The IKE versions that are permitted for the first VPN tunnel. Valid values are `ikev1 | ikev2`.
     */
    @JvmName("gxfccoitpdppplkr")
    public suspend fun tunnel1IkeVersions(values: List<Output<String>>) {
        this.tunnel1IkeVersions = Output.all(values)
    }

    /**
     * @param value The CIDR block of the inside IP addresses for the first VPN tunnel. Valid value is a size /30 CIDR block from the 169.254.0.0/16 range.
     */
    @JvmName("dyyvvdnsytrreqnf")
    public suspend fun tunnel1InsideCidr(`value`: Output<String>) {
        this.tunnel1InsideCidr = value
    }

    /**
     * @param value The range of inside IPv6 addresses for the first VPN tunnel. Supports only EC2 Transit Gateway. Valid value is a size /126 CIDR block from the local fd00::/8 range.
     */
    @JvmName("okidawcqklkxxnfa")
    public suspend fun tunnel1InsideIpv6Cidr(`value`: Output<String>) {
        this.tunnel1InsideIpv6Cidr = value
    }

    /**
     * @param value Options for logging VPN tunnel activity. See Log Options below for more details.
     */
    @JvmName("ipmwkweeybfjcsqt")
    public suspend fun tunnel1LogOptions(`value`: Output<VpnConnectionTunnel1LogOptionsArgs>) {
        this.tunnel1LogOptions = value
    }

    /**
     * @param value List of one or more Diffie-Hellman group numbers that are permitted for the first VPN tunnel for phase 1 IKE negotiations. Valid values are ` 2 | 14 | 15 | 16 | 17 | 18 | 19 | 20 | 21 | 22 | 23 | 24`.
     */
    @JvmName("qdnrsahygnvovhyi")
    public suspend fun tunnel1Phase1DhGroupNumbers(`value`: Output<List<Int>>) {
        this.tunnel1Phase1DhGroupNumbers = value
    }

    @JvmName("yeqynhvjnqexyauf")
    public suspend fun tunnel1Phase1DhGroupNumbers(vararg values: Output<Int>) {
        this.tunnel1Phase1DhGroupNumbers = Output.all(values.asList())
    }

    /**
     * @param values List of one or more Diffie-Hellman group numbers that are permitted for the first VPN tunnel for phase 1 IKE negotiations. Valid values are ` 2 | 14 | 15 | 16 | 17 | 18 | 19 | 20 | 21 | 22 | 23 | 24`.
     */
    @JvmName("kydmvglxjtsskrgp")
    public suspend fun tunnel1Phase1DhGroupNumbers(values: List<Output<Int>>) {
        this.tunnel1Phase1DhGroupNumbers = Output.all(values)
    }

    /**
     * @param value List of one or more encryption algorithms that are permitted for the first VPN tunnel for phase 1 IKE negotiations. Valid values are `AES128 | AES256 | AES128-GCM-16 | AES256-GCM-16`.
     */
    @JvmName("wqihvhhqyumyfkel")
    public suspend fun tunnel1Phase1EncryptionAlgorithms(`value`: Output<List<String>>) {
        this.tunnel1Phase1EncryptionAlgorithms = value
    }

    @JvmName("oplaeeboywgusouu")
    public suspend fun tunnel1Phase1EncryptionAlgorithms(vararg values: Output<String>) {
        this.tunnel1Phase1EncryptionAlgorithms = Output.all(values.asList())
    }

    /**
     * @param values List of one or more encryption algorithms that are permitted for the first VPN tunnel for phase 1 IKE negotiations. Valid values are `AES128 | AES256 | AES128-GCM-16 | AES256-GCM-16`.
     */
    @JvmName("dgxdwkmgnpjudqbh")
    public suspend fun tunnel1Phase1EncryptionAlgorithms(values: List<Output<String>>) {
        this.tunnel1Phase1EncryptionAlgorithms = Output.all(values)
    }

    /**
     * @param value One or more integrity algorithms that are permitted for the first VPN tunnel for phase 1 IKE negotiations. Valid values are `SHA1 | SHA2-256 | SHA2-384 | SHA2-512`.
     */
    @JvmName("bevpqfgcickfjjjd")
    public suspend fun tunnel1Phase1IntegrityAlgorithms(`value`: Output<List<String>>) {
        this.tunnel1Phase1IntegrityAlgorithms = value
    }

    @JvmName("yclfuxiilrhutbaa")
    public suspend fun tunnel1Phase1IntegrityAlgorithms(vararg values: Output<String>) {
        this.tunnel1Phase1IntegrityAlgorithms = Output.all(values.asList())
    }

    /**
     * @param values One or more integrity algorithms that are permitted for the first VPN tunnel for phase 1 IKE negotiations. Valid values are `SHA1 | SHA2-256 | SHA2-384 | SHA2-512`.
     */
    @JvmName("sfgenhyljdusioxo")
    public suspend fun tunnel1Phase1IntegrityAlgorithms(values: List<Output<String>>) {
        this.tunnel1Phase1IntegrityAlgorithms = Output.all(values)
    }

    /**
     * @param value The lifetime for phase 1 of the IKE negotiation for the first VPN tunnel, in seconds. Valid value is between `900` and `28800`.
     */
    @JvmName("hjbuledtcypexevl")
    public suspend fun tunnel1Phase1LifetimeSeconds(`value`: Output<Int>) {
        this.tunnel1Phase1LifetimeSeconds = value
    }

    /**
     * @param value List of one or more Diffie-Hellman group numbers that are permitted for the first VPN tunnel for phase 2 IKE negotiations. Valid values are `2 | 5 | 14 | 15 | 16 | 17 | 18 | 19 | 20 | 21 | 22 | 23 | 24`.
     */
    @JvmName("mbvpvofiwfiiofac")
    public suspend fun tunnel1Phase2DhGroupNumbers(`value`: Output<List<Int>>) {
        this.tunnel1Phase2DhGroupNumbers = value
    }

    @JvmName("hqfmagvkklyqqalp")
    public suspend fun tunnel1Phase2DhGroupNumbers(vararg values: Output<Int>) {
        this.tunnel1Phase2DhGroupNumbers = Output.all(values.asList())
    }

    /**
     * @param values List of one or more Diffie-Hellman group numbers that are permitted for the first VPN tunnel for phase 2 IKE negotiations. Valid values are `2 | 5 | 14 | 15 | 16 | 17 | 18 | 19 | 20 | 21 | 22 | 23 | 24`.
     */
    @JvmName("aclfhssthkbsihtc")
    public suspend fun tunnel1Phase2DhGroupNumbers(values: List<Output<Int>>) {
        this.tunnel1Phase2DhGroupNumbers = Output.all(values)
    }

    /**
     * @param value List of one or more encryption algorithms that are permitted for the first VPN tunnel for phase 2 IKE negotiations. Valid values are `AES128 | AES256 | AES128-GCM-16 | AES256-GCM-16`.
     */
    @JvmName("hjkdiicqiqsrnrsh")
    public suspend fun tunnel1Phase2EncryptionAlgorithms(`value`: Output<List<String>>) {
        this.tunnel1Phase2EncryptionAlgorithms = value
    }

    @JvmName("usekvylqdunpaihk")
    public suspend fun tunnel1Phase2EncryptionAlgorithms(vararg values: Output<String>) {
        this.tunnel1Phase2EncryptionAlgorithms = Output.all(values.asList())
    }

    /**
     * @param values List of one or more encryption algorithms that are permitted for the first VPN tunnel for phase 2 IKE negotiations. Valid values are `AES128 | AES256 | AES128-GCM-16 | AES256-GCM-16`.
     */
    @JvmName("kfiwfacxudvadncv")
    public suspend fun tunnel1Phase2EncryptionAlgorithms(values: List<Output<String>>) {
        this.tunnel1Phase2EncryptionAlgorithms = Output.all(values)
    }

    /**
     * @param value List of one or more integrity algorithms that are permitted for the first VPN tunnel for phase 2 IKE negotiations. Valid values are `SHA1 | SHA2-256 | SHA2-384 | SHA2-512`.
     */
    @JvmName("kmteliohttdgygre")
    public suspend fun tunnel1Phase2IntegrityAlgorithms(`value`: Output<List<String>>) {
        this.tunnel1Phase2IntegrityAlgorithms = value
    }

    @JvmName("vngkcyrwdtftuyib")
    public suspend fun tunnel1Phase2IntegrityAlgorithms(vararg values: Output<String>) {
        this.tunnel1Phase2IntegrityAlgorithms = Output.all(values.asList())
    }

    /**
     * @param values List of one or more integrity algorithms that are permitted for the first VPN tunnel for phase 2 IKE negotiations. Valid values are `SHA1 | SHA2-256 | SHA2-384 | SHA2-512`.
     */
    @JvmName("opmfhpngjlkuidjk")
    public suspend fun tunnel1Phase2IntegrityAlgorithms(values: List<Output<String>>) {
        this.tunnel1Phase2IntegrityAlgorithms = Output.all(values)
    }

    /**
     * @param value The lifetime for phase 2 of the IKE negotiation for the first VPN tunnel, in seconds. Valid value is between `900` and `3600`.
     */
    @JvmName("qjmaiuleggicghmo")
    public suspend fun tunnel1Phase2LifetimeSeconds(`value`: Output<Int>) {
        this.tunnel1Phase2LifetimeSeconds = value
    }

    /**
     * @param value The preshared key of the first VPN tunnel. The preshared key must be between 8 and 64 characters in length and cannot start with zero(0). Allowed characters are alphanumeric characters, periods(.) and underscores(_).
     */
    @JvmName("idnjgmjbnyamqcfk")
    public suspend fun tunnel1PresharedKey(`value`: Output<String>) {
        this.tunnel1PresharedKey = value
    }

    /**
     * @param value The percentage of the rekey window for the first VPN tunnel (determined by `tunnel1_rekey_margin_time_seconds`) during which the rekey time is randomly selected. Valid value is between `0` and `100`.
     */
    @JvmName("lsgltctwkviufskr")
    public suspend fun tunnel1RekeyFuzzPercentage(`value`: Output<Int>) {
        this.tunnel1RekeyFuzzPercentage = value
    }

    /**
     * @param value The margin time, in seconds, before the phase 2 lifetime expires, during which the AWS side of the first VPN connection performs an IKE rekey. The exact time of the rekey is randomly selected based on the value for `tunnel1_rekey_fuzz_percentage`. Valid value is between `60` and half of `tunnel1_phase2_lifetime_seconds`.
     */
    @JvmName("osrtwjltveyokmmv")
    public suspend fun tunnel1RekeyMarginTimeSeconds(`value`: Output<Int>) {
        this.tunnel1RekeyMarginTimeSeconds = value
    }

    /**
     * @param value The number of packets in an IKE replay window for the first VPN tunnel. Valid value is between `64` and `2048`.
     */
    @JvmName("jbdpjevtsmaeycyp")
    public suspend fun tunnel1ReplayWindowSize(`value`: Output<Int>) {
        this.tunnel1ReplayWindowSize = value
    }

    /**
     * @param value The action to take when the establishing the tunnel for the first VPN connection. By default, your customer gateway device must initiate the IKE negotiation and bring up the tunnel. Specify start for AWS to initiate the IKE negotiation. Valid values are `add | start`.
     */
    @JvmName("ijgjgpxmfpsflgux")
    public suspend fun tunnel1StartupAction(`value`: Output<String>) {
        this.tunnel1StartupAction = value
    }

    /**
     * @param value The action to take after DPD timeout occurs for the second VPN tunnel. Specify restart to restart the IKE initiation. Specify clear to end the IKE session. Valid values are `clear | none | restart`.
     */
    @JvmName("upwogetshrxdbxhv")
    public suspend fun tunnel2DpdTimeoutAction(`value`: Output<String>) {
        this.tunnel2DpdTimeoutAction = value
    }

    /**
     * @param value The number of seconds after which a DPD timeout occurs for the second VPN tunnel. Valid value is equal or higher than `30`.
     */
    @JvmName("ehwboidjtffdphkh")
    public suspend fun tunnel2DpdTimeoutSeconds(`value`: Output<Int>) {
        this.tunnel2DpdTimeoutSeconds = value
    }

    /**
     * @param value Turn on or off tunnel endpoint lifecycle control feature for the second VPN tunnel. Valid values are `true | false`.
     */
    @JvmName("cwwxnmgijpcslbsn")
    public suspend fun tunnel2EnableTunnelLifecycleControl(`value`: Output<Boolean>) {
        this.tunnel2EnableTunnelLifecycleControl = value
    }

    /**
     * @param value The IKE versions that are permitted for the second VPN tunnel. Valid values are `ikev1 | ikev2`.
     */
    @JvmName("lbjgsithgmmkbtcy")
    public suspend fun tunnel2IkeVersions(`value`: Output<List<String>>) {
        this.tunnel2IkeVersions = value
    }

    @JvmName("viqdpwuaagembila")
    public suspend fun tunnel2IkeVersions(vararg values: Output<String>) {
        this.tunnel2IkeVersions = Output.all(values.asList())
    }

    /**
     * @param values The IKE versions that are permitted for the second VPN tunnel. Valid values are `ikev1 | ikev2`.
     */
    @JvmName("kxangvolbtqelmgf")
    public suspend fun tunnel2IkeVersions(values: List<Output<String>>) {
        this.tunnel2IkeVersions = Output.all(values)
    }

    /**
     * @param value The CIDR block of the inside IP addresses for the second VPN tunnel. Valid value is a size /30 CIDR block from the 169.254.0.0/16 range.
     */
    @JvmName("pygoboajryxopfqp")
    public suspend fun tunnel2InsideCidr(`value`: Output<String>) {
        this.tunnel2InsideCidr = value
    }

    /**
     * @param value The range of inside IPv6 addresses for the second VPN tunnel. Supports only EC2 Transit Gateway. Valid value is a size /126 CIDR block from the local fd00::/8 range.
     */
    @JvmName("gxrjbdscseptsdsg")
    public suspend fun tunnel2InsideIpv6Cidr(`value`: Output<String>) {
        this.tunnel2InsideIpv6Cidr = value
    }

    /**
     * @param value Options for logging VPN tunnel activity. See Log Options below for more details.
     */
    @JvmName("ciusuukukxgghmdl")
    public suspend fun tunnel2LogOptions(`value`: Output<VpnConnectionTunnel2LogOptionsArgs>) {
        this.tunnel2LogOptions = value
    }

    /**
     * @param value List of one or more Diffie-Hellman group numbers that are permitted for the second VPN tunnel for phase 1 IKE negotiations. Valid values are ` 2 | 14 | 15 | 16 | 17 | 18 | 19 | 20 | 21 | 22 | 23 | 24`.
     */
    @JvmName("hqbdovulqulmnfqr")
    public suspend fun tunnel2Phase1DhGroupNumbers(`value`: Output<List<Int>>) {
        this.tunnel2Phase1DhGroupNumbers = value
    }

    @JvmName("srcevtewgiwcpoie")
    public suspend fun tunnel2Phase1DhGroupNumbers(vararg values: Output<Int>) {
        this.tunnel2Phase1DhGroupNumbers = Output.all(values.asList())
    }

    /**
     * @param values List of one or more Diffie-Hellman group numbers that are permitted for the second VPN tunnel for phase 1 IKE negotiations. Valid values are ` 2 | 14 | 15 | 16 | 17 | 18 | 19 | 20 | 21 | 22 | 23 | 24`.
     */
    @JvmName("jfkgobmkwbgesbgm")
    public suspend fun tunnel2Phase1DhGroupNumbers(values: List<Output<Int>>) {
        this.tunnel2Phase1DhGroupNumbers = Output.all(values)
    }

    /**
     * @param value List of one or more encryption algorithms that are permitted for the second VPN tunnel for phase 1 IKE negotiations. Valid values are `AES128 | AES256 | AES128-GCM-16 | AES256-GCM-16`.
     */
    @JvmName("galysbnavlvpwjoe")
    public suspend fun tunnel2Phase1EncryptionAlgorithms(`value`: Output<List<String>>) {
        this.tunnel2Phase1EncryptionAlgorithms = value
    }

    @JvmName("nehltfptydurfbxr")
    public suspend fun tunnel2Phase1EncryptionAlgorithms(vararg values: Output<String>) {
        this.tunnel2Phase1EncryptionAlgorithms = Output.all(values.asList())
    }

    /**
     * @param values List of one or more encryption algorithms that are permitted for the second VPN tunnel for phase 1 IKE negotiations. Valid values are `AES128 | AES256 | AES128-GCM-16 | AES256-GCM-16`.
     */
    @JvmName("dmnpcyfnaoomevya")
    public suspend fun tunnel2Phase1EncryptionAlgorithms(values: List<Output<String>>) {
        this.tunnel2Phase1EncryptionAlgorithms = Output.all(values)
    }

    /**
     * @param value One or more integrity algorithms that are permitted for the second VPN tunnel for phase 1 IKE negotiations. Valid values are `SHA1 | SHA2-256 | SHA2-384 | SHA2-512`.
     */
    @JvmName("kwbjbfqldyulcbtc")
    public suspend fun tunnel2Phase1IntegrityAlgorithms(`value`: Output<List<String>>) {
        this.tunnel2Phase1IntegrityAlgorithms = value
    }

    @JvmName("snbmwhoyfdokbexn")
    public suspend fun tunnel2Phase1IntegrityAlgorithms(vararg values: Output<String>) {
        this.tunnel2Phase1IntegrityAlgorithms = Output.all(values.asList())
    }

    /**
     * @param values One or more integrity algorithms that are permitted for the second VPN tunnel for phase 1 IKE negotiations. Valid values are `SHA1 | SHA2-256 | SHA2-384 | SHA2-512`.
     */
    @JvmName("doqegmsyhbjltxjk")
    public suspend fun tunnel2Phase1IntegrityAlgorithms(values: List<Output<String>>) {
        this.tunnel2Phase1IntegrityAlgorithms = Output.all(values)
    }

    /**
     * @param value The lifetime for phase 1 of the IKE negotiation for the second VPN tunnel, in seconds. Valid value is between `900` and `28800`.
     */
    @JvmName("tarktchveafjxlak")
    public suspend fun tunnel2Phase1LifetimeSeconds(`value`: Output<Int>) {
        this.tunnel2Phase1LifetimeSeconds = value
    }

    /**
     * @param value List of one or more Diffie-Hellman group numbers that are permitted for the second VPN tunnel for phase 2 IKE negotiations. Valid values are `2 | 5 | 14 | 15 | 16 | 17 | 18 | 19 | 20 | 21 | 22 | 23 | 24`.
     */
    @JvmName("mwdqymuochqfqupu")
    public suspend fun tunnel2Phase2DhGroupNumbers(`value`: Output<List<Int>>) {
        this.tunnel2Phase2DhGroupNumbers = value
    }

    @JvmName("ncxswgiabubiaisu")
    public suspend fun tunnel2Phase2DhGroupNumbers(vararg values: Output<Int>) {
        this.tunnel2Phase2DhGroupNumbers = Output.all(values.asList())
    }

    /**
     * @param values List of one or more Diffie-Hellman group numbers that are permitted for the second VPN tunnel for phase 2 IKE negotiations. Valid values are `2 | 5 | 14 | 15 | 16 | 17 | 18 | 19 | 20 | 21 | 22 | 23 | 24`.
     */
    @JvmName("fctamqhvljfkksui")
    public suspend fun tunnel2Phase2DhGroupNumbers(values: List<Output<Int>>) {
        this.tunnel2Phase2DhGroupNumbers = Output.all(values)
    }

    /**
     * @param value List of one or more encryption algorithms that are permitted for the second VPN tunnel for phase 2 IKE negotiations. Valid values are `AES128 | AES256 | AES128-GCM-16 | AES256-GCM-16`.
     */
    @JvmName("rclcksrjuyjsgjbg")
    public suspend fun tunnel2Phase2EncryptionAlgorithms(`value`: Output<List<String>>) {
        this.tunnel2Phase2EncryptionAlgorithms = value
    }

    @JvmName("krjllurxetiqmhhw")
    public suspend fun tunnel2Phase2EncryptionAlgorithms(vararg values: Output<String>) {
        this.tunnel2Phase2EncryptionAlgorithms = Output.all(values.asList())
    }

    /**
     * @param values List of one or more encryption algorithms that are permitted for the second VPN tunnel for phase 2 IKE negotiations. Valid values are `AES128 | AES256 | AES128-GCM-16 | AES256-GCM-16`.
     */
    @JvmName("fcytxvmhtnegseai")
    public suspend fun tunnel2Phase2EncryptionAlgorithms(values: List<Output<String>>) {
        this.tunnel2Phase2EncryptionAlgorithms = Output.all(values)
    }

    /**
     * @param value List of one or more integrity algorithms that are permitted for the second VPN tunnel for phase 2 IKE negotiations. Valid values are `SHA1 | SHA2-256 | SHA2-384 | SHA2-512`.
     */
    @JvmName("kajnvemqymflckfj")
    public suspend fun tunnel2Phase2IntegrityAlgorithms(`value`: Output<List<String>>) {
        this.tunnel2Phase2IntegrityAlgorithms = value
    }

    @JvmName("rrvylvobkngcmymi")
    public suspend fun tunnel2Phase2IntegrityAlgorithms(vararg values: Output<String>) {
        this.tunnel2Phase2IntegrityAlgorithms = Output.all(values.asList())
    }

    /**
     * @param values List of one or more integrity algorithms that are permitted for the second VPN tunnel for phase 2 IKE negotiations. Valid values are `SHA1 | SHA2-256 | SHA2-384 | SHA2-512`.
     */
    @JvmName("heafkojmhigphrqj")
    public suspend fun tunnel2Phase2IntegrityAlgorithms(values: List<Output<String>>) {
        this.tunnel2Phase2IntegrityAlgorithms = Output.all(values)
    }

    /**
     * @param value The lifetime for phase 2 of the IKE negotiation for the second VPN tunnel, in seconds. Valid value is between `900` and `3600`.
     */
    @JvmName("wmofwrpaunwosqay")
    public suspend fun tunnel2Phase2LifetimeSeconds(`value`: Output<Int>) {
        this.tunnel2Phase2LifetimeSeconds = value
    }

    /**
     * @param value The preshared key of the second VPN tunnel. The preshared key must be between 8 and 64 characters in length and cannot start with zero(0). Allowed characters are alphanumeric characters, periods(.) and underscores(_).
     */
    @JvmName("ycetrbsspkgdboar")
    public suspend fun tunnel2PresharedKey(`value`: Output<String>) {
        this.tunnel2PresharedKey = value
    }

    /**
     * @param value The percentage of the rekey window for the second VPN tunnel (determined by `tunnel2_rekey_margin_time_seconds`) during which the rekey time is randomly selected. Valid value is between `0` and `100`.
     */
    @JvmName("ipspxytrqyuugsnd")
    public suspend fun tunnel2RekeyFuzzPercentage(`value`: Output<Int>) {
        this.tunnel2RekeyFuzzPercentage = value
    }

    /**
     * @param value The margin time, in seconds, before the phase 2 lifetime expires, during which the AWS side of the second VPN connection performs an IKE rekey. The exact time of the rekey is randomly selected based on the value for `tunnel2_rekey_fuzz_percentage`. Valid value is between `60` and half of `tunnel2_phase2_lifetime_seconds`.
     */
    @JvmName("xtsqomolyfvluidt")
    public suspend fun tunnel2RekeyMarginTimeSeconds(`value`: Output<Int>) {
        this.tunnel2RekeyMarginTimeSeconds = value
    }

    /**
     * @param value The number of packets in an IKE replay window for the second VPN tunnel. Valid value is between `64` and `2048`.
     */
    @JvmName("dnvmggduotlnuwub")
    public suspend fun tunnel2ReplayWindowSize(`value`: Output<Int>) {
        this.tunnel2ReplayWindowSize = value
    }

    /**
     * @param value The action to take when the establishing the tunnel for the second VPN connection. By default, your customer gateway device must initiate the IKE negotiation and bring up the tunnel. Specify start for AWS to initiate the IKE negotiation. Valid values are `add | start`.
     */
    @JvmName("mwbnpgxuljkmkqvr")
    public suspend fun tunnel2StartupAction(`value`: Output<String>) {
        this.tunnel2StartupAction = value
    }

    /**
     * @param value Indicate whether the VPN tunnels process IPv4 or IPv6 traffic. Valid values are `ipv4 | ipv6`. `ipv6` Supports only EC2 Transit Gateway.
     */
    @JvmName("xvutrnlqmsaucsal")
    public suspend fun tunnelInsideIpVersion(`value`: Output<String>) {
        this.tunnelInsideIpVersion = value
    }

    /**
     * @param value The type of VPN connection. The only type AWS supports at this time is "ipsec.1".
     */
    @JvmName("etrsqefiruqmfdyh")
    public suspend fun type(`value`: Output<String>) {
        this.type = value
    }

    /**
     * @param value The ID of the Virtual Private Gateway.
     */
    @JvmName("ksafkptjbymuyomr")
    public suspend fun vpnGatewayId(`value`: Output<String>) {
        this.vpnGatewayId = value
    }

    /**
     * @param value The ID of the customer gateway.
     */
    @JvmName("gbhlljqbmhdrscra")
    public suspend fun customerGatewayId(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.customerGatewayId = mapped
    }

    /**
     * @param value Indicate whether to enable acceleration for the VPN connection. Supports only EC2 Transit Gateway.
     */
    @JvmName("xbgxwpaosfgdhdaw")
    public suspend fun enableAcceleration(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.enableAcceleration = mapped
    }

    /**
     * @param value The IPv4 CIDR on the customer gateway (on-premises) side of the VPN connection.
     */
    @JvmName("lxgmgcvadymxkacs")
    public suspend fun localIpv4NetworkCidr(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.localIpv4NetworkCidr = mapped
    }

    /**
     * @param value The IPv6 CIDR on the customer gateway (on-premises) side of the VPN connection.
     */
    @JvmName("atbbfslykihfqcfe")
    public suspend fun localIpv6NetworkCidr(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.localIpv6NetworkCidr = mapped
    }

    /**
     * @param value Indicates if a Public S2S VPN or Private S2S VPN over AWS Direct Connect. Valid values are `PublicIpv4 | PrivateIpv4`
     */
    @JvmName("wauhggxcjbjmynfe")
    public suspend fun outsideIpAddressType(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.outsideIpAddressType = mapped
    }

    /**
     * @param value The IPv4 CIDR on the AWS side of the VPN connection.
     */
    @JvmName("lpttxpsvxbinwomh")
    public suspend fun remoteIpv4NetworkCidr(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.remoteIpv4NetworkCidr = mapped
    }

    /**
     * @param value The IPv6 CIDR on the customer gateway (on-premises) side of the VPN connection.
     */
    @JvmName("ugppukprnhxpdpes")
    public suspend fun remoteIpv6NetworkCidr(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.remoteIpv6NetworkCidr = mapped
    }

    /**
     * @param value Whether the VPN connection uses static routes exclusively. Static routes must be used for devices that don't support BGP.
     */
    @JvmName("mammaoxqrswchchu")
    public suspend fun staticRoutesOnly(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.staticRoutesOnly = mapped
    }

    /**
     * @param value Tags to apply to the connection. If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
     */
    @JvmName("xqyplmbhspxofjor")
    public suspend fun tags(`value`: Map<String, String>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.tags = mapped
    }

    /**
     * @param values Tags to apply to the connection. If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
     */
    @JvmName("eqiifbyowfgroydi")
    public fun tags(vararg values: Pair<String, String>) {
        val toBeMapped = values.toMap()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.tags = mapped
    }

    /**
     * @param value The ID of the EC2 Transit Gateway.
     */
    @JvmName("cccynflyyooymlfg")
    public suspend fun transitGatewayId(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.transitGatewayId = mapped
    }

    /**
     * @param value . The attachment ID of the Transit Gateway attachment to Direct Connect Gateway. The ID is obtained through a data source only.
     */
    @JvmName("rfgetefrgpsedcfs")
    public suspend fun transportTransitGatewayAttachmentId(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.transportTransitGatewayAttachmentId = mapped
    }

    /**
     * @param value The action to take after DPD timeout occurs for the first VPN tunnel. Specify restart to restart the IKE initiation. Specify clear to end the IKE session. Valid values are `clear | none | restart`.
     */
    @JvmName("aajxnysqoxwnlekm")
    public suspend fun tunnel1DpdTimeoutAction(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.tunnel1DpdTimeoutAction = mapped
    }

    /**
     * @param value The number of seconds after which a DPD timeout occurs for the first VPN tunnel. Valid value is equal or higher than `30`.
     */
    @JvmName("mtaovlfihqifxxcs")
    public suspend fun tunnel1DpdTimeoutSeconds(`value`: Int?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.tunnel1DpdTimeoutSeconds = mapped
    }

    /**
     * @param value Turn on or off tunnel endpoint lifecycle control feature for the first VPN tunnel. Valid values are `true | false`.
     */
    @JvmName("xhtjvebocpnlfvuh")
    public suspend fun tunnel1EnableTunnelLifecycleControl(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.tunnel1EnableTunnelLifecycleControl = mapped
    }

    /**
     * @param value The IKE versions that are permitted for the first VPN tunnel. Valid values are `ikev1 | ikev2`.
     */
    @JvmName("boobbnekfbijxotb")
    public suspend fun tunnel1IkeVersions(`value`: List<String>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.tunnel1IkeVersions = mapped
    }

    /**
     * @param values The IKE versions that are permitted for the first VPN tunnel. Valid values are `ikev1 | ikev2`.
     */
    @JvmName("twuifmkinprbfwsj")
    public suspend fun tunnel1IkeVersions(vararg values: String) {
        val toBeMapped = values.toList()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.tunnel1IkeVersions = mapped
    }

    /**
     * @param value The CIDR block of the inside IP addresses for the first VPN tunnel. Valid value is a size /30 CIDR block from the 169.254.0.0/16 range.
     */
    @JvmName("uxqnctvnpkpjmwah")
    public suspend fun tunnel1InsideCidr(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.tunnel1InsideCidr = mapped
    }

    /**
     * @param value The range of inside IPv6 addresses for the first VPN tunnel. Supports only EC2 Transit Gateway. Valid value is a size /126 CIDR block from the local fd00::/8 range.
     */
    @JvmName("jxrqjjipbbajwktd")
    public suspend fun tunnel1InsideIpv6Cidr(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.tunnel1InsideIpv6Cidr = mapped
    }

    /**
     * @param value Options for logging VPN tunnel activity. See Log Options below for more details.
     */
    @JvmName("pfprxydcxifewaky")
    public suspend fun tunnel1LogOptions(`value`: VpnConnectionTunnel1LogOptionsArgs?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.tunnel1LogOptions = mapped
    }

    /**
     * @param argument Options for logging VPN tunnel activity. See Log Options below for more details.
     */
    @JvmName("yrfvttwsdetsjfeb")
    public suspend
    fun tunnel1LogOptions(argument: suspend VpnConnectionTunnel1LogOptionsArgsBuilder.() -> Unit) {
        val toBeMapped = VpnConnectionTunnel1LogOptionsArgsBuilder().applySuspend { argument() }.build()
        val mapped = of(toBeMapped)
        this.tunnel1LogOptions = mapped
    }

    /**
     * @param value List of one or more Diffie-Hellman group numbers that are permitted for the first VPN tunnel for phase 1 IKE negotiations. Valid values are ` 2 | 14 | 15 | 16 | 17 | 18 | 19 | 20 | 21 | 22 | 23 | 24`.
     */
    @JvmName("luxhkfkvefejhmpr")
    public suspend fun tunnel1Phase1DhGroupNumbers(`value`: List<Int>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.tunnel1Phase1DhGroupNumbers = mapped
    }

    /**
     * @param values List of one or more Diffie-Hellman group numbers that are permitted for the first VPN tunnel for phase 1 IKE negotiations. Valid values are ` 2 | 14 | 15 | 16 | 17 | 18 | 19 | 20 | 21 | 22 | 23 | 24`.
     */
    @JvmName("nhjwwahlgbqxvaqp")
    public suspend fun tunnel1Phase1DhGroupNumbers(vararg values: Int) {
        val toBeMapped = values.toList()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.tunnel1Phase1DhGroupNumbers = mapped
    }

    /**
     * @param value List of one or more encryption algorithms that are permitted for the first VPN tunnel for phase 1 IKE negotiations. Valid values are `AES128 | AES256 | AES128-GCM-16 | AES256-GCM-16`.
     */
    @JvmName("afmwvoaibhdodlwe")
    public suspend fun tunnel1Phase1EncryptionAlgorithms(`value`: List<String>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.tunnel1Phase1EncryptionAlgorithms = mapped
    }

    /**
     * @param values List of one or more encryption algorithms that are permitted for the first VPN tunnel for phase 1 IKE negotiations. Valid values are `AES128 | AES256 | AES128-GCM-16 | AES256-GCM-16`.
     */
    @JvmName("xnixtniedccxavmn")
    public suspend fun tunnel1Phase1EncryptionAlgorithms(vararg values: String) {
        val toBeMapped = values.toList()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.tunnel1Phase1EncryptionAlgorithms = mapped
    }

    /**
     * @param value One or more integrity algorithms that are permitted for the first VPN tunnel for phase 1 IKE negotiations. Valid values are `SHA1 | SHA2-256 | SHA2-384 | SHA2-512`.
     */
    @JvmName("eeprypmvpftuswsv")
    public suspend fun tunnel1Phase1IntegrityAlgorithms(`value`: List<String>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.tunnel1Phase1IntegrityAlgorithms = mapped
    }

    /**
     * @param values One or more integrity algorithms that are permitted for the first VPN tunnel for phase 1 IKE negotiations. Valid values are `SHA1 | SHA2-256 | SHA2-384 | SHA2-512`.
     */
    @JvmName("ulyhevywkiirwffo")
    public suspend fun tunnel1Phase1IntegrityAlgorithms(vararg values: String) {
        val toBeMapped = values.toList()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.tunnel1Phase1IntegrityAlgorithms = mapped
    }

    /**
     * @param value The lifetime for phase 1 of the IKE negotiation for the first VPN tunnel, in seconds. Valid value is between `900` and `28800`.
     */
    @JvmName("ncjsalbbmfjvjqwo")
    public suspend fun tunnel1Phase1LifetimeSeconds(`value`: Int?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.tunnel1Phase1LifetimeSeconds = mapped
    }

    /**
     * @param value List of one or more Diffie-Hellman group numbers that are permitted for the first VPN tunnel for phase 2 IKE negotiations. Valid values are `2 | 5 | 14 | 15 | 16 | 17 | 18 | 19 | 20 | 21 | 22 | 23 | 24`.
     */
    @JvmName("ogyaayfmvesqyxwr")
    public suspend fun tunnel1Phase2DhGroupNumbers(`value`: List<Int>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.tunnel1Phase2DhGroupNumbers = mapped
    }

    /**
     * @param values List of one or more Diffie-Hellman group numbers that are permitted for the first VPN tunnel for phase 2 IKE negotiations. Valid values are `2 | 5 | 14 | 15 | 16 | 17 | 18 | 19 | 20 | 21 | 22 | 23 | 24`.
     */
    @JvmName("oxsaamojmqoartsh")
    public suspend fun tunnel1Phase2DhGroupNumbers(vararg values: Int) {
        val toBeMapped = values.toList()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.tunnel1Phase2DhGroupNumbers = mapped
    }

    /**
     * @param value List of one or more encryption algorithms that are permitted for the first VPN tunnel for phase 2 IKE negotiations. Valid values are `AES128 | AES256 | AES128-GCM-16 | AES256-GCM-16`.
     */
    @JvmName("varixwuddaconqvg")
    public suspend fun tunnel1Phase2EncryptionAlgorithms(`value`: List<String>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.tunnel1Phase2EncryptionAlgorithms = mapped
    }

    /**
     * @param values List of one or more encryption algorithms that are permitted for the first VPN tunnel for phase 2 IKE negotiations. Valid values are `AES128 | AES256 | AES128-GCM-16 | AES256-GCM-16`.
     */
    @JvmName("robbisbjcsadcamw")
    public suspend fun tunnel1Phase2EncryptionAlgorithms(vararg values: String) {
        val toBeMapped = values.toList()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.tunnel1Phase2EncryptionAlgorithms = mapped
    }

    /**
     * @param value List of one or more integrity algorithms that are permitted for the first VPN tunnel for phase 2 IKE negotiations. Valid values are `SHA1 | SHA2-256 | SHA2-384 | SHA2-512`.
     */
    @JvmName("qwkrxypnqmagxels")
    public suspend fun tunnel1Phase2IntegrityAlgorithms(`value`: List<String>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.tunnel1Phase2IntegrityAlgorithms = mapped
    }

    /**
     * @param values List of one or more integrity algorithms that are permitted for the first VPN tunnel for phase 2 IKE negotiations. Valid values are `SHA1 | SHA2-256 | SHA2-384 | SHA2-512`.
     */
    @JvmName("mtqrhhthwrmebqsm")
    public suspend fun tunnel1Phase2IntegrityAlgorithms(vararg values: String) {
        val toBeMapped = values.toList()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.tunnel1Phase2IntegrityAlgorithms = mapped
    }

    /**
     * @param value The lifetime for phase 2 of the IKE negotiation for the first VPN tunnel, in seconds. Valid value is between `900` and `3600`.
     */
    @JvmName("eefwleknkammctsu")
    public suspend fun tunnel1Phase2LifetimeSeconds(`value`: Int?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.tunnel1Phase2LifetimeSeconds = mapped
    }

    /**
     * @param value The preshared key of the first VPN tunnel. The preshared key must be between 8 and 64 characters in length and cannot start with zero(0). Allowed characters are alphanumeric characters, periods(.) and underscores(_).
     */
    @JvmName("ucnyqwwqdqtgnvrp")
    public suspend fun tunnel1PresharedKey(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.tunnel1PresharedKey = mapped
    }

    /**
     * @param value The percentage of the rekey window for the first VPN tunnel (determined by `tunnel1_rekey_margin_time_seconds`) during which the rekey time is randomly selected. Valid value is between `0` and `100`.
     */
    @JvmName("jwcbtpicfaeaetvf")
    public suspend fun tunnel1RekeyFuzzPercentage(`value`: Int?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.tunnel1RekeyFuzzPercentage = mapped
    }

    /**
     * @param value The margin time, in seconds, before the phase 2 lifetime expires, during which the AWS side of the first VPN connection performs an IKE rekey. The exact time of the rekey is randomly selected based on the value for `tunnel1_rekey_fuzz_percentage`. Valid value is between `60` and half of `tunnel1_phase2_lifetime_seconds`.
     */
    @JvmName("ohrexotqmaxefedu")
    public suspend fun tunnel1RekeyMarginTimeSeconds(`value`: Int?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.tunnel1RekeyMarginTimeSeconds = mapped
    }

    /**
     * @param value The number of packets in an IKE replay window for the first VPN tunnel. Valid value is between `64` and `2048`.
     */
    @JvmName("iwgvymbrtidmofxn")
    public suspend fun tunnel1ReplayWindowSize(`value`: Int?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.tunnel1ReplayWindowSize = mapped
    }

    /**
     * @param value The action to take when the establishing the tunnel for the first VPN connection. By default, your customer gateway device must initiate the IKE negotiation and bring up the tunnel. Specify start for AWS to initiate the IKE negotiation. Valid values are `add | start`.
     */
    @JvmName("opmgprujhroesngh")
    public suspend fun tunnel1StartupAction(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.tunnel1StartupAction = mapped
    }

    /**
     * @param value The action to take after DPD timeout occurs for the second VPN tunnel. Specify restart to restart the IKE initiation. Specify clear to end the IKE session. Valid values are `clear | none | restart`.
     */
    @JvmName("rmhfcdkittnxggaf")
    public suspend fun tunnel2DpdTimeoutAction(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.tunnel2DpdTimeoutAction = mapped
    }

    /**
     * @param value The number of seconds after which a DPD timeout occurs for the second VPN tunnel. Valid value is equal or higher than `30`.
     */
    @JvmName("bvlwthafmqcdjjgx")
    public suspend fun tunnel2DpdTimeoutSeconds(`value`: Int?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.tunnel2DpdTimeoutSeconds = mapped
    }

    /**
     * @param value Turn on or off tunnel endpoint lifecycle control feature for the second VPN tunnel. Valid values are `true | false`.
     */
    @JvmName("otyorunceudvdqrv")
    public suspend fun tunnel2EnableTunnelLifecycleControl(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.tunnel2EnableTunnelLifecycleControl = mapped
    }

    /**
     * @param value The IKE versions that are permitted for the second VPN tunnel. Valid values are `ikev1 | ikev2`.
     */
    @JvmName("oymdvdkebgyjuemm")
    public suspend fun tunnel2IkeVersions(`value`: List<String>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.tunnel2IkeVersions = mapped
    }

    /**
     * @param values The IKE versions that are permitted for the second VPN tunnel. Valid values are `ikev1 | ikev2`.
     */
    @JvmName("sgnbqhyhcdtgsmwp")
    public suspend fun tunnel2IkeVersions(vararg values: String) {
        val toBeMapped = values.toList()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.tunnel2IkeVersions = mapped
    }

    /**
     * @param value The CIDR block of the inside IP addresses for the second VPN tunnel. Valid value is a size /30 CIDR block from the 169.254.0.0/16 range.
     */
    @JvmName("oslcomkpccbthiyv")
    public suspend fun tunnel2InsideCidr(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.tunnel2InsideCidr = mapped
    }

    /**
     * @param value The range of inside IPv6 addresses for the second VPN tunnel. Supports only EC2 Transit Gateway. Valid value is a size /126 CIDR block from the local fd00::/8 range.
     */
    @JvmName("hnpokdsdiwsvbgsj")
    public suspend fun tunnel2InsideIpv6Cidr(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.tunnel2InsideIpv6Cidr = mapped
    }

    /**
     * @param value Options for logging VPN tunnel activity. See Log Options below for more details.
     */
    @JvmName("gmeesscpopudvxht")
    public suspend fun tunnel2LogOptions(`value`: VpnConnectionTunnel2LogOptionsArgs?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.tunnel2LogOptions = mapped
    }

    /**
     * @param argument Options for logging VPN tunnel activity. See Log Options below for more details.
     */
    @JvmName("fqapggcausgktphf")
    public suspend
    fun tunnel2LogOptions(argument: suspend VpnConnectionTunnel2LogOptionsArgsBuilder.() -> Unit) {
        val toBeMapped = VpnConnectionTunnel2LogOptionsArgsBuilder().applySuspend { argument() }.build()
        val mapped = of(toBeMapped)
        this.tunnel2LogOptions = mapped
    }

    /**
     * @param value List of one or more Diffie-Hellman group numbers that are permitted for the second VPN tunnel for phase 1 IKE negotiations. Valid values are ` 2 | 14 | 15 | 16 | 17 | 18 | 19 | 20 | 21 | 22 | 23 | 24`.
     */
    @JvmName("rxassmacmhxtgcso")
    public suspend fun tunnel2Phase1DhGroupNumbers(`value`: List<Int>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.tunnel2Phase1DhGroupNumbers = mapped
    }

    /**
     * @param values List of one or more Diffie-Hellman group numbers that are permitted for the second VPN tunnel for phase 1 IKE negotiations. Valid values are ` 2 | 14 | 15 | 16 | 17 | 18 | 19 | 20 | 21 | 22 | 23 | 24`.
     */
    @JvmName("ivklrjoexdtjrwsa")
    public suspend fun tunnel2Phase1DhGroupNumbers(vararg values: Int) {
        val toBeMapped = values.toList()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.tunnel2Phase1DhGroupNumbers = mapped
    }

    /**
     * @param value List of one or more encryption algorithms that are permitted for the second VPN tunnel for phase 1 IKE negotiations. Valid values are `AES128 | AES256 | AES128-GCM-16 | AES256-GCM-16`.
     */
    @JvmName("iodtnapditwtnbyg")
    public suspend fun tunnel2Phase1EncryptionAlgorithms(`value`: List<String>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.tunnel2Phase1EncryptionAlgorithms = mapped
    }

    /**
     * @param values List of one or more encryption algorithms that are permitted for the second VPN tunnel for phase 1 IKE negotiations. Valid values are `AES128 | AES256 | AES128-GCM-16 | AES256-GCM-16`.
     */
    @JvmName("ojtcfaxducgdlsqj")
    public suspend fun tunnel2Phase1EncryptionAlgorithms(vararg values: String) {
        val toBeMapped = values.toList()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.tunnel2Phase1EncryptionAlgorithms = mapped
    }

    /**
     * @param value One or more integrity algorithms that are permitted for the second VPN tunnel for phase 1 IKE negotiations. Valid values are `SHA1 | SHA2-256 | SHA2-384 | SHA2-512`.
     */
    @JvmName("qxwodpfvynddwycs")
    public suspend fun tunnel2Phase1IntegrityAlgorithms(`value`: List<String>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.tunnel2Phase1IntegrityAlgorithms = mapped
    }

    /**
     * @param values One or more integrity algorithms that are permitted for the second VPN tunnel for phase 1 IKE negotiations. Valid values are `SHA1 | SHA2-256 | SHA2-384 | SHA2-512`.
     */
    @JvmName("dkhffwucwcywhrjt")
    public suspend fun tunnel2Phase1IntegrityAlgorithms(vararg values: String) {
        val toBeMapped = values.toList()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.tunnel2Phase1IntegrityAlgorithms = mapped
    }

    /**
     * @param value The lifetime for phase 1 of the IKE negotiation for the second VPN tunnel, in seconds. Valid value is between `900` and `28800`.
     */
    @JvmName("bfonctkpkmjuhodt")
    public suspend fun tunnel2Phase1LifetimeSeconds(`value`: Int?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.tunnel2Phase1LifetimeSeconds = mapped
    }

    /**
     * @param value List of one or more Diffie-Hellman group numbers that are permitted for the second VPN tunnel for phase 2 IKE negotiations. Valid values are `2 | 5 | 14 | 15 | 16 | 17 | 18 | 19 | 20 | 21 | 22 | 23 | 24`.
     */
    @JvmName("pgqpkpudtbuhgmrl")
    public suspend fun tunnel2Phase2DhGroupNumbers(`value`: List<Int>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.tunnel2Phase2DhGroupNumbers = mapped
    }

    /**
     * @param values List of one or more Diffie-Hellman group numbers that are permitted for the second VPN tunnel for phase 2 IKE negotiations. Valid values are `2 | 5 | 14 | 15 | 16 | 17 | 18 | 19 | 20 | 21 | 22 | 23 | 24`.
     */
    @JvmName("ssbnermftkwbhhlv")
    public suspend fun tunnel2Phase2DhGroupNumbers(vararg values: Int) {
        val toBeMapped = values.toList()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.tunnel2Phase2DhGroupNumbers = mapped
    }

    /**
     * @param value List of one or more encryption algorithms that are permitted for the second VPN tunnel for phase 2 IKE negotiations. Valid values are `AES128 | AES256 | AES128-GCM-16 | AES256-GCM-16`.
     */
    @JvmName("hambxkurawyrbbwt")
    public suspend fun tunnel2Phase2EncryptionAlgorithms(`value`: List<String>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.tunnel2Phase2EncryptionAlgorithms = mapped
    }

    /**
     * @param values List of one or more encryption algorithms that are permitted for the second VPN tunnel for phase 2 IKE negotiations. Valid values are `AES128 | AES256 | AES128-GCM-16 | AES256-GCM-16`.
     */
    @JvmName("sxnbwqqavqtnatww")
    public suspend fun tunnel2Phase2EncryptionAlgorithms(vararg values: String) {
        val toBeMapped = values.toList()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.tunnel2Phase2EncryptionAlgorithms = mapped
    }

    /**
     * @param value List of one or more integrity algorithms that are permitted for the second VPN tunnel for phase 2 IKE negotiations. Valid values are `SHA1 | SHA2-256 | SHA2-384 | SHA2-512`.
     */
    @JvmName("rlgvnqbuejwocmav")
    public suspend fun tunnel2Phase2IntegrityAlgorithms(`value`: List<String>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.tunnel2Phase2IntegrityAlgorithms = mapped
    }

    /**
     * @param values List of one or more integrity algorithms that are permitted for the second VPN tunnel for phase 2 IKE negotiations. Valid values are `SHA1 | SHA2-256 | SHA2-384 | SHA2-512`.
     */
    @JvmName("duepwskgqyrcpdjx")
    public suspend fun tunnel2Phase2IntegrityAlgorithms(vararg values: String) {
        val toBeMapped = values.toList()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.tunnel2Phase2IntegrityAlgorithms = mapped
    }

    /**
     * @param value The lifetime for phase 2 of the IKE negotiation for the second VPN tunnel, in seconds. Valid value is between `900` and `3600`.
     */
    @JvmName("ruaytoceyaldfryh")
    public suspend fun tunnel2Phase2LifetimeSeconds(`value`: Int?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.tunnel2Phase2LifetimeSeconds = mapped
    }

    /**
     * @param value The preshared key of the second VPN tunnel. The preshared key must be between 8 and 64 characters in length and cannot start with zero(0). Allowed characters are alphanumeric characters, periods(.) and underscores(_).
     */
    @JvmName("hnkwrwvocbngxmhf")
    public suspend fun tunnel2PresharedKey(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.tunnel2PresharedKey = mapped
    }

    /**
     * @param value The percentage of the rekey window for the second VPN tunnel (determined by `tunnel2_rekey_margin_time_seconds`) during which the rekey time is randomly selected. Valid value is between `0` and `100`.
     */
    @JvmName("unlhiulrhrdfhylt")
    public suspend fun tunnel2RekeyFuzzPercentage(`value`: Int?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.tunnel2RekeyFuzzPercentage = mapped
    }

    /**
     * @param value The margin time, in seconds, before the phase 2 lifetime expires, during which the AWS side of the second VPN connection performs an IKE rekey. The exact time of the rekey is randomly selected based on the value for `tunnel2_rekey_fuzz_percentage`. Valid value is between `60` and half of `tunnel2_phase2_lifetime_seconds`.
     */
    @JvmName("ytljkuckltsbrfsf")
    public suspend fun tunnel2RekeyMarginTimeSeconds(`value`: Int?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.tunnel2RekeyMarginTimeSeconds = mapped
    }

    /**
     * @param value The number of packets in an IKE replay window for the second VPN tunnel. Valid value is between `64` and `2048`.
     */
    @JvmName("ngdwrtlrifytvoyw")
    public suspend fun tunnel2ReplayWindowSize(`value`: Int?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.tunnel2ReplayWindowSize = mapped
    }

    /**
     * @param value The action to take when the establishing the tunnel for the second VPN connection. By default, your customer gateway device must initiate the IKE negotiation and bring up the tunnel. Specify start for AWS to initiate the IKE negotiation. Valid values are `add | start`.
     */
    @JvmName("qbdcvawefmplajcr")
    public suspend fun tunnel2StartupAction(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.tunnel2StartupAction = mapped
    }

    /**
     * @param value Indicate whether the VPN tunnels process IPv4 or IPv6 traffic. Valid values are `ipv4 | ipv6`. `ipv6` Supports only EC2 Transit Gateway.
     */
    @JvmName("qvyvuolfosiuiqxj")
    public suspend fun tunnelInsideIpVersion(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.tunnelInsideIpVersion = mapped
    }

    /**
     * @param value The type of VPN connection. The only type AWS supports at this time is "ipsec.1".
     */
    @JvmName("cglimqtussxjjblb")
    public suspend fun type(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.type = mapped
    }

    /**
     * @param value The ID of the Virtual Private Gateway.
     */
    @JvmName("kjvwdqgncpxhsuaf")
    public suspend fun vpnGatewayId(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.vpnGatewayId = mapped
    }

    internal fun build(): VpnConnectionArgs = VpnConnectionArgs(
        customerGatewayId = customerGatewayId,
        enableAcceleration = enableAcceleration,
        localIpv4NetworkCidr = localIpv4NetworkCidr,
        localIpv6NetworkCidr = localIpv6NetworkCidr,
        outsideIpAddressType = outsideIpAddressType,
        remoteIpv4NetworkCidr = remoteIpv4NetworkCidr,
        remoteIpv6NetworkCidr = remoteIpv6NetworkCidr,
        staticRoutesOnly = staticRoutesOnly,
        tags = tags,
        transitGatewayId = transitGatewayId,
        transportTransitGatewayAttachmentId = transportTransitGatewayAttachmentId,
        tunnel1DpdTimeoutAction = tunnel1DpdTimeoutAction,
        tunnel1DpdTimeoutSeconds = tunnel1DpdTimeoutSeconds,
        tunnel1EnableTunnelLifecycleControl = tunnel1EnableTunnelLifecycleControl,
        tunnel1IkeVersions = tunnel1IkeVersions,
        tunnel1InsideCidr = tunnel1InsideCidr,
        tunnel1InsideIpv6Cidr = tunnel1InsideIpv6Cidr,
        tunnel1LogOptions = tunnel1LogOptions,
        tunnel1Phase1DhGroupNumbers = tunnel1Phase1DhGroupNumbers,
        tunnel1Phase1EncryptionAlgorithms = tunnel1Phase1EncryptionAlgorithms,
        tunnel1Phase1IntegrityAlgorithms = tunnel1Phase1IntegrityAlgorithms,
        tunnel1Phase1LifetimeSeconds = tunnel1Phase1LifetimeSeconds,
        tunnel1Phase2DhGroupNumbers = tunnel1Phase2DhGroupNumbers,
        tunnel1Phase2EncryptionAlgorithms = tunnel1Phase2EncryptionAlgorithms,
        tunnel1Phase2IntegrityAlgorithms = tunnel1Phase2IntegrityAlgorithms,
        tunnel1Phase2LifetimeSeconds = tunnel1Phase2LifetimeSeconds,
        tunnel1PresharedKey = tunnel1PresharedKey,
        tunnel1RekeyFuzzPercentage = tunnel1RekeyFuzzPercentage,
        tunnel1RekeyMarginTimeSeconds = tunnel1RekeyMarginTimeSeconds,
        tunnel1ReplayWindowSize = tunnel1ReplayWindowSize,
        tunnel1StartupAction = tunnel1StartupAction,
        tunnel2DpdTimeoutAction = tunnel2DpdTimeoutAction,
        tunnel2DpdTimeoutSeconds = tunnel2DpdTimeoutSeconds,
        tunnel2EnableTunnelLifecycleControl = tunnel2EnableTunnelLifecycleControl,
        tunnel2IkeVersions = tunnel2IkeVersions,
        tunnel2InsideCidr = tunnel2InsideCidr,
        tunnel2InsideIpv6Cidr = tunnel2InsideIpv6Cidr,
        tunnel2LogOptions = tunnel2LogOptions,
        tunnel2Phase1DhGroupNumbers = tunnel2Phase1DhGroupNumbers,
        tunnel2Phase1EncryptionAlgorithms = tunnel2Phase1EncryptionAlgorithms,
        tunnel2Phase1IntegrityAlgorithms = tunnel2Phase1IntegrityAlgorithms,
        tunnel2Phase1LifetimeSeconds = tunnel2Phase1LifetimeSeconds,
        tunnel2Phase2DhGroupNumbers = tunnel2Phase2DhGroupNumbers,
        tunnel2Phase2EncryptionAlgorithms = tunnel2Phase2EncryptionAlgorithms,
        tunnel2Phase2IntegrityAlgorithms = tunnel2Phase2IntegrityAlgorithms,
        tunnel2Phase2LifetimeSeconds = tunnel2Phase2LifetimeSeconds,
        tunnel2PresharedKey = tunnel2PresharedKey,
        tunnel2RekeyFuzzPercentage = tunnel2RekeyFuzzPercentage,
        tunnel2RekeyMarginTimeSeconds = tunnel2RekeyMarginTimeSeconds,
        tunnel2ReplayWindowSize = tunnel2ReplayWindowSize,
        tunnel2StartupAction = tunnel2StartupAction,
        tunnelInsideIpVersion = tunnelInsideIpVersion,
        type = type,
        vpnGatewayId = vpnGatewayId,
    )
}
