@file:Suppress("NAME_SHADOWING", "DEPRECATION")

package com.pulumi.aws.eks.kotlin

import com.pulumi.aws.eks.AddonArgs.builder
import com.pulumi.core.Output
import com.pulumi.core.Output.of
import com.pulumi.kotlin.ConvertibleToJava
import com.pulumi.kotlin.PulumiTagMarker
import kotlin.Boolean
import kotlin.Deprecated
import kotlin.Pair
import kotlin.String
import kotlin.Suppress
import kotlin.collections.Map
import kotlin.jvm.JvmName

/**
 * Manages an EKS add-on.
 * ## Example Usage
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.eks.Addon;
 * import com.pulumi.aws.eks.AddonArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var example = new Addon("example", AddonArgs.builder()
 *             .clusterName(aws_eks_cluster.example().name())
 *             .addonName("vpc-cni")
 *             .build());
 *     }
 * }
 * ```
 * ## Example Update add-on usage with resolve_conflicts_on_update and PRESERVE
 * `resolve_conflicts_on_update` with `PRESERVE` can be used to retain the config changes applied to the add-on with kubectl while upgrading to a newer version of the add-on.
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const example = new aws.eks.Addon("example", {
 *     clusterName: aws_eks_cluster.example.name,
 *     addonName: "coredns",
 *     addonVersion: "v1.10.1-eksbuild.1",
 *     resolveConflictsOnUpdate: "PRESERVE",
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * example = aws.eks.Addon("example",
 *     cluster_name=aws_eks_cluster["example"]["name"],
 *     addon_name="coredns",
 *     addon_version="v1.10.1-eksbuild.1",
 *     resolve_conflicts_on_update="PRESERVE")
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var example = new Aws.Eks.Addon("example", new()
 *     {
 *         ClusterName = aws_eks_cluster.Example.Name,
 *         AddonName = "coredns",
 *         AddonVersion = "v1.10.1-eksbuild.1",
 *         ResolveConflictsOnUpdate = "PRESERVE",
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/eks"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		_, err := eks.NewAddon(ctx, "example", &eks.AddonArgs{
 * 			ClusterName:              pulumi.Any(aws_eks_cluster.Example.Name),
 * 			AddonName:                pulumi.String("coredns"),
 * 			AddonVersion:             pulumi.String("v1.10.1-eksbuild.1"),
 * 			ResolveConflictsOnUpdate: pulumi.String("PRESERVE"),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.eks.Addon;
 * import com.pulumi.aws.eks.AddonArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var example = new Addon("example", AddonArgs.builder()
 *             .clusterName(aws_eks_cluster.example().name())
 *             .addonName("coredns")
 *             .addonVersion("v1.10.1-eksbuild.1")
 *             .resolveConflictsOnUpdate("PRESERVE")
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   example:
 *     type: aws:eks:Addon
 *     properties:
 *       clusterName: ${aws_eks_cluster.example.name}
 *       addonName: coredns
 *       addonVersion: v1.10.1-eksbuild.1
 *       #e.g., previous version v1.9.3-eksbuild.3 and the new version is v1.10.1-eksbuild.1
 *       resolveConflictsOnUpdate: PRESERVE
 * ```
 * ## Example add-on usage with custom configuration_values
 * Custom add-on configuration can be passed using `configuration_values` as a single JSON string while creating or updating the add-on.
 * > **Note:** `configuration_values` is a single JSON string should match the valid JSON schema for each add-on with specific version.
 * To find the correct JSON schema for each add-on can be extracted using [describe-addon-configuration](https://docs.aws.amazon.com/cli/latest/reference/eks/describe-addon-configuration.html) call.
 * This below is an example for extracting the `configuration_values` schema for `coredns`.
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * ```
 * ```python
 * import pulumi
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * return await Deployment.RunAsync(() =>
 * {
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *     }
 * }
 * ```
 * ```yaml
 * {}
 * ```
 * Example to create a `coredns` managed addon with custom `configuration_values`.
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const example = new aws.eks.Addon("example", {
 *     clusterName: "mycluster",
 *     addonName: "coredns",
 *     addonVersion: "v1.10.1-eksbuild.1",
 *     resolveConflictsOnCreate: "OVERWRITE",
 *     configurationValues: JSON.stringify({
 *         replicaCount: 4,
 *         resources: {
 *             limits: {
 *                 cpu: "100m",
 *                 memory: "150Mi",
 *             },
 *             requests: {
 *                 cpu: "100m",
 *                 memory: "150Mi",
 *             },
 *         },
 *     }),
 * });
 * ```
 * ```python
 * import pulumi
 * import json
 * import pulumi_aws as aws
 * example = aws.eks.Addon("example",
 *     cluster_name="mycluster",
 *     addon_name="coredns",
 *     addon_version="v1.10.1-eksbuild.1",
 *     resolve_conflicts_on_create="OVERWRITE",
 *     configuration_values=json.dumps({
 *         "replicaCount": 4,
 *         "resources": {
 *             "limits": {
 *                 "cpu": "100m",
 *                 "memory": "150Mi",
 *             },
 *             "requests": {
 *                 "cpu": "100m",
 *                 "memory": "150Mi",
 *             },
 *         },
 *     }))
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using System.Text.Json;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var example = new Aws.Eks.Addon("example", new()
 *     {
 *         ClusterName = "mycluster",
 *         AddonName = "coredns",
 *         AddonVersion = "v1.10.1-eksbuild.1",
 *         ResolveConflictsOnCreate = "OVERWRITE",
 *         ConfigurationValues = JsonSerializer.Serialize(new Dictionary<string, object?>
 *         {
 *             ["replicaCount"] = 4,
 *             ["resources"] = new Dictionary<string, object?>
 *             {
 *                 ["limits"] = new Dictionary<string, object?>
 *                 {
 *                     ["cpu"] = "100m",
 *                     ["memory"] = "150Mi",
 *                 },
 *                 ["requests"] = new Dictionary<string, object?>
 *                 {
 *                     ["cpu"] = "100m",
 *                     ["memory"] = "150Mi",
 *                 },
 *             },
 *         }),
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"encoding/json"
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/eks"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		tmpJSON0, err := json.Marshal(map[string]interface{}{
 * 			"replicaCount": 4,
 * 			"resources": map[string]interface{}{
 * 				"limits": map[string]interface{}{
 * 					"cpu":    "100m",
 * 					"memory": "150Mi",
 * 				},
 * 				"requests": map[string]interface{}{
 * 					"cpu":    "100m",
 * 					"memory": "150Mi",
 * 				},
 * 			},
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		json0 := string(tmpJSON0)
 * 		_, err = eks.NewAddon(ctx, "example", &eks.AddonArgs{
 * 			ClusterName:              pulumi.String("mycluster"),
 * 			AddonName:                pulumi.String("coredns"),
 * 			AddonVersion:             pulumi.String("v1.10.1-eksbuild.1"),
 * 			ResolveConflictsOnCreate: pulumi.String("OVERWRITE"),
 * 			ConfigurationValues:      pulumi.String(json0),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.eks.Addon;
 * import com.pulumi.aws.eks.AddonArgs;
 * import static com.pulumi.codegen.internal.Serialization.*;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var example = new Addon("example", AddonArgs.builder()
 *             .clusterName("mycluster")
 *             .addonName("coredns")
 *             .addonVersion("v1.10.1-eksbuild.1")
 *             .resolveConflictsOnCreate("OVERWRITE")
 *             .configurationValues(serializeJson(
 *                 jsonObject(
 *                     jsonProperty("replicaCount", 4),
 *                     jsonProperty("resources", jsonObject(
 *                         jsonProperty("limits", jsonObject(
 *                             jsonProperty("cpu", "100m"),
 *                             jsonProperty("memory", "150Mi")
 *                         )),
 *                         jsonProperty("requests", jsonObject(
 *                             jsonProperty("cpu", "100m"),
 *                             jsonProperty("memory", "150Mi")
 *                         ))
 *                     ))
 *                 )))
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   example:
 *     type: aws:eks:Addon
 *     properties:
 *       clusterName: mycluster
 *       addonName: coredns
 *       addonVersion: v1.10.1-eksbuild.1
 *       resolveConflictsOnCreate: OVERWRITE
 *       configurationValues:
 *         fn::toJSON:
 *           replicaCount: 4
 *           resources:
 *             limits:
 *               cpu: 100m
 *               memory: 150Mi
 *             requests:
 *               cpu: 100m
 *               memory: 150Mi
 * ```
 * ## Import
 * Using `pulumi import`, import EKS add-on using the `cluster_name` and `addon_name` separated by a colon (`:`). For example:
 * ```sh
 *  $ pulumi import aws:eks/addon:Addon my_eks_addon my_cluster_name:my_addon_name
 * ```
 * @property addonName Name of the EKS add-on. The name must match one of
 * the names returned by [describe-addon-versions](https://docs.aws.amazon.com/cli/latest/reference/eks/describe-addon-versions.html).
 * @property addonVersion The version of the EKS add-on. The version must
 * match one of the versions returned by [describe-addon-versions](https://docs.aws.amazon.com/cli/latest/reference/eks/describe-addon-versions.html).
 * @property clusterName Name of the EKS Cluster. Must be between 1-100 characters in length. Must begin with an alphanumeric character, and must only contain alphanumeric characters, dashes and underscores (`^[0-9A-Za-z][A-Za-z0-9\-_]+$`).
 * The following arguments are optional:
 * @property configurationValues custom configuration values for addons with single JSON string. This JSON string value must match the JSON schema derived from [describe-addon-configuration](https://docs.aws.amazon.com/cli/latest/reference/eks/describe-addon-configuration.html).
 * @property preserve Indicates if you want to preserve the created resources when deleting the EKS add-on.
 * @property resolveConflicts Define how to resolve parameter value conflicts when migrating an existing add-on to an Amazon EKS add-on or when applying version updates to the add-on. Valid values are `NONE`, `OVERWRITE` and `PRESERVE`. Note that `PRESERVE` is only valid on addon update, not for initial addon creation. If you need to set this to `PRESERVE`, use the `resolve_conflicts_on_create` and `resolve_conflicts_on_update` attributes instead. For more details check [UpdateAddon](https://docs.aws.amazon.com/eks/latest/APIReference/API_UpdateAddon.html) API Docs.
 * @property resolveConflictsOnCreate How to resolve field value conflicts when migrating a self-managed add-on to an Amazon EKS add-on. Valid values are `NONE` and `OVERWRITE`. For more details see the [CreateAddon](https://docs.aws.amazon.com/eks/latest/APIReference/API_CreateAddon.html) API Docs.
 * @property resolveConflictsOnUpdate How to resolve field value conflicts for an Amazon EKS add-on if you've changed a value from the Amazon EKS default value. Valid values are `NONE`, `OVERWRITE`, and `PRESERVE`. For more details see the [UpdateAddon](https://docs.aws.amazon.com/eks/latest/APIReference/API_UpdateAddon.html) API Docs.
 * @property serviceAccountRoleArn The Amazon Resource Name (ARN) of an
 * existing IAM role to bind to the add-on's service account. The role must be
 * assigned the IAM permissions required by the add-on. If you don't specify
 * an existing IAM role, then the add-on uses the permissions assigned to the node
 * IAM role. For more information, see [Amazon EKS node IAM role](https://docs.aws.amazon.com/eks/latest/userguide/create-node-role.html)
 * in the Amazon EKS User Guide.
 * > **Note:** To specify an existing IAM role, you must have an IAM OpenID Connect (OIDC)
 * provider created for your cluster. For more information, [see Enabling IAM roles
 * for service accounts on your cluster](https://docs.aws.amazon.com/eks/latest/userguide/enable-iam-roles-for-service-accounts.html)
 * in the Amazon EKS User Guide.
 * @property tags Key-value map of resource tags. If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
 */
public data class AddonArgs(
    public val addonName: Output<String>? = null,
    public val addonVersion: Output<String>? = null,
    public val clusterName: Output<String>? = null,
    public val configurationValues: Output<String>? = null,
    public val preserve: Output<Boolean>? = null,
    @Deprecated(
        message = """
  The "resolve_conflicts" attribute can't be set to "PRESERVE" on initial resource creation. Use
      "resolve_conflicts_on_create" and/or "resolve_conflicts_on_update" instead
  """,
    )
    public val resolveConflicts: Output<String>? = null,
    public val resolveConflictsOnCreate: Output<String>? = null,
    public val resolveConflictsOnUpdate: Output<String>? = null,
    public val serviceAccountRoleArn: Output<String>? = null,
    public val tags: Output<Map<String, String>>? = null,
) : ConvertibleToJava<com.pulumi.aws.eks.AddonArgs> {
    override fun toJava(): com.pulumi.aws.eks.AddonArgs = com.pulumi.aws.eks.AddonArgs.builder()
        .addonName(addonName?.applyValue({ args0 -> args0 }))
        .addonVersion(addonVersion?.applyValue({ args0 -> args0 }))
        .clusterName(clusterName?.applyValue({ args0 -> args0 }))
        .configurationValues(configurationValues?.applyValue({ args0 -> args0 }))
        .preserve(preserve?.applyValue({ args0 -> args0 }))
        .resolveConflicts(resolveConflicts?.applyValue({ args0 -> args0 }))
        .resolveConflictsOnCreate(resolveConflictsOnCreate?.applyValue({ args0 -> args0 }))
        .resolveConflictsOnUpdate(resolveConflictsOnUpdate?.applyValue({ args0 -> args0 }))
        .serviceAccountRoleArn(serviceAccountRoleArn?.applyValue({ args0 -> args0 }))
        .tags(
            tags?.applyValue({ args0 ->
                args0.map({ args0 ->
                    args0.key.to(args0.value)
                }).toMap()
            }),
        ).build()
}

/**
 * Builder for [AddonArgs].
 */
@PulumiTagMarker
public class AddonArgsBuilder internal constructor() {
    private var addonName: Output<String>? = null

    private var addonVersion: Output<String>? = null

    private var clusterName: Output<String>? = null

    private var configurationValues: Output<String>? = null

    private var preserve: Output<Boolean>? = null

    private var resolveConflicts: Output<String>? = null

    private var resolveConflictsOnCreate: Output<String>? = null

    private var resolveConflictsOnUpdate: Output<String>? = null

    private var serviceAccountRoleArn: Output<String>? = null

    private var tags: Output<Map<String, String>>? = null

    /**
     * @param value Name of the EKS add-on. The name must match one of
     * the names returned by [describe-addon-versions](https://docs.aws.amazon.com/cli/latest/reference/eks/describe-addon-versions.html).
     */
    @JvmName("eypmsayivcvtegvt")
    public suspend fun addonName(`value`: Output<String>) {
        this.addonName = value
    }

    /**
     * @param value The version of the EKS add-on. The version must
     * match one of the versions returned by [describe-addon-versions](https://docs.aws.amazon.com/cli/latest/reference/eks/describe-addon-versions.html).
     */
    @JvmName("cdtfesqdovatcwvl")
    public suspend fun addonVersion(`value`: Output<String>) {
        this.addonVersion = value
    }

    /**
     * @param value Name of the EKS Cluster. Must be between 1-100 characters in length. Must begin with an alphanumeric character, and must only contain alphanumeric characters, dashes and underscores (`^[0-9A-Za-z][A-Za-z0-9\-_]+$`).
     * The following arguments are optional:
     */
    @JvmName("nsoxikamttgmlynq")
    public suspend fun clusterName(`value`: Output<String>) {
        this.clusterName = value
    }

    /**
     * @param value custom configuration values for addons with single JSON string. This JSON string value must match the JSON schema derived from [describe-addon-configuration](https://docs.aws.amazon.com/cli/latest/reference/eks/describe-addon-configuration.html).
     */
    @JvmName("hucndrogqqkkowiu")
    public suspend fun configurationValues(`value`: Output<String>) {
        this.configurationValues = value
    }

    /**
     * @param value Indicates if you want to preserve the created resources when deleting the EKS add-on.
     */
    @JvmName("pcysjnjyojqxqwlm")
    public suspend fun preserve(`value`: Output<Boolean>) {
        this.preserve = value
    }

    /**
     * @param value Define how to resolve parameter value conflicts when migrating an existing add-on to an Amazon EKS add-on or when applying version updates to the add-on. Valid values are `NONE`, `OVERWRITE` and `PRESERVE`. Note that `PRESERVE` is only valid on addon update, not for initial addon creation. If you need to set this to `PRESERVE`, use the `resolve_conflicts_on_create` and `resolve_conflicts_on_update` attributes instead. For more details check [UpdateAddon](https://docs.aws.amazon.com/eks/latest/APIReference/API_UpdateAddon.html) API Docs.
     */
    @Deprecated(
        message = """
  The "resolve_conflicts" attribute can't be set to "PRESERVE" on initial resource creation. Use
      "resolve_conflicts_on_create" and/or "resolve_conflicts_on_update" instead
  """,
    )
    @JvmName("imdmbahmyobapuds")
    public suspend fun resolveConflicts(`value`: Output<String>) {
        this.resolveConflicts = value
    }

    /**
     * @param value How to resolve field value conflicts when migrating a self-managed add-on to an Amazon EKS add-on. Valid values are `NONE` and `OVERWRITE`. For more details see the [CreateAddon](https://docs.aws.amazon.com/eks/latest/APIReference/API_CreateAddon.html) API Docs.
     */
    @JvmName("pbwqbxclicuqgbpq")
    public suspend fun resolveConflictsOnCreate(`value`: Output<String>) {
        this.resolveConflictsOnCreate = value
    }

    /**
     * @param value How to resolve field value conflicts for an Amazon EKS add-on if you've changed a value from the Amazon EKS default value. Valid values are `NONE`, `OVERWRITE`, and `PRESERVE`. For more details see the [UpdateAddon](https://docs.aws.amazon.com/eks/latest/APIReference/API_UpdateAddon.html) API Docs.
     */
    @JvmName("tjxgtfaiuwwnidwb")
    public suspend fun resolveConflictsOnUpdate(`value`: Output<String>) {
        this.resolveConflictsOnUpdate = value
    }

    /**
     * @param value The Amazon Resource Name (ARN) of an
     * existing IAM role to bind to the add-on's service account. The role must be
     * assigned the IAM permissions required by the add-on. If you don't specify
     * an existing IAM role, then the add-on uses the permissions assigned to the node
     * IAM role. For more information, see [Amazon EKS node IAM role](https://docs.aws.amazon.com/eks/latest/userguide/create-node-role.html)
     * in the Amazon EKS User Guide.
     * > **Note:** To specify an existing IAM role, you must have an IAM OpenID Connect (OIDC)
     * provider created for your cluster. For more information, [see Enabling IAM roles
   * for service accounts on your cluster](https://docs.aws.amazon.com/eks/latest/userguide/enable-iam-roles-for-service-accounts.html)
     * in the Amazon EKS User Guide.
     */
    @JvmName("lnyhbebyavengpdg")
    public suspend fun serviceAccountRoleArn(`value`: Output<String>) {
        this.serviceAccountRoleArn = value
    }

    /**
     * @param value Key-value map of resource tags. If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
     */
    @JvmName("spmlaajmcnwnobmp")
    public suspend fun tags(`value`: Output<Map<String, String>>) {
        this.tags = value
    }

    /**
     * @param value Name of the EKS add-on. The name must match one of
     * the names returned by [describe-addon-versions](https://docs.aws.amazon.com/cli/latest/reference/eks/describe-addon-versions.html).
     */
    @JvmName("mxobuengspicmlrj")
    public suspend fun addonName(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.addonName = mapped
    }

    /**
     * @param value The version of the EKS add-on. The version must
     * match one of the versions returned by [describe-addon-versions](https://docs.aws.amazon.com/cli/latest/reference/eks/describe-addon-versions.html).
     */
    @JvmName("ofokpsfmhalsuvvy")
    public suspend fun addonVersion(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.addonVersion = mapped
    }

    /**
     * @param value Name of the EKS Cluster. Must be between 1-100 characters in length. Must begin with an alphanumeric character, and must only contain alphanumeric characters, dashes and underscores (`^[0-9A-Za-z][A-Za-z0-9\-_]+$`).
     * The following arguments are optional:
     */
    @JvmName("wywdqmafuqvetnde")
    public suspend fun clusterName(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.clusterName = mapped
    }

    /**
     * @param value custom configuration values for addons with single JSON string. This JSON string value must match the JSON schema derived from [describe-addon-configuration](https://docs.aws.amazon.com/cli/latest/reference/eks/describe-addon-configuration.html).
     */
    @JvmName("wmuphalcfwfmibti")
    public suspend fun configurationValues(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.configurationValues = mapped
    }

    /**
     * @param value Indicates if you want to preserve the created resources when deleting the EKS add-on.
     */
    @JvmName("yugmhqjbhfdgusbl")
    public suspend fun preserve(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.preserve = mapped
    }

    /**
     * @param value Define how to resolve parameter value conflicts when migrating an existing add-on to an Amazon EKS add-on or when applying version updates to the add-on. Valid values are `NONE`, `OVERWRITE` and `PRESERVE`. Note that `PRESERVE` is only valid on addon update, not for initial addon creation. If you need to set this to `PRESERVE`, use the `resolve_conflicts_on_create` and `resolve_conflicts_on_update` attributes instead. For more details check [UpdateAddon](https://docs.aws.amazon.com/eks/latest/APIReference/API_UpdateAddon.html) API Docs.
     */
    @Deprecated(
        message = """
  The "resolve_conflicts" attribute can't be set to "PRESERVE" on initial resource creation. Use
      "resolve_conflicts_on_create" and/or "resolve_conflicts_on_update" instead
  """,
    )
    @JvmName("bystlryvmjedocwe")
    public suspend fun resolveConflicts(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.resolveConflicts = mapped
    }

    /**
     * @param value How to resolve field value conflicts when migrating a self-managed add-on to an Amazon EKS add-on. Valid values are `NONE` and `OVERWRITE`. For more details see the [CreateAddon](https://docs.aws.amazon.com/eks/latest/APIReference/API_CreateAddon.html) API Docs.
     */
    @JvmName("nyivolbeynesxikm")
    public suspend fun resolveConflictsOnCreate(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.resolveConflictsOnCreate = mapped
    }

    /**
     * @param value How to resolve field value conflicts for an Amazon EKS add-on if you've changed a value from the Amazon EKS default value. Valid values are `NONE`, `OVERWRITE`, and `PRESERVE`. For more details see the [UpdateAddon](https://docs.aws.amazon.com/eks/latest/APIReference/API_UpdateAddon.html) API Docs.
     */
    @JvmName("vhnhvvqawcprqcxo")
    public suspend fun resolveConflictsOnUpdate(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.resolveConflictsOnUpdate = mapped
    }

    /**
     * @param value The Amazon Resource Name (ARN) of an
     * existing IAM role to bind to the add-on's service account. The role must be
     * assigned the IAM permissions required by the add-on. If you don't specify
     * an existing IAM role, then the add-on uses the permissions assigned to the node
     * IAM role. For more information, see [Amazon EKS node IAM role](https://docs.aws.amazon.com/eks/latest/userguide/create-node-role.html)
     * in the Amazon EKS User Guide.
     * > **Note:** To specify an existing IAM role, you must have an IAM OpenID Connect (OIDC)
     * provider created for your cluster. For more information, [see Enabling IAM roles
   * for service accounts on your cluster](https://docs.aws.amazon.com/eks/latest/userguide/enable-iam-roles-for-service-accounts.html)
     * in the Amazon EKS User Guide.
     */
    @JvmName("dbgxpfjnknjuprfp")
    public suspend fun serviceAccountRoleArn(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.serviceAccountRoleArn = mapped
    }

    /**
     * @param value Key-value map of resource tags. If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
     */
    @JvmName("pijrswxiulxvcwsr")
    public suspend fun tags(`value`: Map<String, String>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.tags = mapped
    }

    /**
     * @param values Key-value map of resource tags. If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
     */
    @JvmName("cvtvwcsweabngbsq")
    public fun tags(vararg values: Pair<String, String>) {
        val toBeMapped = values.toMap()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.tags = mapped
    }

    internal fun build(): AddonArgs = AddonArgs(
        addonName = addonName,
        addonVersion = addonVersion,
        clusterName = clusterName,
        configurationValues = configurationValues,
        preserve = preserve,
        resolveConflicts = resolveConflicts,
        resolveConflictsOnCreate = resolveConflictsOnCreate,
        resolveConflictsOnUpdate = resolveConflictsOnUpdate,
        serviceAccountRoleArn = serviceAccountRoleArn,
        tags = tags,
    )
}
