@file:Suppress("NAME_SHADOWING", "DEPRECATION")

package com.pulumi.aws.eks.kotlin

import com.pulumi.aws.eks.ClusterArgs.builder
import com.pulumi.aws.eks.kotlin.inputs.ClusterEncryptionConfigArgs
import com.pulumi.aws.eks.kotlin.inputs.ClusterEncryptionConfigArgsBuilder
import com.pulumi.aws.eks.kotlin.inputs.ClusterKubernetesNetworkConfigArgs
import com.pulumi.aws.eks.kotlin.inputs.ClusterKubernetesNetworkConfigArgsBuilder
import com.pulumi.aws.eks.kotlin.inputs.ClusterOutpostConfigArgs
import com.pulumi.aws.eks.kotlin.inputs.ClusterOutpostConfigArgsBuilder
import com.pulumi.aws.eks.kotlin.inputs.ClusterVpcConfigArgs
import com.pulumi.aws.eks.kotlin.inputs.ClusterVpcConfigArgsBuilder
import com.pulumi.core.Output
import com.pulumi.core.Output.of
import com.pulumi.kotlin.ConvertibleToJava
import com.pulumi.kotlin.PulumiTagMarker
import com.pulumi.kotlin.applySuspend
import kotlin.Pair
import kotlin.String
import kotlin.Suppress
import kotlin.Unit
import kotlin.collections.List
import kotlin.collections.Map
import kotlin.jvm.JvmName

/**
 * Manages an EKS Cluster.
 * ## Example Usage
 * ### Basic Usage
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.eks.Cluster;
 * import com.pulumi.aws.eks.ClusterArgs;
 * import com.pulumi.aws.eks.inputs.ClusterVpcConfigArgs;
 * import com.pulumi.resources.CustomResourceOptions;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var example = new Cluster("example", ClusterArgs.builder()
 *             .roleArn(aws_iam_role.example().arn())
 *             .vpcConfig(ClusterVpcConfigArgs.builder()
 *                 .subnetIds(
 *                     aws_subnet.example1().id(),
 *                     aws_subnet.example2().id())
 *                 .build())
 *             .build(), CustomResourceOptions.builder()
 *                 .dependsOn(
 *                     aws_iam_role_policy_attachment.example-AmazonEKSClusterPolicy(),
 *                     aws_iam_role_policy_attachment.example-AmazonEKSVPCResourceController())
 *                 .build());
 *         ctx.export("endpoint", example.endpoint());
 *         ctx.export("kubeconfig-certificate-authority-data", example.certificateAuthority().applyValue(certificateAuthority -> certificateAuthority.data()));
 *     }
 * }
 * ```
 * ### Example IAM Role for EKS Cluster
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.iam.IamFunctions;
 * import com.pulumi.aws.iam.inputs.GetPolicyDocumentArgs;
 * import com.pulumi.aws.iam.Role;
 * import com.pulumi.aws.iam.RoleArgs;
 * import com.pulumi.aws.iam.RolePolicyAttachment;
 * import com.pulumi.aws.iam.RolePolicyAttachmentArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         final var assumeRole = IamFunctions.getPolicyDocument(GetPolicyDocumentArgs.builder()
 *             .statements(GetPolicyDocumentStatementArgs.builder()
 *                 .effect("Allow")
 *                 .principals(GetPolicyDocumentStatementPrincipalArgs.builder()
 *                     .type("Service")
 *                     .identifiers("eks.amazonaws.com")
 *                     .build())
 *                 .actions("sts:AssumeRole")
 *                 .build())
 *             .build());
 *         var example = new Role("example", RoleArgs.builder()
 *             .assumeRolePolicy(assumeRole.applyValue(getPolicyDocumentResult -> getPolicyDocumentResult.json()))
 *             .build());
 *         var example_AmazonEKSClusterPolicy = new RolePolicyAttachment("example-AmazonEKSClusterPolicy", RolePolicyAttachmentArgs.builder()
 *             .policyArn("arn:aws:iam::aws:policy/AmazonEKSClusterPolicy")
 *             .role(example.name())
 *             .build());
 *         var example_AmazonEKSVPCResourceController = new RolePolicyAttachment("example-AmazonEKSVPCResourceController", RolePolicyAttachmentArgs.builder()
 *             .policyArn("arn:aws:iam::aws:policy/AmazonEKSVPCResourceController")
 *             .role(example.name())
 *             .build());
 *     }
 * }
 * ```
 * ### Enabling Control Plane Logging
 * [EKS Control Plane Logging](https://docs.aws.amazon.com/eks/latest/userguide/control-plane-logs.html) can be enabled via the `enabled_cluster_log_types` argument. To manage the CloudWatch Log Group retention period, the `aws.cloudwatch.LogGroup` resource can be used.
 * > The below configuration uses [`dependsOn`](https://www.pulumi.com/docs/intro/concepts/programming-model/#dependson) to prevent ordering issues with EKS automatically creating the log group first and a variable for naming consistency. Other ordering and naming methodologies may be more appropriate for your environment.
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.cloudwatch.LogGroup;
 * import com.pulumi.aws.cloudwatch.LogGroupArgs;
 * import com.pulumi.aws.eks.Cluster;
 * import com.pulumi.aws.eks.ClusterArgs;
 * import com.pulumi.resources.CustomResourceOptions;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         final var config = ctx.config();
 *         final var clusterName = config.get("clusterName").orElse("example");
 *         var exampleLogGroup = new LogGroup("exampleLogGroup", LogGroupArgs.builder()
 *             .retentionInDays(7)
 *             .build());
 *         var exampleCluster = new Cluster("exampleCluster", ClusterArgs.builder()
 *             .enabledClusterLogTypes(
 *                 "api",
 *                 "audit")
 *             .build(), CustomResourceOptions.builder()
 *                 .dependsOn(exampleLogGroup)
 *                 .build());
 *     }
 * }
 * ```
 * ### EKS Cluster on AWS Outpost
 * [Creating a local Amazon EKS cluster on an AWS Outpost](https://docs.aws.amazon.com/eks/latest/userguide/create-cluster-outpost.html)
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.iam.Role;
 * import com.pulumi.aws.iam.RoleArgs;
 * import com.pulumi.aws.eks.Cluster;
 * import com.pulumi.aws.eks.ClusterArgs;
 * import com.pulumi.aws.eks.inputs.ClusterVpcConfigArgs;
 * import com.pulumi.aws.eks.inputs.ClusterOutpostConfigArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var exampleRole = new Role("exampleRole", RoleArgs.builder()
 *             .assumeRolePolicy(data.aws_iam_policy_document().example_assume_role_policy().json())
 *             .build());
 *         var exampleCluster = new Cluster("exampleCluster", ClusterArgs.builder()
 *             .roleArn(exampleRole.arn())
 *             .vpcConfig(ClusterVpcConfigArgs.builder()
 *                 .endpointPrivateAccess(true)
 *                 .endpointPublicAccess(false)
 *                 .build())
 *             .outpostConfig(ClusterOutpostConfigArgs.builder()
 *                 .controlPlaneInstanceType("m5d.large")
 *                 .outpostArns(data.aws_outposts_outpost().example().arn())
 *                 .build())
 *             .build());
 *     }
 * }
 * ```
 * ## Import
 * Using `pulumi import`, import EKS Clusters using the `name`. For example:
 * ```sh
 *  $ pulumi import aws:eks/cluster:Cluster my_cluster my_cluster
 * ```
 * @property defaultAddonsToRemoves
 * @property enabledClusterLogTypes List of the desired control plane logging to enable. For more information, see [Amazon EKS Control Plane Logging](https://docs.aws.amazon.com/eks/latest/userguide/control-plane-logs.html).
 * @property encryptionConfig Configuration block with encryption configuration for the cluster. Only available on Kubernetes 1.13 and above clusters created after March 6, 2020. Detailed below.
 * @property kubernetesNetworkConfig Configuration block with kubernetes network configuration for the cluster. Detailed below. If removed, this provider will only perform drift detection if a configuration value is provided.
 * @property name Name of the cluster. Must be between 1-100 characters in length. Must begin with an alphanumeric character, and must only contain alphanumeric characters, dashes and underscores (`^[0-9A-Za-z][A-Za-z0-9\-_]+$`).
 * @property outpostConfig Configuration block representing the configuration of your local Amazon EKS cluster on an AWS Outpost. This block isn't available for creating Amazon EKS clusters on the AWS cloud.
 * @property roleArn ARN of the IAM role that provides permissions for the Kubernetes control plane to make calls to AWS API operations on your behalf. Ensure the resource configuration includes explicit dependencies on the IAM Role permissions by adding `depends_on` if using the `aws.iam.RolePolicy` resource or `aws.iam.RolePolicyAttachment` resource, otherwise EKS cannot delete EKS managed EC2 infrastructure such as Security Groups on EKS Cluster deletion.
 * @property tags Key-value map of resource tags. If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
 * @property version Desired Kubernetes master version. If you do not specify a value, the latest available version at resource creation is used and no upgrades will occur except those automatically triggered by EKS. The value must be configured and increased to upgrade the version when desired. Downgrades are not supported by EKS.
 * @property vpcConfig Configuration block for the VPC associated with your cluster. Amazon EKS VPC resources have specific requirements to work properly with Kubernetes. For more information, see [Cluster VPC Considerations](https://docs&#46;aws&#46;amazon&#46;com/eks/latest/userguide/network_reqs&#46;html) and [Cluster Security Group Considerations](https://docs.aws.amazon.com/eks/latest/userguide/sec-group-reqs.html) in the Amazon EKS User Guide. Detailed below. Also contains attributes detailed in the Attributes section.
 * The following arguments are optional:
 */
public data class ClusterArgs(
    public val defaultAddonsToRemoves: Output<List<String>>? = null,
    public val enabledClusterLogTypes: Output<List<String>>? = null,
    public val encryptionConfig: Output<ClusterEncryptionConfigArgs>? = null,
    public val kubernetesNetworkConfig: Output<ClusterKubernetesNetworkConfigArgs>? = null,
    public val name: Output<String>? = null,
    public val outpostConfig: Output<ClusterOutpostConfigArgs>? = null,
    public val roleArn: Output<String>? = null,
    public val tags: Output<Map<String, String>>? = null,
    public val version: Output<String>? = null,
    public val vpcConfig: Output<ClusterVpcConfigArgs>? = null,
) : ConvertibleToJava<com.pulumi.aws.eks.ClusterArgs> {
    override fun toJava(): com.pulumi.aws.eks.ClusterArgs = com.pulumi.aws.eks.ClusterArgs.builder()
        .defaultAddonsToRemoves(
            defaultAddonsToRemoves?.applyValue({ args0 ->
                args0.map({ args0 ->
                    args0
                })
            }),
        )
        .enabledClusterLogTypes(
            enabledClusterLogTypes?.applyValue({ args0 ->
                args0.map({ args0 ->
                    args0
                })
            }),
        )
        .encryptionConfig(encryptionConfig?.applyValue({ args0 -> args0.let({ args0 -> args0.toJava() }) }))
        .kubernetesNetworkConfig(
            kubernetesNetworkConfig?.applyValue({ args0 ->
                args0.let({ args0 ->
                    args0.toJava()
                })
            }),
        )
        .name(name?.applyValue({ args0 -> args0 }))
        .outpostConfig(outpostConfig?.applyValue({ args0 -> args0.let({ args0 -> args0.toJava() }) }))
        .roleArn(roleArn?.applyValue({ args0 -> args0 }))
        .tags(tags?.applyValue({ args0 -> args0.map({ args0 -> args0.key.to(args0.value) }).toMap() }))
        .version(version?.applyValue({ args0 -> args0 }))
        .vpcConfig(vpcConfig?.applyValue({ args0 -> args0.let({ args0 -> args0.toJava() }) })).build()
}

/**
 * Builder for [ClusterArgs].
 */
@PulumiTagMarker
public class ClusterArgsBuilder internal constructor() {
    private var defaultAddonsToRemoves: Output<List<String>>? = null

    private var enabledClusterLogTypes: Output<List<String>>? = null

    private var encryptionConfig: Output<ClusterEncryptionConfigArgs>? = null

    private var kubernetesNetworkConfig: Output<ClusterKubernetesNetworkConfigArgs>? = null

    private var name: Output<String>? = null

    private var outpostConfig: Output<ClusterOutpostConfigArgs>? = null

    private var roleArn: Output<String>? = null

    private var tags: Output<Map<String, String>>? = null

    private var version: Output<String>? = null

    private var vpcConfig: Output<ClusterVpcConfigArgs>? = null

    /**
     * @param value
     */
    @JvmName("pyildrgseldlrvwd")
    public suspend fun defaultAddonsToRemoves(`value`: Output<List<String>>) {
        this.defaultAddonsToRemoves = value
    }

    @JvmName("hbciaaqlmjwhmvjx")
    public suspend fun defaultAddonsToRemoves(vararg values: Output<String>) {
        this.defaultAddonsToRemoves = Output.all(values.asList())
    }

    /**
     * @param values
     */
    @JvmName("onacgkitlpyngdhi")
    public suspend fun defaultAddonsToRemoves(values: List<Output<String>>) {
        this.defaultAddonsToRemoves = Output.all(values)
    }

    /**
     * @param value List of the desired control plane logging to enable. For more information, see [Amazon EKS Control Plane Logging](https://docs.aws.amazon.com/eks/latest/userguide/control-plane-logs.html).
     */
    @JvmName("adiglfxhqnqjndrm")
    public suspend fun enabledClusterLogTypes(`value`: Output<List<String>>) {
        this.enabledClusterLogTypes = value
    }

    @JvmName("pqabrycehujvyccv")
    public suspend fun enabledClusterLogTypes(vararg values: Output<String>) {
        this.enabledClusterLogTypes = Output.all(values.asList())
    }

    /**
     * @param values List of the desired control plane logging to enable. For more information, see [Amazon EKS Control Plane Logging](https://docs.aws.amazon.com/eks/latest/userguide/control-plane-logs.html).
     */
    @JvmName("wnwaalnhxfbgncia")
    public suspend fun enabledClusterLogTypes(values: List<Output<String>>) {
        this.enabledClusterLogTypes = Output.all(values)
    }

    /**
     * @param value Configuration block with encryption configuration for the cluster. Only available on Kubernetes 1.13 and above clusters created after March 6, 2020. Detailed below.
     */
    @JvmName("giomupuspskyksiw")
    public suspend fun encryptionConfig(`value`: Output<ClusterEncryptionConfigArgs>) {
        this.encryptionConfig = value
    }

    /**
     * @param value Configuration block with kubernetes network configuration for the cluster. Detailed below. If removed, this provider will only perform drift detection if a configuration value is provided.
     */
    @JvmName("wsckrqikrygaetfd")
    public suspend fun kubernetesNetworkConfig(`value`: Output<ClusterKubernetesNetworkConfigArgs>) {
        this.kubernetesNetworkConfig = value
    }

    /**
     * @param value Name of the cluster. Must be between 1-100 characters in length. Must begin with an alphanumeric character, and must only contain alphanumeric characters, dashes and underscores (`^[0-9A-Za-z][A-Za-z0-9\-_]+$`).
     */
    @JvmName("dfqgppahitmbtlgq")
    public suspend fun name(`value`: Output<String>) {
        this.name = value
    }

    /**
     * @param value Configuration block representing the configuration of your local Amazon EKS cluster on an AWS Outpost. This block isn't available for creating Amazon EKS clusters on the AWS cloud.
     */
    @JvmName("ktuvvuugcktryjll")
    public suspend fun outpostConfig(`value`: Output<ClusterOutpostConfigArgs>) {
        this.outpostConfig = value
    }

    /**
     * @param value ARN of the IAM role that provides permissions for the Kubernetes control plane to make calls to AWS API operations on your behalf. Ensure the resource configuration includes explicit dependencies on the IAM Role permissions by adding `depends_on` if using the `aws.iam.RolePolicy` resource or `aws.iam.RolePolicyAttachment` resource, otherwise EKS cannot delete EKS managed EC2 infrastructure such as Security Groups on EKS Cluster deletion.
     */
    @JvmName("fdcfqvpvfbtcrtwo")
    public suspend fun roleArn(`value`: Output<String>) {
        this.roleArn = value
    }

    /**
     * @param value Key-value map of resource tags. If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
     */
    @JvmName("pgdunsiedstikudc")
    public suspend fun tags(`value`: Output<Map<String, String>>) {
        this.tags = value
    }

    /**
     * @param value Desired Kubernetes master version. If you do not specify a value, the latest available version at resource creation is used and no upgrades will occur except those automatically triggered by EKS. The value must be configured and increased to upgrade the version when desired. Downgrades are not supported by EKS.
     */
    @JvmName("bvvnfwgsbdjbwucc")
    public suspend fun version(`value`: Output<String>) {
        this.version = value
    }

    /**
     * @param value Configuration block for the VPC associated with your cluster. Amazon EKS VPC resources have specific requirements to work properly with Kubernetes. For more information, see [Cluster VPC Considerations](https://docs&#46;aws&#46;amazon&#46;com/eks/latest/userguide/network_reqs&#46;html) and [Cluster Security Group Considerations](https://docs.aws.amazon.com/eks/latest/userguide/sec-group-reqs.html) in the Amazon EKS User Guide. Detailed below. Also contains attributes detailed in the Attributes section.
     * The following arguments are optional:
     */
    @JvmName("ghayeubjbahxmmjs")
    public suspend fun vpcConfig(`value`: Output<ClusterVpcConfigArgs>) {
        this.vpcConfig = value
    }

    /**
     * @param value
     */
    @JvmName("lotrwsdhaksqkiub")
    public suspend fun defaultAddonsToRemoves(`value`: List<String>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.defaultAddonsToRemoves = mapped
    }

    /**
     * @param values
     */
    @JvmName("gyqciomyipkjacnq")
    public suspend fun defaultAddonsToRemoves(vararg values: String) {
        val toBeMapped = values.toList()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.defaultAddonsToRemoves = mapped
    }

    /**
     * @param value List of the desired control plane logging to enable. For more information, see [Amazon EKS Control Plane Logging](https://docs.aws.amazon.com/eks/latest/userguide/control-plane-logs.html).
     */
    @JvmName("xwjfbekjlpodnles")
    public suspend fun enabledClusterLogTypes(`value`: List<String>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.enabledClusterLogTypes = mapped
    }

    /**
     * @param values List of the desired control plane logging to enable. For more information, see [Amazon EKS Control Plane Logging](https://docs.aws.amazon.com/eks/latest/userguide/control-plane-logs.html).
     */
    @JvmName("bxrfdyhlvtmlmbgn")
    public suspend fun enabledClusterLogTypes(vararg values: String) {
        val toBeMapped = values.toList()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.enabledClusterLogTypes = mapped
    }

    /**
     * @param value Configuration block with encryption configuration for the cluster. Only available on Kubernetes 1.13 and above clusters created after March 6, 2020. Detailed below.
     */
    @JvmName("teawwmedqacubtbw")
    public suspend fun encryptionConfig(`value`: ClusterEncryptionConfigArgs?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.encryptionConfig = mapped
    }

    /**
     * @param argument Configuration block with encryption configuration for the cluster. Only available on Kubernetes 1.13 and above clusters created after March 6, 2020. Detailed below.
     */
    @JvmName("lklmdekjvqgpptjb")
    public suspend
    fun encryptionConfig(argument: suspend ClusterEncryptionConfigArgsBuilder.() -> Unit) {
        val toBeMapped = ClusterEncryptionConfigArgsBuilder().applySuspend { argument() }.build()
        val mapped = of(toBeMapped)
        this.encryptionConfig = mapped
    }

    /**
     * @param value Configuration block with kubernetes network configuration for the cluster. Detailed below. If removed, this provider will only perform drift detection if a configuration value is provided.
     */
    @JvmName("yxghidcbtamalcwh")
    public suspend fun kubernetesNetworkConfig(`value`: ClusterKubernetesNetworkConfigArgs?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.kubernetesNetworkConfig = mapped
    }

    /**
     * @param argument Configuration block with kubernetes network configuration for the cluster. Detailed below. If removed, this provider will only perform drift detection if a configuration value is provided.
     */
    @JvmName("lplyyqsahfutyatt")
    public suspend
    fun kubernetesNetworkConfig(argument: suspend ClusterKubernetesNetworkConfigArgsBuilder.() -> Unit) {
        val toBeMapped = ClusterKubernetesNetworkConfigArgsBuilder().applySuspend { argument() }.build()
        val mapped = of(toBeMapped)
        this.kubernetesNetworkConfig = mapped
    }

    /**
     * @param value Name of the cluster. Must be between 1-100 characters in length. Must begin with an alphanumeric character, and must only contain alphanumeric characters, dashes and underscores (`^[0-9A-Za-z][A-Za-z0-9\-_]+$`).
     */
    @JvmName("ixlnmltkfqnddvjq")
    public suspend fun name(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.name = mapped
    }

    /**
     * @param value Configuration block representing the configuration of your local Amazon EKS cluster on an AWS Outpost. This block isn't available for creating Amazon EKS clusters on the AWS cloud.
     */
    @JvmName("xqlkcvjoqroovibx")
    public suspend fun outpostConfig(`value`: ClusterOutpostConfigArgs?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.outpostConfig = mapped
    }

    /**
     * @param argument Configuration block representing the configuration of your local Amazon EKS cluster on an AWS Outpost. This block isn't available for creating Amazon EKS clusters on the AWS cloud.
     */
    @JvmName("dsmgtjumsarmbkkh")
    public suspend fun outpostConfig(argument: suspend ClusterOutpostConfigArgsBuilder.() -> Unit) {
        val toBeMapped = ClusterOutpostConfigArgsBuilder().applySuspend { argument() }.build()
        val mapped = of(toBeMapped)
        this.outpostConfig = mapped
    }

    /**
     * @param value ARN of the IAM role that provides permissions for the Kubernetes control plane to make calls to AWS API operations on your behalf. Ensure the resource configuration includes explicit dependencies on the IAM Role permissions by adding `depends_on` if using the `aws.iam.RolePolicy` resource or `aws.iam.RolePolicyAttachment` resource, otherwise EKS cannot delete EKS managed EC2 infrastructure such as Security Groups on EKS Cluster deletion.
     */
    @JvmName("gsmxfpawopiijhcp")
    public suspend fun roleArn(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.roleArn = mapped
    }

    /**
     * @param value Key-value map of resource tags. If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
     */
    @JvmName("mjygldqoiibxiols")
    public suspend fun tags(`value`: Map<String, String>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.tags = mapped
    }

    /**
     * @param values Key-value map of resource tags. If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
     */
    @JvmName("ydetlnmpcavlmdtw")
    public fun tags(vararg values: Pair<String, String>) {
        val toBeMapped = values.toMap()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.tags = mapped
    }

    /**
     * @param value Desired Kubernetes master version. If you do not specify a value, the latest available version at resource creation is used and no upgrades will occur except those automatically triggered by EKS. The value must be configured and increased to upgrade the version when desired. Downgrades are not supported by EKS.
     */
    @JvmName("cnhnvjbbbpvljfon")
    public suspend fun version(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.version = mapped
    }

    /**
     * @param value Configuration block for the VPC associated with your cluster. Amazon EKS VPC resources have specific requirements to work properly with Kubernetes. For more information, see [Cluster VPC Considerations](https://docs&#46;aws&#46;amazon&#46;com/eks/latest/userguide/network_reqs&#46;html) and [Cluster Security Group Considerations](https://docs.aws.amazon.com/eks/latest/userguide/sec-group-reqs.html) in the Amazon EKS User Guide. Detailed below. Also contains attributes detailed in the Attributes section.
     * The following arguments are optional:
     */
    @JvmName("acysidnseeugdayq")
    public suspend fun vpcConfig(`value`: ClusterVpcConfigArgs?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.vpcConfig = mapped
    }

    /**
     * @param argument Configuration block for the VPC associated with your cluster. Amazon EKS VPC resources have specific requirements to work properly with Kubernetes. For more information, see [Cluster VPC Considerations](https://docs&#46;aws&#46;amazon&#46;com/eks/latest/userguide/network_reqs&#46;html) and [Cluster Security Group Considerations](https://docs.aws.amazon.com/eks/latest/userguide/sec-group-reqs.html) in the Amazon EKS User Guide. Detailed below. Also contains attributes detailed in the Attributes section.
     * The following arguments are optional:
     */
    @JvmName("nyumqrvknngocbae")
    public suspend fun vpcConfig(argument: suspend ClusterVpcConfigArgsBuilder.() -> Unit) {
        val toBeMapped = ClusterVpcConfigArgsBuilder().applySuspend { argument() }.build()
        val mapped = of(toBeMapped)
        this.vpcConfig = mapped
    }

    internal fun build(): ClusterArgs = ClusterArgs(
        defaultAddonsToRemoves = defaultAddonsToRemoves,
        enabledClusterLogTypes = enabledClusterLogTypes,
        encryptionConfig = encryptionConfig,
        kubernetesNetworkConfig = kubernetesNetworkConfig,
        name = name,
        outpostConfig = outpostConfig,
        roleArn = roleArn,
        tags = tags,
        version = version,
        vpcConfig = vpcConfig,
    )
}
