@file:Suppress("NAME_SHADOWING", "DEPRECATION")

package com.pulumi.aws.elasticache.kotlin

import com.pulumi.aws.elasticache.ClusterArgs.builder
import com.pulumi.aws.elasticache.kotlin.inputs.ClusterLogDeliveryConfigurationArgs
import com.pulumi.aws.elasticache.kotlin.inputs.ClusterLogDeliveryConfigurationArgsBuilder
import com.pulumi.core.Output
import com.pulumi.core.Output.of
import com.pulumi.kotlin.ConvertibleToJava
import com.pulumi.kotlin.PulumiTagMarker
import com.pulumi.kotlin.applySuspend
import kotlin.Boolean
import kotlin.Int
import kotlin.Pair
import kotlin.String
import kotlin.Suppress
import kotlin.Unit
import kotlin.collections.List
import kotlin.collections.Map
import kotlin.jvm.JvmName

/**
 * Provides an ElastiCache Cluster resource, which manages either a
 * [Memcached cluster](https://docs.aws.amazon.com/AmazonElastiCache/latest/mem-ug/WhatIs.html), a
 * [single-node Redis instance](https://docs.aws.amazon.com/AmazonElastiCache/latest/red-ug/WhatIs.html), or a
 * [read replica in a Redis (Cluster Mode Enabled) replication group].
 * For working with Redis (Cluster Mode Enabled) replication groups, see the
 * `aws.elasticache.ReplicationGroup` resource.
 * > **Note:** When you change an attribute, such as `num_cache_nodes`, by default
 * it is applied in the next maintenance window. Because of this, this provider may report
 * a difference in its planning phase because the actual modification has not yet taken
 * place. You can use the `apply_immediately` flag to instruct the service to apply the
 * change immediately. Using `apply_immediately` can result in a brief downtime as the server reboots.
 * See the AWS Documentation on Modifying an ElastiCache Cache Cluster for
 * [ElastiCache for Memcached](https://docs.aws.amazon.com/AmazonElastiCache/latest/mem-ug/Clusters.Modify.html) or
 * [ElastiCache for Redis](https://docs.aws.amazon.com/AmazonElastiCache/latest/red-ug/Clusters.Modify.html)
 * for more information.
 * > **Note:** Any attribute changes that re-create the resource will be applied immediately, regardless of the value of `apply_immediately`.
 * ## Example Usage
 * ### Memcached Cluster
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.elasticache.Cluster;
 * import com.pulumi.aws.elasticache.ClusterArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var example = new Cluster("example", ClusterArgs.builder()
 *             .engine("memcached")
 *             .nodeType("cache.m4.large")
 *             .numCacheNodes(2)
 *             .parameterGroupName("default.memcached1.4")
 *             .port(11211)
 *             .build());
 *     }
 * }
 * ```
 * ### Redis Instance
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.elasticache.Cluster;
 * import com.pulumi.aws.elasticache.ClusterArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var example = new Cluster("example", ClusterArgs.builder()
 *             .engine("redis")
 *             .engineVersion("3.2.10")
 *             .nodeType("cache.m4.large")
 *             .numCacheNodes(1)
 *             .parameterGroupName("default.redis3.2")
 *             .port(6379)
 *             .build());
 *     }
 * }
 * ```
 * ### Redis Cluster Mode Disabled Read Replica Instance
 * These inherit their settings from the replication group.
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.elasticache.Cluster;
 * import com.pulumi.aws.elasticache.ClusterArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var replica = new Cluster("replica", ClusterArgs.builder()
 *             .replicationGroupId(aws_elasticache_replication_group.example().id())
 *             .build());
 *     }
 * }
 * ```
 * ### Redis Log Delivery configuration
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.elasticache.Cluster;
 * import com.pulumi.aws.elasticache.ClusterArgs;
 * import com.pulumi.aws.elasticache.inputs.ClusterLogDeliveryConfigurationArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var test = new Cluster("test", ClusterArgs.builder()
 *             .engine("redis")
 *             .nodeType("cache.t3.micro")
 *             .numCacheNodes(1)
 *             .port(6379)
 *             .applyImmediately(true)
 *             .logDeliveryConfigurations(
 *                 ClusterLogDeliveryConfigurationArgs.builder()
 *                     .destination(aws_cloudwatch_log_group.example().name())
 *                     .destinationType("cloudwatch-logs")
 *                     .logFormat("text")
 *                     .logType("slow-log")
 *                     .build(),
 *                 ClusterLogDeliveryConfigurationArgs.builder()
 *                     .destination(aws_kinesis_firehose_delivery_stream.example().name())
 *                     .destinationType("kinesis-firehose")
 *                     .logFormat("json")
 *                     .logType("engine-log")
 *                     .build())
 *             .build());
 *     }
 * }
 * ```
 * ## Import
 * Using `pulumi import`, import ElastiCache Clusters using the `cluster_id`. For example:
 * ```sh
 *  $ pulumi import aws:elasticache/cluster:Cluster my_cluster my_cluster
 * ```
 * @property applyImmediately Whether any database modifications are applied immediately, or during the next maintenance window. Default is `false`. See [Amazon ElastiCache Documentation for more information&#46;](https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_ModifyCacheCluster.html).
 * @property autoMinorVersionUpgrade Specifies whether minor version engine upgrades will be applied automatically to the underlying Cache Cluster instances during the maintenance window.
 * Only supported for engine type `"redis"` and if the engine version is 6 or higher.
 * Defaults to `true`.
 * @property availabilityZone Availability Zone for the cache cluster. If you want to create cache nodes in multi-az, use `preferred_availability_zones` instead. Default: System chosen Availability Zone. Changing this value will re-create the resource.
 * @property azMode Whether the nodes in this Memcached node group are created in a single Availability Zone or created across multiple Availability Zones in the cluster's region. Valid values for this parameter are `single-az` or `cross-az`, default is `single-az`. If you want to choose `cross-az`, `num_cache_nodes` must be greater than `1`.
 * @property clusterId Group identifier. ElastiCache converts this name to lowercase. Changing this value will re-create the resource.
 * @property engine Name of the cache engine to be used for this cache cluster. Valid values are `memcached` or `redis`.
 * @property engineVersion Version number of the cache engine to be used.
 * If not set, defaults to the latest version.
 * See [Describe Cache Engine Versions](https://docs.aws.amazon.com/cli/latest/reference/elasticache/describe-cache-engine-versions.html) in the AWS Documentation for supported versions.
 * When `engine` is `redis` and the version is 7 or higher, the major and minor version should be set, e.g., `7.2`.
 * When the version is 6, the major and minor version can be set, e.g., `6.2`,
 * or the minor version can be unspecified which will use the latest version at creation time, e.g., `6.x`.
 * Otherwise, specify the full version desired, e.g., `5.0.6`.
 * The actual engine version used is returned in the attribute `engine_version_actual`, see Attribute Reference below. Cannot be provided with `replication_group_id.`
 * @property finalSnapshotIdentifier Name of your final cluster snapshot. If omitted, no final snapshot will be made.
 * @property ipDiscovery The IP version to advertise in the discovery protocol. Valid values are `ipv4` or `ipv6`.
 * @property logDeliveryConfigurations Specifies the destination and format of Redis [SLOWLOG](https://redis&#46;io/commands/slowlog) or Redis [Engine Log](https://docs&#46;aws&#46;amazon&#46;com/AmazonElastiCache/latest/red-ug/Log_Delivery&#46;html#Log_contents-engine-log)&#46; See the documentation on [Amazon ElastiCache](https://docs.aws.amazon.com/AmazonElastiCache/latest/red-ug/Log_Delivery.html). See Log Delivery Configuration below for more details.
 * @property maintenanceWindow Specifies the weekly time range for when maintenance
 * on the cache cluster is performed. The format is `ddd:hh24:mi-ddd:hh24:mi` (24H Clock UTC).
 * The minimum maintenance window is a 60 minute period. Example: `sun:05:00-sun:09:00`.
 * @property networkType The IP versions for cache cluster connections. IPv6 is supported with Redis engine `6.2` onword or Memcached version `1.6.6` for all [Nitro system](https://aws.amazon.com/ec2/nitro/) instances. Valid values are `ipv4`, `ipv6` or `dual_stack`.
 * @property nodeType The instance class used. See AWS documentation for information on [supported node types for Redis](https://docs&#46;aws&#46;amazon&#46;com/AmazonElastiCache/latest/red-ug/CacheNodes&#46;SupportedTypes&#46;html) and [guidance on selecting node types for Redis](https://docs&#46;aws&#46;amazon&#46;com/AmazonElastiCache/latest/red-ug/nodes-select-size&#46;html)&#46; See AWS documentation for information on [supported node types for Memcached](https://docs&#46;aws&#46;amazon&#46;com/AmazonElastiCache/latest/mem-ug/CacheNodes&#46;SupportedTypes&#46;html) and [guidance on selecting node types for Memcached](https://docs.aws.amazon.com/AmazonElastiCache/latest/mem-ug/nodes-select-size.html). For Memcached, changing this value will re-create the resource.
 * @property notificationTopicArn ARN of an SNS topic to send ElastiCache notifications to. Example: `arn:aws:sns:us-east-1:012345678999:my_sns_topic`.
 * @property numCacheNodes The initial number of cache nodes that the cache cluster will have. For Redis, this value must be 1. For Memcached, this value must be between 1 and 40. If this number is reduced on subsequent runs, the highest numbered nodes will be removed.
 * @property outpostMode Specify the outpost mode that will apply to the cache cluster creation. Valid values are `"single-outpost"` and `"cross-outpost"`, however AWS currently only supports `"single-outpost"` mode.
 * @property parameterGroupName The name of the parameter group to associate with this cache cluster.
 * The following arguments are optional:
 * @property port The port number on which each of the cache nodes will accept connections. For Memcached the default is 11211, and for Redis the default port is 6379. Cannot be provided with `replication_group_id`. Changing this value will re-create the resource.
 * @property preferredAvailabilityZones List of the Availability Zones in which cache nodes are created. If you are creating your cluster in an Amazon VPC you can only locate nodes in Availability Zones that are associated with the subnets in the selected subnet group. The number of Availability Zones listed must equal the value of `num_cache_nodes`. If you want all the nodes in the same Availability Zone, use `availability_zone` instead, or repeat the Availability Zone multiple times in the list. Default: System chosen Availability Zones. Detecting drift of existing node availability zone is not currently supported. Updating this argument by itself to migrate existing node availability zones is not currently supported and will show a perpetual difference.
 * @property preferredOutpostArn The outpost ARN in which the cache cluster will be created.
 * @property replicationGroupId ID of the replication group to which this cluster should belong. If this parameter is specified, the cluster is added to the specified replication group as a read replica; otherwise, the cluster is a standalone primary that is not part of any replication group.
 * @property securityGroupIds One or more VPC security groups associated with the cache cluster. Cannot be provided with `replication_group_id.`
 * @property snapshotArns Single-element string list containing an Amazon Resource Name (ARN) of a Redis RDB snapshot file stored in Amazon S3. The object name cannot contain any commas. Changing `snapshot_arns` forces a new resource.
 * @property snapshotName Name of a snapshot from which to restore data into the new node group. Changing `snapshot_name` forces a new resource.
 * @property snapshotRetentionLimit Number of days for which ElastiCache will retain automatic cache cluster snapshots before deleting them. For example, if you set SnapshotRetentionLimit to 5, then a snapshot that was taken today will be retained for 5 days before being deleted. If the value of SnapshotRetentionLimit is set to zero (0), backups are turned off. Please note that setting a `snapshot_retention_limit` is not supported on cache.t1.micro cache nodes
 * @property snapshotWindow Daily time range (in UTC) during which ElastiCache will begin taking a daily snapshot of your cache cluster. Example: 05:00-09:00
 * @property subnetGroupName Name of the subnet group to be used for the cache cluster. Changing this value will re-create the resource. Cannot be provided with `replication_group_id.`
 * @property tags Map of tags to assign to the resource. If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
 * @property transitEncryptionEnabled Enable encryption in-transit. Supported only with Memcached versions `1.6.12` and later, running in a VPC. See the [ElastiCache in-transit encryption](https://docs.aws.amazon.com/AmazonElastiCache/latest/mem-ug/in-transit-encryption-mc.html) documentation for more details.
 */
public data class ClusterArgs(
    public val applyImmediately: Output<Boolean>? = null,
    public val autoMinorVersionUpgrade: Output<String>? = null,
    public val availabilityZone: Output<String>? = null,
    public val azMode: Output<String>? = null,
    public val clusterId: Output<String>? = null,
    public val engine: Output<String>? = null,
    public val engineVersion: Output<String>? = null,
    public val finalSnapshotIdentifier: Output<String>? = null,
    public val ipDiscovery: Output<String>? = null,
    public val logDeliveryConfigurations: Output<List<ClusterLogDeliveryConfigurationArgs>>? = null,
    public val maintenanceWindow: Output<String>? = null,
    public val networkType: Output<String>? = null,
    public val nodeType: Output<String>? = null,
    public val notificationTopicArn: Output<String>? = null,
    public val numCacheNodes: Output<Int>? = null,
    public val outpostMode: Output<String>? = null,
    public val parameterGroupName: Output<String>? = null,
    public val port: Output<Int>? = null,
    public val preferredAvailabilityZones: Output<List<String>>? = null,
    public val preferredOutpostArn: Output<String>? = null,
    public val replicationGroupId: Output<String>? = null,
    public val securityGroupIds: Output<List<String>>? = null,
    public val snapshotArns: Output<String>? = null,
    public val snapshotName: Output<String>? = null,
    public val snapshotRetentionLimit: Output<Int>? = null,
    public val snapshotWindow: Output<String>? = null,
    public val subnetGroupName: Output<String>? = null,
    public val tags: Output<Map<String, String>>? = null,
    public val transitEncryptionEnabled: Output<Boolean>? = null,
) : ConvertibleToJava<com.pulumi.aws.elasticache.ClusterArgs> {
    override fun toJava(): com.pulumi.aws.elasticache.ClusterArgs =
        com.pulumi.aws.elasticache.ClusterArgs.builder()
            .applyImmediately(applyImmediately?.applyValue({ args0 -> args0 }))
            .autoMinorVersionUpgrade(autoMinorVersionUpgrade?.applyValue({ args0 -> args0 }))
            .availabilityZone(availabilityZone?.applyValue({ args0 -> args0 }))
            .azMode(azMode?.applyValue({ args0 -> args0 }))
            .clusterId(clusterId?.applyValue({ args0 -> args0 }))
            .engine(engine?.applyValue({ args0 -> args0 }))
            .engineVersion(engineVersion?.applyValue({ args0 -> args0 }))
            .finalSnapshotIdentifier(finalSnapshotIdentifier?.applyValue({ args0 -> args0 }))
            .ipDiscovery(ipDiscovery?.applyValue({ args0 -> args0 }))
            .logDeliveryConfigurations(
                logDeliveryConfigurations?.applyValue({ args0 ->
                    args0.map({ args0 ->
                        args0.let({ args0 -> args0.toJava() })
                    })
                }),
            )
            .maintenanceWindow(maintenanceWindow?.applyValue({ args0 -> args0 }))
            .networkType(networkType?.applyValue({ args0 -> args0 }))
            .nodeType(nodeType?.applyValue({ args0 -> args0 }))
            .notificationTopicArn(notificationTopicArn?.applyValue({ args0 -> args0 }))
            .numCacheNodes(numCacheNodes?.applyValue({ args0 -> args0 }))
            .outpostMode(outpostMode?.applyValue({ args0 -> args0 }))
            .parameterGroupName(parameterGroupName?.applyValue({ args0 -> args0 }))
            .port(port?.applyValue({ args0 -> args0 }))
            .preferredAvailabilityZones(
                preferredAvailabilityZones?.applyValue({ args0 ->
                    args0.map({ args0 ->
                        args0
                    })
                }),
            )
            .preferredOutpostArn(preferredOutpostArn?.applyValue({ args0 -> args0 }))
            .replicationGroupId(replicationGroupId?.applyValue({ args0 -> args0 }))
            .securityGroupIds(securityGroupIds?.applyValue({ args0 -> args0.map({ args0 -> args0 }) }))
            .snapshotArns(snapshotArns?.applyValue({ args0 -> args0 }))
            .snapshotName(snapshotName?.applyValue({ args0 -> args0 }))
            .snapshotRetentionLimit(snapshotRetentionLimit?.applyValue({ args0 -> args0 }))
            .snapshotWindow(snapshotWindow?.applyValue({ args0 -> args0 }))
            .subnetGroupName(subnetGroupName?.applyValue({ args0 -> args0 }))
            .tags(tags?.applyValue({ args0 -> args0.map({ args0 -> args0.key.to(args0.value) }).toMap() }))
            .transitEncryptionEnabled(transitEncryptionEnabled?.applyValue({ args0 -> args0 })).build()
}

/**
 * Builder for [ClusterArgs].
 */
@PulumiTagMarker
public class ClusterArgsBuilder internal constructor() {
    private var applyImmediately: Output<Boolean>? = null

    private var autoMinorVersionUpgrade: Output<String>? = null

    private var availabilityZone: Output<String>? = null

    private var azMode: Output<String>? = null

    private var clusterId: Output<String>? = null

    private var engine: Output<String>? = null

    private var engineVersion: Output<String>? = null

    private var finalSnapshotIdentifier: Output<String>? = null

    private var ipDiscovery: Output<String>? = null

    private var logDeliveryConfigurations: Output<List<ClusterLogDeliveryConfigurationArgs>>? = null

    private var maintenanceWindow: Output<String>? = null

    private var networkType: Output<String>? = null

    private var nodeType: Output<String>? = null

    private var notificationTopicArn: Output<String>? = null

    private var numCacheNodes: Output<Int>? = null

    private var outpostMode: Output<String>? = null

    private var parameterGroupName: Output<String>? = null

    private var port: Output<Int>? = null

    private var preferredAvailabilityZones: Output<List<String>>? = null

    private var preferredOutpostArn: Output<String>? = null

    private var replicationGroupId: Output<String>? = null

    private var securityGroupIds: Output<List<String>>? = null

    private var snapshotArns: Output<String>? = null

    private var snapshotName: Output<String>? = null

    private var snapshotRetentionLimit: Output<Int>? = null

    private var snapshotWindow: Output<String>? = null

    private var subnetGroupName: Output<String>? = null

    private var tags: Output<Map<String, String>>? = null

    private var transitEncryptionEnabled: Output<Boolean>? = null

    /**
     * @param value Whether any database modifications are applied immediately, or during the next maintenance window. Default is `false`. See [Amazon ElastiCache Documentation for more information&#46;](https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_ModifyCacheCluster.html).
     */
    @JvmName("tiknsdbarmdnilod")
    public suspend fun applyImmediately(`value`: Output<Boolean>) {
        this.applyImmediately = value
    }

    /**
     * @param value Specifies whether minor version engine upgrades will be applied automatically to the underlying Cache Cluster instances during the maintenance window.
     * Only supported for engine type `"redis"` and if the engine version is 6 or higher.
     * Defaults to `true`.
     */
    @JvmName("fupkcmnayivvscgr")
    public suspend fun autoMinorVersionUpgrade(`value`: Output<String>) {
        this.autoMinorVersionUpgrade = value
    }

    /**
     * @param value Availability Zone for the cache cluster. If you want to create cache nodes in multi-az, use `preferred_availability_zones` instead. Default: System chosen Availability Zone. Changing this value will re-create the resource.
     */
    @JvmName("sxmcwuaowkfwjjbx")
    public suspend fun availabilityZone(`value`: Output<String>) {
        this.availabilityZone = value
    }

    /**
     * @param value Whether the nodes in this Memcached node group are created in a single Availability Zone or created across multiple Availability Zones in the cluster's region. Valid values for this parameter are `single-az` or `cross-az`, default is `single-az`. If you want to choose `cross-az`, `num_cache_nodes` must be greater than `1`.
     */
    @JvmName("ynkuiaayibjsutxr")
    public suspend fun azMode(`value`: Output<String>) {
        this.azMode = value
    }

    /**
     * @param value Group identifier. ElastiCache converts this name to lowercase. Changing this value will re-create the resource.
     */
    @JvmName("wfyarwkmcvsvprad")
    public suspend fun clusterId(`value`: Output<String>) {
        this.clusterId = value
    }

    /**
     * @param value Name of the cache engine to be used for this cache cluster. Valid values are `memcached` or `redis`.
     */
    @JvmName("miemjjchwoumubdd")
    public suspend fun engine(`value`: Output<String>) {
        this.engine = value
    }

    /**
     * @param value Version number of the cache engine to be used.
     * If not set, defaults to the latest version.
     * See [Describe Cache Engine Versions](https://docs.aws.amazon.com/cli/latest/reference/elasticache/describe-cache-engine-versions.html) in the AWS Documentation for supported versions.
     * When `engine` is `redis` and the version is 7 or higher, the major and minor version should be set, e.g., `7.2`.
     * When the version is 6, the major and minor version can be set, e.g., `6.2`,
     * or the minor version can be unspecified which will use the latest version at creation time, e.g., `6.x`.
     * Otherwise, specify the full version desired, e.g., `5.0.6`.
     * The actual engine version used is returned in the attribute `engine_version_actual`, see Attribute Reference below. Cannot be provided with `replication_group_id.`
     */
    @JvmName("gemcdrnfwwvgcodk")
    public suspend fun engineVersion(`value`: Output<String>) {
        this.engineVersion = value
    }

    /**
     * @param value Name of your final cluster snapshot. If omitted, no final snapshot will be made.
     */
    @JvmName("kfiqfphwmgfqrirf")
    public suspend fun finalSnapshotIdentifier(`value`: Output<String>) {
        this.finalSnapshotIdentifier = value
    }

    /**
     * @param value The IP version to advertise in the discovery protocol. Valid values are `ipv4` or `ipv6`.
     */
    @JvmName("hexrtmtuitgrgesm")
    public suspend fun ipDiscovery(`value`: Output<String>) {
        this.ipDiscovery = value
    }

    /**
     * @param value Specifies the destination and format of Redis [SLOWLOG](https://redis&#46;io/commands/slowlog) or Redis [Engine Log](https://docs&#46;aws&#46;amazon&#46;com/AmazonElastiCache/latest/red-ug/Log_Delivery&#46;html#Log_contents-engine-log)&#46; See the documentation on [Amazon ElastiCache](https://docs.aws.amazon.com/AmazonElastiCache/latest/red-ug/Log_Delivery.html). See Log Delivery Configuration below for more details.
     */
    @JvmName("cxncmmggodikvqid")
    public suspend
    fun logDeliveryConfigurations(`value`: Output<List<ClusterLogDeliveryConfigurationArgs>>) {
        this.logDeliveryConfigurations = value
    }

    @JvmName("ibteikprspgmixjp")
    public suspend fun logDeliveryConfigurations(
        vararg
        values: Output<ClusterLogDeliveryConfigurationArgs>,
    ) {
        this.logDeliveryConfigurations = Output.all(values.asList())
    }

    /**
     * @param values Specifies the destination and format of Redis [SLOWLOG](https://redis&#46;io/commands/slowlog) or Redis [Engine Log](https://docs&#46;aws&#46;amazon&#46;com/AmazonElastiCache/latest/red-ug/Log_Delivery&#46;html#Log_contents-engine-log)&#46; See the documentation on [Amazon ElastiCache](https://docs.aws.amazon.com/AmazonElastiCache/latest/red-ug/Log_Delivery.html). See Log Delivery Configuration below for more details.
     */
    @JvmName("ttsmsfoaqjjovvxg")
    public suspend
    fun logDeliveryConfigurations(values: List<Output<ClusterLogDeliveryConfigurationArgs>>) {
        this.logDeliveryConfigurations = Output.all(values)
    }

    /**
     * @param value Specifies the weekly time range for when maintenance
     * on the cache cluster is performed. The format is `ddd:hh24:mi-ddd:hh24:mi` (24H Clock UTC).
     * The minimum maintenance window is a 60 minute period. Example: `sun:05:00-sun:09:00`.
     */
    @JvmName("mqlhvpmibuvcmsqo")
    public suspend fun maintenanceWindow(`value`: Output<String>) {
        this.maintenanceWindow = value
    }

    /**
     * @param value The IP versions for cache cluster connections. IPv6 is supported with Redis engine `6.2` onword or Memcached version `1.6.6` for all [Nitro system](https://aws.amazon.com/ec2/nitro/) instances. Valid values are `ipv4`, `ipv6` or `dual_stack`.
     */
    @JvmName("hdnshcdgiwmbotcg")
    public suspend fun networkType(`value`: Output<String>) {
        this.networkType = value
    }

    /**
     * @param value The instance class used. See AWS documentation for information on [supported node types for Redis](https://docs&#46;aws&#46;amazon&#46;com/AmazonElastiCache/latest/red-ug/CacheNodes&#46;SupportedTypes&#46;html) and [guidance on selecting node types for Redis](https://docs&#46;aws&#46;amazon&#46;com/AmazonElastiCache/latest/red-ug/nodes-select-size&#46;html)&#46; See AWS documentation for information on [supported node types for Memcached](https://docs&#46;aws&#46;amazon&#46;com/AmazonElastiCache/latest/mem-ug/CacheNodes&#46;SupportedTypes&#46;html) and [guidance on selecting node types for Memcached](https://docs.aws.amazon.com/AmazonElastiCache/latest/mem-ug/nodes-select-size.html). For Memcached, changing this value will re-create the resource.
     */
    @JvmName("lkkwhigeohlfcmmy")
    public suspend fun nodeType(`value`: Output<String>) {
        this.nodeType = value
    }

    /**
     * @param value ARN of an SNS topic to send ElastiCache notifications to. Example: `arn:aws:sns:us-east-1:012345678999:my_sns_topic`.
     */
    @JvmName("goufewwbssmmqwep")
    public suspend fun notificationTopicArn(`value`: Output<String>) {
        this.notificationTopicArn = value
    }

    /**
     * @param value The initial number of cache nodes that the cache cluster will have. For Redis, this value must be 1. For Memcached, this value must be between 1 and 40. If this number is reduced on subsequent runs, the highest numbered nodes will be removed.
     */
    @JvmName("bpavsmcoppffxryo")
    public suspend fun numCacheNodes(`value`: Output<Int>) {
        this.numCacheNodes = value
    }

    /**
     * @param value Specify the outpost mode that will apply to the cache cluster creation. Valid values are `"single-outpost"` and `"cross-outpost"`, however AWS currently only supports `"single-outpost"` mode.
     */
    @JvmName("dnwakxwqhfxnwxsf")
    public suspend fun outpostMode(`value`: Output<String>) {
        this.outpostMode = value
    }

    /**
     * @param value The name of the parameter group to associate with this cache cluster.
     * The following arguments are optional:
     */
    @JvmName("wedgloevfttvwyyj")
    public suspend fun parameterGroupName(`value`: Output<String>) {
        this.parameterGroupName = value
    }

    /**
     * @param value The port number on which each of the cache nodes will accept connections. For Memcached the default is 11211, and for Redis the default port is 6379. Cannot be provided with `replication_group_id`. Changing this value will re-create the resource.
     */
    @JvmName("ooqpblftemwfkuyg")
    public suspend fun port(`value`: Output<Int>) {
        this.port = value
    }

    /**
     * @param value List of the Availability Zones in which cache nodes are created. If you are creating your cluster in an Amazon VPC you can only locate nodes in Availability Zones that are associated with the subnets in the selected subnet group. The number of Availability Zones listed must equal the value of `num_cache_nodes`. If you want all the nodes in the same Availability Zone, use `availability_zone` instead, or repeat the Availability Zone multiple times in the list. Default: System chosen Availability Zones. Detecting drift of existing node availability zone is not currently supported. Updating this argument by itself to migrate existing node availability zones is not currently supported and will show a perpetual difference.
     */
    @JvmName("nvjbcddlekbujtjo")
    public suspend fun preferredAvailabilityZones(`value`: Output<List<String>>) {
        this.preferredAvailabilityZones = value
    }

    @JvmName("ngtycjbggqwstsis")
    public suspend fun preferredAvailabilityZones(vararg values: Output<String>) {
        this.preferredAvailabilityZones = Output.all(values.asList())
    }

    /**
     * @param values List of the Availability Zones in which cache nodes are created. If you are creating your cluster in an Amazon VPC you can only locate nodes in Availability Zones that are associated with the subnets in the selected subnet group. The number of Availability Zones listed must equal the value of `num_cache_nodes`. If you want all the nodes in the same Availability Zone, use `availability_zone` instead, or repeat the Availability Zone multiple times in the list. Default: System chosen Availability Zones. Detecting drift of existing node availability zone is not currently supported. Updating this argument by itself to migrate existing node availability zones is not currently supported and will show a perpetual difference.
     */
    @JvmName("nkjgliianyyjimng")
    public suspend fun preferredAvailabilityZones(values: List<Output<String>>) {
        this.preferredAvailabilityZones = Output.all(values)
    }

    /**
     * @param value The outpost ARN in which the cache cluster will be created.
     */
    @JvmName("frpouoowmskqexei")
    public suspend fun preferredOutpostArn(`value`: Output<String>) {
        this.preferredOutpostArn = value
    }

    /**
     * @param value ID of the replication group to which this cluster should belong. If this parameter is specified, the cluster is added to the specified replication group as a read replica; otherwise, the cluster is a standalone primary that is not part of any replication group.
     */
    @JvmName("bryfixreimkkauqq")
    public suspend fun replicationGroupId(`value`: Output<String>) {
        this.replicationGroupId = value
    }

    /**
     * @param value One or more VPC security groups associated with the cache cluster. Cannot be provided with `replication_group_id.`
     */
    @JvmName("thxqxbyfadyhemwl")
    public suspend fun securityGroupIds(`value`: Output<List<String>>) {
        this.securityGroupIds = value
    }

    @JvmName("qmlwuvyrfadqapug")
    public suspend fun securityGroupIds(vararg values: Output<String>) {
        this.securityGroupIds = Output.all(values.asList())
    }

    /**
     * @param values One or more VPC security groups associated with the cache cluster. Cannot be provided with `replication_group_id.`
     */
    @JvmName("pmrnueniwinvefga")
    public suspend fun securityGroupIds(values: List<Output<String>>) {
        this.securityGroupIds = Output.all(values)
    }

    /**
     * @param value Single-element string list containing an Amazon Resource Name (ARN) of a Redis RDB snapshot file stored in Amazon S3. The object name cannot contain any commas. Changing `snapshot_arns` forces a new resource.
     */
    @JvmName("blthfthsfrydjtsr")
    public suspend fun snapshotArns(`value`: Output<String>) {
        this.snapshotArns = value
    }

    /**
     * @param value Name of a snapshot from which to restore data into the new node group. Changing `snapshot_name` forces a new resource.
     */
    @JvmName("gimbjrptjclnjawr")
    public suspend fun snapshotName(`value`: Output<String>) {
        this.snapshotName = value
    }

    /**
     * @param value Number of days for which ElastiCache will retain automatic cache cluster snapshots before deleting them. For example, if you set SnapshotRetentionLimit to 5, then a snapshot that was taken today will be retained for 5 days before being deleted. If the value of SnapshotRetentionLimit is set to zero (0), backups are turned off. Please note that setting a `snapshot_retention_limit` is not supported on cache.t1.micro cache nodes
     */
    @JvmName("filagddjgpjfpasc")
    public suspend fun snapshotRetentionLimit(`value`: Output<Int>) {
        this.snapshotRetentionLimit = value
    }

    /**
     * @param value Daily time range (in UTC) during which ElastiCache will begin taking a daily snapshot of your cache cluster. Example: 05:00-09:00
     */
    @JvmName("wpcybvgdrrbokkvk")
    public suspend fun snapshotWindow(`value`: Output<String>) {
        this.snapshotWindow = value
    }

    /**
     * @param value Name of the subnet group to be used for the cache cluster. Changing this value will re-create the resource. Cannot be provided with `replication_group_id.`
     */
    @JvmName("dihdanxrefajwuip")
    public suspend fun subnetGroupName(`value`: Output<String>) {
        this.subnetGroupName = value
    }

    /**
     * @param value Map of tags to assign to the resource. If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
     */
    @JvmName("fjmhjkuixghnxyev")
    public suspend fun tags(`value`: Output<Map<String, String>>) {
        this.tags = value
    }

    /**
     * @param value Enable encryption in-transit. Supported only with Memcached versions `1.6.12` and later, running in a VPC. See the [ElastiCache in-transit encryption](https://docs.aws.amazon.com/AmazonElastiCache/latest/mem-ug/in-transit-encryption-mc.html) documentation for more details.
     */
    @JvmName("cgwcjwifyjsxgwxb")
    public suspend fun transitEncryptionEnabled(`value`: Output<Boolean>) {
        this.transitEncryptionEnabled = value
    }

    /**
     * @param value Whether any database modifications are applied immediately, or during the next maintenance window. Default is `false`. See [Amazon ElastiCache Documentation for more information&#46;](https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_ModifyCacheCluster.html).
     */
    @JvmName("yeyffmwrybqdjmrq")
    public suspend fun applyImmediately(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.applyImmediately = mapped
    }

    /**
     * @param value Specifies whether minor version engine upgrades will be applied automatically to the underlying Cache Cluster instances during the maintenance window.
     * Only supported for engine type `"redis"` and if the engine version is 6 or higher.
     * Defaults to `true`.
     */
    @JvmName("lrsyowpbfrtukyhw")
    public suspend fun autoMinorVersionUpgrade(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.autoMinorVersionUpgrade = mapped
    }

    /**
     * @param value Availability Zone for the cache cluster. If you want to create cache nodes in multi-az, use `preferred_availability_zones` instead. Default: System chosen Availability Zone. Changing this value will re-create the resource.
     */
    @JvmName("wdkqjxtctudcfuco")
    public suspend fun availabilityZone(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.availabilityZone = mapped
    }

    /**
     * @param value Whether the nodes in this Memcached node group are created in a single Availability Zone or created across multiple Availability Zones in the cluster's region. Valid values for this parameter are `single-az` or `cross-az`, default is `single-az`. If you want to choose `cross-az`, `num_cache_nodes` must be greater than `1`.
     */
    @JvmName("hqxvdgsomgilfwtb")
    public suspend fun azMode(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.azMode = mapped
    }

    /**
     * @param value Group identifier. ElastiCache converts this name to lowercase. Changing this value will re-create the resource.
     */
    @JvmName("yrxrxjnolkreoqxf")
    public suspend fun clusterId(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.clusterId = mapped
    }

    /**
     * @param value Name of the cache engine to be used for this cache cluster. Valid values are `memcached` or `redis`.
     */
    @JvmName("ytoxyafjklgjfogo")
    public suspend fun engine(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.engine = mapped
    }

    /**
     * @param value Version number of the cache engine to be used.
     * If not set, defaults to the latest version.
     * See [Describe Cache Engine Versions](https://docs.aws.amazon.com/cli/latest/reference/elasticache/describe-cache-engine-versions.html) in the AWS Documentation for supported versions.
     * When `engine` is `redis` and the version is 7 or higher, the major and minor version should be set, e.g., `7.2`.
     * When the version is 6, the major and minor version can be set, e.g., `6.2`,
     * or the minor version can be unspecified which will use the latest version at creation time, e.g., `6.x`.
     * Otherwise, specify the full version desired, e.g., `5.0.6`.
     * The actual engine version used is returned in the attribute `engine_version_actual`, see Attribute Reference below. Cannot be provided with `replication_group_id.`
     */
    @JvmName("vlpyscyjgyfcibuy")
    public suspend fun engineVersion(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.engineVersion = mapped
    }

    /**
     * @param value Name of your final cluster snapshot. If omitted, no final snapshot will be made.
     */
    @JvmName("lduthjvcpyjcdwmy")
    public suspend fun finalSnapshotIdentifier(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.finalSnapshotIdentifier = mapped
    }

    /**
     * @param value The IP version to advertise in the discovery protocol. Valid values are `ipv4` or `ipv6`.
     */
    @JvmName("nxrwgqxpqrqxhysp")
    public suspend fun ipDiscovery(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.ipDiscovery = mapped
    }

    /**
     * @param value Specifies the destination and format of Redis [SLOWLOG](https://redis&#46;io/commands/slowlog) or Redis [Engine Log](https://docs&#46;aws&#46;amazon&#46;com/AmazonElastiCache/latest/red-ug/Log_Delivery&#46;html#Log_contents-engine-log)&#46; See the documentation on [Amazon ElastiCache](https://docs.aws.amazon.com/AmazonElastiCache/latest/red-ug/Log_Delivery.html). See Log Delivery Configuration below for more details.
     */
    @JvmName("bwyvylmiiregyhtl")
    public suspend
    fun logDeliveryConfigurations(`value`: List<ClusterLogDeliveryConfigurationArgs>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.logDeliveryConfigurations = mapped
    }

    /**
     * @param argument Specifies the destination and format of Redis [SLOWLOG](https://redis&#46;io/commands/slowlog) or Redis [Engine Log](https://docs&#46;aws&#46;amazon&#46;com/AmazonElastiCache/latest/red-ug/Log_Delivery&#46;html#Log_contents-engine-log)&#46; See the documentation on [Amazon ElastiCache](https://docs.aws.amazon.com/AmazonElastiCache/latest/red-ug/Log_Delivery.html). See Log Delivery Configuration below for more details.
     */
    @JvmName("ckusscrvkcautywc")
    public suspend
    fun logDeliveryConfigurations(argument: List<suspend ClusterLogDeliveryConfigurationArgsBuilder.() -> Unit>) {
        val toBeMapped = argument.toList().map {
            ClusterLogDeliveryConfigurationArgsBuilder().applySuspend { it() }.build()
        }
        val mapped = of(toBeMapped)
        this.logDeliveryConfigurations = mapped
    }

    /**
     * @param argument Specifies the destination and format of Redis [SLOWLOG](https://redis&#46;io/commands/slowlog) or Redis [Engine Log](https://docs&#46;aws&#46;amazon&#46;com/AmazonElastiCache/latest/red-ug/Log_Delivery&#46;html#Log_contents-engine-log)&#46; See the documentation on [Amazon ElastiCache](https://docs.aws.amazon.com/AmazonElastiCache/latest/red-ug/Log_Delivery.html). See Log Delivery Configuration below for more details.
     */
    @JvmName("pavhhpsyrkqaqgnx")
    public suspend fun logDeliveryConfigurations(
        vararg
        argument: suspend ClusterLogDeliveryConfigurationArgsBuilder.() -> Unit,
    ) {
        val toBeMapped = argument.toList().map {
            ClusterLogDeliveryConfigurationArgsBuilder().applySuspend { it() }.build()
        }
        val mapped = of(toBeMapped)
        this.logDeliveryConfigurations = mapped
    }

    /**
     * @param argument Specifies the destination and format of Redis [SLOWLOG](https://redis&#46;io/commands/slowlog) or Redis [Engine Log](https://docs&#46;aws&#46;amazon&#46;com/AmazonElastiCache/latest/red-ug/Log_Delivery&#46;html#Log_contents-engine-log)&#46; See the documentation on [Amazon ElastiCache](https://docs.aws.amazon.com/AmazonElastiCache/latest/red-ug/Log_Delivery.html). See Log Delivery Configuration below for more details.
     */
    @JvmName("obqvbrjgjxyeuter")
    public suspend
    fun logDeliveryConfigurations(argument: suspend ClusterLogDeliveryConfigurationArgsBuilder.() -> Unit) {
        val toBeMapped = listOf(
            ClusterLogDeliveryConfigurationArgsBuilder().applySuspend {
                argument()
            }.build(),
        )
        val mapped = of(toBeMapped)
        this.logDeliveryConfigurations = mapped
    }

    /**
     * @param values Specifies the destination and format of Redis [SLOWLOG](https://redis&#46;io/commands/slowlog) or Redis [Engine Log](https://docs&#46;aws&#46;amazon&#46;com/AmazonElastiCache/latest/red-ug/Log_Delivery&#46;html#Log_contents-engine-log)&#46; See the documentation on [Amazon ElastiCache](https://docs.aws.amazon.com/AmazonElastiCache/latest/red-ug/Log_Delivery.html). See Log Delivery Configuration below for more details.
     */
    @JvmName("oshrcoeckwmnnomo")
    public suspend fun logDeliveryConfigurations(vararg values: ClusterLogDeliveryConfigurationArgs) {
        val toBeMapped = values.toList()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.logDeliveryConfigurations = mapped
    }

    /**
     * @param value Specifies the weekly time range for when maintenance
     * on the cache cluster is performed. The format is `ddd:hh24:mi-ddd:hh24:mi` (24H Clock UTC).
     * The minimum maintenance window is a 60 minute period. Example: `sun:05:00-sun:09:00`.
     */
    @JvmName("vhkhmurjiltrcupn")
    public suspend fun maintenanceWindow(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.maintenanceWindow = mapped
    }

    /**
     * @param value The IP versions for cache cluster connections. IPv6 is supported with Redis engine `6.2` onword or Memcached version `1.6.6` for all [Nitro system](https://aws.amazon.com/ec2/nitro/) instances. Valid values are `ipv4`, `ipv6` or `dual_stack`.
     */
    @JvmName("ymerefjbmlchgahe")
    public suspend fun networkType(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.networkType = mapped
    }

    /**
     * @param value The instance class used. See AWS documentation for information on [supported node types for Redis](https://docs&#46;aws&#46;amazon&#46;com/AmazonElastiCache/latest/red-ug/CacheNodes&#46;SupportedTypes&#46;html) and [guidance on selecting node types for Redis](https://docs&#46;aws&#46;amazon&#46;com/AmazonElastiCache/latest/red-ug/nodes-select-size&#46;html)&#46; See AWS documentation for information on [supported node types for Memcached](https://docs&#46;aws&#46;amazon&#46;com/AmazonElastiCache/latest/mem-ug/CacheNodes&#46;SupportedTypes&#46;html) and [guidance on selecting node types for Memcached](https://docs.aws.amazon.com/AmazonElastiCache/latest/mem-ug/nodes-select-size.html). For Memcached, changing this value will re-create the resource.
     */
    @JvmName("rfniptsmrxjryxef")
    public suspend fun nodeType(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.nodeType = mapped
    }

    /**
     * @param value ARN of an SNS topic to send ElastiCache notifications to. Example: `arn:aws:sns:us-east-1:012345678999:my_sns_topic`.
     */
    @JvmName("cewsjbfwnedkvvig")
    public suspend fun notificationTopicArn(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.notificationTopicArn = mapped
    }

    /**
     * @param value The initial number of cache nodes that the cache cluster will have. For Redis, this value must be 1. For Memcached, this value must be between 1 and 40. If this number is reduced on subsequent runs, the highest numbered nodes will be removed.
     */
    @JvmName("imfrnqwymkpugeey")
    public suspend fun numCacheNodes(`value`: Int?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.numCacheNodes = mapped
    }

    /**
     * @param value Specify the outpost mode that will apply to the cache cluster creation. Valid values are `"single-outpost"` and `"cross-outpost"`, however AWS currently only supports `"single-outpost"` mode.
     */
    @JvmName("hmoqspnmhdnmlfdo")
    public suspend fun outpostMode(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.outpostMode = mapped
    }

    /**
     * @param value The name of the parameter group to associate with this cache cluster.
     * The following arguments are optional:
     */
    @JvmName("jonoeykjcjgipsrt")
    public suspend fun parameterGroupName(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.parameterGroupName = mapped
    }

    /**
     * @param value The port number on which each of the cache nodes will accept connections. For Memcached the default is 11211, and for Redis the default port is 6379. Cannot be provided with `replication_group_id`. Changing this value will re-create the resource.
     */
    @JvmName("onuycgucqdiishgn")
    public suspend fun port(`value`: Int?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.port = mapped
    }

    /**
     * @param value List of the Availability Zones in which cache nodes are created. If you are creating your cluster in an Amazon VPC you can only locate nodes in Availability Zones that are associated with the subnets in the selected subnet group. The number of Availability Zones listed must equal the value of `num_cache_nodes`. If you want all the nodes in the same Availability Zone, use `availability_zone` instead, or repeat the Availability Zone multiple times in the list. Default: System chosen Availability Zones. Detecting drift of existing node availability zone is not currently supported. Updating this argument by itself to migrate existing node availability zones is not currently supported and will show a perpetual difference.
     */
    @JvmName("fkkkrmatoswpgvka")
    public suspend fun preferredAvailabilityZones(`value`: List<String>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.preferredAvailabilityZones = mapped
    }

    /**
     * @param values List of the Availability Zones in which cache nodes are created. If you are creating your cluster in an Amazon VPC you can only locate nodes in Availability Zones that are associated with the subnets in the selected subnet group. The number of Availability Zones listed must equal the value of `num_cache_nodes`. If you want all the nodes in the same Availability Zone, use `availability_zone` instead, or repeat the Availability Zone multiple times in the list. Default: System chosen Availability Zones. Detecting drift of existing node availability zone is not currently supported. Updating this argument by itself to migrate existing node availability zones is not currently supported and will show a perpetual difference.
     */
    @JvmName("ajsnouipxemkrokd")
    public suspend fun preferredAvailabilityZones(vararg values: String) {
        val toBeMapped = values.toList()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.preferredAvailabilityZones = mapped
    }

    /**
     * @param value The outpost ARN in which the cache cluster will be created.
     */
    @JvmName("qmkqtuwonkejpbrq")
    public suspend fun preferredOutpostArn(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.preferredOutpostArn = mapped
    }

    /**
     * @param value ID of the replication group to which this cluster should belong. If this parameter is specified, the cluster is added to the specified replication group as a read replica; otherwise, the cluster is a standalone primary that is not part of any replication group.
     */
    @JvmName("doqdfoiokhlcyonu")
    public suspend fun replicationGroupId(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.replicationGroupId = mapped
    }

    /**
     * @param value One or more VPC security groups associated with the cache cluster. Cannot be provided with `replication_group_id.`
     */
    @JvmName("tbjbctfljifhohrc")
    public suspend fun securityGroupIds(`value`: List<String>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.securityGroupIds = mapped
    }

    /**
     * @param values One or more VPC security groups associated with the cache cluster. Cannot be provided with `replication_group_id.`
     */
    @JvmName("dfxcxtskxtgowfql")
    public suspend fun securityGroupIds(vararg values: String) {
        val toBeMapped = values.toList()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.securityGroupIds = mapped
    }

    /**
     * @param value Single-element string list containing an Amazon Resource Name (ARN) of a Redis RDB snapshot file stored in Amazon S3. The object name cannot contain any commas. Changing `snapshot_arns` forces a new resource.
     */
    @JvmName("ljitjbhcsxmnxlji")
    public suspend fun snapshotArns(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.snapshotArns = mapped
    }

    /**
     * @param value Name of a snapshot from which to restore data into the new node group. Changing `snapshot_name` forces a new resource.
     */
    @JvmName("iwcikfhumepxpcag")
    public suspend fun snapshotName(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.snapshotName = mapped
    }

    /**
     * @param value Number of days for which ElastiCache will retain automatic cache cluster snapshots before deleting them. For example, if you set SnapshotRetentionLimit to 5, then a snapshot that was taken today will be retained for 5 days before being deleted. If the value of SnapshotRetentionLimit is set to zero (0), backups are turned off. Please note that setting a `snapshot_retention_limit` is not supported on cache.t1.micro cache nodes
     */
    @JvmName("tfmnqrhtkyasevft")
    public suspend fun snapshotRetentionLimit(`value`: Int?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.snapshotRetentionLimit = mapped
    }

    /**
     * @param value Daily time range (in UTC) during which ElastiCache will begin taking a daily snapshot of your cache cluster. Example: 05:00-09:00
     */
    @JvmName("bjnqtfjiakcioxwp")
    public suspend fun snapshotWindow(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.snapshotWindow = mapped
    }

    /**
     * @param value Name of the subnet group to be used for the cache cluster. Changing this value will re-create the resource. Cannot be provided with `replication_group_id.`
     */
    @JvmName("ajgthtsyrshxdtet")
    public suspend fun subnetGroupName(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.subnetGroupName = mapped
    }

    /**
     * @param value Map of tags to assign to the resource. If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
     */
    @JvmName("rnlhnnpwpfgupaeb")
    public suspend fun tags(`value`: Map<String, String>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.tags = mapped
    }

    /**
     * @param values Map of tags to assign to the resource. If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
     */
    @JvmName("jllwgoplkrerldkt")
    public fun tags(vararg values: Pair<String, String>) {
        val toBeMapped = values.toMap()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.tags = mapped
    }

    /**
     * @param value Enable encryption in-transit. Supported only with Memcached versions `1.6.12` and later, running in a VPC. See the [ElastiCache in-transit encryption](https://docs.aws.amazon.com/AmazonElastiCache/latest/mem-ug/in-transit-encryption-mc.html) documentation for more details.
     */
    @JvmName("ogjirqqqdevubmbf")
    public suspend fun transitEncryptionEnabled(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.transitEncryptionEnabled = mapped
    }

    internal fun build(): ClusterArgs = ClusterArgs(
        applyImmediately = applyImmediately,
        autoMinorVersionUpgrade = autoMinorVersionUpgrade,
        availabilityZone = availabilityZone,
        azMode = azMode,
        clusterId = clusterId,
        engine = engine,
        engineVersion = engineVersion,
        finalSnapshotIdentifier = finalSnapshotIdentifier,
        ipDiscovery = ipDiscovery,
        logDeliveryConfigurations = logDeliveryConfigurations,
        maintenanceWindow = maintenanceWindow,
        networkType = networkType,
        nodeType = nodeType,
        notificationTopicArn = notificationTopicArn,
        numCacheNodes = numCacheNodes,
        outpostMode = outpostMode,
        parameterGroupName = parameterGroupName,
        port = port,
        preferredAvailabilityZones = preferredAvailabilityZones,
        preferredOutpostArn = preferredOutpostArn,
        replicationGroupId = replicationGroupId,
        securityGroupIds = securityGroupIds,
        snapshotArns = snapshotArns,
        snapshotName = snapshotName,
        snapshotRetentionLimit = snapshotRetentionLimit,
        snapshotWindow = snapshotWindow,
        subnetGroupName = subnetGroupName,
        tags = tags,
        transitEncryptionEnabled = transitEncryptionEnabled,
    )
}
