@file:Suppress("NAME_SHADOWING", "DEPRECATION")

package com.pulumi.aws.elasticache.kotlin

import com.pulumi.aws.elasticache.ReplicationGroupArgs.builder
import com.pulumi.aws.elasticache.kotlin.inputs.ReplicationGroupLogDeliveryConfigurationArgs
import com.pulumi.aws.elasticache.kotlin.inputs.ReplicationGroupLogDeliveryConfigurationArgsBuilder
import com.pulumi.core.Output
import com.pulumi.core.Output.of
import com.pulumi.kotlin.ConvertibleToJava
import com.pulumi.kotlin.PulumiTagMarker
import com.pulumi.kotlin.applySuspend
import kotlin.Boolean
import kotlin.Int
import kotlin.Pair
import kotlin.String
import kotlin.Suppress
import kotlin.Unit
import kotlin.collections.List
import kotlin.collections.Map
import kotlin.jvm.JvmName

/**
 * Provides an ElastiCache Replication Group resource.
 * For working with a [Memcached cluster](https://docs.aws.amazon.com/AmazonElastiCache/latest/mem-ug/WhatIs.html) or a
 * [single-node Redis instance (Cluster Mode Disabled)](https://docs.aws.amazon.com/AmazonElastiCache/latest/red-ug/WhatIs.html),
 * see the `aws.elasticache.Cluster` resource.
 * > **Note:** When you change an attribute, such as `engine_version`, by
 * default the ElastiCache API applies it in the next maintenance window. Because
 * of this, this provider may report a difference in its planning phase because the
 * actual modification has not yet taken place. You can use the
 * `apply_immediately` flag to instruct the service to apply the change
 * immediately. Using `apply_immediately` can result in a brief downtime as
 * servers reboots.
 * See the AWS Documentation on
 * [Modifying an ElastiCache Cache Cluster](https://docs.aws.amazon.com/AmazonElastiCache/latest/red-ug/Clusters.Modify.html)
 * for more information.
 * > **Note:** Any attribute changes that re-create the resource will be applied immediately, regardless of the value of `apply_immediately`.
 * > **Note:** Be aware of the terminology collision around "cluster" for `aws.elasticache.ReplicationGroup`. For example, it is possible to create a ["Cluster Mode Disabled [Redis] Cluster"](https://docs&#46;aws&#46;amazon&#46;com/AmazonElastiCache/latest/red-ug/Clusters&#46;Create&#46;CON&#46;Redis&#46;html)&#46; With "Cluster Mode Enabled", the data will be stored in shards (called "node groups")&#46; See [Redis Cluster Configuration](https://docs.aws.amazon.com/AmazonElastiCache/latest/red-ug/cluster-create-determine-requirements.html#redis-cluster-configuration) for a diagram of the differences. To enable cluster mode, use a parameter group that has cluster mode enabled. The default parameter groups provided by AWS end with ".cluster.on", for example `default.redis6.x.cluster.on`.
 * ## Example Usage
 * ### Redis Cluster Mode Disabled
 * To create a single shard primary with single read replica:
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.elasticache.ReplicationGroup;
 * import com.pulumi.aws.elasticache.ReplicationGroupArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var example = new ReplicationGroup("example", ReplicationGroupArgs.builder()
 *             .automaticFailoverEnabled(true)
 *             .description("example description")
 *             .nodeType("cache.m4.large")
 *             .numCacheClusters(2)
 *             .parameterGroupName("default.redis3.2")
 *             .port(6379)
 *             .preferredCacheClusterAzs(
 *                 "us-west-2a",
 *                 "us-west-2b")
 *             .build());
 *     }
 * }
 * ```
 * ### Redis Cluster Mode Enabled
 * To create two shards with a primary and a single read replica each:
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.elasticache.ReplicationGroup;
 * import com.pulumi.aws.elasticache.ReplicationGroupArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var baz = new ReplicationGroup("baz", ReplicationGroupArgs.builder()
 *             .automaticFailoverEnabled(true)
 *             .description("example description")
 *             .nodeType("cache.t2.small")
 *             .numNodeGroups(2)
 *             .parameterGroupName("default.redis3.2.cluster.on")
 *             .port(6379)
 *             .replicasPerNodeGroup(1)
 *             .build());
 *     }
 * }
 * ```
 * ### Redis Log Delivery configuration
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.elasticache.ReplicationGroup;
 * import com.pulumi.aws.elasticache.ReplicationGroupArgs;
 * import com.pulumi.aws.elasticache.inputs.ReplicationGroupLogDeliveryConfigurationArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var test = new ReplicationGroup("test", ReplicationGroupArgs.builder()
 *             .description("test description")
 *             .nodeType("cache.t3.small")
 *             .port(6379)
 *             .applyImmediately(true)
 *             .autoMinorVersionUpgrade(false)
 *             .maintenanceWindow("tue:06:30-tue:07:30")
 *             .snapshotWindow("01:00-02:00")
 *             .logDeliveryConfigurations(
 *                 ReplicationGroupLogDeliveryConfigurationArgs.builder()
 *                     .destination(aws_cloudwatch_log_group.example().name())
 *                     .destinationType("cloudwatch-logs")
 *                     .logFormat("text")
 *                     .logType("slow-log")
 *                     .build(),
 *                 ReplicationGroupLogDeliveryConfigurationArgs.builder()
 *                     .destination(aws_kinesis_firehose_delivery_stream.example().name())
 *                     .destinationType("kinesis-firehose")
 *                     .logFormat("json")
 *                     .logType("engine-log")
 *                     .build())
 *             .build());
 *     }
 * }
 * ```
 * ### Creating a secondary replication group for a global replication group
 * A Global Replication Group can have one one two secondary Replication Groups in different regions. These are added to an existing Global Replication Group.
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.elasticache.ReplicationGroup;
 * import com.pulumi.aws.elasticache.ReplicationGroupArgs;
 * import com.pulumi.aws.elasticache.GlobalReplicationGroup;
 * import com.pulumi.aws.elasticache.GlobalReplicationGroupArgs;
 * import com.pulumi.resources.CustomResourceOptions;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var primary = new ReplicationGroup("primary", ReplicationGroupArgs.builder()
 *             .description("primary replication group")
 *             .engine("redis")
 *             .engineVersion("5.0.6")
 *             .nodeType("cache.m5.large")
 *             .numCacheClusters(1)
 *             .build(), CustomResourceOptions.builder()
 *                 .provider(aws.other_region())
 *                 .build());
 *         var example = new GlobalReplicationGroup("example", GlobalReplicationGroupArgs.builder()
 *             .globalReplicationGroupIdSuffix("example")
 *             .primaryReplicationGroupId(primary.id())
 *             .build(), CustomResourceOptions.builder()
 *                 .provider(aws.other_region())
 *                 .build());
 *         var secondary = new ReplicationGroup("secondary", ReplicationGroupArgs.builder()
 *             .description("secondary replication group")
 *             .globalReplicationGroupId(example.globalReplicationGroupId())
 *             .numCacheClusters(1)
 *             .build());
 *     }
 * }
 * ```
 * ### Redis AUTH and In-Transit Encryption Enabled
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.elasticache.ReplicationGroup;
 * import com.pulumi.aws.elasticache.ReplicationGroupArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var example = new ReplicationGroup("example", ReplicationGroupArgs.builder()
 *             .description("example with authentication")
 *             .nodeType("cache.t2.micro")
 *             .numCacheClusters(1)
 *             .port(6379)
 *             .subnetGroupName(aws_elasticache_subnet_group.example().name())
 *             .securityGroupIds(aws_security_group.example().id())
 *             .parameterGroupName("default.redis5.0")
 *             .engineVersion("5.0.6")
 *             .transitEncryptionEnabled(true)
 *             .authToken("abcdefgh1234567890")
 *             .authTokenUpdateStrategy("ROTATE")
 *             .build());
 *     }
 * }
 * ```
 * ## Import
 * Using `pulumi import`, import ElastiCache Replication Groups using the `replication_group_id`. For example:
 * ```sh
 *  $ pulumi import aws:elasticache/replicationGroup:ReplicationGroup my_replication_group replication-group-1
 * ```
 * @property applyImmediately Specifies whether any modifications are applied immediately, or during the next maintenance window. Default is `false`.
 * @property atRestEncryptionEnabled Whether to enable encryption at rest.
 * @property authToken Password used to access a password protected server. Can be specified only if `transit_encryption_enabled = true`.
 * @property authTokenUpdateStrategy Strategy to use when updating the `auth_token`. Valid values are `SET`, `ROTATE`, and `DELETE`. Defaults to `ROTATE`.
 * @property autoMinorVersionUpgrade Specifies whether minor version engine upgrades will be applied automatically to the underlying Cache Cluster instances during the maintenance window.
 * Only supported for engine type `"redis"` and if the engine version is 6 or higher.
 * Defaults to `true`.
 * @property automaticFailoverEnabled Specifies whether a read-only replica will be automatically promoted to read/write primary if the existing primary fails. If enabled, `num_cache_clusters` must be greater than 1. Must be enabled for Redis (cluster mode enabled) replication groups. Defaults to `false`.
 * @property dataTieringEnabled Enables data tiering. Data tiering is only supported for replication groups using the r6gd node type. This parameter must be set to `true` when using r6gd nodes.
 * @property description User-created description for the replication group. Must not be empty.
 * @property engine Name of the cache engine to be used for the clusters in this replication group. The only valid value is `redis`.
 * @property engineVersion Version number of the cache engine to be used for the cache clusters in this replication group.
 * If the version is 7 or higher, the major and minor version should be set, e.g., `7.2`.
 * If the version is 6, the major and minor version can be set, e.g., `6.2`,
 * or the minor version can be unspecified which will use the latest version at creation time, e.g., `6.x`.
 * Otherwise, specify the full version desired, e.g., `5.0.6`.
 * The actual engine version used is returned in the attribute `engine_version_actual`, see Attribute Reference below.
 * @property finalSnapshotIdentifier The name of your final node group (shard) snapshot. ElastiCache creates the snapshot from the primary node in the cluster. If omitted, no final snapshot will be made.
 * @property globalReplicationGroupId The ID of the global replication group to which this replication group should belong. If this parameter is specified, the replication group is added to the specified global replication group as a secondary replication group; otherwise, the replication group is not part of any global replication group. If `global_replication_group_id` is set, the `num_node_groups` parameter cannot be set.
 * @property ipDiscovery The IP version to advertise in the discovery protocol. Valid values are `ipv4` or `ipv6`.
 * @property kmsKeyId The ARN of the key that you wish to use if encrypting at rest. If not supplied, uses service managed encryption. Can be specified only if `at_rest_encryption_enabled = true`.
 * @property logDeliveryConfigurations Specifies the destination and format of Redis [SLOWLOG](https://redis&#46;io/commands/slowlog) or Redis [Engine Log](https://docs&#46;aws&#46;amazon&#46;com/AmazonElastiCache/latest/red-ug/Log_Delivery&#46;html#Log_contents-engine-log)&#46; See the documentation on [Amazon ElastiCache](https://docs.aws.amazon.com/AmazonElastiCache/latest/red-ug/Log_Delivery.html#Log_contents-engine-log). See Log Delivery Configuration below for more details.
 * @property maintenanceWindow Specifies the weekly time range for when maintenance on the cache cluster is performed. The format is `ddd:hh24:mi-ddd:hh24:mi` (24H Clock UTC). The minimum maintenance window is a 60 minute period. Example: `sun:05:00-sun:09:00`
 * @property multiAzEnabled Specifies whether to enable Multi-AZ Support for the replication group. If `true`, `automatic_failover_enabled` must also be enabled. Defaults to `false`.
 * @property networkType The IP versions for cache cluster connections. Valid values are `ipv4`, `ipv6` or `dual_stack`.
 * @property nodeType Instance class to be used. See AWS documentation for information on [supported node types](https://docs&#46;aws&#46;amazon&#46;com/AmazonElastiCache/latest/red-ug/CacheNodes&#46;SupportedTypes&#46;html) and [guidance on selecting node types](https://docs.aws.amazon.com/AmazonElastiCache/latest/red-ug/nodes-select-size.html). Required unless `global_replication_group_id` is set. Cannot be set if `global_replication_group_id` is set.
 * @property notificationTopicArn ARN of an SNS topic to send ElastiCache notifications to. Example: `arn:aws:sns:us-east-1:012345678999:my_sns_topic`
 * @property numCacheClusters Number of cache clusters (primary and replicas) this replication group will have. If Multi-AZ is enabled, the value of this parameter must be at least 2. Updates will occur before other modifications. Conflicts with `num_node_groups`. Defaults to `1`.
 * @property numNodeGroups Number of node groups (shards) for this Redis replication group.
 * Changing this number will trigger a resizing operation before other settings modifications.
 * @property parameterGroupName Name of the parameter group to associate with this replication group. If this argument is omitted, the default cache parameter group for the specified engine is used. To enable "cluster mode", i.e., data sharding, use a parameter group that has the parameter `cluster-enabled` set to true.
 * @property port Port number on which each of the cache nodes will accept connections. For Memcache the default is 11211, and for Redis the default port is 6379.
 * @property preferredCacheClusterAzs List of EC2 availability zones in which the replication group's cache clusters will be created. The order of the availability zones in the list is considered. The first item in the list will be the primary node. Ignored when updating.
 * @property replicasPerNodeGroup Number of replica nodes in each node group.
 * Changing this number will trigger a resizing operation before other settings modifications.
 * Valid values are 0 to 5.
 * @property replicationGroupId Replication group identifier. This parameter is stored as a lowercase string.
 * The following arguments are optional:
 * @property securityGroupIds IDs of one or more Amazon VPC security groups associated with this replication group. Use this parameter only when you are creating a replication group in an Amazon Virtual Private Cloud.
 * @property securityGroupNames Names of one or more Amazon VPC security groups associated with this replication group. Use this parameter only when you are creating a replication group in an Amazon Virtual Private Cloud.
 * @property snapshotArns List of ARNs that identify Redis RDB snapshot files stored in Amazon S3. The names object names cannot contain any commas.
 * @property snapshotName Name of a snapshot from which to restore data into the new node group. Changing the `snapshot_name` forces a new resource.
 * @property snapshotRetentionLimit Number of days for which ElastiCache will retain automatic cache cluster snapshots before deleting them. For example, if you set SnapshotRetentionLimit to 5, then a snapshot that was taken today will be retained for 5 days before being deleted. If the value of `snapshot_retention_limit` is set to zero (0), backups are turned off. Please note that setting a `snapshot_retention_limit` is not supported on cache.t1.micro cache nodes
 * @property snapshotWindow Daily time range (in UTC) during which ElastiCache will begin taking a daily snapshot of your cache cluster. The minimum snapshot window is a 60 minute period. Example: `05:00-09:00`
 * @property subnetGroupName Name of the cache subnet group to be used for the replication group.
 * @property tags Map of tags to assign to the resource. Adding tags to this resource will add or overwrite any existing tags on the clusters in the replication group and not to the group itself. If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
 * @property transitEncryptionEnabled Whether to enable encryption in transit.
 * @property userGroupIds User Group ID to associate with the replication group. Only a maximum of one (1) user group ID is valid. **NOTE:** This argument _is_ a set because the AWS specification allows for multiple IDs. However, in practice, AWS only allows a maximum size of one.
 */
public data class ReplicationGroupArgs(
    public val applyImmediately: Output<Boolean>? = null,
    public val atRestEncryptionEnabled: Output<Boolean>? = null,
    public val authToken: Output<String>? = null,
    public val authTokenUpdateStrategy: Output<String>? = null,
    public val autoMinorVersionUpgrade: Output<Boolean>? = null,
    public val automaticFailoverEnabled: Output<Boolean>? = null,
    public val dataTieringEnabled: Output<Boolean>? = null,
    public val description: Output<String>? = null,
    public val engine: Output<String>? = null,
    public val engineVersion: Output<String>? = null,
    public val finalSnapshotIdentifier: Output<String>? = null,
    public val globalReplicationGroupId: Output<String>? = null,
    public val ipDiscovery: Output<String>? = null,
    public val kmsKeyId: Output<String>? = null,
    public val logDeliveryConfigurations: Output<List<ReplicationGroupLogDeliveryConfigurationArgs>>? =
        null,
    public val maintenanceWindow: Output<String>? = null,
    public val multiAzEnabled: Output<Boolean>? = null,
    public val networkType: Output<String>? = null,
    public val nodeType: Output<String>? = null,
    public val notificationTopicArn: Output<String>? = null,
    public val numCacheClusters: Output<Int>? = null,
    public val numNodeGroups: Output<Int>? = null,
    public val parameterGroupName: Output<String>? = null,
    public val port: Output<Int>? = null,
    public val preferredCacheClusterAzs: Output<List<String>>? = null,
    public val replicasPerNodeGroup: Output<Int>? = null,
    public val replicationGroupId: Output<String>? = null,
    public val securityGroupIds: Output<List<String>>? = null,
    public val securityGroupNames: Output<List<String>>? = null,
    public val snapshotArns: Output<List<String>>? = null,
    public val snapshotName: Output<String>? = null,
    public val snapshotRetentionLimit: Output<Int>? = null,
    public val snapshotWindow: Output<String>? = null,
    public val subnetGroupName: Output<String>? = null,
    public val tags: Output<Map<String, String>>? = null,
    public val transitEncryptionEnabled: Output<Boolean>? = null,
    public val userGroupIds: Output<List<String>>? = null,
) : ConvertibleToJava<com.pulumi.aws.elasticache.ReplicationGroupArgs> {
    override fun toJava(): com.pulumi.aws.elasticache.ReplicationGroupArgs =
        com.pulumi.aws.elasticache.ReplicationGroupArgs.builder()
            .applyImmediately(applyImmediately?.applyValue({ args0 -> args0 }))
            .atRestEncryptionEnabled(atRestEncryptionEnabled?.applyValue({ args0 -> args0 }))
            .authToken(authToken?.applyValue({ args0 -> args0 }))
            .authTokenUpdateStrategy(authTokenUpdateStrategy?.applyValue({ args0 -> args0 }))
            .autoMinorVersionUpgrade(autoMinorVersionUpgrade?.applyValue({ args0 -> args0 }))
            .automaticFailoverEnabled(automaticFailoverEnabled?.applyValue({ args0 -> args0 }))
            .dataTieringEnabled(dataTieringEnabled?.applyValue({ args0 -> args0 }))
            .description(description?.applyValue({ args0 -> args0 }))
            .engine(engine?.applyValue({ args0 -> args0 }))
            .engineVersion(engineVersion?.applyValue({ args0 -> args0 }))
            .finalSnapshotIdentifier(finalSnapshotIdentifier?.applyValue({ args0 -> args0 }))
            .globalReplicationGroupId(globalReplicationGroupId?.applyValue({ args0 -> args0 }))
            .ipDiscovery(ipDiscovery?.applyValue({ args0 -> args0 }))
            .kmsKeyId(kmsKeyId?.applyValue({ args0 -> args0 }))
            .logDeliveryConfigurations(
                logDeliveryConfigurations?.applyValue({ args0 ->
                    args0.map({ args0 ->
                        args0.let({ args0 -> args0.toJava() })
                    })
                }),
            )
            .maintenanceWindow(maintenanceWindow?.applyValue({ args0 -> args0 }))
            .multiAzEnabled(multiAzEnabled?.applyValue({ args0 -> args0 }))
            .networkType(networkType?.applyValue({ args0 -> args0 }))
            .nodeType(nodeType?.applyValue({ args0 -> args0 }))
            .notificationTopicArn(notificationTopicArn?.applyValue({ args0 -> args0 }))
            .numCacheClusters(numCacheClusters?.applyValue({ args0 -> args0 }))
            .numNodeGroups(numNodeGroups?.applyValue({ args0 -> args0 }))
            .parameterGroupName(parameterGroupName?.applyValue({ args0 -> args0 }))
            .port(port?.applyValue({ args0 -> args0 }))
            .preferredCacheClusterAzs(
                preferredCacheClusterAzs?.applyValue({ args0 ->
                    args0.map({ args0 ->
                        args0
                    })
                }),
            )
            .replicasPerNodeGroup(replicasPerNodeGroup?.applyValue({ args0 -> args0 }))
            .replicationGroupId(replicationGroupId?.applyValue({ args0 -> args0 }))
            .securityGroupIds(securityGroupIds?.applyValue({ args0 -> args0.map({ args0 -> args0 }) }))
            .securityGroupNames(securityGroupNames?.applyValue({ args0 -> args0.map({ args0 -> args0 }) }))
            .snapshotArns(snapshotArns?.applyValue({ args0 -> args0.map({ args0 -> args0 }) }))
            .snapshotName(snapshotName?.applyValue({ args0 -> args0 }))
            .snapshotRetentionLimit(snapshotRetentionLimit?.applyValue({ args0 -> args0 }))
            .snapshotWindow(snapshotWindow?.applyValue({ args0 -> args0 }))
            .subnetGroupName(subnetGroupName?.applyValue({ args0 -> args0 }))
            .tags(tags?.applyValue({ args0 -> args0.map({ args0 -> args0.key.to(args0.value) }).toMap() }))
            .transitEncryptionEnabled(transitEncryptionEnabled?.applyValue({ args0 -> args0 }))
            .userGroupIds(userGroupIds?.applyValue({ args0 -> args0.map({ args0 -> args0 }) })).build()
}

/**
 * Builder for [ReplicationGroupArgs].
 */
@PulumiTagMarker
public class ReplicationGroupArgsBuilder internal constructor() {
    private var applyImmediately: Output<Boolean>? = null

    private var atRestEncryptionEnabled: Output<Boolean>? = null

    private var authToken: Output<String>? = null

    private var authTokenUpdateStrategy: Output<String>? = null

    private var autoMinorVersionUpgrade: Output<Boolean>? = null

    private var automaticFailoverEnabled: Output<Boolean>? = null

    private var dataTieringEnabled: Output<Boolean>? = null

    private var description: Output<String>? = null

    private var engine: Output<String>? = null

    private var engineVersion: Output<String>? = null

    private var finalSnapshotIdentifier: Output<String>? = null

    private var globalReplicationGroupId: Output<String>? = null

    private var ipDiscovery: Output<String>? = null

    private var kmsKeyId: Output<String>? = null

    private var logDeliveryConfigurations: Output<List<ReplicationGroupLogDeliveryConfigurationArgs>>? =
        null

    private var maintenanceWindow: Output<String>? = null

    private var multiAzEnabled: Output<Boolean>? = null

    private var networkType: Output<String>? = null

    private var nodeType: Output<String>? = null

    private var notificationTopicArn: Output<String>? = null

    private var numCacheClusters: Output<Int>? = null

    private var numNodeGroups: Output<Int>? = null

    private var parameterGroupName: Output<String>? = null

    private var port: Output<Int>? = null

    private var preferredCacheClusterAzs: Output<List<String>>? = null

    private var replicasPerNodeGroup: Output<Int>? = null

    private var replicationGroupId: Output<String>? = null

    private var securityGroupIds: Output<List<String>>? = null

    private var securityGroupNames: Output<List<String>>? = null

    private var snapshotArns: Output<List<String>>? = null

    private var snapshotName: Output<String>? = null

    private var snapshotRetentionLimit: Output<Int>? = null

    private var snapshotWindow: Output<String>? = null

    private var subnetGroupName: Output<String>? = null

    private var tags: Output<Map<String, String>>? = null

    private var transitEncryptionEnabled: Output<Boolean>? = null

    private var userGroupIds: Output<List<String>>? = null

    /**
     * @param value Specifies whether any modifications are applied immediately, or during the next maintenance window. Default is `false`.
     */
    @JvmName("lsxxiobkdrwqylhb")
    public suspend fun applyImmediately(`value`: Output<Boolean>) {
        this.applyImmediately = value
    }

    /**
     * @param value Whether to enable encryption at rest.
     */
    @JvmName("tshqidforyxucjao")
    public suspend fun atRestEncryptionEnabled(`value`: Output<Boolean>) {
        this.atRestEncryptionEnabled = value
    }

    /**
     * @param value Password used to access a password protected server. Can be specified only if `transit_encryption_enabled = true`.
     */
    @JvmName("miyusroyoabvmxhw")
    public suspend fun authToken(`value`: Output<String>) {
        this.authToken = value
    }

    /**
     * @param value Strategy to use when updating the `auth_token`. Valid values are `SET`, `ROTATE`, and `DELETE`. Defaults to `ROTATE`.
     */
    @JvmName("rxhxahpynltgywct")
    public suspend fun authTokenUpdateStrategy(`value`: Output<String>) {
        this.authTokenUpdateStrategy = value
    }

    /**
     * @param value Specifies whether minor version engine upgrades will be applied automatically to the underlying Cache Cluster instances during the maintenance window.
     * Only supported for engine type `"redis"` and if the engine version is 6 or higher.
     * Defaults to `true`.
     */
    @JvmName("ibwgvgjydstifrje")
    public suspend fun autoMinorVersionUpgrade(`value`: Output<Boolean>) {
        this.autoMinorVersionUpgrade = value
    }

    /**
     * @param value Specifies whether a read-only replica will be automatically promoted to read/write primary if the existing primary fails. If enabled, `num_cache_clusters` must be greater than 1. Must be enabled for Redis (cluster mode enabled) replication groups. Defaults to `false`.
     */
    @JvmName("qgbkrtvtoseaypwm")
    public suspend fun automaticFailoverEnabled(`value`: Output<Boolean>) {
        this.automaticFailoverEnabled = value
    }

    /**
     * @param value Enables data tiering. Data tiering is only supported for replication groups using the r6gd node type. This parameter must be set to `true` when using r6gd nodes.
     */
    @JvmName("tlshhcccoyncjcvd")
    public suspend fun dataTieringEnabled(`value`: Output<Boolean>) {
        this.dataTieringEnabled = value
    }

    /**
     * @param value User-created description for the replication group. Must not be empty.
     */
    @JvmName("hnygxblyoacbqbiv")
    public suspend fun description(`value`: Output<String>) {
        this.description = value
    }

    /**
     * @param value Name of the cache engine to be used for the clusters in this replication group. The only valid value is `redis`.
     */
    @JvmName("ccdqmrpbmbdqnvhm")
    public suspend fun engine(`value`: Output<String>) {
        this.engine = value
    }

    /**
     * @param value Version number of the cache engine to be used for the cache clusters in this replication group.
     * If the version is 7 or higher, the major and minor version should be set, e.g., `7.2`.
     * If the version is 6, the major and minor version can be set, e.g., `6.2`,
     * or the minor version can be unspecified which will use the latest version at creation time, e.g., `6.x`.
     * Otherwise, specify the full version desired, e.g., `5.0.6`.
     * The actual engine version used is returned in the attribute `engine_version_actual`, see Attribute Reference below.
     */
    @JvmName("vpvhoaldlrnudnow")
    public suspend fun engineVersion(`value`: Output<String>) {
        this.engineVersion = value
    }

    /**
     * @param value The name of your final node group (shard) snapshot. ElastiCache creates the snapshot from the primary node in the cluster. If omitted, no final snapshot will be made.
     */
    @JvmName("edajdryplblbsaqh")
    public suspend fun finalSnapshotIdentifier(`value`: Output<String>) {
        this.finalSnapshotIdentifier = value
    }

    /**
     * @param value The ID of the global replication group to which this replication group should belong. If this parameter is specified, the replication group is added to the specified global replication group as a secondary replication group; otherwise, the replication group is not part of any global replication group. If `global_replication_group_id` is set, the `num_node_groups` parameter cannot be set.
     */
    @JvmName("ohfelynsobqwfyge")
    public suspend fun globalReplicationGroupId(`value`: Output<String>) {
        this.globalReplicationGroupId = value
    }

    /**
     * @param value The IP version to advertise in the discovery protocol. Valid values are `ipv4` or `ipv6`.
     */
    @JvmName("llgttuouijfnvtxc")
    public suspend fun ipDiscovery(`value`: Output<String>) {
        this.ipDiscovery = value
    }

    /**
     * @param value The ARN of the key that you wish to use if encrypting at rest. If not supplied, uses service managed encryption. Can be specified only if `at_rest_encryption_enabled = true`.
     */
    @JvmName("cdwceqdkavceqsxe")
    public suspend fun kmsKeyId(`value`: Output<String>) {
        this.kmsKeyId = value
    }

    /**
     * @param value Specifies the destination and format of Redis [SLOWLOG](https://redis&#46;io/commands/slowlog) or Redis [Engine Log](https://docs&#46;aws&#46;amazon&#46;com/AmazonElastiCache/latest/red-ug/Log_Delivery&#46;html#Log_contents-engine-log)&#46; See the documentation on [Amazon ElastiCache](https://docs.aws.amazon.com/AmazonElastiCache/latest/red-ug/Log_Delivery.html#Log_contents-engine-log). See Log Delivery Configuration below for more details.
     */
    @JvmName("civyrckgnvjkvmut")
    public suspend
    fun logDeliveryConfigurations(`value`: Output<List<ReplicationGroupLogDeliveryConfigurationArgs>>) {
        this.logDeliveryConfigurations = value
    }

    @JvmName("ligmceesdjgnyidt")
    public suspend fun logDeliveryConfigurations(
        vararg
        values: Output<ReplicationGroupLogDeliveryConfigurationArgs>,
    ) {
        this.logDeliveryConfigurations = Output.all(values.asList())
    }

    /**
     * @param values Specifies the destination and format of Redis [SLOWLOG](https://redis&#46;io/commands/slowlog) or Redis [Engine Log](https://docs&#46;aws&#46;amazon&#46;com/AmazonElastiCache/latest/red-ug/Log_Delivery&#46;html#Log_contents-engine-log)&#46; See the documentation on [Amazon ElastiCache](https://docs.aws.amazon.com/AmazonElastiCache/latest/red-ug/Log_Delivery.html#Log_contents-engine-log). See Log Delivery Configuration below for more details.
     */
    @JvmName("rrqnrhfillvrqcrp")
    public suspend
    fun logDeliveryConfigurations(values: List<Output<ReplicationGroupLogDeliveryConfigurationArgs>>) {
        this.logDeliveryConfigurations = Output.all(values)
    }

    /**
     * @param value Specifies the weekly time range for when maintenance on the cache cluster is performed. The format is `ddd:hh24:mi-ddd:hh24:mi` (24H Clock UTC). The minimum maintenance window is a 60 minute period. Example: `sun:05:00-sun:09:00`
     */
    @JvmName("gtinfiwunamyoinl")
    public suspend fun maintenanceWindow(`value`: Output<String>) {
        this.maintenanceWindow = value
    }

    /**
     * @param value Specifies whether to enable Multi-AZ Support for the replication group. If `true`, `automatic_failover_enabled` must also be enabled. Defaults to `false`.
     */
    @JvmName("asjxcdfbrnintcio")
    public suspend fun multiAzEnabled(`value`: Output<Boolean>) {
        this.multiAzEnabled = value
    }

    /**
     * @param value The IP versions for cache cluster connections. Valid values are `ipv4`, `ipv6` or `dual_stack`.
     */
    @JvmName("eunhsxmnqtsgknmv")
    public suspend fun networkType(`value`: Output<String>) {
        this.networkType = value
    }

    /**
     * @param value Instance class to be used. See AWS documentation for information on [supported node types](https://docs&#46;aws&#46;amazon&#46;com/AmazonElastiCache/latest/red-ug/CacheNodes&#46;SupportedTypes&#46;html) and [guidance on selecting node types](https://docs.aws.amazon.com/AmazonElastiCache/latest/red-ug/nodes-select-size.html). Required unless `global_replication_group_id` is set. Cannot be set if `global_replication_group_id` is set.
     */
    @JvmName("oqxkcvofkevwldru")
    public suspend fun nodeType(`value`: Output<String>) {
        this.nodeType = value
    }

    /**
     * @param value ARN of an SNS topic to send ElastiCache notifications to. Example: `arn:aws:sns:us-east-1:012345678999:my_sns_topic`
     */
    @JvmName("mtdybkktiaitgucn")
    public suspend fun notificationTopicArn(`value`: Output<String>) {
        this.notificationTopicArn = value
    }

    /**
     * @param value Number of cache clusters (primary and replicas) this replication group will have. If Multi-AZ is enabled, the value of this parameter must be at least 2. Updates will occur before other modifications. Conflicts with `num_node_groups`. Defaults to `1`.
     */
    @JvmName("pafofukajrhplbom")
    public suspend fun numCacheClusters(`value`: Output<Int>) {
        this.numCacheClusters = value
    }

    /**
     * @param value Number of node groups (shards) for this Redis replication group.
     * Changing this number will trigger a resizing operation before other settings modifications.
     */
    @JvmName("lkplmjcxhscktuoy")
    public suspend fun numNodeGroups(`value`: Output<Int>) {
        this.numNodeGroups = value
    }

    /**
     * @param value Name of the parameter group to associate with this replication group. If this argument is omitted, the default cache parameter group for the specified engine is used. To enable "cluster mode", i.e., data sharding, use a parameter group that has the parameter `cluster-enabled` set to true.
     */
    @JvmName("stydprvodggcxspf")
    public suspend fun parameterGroupName(`value`: Output<String>) {
        this.parameterGroupName = value
    }

    /**
     * @param value Port number on which each of the cache nodes will accept connections. For Memcache the default is 11211, and for Redis the default port is 6379.
     */
    @JvmName("aytjeapgfwotiowq")
    public suspend fun port(`value`: Output<Int>) {
        this.port = value
    }

    /**
     * @param value List of EC2 availability zones in which the replication group's cache clusters will be created. The order of the availability zones in the list is considered. The first item in the list will be the primary node. Ignored when updating.
     */
    @JvmName("uvtqhqkqpksxmwbk")
    public suspend fun preferredCacheClusterAzs(`value`: Output<List<String>>) {
        this.preferredCacheClusterAzs = value
    }

    @JvmName("xxansfgqvolreqqm")
    public suspend fun preferredCacheClusterAzs(vararg values: Output<String>) {
        this.preferredCacheClusterAzs = Output.all(values.asList())
    }

    /**
     * @param values List of EC2 availability zones in which the replication group's cache clusters will be created. The order of the availability zones in the list is considered. The first item in the list will be the primary node. Ignored when updating.
     */
    @JvmName("pyrrrdukruhkaqjh")
    public suspend fun preferredCacheClusterAzs(values: List<Output<String>>) {
        this.preferredCacheClusterAzs = Output.all(values)
    }

    /**
     * @param value Number of replica nodes in each node group.
     * Changing this number will trigger a resizing operation before other settings modifications.
     * Valid values are 0 to 5.
     */
    @JvmName("fybukpmaaautpbwq")
    public suspend fun replicasPerNodeGroup(`value`: Output<Int>) {
        this.replicasPerNodeGroup = value
    }

    /**
     * @param value Replication group identifier. This parameter is stored as a lowercase string.
     * The following arguments are optional:
     */
    @JvmName("rnfwvtlxemcasirx")
    public suspend fun replicationGroupId(`value`: Output<String>) {
        this.replicationGroupId = value
    }

    /**
     * @param value IDs of one or more Amazon VPC security groups associated with this replication group. Use this parameter only when you are creating a replication group in an Amazon Virtual Private Cloud.
     */
    @JvmName("vvlkuqfahtbgesne")
    public suspend fun securityGroupIds(`value`: Output<List<String>>) {
        this.securityGroupIds = value
    }

    @JvmName("hpbpwujeecrvjgeq")
    public suspend fun securityGroupIds(vararg values: Output<String>) {
        this.securityGroupIds = Output.all(values.asList())
    }

    /**
     * @param values IDs of one or more Amazon VPC security groups associated with this replication group. Use this parameter only when you are creating a replication group in an Amazon Virtual Private Cloud.
     */
    @JvmName("tinskseambysgltm")
    public suspend fun securityGroupIds(values: List<Output<String>>) {
        this.securityGroupIds = Output.all(values)
    }

    /**
     * @param value Names of one or more Amazon VPC security groups associated with this replication group. Use this parameter only when you are creating a replication group in an Amazon Virtual Private Cloud.
     */
    @JvmName("alnnchlrirepibgf")
    public suspend fun securityGroupNames(`value`: Output<List<String>>) {
        this.securityGroupNames = value
    }

    @JvmName("gcthlkxvvqbeuplm")
    public suspend fun securityGroupNames(vararg values: Output<String>) {
        this.securityGroupNames = Output.all(values.asList())
    }

    /**
     * @param values Names of one or more Amazon VPC security groups associated with this replication group. Use this parameter only when you are creating a replication group in an Amazon Virtual Private Cloud.
     */
    @JvmName("uebhbcinxxhnmudy")
    public suspend fun securityGroupNames(values: List<Output<String>>) {
        this.securityGroupNames = Output.all(values)
    }

    /**
     * @param value List of ARNs that identify Redis RDB snapshot files stored in Amazon S3. The names object names cannot contain any commas.
     */
    @JvmName("rkfrqsqsjqgcqfeq")
    public suspend fun snapshotArns(`value`: Output<List<String>>) {
        this.snapshotArns = value
    }

    @JvmName("unfhmamvcdmlmqjg")
    public suspend fun snapshotArns(vararg values: Output<String>) {
        this.snapshotArns = Output.all(values.asList())
    }

    /**
     * @param values List of ARNs that identify Redis RDB snapshot files stored in Amazon S3. The names object names cannot contain any commas.
     */
    @JvmName("wtbrlnqqlgdaefcc")
    public suspend fun snapshotArns(values: List<Output<String>>) {
        this.snapshotArns = Output.all(values)
    }

    /**
     * @param value Name of a snapshot from which to restore data into the new node group. Changing the `snapshot_name` forces a new resource.
     */
    @JvmName("wjnqxisxaksenspa")
    public suspend fun snapshotName(`value`: Output<String>) {
        this.snapshotName = value
    }

    /**
     * @param value Number of days for which ElastiCache will retain automatic cache cluster snapshots before deleting them. For example, if you set SnapshotRetentionLimit to 5, then a snapshot that was taken today will be retained for 5 days before being deleted. If the value of `snapshot_retention_limit` is set to zero (0), backups are turned off. Please note that setting a `snapshot_retention_limit` is not supported on cache.t1.micro cache nodes
     */
    @JvmName("rxbdyyeohrhhpmhi")
    public suspend fun snapshotRetentionLimit(`value`: Output<Int>) {
        this.snapshotRetentionLimit = value
    }

    /**
     * @param value Daily time range (in UTC) during which ElastiCache will begin taking a daily snapshot of your cache cluster. The minimum snapshot window is a 60 minute period. Example: `05:00-09:00`
     */
    @JvmName("dblmdfartfxblvdu")
    public suspend fun snapshotWindow(`value`: Output<String>) {
        this.snapshotWindow = value
    }

    /**
     * @param value Name of the cache subnet group to be used for the replication group.
     */
    @JvmName("jbttlsfwmtfnksxj")
    public suspend fun subnetGroupName(`value`: Output<String>) {
        this.subnetGroupName = value
    }

    /**
     * @param value Map of tags to assign to the resource. Adding tags to this resource will add or overwrite any existing tags on the clusters in the replication group and not to the group itself. If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
     */
    @JvmName("uvgfvbciaqldkdkx")
    public suspend fun tags(`value`: Output<Map<String, String>>) {
        this.tags = value
    }

    /**
     * @param value Whether to enable encryption in transit.
     */
    @JvmName("itbbhqevhmusajhe")
    public suspend fun transitEncryptionEnabled(`value`: Output<Boolean>) {
        this.transitEncryptionEnabled = value
    }

    /**
     * @param value User Group ID to associate with the replication group. Only a maximum of one (1) user group ID is valid. **NOTE:** This argument _is_ a set because the AWS specification allows for multiple IDs. However, in practice, AWS only allows a maximum size of one.
     */
    @JvmName("luioqrgxivhtatlo")
    public suspend fun userGroupIds(`value`: Output<List<String>>) {
        this.userGroupIds = value
    }

    @JvmName("iwjisurbrsfyflie")
    public suspend fun userGroupIds(vararg values: Output<String>) {
        this.userGroupIds = Output.all(values.asList())
    }

    /**
     * @param values User Group ID to associate with the replication group. Only a maximum of one (1) user group ID is valid. **NOTE:** This argument _is_ a set because the AWS specification allows for multiple IDs. However, in practice, AWS only allows a maximum size of one.
     */
    @JvmName("fupcsxuuyjwvuoll")
    public suspend fun userGroupIds(values: List<Output<String>>) {
        this.userGroupIds = Output.all(values)
    }

    /**
     * @param value Specifies whether any modifications are applied immediately, or during the next maintenance window. Default is `false`.
     */
    @JvmName("quwbcobtfxlowoaw")
    public suspend fun applyImmediately(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.applyImmediately = mapped
    }

    /**
     * @param value Whether to enable encryption at rest.
     */
    @JvmName("nadjsiwlrjntxktj")
    public suspend fun atRestEncryptionEnabled(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.atRestEncryptionEnabled = mapped
    }

    /**
     * @param value Password used to access a password protected server. Can be specified only if `transit_encryption_enabled = true`.
     */
    @JvmName("nqgdwjmwswybfgef")
    public suspend fun authToken(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.authToken = mapped
    }

    /**
     * @param value Strategy to use when updating the `auth_token`. Valid values are `SET`, `ROTATE`, and `DELETE`. Defaults to `ROTATE`.
     */
    @JvmName("gydqthdhutyawvuy")
    public suspend fun authTokenUpdateStrategy(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.authTokenUpdateStrategy = mapped
    }

    /**
     * @param value Specifies whether minor version engine upgrades will be applied automatically to the underlying Cache Cluster instances during the maintenance window.
     * Only supported for engine type `"redis"` and if the engine version is 6 or higher.
     * Defaults to `true`.
     */
    @JvmName("myjemyemhesmhcuq")
    public suspend fun autoMinorVersionUpgrade(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.autoMinorVersionUpgrade = mapped
    }

    /**
     * @param value Specifies whether a read-only replica will be automatically promoted to read/write primary if the existing primary fails. If enabled, `num_cache_clusters` must be greater than 1. Must be enabled for Redis (cluster mode enabled) replication groups. Defaults to `false`.
     */
    @JvmName("fdwiqosgqetvhlqp")
    public suspend fun automaticFailoverEnabled(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.automaticFailoverEnabled = mapped
    }

    /**
     * @param value Enables data tiering. Data tiering is only supported for replication groups using the r6gd node type. This parameter must be set to `true` when using r6gd nodes.
     */
    @JvmName("qumrvtvwgdptlask")
    public suspend fun dataTieringEnabled(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.dataTieringEnabled = mapped
    }

    /**
     * @param value User-created description for the replication group. Must not be empty.
     */
    @JvmName("fkyumjxyklrtebxs")
    public suspend fun description(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.description = mapped
    }

    /**
     * @param value Name of the cache engine to be used for the clusters in this replication group. The only valid value is `redis`.
     */
    @JvmName("udymcrhpvpbnyeue")
    public suspend fun engine(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.engine = mapped
    }

    /**
     * @param value Version number of the cache engine to be used for the cache clusters in this replication group.
     * If the version is 7 or higher, the major and minor version should be set, e.g., `7.2`.
     * If the version is 6, the major and minor version can be set, e.g., `6.2`,
     * or the minor version can be unspecified which will use the latest version at creation time, e.g., `6.x`.
     * Otherwise, specify the full version desired, e.g., `5.0.6`.
     * The actual engine version used is returned in the attribute `engine_version_actual`, see Attribute Reference below.
     */
    @JvmName("mdmgqqwvujrnchyq")
    public suspend fun engineVersion(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.engineVersion = mapped
    }

    /**
     * @param value The name of your final node group (shard) snapshot. ElastiCache creates the snapshot from the primary node in the cluster. If omitted, no final snapshot will be made.
     */
    @JvmName("dpnxfavyleetdphr")
    public suspend fun finalSnapshotIdentifier(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.finalSnapshotIdentifier = mapped
    }

    /**
     * @param value The ID of the global replication group to which this replication group should belong. If this parameter is specified, the replication group is added to the specified global replication group as a secondary replication group; otherwise, the replication group is not part of any global replication group. If `global_replication_group_id` is set, the `num_node_groups` parameter cannot be set.
     */
    @JvmName("fvxxongtlbgmqhgs")
    public suspend fun globalReplicationGroupId(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.globalReplicationGroupId = mapped
    }

    /**
     * @param value The IP version to advertise in the discovery protocol. Valid values are `ipv4` or `ipv6`.
     */
    @JvmName("xdidobgsmfwqcsvi")
    public suspend fun ipDiscovery(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.ipDiscovery = mapped
    }

    /**
     * @param value The ARN of the key that you wish to use if encrypting at rest. If not supplied, uses service managed encryption. Can be specified only if `at_rest_encryption_enabled = true`.
     */
    @JvmName("tkfyxrgnrbrbfrlw")
    public suspend fun kmsKeyId(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.kmsKeyId = mapped
    }

    /**
     * @param value Specifies the destination and format of Redis [SLOWLOG](https://redis&#46;io/commands/slowlog) or Redis [Engine Log](https://docs&#46;aws&#46;amazon&#46;com/AmazonElastiCache/latest/red-ug/Log_Delivery&#46;html#Log_contents-engine-log)&#46; See the documentation on [Amazon ElastiCache](https://docs.aws.amazon.com/AmazonElastiCache/latest/red-ug/Log_Delivery.html#Log_contents-engine-log). See Log Delivery Configuration below for more details.
     */
    @JvmName("wyfhfbyxjddhemmf")
    public suspend
    fun logDeliveryConfigurations(`value`: List<ReplicationGroupLogDeliveryConfigurationArgs>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.logDeliveryConfigurations = mapped
    }

    /**
     * @param argument Specifies the destination and format of Redis [SLOWLOG](https://redis&#46;io/commands/slowlog) or Redis [Engine Log](https://docs&#46;aws&#46;amazon&#46;com/AmazonElastiCache/latest/red-ug/Log_Delivery&#46;html#Log_contents-engine-log)&#46; See the documentation on [Amazon ElastiCache](https://docs.aws.amazon.com/AmazonElastiCache/latest/red-ug/Log_Delivery.html#Log_contents-engine-log). See Log Delivery Configuration below for more details.
     */
    @JvmName("lfbeyexqugjpdmxj")
    public suspend
    fun logDeliveryConfigurations(argument: List<suspend ReplicationGroupLogDeliveryConfigurationArgsBuilder.() -> Unit>) {
        val toBeMapped = argument.toList().map {
            ReplicationGroupLogDeliveryConfigurationArgsBuilder().applySuspend { it() }.build()
        }
        val mapped = of(toBeMapped)
        this.logDeliveryConfigurations = mapped
    }

    /**
     * @param argument Specifies the destination and format of Redis [SLOWLOG](https://redis&#46;io/commands/slowlog) or Redis [Engine Log](https://docs&#46;aws&#46;amazon&#46;com/AmazonElastiCache/latest/red-ug/Log_Delivery&#46;html#Log_contents-engine-log)&#46; See the documentation on [Amazon ElastiCache](https://docs.aws.amazon.com/AmazonElastiCache/latest/red-ug/Log_Delivery.html#Log_contents-engine-log). See Log Delivery Configuration below for more details.
     */
    @JvmName("odpokaacgocpmfur")
    public suspend fun logDeliveryConfigurations(
        vararg
        argument: suspend ReplicationGroupLogDeliveryConfigurationArgsBuilder.() -> Unit,
    ) {
        val toBeMapped = argument.toList().map {
            ReplicationGroupLogDeliveryConfigurationArgsBuilder().applySuspend { it() }.build()
        }
        val mapped = of(toBeMapped)
        this.logDeliveryConfigurations = mapped
    }

    /**
     * @param argument Specifies the destination and format of Redis [SLOWLOG](https://redis&#46;io/commands/slowlog) or Redis [Engine Log](https://docs&#46;aws&#46;amazon&#46;com/AmazonElastiCache/latest/red-ug/Log_Delivery&#46;html#Log_contents-engine-log)&#46; See the documentation on [Amazon ElastiCache](https://docs.aws.amazon.com/AmazonElastiCache/latest/red-ug/Log_Delivery.html#Log_contents-engine-log). See Log Delivery Configuration below for more details.
     */
    @JvmName("tiiitasuvbnyhtbc")
    public suspend
    fun logDeliveryConfigurations(argument: suspend ReplicationGroupLogDeliveryConfigurationArgsBuilder.() -> Unit) {
        val toBeMapped = listOf(
            ReplicationGroupLogDeliveryConfigurationArgsBuilder().applySuspend {
                argument()
            }.build(),
        )
        val mapped = of(toBeMapped)
        this.logDeliveryConfigurations = mapped
    }

    /**
     * @param values Specifies the destination and format of Redis [SLOWLOG](https://redis&#46;io/commands/slowlog) or Redis [Engine Log](https://docs&#46;aws&#46;amazon&#46;com/AmazonElastiCache/latest/red-ug/Log_Delivery&#46;html#Log_contents-engine-log)&#46; See the documentation on [Amazon ElastiCache](https://docs.aws.amazon.com/AmazonElastiCache/latest/red-ug/Log_Delivery.html#Log_contents-engine-log). See Log Delivery Configuration below for more details.
     */
    @JvmName("nqjwlfwkcxfxlqgg")
    public suspend fun logDeliveryConfigurations(
        vararg
        values: ReplicationGroupLogDeliveryConfigurationArgs,
    ) {
        val toBeMapped = values.toList()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.logDeliveryConfigurations = mapped
    }

    /**
     * @param value Specifies the weekly time range for when maintenance on the cache cluster is performed. The format is `ddd:hh24:mi-ddd:hh24:mi` (24H Clock UTC). The minimum maintenance window is a 60 minute period. Example: `sun:05:00-sun:09:00`
     */
    @JvmName("ltjkbnesdgdmejfk")
    public suspend fun maintenanceWindow(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.maintenanceWindow = mapped
    }

    /**
     * @param value Specifies whether to enable Multi-AZ Support for the replication group. If `true`, `automatic_failover_enabled` must also be enabled. Defaults to `false`.
     */
    @JvmName("ydiyafqcujhbadyu")
    public suspend fun multiAzEnabled(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.multiAzEnabled = mapped
    }

    /**
     * @param value The IP versions for cache cluster connections. Valid values are `ipv4`, `ipv6` or `dual_stack`.
     */
    @JvmName("vakoyuxvbbynepjv")
    public suspend fun networkType(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.networkType = mapped
    }

    /**
     * @param value Instance class to be used. See AWS documentation for information on [supported node types](https://docs&#46;aws&#46;amazon&#46;com/AmazonElastiCache/latest/red-ug/CacheNodes&#46;SupportedTypes&#46;html) and [guidance on selecting node types](https://docs.aws.amazon.com/AmazonElastiCache/latest/red-ug/nodes-select-size.html). Required unless `global_replication_group_id` is set. Cannot be set if `global_replication_group_id` is set.
     */
    @JvmName("xunijsdvvksosadd")
    public suspend fun nodeType(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.nodeType = mapped
    }

    /**
     * @param value ARN of an SNS topic to send ElastiCache notifications to. Example: `arn:aws:sns:us-east-1:012345678999:my_sns_topic`
     */
    @JvmName("uflptseotbpfyshf")
    public suspend fun notificationTopicArn(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.notificationTopicArn = mapped
    }

    /**
     * @param value Number of cache clusters (primary and replicas) this replication group will have. If Multi-AZ is enabled, the value of this parameter must be at least 2. Updates will occur before other modifications. Conflicts with `num_node_groups`. Defaults to `1`.
     */
    @JvmName("twvoingardvfcdti")
    public suspend fun numCacheClusters(`value`: Int?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.numCacheClusters = mapped
    }

    /**
     * @param value Number of node groups (shards) for this Redis replication group.
     * Changing this number will trigger a resizing operation before other settings modifications.
     */
    @JvmName("xvxnubtviirjmynb")
    public suspend fun numNodeGroups(`value`: Int?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.numNodeGroups = mapped
    }

    /**
     * @param value Name of the parameter group to associate with this replication group. If this argument is omitted, the default cache parameter group for the specified engine is used. To enable "cluster mode", i.e., data sharding, use a parameter group that has the parameter `cluster-enabled` set to true.
     */
    @JvmName("iypwnpmcxougytno")
    public suspend fun parameterGroupName(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.parameterGroupName = mapped
    }

    /**
     * @param value Port number on which each of the cache nodes will accept connections. For Memcache the default is 11211, and for Redis the default port is 6379.
     */
    @JvmName("nrvfxlhbmprajwsr")
    public suspend fun port(`value`: Int?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.port = mapped
    }

    /**
     * @param value List of EC2 availability zones in which the replication group's cache clusters will be created. The order of the availability zones in the list is considered. The first item in the list will be the primary node. Ignored when updating.
     */
    @JvmName("dmvtssnfbbxeffng")
    public suspend fun preferredCacheClusterAzs(`value`: List<String>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.preferredCacheClusterAzs = mapped
    }

    /**
     * @param values List of EC2 availability zones in which the replication group's cache clusters will be created. The order of the availability zones in the list is considered. The first item in the list will be the primary node. Ignored when updating.
     */
    @JvmName("fnojkwtsqwllrudc")
    public suspend fun preferredCacheClusterAzs(vararg values: String) {
        val toBeMapped = values.toList()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.preferredCacheClusterAzs = mapped
    }

    /**
     * @param value Number of replica nodes in each node group.
     * Changing this number will trigger a resizing operation before other settings modifications.
     * Valid values are 0 to 5.
     */
    @JvmName("knsxsustlrvpkkgi")
    public suspend fun replicasPerNodeGroup(`value`: Int?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.replicasPerNodeGroup = mapped
    }

    /**
     * @param value Replication group identifier. This parameter is stored as a lowercase string.
     * The following arguments are optional:
     */
    @JvmName("igmmiroevafhqsca")
    public suspend fun replicationGroupId(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.replicationGroupId = mapped
    }

    /**
     * @param value IDs of one or more Amazon VPC security groups associated with this replication group. Use this parameter only when you are creating a replication group in an Amazon Virtual Private Cloud.
     */
    @JvmName("fkybovlhlkixkaxk")
    public suspend fun securityGroupIds(`value`: List<String>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.securityGroupIds = mapped
    }

    /**
     * @param values IDs of one or more Amazon VPC security groups associated with this replication group. Use this parameter only when you are creating a replication group in an Amazon Virtual Private Cloud.
     */
    @JvmName("bsvubimblibdonxl")
    public suspend fun securityGroupIds(vararg values: String) {
        val toBeMapped = values.toList()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.securityGroupIds = mapped
    }

    /**
     * @param value Names of one or more Amazon VPC security groups associated with this replication group. Use this parameter only when you are creating a replication group in an Amazon Virtual Private Cloud.
     */
    @JvmName("yihbajirntkkjxeg")
    public suspend fun securityGroupNames(`value`: List<String>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.securityGroupNames = mapped
    }

    /**
     * @param values Names of one or more Amazon VPC security groups associated with this replication group. Use this parameter only when you are creating a replication group in an Amazon Virtual Private Cloud.
     */
    @JvmName("tnuvhkmpogjtanrq")
    public suspend fun securityGroupNames(vararg values: String) {
        val toBeMapped = values.toList()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.securityGroupNames = mapped
    }

    /**
     * @param value List of ARNs that identify Redis RDB snapshot files stored in Amazon S3. The names object names cannot contain any commas.
     */
    @JvmName("ddhmqlcqksymxsoy")
    public suspend fun snapshotArns(`value`: List<String>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.snapshotArns = mapped
    }

    /**
     * @param values List of ARNs that identify Redis RDB snapshot files stored in Amazon S3. The names object names cannot contain any commas.
     */
    @JvmName("uajmnyrpaywrjpcg")
    public suspend fun snapshotArns(vararg values: String) {
        val toBeMapped = values.toList()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.snapshotArns = mapped
    }

    /**
     * @param value Name of a snapshot from which to restore data into the new node group. Changing the `snapshot_name` forces a new resource.
     */
    @JvmName("ynonrgbyjfrshalp")
    public suspend fun snapshotName(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.snapshotName = mapped
    }

    /**
     * @param value Number of days for which ElastiCache will retain automatic cache cluster snapshots before deleting them. For example, if you set SnapshotRetentionLimit to 5, then a snapshot that was taken today will be retained for 5 days before being deleted. If the value of `snapshot_retention_limit` is set to zero (0), backups are turned off. Please note that setting a `snapshot_retention_limit` is not supported on cache.t1.micro cache nodes
     */
    @JvmName("nngjayanstdaklgu")
    public suspend fun snapshotRetentionLimit(`value`: Int?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.snapshotRetentionLimit = mapped
    }

    /**
     * @param value Daily time range (in UTC) during which ElastiCache will begin taking a daily snapshot of your cache cluster. The minimum snapshot window is a 60 minute period. Example: `05:00-09:00`
     */
    @JvmName("fmosatevgqdgxamu")
    public suspend fun snapshotWindow(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.snapshotWindow = mapped
    }

    /**
     * @param value Name of the cache subnet group to be used for the replication group.
     */
    @JvmName("pbgmtdcryfpxxopm")
    public suspend fun subnetGroupName(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.subnetGroupName = mapped
    }

    /**
     * @param value Map of tags to assign to the resource. Adding tags to this resource will add or overwrite any existing tags on the clusters in the replication group and not to the group itself. If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
     */
    @JvmName("hybfgomgvbxsoomt")
    public suspend fun tags(`value`: Map<String, String>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.tags = mapped
    }

    /**
     * @param values Map of tags to assign to the resource. Adding tags to this resource will add or overwrite any existing tags on the clusters in the replication group and not to the group itself. If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
     */
    @JvmName("ormbeajkipgevnqa")
    public fun tags(vararg values: Pair<String, String>) {
        val toBeMapped = values.toMap()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.tags = mapped
    }

    /**
     * @param value Whether to enable encryption in transit.
     */
    @JvmName("epkjbsfjymlitkjm")
    public suspend fun transitEncryptionEnabled(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.transitEncryptionEnabled = mapped
    }

    /**
     * @param value User Group ID to associate with the replication group. Only a maximum of one (1) user group ID is valid. **NOTE:** This argument _is_ a set because the AWS specification allows for multiple IDs. However, in practice, AWS only allows a maximum size of one.
     */
    @JvmName("uudkeoirgxxysddg")
    public suspend fun userGroupIds(`value`: List<String>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.userGroupIds = mapped
    }

    /**
     * @param values User Group ID to associate with the replication group. Only a maximum of one (1) user group ID is valid. **NOTE:** This argument _is_ a set because the AWS specification allows for multiple IDs. However, in practice, AWS only allows a maximum size of one.
     */
    @JvmName("gnafuorfmlwsephe")
    public suspend fun userGroupIds(vararg values: String) {
        val toBeMapped = values.toList()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.userGroupIds = mapped
    }

    internal fun build(): ReplicationGroupArgs = ReplicationGroupArgs(
        applyImmediately = applyImmediately,
        atRestEncryptionEnabled = atRestEncryptionEnabled,
        authToken = authToken,
        authTokenUpdateStrategy = authTokenUpdateStrategy,
        autoMinorVersionUpgrade = autoMinorVersionUpgrade,
        automaticFailoverEnabled = automaticFailoverEnabled,
        dataTieringEnabled = dataTieringEnabled,
        description = description,
        engine = engine,
        engineVersion = engineVersion,
        finalSnapshotIdentifier = finalSnapshotIdentifier,
        globalReplicationGroupId = globalReplicationGroupId,
        ipDiscovery = ipDiscovery,
        kmsKeyId = kmsKeyId,
        logDeliveryConfigurations = logDeliveryConfigurations,
        maintenanceWindow = maintenanceWindow,
        multiAzEnabled = multiAzEnabled,
        networkType = networkType,
        nodeType = nodeType,
        notificationTopicArn = notificationTopicArn,
        numCacheClusters = numCacheClusters,
        numNodeGroups = numNodeGroups,
        parameterGroupName = parameterGroupName,
        port = port,
        preferredCacheClusterAzs = preferredCacheClusterAzs,
        replicasPerNodeGroup = replicasPerNodeGroup,
        replicationGroupId = replicationGroupId,
        securityGroupIds = securityGroupIds,
        securityGroupNames = securityGroupNames,
        snapshotArns = snapshotArns,
        snapshotName = snapshotName,
        snapshotRetentionLimit = snapshotRetentionLimit,
        snapshotWindow = snapshotWindow,
        subnetGroupName = subnetGroupName,
        tags = tags,
        transitEncryptionEnabled = transitEncryptionEnabled,
        userGroupIds = userGroupIds,
    )
}
