@file:Suppress("NAME_SHADOWING", "DEPRECATION")

package com.pulumi.aws.opensearch.kotlin

import com.pulumi.aws.opensearch.DomainArgs.builder
import com.pulumi.aws.opensearch.kotlin.inputs.DomainAdvancedSecurityOptionsArgs
import com.pulumi.aws.opensearch.kotlin.inputs.DomainAdvancedSecurityOptionsArgsBuilder
import com.pulumi.aws.opensearch.kotlin.inputs.DomainAutoTuneOptionsArgs
import com.pulumi.aws.opensearch.kotlin.inputs.DomainAutoTuneOptionsArgsBuilder
import com.pulumi.aws.opensearch.kotlin.inputs.DomainClusterConfigArgs
import com.pulumi.aws.opensearch.kotlin.inputs.DomainClusterConfigArgsBuilder
import com.pulumi.aws.opensearch.kotlin.inputs.DomainCognitoOptionsArgs
import com.pulumi.aws.opensearch.kotlin.inputs.DomainCognitoOptionsArgsBuilder
import com.pulumi.aws.opensearch.kotlin.inputs.DomainDomainEndpointOptionsArgs
import com.pulumi.aws.opensearch.kotlin.inputs.DomainDomainEndpointOptionsArgsBuilder
import com.pulumi.aws.opensearch.kotlin.inputs.DomainEbsOptionsArgs
import com.pulumi.aws.opensearch.kotlin.inputs.DomainEbsOptionsArgsBuilder
import com.pulumi.aws.opensearch.kotlin.inputs.DomainEncryptAtRestArgs
import com.pulumi.aws.opensearch.kotlin.inputs.DomainEncryptAtRestArgsBuilder
import com.pulumi.aws.opensearch.kotlin.inputs.DomainLogPublishingOptionArgs
import com.pulumi.aws.opensearch.kotlin.inputs.DomainLogPublishingOptionArgsBuilder
import com.pulumi.aws.opensearch.kotlin.inputs.DomainNodeToNodeEncryptionArgs
import com.pulumi.aws.opensearch.kotlin.inputs.DomainNodeToNodeEncryptionArgsBuilder
import com.pulumi.aws.opensearch.kotlin.inputs.DomainOffPeakWindowOptionsArgs
import com.pulumi.aws.opensearch.kotlin.inputs.DomainOffPeakWindowOptionsArgsBuilder
import com.pulumi.aws.opensearch.kotlin.inputs.DomainSnapshotOptionsArgs
import com.pulumi.aws.opensearch.kotlin.inputs.DomainSnapshotOptionsArgsBuilder
import com.pulumi.aws.opensearch.kotlin.inputs.DomainSoftwareUpdateOptionsArgs
import com.pulumi.aws.opensearch.kotlin.inputs.DomainSoftwareUpdateOptionsArgsBuilder
import com.pulumi.aws.opensearch.kotlin.inputs.DomainVpcOptionsArgs
import com.pulumi.aws.opensearch.kotlin.inputs.DomainVpcOptionsArgsBuilder
import com.pulumi.core.Output
import com.pulumi.core.Output.of
import com.pulumi.kotlin.ConvertibleToJava
import com.pulumi.kotlin.PulumiTagMarker
import com.pulumi.kotlin.applySuspend
import kotlin.Pair
import kotlin.String
import kotlin.Suppress
import kotlin.Unit
import kotlin.collections.List
import kotlin.collections.Map
import kotlin.jvm.JvmName

/**
 * Manages an Amazon OpenSearch Domain.
 * ## Elasticsearch vs. OpenSearch
 * Amazon OpenSearch Service is the successor to Amazon Elasticsearch Service and supports OpenSearch and legacy Elasticsearch OSS (up to 7.10, the final open source version of the software).
 * OpenSearch Domain configurations are similar in many ways to Elasticsearch Domain configurations. However, there are important differences including these:
 * * OpenSearch has `engine_version` while Elasticsearch has `elasticsearch_version`
 * * Versions are specified differently - _e.g._, `Elasticsearch_7.10` with OpenSearch vs. `7.10` for Elasticsearch.
 * * `instance_type` argument values end in `search` for OpenSearch vs. `elasticsearch` for Elasticsearch (_e.g._, `t2.micro.search` vs. `t2.micro.elasticsearch`).
 * * The AWS-managed service-linked role for OpenSearch is called `AWSServiceRoleForAmazonOpenSearchService` instead of `AWSServiceRoleForAmazonElasticsearchService` for Elasticsearch.
 * There are also some potentially unexpected similarities in configurations:
 * * ARNs for both are prefaced with `arn:aws:es:`.
 * * Both OpenSearch and Elasticsearch use assume role policies that refer to the `Principal` `Service` as `es.amazonaws.com`.
 * * IAM policy actions, such as those you will find in `access_policies`, are prefaced with `es:` for both.
 * ## Example Usage
 * ### Basic Usage
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.opensearch.Domain;
 * import com.pulumi.aws.opensearch.DomainArgs;
 * import com.pulumi.aws.opensearch.inputs.DomainClusterConfigArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var example = new Domain("example", DomainArgs.builder()
 *             .clusterConfig(DomainClusterConfigArgs.builder()
 *                 .instanceType("r4.large.search")
 *                 .build())
 *             .engineVersion("Elasticsearch_7.10")
 *             .tags(Map.of("Domain", "TestDomain"))
 *             .build());
 *     }
 * }
 * ```
 * ### Access Policy
 * > See also: `aws.opensearch.DomainPolicy` resource
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.AwsFunctions;
 * import com.pulumi.aws.inputs.GetRegionArgs;
 * import com.pulumi.aws.inputs.GetCallerIdentityArgs;
 * import com.pulumi.aws.iam.IamFunctions;
 * import com.pulumi.aws.iam.inputs.GetPolicyDocumentArgs;
 * import com.pulumi.aws.opensearch.Domain;
 * import com.pulumi.aws.opensearch.DomainArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         final var config = ctx.config();
 *         final var domain = config.get("domain").orElse("tf-test");
 *         final var currentRegion = AwsFunctions.getRegion();
 *         final var currentCallerIdentity = AwsFunctions.getCallerIdentity();
 *         final var examplePolicyDocument = IamFunctions.getPolicyDocument(GetPolicyDocumentArgs.builder()
 *             .statements(GetPolicyDocumentStatementArgs.builder()
 *                 .effect("Allow")
 *                 .principals(GetPolicyDocumentStatementPrincipalArgs.builder()
 *                     .type("*")
 *                     .identifiers("*")
 *                     .build())
 *                 .actions("es:*")
 *                 .resources(String.format("arn:aws:es:%s:%s:domain/%s/*", currentRegion.applyValue(getRegionResult -> getRegionResult.name()),currentCallerIdentity.applyValue(getCallerIdentityResult -> getCallerIdentityResult.accountId()),domain))
 *                 .conditions(GetPolicyDocumentStatementConditionArgs.builder()
 *                     .test("IpAddress")
 *                     .variable("aws:SourceIp")
 *                     .values("66.193.100.22/32")
 *                     .build())
 *                 .build())
 *             .build());
 *         var exampleDomain = new Domain("exampleDomain", DomainArgs.builder()
 *             .accessPolicies(examplePolicyDocument.applyValue(getPolicyDocumentResult -> getPolicyDocumentResult.json()))
 *             .build());
 *     }
 * }
 * ```
 * ### Log publishing to CloudWatch Logs
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.cloudwatch.LogGroup;
 * import com.pulumi.aws.iam.IamFunctions;
 * import com.pulumi.aws.iam.inputs.GetPolicyDocumentArgs;
 * import com.pulumi.aws.cloudwatch.LogResourcePolicy;
 * import com.pulumi.aws.cloudwatch.LogResourcePolicyArgs;
 * import com.pulumi.aws.opensearch.Domain;
 * import com.pulumi.aws.opensearch.DomainArgs;
 * import com.pulumi.aws.opensearch.inputs.DomainLogPublishingOptionArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var exampleLogGroup = new LogGroup("exampleLogGroup");
 *         final var examplePolicyDocument = IamFunctions.getPolicyDocument(GetPolicyDocumentArgs.builder()
 *             .statements(GetPolicyDocumentStatementArgs.builder()
 *                 .effect("Allow")
 *                 .principals(GetPolicyDocumentStatementPrincipalArgs.builder()
 *                     .type("Service")
 *                     .identifiers("es.amazonaws.com")
 *                     .build())
 *                 .actions(
 *                     "logs:PutLogEvents",
 *                     "logs:PutLogEventsBatch",
 *                     "logs:CreateLogStream")
 *                 .resources("arn:aws:logs:*")
 *                 .build())
 *             .build());
 *         var exampleLogResourcePolicy = new LogResourcePolicy("exampleLogResourcePolicy", LogResourcePolicyArgs.builder()
 *             .policyName("example")
 *             .policyDocument(examplePolicyDocument.applyValue(getPolicyDocumentResult -> getPolicyDocumentResult.json()))
 *             .build());
 *         var exampleDomain = new Domain("exampleDomain", DomainArgs.builder()
 *             .logPublishingOptions(DomainLogPublishingOptionArgs.builder()
 *                 .cloudwatchLogGroupArn(exampleLogGroup.arn())
 *                 .logType("INDEX_SLOW_LOGS")
 *                 .build())
 *             .build());
 *     }
 * }
 * ```
 * ### VPC based OpenSearch
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.ec2.Ec2Functions;
 * import com.pulumi.aws.ec2.inputs.GetVpcArgs;
 * import com.pulumi.aws.ec2.inputs.GetSubnetsArgs;
 * import com.pulumi.aws.AwsFunctions;
 * import com.pulumi.aws.inputs.GetRegionArgs;
 * import com.pulumi.aws.inputs.GetCallerIdentityArgs;
 * import com.pulumi.aws.ec2.SecurityGroup;
 * import com.pulumi.aws.ec2.SecurityGroupArgs;
 * import com.pulumi.aws.ec2.inputs.SecurityGroupIngressArgs;
 * import com.pulumi.aws.iam.ServiceLinkedRole;
 * import com.pulumi.aws.iam.ServiceLinkedRoleArgs;
 * import com.pulumi.aws.iam.IamFunctions;
 * import com.pulumi.aws.iam.inputs.GetPolicyDocumentArgs;
 * import com.pulumi.aws.opensearch.Domain;
 * import com.pulumi.aws.opensearch.DomainArgs;
 * import com.pulumi.aws.opensearch.inputs.DomainClusterConfigArgs;
 * import com.pulumi.aws.opensearch.inputs.DomainVpcOptionsArgs;
 * import com.pulumi.resources.CustomResourceOptions;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         final var config = ctx.config();
 *         final var vpc = config.get("vpc");
 *         final var domain = config.get("domain").orElse("tf-test");
 *         final var exampleVpc = Ec2Functions.getVpc(GetVpcArgs.builder()
 *             .tags(Map.of("Name", vpc))
 *             .build());
 *         final var exampleSubnets = Ec2Functions.getSubnets(GetSubnetsArgs.builder()
 *             .filters(GetSubnetsFilterArgs.builder()
 *                 .name("vpc-id")
 *                 .values(exampleVpc.applyValue(getVpcResult -> getVpcResult.id()))
 *                 .build())
 *             .tags(Map.of("Tier", "private"))
 *             .build());
 *         final var currentRegion = AwsFunctions.getRegion();
 *         final var currentCallerIdentity = AwsFunctions.getCallerIdentity();
 *         var exampleSecurityGroup = new SecurityGroup("exampleSecurityGroup", SecurityGroupArgs.builder()
 *             .description("Managed by Pulumi")
 *             .vpcId(exampleVpc.applyValue(getVpcResult -> getVpcResult.id()))
 *             .ingress(SecurityGroupIngressArgs.builder()
 *                 .fromPort(443)
 *                 .toPort(443)
 *                 .protocol("tcp")
 *                 .cidrBlocks(exampleVpc.applyValue(getVpcResult -> getVpcResult.cidrBlock()))
 *                 .build())
 *             .build());
 *         var exampleServiceLinkedRole = new ServiceLinkedRole("exampleServiceLinkedRole", ServiceLinkedRoleArgs.builder()
 *             .awsServiceName("opensearchservice.amazonaws.com")
 *             .build());
 *         final var examplePolicyDocument = IamFunctions.getPolicyDocument(GetPolicyDocumentArgs.builder()
 *             .statements(GetPolicyDocumentStatementArgs.builder()
 *                 .effect("Allow")
 *                 .principals(GetPolicyDocumentStatementPrincipalArgs.builder()
 *                     .type("*")
 *                     .identifiers("*")
 *                     .build())
 *                 .actions("es:*")
 *                 .resources(String.format("arn:aws:es:%s:%s:domain/%s/*", currentRegion.applyValue(getRegionResult -> getRegionResult.name()),currentCallerIdentity.applyValue(getCallerIdentityResult -> getCallerIdentityResult.accountId()),domain))
 *                 .build())
 *             .build());
 *         var exampleDomain = new Domain("exampleDomain", DomainArgs.builder()
 *             .engineVersion("OpenSearch_1.0")
 *             .clusterConfig(DomainClusterConfigArgs.builder()
 *                 .instanceType("m4.large.search")
 *                 .zoneAwarenessEnabled(true)
 *                 .build())
 *             .vpcOptions(DomainVpcOptionsArgs.builder()
 *                 .subnetIds(
 *                     exampleSubnets.applyValue(getSubnetsResult -> getSubnetsResult.ids()[0]),
 *                     exampleSubnets.applyValue(getSubnetsResult -> getSubnetsResult.ids()[1]))
 *                 .securityGroupIds(exampleSecurityGroup.id())
 *                 .build())
 *             .advancedOptions(Map.of("rest.action.multi.allow_explicit_index", "true"))
 *             .accessPolicies(examplePolicyDocument.applyValue(getPolicyDocumentResult -> getPolicyDocumentResult.json()))
 *             .tags(Map.of("Domain", "TestDomain"))
 *             .build(), CustomResourceOptions.builder()
 *                 .dependsOn(exampleServiceLinkedRole)
 *                 .build());
 *     }
 * }
 * ```
 * ### First apply
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.opensearch.Domain;
 * import com.pulumi.aws.opensearch.DomainArgs;
 * import com.pulumi.aws.opensearch.inputs.DomainAdvancedSecurityOptionsArgs;
 * import com.pulumi.aws.opensearch.inputs.DomainAdvancedSecurityOptionsMasterUserOptionsArgs;
 * import com.pulumi.aws.opensearch.inputs.DomainClusterConfigArgs;
 * import com.pulumi.aws.opensearch.inputs.DomainDomainEndpointOptionsArgs;
 * import com.pulumi.aws.opensearch.inputs.DomainEbsOptionsArgs;
 * import com.pulumi.aws.opensearch.inputs.DomainEncryptAtRestArgs;
 * import com.pulumi.aws.opensearch.inputs.DomainNodeToNodeEncryptionArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var example = new Domain("example", DomainArgs.builder()
 *             .advancedSecurityOptions(DomainAdvancedSecurityOptionsArgs.builder()
 *                 .anonymousAuthEnabled(true)
 *                 .enabled(false)
 *                 .internalUserDatabaseEnabled(true)
 *                 .masterUserOptions(DomainAdvancedSecurityOptionsMasterUserOptionsArgs.builder()
 *                     .masterUserName("example")
 *                     .masterUserPassword("Barbarbarbar1!")
 *                     .build())
 *                 .build())
 *             .clusterConfig(DomainClusterConfigArgs.builder()
 *                 .instanceType("r5.large.search")
 *                 .build())
 *             .domainEndpointOptions(DomainDomainEndpointOptionsArgs.builder()
 *                 .enforceHttps(true)
 *                 .tlsSecurityPolicy("Policy-Min-TLS-1-2-2019-07")
 *                 .build())
 *             .ebsOptions(DomainEbsOptionsArgs.builder()
 *                 .ebsEnabled(true)
 *                 .volumeSize(10)
 *                 .build())
 *             .encryptAtRest(DomainEncryptAtRestArgs.builder()
 *                 .enabled(true)
 *                 .build())
 *             .engineVersion("Elasticsearch_7.1")
 *             .nodeToNodeEncryption(DomainNodeToNodeEncryptionArgs.builder()
 *                 .enabled(true)
 *                 .build())
 *             .build());
 *     }
 * }
 * ```
 * ### Second apply
 * Notice that the only change is `advanced_security_options.0.enabled` is now set to `true`.
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.opensearch.Domain;
 * import com.pulumi.aws.opensearch.DomainArgs;
 * import com.pulumi.aws.opensearch.inputs.DomainAdvancedSecurityOptionsArgs;
 * import com.pulumi.aws.opensearch.inputs.DomainAdvancedSecurityOptionsMasterUserOptionsArgs;
 * import com.pulumi.aws.opensearch.inputs.DomainClusterConfigArgs;
 * import com.pulumi.aws.opensearch.inputs.DomainDomainEndpointOptionsArgs;
 * import com.pulumi.aws.opensearch.inputs.DomainEbsOptionsArgs;
 * import com.pulumi.aws.opensearch.inputs.DomainEncryptAtRestArgs;
 * import com.pulumi.aws.opensearch.inputs.DomainNodeToNodeEncryptionArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var example = new Domain("example", DomainArgs.builder()
 *             .advancedSecurityOptions(DomainAdvancedSecurityOptionsArgs.builder()
 *                 .anonymousAuthEnabled(true)
 *                 .enabled(true)
 *                 .internalUserDatabaseEnabled(true)
 *                 .masterUserOptions(DomainAdvancedSecurityOptionsMasterUserOptionsArgs.builder()
 *                     .masterUserName("example")
 *                     .masterUserPassword("Barbarbarbar1!")
 *                     .build())
 *                 .build())
 *             .clusterConfig(DomainClusterConfigArgs.builder()
 *                 .instanceType("r5.large.search")
 *                 .build())
 *             .domainEndpointOptions(DomainDomainEndpointOptionsArgs.builder()
 *                 .enforceHttps(true)
 *                 .tlsSecurityPolicy("Policy-Min-TLS-1-2-2019-07")
 *                 .build())
 *             .ebsOptions(DomainEbsOptionsArgs.builder()
 *                 .ebsEnabled(true)
 *                 .volumeSize(10)
 *                 .build())
 *             .encryptAtRest(DomainEncryptAtRestArgs.builder()
 *                 .enabled(true)
 *                 .build())
 *             .engineVersion("Elasticsearch_7.1")
 *             .nodeToNodeEncryption(DomainNodeToNodeEncryptionArgs.builder()
 *                 .enabled(true)
 *                 .build())
 *             .build());
 *     }
 * }
 * ```
 * ## Import
 * Using `pulumi import`, import OpenSearch domains using the `domain_name`. For example:
 * ```sh
 *  $ pulumi import aws:opensearch/domain:Domain example domain_name
 * ```
 * @property accessPolicies IAM policy document specifying the access policies for the domain.
 * @property advancedOptions Key-value string pairs to specify advanced configuration options. Note that the values for these configuration options must be strings (wrapped in quotes) or they may be wrong and cause a perpetual diff, causing the provider to want to recreate your OpenSearch domain on every apply.
 * @property advancedSecurityOptions Configuration block for [fine-grained access control](https://docs.aws.amazon.com/opensearch-service/latest/developerguide/fgac.html). Detailed below.
 * @property autoTuneOptions Configuration block for the Auto-Tune options of the domain. Detailed below.
 * @property clusterConfig Configuration block for the cluster of the domain. Detailed below.
 * @property cognitoOptions Configuration block for authenticating dashboard with Cognito. Detailed below.
 * @property domainEndpointOptions Configuration block for domain endpoint HTTP(S) related options. Detailed below.
 * @property domainName Name of the domain.
 * The following arguments are optional:
 * @property ebsOptions Configuration block for EBS related options, may be required based on chosen [instance size](https://aws.amazon.com/opensearch-service/pricing/). Detailed below.
 * @property encryptAtRest Configuration block for encrypt at rest options. Only available for [certain instance types](https://docs.aws.amazon.com/opensearch-service/latest/developerguide/encryption-at-rest.html). Detailed below.
 * @property engineVersion Either `Elasticsearch_X.Y` or `OpenSearch_X.Y` to specify the engine version for the Amazon OpenSearch Service domain. For example, `OpenSearch_1.0` or `Elasticsearch_7.9`.
 * See [Creating and managing Amazon OpenSearch Service domains](http://docs.aws.amazon.com/opensearch-service/latest/developerguide/createupdatedomains.html#createdomains).
 * Defaults to the lastest version of OpenSearch.
 * @property logPublishingOptions Configuration block for publishing slow and application logs to CloudWatch Logs. This block can be declared multiple times, for each log_type, within the same resource. Detailed below.
 * @property nodeToNodeEncryption Configuration block for node-to-node encryption options. Detailed below.
 * @property offPeakWindowOptions Configuration to add Off Peak update options. ([documentation](https://docs.aws.amazon.com/opensearch-service/latest/developerguide/off-peak.html)). Detailed below.
 * @property snapshotOptions Configuration block for snapshot related options. Detailed below. DEPRECATED. For domains running OpenSearch 5.3 and later, Amazon OpenSearch takes hourly automated snapshots, making this setting irrelevant. For domains running earlier versions, OpenSearch takes daily automated snapshots.
 * @property softwareUpdateOptions Software update options for the domain. Detailed below.
 * @property tags Map of tags to assign to the resource. If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
 * @property vpcOptions Configuration block for VPC related options. Adding or removing this configuration forces a new resource ([documentation](https://docs.aws.amazon.com/opensearch-service/latest/developerguide/vpc.html)). Detailed below.
 * */*/
 */
public data class DomainArgs(
    public val accessPolicies: Output<String>? = null,
    public val advancedOptions: Output<Map<String, String>>? = null,
    public val advancedSecurityOptions: Output<DomainAdvancedSecurityOptionsArgs>? = null,
    public val autoTuneOptions: Output<DomainAutoTuneOptionsArgs>? = null,
    public val clusterConfig: Output<DomainClusterConfigArgs>? = null,
    public val cognitoOptions: Output<DomainCognitoOptionsArgs>? = null,
    public val domainEndpointOptions: Output<DomainDomainEndpointOptionsArgs>? = null,
    public val domainName: Output<String>? = null,
    public val ebsOptions: Output<DomainEbsOptionsArgs>? = null,
    public val encryptAtRest: Output<DomainEncryptAtRestArgs>? = null,
    public val engineVersion: Output<String>? = null,
    public val logPublishingOptions: Output<List<DomainLogPublishingOptionArgs>>? = null,
    public val nodeToNodeEncryption: Output<DomainNodeToNodeEncryptionArgs>? = null,
    public val offPeakWindowOptions: Output<DomainOffPeakWindowOptionsArgs>? = null,
    public val snapshotOptions: Output<DomainSnapshotOptionsArgs>? = null,
    public val softwareUpdateOptions: Output<DomainSoftwareUpdateOptionsArgs>? = null,
    public val tags: Output<Map<String, String>>? = null,
    public val vpcOptions: Output<DomainVpcOptionsArgs>? = null,
) : ConvertibleToJava<com.pulumi.aws.opensearch.DomainArgs> {
    override fun toJava(): com.pulumi.aws.opensearch.DomainArgs =
        com.pulumi.aws.opensearch.DomainArgs.builder()
            .accessPolicies(accessPolicies?.applyValue({ args0 -> args0 }))
            .advancedOptions(
                advancedOptions?.applyValue({ args0 ->
                    args0.map({ args0 ->
                        args0.key.to(args0.value)
                    }).toMap()
                }),
            )
            .advancedSecurityOptions(
                advancedSecurityOptions?.applyValue({ args0 ->
                    args0.let({ args0 ->
                        args0.toJava()
                    })
                }),
            )
            .autoTuneOptions(autoTuneOptions?.applyValue({ args0 -> args0.let({ args0 -> args0.toJava() }) }))
            .clusterConfig(clusterConfig?.applyValue({ args0 -> args0.let({ args0 -> args0.toJava() }) }))
            .cognitoOptions(cognitoOptions?.applyValue({ args0 -> args0.let({ args0 -> args0.toJava() }) }))
            .domainEndpointOptions(
                domainEndpointOptions?.applyValue({ args0 ->
                    args0.let({ args0 ->
                        args0.toJava()
                    })
                }),
            )
            .domainName(domainName?.applyValue({ args0 -> args0 }))
            .ebsOptions(ebsOptions?.applyValue({ args0 -> args0.let({ args0 -> args0.toJava() }) }))
            .encryptAtRest(encryptAtRest?.applyValue({ args0 -> args0.let({ args0 -> args0.toJava() }) }))
            .engineVersion(engineVersion?.applyValue({ args0 -> args0 }))
            .logPublishingOptions(
                logPublishingOptions?.applyValue({ args0 ->
                    args0.map({ args0 ->
                        args0.let({ args0 -> args0.toJava() })
                    })
                }),
            )
            .nodeToNodeEncryption(
                nodeToNodeEncryption?.applyValue({ args0 ->
                    args0.let({ args0 ->
                        args0.toJava()
                    })
                }),
            )
            .offPeakWindowOptions(
                offPeakWindowOptions?.applyValue({ args0 ->
                    args0.let({ args0 ->
                        args0.toJava()
                    })
                }),
            )
            .snapshotOptions(snapshotOptions?.applyValue({ args0 -> args0.let({ args0 -> args0.toJava() }) }))
            .softwareUpdateOptions(
                softwareUpdateOptions?.applyValue({ args0 ->
                    args0.let({ args0 ->
                        args0.toJava()
                    })
                }),
            )
            .tags(tags?.applyValue({ args0 -> args0.map({ args0 -> args0.key.to(args0.value) }).toMap() }))
            .vpcOptions(vpcOptions?.applyValue({ args0 -> args0.let({ args0 -> args0.toJava() }) })).build()
}

/**
 * Builder for [DomainArgs].
 */
@PulumiTagMarker
public class DomainArgsBuilder internal constructor() {
    private var accessPolicies: Output<String>? = null

    private var advancedOptions: Output<Map<String, String>>? = null

    private var advancedSecurityOptions: Output<DomainAdvancedSecurityOptionsArgs>? = null

    private var autoTuneOptions: Output<DomainAutoTuneOptionsArgs>? = null

    private var clusterConfig: Output<DomainClusterConfigArgs>? = null

    private var cognitoOptions: Output<DomainCognitoOptionsArgs>? = null

    private var domainEndpointOptions: Output<DomainDomainEndpointOptionsArgs>? = null

    private var domainName: Output<String>? = null

    private var ebsOptions: Output<DomainEbsOptionsArgs>? = null

    private var encryptAtRest: Output<DomainEncryptAtRestArgs>? = null

    private var engineVersion: Output<String>? = null

    private var logPublishingOptions: Output<List<DomainLogPublishingOptionArgs>>? = null

    private var nodeToNodeEncryption: Output<DomainNodeToNodeEncryptionArgs>? = null

    private var offPeakWindowOptions: Output<DomainOffPeakWindowOptionsArgs>? = null

    private var snapshotOptions: Output<DomainSnapshotOptionsArgs>? = null

    private var softwareUpdateOptions: Output<DomainSoftwareUpdateOptionsArgs>? = null

    private var tags: Output<Map<String, String>>? = null

    private var vpcOptions: Output<DomainVpcOptionsArgs>? = null

    /**
     * @param value IAM policy document specifying the access policies for the domain.
     */
    @JvmName("reimanskktijjoyq")
    public suspend fun accessPolicies(`value`: Output<String>) {
        this.accessPolicies = value
    }

    /**
     * @param value Key-value string pairs to specify advanced configuration options. Note that the values for these configuration options must be strings (wrapped in quotes) or they may be wrong and cause a perpetual diff, causing the provider to want to recreate your OpenSearch domain on every apply.
     */
    @JvmName("dyjseckiibcnseml")
    public suspend fun advancedOptions(`value`: Output<Map<String, String>>) {
        this.advancedOptions = value
    }

    /**
     * @param value Configuration block for [fine-grained access control](https://docs.aws.amazon.com/opensearch-service/latest/developerguide/fgac.html). Detailed below.
     */
    @JvmName("tthykscbodimgcfv")
    public suspend fun advancedSecurityOptions(`value`: Output<DomainAdvancedSecurityOptionsArgs>) {
        this.advancedSecurityOptions = value
    }

    /**
     * @param value Configuration block for the Auto-Tune options of the domain. Detailed below.
     */
    @JvmName("qqqjtnsphwndofcq")
    public suspend fun autoTuneOptions(`value`: Output<DomainAutoTuneOptionsArgs>) {
        this.autoTuneOptions = value
    }

    /**
     * @param value Configuration block for the cluster of the domain. Detailed below.
     */
    @JvmName("dahwrkrcfjhhfwnp")
    public suspend fun clusterConfig(`value`: Output<DomainClusterConfigArgs>) {
        this.clusterConfig = value
    }

    /**
     * @param value Configuration block for authenticating dashboard with Cognito. Detailed below.
     */
    @JvmName("avondfsennqjbhxq")
    public suspend fun cognitoOptions(`value`: Output<DomainCognitoOptionsArgs>) {
        this.cognitoOptions = value
    }

    /**
     * @param value Configuration block for domain endpoint HTTP(S) related options. Detailed below.
     */
    @JvmName("hfwwyemrtaoluyjy")
    public suspend fun domainEndpointOptions(`value`: Output<DomainDomainEndpointOptionsArgs>) {
        this.domainEndpointOptions = value
    }

    /**
     * @param value Name of the domain.
     * The following arguments are optional:
     */
    @JvmName("kiggayqhxhddpdts")
    public suspend fun domainName(`value`: Output<String>) {
        this.domainName = value
    }

    /**
     * @param value Configuration block for EBS related options, may be required based on chosen [instance size](https://aws.amazon.com/opensearch-service/pricing/). Detailed below.
     */
    @JvmName("vuudqfesbfjmjohx")
    public suspend fun ebsOptions(`value`: Output<DomainEbsOptionsArgs>) {
        this.ebsOptions = value
    }

    /**
     * @param value Configuration block for encrypt at rest options. Only available for [certain instance types](https://docs.aws.amazon.com/opensearch-service/latest/developerguide/encryption-at-rest.html). Detailed below.
     */
    @JvmName("jjrdvbdiuwmwkeke")
    public suspend fun encryptAtRest(`value`: Output<DomainEncryptAtRestArgs>) {
        this.encryptAtRest = value
    }

    /**
     * @param value Either `Elasticsearch_X.Y` or `OpenSearch_X.Y` to specify the engine version for the Amazon OpenSearch Service domain. For example, `OpenSearch_1.0` or `Elasticsearch_7.9`.
     * See [Creating and managing Amazon OpenSearch Service domains](http://docs.aws.amazon.com/opensearch-service/latest/developerguide/createupdatedomains.html#createdomains).
     * Defaults to the lastest version of OpenSearch.
     */
    @JvmName("yoldysttxcvjxdcw")
    public suspend fun engineVersion(`value`: Output<String>) {
        this.engineVersion = value
    }

    /**
     * @param value Configuration block for publishing slow and application logs to CloudWatch Logs. This block can be declared multiple times, for each log_type, within the same resource. Detailed below.
     */
    @JvmName("ekbcermqgpgoxiuu")
    public suspend fun logPublishingOptions(`value`: Output<List<DomainLogPublishingOptionArgs>>) {
        this.logPublishingOptions = value
    }

    @JvmName("jspbnsbjvavyuqsj")
    public suspend fun logPublishingOptions(vararg values: Output<DomainLogPublishingOptionArgs>) {
        this.logPublishingOptions = Output.all(values.asList())
    }

    /**
     * @param values Configuration block for publishing slow and application logs to CloudWatch Logs. This block can be declared multiple times, for each log_type, within the same resource. Detailed below.
     */
    @JvmName("syukpbhrwasbvoqj")
    public suspend fun logPublishingOptions(values: List<Output<DomainLogPublishingOptionArgs>>) {
        this.logPublishingOptions = Output.all(values)
    }

    /**
     * @param value Configuration block for node-to-node encryption options. Detailed below.
     */
    @JvmName("vbofjntwbfmlhdug")
    public suspend fun nodeToNodeEncryption(`value`: Output<DomainNodeToNodeEncryptionArgs>) {
        this.nodeToNodeEncryption = value
    }

    /**
     * @param value Configuration to add Off Peak update options. ([documentation](https://docs.aws.amazon.com/opensearch-service/latest/developerguide/off-peak.html)). Detailed below.
     */
    @JvmName("jephksaurvcfhmmx")
    public suspend fun offPeakWindowOptions(`value`: Output<DomainOffPeakWindowOptionsArgs>) {
        this.offPeakWindowOptions = value
    }

    /**
     * @param value Configuration block for snapshot related options. Detailed below. DEPRECATED. For domains running OpenSearch 5.3 and later, Amazon OpenSearch takes hourly automated snapshots, making this setting irrelevant. For domains running earlier versions, OpenSearch takes daily automated snapshots.
     */
    @JvmName("jmqaftmoawvlolcr")
    public suspend fun snapshotOptions(`value`: Output<DomainSnapshotOptionsArgs>) {
        this.snapshotOptions = value
    }

    /**
     * @param value Software update options for the domain. Detailed below.
     */
    @JvmName("aijdmfrschnawcyk")
    public suspend fun softwareUpdateOptions(`value`: Output<DomainSoftwareUpdateOptionsArgs>) {
        this.softwareUpdateOptions = value
    }

    /**
     * @param value Map of tags to assign to the resource. If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
     */
    @JvmName("kybucstgjqpesfcv")
    public suspend fun tags(`value`: Output<Map<String, String>>) {
        this.tags = value
    }

    /**
     * @param value Configuration block for VPC related options. Adding or removing this configuration forces a new resource ([documentation](https://docs.aws.amazon.com/opensearch-service/latest/developerguide/vpc.html)). Detailed below.
     */
    @JvmName("rqoidichrdgvymum")
    public suspend fun vpcOptions(`value`: Output<DomainVpcOptionsArgs>) {
        this.vpcOptions = value
    }

    /**
     * @param value IAM policy document specifying the access policies for the domain.
     */
    @JvmName("rglbiterklrtednk")
    public suspend fun accessPolicies(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.accessPolicies = mapped
    }

    /**
     * @param value Key-value string pairs to specify advanced configuration options. Note that the values for these configuration options must be strings (wrapped in quotes) or they may be wrong and cause a perpetual diff, causing the provider to want to recreate your OpenSearch domain on every apply.
     */
    @JvmName("aejhkuedsdvxrwlm")
    public suspend fun advancedOptions(`value`: Map<String, String>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.advancedOptions = mapped
    }

    /**
     * @param values Key-value string pairs to specify advanced configuration options. Note that the values for these configuration options must be strings (wrapped in quotes) or they may be wrong and cause a perpetual diff, causing the provider to want to recreate your OpenSearch domain on every apply.
     */
    @JvmName("yoepqiglvpubxdtr")
    public fun advancedOptions(vararg values: Pair<String, String>) {
        val toBeMapped = values.toMap()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.advancedOptions = mapped
    }

    /**
     * @param value Configuration block for [fine-grained access control](https://docs.aws.amazon.com/opensearch-service/latest/developerguide/fgac.html). Detailed below.
     */
    @JvmName("bfnjolhojuibsytt")
    public suspend fun advancedSecurityOptions(`value`: DomainAdvancedSecurityOptionsArgs?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.advancedSecurityOptions = mapped
    }

    /**
     * @param argument Configuration block for [fine-grained access control](https://docs.aws.amazon.com/opensearch-service/latest/developerguide/fgac.html). Detailed below.
     */
    @JvmName("sgpxbkdtrnqfuxqb")
    public suspend
    fun advancedSecurityOptions(argument: suspend DomainAdvancedSecurityOptionsArgsBuilder.() -> Unit) {
        val toBeMapped = DomainAdvancedSecurityOptionsArgsBuilder().applySuspend { argument() }.build()
        val mapped = of(toBeMapped)
        this.advancedSecurityOptions = mapped
    }

    /**
     * @param value Configuration block for the Auto-Tune options of the domain. Detailed below.
     */
    @JvmName("scfquapbclqqifph")
    public suspend fun autoTuneOptions(`value`: DomainAutoTuneOptionsArgs?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.autoTuneOptions = mapped
    }

    /**
     * @param argument Configuration block for the Auto-Tune options of the domain. Detailed below.
     */
    @JvmName("tmpmurtddxfsacwq")
    public suspend
    fun autoTuneOptions(argument: suspend DomainAutoTuneOptionsArgsBuilder.() -> Unit) {
        val toBeMapped = DomainAutoTuneOptionsArgsBuilder().applySuspend { argument() }.build()
        val mapped = of(toBeMapped)
        this.autoTuneOptions = mapped
    }

    /**
     * @param value Configuration block for the cluster of the domain. Detailed below.
     */
    @JvmName("sdyjarducxcesfts")
    public suspend fun clusterConfig(`value`: DomainClusterConfigArgs?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.clusterConfig = mapped
    }

    /**
     * @param argument Configuration block for the cluster of the domain. Detailed below.
     */
    @JvmName("wyqujpufouteofva")
    public suspend fun clusterConfig(argument: suspend DomainClusterConfigArgsBuilder.() -> Unit) {
        val toBeMapped = DomainClusterConfigArgsBuilder().applySuspend { argument() }.build()
        val mapped = of(toBeMapped)
        this.clusterConfig = mapped
    }

    /**
     * @param value Configuration block for authenticating dashboard with Cognito. Detailed below.
     */
    @JvmName("lmlvcjcdponqfxmf")
    public suspend fun cognitoOptions(`value`: DomainCognitoOptionsArgs?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.cognitoOptions = mapped
    }

    /**
     * @param argument Configuration block for authenticating dashboard with Cognito. Detailed below.
     */
    @JvmName("bapqfrcgxkjchcqx")
    public suspend fun cognitoOptions(argument: suspend DomainCognitoOptionsArgsBuilder.() -> Unit) {
        val toBeMapped = DomainCognitoOptionsArgsBuilder().applySuspend { argument() }.build()
        val mapped = of(toBeMapped)
        this.cognitoOptions = mapped
    }

    /**
     * @param value Configuration block for domain endpoint HTTP(S) related options. Detailed below.
     */
    @JvmName("xpqbmpvdmtjwxwow")
    public suspend fun domainEndpointOptions(`value`: DomainDomainEndpointOptionsArgs?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.domainEndpointOptions = mapped
    }

    /**
     * @param argument Configuration block for domain endpoint HTTP(S) related options. Detailed below.
     */
    @JvmName("tyeyyjxbjqcbgwxs")
    public suspend
    fun domainEndpointOptions(argument: suspend DomainDomainEndpointOptionsArgsBuilder.() -> Unit) {
        val toBeMapped = DomainDomainEndpointOptionsArgsBuilder().applySuspend { argument() }.build()
        val mapped = of(toBeMapped)
        this.domainEndpointOptions = mapped
    }

    /**
     * @param value Name of the domain.
     * The following arguments are optional:
     */
    @JvmName("spglhhyfivsvgfdv")
    public suspend fun domainName(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.domainName = mapped
    }

    /**
     * @param value Configuration block for EBS related options, may be required based on chosen [instance size](https://aws.amazon.com/opensearch-service/pricing/). Detailed below.
     */
    @JvmName("mnldsdlusdyxaoll")
    public suspend fun ebsOptions(`value`: DomainEbsOptionsArgs?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.ebsOptions = mapped
    }

    /**
     * @param argument Configuration block for EBS related options, may be required based on chosen [instance size](https://aws.amazon.com/opensearch-service/pricing/). Detailed below.
     */
    @JvmName("bjexqghhksphpgro")
    public suspend fun ebsOptions(argument: suspend DomainEbsOptionsArgsBuilder.() -> Unit) {
        val toBeMapped = DomainEbsOptionsArgsBuilder().applySuspend { argument() }.build()
        val mapped = of(toBeMapped)
        this.ebsOptions = mapped
    }

    /**
     * @param value Configuration block for encrypt at rest options. Only available for [certain instance types](https://docs.aws.amazon.com/opensearch-service/latest/developerguide/encryption-at-rest.html). Detailed below.
     */
    @JvmName("symjmsrlilnsswwy")
    public suspend fun encryptAtRest(`value`: DomainEncryptAtRestArgs?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.encryptAtRest = mapped
    }

    /**
     * @param argument Configuration block for encrypt at rest options. Only available for [certain instance types](https://docs.aws.amazon.com/opensearch-service/latest/developerguide/encryption-at-rest.html). Detailed below.
     */
    @JvmName("pgrfsvoviujpgklr")
    public suspend fun encryptAtRest(argument: suspend DomainEncryptAtRestArgsBuilder.() -> Unit) {
        val toBeMapped = DomainEncryptAtRestArgsBuilder().applySuspend { argument() }.build()
        val mapped = of(toBeMapped)
        this.encryptAtRest = mapped
    }

    /**
     * @param value Either `Elasticsearch_X.Y` or `OpenSearch_X.Y` to specify the engine version for the Amazon OpenSearch Service domain. For example, `OpenSearch_1.0` or `Elasticsearch_7.9`.
     * See [Creating and managing Amazon OpenSearch Service domains](http://docs.aws.amazon.com/opensearch-service/latest/developerguide/createupdatedomains.html#createdomains).
     * Defaults to the lastest version of OpenSearch.
     */
    @JvmName("eafulvayrnujvlyh")
    public suspend fun engineVersion(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.engineVersion = mapped
    }

    /**
     * @param value Configuration block for publishing slow and application logs to CloudWatch Logs. This block can be declared multiple times, for each log_type, within the same resource. Detailed below.
     */
    @JvmName("whobnramwgjinbwt")
    public suspend fun logPublishingOptions(`value`: List<DomainLogPublishingOptionArgs>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.logPublishingOptions = mapped
    }

    /**
     * @param argument Configuration block for publishing slow and application logs to CloudWatch Logs. This block can be declared multiple times, for each log_type, within the same resource. Detailed below.
     */
    @JvmName("bdamkrnnrlmgqkfg")
    public suspend
    fun logPublishingOptions(argument: List<suspend DomainLogPublishingOptionArgsBuilder.() -> Unit>) {
        val toBeMapped = argument.toList().map {
            DomainLogPublishingOptionArgsBuilder().applySuspend {
                it()
            }.build()
        }
        val mapped = of(toBeMapped)
        this.logPublishingOptions = mapped
    }

    /**
     * @param argument Configuration block for publishing slow and application logs to CloudWatch Logs. This block can be declared multiple times, for each log_type, within the same resource. Detailed below.
     */
    @JvmName("gxhsfoyngvhriebf")
    public suspend fun logPublishingOptions(
        vararg
        argument: suspend DomainLogPublishingOptionArgsBuilder.() -> Unit,
    ) {
        val toBeMapped = argument.toList().map {
            DomainLogPublishingOptionArgsBuilder().applySuspend {
                it()
            }.build()
        }
        val mapped = of(toBeMapped)
        this.logPublishingOptions = mapped
    }

    /**
     * @param argument Configuration block for publishing slow and application logs to CloudWatch Logs. This block can be declared multiple times, for each log_type, within the same resource. Detailed below.
     */
    @JvmName("ypldfoohqjagoyqu")
    public suspend
    fun logPublishingOptions(argument: suspend DomainLogPublishingOptionArgsBuilder.() -> Unit) {
        val toBeMapped = listOf(
            DomainLogPublishingOptionArgsBuilder().applySuspend {
                argument()
            }.build(),
        )
        val mapped = of(toBeMapped)
        this.logPublishingOptions = mapped
    }

    /**
     * @param values Configuration block for publishing slow and application logs to CloudWatch Logs. This block can be declared multiple times, for each log_type, within the same resource. Detailed below.
     */
    @JvmName("bannefrbdkbbtrws")
    public suspend fun logPublishingOptions(vararg values: DomainLogPublishingOptionArgs) {
        val toBeMapped = values.toList()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.logPublishingOptions = mapped
    }

    /**
     * @param value Configuration block for node-to-node encryption options. Detailed below.
     */
    @JvmName("iewsinhjnxcrgtet")
    public suspend fun nodeToNodeEncryption(`value`: DomainNodeToNodeEncryptionArgs?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.nodeToNodeEncryption = mapped
    }

    /**
     * @param argument Configuration block for node-to-node encryption options. Detailed below.
     */
    @JvmName("vpnpbquicheghqlw")
    public suspend
    fun nodeToNodeEncryption(argument: suspend DomainNodeToNodeEncryptionArgsBuilder.() -> Unit) {
        val toBeMapped = DomainNodeToNodeEncryptionArgsBuilder().applySuspend { argument() }.build()
        val mapped = of(toBeMapped)
        this.nodeToNodeEncryption = mapped
    }

    /**
     * @param value Configuration to add Off Peak update options. ([documentation](https://docs.aws.amazon.com/opensearch-service/latest/developerguide/off-peak.html)). Detailed below.
     */
    @JvmName("obponajkvsjynowm")
    public suspend fun offPeakWindowOptions(`value`: DomainOffPeakWindowOptionsArgs?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.offPeakWindowOptions = mapped
    }

    /**
     * @param argument Configuration to add Off Peak update options. ([documentation](https://docs.aws.amazon.com/opensearch-service/latest/developerguide/off-peak.html)). Detailed below.
     */
    @JvmName("dmytyenyhxrdohuk")
    public suspend
    fun offPeakWindowOptions(argument: suspend DomainOffPeakWindowOptionsArgsBuilder.() -> Unit) {
        val toBeMapped = DomainOffPeakWindowOptionsArgsBuilder().applySuspend { argument() }.build()
        val mapped = of(toBeMapped)
        this.offPeakWindowOptions = mapped
    }

    /**
     * @param value Configuration block for snapshot related options. Detailed below. DEPRECATED. For domains running OpenSearch 5.3 and later, Amazon OpenSearch takes hourly automated snapshots, making this setting irrelevant. For domains running earlier versions, OpenSearch takes daily automated snapshots.
     */
    @JvmName("blydmgcpugqtdkrs")
    public suspend fun snapshotOptions(`value`: DomainSnapshotOptionsArgs?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.snapshotOptions = mapped
    }

    /**
     * @param argument Configuration block for snapshot related options. Detailed below. DEPRECATED. For domains running OpenSearch 5.3 and later, Amazon OpenSearch takes hourly automated snapshots, making this setting irrelevant. For domains running earlier versions, OpenSearch takes daily automated snapshots.
     */
    @JvmName("lwtfkyafpgpmlkao")
    public suspend
    fun snapshotOptions(argument: suspend DomainSnapshotOptionsArgsBuilder.() -> Unit) {
        val toBeMapped = DomainSnapshotOptionsArgsBuilder().applySuspend { argument() }.build()
        val mapped = of(toBeMapped)
        this.snapshotOptions = mapped
    }

    /**
     * @param value Software update options for the domain. Detailed below.
     */
    @JvmName("aomkramemyhwvfde")
    public suspend fun softwareUpdateOptions(`value`: DomainSoftwareUpdateOptionsArgs?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.softwareUpdateOptions = mapped
    }

    /**
     * @param argument Software update options for the domain. Detailed below.
     */
    @JvmName("skkfphomrsjmlgxw")
    public suspend
    fun softwareUpdateOptions(argument: suspend DomainSoftwareUpdateOptionsArgsBuilder.() -> Unit) {
        val toBeMapped = DomainSoftwareUpdateOptionsArgsBuilder().applySuspend { argument() }.build()
        val mapped = of(toBeMapped)
        this.softwareUpdateOptions = mapped
    }

    /**
     * @param value Map of tags to assign to the resource. If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
     */
    @JvmName("wdggbnifolxhdwlj")
    public suspend fun tags(`value`: Map<String, String>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.tags = mapped
    }

    /**
     * @param values Map of tags to assign to the resource. If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
     */
    @JvmName("whnwgsvouuoecbef")
    public fun tags(vararg values: Pair<String, String>) {
        val toBeMapped = values.toMap()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.tags = mapped
    }

    /**
     * @param value Configuration block for VPC related options. Adding or removing this configuration forces a new resource ([documentation](https://docs.aws.amazon.com/opensearch-service/latest/developerguide/vpc.html)). Detailed below.
     */
    @JvmName("agocxifwiriquwpx")
    public suspend fun vpcOptions(`value`: DomainVpcOptionsArgs?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.vpcOptions = mapped
    }

    /**
     * @param argument Configuration block for VPC related options. Adding or removing this configuration forces a new resource ([documentation](https://docs.aws.amazon.com/opensearch-service/latest/developerguide/vpc.html)). Detailed below.
     */
    @JvmName("iebvodwaulpawqbm")
    public suspend fun vpcOptions(argument: suspend DomainVpcOptionsArgsBuilder.() -> Unit) {
        val toBeMapped = DomainVpcOptionsArgsBuilder().applySuspend { argument() }.build()
        val mapped = of(toBeMapped)
        this.vpcOptions = mapped
    }

    internal fun build(): DomainArgs = DomainArgs(
        accessPolicies = accessPolicies,
        advancedOptions = advancedOptions,
        advancedSecurityOptions = advancedSecurityOptions,
        autoTuneOptions = autoTuneOptions,
        clusterConfig = clusterConfig,
        cognitoOptions = cognitoOptions,
        domainEndpointOptions = domainEndpointOptions,
        domainName = domainName,
        ebsOptions = ebsOptions,
        encryptAtRest = encryptAtRest,
        engineVersion = engineVersion,
        logPublishingOptions = logPublishingOptions,
        nodeToNodeEncryption = nodeToNodeEncryption,
        offPeakWindowOptions = offPeakWindowOptions,
        snapshotOptions = snapshotOptions,
        softwareUpdateOptions = softwareUpdateOptions,
        tags = tags,
        vpcOptions = vpcOptions,
    )
}
