@file:Suppress("NAME_SHADOWING", "DEPRECATION")

package com.pulumi.aws.rds.kotlin

import com.pulumi.aws.rds.ClusterArgs.builder
import com.pulumi.aws.rds.kotlin.enums.EngineMode
import com.pulumi.aws.rds.kotlin.enums.EngineType
import com.pulumi.aws.rds.kotlin.inputs.ClusterRestoreToPointInTimeArgs
import com.pulumi.aws.rds.kotlin.inputs.ClusterRestoreToPointInTimeArgsBuilder
import com.pulumi.aws.rds.kotlin.inputs.ClusterS3ImportArgs
import com.pulumi.aws.rds.kotlin.inputs.ClusterS3ImportArgsBuilder
import com.pulumi.aws.rds.kotlin.inputs.ClusterScalingConfigurationArgs
import com.pulumi.aws.rds.kotlin.inputs.ClusterScalingConfigurationArgsBuilder
import com.pulumi.aws.rds.kotlin.inputs.ClusterServerlessv2ScalingConfigurationArgs
import com.pulumi.aws.rds.kotlin.inputs.ClusterServerlessv2ScalingConfigurationArgsBuilder
import com.pulumi.core.Either
import com.pulumi.core.Output
import com.pulumi.core.Output.of
import com.pulumi.kotlin.ConvertibleToJava
import com.pulumi.kotlin.PulumiTagMarker
import com.pulumi.kotlin.applySuspend
import kotlin.Boolean
import kotlin.Int
import kotlin.Pair
import kotlin.String
import kotlin.Suppress
import kotlin.Unit
import kotlin.collections.List
import kotlin.collections.Map
import kotlin.jvm.JvmName

/**
 * Manages a [RDS Aurora Cluster](https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/CHAP_Aurora.html). To manage cluster instances that inherit configuration from the cluster (when not running the cluster in `serverless` engine mode), see the `aws.rds.ClusterInstance` resource. To manage non-Aurora databases (e.g., MySQL, PostgreSQL, SQL Server, etc.), see the `aws.rds.Instance` resource.
 * For information on the difference between the available Aurora MySQL engines
 * see [Comparison between Aurora MySQL 1 and Aurora MySQL 2](https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/AuroraMySQL.Updates.20180206.html)
 * in the Amazon RDS User Guide.
 * Changes to an RDS Cluster can occur when you manually change a
 * parameter, such as `port`, and are reflected in the next maintenance
 * window. Because of this, this provider may report a difference in its planning
 * phase because a modification has not yet taken place. You can use the
 * `apply_immediately` flag to instruct the service to apply the change immediately
 * (see documentation below).
 * > **Note:** using `apply_immediately` can result in a
 * brief downtime as the server reboots. See the AWS Docs on [RDS Maintenance](https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_UpgradeDBInstance.Maintenance.html)
 * for more information.
 * ## Example Usage
 * ### Aurora MySQL 2.x (MySQL 5.7)
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.rds.Cluster;
 * import com.pulumi.aws.rds.ClusterArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var default_ = new Cluster("default", ClusterArgs.builder()
 *             .availabilityZones(
 *                 "us-west-2a",
 *                 "us-west-2b",
 *                 "us-west-2c")
 *             .backupRetentionPeriod(5)
 *             .clusterIdentifier("aurora-cluster-demo")
 *             .databaseName("mydb")
 *             .engine("aurora-mysql")
 *             .engineVersion("5.7.mysql_aurora.2.03.2")
 *             .masterPassword("bar")
 *             .masterUsername("foo")
 *             .preferredBackupWindow("07:00-09:00")
 *             .build());
 *     }
 * }
 * ```
 * ### Aurora MySQL 1.x (MySQL 5.6)
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.rds.Cluster;
 * import com.pulumi.aws.rds.ClusterArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var default_ = new Cluster("default", ClusterArgs.builder()
 *             .availabilityZones(
 *                 "us-west-2a",
 *                 "us-west-2b",
 *                 "us-west-2c")
 *             .backupRetentionPeriod(5)
 *             .clusterIdentifier("aurora-cluster-demo")
 *             .databaseName("mydb")
 *             .masterPassword("bar")
 *             .masterUsername("foo")
 *             .preferredBackupWindow("07:00-09:00")
 *             .build());
 *     }
 * }
 * ```
 * ### Aurora with PostgreSQL engine
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.rds.Cluster;
 * import com.pulumi.aws.rds.ClusterArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var postgresql = new Cluster("postgresql", ClusterArgs.builder()
 *             .availabilityZones(
 *                 "us-west-2a",
 *                 "us-west-2b",
 *                 "us-west-2c")
 *             .backupRetentionPeriod(5)
 *             .clusterIdentifier("aurora-cluster-demo")
 *             .databaseName("mydb")
 *             .engine("aurora-postgresql")
 *             .masterPassword("bar")
 *             .masterUsername("foo")
 *             .preferredBackupWindow("07:00-09:00")
 *             .build());
 *     }
 * }
 * ```
 * ### RDS Multi-AZ Cluster
 * > More information about RDS Multi-AZ Clusters can be found in the [RDS User Guide](https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/multi-az-db-clusters-concepts.html).
 * To create a Multi-AZ RDS cluster, you must additionally specify the `engine`, `storage_type`, `allocated_storage`, `iops` and `db_cluster_instance_class` attributes.
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.rds.Cluster;
 * import com.pulumi.aws.rds.ClusterArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var example = new Cluster("example", ClusterArgs.builder()
 *             .allocatedStorage(100)
 *             .availabilityZones(
 *                 "us-west-2a",
 *                 "us-west-2b",
 *                 "us-west-2c")
 *             .clusterIdentifier("example")
 *             .dbClusterInstanceClass("db.r6gd.xlarge")
 *             .engine("mysql")
 *             .iops(1000)
 *             .masterPassword("mustbeeightcharaters")
 *             .masterUsername("test")
 *             .storageType("io1")
 *             .build());
 *     }
 * }
 * ```
 * ### RDS Serverless v2 Cluster
 * > More information about RDS Serverless v2 Clusters can be found in the [RDS User Guide](https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/aurora-serverless-v2.html).
 * To create a Serverless v2 RDS cluster, you must additionally specify the `engine_mode` and `serverlessv2_scaling_configuration` attributes. An `aws.rds.ClusterInstance` resource must also be added to the cluster with the `instance_class` attribute specified.
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.rds.Cluster;
 * import com.pulumi.aws.rds.ClusterArgs;
 * import com.pulumi.aws.rds.inputs.ClusterServerlessv2ScalingConfigurationArgs;
 * import com.pulumi.aws.rds.ClusterInstance;
 * import com.pulumi.aws.rds.ClusterInstanceArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var exampleCluster = new Cluster("exampleCluster", ClusterArgs.builder()
 *             .clusterIdentifier("example")
 *             .engine("aurora-postgresql")
 *             .engineMode("provisioned")
 *             .engineVersion("13.6")
 *             .databaseName("test")
 *             .masterUsername("test")
 *             .masterPassword("must_be_eight_characters")
 *             .serverlessv2ScalingConfiguration(ClusterServerlessv2ScalingConfigurationArgs.builder()
 *                 .maxCapacity(1)
 *                 .minCapacity(0.5)
 *                 .build())
 *             .build());
 *         var exampleClusterInstance = new ClusterInstance("exampleClusterInstance", ClusterInstanceArgs.builder()
 *             .clusterIdentifier(exampleCluster.id())
 *             .instanceClass("db.serverless")
 *             .engine(exampleCluster.engine())
 *             .engineVersion(exampleCluster.engineVersion())
 *             .build());
 *     }
 * }
 * ```
 * ### RDS/Aurora Managed Master Passwords via Secrets Manager, default KMS Key
 * > More information about RDS/Aurora Aurora integrates with Secrets Manager to manage master user passwords for your DB clusters can be found in the [RDS User Guide](https://aws.amazon.com/about-aws/whats-new/2022/12/amazon-rds-integration-aws-secrets-manager/) and [Aurora User Guide](https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/rds-secrets-manager.html).
 * You can specify the `manage_master_user_password` attribute to enable managing the master password with Secrets Manager. You can also update an existing cluster to use Secrets Manager by specify the `manage_master_user_password` attribute and removing the `master_password` attribute (removal is required).
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.rds.Cluster;
 * import com.pulumi.aws.rds.ClusterArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var test = new Cluster("test", ClusterArgs.builder()
 *             .clusterIdentifier("example")
 *             .databaseName("test")
 *             .manageMasterUserPassword(true)
 *             .masterUsername("test")
 *             .build());
 *     }
 * }
 * ```
 * ### RDS/Aurora Managed Master Passwords via Secrets Manager, specific KMS Key
 * > More information about RDS/Aurora Aurora integrates with Secrets Manager to manage master user passwords for your DB clusters can be found in the [RDS User Guide](https://aws.amazon.com/about-aws/whats-new/2022/12/amazon-rds-integration-aws-secrets-manager/) and [Aurora User Guide](https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/rds-secrets-manager.html).
 * You can specify the `master_user_secret_kms_key_id` attribute to specify a specific KMS Key.
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.kms.Key;
 * import com.pulumi.aws.kms.KeyArgs;
 * import com.pulumi.aws.rds.Cluster;
 * import com.pulumi.aws.rds.ClusterArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var example = new Key("example", KeyArgs.builder()
 *             .description("Example KMS Key")
 *             .build());
 *         var test = new Cluster("test", ClusterArgs.builder()
 *             .clusterIdentifier("example")
 *             .databaseName("test")
 *             .manageMasterUserPassword(true)
 *             .masterUsername("test")
 *             .masterUserSecretKmsKeyId(example.keyId())
 *             .build());
 *     }
 * }
 * ```
 * ### Global Cluster Restored From Snapshot
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.rds.RdsFunctions;
 * import com.pulumi.aws.rds.inputs.GetClusterSnapshotArgs;
 * import com.pulumi.aws.rds.Cluster;
 * import com.pulumi.aws.rds.ClusterArgs;
 * import com.pulumi.aws.rds.GlobalCluster;
 * import com.pulumi.aws.rds.GlobalClusterArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         final var exampleClusterSnapshot = RdsFunctions.getClusterSnapshot(GetClusterSnapshotArgs.builder()
 *             .dbClusterIdentifier("example-original-cluster")
 *             .mostRecent(true)
 *             .build());
 *         var exampleCluster = new Cluster("exampleCluster", ClusterArgs.builder()
 *             .engine("aurora")
 *             .engineVersion("5.6.mysql_aurora.1.22.4")
 *             .clusterIdentifier("example")
 *             .snapshotIdentifier(exampleClusterSnapshot.applyValue(getClusterSnapshotResult -> getClusterSnapshotResult.id()))
 *             .build());
 *         var exampleGlobalCluster = new GlobalCluster("exampleGlobalCluster", GlobalClusterArgs.builder()
 *             .globalClusterIdentifier("example")
 *             .sourceDbClusterIdentifier(exampleCluster.arn())
 *             .forceDestroy(true)
 *             .build());
 *     }
 * }
 * ```
 * ## Import
 * Using `pulumi import`, import RDS Clusters using the `cluster_identifier`. For example:
 * ```sh
 *  $ pulumi import aws:rds/cluster:Cluster aurora_cluster aurora-prod-cluster
 * ```
 * @property allocatedStorage The amount of storage in gibibytes (GiB) to allocate to each DB instance in the Multi-AZ DB cluster.
 * @property allowMajorVersionUpgrade Enable to allow major engine version upgrades when changing engine versions. Defaults to `false`.
 * @property applyImmediately Specifies whether any cluster modifications are applied immediately, or during the next maintenance window. Default is `false`. See [Amazon RDS Documentation for more information&#46;](https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/Overview.DBInstance.Modifying.html)
 * @property availabilityZones List of EC2 Availability Zones for the DB cluster storage where DB cluster instances can be created.
 * RDS automatically assigns 3 AZs if less than 3 AZs are configured, which will show as a difference requiring resource recreation next pulumi up.
 * We recommend specifying 3 AZs or using the `lifecycle` configuration block `ignore_changes` argument if necessary.
 * A maximum of 3 AZs can be configured.
 * @property backtrackWindow Target backtrack window, in seconds. Only available for `aurora` and `aurora-mysql` engines currently. To disable backtracking, set this value to `0`. Defaults to `0`. Must be between `0` and `259200` (72 hours)
 * @property backupRetentionPeriod Days to retain backups for. Default `1`
 * @property clusterIdentifier The cluster identifier. If omitted, this provider will assign a random, unique identifier.
 * @property clusterIdentifierPrefix Creates a unique cluster identifier beginning with the specified prefix. Conflicts with `cluster_identifier`.
 * @property clusterMembers List of RDS Instances that are a part of this cluster
 * @property copyTagsToSnapshot Copy all Cluster `tags` to snapshots. Default is `false`.
 * @property databaseName Name for an automatically created database on cluster creation. There are different naming restrictions per database engine: [RDS Naming Constraints](http://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/CHAP_Limits.html#RDS_Limits.Constraints)
 * @property dbClusterInstanceClass The compute and memory capacity of each DB instance in the Multi-AZ DB cluster, for example `db.m6g.xlarge`. Not all DB instance classes are available in all AWS Regions, or for all database engines. For the full list of DB instance classes and availability for your engine, see [DB instance class](https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/Concepts.DBInstanceClass.html) in the Amazon RDS User Guide.
 * @property dbClusterParameterGroupName A cluster parameter group to associate with the cluster.
 * @property dbInstanceParameterGroupName Instance parameter group to associate with all instances of the DB cluster. The `db_instance_parameter_group_name` parameter is only valid in combination with the `allow_major_version_upgrade` parameter.
 * @property dbSubnetGroupName DB subnet group to associate with this DB cluster.
 * **NOTE:** This must match the `db_subnet_group_name` specified on every `aws.rds.ClusterInstance` in the cluster.
 * @property dbSystemId For use with RDS Custom.
 * @property deleteAutomatedBackups Specifies whether to remove automated backups immediately after the DB cluster is deleted. Default is `true`.
 * @property deletionProtection If the DB cluster should have deletion protection enabled.
 * The database can't be deleted when this value is set to `true`.
 * The default is `false`.
 * @property enableGlobalWriteForwarding Whether cluster should forward writes to an associated global cluster. Applied to secondary clusters to enable them to forward writes to an `aws.rds.GlobalCluster`'s primary cluster. See the [Aurora Userguide documentation](https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/aurora-global-database-write-forwarding.html) for more information.
 * @property enableHttpEndpoint Enable HTTP endpoint (data API). Only valid when `engine_mode` is set to `serverless`.
 * @property enabledCloudwatchLogsExports Set of log types to export to cloudwatch. If omitted, no logs will be exported. The following log types are supported: `audit`, `error`, `general`, `slowquery`, `postgresql` (PostgreSQL).
 * @property engine Name of the database engine to be used for this DB cluster. Valid Values: `aurora-mysql`, `aurora-postgresql`, `mysql`, `postgres`. (Note that `mysql` and `postgres` are Multi-AZ RDS clusters).
 * @property engineMode Database engine mode. Valid values: `global` (only valid for Aurora MySQL 1.21 and earlier), `parallelquery`, `provisioned`, `serverless`. Defaults to: `provisioned`. See the [RDS User Guide](https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/aurora-serverless.html) for limitations when using `serverless`.
 * @property engineVersion Database engine version. Updating this argument results in an outage. See the [Aurora MySQL](https://docs&#46;aws&#46;amazon&#46;com/AmazonRDS/latest/AuroraUserGuide/AuroraMySQL&#46;Updates&#46;html) and [Aurora Postgres](https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/AuroraPostgreSQL.Updates.html) documentation for your configured engine to determine this value, or by running `aws rds describe-db-engine-versions`. For example with Aurora MySQL 2, a potential value for this argument is `5.7.mysql_aurora.2.03.2`. The value can contain a partial version where supported by the API. The actual engine version used is returned in the attribute `engine_version_actual`, , see Attribute Reference below.
 * @property finalSnapshotIdentifier Name of your final DB snapshot when this DB cluster is deleted. If omitted, no final snapshot will be made.
 * @property globalClusterIdentifier Global cluster identifier specified on `aws.rds.GlobalCluster`.
 * @property iamDatabaseAuthenticationEnabled Specifies whether or not mappings of AWS Identity and Access Management (IAM) accounts to database accounts is enabled. Please see [AWS Documentation](https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/UsingWithRDS.IAMDBAuth.html) for availability and limitations.
 * @property iamRoles List of ARNs for the IAM roles to associate to the RDS Cluster.
 * @property iops Amount of Provisioned IOPS (input/output operations per second) to be initially allocated for each DB instance in the Multi-AZ DB cluster. For information about valid Iops values, see [Amazon RDS Provisioned IOPS storage to improve performance](https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/CHAP_Storage.html#USER_PIOPS) in the Amazon RDS User Guide. (This setting is required to create a Multi-AZ DB cluster). Must be a multiple between .5 and 50 of the storage amount for the DB cluster.
 * @property kmsKeyId ARN for the KMS encryption key. When specifying `kms_key_id`, `storage_encrypted` needs to be set to true.
 * @property manageMasterUserPassword Set to true to allow RDS to manage the master user password in Secrets Manager. Cannot be set if `master_password` is provided.
 * @property masterPassword Password for the master DB user. Note that this may show up in logs, and it will be stored in the state file. Please refer to the [RDS Naming Constraints](http://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/CHAP_Limits.html#RDS_Limits.Constraints). Cannot be set if `manage_master_user_password` is set to `true`.
 * @property masterUserSecretKmsKeyId Amazon Web Services KMS key identifier is the key ARN, key ID, alias ARN, or alias name for the KMS key. To use a KMS key in a different Amazon Web Services account, specify the key ARN or alias ARN. If not specified, the default KMS key for your Amazon Web Services account is used.
 * @property masterUsername Username for the master DB user. Please refer to the [RDS Naming Constraints](http://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/CHAP_Limits.html#RDS_Limits.Constraints). This argument does not support in-place updates and cannot be changed during a restore from snapshot.
 * @property networkType Network type of the cluster. Valid values: `IPV4`, `DUAL`.
 * @property port Port on which the DB accepts connections
 * @property preferredBackupWindow Daily time range during which automated backups are created if automated backups are enabled using the BackupRetentionPeriod parameter.Time in UTC. Default: A 30-minute window selected at random from an 8-hour block of time per regionE.g., 04:00-09:00
 * @property preferredMaintenanceWindow Weekly time range during which system maintenance can occur, in (UTC) e.g., wed:04:00-wed:04:30
 * @property replicationSourceIdentifier ARN of a source DB cluster or DB instance if this DB cluster is to be created as a Read Replica. If DB Cluster is part of a Global Cluster, use the `lifecycle` configuration block `ignore_changes` argument to prevent this provider from showing differences for this argument instead of configuring this value.
 * @property restoreToPointInTime Nested attribute for [point in time restore](https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/USER_PIT.html). More details below.
 * @property s3Import
 * @property scalingConfiguration Nested attribute with scaling properties. Only valid when `engine_mode` is set to `serverless`. More details below.
 * @property serverlessv2ScalingConfiguration Nested attribute with scaling properties for ServerlessV2. Only valid when `engine_mode` is set to `provisioned`. More details below.
 * @property skipFinalSnapshot Determines whether a final DB snapshot is created before the DB cluster is deleted. If true is specified, no DB snapshot is created. If false is specified, a DB snapshot is created before the DB cluster is deleted, using the value from `final_snapshot_identifier`. Default is `false`.
 * @property snapshotIdentifier Specifies whether or not to create this cluster from a snapshot. You can use either the name or ARN when specifying a DB cluster snapshot, or the ARN when specifying a DB snapshot. Conflicts with `global_cluster_identifier`. Clusters cannot be restored from snapshot **and** joined to an existing global cluster in a single operation. See the [AWS documentation](https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/aurora-global-database-getting-started.html#aurora-global-database.use-snapshot) or the Global Cluster Restored From Snapshot example for instructions on building a global cluster starting with a snapshot.
 * @property sourceRegion The source region for an encrypted replica DB cluster.
 * @property storageEncrypted Specifies whether the DB cluster is encrypted. The default is `false` for `provisioned` `engine_mode` and `true` for `serverless` `engine_mode`. When restoring an unencrypted `snapshot_identifier`, the `kms_key_id` argument must be provided to encrypt the restored cluster. The provider will only perform drift detection if a configuration value is provided.
 * @property storageType (Forces new for Multi-AZ DB clusters) Specifies the storage type to be associated with the DB cluster. For Aurora DB clusters, `storage_type` modifications can be done in-place. For Multi-AZ DB Clusters, the `iops` argument must also be set. Valid values are: `""`, `aurora-iopt1` (Aurora DB Clusters); `io1` (Multi-AZ DB Clusters). Default: `""` (Aurora DB Clusters); `io1` (Multi-AZ DB Clusters).
 * @property tags A map of tags to assign to the DB cluster. If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
 * @property vpcSecurityGroupIds List of VPC security groups to associate with the Cluster
 */
public data class ClusterArgs(
    public val allocatedStorage: Output<Int>? = null,
    public val allowMajorVersionUpgrade: Output<Boolean>? = null,
    public val applyImmediately: Output<Boolean>? = null,
    public val availabilityZones: Output<List<String>>? = null,
    public val backtrackWindow: Output<Int>? = null,
    public val backupRetentionPeriod: Output<Int>? = null,
    public val clusterIdentifier: Output<String>? = null,
    public val clusterIdentifierPrefix: Output<String>? = null,
    public val clusterMembers: Output<List<String>>? = null,
    public val copyTagsToSnapshot: Output<Boolean>? = null,
    public val databaseName: Output<String>? = null,
    public val dbClusterInstanceClass: Output<String>? = null,
    public val dbClusterParameterGroupName: Output<String>? = null,
    public val dbInstanceParameterGroupName: Output<String>? = null,
    public val dbSubnetGroupName: Output<String>? = null,
    public val dbSystemId: Output<String>? = null,
    public val deleteAutomatedBackups: Output<Boolean>? = null,
    public val deletionProtection: Output<Boolean>? = null,
    public val enableGlobalWriteForwarding: Output<Boolean>? = null,
    public val enableHttpEndpoint: Output<Boolean>? = null,
    public val enabledCloudwatchLogsExports: Output<List<String>>? = null,
    public val engine: Output<Either<String, EngineType>>? = null,
    public val engineMode: Output<Either<String, EngineMode>>? = null,
    public val engineVersion: Output<String>? = null,
    public val finalSnapshotIdentifier: Output<String>? = null,
    public val globalClusterIdentifier: Output<String>? = null,
    public val iamDatabaseAuthenticationEnabled: Output<Boolean>? = null,
    public val iamRoles: Output<List<String>>? = null,
    public val iops: Output<Int>? = null,
    public val kmsKeyId: Output<String>? = null,
    public val manageMasterUserPassword: Output<Boolean>? = null,
    public val masterPassword: Output<String>? = null,
    public val masterUserSecretKmsKeyId: Output<String>? = null,
    public val masterUsername: Output<String>? = null,
    public val networkType: Output<String>? = null,
    public val port: Output<Int>? = null,
    public val preferredBackupWindow: Output<String>? = null,
    public val preferredMaintenanceWindow: Output<String>? = null,
    public val replicationSourceIdentifier: Output<String>? = null,
    public val restoreToPointInTime: Output<ClusterRestoreToPointInTimeArgs>? = null,
    public val s3Import: Output<ClusterS3ImportArgs>? = null,
    public val scalingConfiguration: Output<ClusterScalingConfigurationArgs>? = null,
    public val serverlessv2ScalingConfiguration: Output<ClusterServerlessv2ScalingConfigurationArgs>? =
        null,
    public val skipFinalSnapshot: Output<Boolean>? = null,
    public val snapshotIdentifier: Output<String>? = null,
    public val sourceRegion: Output<String>? = null,
    public val storageEncrypted: Output<Boolean>? = null,
    public val storageType: Output<String>? = null,
    public val tags: Output<Map<String, String>>? = null,
    public val vpcSecurityGroupIds: Output<List<String>>? = null,
) : ConvertibleToJava<com.pulumi.aws.rds.ClusterArgs> {
    override fun toJava(): com.pulumi.aws.rds.ClusterArgs = com.pulumi.aws.rds.ClusterArgs.builder()
        .allocatedStorage(allocatedStorage?.applyValue({ args0 -> args0 }))
        .allowMajorVersionUpgrade(allowMajorVersionUpgrade?.applyValue({ args0 -> args0 }))
        .applyImmediately(applyImmediately?.applyValue({ args0 -> args0 }))
        .availabilityZones(availabilityZones?.applyValue({ args0 -> args0.map({ args0 -> args0 }) }))
        .backtrackWindow(backtrackWindow?.applyValue({ args0 -> args0 }))
        .backupRetentionPeriod(backupRetentionPeriod?.applyValue({ args0 -> args0 }))
        .clusterIdentifier(clusterIdentifier?.applyValue({ args0 -> args0 }))
        .clusterIdentifierPrefix(clusterIdentifierPrefix?.applyValue({ args0 -> args0 }))
        .clusterMembers(clusterMembers?.applyValue({ args0 -> args0.map({ args0 -> args0 }) }))
        .copyTagsToSnapshot(copyTagsToSnapshot?.applyValue({ args0 -> args0 }))
        .databaseName(databaseName?.applyValue({ args0 -> args0 }))
        .dbClusterInstanceClass(dbClusterInstanceClass?.applyValue({ args0 -> args0 }))
        .dbClusterParameterGroupName(dbClusterParameterGroupName?.applyValue({ args0 -> args0 }))
        .dbInstanceParameterGroupName(dbInstanceParameterGroupName?.applyValue({ args0 -> args0 }))
        .dbSubnetGroupName(dbSubnetGroupName?.applyValue({ args0 -> args0 }))
        .dbSystemId(dbSystemId?.applyValue({ args0 -> args0 }))
        .deleteAutomatedBackups(deleteAutomatedBackups?.applyValue({ args0 -> args0 }))
        .deletionProtection(deletionProtection?.applyValue({ args0 -> args0 }))
        .enableGlobalWriteForwarding(enableGlobalWriteForwarding?.applyValue({ args0 -> args0 }))
        .enableHttpEndpoint(enableHttpEndpoint?.applyValue({ args0 -> args0 }))
        .enabledCloudwatchLogsExports(
            enabledCloudwatchLogsExports?.applyValue({ args0 ->
                args0.map({ args0 -> args0 })
            }),
        )
        .engine(
            engine?.applyValue({ args0 ->
                args0.transform({ args0 -> args0 }, { args0 ->
                    args0.let({ args0 -> args0.toJava() })
                })
            }),
        )
        .engineMode(
            engineMode?.applyValue({ args0 ->
                args0.transform({ args0 -> args0 }, { args0 ->
                    args0.let({ args0 -> args0.toJava() })
                })
            }),
        )
        .engineVersion(engineVersion?.applyValue({ args0 -> args0 }))
        .finalSnapshotIdentifier(finalSnapshotIdentifier?.applyValue({ args0 -> args0 }))
        .globalClusterIdentifier(globalClusterIdentifier?.applyValue({ args0 -> args0 }))
        .iamDatabaseAuthenticationEnabled(iamDatabaseAuthenticationEnabled?.applyValue({ args0 -> args0 }))
        .iamRoles(iamRoles?.applyValue({ args0 -> args0.map({ args0 -> args0 }) }))
        .iops(iops?.applyValue({ args0 -> args0 }))
        .kmsKeyId(kmsKeyId?.applyValue({ args0 -> args0 }))
        .manageMasterUserPassword(manageMasterUserPassword?.applyValue({ args0 -> args0 }))
        .masterPassword(masterPassword?.applyValue({ args0 -> args0 }))
        .masterUserSecretKmsKeyId(masterUserSecretKmsKeyId?.applyValue({ args0 -> args0 }))
        .masterUsername(masterUsername?.applyValue({ args0 -> args0 }))
        .networkType(networkType?.applyValue({ args0 -> args0 }))
        .port(port?.applyValue({ args0 -> args0 }))
        .preferredBackupWindow(preferredBackupWindow?.applyValue({ args0 -> args0 }))
        .preferredMaintenanceWindow(preferredMaintenanceWindow?.applyValue({ args0 -> args0 }))
        .replicationSourceIdentifier(replicationSourceIdentifier?.applyValue({ args0 -> args0 }))
        .restoreToPointInTime(
            restoreToPointInTime?.applyValue({ args0 ->
                args0.let({ args0 ->
                    args0.toJava()
                })
            }),
        )
        .s3Import(s3Import?.applyValue({ args0 -> args0.let({ args0 -> args0.toJava() }) }))
        .scalingConfiguration(
            scalingConfiguration?.applyValue({ args0 ->
                args0.let({ args0 ->
                    args0.toJava()
                })
            }),
        )
        .serverlessv2ScalingConfiguration(
            serverlessv2ScalingConfiguration?.applyValue({ args0 ->
                args0.let({ args0 -> args0.toJava() })
            }),
        )
        .skipFinalSnapshot(skipFinalSnapshot?.applyValue({ args0 -> args0 }))
        .snapshotIdentifier(snapshotIdentifier?.applyValue({ args0 -> args0 }))
        .sourceRegion(sourceRegion?.applyValue({ args0 -> args0 }))
        .storageEncrypted(storageEncrypted?.applyValue({ args0 -> args0 }))
        .storageType(storageType?.applyValue({ args0 -> args0 }))
        .tags(tags?.applyValue({ args0 -> args0.map({ args0 -> args0.key.to(args0.value) }).toMap() }))
        .vpcSecurityGroupIds(
            vpcSecurityGroupIds?.applyValue({ args0 ->
                args0.map({ args0 ->
                    args0
                })
            }),
        ).build()
}

/**
 * Builder for [ClusterArgs].
 */
@PulumiTagMarker
public class ClusterArgsBuilder internal constructor() {
    private var allocatedStorage: Output<Int>? = null

    private var allowMajorVersionUpgrade: Output<Boolean>? = null

    private var applyImmediately: Output<Boolean>? = null

    private var availabilityZones: Output<List<String>>? = null

    private var backtrackWindow: Output<Int>? = null

    private var backupRetentionPeriod: Output<Int>? = null

    private var clusterIdentifier: Output<String>? = null

    private var clusterIdentifierPrefix: Output<String>? = null

    private var clusterMembers: Output<List<String>>? = null

    private var copyTagsToSnapshot: Output<Boolean>? = null

    private var databaseName: Output<String>? = null

    private var dbClusterInstanceClass: Output<String>? = null

    private var dbClusterParameterGroupName: Output<String>? = null

    private var dbInstanceParameterGroupName: Output<String>? = null

    private var dbSubnetGroupName: Output<String>? = null

    private var dbSystemId: Output<String>? = null

    private var deleteAutomatedBackups: Output<Boolean>? = null

    private var deletionProtection: Output<Boolean>? = null

    private var enableGlobalWriteForwarding: Output<Boolean>? = null

    private var enableHttpEndpoint: Output<Boolean>? = null

    private var enabledCloudwatchLogsExports: Output<List<String>>? = null

    private var engine: Output<Either<String, EngineType>>? = null

    private var engineMode: Output<Either<String, EngineMode>>? = null

    private var engineVersion: Output<String>? = null

    private var finalSnapshotIdentifier: Output<String>? = null

    private var globalClusterIdentifier: Output<String>? = null

    private var iamDatabaseAuthenticationEnabled: Output<Boolean>? = null

    private var iamRoles: Output<List<String>>? = null

    private var iops: Output<Int>? = null

    private var kmsKeyId: Output<String>? = null

    private var manageMasterUserPassword: Output<Boolean>? = null

    private var masterPassword: Output<String>? = null

    private var masterUserSecretKmsKeyId: Output<String>? = null

    private var masterUsername: Output<String>? = null

    private var networkType: Output<String>? = null

    private var port: Output<Int>? = null

    private var preferredBackupWindow: Output<String>? = null

    private var preferredMaintenanceWindow: Output<String>? = null

    private var replicationSourceIdentifier: Output<String>? = null

    private var restoreToPointInTime: Output<ClusterRestoreToPointInTimeArgs>? = null

    private var s3Import: Output<ClusterS3ImportArgs>? = null

    private var scalingConfiguration: Output<ClusterScalingConfigurationArgs>? = null

    private var serverlessv2ScalingConfiguration: Output<ClusterServerlessv2ScalingConfigurationArgs>? =
        null

    private var skipFinalSnapshot: Output<Boolean>? = null

    private var snapshotIdentifier: Output<String>? = null

    private var sourceRegion: Output<String>? = null

    private var storageEncrypted: Output<Boolean>? = null

    private var storageType: Output<String>? = null

    private var tags: Output<Map<String, String>>? = null

    private var vpcSecurityGroupIds: Output<List<String>>? = null

    /**
     * @param value The amount of storage in gibibytes (GiB) to allocate to each DB instance in the Multi-AZ DB cluster.
     */
    @JvmName("kdpcdccfvuwlwetw")
    public suspend fun allocatedStorage(`value`: Output<Int>) {
        this.allocatedStorage = value
    }

    /**
     * @param value Enable to allow major engine version upgrades when changing engine versions. Defaults to `false`.
     */
    @JvmName("swjvavawjhjfjyqx")
    public suspend fun allowMajorVersionUpgrade(`value`: Output<Boolean>) {
        this.allowMajorVersionUpgrade = value
    }

    /**
     * @param value Specifies whether any cluster modifications are applied immediately, or during the next maintenance window. Default is `false`. See [Amazon RDS Documentation for more information&#46;](https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/Overview.DBInstance.Modifying.html)
     */
    @JvmName("neeyuaywqkhrjrub")
    public suspend fun applyImmediately(`value`: Output<Boolean>) {
        this.applyImmediately = value
    }

    /**
     * @param value List of EC2 Availability Zones for the DB cluster storage where DB cluster instances can be created.
     * RDS automatically assigns 3 AZs if less than 3 AZs are configured, which will show as a difference requiring resource recreation next pulumi up.
     * We recommend specifying 3 AZs or using the `lifecycle` configuration block `ignore_changes` argument if necessary.
     * A maximum of 3 AZs can be configured.
     */
    @JvmName("hyjdhaaftfxsefie")
    public suspend fun availabilityZones(`value`: Output<List<String>>) {
        this.availabilityZones = value
    }

    @JvmName("txlinidyliqaimdn")
    public suspend fun availabilityZones(vararg values: Output<String>) {
        this.availabilityZones = Output.all(values.asList())
    }

    /**
     * @param values List of EC2 Availability Zones for the DB cluster storage where DB cluster instances can be created.
     * RDS automatically assigns 3 AZs if less than 3 AZs are configured, which will show as a difference requiring resource recreation next pulumi up.
     * We recommend specifying 3 AZs or using the `lifecycle` configuration block `ignore_changes` argument if necessary.
     * A maximum of 3 AZs can be configured.
     */
    @JvmName("sutrtwdsxstcxuey")
    public suspend fun availabilityZones(values: List<Output<String>>) {
        this.availabilityZones = Output.all(values)
    }

    /**
     * @param value Target backtrack window, in seconds. Only available for `aurora` and `aurora-mysql` engines currently. To disable backtracking, set this value to `0`. Defaults to `0`. Must be between `0` and `259200` (72 hours)
     */
    @JvmName("bubkbunenadwkxhk")
    public suspend fun backtrackWindow(`value`: Output<Int>) {
        this.backtrackWindow = value
    }

    /**
     * @param value Days to retain backups for. Default `1`
     */
    @JvmName("nticcpblrjpclome")
    public suspend fun backupRetentionPeriod(`value`: Output<Int>) {
        this.backupRetentionPeriod = value
    }

    /**
     * @param value The cluster identifier. If omitted, this provider will assign a random, unique identifier.
     */
    @JvmName("nhlbtxdddwxhcexe")
    public suspend fun clusterIdentifier(`value`: Output<String>) {
        this.clusterIdentifier = value
    }

    /**
     * @param value Creates a unique cluster identifier beginning with the specified prefix. Conflicts with `cluster_identifier`.
     */
    @JvmName("qerylhkelurghhwr")
    public suspend fun clusterIdentifierPrefix(`value`: Output<String>) {
        this.clusterIdentifierPrefix = value
    }

    /**
     * @param value List of RDS Instances that are a part of this cluster
     */
    @JvmName("knscberdbxfivqho")
    public suspend fun clusterMembers(`value`: Output<List<String>>) {
        this.clusterMembers = value
    }

    @JvmName("fpsirnbjfriasiwp")
    public suspend fun clusterMembers(vararg values: Output<String>) {
        this.clusterMembers = Output.all(values.asList())
    }

    /**
     * @param values List of RDS Instances that are a part of this cluster
     */
    @JvmName("qmvkhtoxhflqxbdg")
    public suspend fun clusterMembers(values: List<Output<String>>) {
        this.clusterMembers = Output.all(values)
    }

    /**
     * @param value Copy all Cluster `tags` to snapshots. Default is `false`.
     */
    @JvmName("thayvwxwabuqjiyj")
    public suspend fun copyTagsToSnapshot(`value`: Output<Boolean>) {
        this.copyTagsToSnapshot = value
    }

    /**
     * @param value Name for an automatically created database on cluster creation. There are different naming restrictions per database engine: [RDS Naming Constraints](http://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/CHAP_Limits.html#RDS_Limits.Constraints)
     */
    @JvmName("ppbdqippdubxktvk")
    public suspend fun databaseName(`value`: Output<String>) {
        this.databaseName = value
    }

    /**
     * @param value The compute and memory capacity of each DB instance in the Multi-AZ DB cluster, for example `db.m6g.xlarge`. Not all DB instance classes are available in all AWS Regions, or for all database engines. For the full list of DB instance classes and availability for your engine, see [DB instance class](https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/Concepts.DBInstanceClass.html) in the Amazon RDS User Guide.
     */
    @JvmName("xbbpukkgpkobqnan")
    public suspend fun dbClusterInstanceClass(`value`: Output<String>) {
        this.dbClusterInstanceClass = value
    }

    /**
     * @param value A cluster parameter group to associate with the cluster.
     */
    @JvmName("kknleeuyrmsqvvrn")
    public suspend fun dbClusterParameterGroupName(`value`: Output<String>) {
        this.dbClusterParameterGroupName = value
    }

    /**
     * @param value Instance parameter group to associate with all instances of the DB cluster. The `db_instance_parameter_group_name` parameter is only valid in combination with the `allow_major_version_upgrade` parameter.
     */
    @JvmName("xvwvbkviixmbfpus")
    public suspend fun dbInstanceParameterGroupName(`value`: Output<String>) {
        this.dbInstanceParameterGroupName = value
    }

    /**
     * @param value DB subnet group to associate with this DB cluster.
     * **NOTE:** This must match the `db_subnet_group_name` specified on every `aws.rds.ClusterInstance` in the cluster.
     */
    @JvmName("rtucgfosfwjndywb")
    public suspend fun dbSubnetGroupName(`value`: Output<String>) {
        this.dbSubnetGroupName = value
    }

    /**
     * @param value For use with RDS Custom.
     */
    @JvmName("ymrdhrchkksxbqpt")
    public suspend fun dbSystemId(`value`: Output<String>) {
        this.dbSystemId = value
    }

    /**
     * @param value Specifies whether to remove automated backups immediately after the DB cluster is deleted. Default is `true`.
     */
    @JvmName("emruykqylxbdbbyy")
    public suspend fun deleteAutomatedBackups(`value`: Output<Boolean>) {
        this.deleteAutomatedBackups = value
    }

    /**
     * @param value If the DB cluster should have deletion protection enabled.
     * The database can't be deleted when this value is set to `true`.
     * The default is `false`.
     */
    @JvmName("bgmemlnnniljunej")
    public suspend fun deletionProtection(`value`: Output<Boolean>) {
        this.deletionProtection = value
    }

    /**
     * @param value Whether cluster should forward writes to an associated global cluster. Applied to secondary clusters to enable them to forward writes to an `aws.rds.GlobalCluster`'s primary cluster. See the [Aurora Userguide documentation](https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/aurora-global-database-write-forwarding.html) for more information.
     */
    @JvmName("motsalnsmwornufb")
    public suspend fun enableGlobalWriteForwarding(`value`: Output<Boolean>) {
        this.enableGlobalWriteForwarding = value
    }

    /**
     * @param value Enable HTTP endpoint (data API). Only valid when `engine_mode` is set to `serverless`.
     */
    @JvmName("rupqsrrfwtmrkaip")
    public suspend fun enableHttpEndpoint(`value`: Output<Boolean>) {
        this.enableHttpEndpoint = value
    }

    /**
     * @param value Set of log types to export to cloudwatch. If omitted, no logs will be exported. The following log types are supported: `audit`, `error`, `general`, `slowquery`, `postgresql` (PostgreSQL).
     */
    @JvmName("uydbvoxtnalqobsi")
    public suspend fun enabledCloudwatchLogsExports(`value`: Output<List<String>>) {
        this.enabledCloudwatchLogsExports = value
    }

    @JvmName("lkdvdrkrnafhypgw")
    public suspend fun enabledCloudwatchLogsExports(vararg values: Output<String>) {
        this.enabledCloudwatchLogsExports = Output.all(values.asList())
    }

    /**
     * @param values Set of log types to export to cloudwatch. If omitted, no logs will be exported. The following log types are supported: `audit`, `error`, `general`, `slowquery`, `postgresql` (PostgreSQL).
     */
    @JvmName("pundvhvdyjkyriwu")
    public suspend fun enabledCloudwatchLogsExports(values: List<Output<String>>) {
        this.enabledCloudwatchLogsExports = Output.all(values)
    }

    /**
     * @param value Name of the database engine to be used for this DB cluster. Valid Values: `aurora-mysql`, `aurora-postgresql`, `mysql`, `postgres`. (Note that `mysql` and `postgres` are Multi-AZ RDS clusters).
     */
    @JvmName("trxvrjmmywpwmlye")
    public suspend fun engine(`value`: Output<Either<String, EngineType>>) {
        this.engine = value
    }

    /**
     * @param value Database engine mode. Valid values: `global` (only valid for Aurora MySQL 1.21 and earlier), `parallelquery`, `provisioned`, `serverless`. Defaults to: `provisioned`. See the [RDS User Guide](https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/aurora-serverless.html) for limitations when using `serverless`.
     */
    @JvmName("oowjpqkpyxmarijj")
    public suspend fun engineMode(`value`: Output<Either<String, EngineMode>>) {
        this.engineMode = value
    }

    /**
     * @param value Database engine version. Updating this argument results in an outage. See the [Aurora MySQL](https://docs&#46;aws&#46;amazon&#46;com/AmazonRDS/latest/AuroraUserGuide/AuroraMySQL&#46;Updates&#46;html) and [Aurora Postgres](https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/AuroraPostgreSQL.Updates.html) documentation for your configured engine to determine this value, or by running `aws rds describe-db-engine-versions`. For example with Aurora MySQL 2, a potential value for this argument is `5.7.mysql_aurora.2.03.2`. The value can contain a partial version where supported by the API. The actual engine version used is returned in the attribute `engine_version_actual`, , see Attribute Reference below.
     */
    @JvmName("ifswadpnqtrvwnvt")
    public suspend fun engineVersion(`value`: Output<String>) {
        this.engineVersion = value
    }

    /**
     * @param value Name of your final DB snapshot when this DB cluster is deleted. If omitted, no final snapshot will be made.
     */
    @JvmName("uqcwhyriqnuadrxv")
    public suspend fun finalSnapshotIdentifier(`value`: Output<String>) {
        this.finalSnapshotIdentifier = value
    }

    /**
     * @param value Global cluster identifier specified on `aws.rds.GlobalCluster`.
     */
    @JvmName("fknkbpvhjooqlmnh")
    public suspend fun globalClusterIdentifier(`value`: Output<String>) {
        this.globalClusterIdentifier = value
    }

    /**
     * @param value Specifies whether or not mappings of AWS Identity and Access Management (IAM) accounts to database accounts is enabled. Please see [AWS Documentation](https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/UsingWithRDS.IAMDBAuth.html) for availability and limitations.
     */
    @JvmName("kgasxsqbvwrdquip")
    public suspend fun iamDatabaseAuthenticationEnabled(`value`: Output<Boolean>) {
        this.iamDatabaseAuthenticationEnabled = value
    }

    /**
     * @param value List of ARNs for the IAM roles to associate to the RDS Cluster.
     */
    @JvmName("kcwsyjkiqwduaski")
    public suspend fun iamRoles(`value`: Output<List<String>>) {
        this.iamRoles = value
    }

    @JvmName("udqydfrusekgrqku")
    public suspend fun iamRoles(vararg values: Output<String>) {
        this.iamRoles = Output.all(values.asList())
    }

    /**
     * @param values List of ARNs for the IAM roles to associate to the RDS Cluster.
     */
    @JvmName("vcijaelpvldkmqko")
    public suspend fun iamRoles(values: List<Output<String>>) {
        this.iamRoles = Output.all(values)
    }

    /**
     * @param value Amount of Provisioned IOPS (input/output operations per second) to be initially allocated for each DB instance in the Multi-AZ DB cluster. For information about valid Iops values, see [Amazon RDS Provisioned IOPS storage to improve performance](https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/CHAP_Storage.html#USER_PIOPS) in the Amazon RDS User Guide. (This setting is required to create a Multi-AZ DB cluster). Must be a multiple between .5 and 50 of the storage amount for the DB cluster.
     */
    @JvmName("hnpugxmbybavdiyw")
    public suspend fun iops(`value`: Output<Int>) {
        this.iops = value
    }

    /**
     * @param value ARN for the KMS encryption key. When specifying `kms_key_id`, `storage_encrypted` needs to be set to true.
     */
    @JvmName("erhqsbwldgiawuoo")
    public suspend fun kmsKeyId(`value`: Output<String>) {
        this.kmsKeyId = value
    }

    /**
     * @param value Set to true to allow RDS to manage the master user password in Secrets Manager. Cannot be set if `master_password` is provided.
     */
    @JvmName("whvsjevffkvkfisl")
    public suspend fun manageMasterUserPassword(`value`: Output<Boolean>) {
        this.manageMasterUserPassword = value
    }

    /**
     * @param value Password for the master DB user. Note that this may show up in logs, and it will be stored in the state file. Please refer to the [RDS Naming Constraints](http://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/CHAP_Limits.html#RDS_Limits.Constraints). Cannot be set if `manage_master_user_password` is set to `true`.
     */
    @JvmName("yskthhkvxugloxdf")
    public suspend fun masterPassword(`value`: Output<String>) {
        this.masterPassword = value
    }

    /**
     * @param value Amazon Web Services KMS key identifier is the key ARN, key ID, alias ARN, or alias name for the KMS key. To use a KMS key in a different Amazon Web Services account, specify the key ARN or alias ARN. If not specified, the default KMS key for your Amazon Web Services account is used.
     */
    @JvmName("omcubwofmxamfyen")
    public suspend fun masterUserSecretKmsKeyId(`value`: Output<String>) {
        this.masterUserSecretKmsKeyId = value
    }

    /**
     * @param value Username for the master DB user. Please refer to the [RDS Naming Constraints](http://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/CHAP_Limits.html#RDS_Limits.Constraints). This argument does not support in-place updates and cannot be changed during a restore from snapshot.
     */
    @JvmName("mfsxjyjdgwrvrege")
    public suspend fun masterUsername(`value`: Output<String>) {
        this.masterUsername = value
    }

    /**
     * @param value Network type of the cluster. Valid values: `IPV4`, `DUAL`.
     */
    @JvmName("phxrumglnkfephrp")
    public suspend fun networkType(`value`: Output<String>) {
        this.networkType = value
    }

    /**
     * @param value Port on which the DB accepts connections
     */
    @JvmName("hmalbnchqosiahgc")
    public suspend fun port(`value`: Output<Int>) {
        this.port = value
    }

    /**
     * @param value Daily time range during which automated backups are created if automated backups are enabled using the BackupRetentionPeriod parameter.Time in UTC. Default: A 30-minute window selected at random from an 8-hour block of time per regionE.g., 04:00-09:00
     */
    @JvmName("cihfkndcgjvktorh")
    public suspend fun preferredBackupWindow(`value`: Output<String>) {
        this.preferredBackupWindow = value
    }

    /**
     * @param value Weekly time range during which system maintenance can occur, in (UTC) e.g., wed:04:00-wed:04:30
     */
    @JvmName("xvslggpkledqquoj")
    public suspend fun preferredMaintenanceWindow(`value`: Output<String>) {
        this.preferredMaintenanceWindow = value
    }

    /**
     * @param value ARN of a source DB cluster or DB instance if this DB cluster is to be created as a Read Replica. If DB Cluster is part of a Global Cluster, use the `lifecycle` configuration block `ignore_changes` argument to prevent this provider from showing differences for this argument instead of configuring this value.
     */
    @JvmName("koffxjnxyiwctppk")
    public suspend fun replicationSourceIdentifier(`value`: Output<String>) {
        this.replicationSourceIdentifier = value
    }

    /**
     * @param value Nested attribute for [point in time restore](https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/USER_PIT.html). More details below.
     */
    @JvmName("immpqvlcbvqupwil")
    public suspend fun restoreToPointInTime(`value`: Output<ClusterRestoreToPointInTimeArgs>) {
        this.restoreToPointInTime = value
    }

    /**
     * @param value
     */
    @JvmName("sjydqeqtralvbhgv")
    public suspend fun s3Import(`value`: Output<ClusterS3ImportArgs>) {
        this.s3Import = value
    }

    /**
     * @param value Nested attribute with scaling properties. Only valid when `engine_mode` is set to `serverless`. More details below.
     */
    @JvmName("ynxxyrbsctmlekhb")
    public suspend fun scalingConfiguration(`value`: Output<ClusterScalingConfigurationArgs>) {
        this.scalingConfiguration = value
    }

    /**
     * @param value Nested attribute with scaling properties for ServerlessV2. Only valid when `engine_mode` is set to `provisioned`. More details below.
     */
    @JvmName("sjklprbkmvsafqei")
    public suspend
    fun serverlessv2ScalingConfiguration(`value`: Output<ClusterServerlessv2ScalingConfigurationArgs>) {
        this.serverlessv2ScalingConfiguration = value
    }

    /**
     * @param value Determines whether a final DB snapshot is created before the DB cluster is deleted. If true is specified, no DB snapshot is created. If false is specified, a DB snapshot is created before the DB cluster is deleted, using the value from `final_snapshot_identifier`. Default is `false`.
     */
    @JvmName("ijfiirryuqotthsr")
    public suspend fun skipFinalSnapshot(`value`: Output<Boolean>) {
        this.skipFinalSnapshot = value
    }

    /**
     * @param value Specifies whether or not to create this cluster from a snapshot. You can use either the name or ARN when specifying a DB cluster snapshot, or the ARN when specifying a DB snapshot. Conflicts with `global_cluster_identifier`. Clusters cannot be restored from snapshot **and** joined to an existing global cluster in a single operation. See the [AWS documentation](https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/aurora-global-database-getting-started.html#aurora-global-database.use-snapshot) or the Global Cluster Restored From Snapshot example for instructions on building a global cluster starting with a snapshot.
     */
    @JvmName("srudljowrgjdrfvf")
    public suspend fun snapshotIdentifier(`value`: Output<String>) {
        this.snapshotIdentifier = value
    }

    /**
     * @param value The source region for an encrypted replica DB cluster.
     */
    @JvmName("ypnsyjajockfwybj")
    public suspend fun sourceRegion(`value`: Output<String>) {
        this.sourceRegion = value
    }

    /**
     * @param value Specifies whether the DB cluster is encrypted. The default is `false` for `provisioned` `engine_mode` and `true` for `serverless` `engine_mode`. When restoring an unencrypted `snapshot_identifier`, the `kms_key_id` argument must be provided to encrypt the restored cluster. The provider will only perform drift detection if a configuration value is provided.
     */
    @JvmName("ysijhjmhvqeiguwv")
    public suspend fun storageEncrypted(`value`: Output<Boolean>) {
        this.storageEncrypted = value
    }

    /**
     * @param value (Forces new for Multi-AZ DB clusters) Specifies the storage type to be associated with the DB cluster. For Aurora DB clusters, `storage_type` modifications can be done in-place. For Multi-AZ DB Clusters, the `iops` argument must also be set. Valid values are: `""`, `aurora-iopt1` (Aurora DB Clusters); `io1` (Multi-AZ DB Clusters). Default: `""` (Aurora DB Clusters); `io1` (Multi-AZ DB Clusters).
     */
    @JvmName("mmipmmodrtffxeml")
    public suspend fun storageType(`value`: Output<String>) {
        this.storageType = value
    }

    /**
     * @param value A map of tags to assign to the DB cluster. If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
     */
    @JvmName("rkyrptofrnbgsfke")
    public suspend fun tags(`value`: Output<Map<String, String>>) {
        this.tags = value
    }

    /**
     * @param value List of VPC security groups to associate with the Cluster
     */
    @JvmName("cgrxukeobigtgqma")
    public suspend fun vpcSecurityGroupIds(`value`: Output<List<String>>) {
        this.vpcSecurityGroupIds = value
    }

    @JvmName("kwktustfxxbexqhl")
    public suspend fun vpcSecurityGroupIds(vararg values: Output<String>) {
        this.vpcSecurityGroupIds = Output.all(values.asList())
    }

    /**
     * @param values List of VPC security groups to associate with the Cluster
     */
    @JvmName("iajvqihwmiohkwdv")
    public suspend fun vpcSecurityGroupIds(values: List<Output<String>>) {
        this.vpcSecurityGroupIds = Output.all(values)
    }

    /**
     * @param value The amount of storage in gibibytes (GiB) to allocate to each DB instance in the Multi-AZ DB cluster.
     */
    @JvmName("cgwiuybppvdagkyw")
    public suspend fun allocatedStorage(`value`: Int?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.allocatedStorage = mapped
    }

    /**
     * @param value Enable to allow major engine version upgrades when changing engine versions. Defaults to `false`.
     */
    @JvmName("cmahdfurrvpclvhw")
    public suspend fun allowMajorVersionUpgrade(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.allowMajorVersionUpgrade = mapped
    }

    /**
     * @param value Specifies whether any cluster modifications are applied immediately, or during the next maintenance window. Default is `false`. See [Amazon RDS Documentation for more information&#46;](https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/Overview.DBInstance.Modifying.html)
     */
    @JvmName("uscaipacneuiuhvr")
    public suspend fun applyImmediately(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.applyImmediately = mapped
    }

    /**
     * @param value List of EC2 Availability Zones for the DB cluster storage where DB cluster instances can be created.
     * RDS automatically assigns 3 AZs if less than 3 AZs are configured, which will show as a difference requiring resource recreation next pulumi up.
     * We recommend specifying 3 AZs or using the `lifecycle` configuration block `ignore_changes` argument if necessary.
     * A maximum of 3 AZs can be configured.
     */
    @JvmName("xuarbwyhvbarhswc")
    public suspend fun availabilityZones(`value`: List<String>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.availabilityZones = mapped
    }

    /**
     * @param values List of EC2 Availability Zones for the DB cluster storage where DB cluster instances can be created.
     * RDS automatically assigns 3 AZs if less than 3 AZs are configured, which will show as a difference requiring resource recreation next pulumi up.
     * We recommend specifying 3 AZs or using the `lifecycle` configuration block `ignore_changes` argument if necessary.
     * A maximum of 3 AZs can be configured.
     */
    @JvmName("wncqxqqirkeemefo")
    public suspend fun availabilityZones(vararg values: String) {
        val toBeMapped = values.toList()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.availabilityZones = mapped
    }

    /**
     * @param value Target backtrack window, in seconds. Only available for `aurora` and `aurora-mysql` engines currently. To disable backtracking, set this value to `0`. Defaults to `0`. Must be between `0` and `259200` (72 hours)
     */
    @JvmName("nycdmrchjbllmhud")
    public suspend fun backtrackWindow(`value`: Int?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.backtrackWindow = mapped
    }

    /**
     * @param value Days to retain backups for. Default `1`
     */
    @JvmName("jxnxwdrgrcxcwvcc")
    public suspend fun backupRetentionPeriod(`value`: Int?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.backupRetentionPeriod = mapped
    }

    /**
     * @param value The cluster identifier. If omitted, this provider will assign a random, unique identifier.
     */
    @JvmName("onyvdpmotevrkqma")
    public suspend fun clusterIdentifier(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.clusterIdentifier = mapped
    }

    /**
     * @param value Creates a unique cluster identifier beginning with the specified prefix. Conflicts with `cluster_identifier`.
     */
    @JvmName("cagvndiasshnwuvi")
    public suspend fun clusterIdentifierPrefix(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.clusterIdentifierPrefix = mapped
    }

    /**
     * @param value List of RDS Instances that are a part of this cluster
     */
    @JvmName("ssusyuodpusjhgnk")
    public suspend fun clusterMembers(`value`: List<String>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.clusterMembers = mapped
    }

    /**
     * @param values List of RDS Instances that are a part of this cluster
     */
    @JvmName("mwxrypijvixykagw")
    public suspend fun clusterMembers(vararg values: String) {
        val toBeMapped = values.toList()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.clusterMembers = mapped
    }

    /**
     * @param value Copy all Cluster `tags` to snapshots. Default is `false`.
     */
    @JvmName("oogyatdsfvwibinx")
    public suspend fun copyTagsToSnapshot(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.copyTagsToSnapshot = mapped
    }

    /**
     * @param value Name for an automatically created database on cluster creation. There are different naming restrictions per database engine: [RDS Naming Constraints](http://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/CHAP_Limits.html#RDS_Limits.Constraints)
     */
    @JvmName("yssrgbjdrcqjkwox")
    public suspend fun databaseName(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.databaseName = mapped
    }

    /**
     * @param value The compute and memory capacity of each DB instance in the Multi-AZ DB cluster, for example `db.m6g.xlarge`. Not all DB instance classes are available in all AWS Regions, or for all database engines. For the full list of DB instance classes and availability for your engine, see [DB instance class](https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/Concepts.DBInstanceClass.html) in the Amazon RDS User Guide.
     */
    @JvmName("taybsqgkunsqqpba")
    public suspend fun dbClusterInstanceClass(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.dbClusterInstanceClass = mapped
    }

    /**
     * @param value A cluster parameter group to associate with the cluster.
     */
    @JvmName("xoagrpjqgmgkobco")
    public suspend fun dbClusterParameterGroupName(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.dbClusterParameterGroupName = mapped
    }

    /**
     * @param value Instance parameter group to associate with all instances of the DB cluster. The `db_instance_parameter_group_name` parameter is only valid in combination with the `allow_major_version_upgrade` parameter.
     */
    @JvmName("ufqtwplodumgkkto")
    public suspend fun dbInstanceParameterGroupName(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.dbInstanceParameterGroupName = mapped
    }

    /**
     * @param value DB subnet group to associate with this DB cluster.
     * **NOTE:** This must match the `db_subnet_group_name` specified on every `aws.rds.ClusterInstance` in the cluster.
     */
    @JvmName("gryvbdctscugcunq")
    public suspend fun dbSubnetGroupName(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.dbSubnetGroupName = mapped
    }

    /**
     * @param value For use with RDS Custom.
     */
    @JvmName("vjfttguungluuaag")
    public suspend fun dbSystemId(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.dbSystemId = mapped
    }

    /**
     * @param value Specifies whether to remove automated backups immediately after the DB cluster is deleted. Default is `true`.
     */
    @JvmName("rujemkvbeybmgiqy")
    public suspend fun deleteAutomatedBackups(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.deleteAutomatedBackups = mapped
    }

    /**
     * @param value If the DB cluster should have deletion protection enabled.
     * The database can't be deleted when this value is set to `true`.
     * The default is `false`.
     */
    @JvmName("gogtljdilmyjeqqy")
    public suspend fun deletionProtection(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.deletionProtection = mapped
    }

    /**
     * @param value Whether cluster should forward writes to an associated global cluster. Applied to secondary clusters to enable them to forward writes to an `aws.rds.GlobalCluster`'s primary cluster. See the [Aurora Userguide documentation](https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/aurora-global-database-write-forwarding.html) for more information.
     */
    @JvmName("vhgbmuwvabgfuhbf")
    public suspend fun enableGlobalWriteForwarding(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.enableGlobalWriteForwarding = mapped
    }

    /**
     * @param value Enable HTTP endpoint (data API). Only valid when `engine_mode` is set to `serverless`.
     */
    @JvmName("wpwltkuittihhrod")
    public suspend fun enableHttpEndpoint(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.enableHttpEndpoint = mapped
    }

    /**
     * @param value Set of log types to export to cloudwatch. If omitted, no logs will be exported. The following log types are supported: `audit`, `error`, `general`, `slowquery`, `postgresql` (PostgreSQL).
     */
    @JvmName("knqhwbyoosbmbgpu")
    public suspend fun enabledCloudwatchLogsExports(`value`: List<String>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.enabledCloudwatchLogsExports = mapped
    }

    /**
     * @param values Set of log types to export to cloudwatch. If omitted, no logs will be exported. The following log types are supported: `audit`, `error`, `general`, `slowquery`, `postgresql` (PostgreSQL).
     */
    @JvmName("klmxkjhrqmcjahlb")
    public suspend fun enabledCloudwatchLogsExports(vararg values: String) {
        val toBeMapped = values.toList()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.enabledCloudwatchLogsExports = mapped
    }

    /**
     * @param value Name of the database engine to be used for this DB cluster. Valid Values: `aurora-mysql`, `aurora-postgresql`, `mysql`, `postgres`. (Note that `mysql` and `postgres` are Multi-AZ RDS clusters).
     */
    @JvmName("rovsmgikpqlonhpo")
    public suspend fun engine(`value`: Either<String, EngineType>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.engine = mapped
    }

    /**
     * @param value Name of the database engine to be used for this DB cluster. Valid Values: `aurora-mysql`, `aurora-postgresql`, `mysql`, `postgres`. (Note that `mysql` and `postgres` are Multi-AZ RDS clusters).
     */
    @JvmName("qucmubjxltvfovhg")
    public fun engine(`value`: String) {
        val toBeMapped = Either.ofLeft<String, EngineType>(value)
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.engine = mapped
    }

    /**
     * @param value Name of the database engine to be used for this DB cluster. Valid Values: `aurora-mysql`, `aurora-postgresql`, `mysql`, `postgres`. (Note that `mysql` and `postgres` are Multi-AZ RDS clusters).
     */
    @JvmName("vmnshcfjcswirhxh")
    public fun engine(`value`: EngineType) {
        val toBeMapped = Either.ofRight<String, EngineType>(value)
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.engine = mapped
    }

    /**
     * @param value Database engine mode. Valid values: `global` (only valid for Aurora MySQL 1.21 and earlier), `parallelquery`, `provisioned`, `serverless`. Defaults to: `provisioned`. See the [RDS User Guide](https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/aurora-serverless.html) for limitations when using `serverless`.
     */
    @JvmName("fvgomvvwitwjbmwx")
    public suspend fun engineMode(`value`: Either<String, EngineMode>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.engineMode = mapped
    }

    /**
     * @param value Database engine mode. Valid values: `global` (only valid for Aurora MySQL 1.21 and earlier), `parallelquery`, `provisioned`, `serverless`. Defaults to: `provisioned`. See the [RDS User Guide](https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/aurora-serverless.html) for limitations when using `serverless`.
     */
    @JvmName("ddoukjthmmnanqkg")
    public fun engineMode(`value`: String) {
        val toBeMapped = Either.ofLeft<String, EngineMode>(value)
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.engineMode = mapped
    }

    /**
     * @param value Database engine mode. Valid values: `global` (only valid for Aurora MySQL 1.21 and earlier), `parallelquery`, `provisioned`, `serverless`. Defaults to: `provisioned`. See the [RDS User Guide](https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/aurora-serverless.html) for limitations when using `serverless`.
     */
    @JvmName("ydlxrotdgyqsbjbd")
    public fun engineMode(`value`: EngineMode) {
        val toBeMapped = Either.ofRight<String, EngineMode>(value)
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.engineMode = mapped
    }

    /**
     * @param value Database engine version. Updating this argument results in an outage. See the [Aurora MySQL](https://docs&#46;aws&#46;amazon&#46;com/AmazonRDS/latest/AuroraUserGuide/AuroraMySQL&#46;Updates&#46;html) and [Aurora Postgres](https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/AuroraPostgreSQL.Updates.html) documentation for your configured engine to determine this value, or by running `aws rds describe-db-engine-versions`. For example with Aurora MySQL 2, a potential value for this argument is `5.7.mysql_aurora.2.03.2`. The value can contain a partial version where supported by the API. The actual engine version used is returned in the attribute `engine_version_actual`, , see Attribute Reference below.
     */
    @JvmName("oxtyhmnonckldfjj")
    public suspend fun engineVersion(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.engineVersion = mapped
    }

    /**
     * @param value Name of your final DB snapshot when this DB cluster is deleted. If omitted, no final snapshot will be made.
     */
    @JvmName("refvuwopyalnlapf")
    public suspend fun finalSnapshotIdentifier(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.finalSnapshotIdentifier = mapped
    }

    /**
     * @param value Global cluster identifier specified on `aws.rds.GlobalCluster`.
     */
    @JvmName("ndyooxwcfcrtryjm")
    public suspend fun globalClusterIdentifier(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.globalClusterIdentifier = mapped
    }

    /**
     * @param value Specifies whether or not mappings of AWS Identity and Access Management (IAM) accounts to database accounts is enabled. Please see [AWS Documentation](https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/UsingWithRDS.IAMDBAuth.html) for availability and limitations.
     */
    @JvmName("uyqstcqblabdajkc")
    public suspend fun iamDatabaseAuthenticationEnabled(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.iamDatabaseAuthenticationEnabled = mapped
    }

    /**
     * @param value List of ARNs for the IAM roles to associate to the RDS Cluster.
     */
    @JvmName("bkkkjkxvyedeuvls")
    public suspend fun iamRoles(`value`: List<String>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.iamRoles = mapped
    }

    /**
     * @param values List of ARNs for the IAM roles to associate to the RDS Cluster.
     */
    @JvmName("iahpijigaecceanv")
    public suspend fun iamRoles(vararg values: String) {
        val toBeMapped = values.toList()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.iamRoles = mapped
    }

    /**
     * @param value Amount of Provisioned IOPS (input/output operations per second) to be initially allocated for each DB instance in the Multi-AZ DB cluster. For information about valid Iops values, see [Amazon RDS Provisioned IOPS storage to improve performance](https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/CHAP_Storage.html#USER_PIOPS) in the Amazon RDS User Guide. (This setting is required to create a Multi-AZ DB cluster). Must be a multiple between .5 and 50 of the storage amount for the DB cluster.
     */
    @JvmName("srdxoohclylfyfpl")
    public suspend fun iops(`value`: Int?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.iops = mapped
    }

    /**
     * @param value ARN for the KMS encryption key. When specifying `kms_key_id`, `storage_encrypted` needs to be set to true.
     */
    @JvmName("myoysxifepssjfmk")
    public suspend fun kmsKeyId(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.kmsKeyId = mapped
    }

    /**
     * @param value Set to true to allow RDS to manage the master user password in Secrets Manager. Cannot be set if `master_password` is provided.
     */
    @JvmName("sagbwewigyekrvfb")
    public suspend fun manageMasterUserPassword(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.manageMasterUserPassword = mapped
    }

    /**
     * @param value Password for the master DB user. Note that this may show up in logs, and it will be stored in the state file. Please refer to the [RDS Naming Constraints](http://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/CHAP_Limits.html#RDS_Limits.Constraints). Cannot be set if `manage_master_user_password` is set to `true`.
     */
    @JvmName("mkvuetfrjaacyoaa")
    public suspend fun masterPassword(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.masterPassword = mapped
    }

    /**
     * @param value Amazon Web Services KMS key identifier is the key ARN, key ID, alias ARN, or alias name for the KMS key. To use a KMS key in a different Amazon Web Services account, specify the key ARN or alias ARN. If not specified, the default KMS key for your Amazon Web Services account is used.
     */
    @JvmName("jftkywoukqphlmxx")
    public suspend fun masterUserSecretKmsKeyId(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.masterUserSecretKmsKeyId = mapped
    }

    /**
     * @param value Username for the master DB user. Please refer to the [RDS Naming Constraints](http://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/CHAP_Limits.html#RDS_Limits.Constraints). This argument does not support in-place updates and cannot be changed during a restore from snapshot.
     */
    @JvmName("elntklbogvgpnrmm")
    public suspend fun masterUsername(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.masterUsername = mapped
    }

    /**
     * @param value Network type of the cluster. Valid values: `IPV4`, `DUAL`.
     */
    @JvmName("dlbjsjpyxdbfjsuk")
    public suspend fun networkType(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.networkType = mapped
    }

    /**
     * @param value Port on which the DB accepts connections
     */
    @JvmName("gomvgdrtlasouseq")
    public suspend fun port(`value`: Int?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.port = mapped
    }

    /**
     * @param value Daily time range during which automated backups are created if automated backups are enabled using the BackupRetentionPeriod parameter.Time in UTC. Default: A 30-minute window selected at random from an 8-hour block of time per regionE.g., 04:00-09:00
     */
    @JvmName("tnjdrxfsqgxgkjxj")
    public suspend fun preferredBackupWindow(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.preferredBackupWindow = mapped
    }

    /**
     * @param value Weekly time range during which system maintenance can occur, in (UTC) e.g., wed:04:00-wed:04:30
     */
    @JvmName("pwbixbxejgqetuwd")
    public suspend fun preferredMaintenanceWindow(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.preferredMaintenanceWindow = mapped
    }

    /**
     * @param value ARN of a source DB cluster or DB instance if this DB cluster is to be created as a Read Replica. If DB Cluster is part of a Global Cluster, use the `lifecycle` configuration block `ignore_changes` argument to prevent this provider from showing differences for this argument instead of configuring this value.
     */
    @JvmName("mcyxodlkcqohbruh")
    public suspend fun replicationSourceIdentifier(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.replicationSourceIdentifier = mapped
    }

    /**
     * @param value Nested attribute for [point in time restore](https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/USER_PIT.html). More details below.
     */
    @JvmName("dkfkvippbkfspfdk")
    public suspend fun restoreToPointInTime(`value`: ClusterRestoreToPointInTimeArgs?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.restoreToPointInTime = mapped
    }

    /**
     * @param argument Nested attribute for [point in time restore](https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/USER_PIT.html). More details below.
     */
    @JvmName("ijwcqnxubdcplnon")
    public suspend
    fun restoreToPointInTime(argument: suspend ClusterRestoreToPointInTimeArgsBuilder.() -> Unit) {
        val toBeMapped = ClusterRestoreToPointInTimeArgsBuilder().applySuspend { argument() }.build()
        val mapped = of(toBeMapped)
        this.restoreToPointInTime = mapped
    }

    /**
     * @param value
     */
    @JvmName("ubgdpvgdexfsnopw")
    public suspend fun s3Import(`value`: ClusterS3ImportArgs?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.s3Import = mapped
    }

    /**
     * @param argument
     */
    @JvmName("rkxesjnadropvrgj")
    public suspend fun s3Import(argument: suspend ClusterS3ImportArgsBuilder.() -> Unit) {
        val toBeMapped = ClusterS3ImportArgsBuilder().applySuspend { argument() }.build()
        val mapped = of(toBeMapped)
        this.s3Import = mapped
    }

    /**
     * @param value Nested attribute with scaling properties. Only valid when `engine_mode` is set to `serverless`. More details below.
     */
    @JvmName("jpddlwtnjqgutlcn")
    public suspend fun scalingConfiguration(`value`: ClusterScalingConfigurationArgs?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.scalingConfiguration = mapped
    }

    /**
     * @param argument Nested attribute with scaling properties. Only valid when `engine_mode` is set to `serverless`. More details below.
     */
    @JvmName("jxygjyoekaxetkwu")
    public suspend
    fun scalingConfiguration(argument: suspend ClusterScalingConfigurationArgsBuilder.() -> Unit) {
        val toBeMapped = ClusterScalingConfigurationArgsBuilder().applySuspend { argument() }.build()
        val mapped = of(toBeMapped)
        this.scalingConfiguration = mapped
    }

    /**
     * @param value Nested attribute with scaling properties for ServerlessV2. Only valid when `engine_mode` is set to `provisioned`. More details below.
     */
    @JvmName("imdomgangemlewfv")
    public suspend
    fun serverlessv2ScalingConfiguration(`value`: ClusterServerlessv2ScalingConfigurationArgs?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.serverlessv2ScalingConfiguration = mapped
    }

    /**
     * @param argument Nested attribute with scaling properties for ServerlessV2. Only valid when `engine_mode` is set to `provisioned`. More details below.
     */
    @JvmName("nchhmpwishftrarf")
    public suspend
    fun serverlessv2ScalingConfiguration(argument: suspend ClusterServerlessv2ScalingConfigurationArgsBuilder.() -> Unit) {
        val toBeMapped = ClusterServerlessv2ScalingConfigurationArgsBuilder().applySuspend {
            argument()
        }.build()
        val mapped = of(toBeMapped)
        this.serverlessv2ScalingConfiguration = mapped
    }

    /**
     * @param value Determines whether a final DB snapshot is created before the DB cluster is deleted. If true is specified, no DB snapshot is created. If false is specified, a DB snapshot is created before the DB cluster is deleted, using the value from `final_snapshot_identifier`. Default is `false`.
     */
    @JvmName("xvkibggslkqbtmyd")
    public suspend fun skipFinalSnapshot(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.skipFinalSnapshot = mapped
    }

    /**
     * @param value Specifies whether or not to create this cluster from a snapshot. You can use either the name or ARN when specifying a DB cluster snapshot, or the ARN when specifying a DB snapshot. Conflicts with `global_cluster_identifier`. Clusters cannot be restored from snapshot **and** joined to an existing global cluster in a single operation. See the [AWS documentation](https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/aurora-global-database-getting-started.html#aurora-global-database.use-snapshot) or the Global Cluster Restored From Snapshot example for instructions on building a global cluster starting with a snapshot.
     */
    @JvmName("rdoqqcwfrranpwth")
    public suspend fun snapshotIdentifier(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.snapshotIdentifier = mapped
    }

    /**
     * @param value The source region for an encrypted replica DB cluster.
     */
    @JvmName("wpsheeinjvyeppak")
    public suspend fun sourceRegion(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.sourceRegion = mapped
    }

    /**
     * @param value Specifies whether the DB cluster is encrypted. The default is `false` for `provisioned` `engine_mode` and `true` for `serverless` `engine_mode`. When restoring an unencrypted `snapshot_identifier`, the `kms_key_id` argument must be provided to encrypt the restored cluster. The provider will only perform drift detection if a configuration value is provided.
     */
    @JvmName("fupfphibwogoviim")
    public suspend fun storageEncrypted(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.storageEncrypted = mapped
    }

    /**
     * @param value (Forces new for Multi-AZ DB clusters) Specifies the storage type to be associated with the DB cluster. For Aurora DB clusters, `storage_type` modifications can be done in-place. For Multi-AZ DB Clusters, the `iops` argument must also be set. Valid values are: `""`, `aurora-iopt1` (Aurora DB Clusters); `io1` (Multi-AZ DB Clusters). Default: `""` (Aurora DB Clusters); `io1` (Multi-AZ DB Clusters).
     */
    @JvmName("tomnortowcktnihu")
    public suspend fun storageType(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.storageType = mapped
    }

    /**
     * @param value A map of tags to assign to the DB cluster. If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
     */
    @JvmName("imotpdaltobostig")
    public suspend fun tags(`value`: Map<String, String>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.tags = mapped
    }

    /**
     * @param values A map of tags to assign to the DB cluster. If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
     */
    @JvmName("snyjxwsctsyitqio")
    public fun tags(vararg values: Pair<String, String>) {
        val toBeMapped = values.toMap()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.tags = mapped
    }

    /**
     * @param value List of VPC security groups to associate with the Cluster
     */
    @JvmName("njlcpbeotnlkkjpo")
    public suspend fun vpcSecurityGroupIds(`value`: List<String>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.vpcSecurityGroupIds = mapped
    }

    /**
     * @param values List of VPC security groups to associate with the Cluster
     */
    @JvmName("ewofkrumjigwkepb")
    public suspend fun vpcSecurityGroupIds(vararg values: String) {
        val toBeMapped = values.toList()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.vpcSecurityGroupIds = mapped
    }

    internal fun build(): ClusterArgs = ClusterArgs(
        allocatedStorage = allocatedStorage,
        allowMajorVersionUpgrade = allowMajorVersionUpgrade,
        applyImmediately = applyImmediately,
        availabilityZones = availabilityZones,
        backtrackWindow = backtrackWindow,
        backupRetentionPeriod = backupRetentionPeriod,
        clusterIdentifier = clusterIdentifier,
        clusterIdentifierPrefix = clusterIdentifierPrefix,
        clusterMembers = clusterMembers,
        copyTagsToSnapshot = copyTagsToSnapshot,
        databaseName = databaseName,
        dbClusterInstanceClass = dbClusterInstanceClass,
        dbClusterParameterGroupName = dbClusterParameterGroupName,
        dbInstanceParameterGroupName = dbInstanceParameterGroupName,
        dbSubnetGroupName = dbSubnetGroupName,
        dbSystemId = dbSystemId,
        deleteAutomatedBackups = deleteAutomatedBackups,
        deletionProtection = deletionProtection,
        enableGlobalWriteForwarding = enableGlobalWriteForwarding,
        enableHttpEndpoint = enableHttpEndpoint,
        enabledCloudwatchLogsExports = enabledCloudwatchLogsExports,
        engine = engine,
        engineMode = engineMode,
        engineVersion = engineVersion,
        finalSnapshotIdentifier = finalSnapshotIdentifier,
        globalClusterIdentifier = globalClusterIdentifier,
        iamDatabaseAuthenticationEnabled = iamDatabaseAuthenticationEnabled,
        iamRoles = iamRoles,
        iops = iops,
        kmsKeyId = kmsKeyId,
        manageMasterUserPassword = manageMasterUserPassword,
        masterPassword = masterPassword,
        masterUserSecretKmsKeyId = masterUserSecretKmsKeyId,
        masterUsername = masterUsername,
        networkType = networkType,
        port = port,
        preferredBackupWindow = preferredBackupWindow,
        preferredMaintenanceWindow = preferredMaintenanceWindow,
        replicationSourceIdentifier = replicationSourceIdentifier,
        restoreToPointInTime = restoreToPointInTime,
        s3Import = s3Import,
        scalingConfiguration = scalingConfiguration,
        serverlessv2ScalingConfiguration = serverlessv2ScalingConfiguration,
        skipFinalSnapshot = skipFinalSnapshot,
        snapshotIdentifier = snapshotIdentifier,
        sourceRegion = sourceRegion,
        storageEncrypted = storageEncrypted,
        storageType = storageType,
        tags = tags,
        vpcSecurityGroupIds = vpcSecurityGroupIds,
    )
}
