@file:Suppress("NAME_SHADOWING", "DEPRECATION")

package com.pulumi.aws.rds.kotlin

import com.pulumi.aws.rds.ClusterInstanceArgs.builder
import com.pulumi.aws.rds.kotlin.enums.InstanceType
import com.pulumi.core.Either
import com.pulumi.core.Output
import com.pulumi.core.Output.of
import com.pulumi.kotlin.ConvertibleToJava
import com.pulumi.kotlin.PulumiTagMarker
import kotlin.Boolean
import kotlin.Int
import kotlin.Pair
import kotlin.String
import kotlin.Suppress
import kotlin.collections.Map
import kotlin.jvm.JvmName

/**
 * Provides an RDS Cluster Instance Resource. A Cluster Instance Resource defines
 * attributes that are specific to a single instance in a RDS Cluster,
 * specifically running Amazon Aurora.
 * Unlike other RDS resources that support replication, with Amazon Aurora you do
 * not designate a primary and subsequent replicas. Instead, you simply add RDS
 * Instances and Aurora manages the replication. You can use the [count][5]
 * meta-parameter to make multiple instances and join them all to the same RDS
 * Cluster, or you may specify different Cluster Instance resources with various
 * `instance_class` sizes.
 * For more information on Amazon Aurora, see [Aurora on Amazon RDS](https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/CHAP_Aurora.html) in the Amazon RDS User Guide.
 * > **NOTE:** Deletion Protection from the RDS service can only be enabled at the cluster level, not for individual cluster instances. You can still add the [`protect` CustomResourceOption](https://www.pulumi.com/docs/intro/concepts/programming-model/#protect) to this resource configuration if you desire protection from accidental deletion.
 * > **NOTE:** `aurora` is no longer a valid `engine` because of [Amazon Aurora's MySQL-Compatible Edition version 1 end of life](https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/Aurora.MySQL56.EOL.html).
 * ## Example Usage
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.rds.Cluster;
 * import com.pulumi.aws.rds.ClusterArgs;
 * import com.pulumi.aws.rds.ClusterInstance;
 * import com.pulumi.aws.rds.ClusterInstanceArgs;
 * import com.pulumi.codegen.internal.KeyedValue;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var default_ = new Cluster("default", ClusterArgs.builder()
 *             .clusterIdentifier("aurora-cluster-demo")
 *             .availabilityZones(
 *                 "us-west-2a",
 *                 "us-west-2b",
 *                 "us-west-2c")
 *             .databaseName("mydb")
 *             .masterUsername("foo")
 *             .masterPassword("barbut8chars")
 *             .build());
 *         for (var i = 0; i < 2; i++) {
 *             new ClusterInstance("clusterInstances-" + i, ClusterInstanceArgs.builder()
 *                 .identifier(String.format("aurora-cluster-demo-%s", range.value()))
 *                 .clusterIdentifier(default_.id())
 *                 .instanceClass("db.r4.large")
 *                 .engine(default_.engine())
 *                 .engineVersion(default_.engineVersion())
 *                 .build());
 * }
 *     }
 * }
 * ```
 * ## Import
 * Using `pulumi import`, import RDS Cluster Instances using the `identifier`. For example:
 * ```sh
 *  $ pulumi import aws:rds/clusterInstance:ClusterInstance prod_instance_1 aurora-cluster-instance-1
 * ```
 * @property applyImmediately Specifies whether any database modifications are applied immediately, or during the next maintenance window. Default is`false`.
 * @property autoMinorVersionUpgrade Indicates that minor engine upgrades will be applied automatically to the DB instance during the maintenance window. Default `true`.
 * @property availabilityZone EC2 Availability Zone that the DB instance is created in. See [docs](https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_CreateDBInstance.html) about the details.
 * @property caCertIdentifier Identifier of the CA certificate for the DB instance.
 * @property clusterIdentifier Identifier of the `aws.rds.Cluster` in which to launch this instance.
 * @property copyTagsToSnapshot Indicates whether to copy all of the user-defined tags from the DB instance to snapshots of the DB instance. Default `false`.
 * @property customIamInstanceProfile Instance profile associated with the underlying Amazon EC2 instance of an RDS Custom DB instance.
 * @property dbParameterGroupName Name of the DB parameter group to associate with this instance.
 * @property dbSubnetGroupName DB subnet group to associate with this DB instance. **NOTE:** This must match the `db_subnet_group_name` of the attached `aws.rds.Cluster`.
 * @property engine Name of the database engine to be used for the RDS cluster instance.
 * Valid Values: `aurora-mysql`, `aurora-postgresql`.
 * @property engineVersion Database engine version.
 * @property identifier Identifier for the RDS instance, if omitted, Pulumi will assign a random, unique identifier.
 * @property identifierPrefix Creates a unique identifier beginning with the specified prefix. Conflicts with `identifier`.
 * @property instanceClass Instance class to use. For details on CPU and memory, see [Scaling Aurora DB Instances](https://docs&#46;aws&#46;amazon&#46;com/AmazonRDS/latest/UserGuide/Aurora&#46;Managing&#46;html)&#46; Aurora uses `db&#46;*` instance classes/types&#46; Please see [AWS Documentation](https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/Concepts.DBInstanceClass.html) for currently available instance classes and complete details.
 * @property monitoringInterval Interval, in seconds, between points when Enhanced Monitoring metrics are collected for the DB instance. To disable collecting Enhanced Monitoring metrics, specify 0. The default is 0. Valid Values: 0, 1, 5, 10, 15, 30, 60.
 * @property monitoringRoleArn ARN for the IAM role that permits RDS to send enhanced monitoring metrics to CloudWatch Logs. You can find more information on the [AWS Documentation](http://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_Monitoring.html) what IAM permissions are needed to allow Enhanced Monitoring for RDS Instances.
 * @property performanceInsightsEnabled Specifies whether Performance Insights is enabled or not.
 * @property performanceInsightsKmsKeyId ARN for the KMS key to encrypt Performance Insights data. When specifying `performance_insights_kms_key_id`, `performance_insights_enabled` needs to be set to true.
 * @property performanceInsightsRetentionPeriod Amount of time in days to retain Performance Insights data. Valid values are `7`, `731` (2 years) or a multiple of `31`. When specifying `performance_insights_retention_period`, `performance_insights_enabled` needs to be set to true. Defaults to '7'.
 * @property preferredBackupWindow Daily time range during which automated backups are created if automated backups are enabled. Eg: "04:00-09:00". **NOTE:** If `preferred_backup_window` is set at the cluster level, this argument **must** be omitted.
 * @property preferredMaintenanceWindow Window to perform maintenance in. Syntax: "ddd:hh24:mi-ddd:hh24:mi". Eg: "Mon:00:00-Mon:03:00".
 * @property promotionTier Default 0. Failover Priority setting on instance level. The reader who has lower tier has higher priority to get promoted to writer.
 * @property publiclyAccessible Bool to control if instance is publicly accessible. Default `false`. See the documentation on [Creating DB Instances](https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_CreateDBInstance.html) for more details on controlling this property.
 * @property tags Map of tags to assign to the instance. If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
 */
public data class ClusterInstanceArgs(
    public val applyImmediately: Output<Boolean>? = null,
    public val autoMinorVersionUpgrade: Output<Boolean>? = null,
    public val availabilityZone: Output<String>? = null,
    public val caCertIdentifier: Output<String>? = null,
    public val clusterIdentifier: Output<String>? = null,
    public val copyTagsToSnapshot: Output<Boolean>? = null,
    public val customIamInstanceProfile: Output<String>? = null,
    public val dbParameterGroupName: Output<String>? = null,
    public val dbSubnetGroupName: Output<String>? = null,
    public val engine: Output<String>? = null,
    public val engineVersion: Output<String>? = null,
    public val identifier: Output<String>? = null,
    public val identifierPrefix: Output<String>? = null,
    public val instanceClass: Output<Either<String, InstanceType>>? = null,
    public val monitoringInterval: Output<Int>? = null,
    public val monitoringRoleArn: Output<String>? = null,
    public val performanceInsightsEnabled: Output<Boolean>? = null,
    public val performanceInsightsKmsKeyId: Output<String>? = null,
    public val performanceInsightsRetentionPeriod: Output<Int>? = null,
    public val preferredBackupWindow: Output<String>? = null,
    public val preferredMaintenanceWindow: Output<String>? = null,
    public val promotionTier: Output<Int>? = null,
    public val publiclyAccessible: Output<Boolean>? = null,
    public val tags: Output<Map<String, String>>? = null,
) : ConvertibleToJava<com.pulumi.aws.rds.ClusterInstanceArgs> {
    override fun toJava(): com.pulumi.aws.rds.ClusterInstanceArgs =
        com.pulumi.aws.rds.ClusterInstanceArgs.builder()
            .applyImmediately(applyImmediately?.applyValue({ args0 -> args0 }))
            .autoMinorVersionUpgrade(autoMinorVersionUpgrade?.applyValue({ args0 -> args0 }))
            .availabilityZone(availabilityZone?.applyValue({ args0 -> args0 }))
            .caCertIdentifier(caCertIdentifier?.applyValue({ args0 -> args0 }))
            .clusterIdentifier(clusterIdentifier?.applyValue({ args0 -> args0 }))
            .copyTagsToSnapshot(copyTagsToSnapshot?.applyValue({ args0 -> args0 }))
            .customIamInstanceProfile(customIamInstanceProfile?.applyValue({ args0 -> args0 }))
            .dbParameterGroupName(dbParameterGroupName?.applyValue({ args0 -> args0 }))
            .dbSubnetGroupName(dbSubnetGroupName?.applyValue({ args0 -> args0 }))
            .engine(engine?.applyValue({ args0 -> args0 }))
            .engineVersion(engineVersion?.applyValue({ args0 -> args0 }))
            .identifier(identifier?.applyValue({ args0 -> args0 }))
            .identifierPrefix(identifierPrefix?.applyValue({ args0 -> args0 }))
            .instanceClass(
                instanceClass?.applyValue({ args0 ->
                    args0.transform({ args0 -> args0 }, { args0 ->
                        args0.let({ args0 -> args0.toJava() })
                    })
                }),
            )
            .monitoringInterval(monitoringInterval?.applyValue({ args0 -> args0 }))
            .monitoringRoleArn(monitoringRoleArn?.applyValue({ args0 -> args0 }))
            .performanceInsightsEnabled(performanceInsightsEnabled?.applyValue({ args0 -> args0 }))
            .performanceInsightsKmsKeyId(performanceInsightsKmsKeyId?.applyValue({ args0 -> args0 }))
            .performanceInsightsRetentionPeriod(
                performanceInsightsRetentionPeriod?.applyValue({ args0 ->
                    args0
                }),
            )
            .preferredBackupWindow(preferredBackupWindow?.applyValue({ args0 -> args0 }))
            .preferredMaintenanceWindow(preferredMaintenanceWindow?.applyValue({ args0 -> args0 }))
            .promotionTier(promotionTier?.applyValue({ args0 -> args0 }))
            .publiclyAccessible(publiclyAccessible?.applyValue({ args0 -> args0 }))
            .tags(
                tags?.applyValue({ args0 ->
                    args0.map({ args0 ->
                        args0.key.to(args0.value)
                    }).toMap()
                }),
            ).build()
}

/**
 * Builder for [ClusterInstanceArgs].
 */
@PulumiTagMarker
public class ClusterInstanceArgsBuilder internal constructor() {
    private var applyImmediately: Output<Boolean>? = null

    private var autoMinorVersionUpgrade: Output<Boolean>? = null

    private var availabilityZone: Output<String>? = null

    private var caCertIdentifier: Output<String>? = null

    private var clusterIdentifier: Output<String>? = null

    private var copyTagsToSnapshot: Output<Boolean>? = null

    private var customIamInstanceProfile: Output<String>? = null

    private var dbParameterGroupName: Output<String>? = null

    private var dbSubnetGroupName: Output<String>? = null

    private var engine: Output<String>? = null

    private var engineVersion: Output<String>? = null

    private var identifier: Output<String>? = null

    private var identifierPrefix: Output<String>? = null

    private var instanceClass: Output<Either<String, InstanceType>>? = null

    private var monitoringInterval: Output<Int>? = null

    private var monitoringRoleArn: Output<String>? = null

    private var performanceInsightsEnabled: Output<Boolean>? = null

    private var performanceInsightsKmsKeyId: Output<String>? = null

    private var performanceInsightsRetentionPeriod: Output<Int>? = null

    private var preferredBackupWindow: Output<String>? = null

    private var preferredMaintenanceWindow: Output<String>? = null

    private var promotionTier: Output<Int>? = null

    private var publiclyAccessible: Output<Boolean>? = null

    private var tags: Output<Map<String, String>>? = null

    /**
     * @param value Specifies whether any database modifications are applied immediately, or during the next maintenance window. Default is`false`.
     */
    @JvmName("kfgckknkhoqcchao")
    public suspend fun applyImmediately(`value`: Output<Boolean>) {
        this.applyImmediately = value
    }

    /**
     * @param value Indicates that minor engine upgrades will be applied automatically to the DB instance during the maintenance window. Default `true`.
     */
    @JvmName("kjkgglshdutnjnwo")
    public suspend fun autoMinorVersionUpgrade(`value`: Output<Boolean>) {
        this.autoMinorVersionUpgrade = value
    }

    /**
     * @param value EC2 Availability Zone that the DB instance is created in. See [docs](https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_CreateDBInstance.html) about the details.
     */
    @JvmName("fuaegcdyxowfrbrk")
    public suspend fun availabilityZone(`value`: Output<String>) {
        this.availabilityZone = value
    }

    /**
     * @param value Identifier of the CA certificate for the DB instance.
     */
    @JvmName("hjjpjoqhvabbjicf")
    public suspend fun caCertIdentifier(`value`: Output<String>) {
        this.caCertIdentifier = value
    }

    /**
     * @param value Identifier of the `aws.rds.Cluster` in which to launch this instance.
     */
    @JvmName("iumjwebfxhnbaspp")
    public suspend fun clusterIdentifier(`value`: Output<String>) {
        this.clusterIdentifier = value
    }

    /**
     * @param value Indicates whether to copy all of the user-defined tags from the DB instance to snapshots of the DB instance. Default `false`.
     */
    @JvmName("soobwvhecexnwhhi")
    public suspend fun copyTagsToSnapshot(`value`: Output<Boolean>) {
        this.copyTagsToSnapshot = value
    }

    /**
     * @param value Instance profile associated with the underlying Amazon EC2 instance of an RDS Custom DB instance.
     */
    @JvmName("mbcdwidqkgxmqlbe")
    public suspend fun customIamInstanceProfile(`value`: Output<String>) {
        this.customIamInstanceProfile = value
    }

    /**
     * @param value Name of the DB parameter group to associate with this instance.
     */
    @JvmName("ryvywjtdqnjkvemd")
    public suspend fun dbParameterGroupName(`value`: Output<String>) {
        this.dbParameterGroupName = value
    }

    /**
     * @param value DB subnet group to associate with this DB instance. **NOTE:** This must match the `db_subnet_group_name` of the attached `aws.rds.Cluster`.
     */
    @JvmName("ubjvpiywccboymdx")
    public suspend fun dbSubnetGroupName(`value`: Output<String>) {
        this.dbSubnetGroupName = value
    }

    /**
     * @param value Name of the database engine to be used for the RDS cluster instance.
     * Valid Values: `aurora-mysql`, `aurora-postgresql`.
     */
    @JvmName("ckwsxgbfiqktideo")
    public suspend fun engine(`value`: Output<String>) {
        this.engine = value
    }

    /**
     * @param value Database engine version.
     */
    @JvmName("vvhbcosjgiddkuny")
    public suspend fun engineVersion(`value`: Output<String>) {
        this.engineVersion = value
    }

    /**
     * @param value Identifier for the RDS instance, if omitted, Pulumi will assign a random, unique identifier.
     */
    @JvmName("awvrpmusmqmbkonv")
    public suspend fun identifier(`value`: Output<String>) {
        this.identifier = value
    }

    /**
     * @param value Creates a unique identifier beginning with the specified prefix. Conflicts with `identifier`.
     */
    @JvmName("pwllxxwkswkwcndb")
    public suspend fun identifierPrefix(`value`: Output<String>) {
        this.identifierPrefix = value
    }

    /**
     * @param value Instance class to use. For details on CPU and memory, see [Scaling Aurora DB Instances](https://docs&#46;aws&#46;amazon&#46;com/AmazonRDS/latest/UserGuide/Aurora&#46;Managing&#46;html)&#46; Aurora uses `db&#46;*` instance classes/types&#46; Please see [AWS Documentation](https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/Concepts.DBInstanceClass.html) for currently available instance classes and complete details.
     */
    @JvmName("oyjebbbwseeeupcm")
    public suspend fun instanceClass(`value`: Output<Either<String, InstanceType>>) {
        this.instanceClass = value
    }

    /**
     * @param value Interval, in seconds, between points when Enhanced Monitoring metrics are collected for the DB instance. To disable collecting Enhanced Monitoring metrics, specify 0. The default is 0. Valid Values: 0, 1, 5, 10, 15, 30, 60.
     */
    @JvmName("mvxjcxuuxmlrciby")
    public suspend fun monitoringInterval(`value`: Output<Int>) {
        this.monitoringInterval = value
    }

    /**
     * @param value ARN for the IAM role that permits RDS to send enhanced monitoring metrics to CloudWatch Logs. You can find more information on the [AWS Documentation](http://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_Monitoring.html) what IAM permissions are needed to allow Enhanced Monitoring for RDS Instances.
     */
    @JvmName("calnwvsboprfhvsr")
    public suspend fun monitoringRoleArn(`value`: Output<String>) {
        this.monitoringRoleArn = value
    }

    /**
     * @param value Specifies whether Performance Insights is enabled or not.
     */
    @JvmName("oeypneylhpyglkpg")
    public suspend fun performanceInsightsEnabled(`value`: Output<Boolean>) {
        this.performanceInsightsEnabled = value
    }

    /**
     * @param value ARN for the KMS key to encrypt Performance Insights data. When specifying `performance_insights_kms_key_id`, `performance_insights_enabled` needs to be set to true.
     */
    @JvmName("amdxtbfnyxdbnkek")
    public suspend fun performanceInsightsKmsKeyId(`value`: Output<String>) {
        this.performanceInsightsKmsKeyId = value
    }

    /**
     * @param value Amount of time in days to retain Performance Insights data. Valid values are `7`, `731` (2 years) or a multiple of `31`. When specifying `performance_insights_retention_period`, `performance_insights_enabled` needs to be set to true. Defaults to '7'.
     */
    @JvmName("dmyneaiypvohkksu")
    public suspend fun performanceInsightsRetentionPeriod(`value`: Output<Int>) {
        this.performanceInsightsRetentionPeriod = value
    }

    /**
     * @param value Daily time range during which automated backups are created if automated backups are enabled. Eg: "04:00-09:00". **NOTE:** If `preferred_backup_window` is set at the cluster level, this argument **must** be omitted.
     */
    @JvmName("rqwsbehixuwrneju")
    public suspend fun preferredBackupWindow(`value`: Output<String>) {
        this.preferredBackupWindow = value
    }

    /**
     * @param value Window to perform maintenance in. Syntax: "ddd:hh24:mi-ddd:hh24:mi". Eg: "Mon:00:00-Mon:03:00".
     */
    @JvmName("xxjxvlbhklpfbnjf")
    public suspend fun preferredMaintenanceWindow(`value`: Output<String>) {
        this.preferredMaintenanceWindow = value
    }

    /**
     * @param value Default 0. Failover Priority setting on instance level. The reader who has lower tier has higher priority to get promoted to writer.
     */
    @JvmName("hjofxmtjvlryjmhw")
    public suspend fun promotionTier(`value`: Output<Int>) {
        this.promotionTier = value
    }

    /**
     * @param value Bool to control if instance is publicly accessible. Default `false`. See the documentation on [Creating DB Instances](https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_CreateDBInstance.html) for more details on controlling this property.
     */
    @JvmName("rwwqqhdliolephvg")
    public suspend fun publiclyAccessible(`value`: Output<Boolean>) {
        this.publiclyAccessible = value
    }

    /**
     * @param value Map of tags to assign to the instance. If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
     */
    @JvmName("ftxflpytnkifjrwu")
    public suspend fun tags(`value`: Output<Map<String, String>>) {
        this.tags = value
    }

    /**
     * @param value Specifies whether any database modifications are applied immediately, or during the next maintenance window. Default is`false`.
     */
    @JvmName("hrxvkkdxaokxafad")
    public suspend fun applyImmediately(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.applyImmediately = mapped
    }

    /**
     * @param value Indicates that minor engine upgrades will be applied automatically to the DB instance during the maintenance window. Default `true`.
     */
    @JvmName("oywobqcbohvlrxmc")
    public suspend fun autoMinorVersionUpgrade(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.autoMinorVersionUpgrade = mapped
    }

    /**
     * @param value EC2 Availability Zone that the DB instance is created in. See [docs](https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_CreateDBInstance.html) about the details.
     */
    @JvmName("lrvoagdyjntiqpxo")
    public suspend fun availabilityZone(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.availabilityZone = mapped
    }

    /**
     * @param value Identifier of the CA certificate for the DB instance.
     */
    @JvmName("oajrmeskxoypcqpe")
    public suspend fun caCertIdentifier(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.caCertIdentifier = mapped
    }

    /**
     * @param value Identifier of the `aws.rds.Cluster` in which to launch this instance.
     */
    @JvmName("jaglehemvastkfqx")
    public suspend fun clusterIdentifier(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.clusterIdentifier = mapped
    }

    /**
     * @param value Indicates whether to copy all of the user-defined tags from the DB instance to snapshots of the DB instance. Default `false`.
     */
    @JvmName("ytxjpqnmnwmgybvh")
    public suspend fun copyTagsToSnapshot(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.copyTagsToSnapshot = mapped
    }

    /**
     * @param value Instance profile associated with the underlying Amazon EC2 instance of an RDS Custom DB instance.
     */
    @JvmName("qtkegxqqjmyipglw")
    public suspend fun customIamInstanceProfile(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.customIamInstanceProfile = mapped
    }

    /**
     * @param value Name of the DB parameter group to associate with this instance.
     */
    @JvmName("gmwnjfrrtqfifugf")
    public suspend fun dbParameterGroupName(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.dbParameterGroupName = mapped
    }

    /**
     * @param value DB subnet group to associate with this DB instance. **NOTE:** This must match the `db_subnet_group_name` of the attached `aws.rds.Cluster`.
     */
    @JvmName("wmpmibaaptrwgygm")
    public suspend fun dbSubnetGroupName(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.dbSubnetGroupName = mapped
    }

    /**
     * @param value Name of the database engine to be used for the RDS cluster instance.
     * Valid Values: `aurora-mysql`, `aurora-postgresql`.
     */
    @JvmName("dbwgkmblafpdmbhs")
    public suspend fun engine(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.engine = mapped
    }

    /**
     * @param value Database engine version.
     */
    @JvmName("abeskunwukaolfxy")
    public suspend fun engineVersion(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.engineVersion = mapped
    }

    /**
     * @param value Identifier for the RDS instance, if omitted, Pulumi will assign a random, unique identifier.
     */
    @JvmName("iudsqdorvbnotabc")
    public suspend fun identifier(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.identifier = mapped
    }

    /**
     * @param value Creates a unique identifier beginning with the specified prefix. Conflicts with `identifier`.
     */
    @JvmName("juocwxunhglfvwvk")
    public suspend fun identifierPrefix(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.identifierPrefix = mapped
    }

    /**
     * @param value Instance class to use. For details on CPU and memory, see [Scaling Aurora DB Instances](https://docs&#46;aws&#46;amazon&#46;com/AmazonRDS/latest/UserGuide/Aurora&#46;Managing&#46;html)&#46; Aurora uses `db&#46;*` instance classes/types&#46; Please see [AWS Documentation](https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/Concepts.DBInstanceClass.html) for currently available instance classes and complete details.
     */
    @JvmName("smithsptsvownlpg")
    public suspend fun instanceClass(`value`: Either<String, InstanceType>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.instanceClass = mapped
    }

    /**
     * @param value Instance class to use. For details on CPU and memory, see [Scaling Aurora DB Instances](https://docs&#46;aws&#46;amazon&#46;com/AmazonRDS/latest/UserGuide/Aurora&#46;Managing&#46;html)&#46; Aurora uses `db&#46;*` instance classes/types&#46; Please see [AWS Documentation](https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/Concepts.DBInstanceClass.html) for currently available instance classes and complete details.
     */
    @JvmName("mrmbapuqimlnyuvo")
    public fun instanceClass(`value`: String) {
        val toBeMapped = Either.ofLeft<String, InstanceType>(value)
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.instanceClass = mapped
    }

    /**
     * @param value Instance class to use. For details on CPU and memory, see [Scaling Aurora DB Instances](https://docs&#46;aws&#46;amazon&#46;com/AmazonRDS/latest/UserGuide/Aurora&#46;Managing&#46;html)&#46; Aurora uses `db&#46;*` instance classes/types&#46; Please see [AWS Documentation](https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/Concepts.DBInstanceClass.html) for currently available instance classes and complete details.
     */
    @JvmName("mhxclvmkvbcsqvrl")
    public fun instanceClass(`value`: InstanceType) {
        val toBeMapped = Either.ofRight<String, InstanceType>(value)
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.instanceClass = mapped
    }

    /**
     * @param value Interval, in seconds, between points when Enhanced Monitoring metrics are collected for the DB instance. To disable collecting Enhanced Monitoring metrics, specify 0. The default is 0. Valid Values: 0, 1, 5, 10, 15, 30, 60.
     */
    @JvmName("ejhawpdabbphaxxa")
    public suspend fun monitoringInterval(`value`: Int?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.monitoringInterval = mapped
    }

    /**
     * @param value ARN for the IAM role that permits RDS to send enhanced monitoring metrics to CloudWatch Logs. You can find more information on the [AWS Documentation](http://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_Monitoring.html) what IAM permissions are needed to allow Enhanced Monitoring for RDS Instances.
     */
    @JvmName("yaicoeqrjmwvrjkc")
    public suspend fun monitoringRoleArn(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.monitoringRoleArn = mapped
    }

    /**
     * @param value Specifies whether Performance Insights is enabled or not.
     */
    @JvmName("evethyjieurpdhvv")
    public suspend fun performanceInsightsEnabled(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.performanceInsightsEnabled = mapped
    }

    /**
     * @param value ARN for the KMS key to encrypt Performance Insights data. When specifying `performance_insights_kms_key_id`, `performance_insights_enabled` needs to be set to true.
     */
    @JvmName("qkyeshwqecnohats")
    public suspend fun performanceInsightsKmsKeyId(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.performanceInsightsKmsKeyId = mapped
    }

    /**
     * @param value Amount of time in days to retain Performance Insights data. Valid values are `7`, `731` (2 years) or a multiple of `31`. When specifying `performance_insights_retention_period`, `performance_insights_enabled` needs to be set to true. Defaults to '7'.
     */
    @JvmName("awnupitulxorgbyj")
    public suspend fun performanceInsightsRetentionPeriod(`value`: Int?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.performanceInsightsRetentionPeriod = mapped
    }

    /**
     * @param value Daily time range during which automated backups are created if automated backups are enabled. Eg: "04:00-09:00". **NOTE:** If `preferred_backup_window` is set at the cluster level, this argument **must** be omitted.
     */
    @JvmName("pohlgbiyxtdxwnkh")
    public suspend fun preferredBackupWindow(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.preferredBackupWindow = mapped
    }

    /**
     * @param value Window to perform maintenance in. Syntax: "ddd:hh24:mi-ddd:hh24:mi". Eg: "Mon:00:00-Mon:03:00".
     */
    @JvmName("svjmmykmdrcomeic")
    public suspend fun preferredMaintenanceWindow(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.preferredMaintenanceWindow = mapped
    }

    /**
     * @param value Default 0. Failover Priority setting on instance level. The reader who has lower tier has higher priority to get promoted to writer.
     */
    @JvmName("qsiruodbxgwiigih")
    public suspend fun promotionTier(`value`: Int?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.promotionTier = mapped
    }

    /**
     * @param value Bool to control if instance is publicly accessible. Default `false`. See the documentation on [Creating DB Instances](https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_CreateDBInstance.html) for more details on controlling this property.
     */
    @JvmName("ttxuumpsthjsxjtn")
    public suspend fun publiclyAccessible(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.publiclyAccessible = mapped
    }

    /**
     * @param value Map of tags to assign to the instance. If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
     */
    @JvmName("dnqfrcdequpqefnm")
    public suspend fun tags(`value`: Map<String, String>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.tags = mapped
    }

    /**
     * @param values Map of tags to assign to the instance. If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
     */
    @JvmName("fxjrprualgilmrnb")
    public fun tags(vararg values: Pair<String, String>) {
        val toBeMapped = values.toMap()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.tags = mapped
    }

    internal fun build(): ClusterInstanceArgs = ClusterInstanceArgs(
        applyImmediately = applyImmediately,
        autoMinorVersionUpgrade = autoMinorVersionUpgrade,
        availabilityZone = availabilityZone,
        caCertIdentifier = caCertIdentifier,
        clusterIdentifier = clusterIdentifier,
        copyTagsToSnapshot = copyTagsToSnapshot,
        customIamInstanceProfile = customIamInstanceProfile,
        dbParameterGroupName = dbParameterGroupName,
        dbSubnetGroupName = dbSubnetGroupName,
        engine = engine,
        engineVersion = engineVersion,
        identifier = identifier,
        identifierPrefix = identifierPrefix,
        instanceClass = instanceClass,
        monitoringInterval = monitoringInterval,
        monitoringRoleArn = monitoringRoleArn,
        performanceInsightsEnabled = performanceInsightsEnabled,
        performanceInsightsKmsKeyId = performanceInsightsKmsKeyId,
        performanceInsightsRetentionPeriod = performanceInsightsRetentionPeriod,
        preferredBackupWindow = preferredBackupWindow,
        preferredMaintenanceWindow = preferredMaintenanceWindow,
        promotionTier = promotionTier,
        publiclyAccessible = publiclyAccessible,
        tags = tags,
    )
}
