@file:Suppress("NAME_SHADOWING", "DEPRECATION")

package com.pulumi.aws.rds.kotlin

import com.pulumi.aws.rds.GlobalClusterArgs.builder
import com.pulumi.core.Output
import com.pulumi.core.Output.of
import com.pulumi.kotlin.ConvertibleToJava
import com.pulumi.kotlin.PulumiTagMarker
import kotlin.Boolean
import kotlin.String
import kotlin.Suppress
import kotlin.jvm.JvmName

/**
 * Manages an RDS Global Cluster, which is an Aurora global database spread across multiple regions. The global database contains a single primary cluster with read-write capability, and a read-only secondary cluster that receives data from the primary cluster through high-speed replication performed by the Aurora storage subsystem.
 * More information about Aurora global databases can be found in the [Aurora User Guide](https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/aurora-global-database.html#aurora-global-database-creating).
 * ## Example Usage
 * ### New MySQL Global Cluster
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.rds.GlobalCluster;
 * import com.pulumi.aws.rds.GlobalClusterArgs;
 * import com.pulumi.aws.rds.Cluster;
 * import com.pulumi.aws.rds.ClusterArgs;
 * import com.pulumi.aws.rds.ClusterInstance;
 * import com.pulumi.aws.rds.ClusterInstanceArgs;
 * import com.pulumi.resources.CustomResourceOptions;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var example = new GlobalCluster("example", GlobalClusterArgs.builder()
 *             .globalClusterIdentifier("global-test")
 *             .engine("aurora")
 *             .engineVersion("5.6.mysql_aurora.1.22.2")
 *             .databaseName("example_db")
 *             .build());
 *         var primaryCluster = new Cluster("primaryCluster", ClusterArgs.builder()
 *             .engine(example.engine())
 *             .engineVersion(example.engineVersion())
 *             .clusterIdentifier("test-primary-cluster")
 *             .masterUsername("username")
 *             .masterPassword("somepass123")
 *             .databaseName("example_db")
 *             .globalClusterIdentifier(example.id())
 *             .dbSubnetGroupName("default")
 *             .build(), CustomResourceOptions.builder()
 *                 .provider(aws.primary())
 *                 .build());
 *         var primaryClusterInstance = new ClusterInstance("primaryClusterInstance", ClusterInstanceArgs.builder()
 *             .engine(example.engine())
 *             .engineVersion(example.engineVersion())
 *             .identifier("test-primary-cluster-instance")
 *             .clusterIdentifier(primaryCluster.id())
 *             .instanceClass("db.r4.large")
 *             .dbSubnetGroupName("default")
 *             .build(), CustomResourceOptions.builder()
 *                 .provider(aws.primary())
 *                 .build());
 *         var secondaryCluster = new Cluster("secondaryCluster", ClusterArgs.builder()
 *             .engine(example.engine())
 *             .engineVersion(example.engineVersion())
 *             .clusterIdentifier("test-secondary-cluster")
 *             .globalClusterIdentifier(example.id())
 *             .dbSubnetGroupName("default")
 *             .build(), CustomResourceOptions.builder()
 *                 .provider(aws.secondary())
 *                 .dependsOn(primaryClusterInstance)
 *                 .build());
 *         var secondaryClusterInstance = new ClusterInstance("secondaryClusterInstance", ClusterInstanceArgs.builder()
 *             .engine(example.engine())
 *             .engineVersion(example.engineVersion())
 *             .identifier("test-secondary-cluster-instance")
 *             .clusterIdentifier(secondaryCluster.id())
 *             .instanceClass("db.r4.large")
 *             .dbSubnetGroupName("default")
 *             .build(), CustomResourceOptions.builder()
 *                 .provider(aws.secondary())
 *                 .build());
 *     }
 * }
 * ```
 * ### New PostgreSQL Global Cluster
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.Provider;
 * import com.pulumi.aws.ProviderArgs;
 * import com.pulumi.aws.rds.GlobalCluster;
 * import com.pulumi.aws.rds.GlobalClusterArgs;
 * import com.pulumi.aws.rds.Cluster;
 * import com.pulumi.aws.rds.ClusterArgs;
 * import com.pulumi.aws.rds.ClusterInstance;
 * import com.pulumi.aws.rds.ClusterInstanceArgs;
 * import com.pulumi.resources.CustomResourceOptions;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var primary = new Provider("primary", ProviderArgs.builder()
 *             .region("us-east-2")
 *             .build());
 *         var secondary = new Provider("secondary", ProviderArgs.builder()
 *             .region("us-east-1")
 *             .build());
 *         var example = new GlobalCluster("example", GlobalClusterArgs.builder()
 *             .globalClusterIdentifier("global-test")
 *             .engine("aurora-postgresql")
 *             .engineVersion("11.9")
 *             .databaseName("example_db")
 *             .build());
 *         var primaryCluster = new Cluster("primaryCluster", ClusterArgs.builder()
 *             .engine(example.engine())
 *             .engineVersion(example.engineVersion())
 *             .clusterIdentifier("test-primary-cluster")
 *             .masterUsername("username")
 *             .masterPassword("somepass123")
 *             .databaseName("example_db")
 *             .globalClusterIdentifier(example.id())
 *             .dbSubnetGroupName("default")
 *             .build(), CustomResourceOptions.builder()
 *                 .provider(aws.primary())
 *                 .build());
 *         var primaryClusterInstance = new ClusterInstance("primaryClusterInstance", ClusterInstanceArgs.builder()
 *             .engine(example.engine())
 *             .engineVersion(example.engineVersion())
 *             .identifier("test-primary-cluster-instance")
 *             .clusterIdentifier(primaryCluster.id())
 *             .instanceClass("db.r4.large")
 *             .dbSubnetGroupName("default")
 *             .build(), CustomResourceOptions.builder()
 *                 .provider(aws.primary())
 *                 .build());
 *         var secondaryCluster = new Cluster("secondaryCluster", ClusterArgs.builder()
 *             .engine(example.engine())
 *             .engineVersion(example.engineVersion())
 *             .clusterIdentifier("test-secondary-cluster")
 *             .globalClusterIdentifier(example.id())
 *             .skipFinalSnapshot(true)
 *             .dbSubnetGroupName("default")
 *             .build(), CustomResourceOptions.builder()
 *                 .provider(aws.secondary())
 *                 .dependsOn(primaryClusterInstance)
 *                 .build());
 *         var secondaryClusterInstance = new ClusterInstance("secondaryClusterInstance", ClusterInstanceArgs.builder()
 *             .engine(example.engine())
 *             .engineVersion(example.engineVersion())
 *             .identifier("test-secondary-cluster-instance")
 *             .clusterIdentifier(secondaryCluster.id())
 *             .instanceClass("db.r4.large")
 *             .dbSubnetGroupName("default")
 *             .build(), CustomResourceOptions.builder()
 *                 .provider(aws.secondary())
 *                 .build());
 *     }
 * }
 * ```
 * ### New Global Cluster From Existing DB Cluster
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.rds.Cluster;
 * import com.pulumi.aws.rds.GlobalCluster;
 * import com.pulumi.aws.rds.GlobalClusterArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var exampleCluster = new Cluster("exampleCluster");
 *         var exampleGlobalCluster = new GlobalCluster("exampleGlobalCluster", GlobalClusterArgs.builder()
 *             .forceDestroy(true)
 *             .globalClusterIdentifier("example")
 *             .sourceDbClusterIdentifier(exampleCluster.arn())
 *             .build());
 *     }
 * }
 * ```
 * ### Upgrading Engine Versions
 * When you upgrade the version of an `aws.rds.GlobalCluster`, the provider will attempt to in-place upgrade the engine versions of all associated clusters. Since the `aws.rds.Cluster` resource is being updated through the `aws.rds.GlobalCluster`, you are likely to get an error (`Provider produced inconsistent final plan`). To avoid this, use the `lifecycle` `ignore_changes` meta argument as shown below on the `aws.rds.Cluster`.
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.rds.GlobalCluster;
 * import com.pulumi.aws.rds.GlobalClusterArgs;
 * import com.pulumi.aws.rds.Cluster;
 * import com.pulumi.aws.rds.ClusterArgs;
 * import com.pulumi.aws.rds.ClusterInstance;
 * import com.pulumi.aws.rds.ClusterInstanceArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var example = new GlobalCluster("example", GlobalClusterArgs.builder()
 *             .globalClusterIdentifier("kyivkharkiv")
 *             .engine("aurora-mysql")
 *             .engineVersion("5.7.mysql_aurora.2.07.5")
 *             .build());
 *         var primaryCluster = new Cluster("primaryCluster", ClusterArgs.builder()
 *             .allowMajorVersionUpgrade(true)
 *             .applyImmediately(true)
 *             .clusterIdentifier("odessadnipro")
 *             .databaseName("totoro")
 *             .engine(example.engine())
 *             .engineVersion(example.engineVersion())
 *             .globalClusterIdentifier(example.id())
 *             .masterPassword("satsukimae")
 *             .masterUsername("maesatsuki")
 *             .skipFinalSnapshot(true)
 *             .build());
 *         var primaryClusterInstance = new ClusterInstance("primaryClusterInstance", ClusterInstanceArgs.builder()
 *             .applyImmediately(true)
 *             .clusterIdentifier(primaryCluster.id())
 *             .engine(primaryCluster.engine())
 *             .engineVersion(primaryCluster.engineVersion())
 *             .identifier("donetsklviv")
 *             .instanceClass("db.r4.large")
 *             .build());
 *     }
 * }
 * ```
 * ## Import
 * Using `pulumi import`, import `aws_rds_global_cluster` using the RDS Global Cluster identifier. For example:
 * ```sh
 *  $ pulumi import aws:rds/globalCluster:GlobalCluster example example
 * ```
 *  Certain resource arguments, like `force_destroy`, only exist within this provider. If the argument is set in the the provider configuration on an imported resource, This provider will show a difference on the first plan after import to update the state value. This change is safe to apply immediately so the state matches the desired configuration.
 * Certain resource arguments, like `source_db_cluster_identifier`, do not have an API method for reading the information after creation. If the argument is set in the Pulumi program on an imported resource, Pulumi will always show a difference. To workaround this behavior, either omit the argument from the Pulumi program or use `ignore_changes` to hide the difference. For example:
 * @property databaseName Name for an automatically created database on cluster creation.
 * @property deletionProtection If the Global Cluster should have deletion protection enabled. The database can't be deleted when this value is set to `true`. The default is `false`.
 * @property engine Name of the database engine to be used for this DB cluster. The provider will only perform drift detection if a configuration value is provided. Valid values: `aurora`, `aurora-mysql`, `aurora-postgresql`. Defaults to `aurora`. Conflicts with `source_db_cluster_identifier`.
 * @property engineVersion Engine version of the Aurora global database. The `engine`, `engine_version`, and `instance_class` (on the `aws.rds.ClusterInstance`) must together support global databases. See [Using Amazon Aurora global databases](https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/aurora-global-database.html) for more information. By upgrading the engine version, the provider will upgrade cluster members. **NOTE:** To avoid an `inconsistent final plan` error while upgrading, use the `lifecycle` `ignore_changes` for `engine_version` meta argument on the associated `aws.rds.Cluster` resource as shown above in Upgrading Engine Versions example.
 * @property forceDestroy Enable to remove DB Cluster members from Global Cluster on destroy. Required with `source_db_cluster_identifier`.
 * @property globalClusterIdentifier Global cluster identifier.
 * @property sourceDbClusterIdentifier Amazon Resource Name (ARN) to use as the primary DB Cluster of the Global Cluster on creation. The provider cannot perform drift detection of this value.
 * @property storageEncrypted Specifies whether the DB cluster is encrypted. The default is `false` unless `source_db_cluster_identifier` is specified and encrypted. The provider will only perform drift detection if a configuration value is provided.
 */
public data class GlobalClusterArgs(
    public val databaseName: Output<String>? = null,
    public val deletionProtection: Output<Boolean>? = null,
    public val engine: Output<String>? = null,
    public val engineVersion: Output<String>? = null,
    public val forceDestroy: Output<Boolean>? = null,
    public val globalClusterIdentifier: Output<String>? = null,
    public val sourceDbClusterIdentifier: Output<String>? = null,
    public val storageEncrypted: Output<Boolean>? = null,
) : ConvertibleToJava<com.pulumi.aws.rds.GlobalClusterArgs> {
    override fun toJava(): com.pulumi.aws.rds.GlobalClusterArgs =
        com.pulumi.aws.rds.GlobalClusterArgs.builder()
            .databaseName(databaseName?.applyValue({ args0 -> args0 }))
            .deletionProtection(deletionProtection?.applyValue({ args0 -> args0 }))
            .engine(engine?.applyValue({ args0 -> args0 }))
            .engineVersion(engineVersion?.applyValue({ args0 -> args0 }))
            .forceDestroy(forceDestroy?.applyValue({ args0 -> args0 }))
            .globalClusterIdentifier(globalClusterIdentifier?.applyValue({ args0 -> args0 }))
            .sourceDbClusterIdentifier(sourceDbClusterIdentifier?.applyValue({ args0 -> args0 }))
            .storageEncrypted(storageEncrypted?.applyValue({ args0 -> args0 })).build()
}

/**
 * Builder for [GlobalClusterArgs].
 */
@PulumiTagMarker
public class GlobalClusterArgsBuilder internal constructor() {
    private var databaseName: Output<String>? = null

    private var deletionProtection: Output<Boolean>? = null

    private var engine: Output<String>? = null

    private var engineVersion: Output<String>? = null

    private var forceDestroy: Output<Boolean>? = null

    private var globalClusterIdentifier: Output<String>? = null

    private var sourceDbClusterIdentifier: Output<String>? = null

    private var storageEncrypted: Output<Boolean>? = null

    /**
     * @param value Name for an automatically created database on cluster creation.
     */
    @JvmName("juaaqjsvdtmvbtqr")
    public suspend fun databaseName(`value`: Output<String>) {
        this.databaseName = value
    }

    /**
     * @param value If the Global Cluster should have deletion protection enabled. The database can't be deleted when this value is set to `true`. The default is `false`.
     */
    @JvmName("klndbpvgjegyqnlm")
    public suspend fun deletionProtection(`value`: Output<Boolean>) {
        this.deletionProtection = value
    }

    /**
     * @param value Name of the database engine to be used for this DB cluster. The provider will only perform drift detection if a configuration value is provided. Valid values: `aurora`, `aurora-mysql`, `aurora-postgresql`. Defaults to `aurora`. Conflicts with `source_db_cluster_identifier`.
     */
    @JvmName("xltmqethedmqeiav")
    public suspend fun engine(`value`: Output<String>) {
        this.engine = value
    }

    /**
     * @param value Engine version of the Aurora global database. The `engine`, `engine_version`, and `instance_class` (on the `aws.rds.ClusterInstance`) must together support global databases. See [Using Amazon Aurora global databases](https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/aurora-global-database.html) for more information. By upgrading the engine version, the provider will upgrade cluster members. **NOTE:** To avoid an `inconsistent final plan` error while upgrading, use the `lifecycle` `ignore_changes` for `engine_version` meta argument on the associated `aws.rds.Cluster` resource as shown above in Upgrading Engine Versions example.
     */
    @JvmName("wexmurfhhwmxyvfv")
    public suspend fun engineVersion(`value`: Output<String>) {
        this.engineVersion = value
    }

    /**
     * @param value Enable to remove DB Cluster members from Global Cluster on destroy. Required with `source_db_cluster_identifier`.
     */
    @JvmName("fepsgkwbucusmvhh")
    public suspend fun forceDestroy(`value`: Output<Boolean>) {
        this.forceDestroy = value
    }

    /**
     * @param value Global cluster identifier.
     */
    @JvmName("qqiroxvtmyqdosgk")
    public suspend fun globalClusterIdentifier(`value`: Output<String>) {
        this.globalClusterIdentifier = value
    }

    /**
     * @param value Amazon Resource Name (ARN) to use as the primary DB Cluster of the Global Cluster on creation. The provider cannot perform drift detection of this value.
     */
    @JvmName("cvmegvfoituakaws")
    public suspend fun sourceDbClusterIdentifier(`value`: Output<String>) {
        this.sourceDbClusterIdentifier = value
    }

    /**
     * @param value Specifies whether the DB cluster is encrypted. The default is `false` unless `source_db_cluster_identifier` is specified and encrypted. The provider will only perform drift detection if a configuration value is provided.
     */
    @JvmName("ljrmlbxoauefogeb")
    public suspend fun storageEncrypted(`value`: Output<Boolean>) {
        this.storageEncrypted = value
    }

    /**
     * @param value Name for an automatically created database on cluster creation.
     */
    @JvmName("cdoyfdunijivlcum")
    public suspend fun databaseName(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.databaseName = mapped
    }

    /**
     * @param value If the Global Cluster should have deletion protection enabled. The database can't be deleted when this value is set to `true`. The default is `false`.
     */
    @JvmName("dargjysgvvhettjh")
    public suspend fun deletionProtection(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.deletionProtection = mapped
    }

    /**
     * @param value Name of the database engine to be used for this DB cluster. The provider will only perform drift detection if a configuration value is provided. Valid values: `aurora`, `aurora-mysql`, `aurora-postgresql`. Defaults to `aurora`. Conflicts with `source_db_cluster_identifier`.
     */
    @JvmName("ylyrqbkddjwuwwrn")
    public suspend fun engine(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.engine = mapped
    }

    /**
     * @param value Engine version of the Aurora global database. The `engine`, `engine_version`, and `instance_class` (on the `aws.rds.ClusterInstance`) must together support global databases. See [Using Amazon Aurora global databases](https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/aurora-global-database.html) for more information. By upgrading the engine version, the provider will upgrade cluster members. **NOTE:** To avoid an `inconsistent final plan` error while upgrading, use the `lifecycle` `ignore_changes` for `engine_version` meta argument on the associated `aws.rds.Cluster` resource as shown above in Upgrading Engine Versions example.
     */
    @JvmName("wslqglehtprebnmj")
    public suspend fun engineVersion(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.engineVersion = mapped
    }

    /**
     * @param value Enable to remove DB Cluster members from Global Cluster on destroy. Required with `source_db_cluster_identifier`.
     */
    @JvmName("jebggyrslprfwhyn")
    public suspend fun forceDestroy(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.forceDestroy = mapped
    }

    /**
     * @param value Global cluster identifier.
     */
    @JvmName("wyuydnwvmnngxhur")
    public suspend fun globalClusterIdentifier(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.globalClusterIdentifier = mapped
    }

    /**
     * @param value Amazon Resource Name (ARN) to use as the primary DB Cluster of the Global Cluster on creation. The provider cannot perform drift detection of this value.
     */
    @JvmName("dipxgkluftnskkfp")
    public suspend fun sourceDbClusterIdentifier(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.sourceDbClusterIdentifier = mapped
    }

    /**
     * @param value Specifies whether the DB cluster is encrypted. The default is `false` unless `source_db_cluster_identifier` is specified and encrypted. The provider will only perform drift detection if a configuration value is provided.
     */
    @JvmName("mwwedeebviqmftik")
    public suspend fun storageEncrypted(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.storageEncrypted = mapped
    }

    internal fun build(): GlobalClusterArgs = GlobalClusterArgs(
        databaseName = databaseName,
        deletionProtection = deletionProtection,
        engine = engine,
        engineVersion = engineVersion,
        forceDestroy = forceDestroy,
        globalClusterIdentifier = globalClusterIdentifier,
        sourceDbClusterIdentifier = sourceDbClusterIdentifier,
        storageEncrypted = storageEncrypted,
    )
}
