@file:Suppress("NAME_SHADOWING", "DEPRECATION")

package com.pulumi.aws.rds.kotlin

import com.pulumi.aws.rds.InstanceArgs.builder
import com.pulumi.aws.rds.kotlin.enums.InstanceType
import com.pulumi.aws.rds.kotlin.enums.StorageType
import com.pulumi.aws.rds.kotlin.inputs.InstanceBlueGreenUpdateArgs
import com.pulumi.aws.rds.kotlin.inputs.InstanceBlueGreenUpdateArgsBuilder
import com.pulumi.aws.rds.kotlin.inputs.InstanceRestoreToPointInTimeArgs
import com.pulumi.aws.rds.kotlin.inputs.InstanceRestoreToPointInTimeArgsBuilder
import com.pulumi.aws.rds.kotlin.inputs.InstanceS3ImportArgs
import com.pulumi.aws.rds.kotlin.inputs.InstanceS3ImportArgsBuilder
import com.pulumi.core.Either
import com.pulumi.core.Output
import com.pulumi.core.Output.of
import com.pulumi.kotlin.ConvertibleToJava
import com.pulumi.kotlin.PulumiTagMarker
import com.pulumi.kotlin.applySuspend
import kotlin.Boolean
import kotlin.Deprecated
import kotlin.Int
import kotlin.Pair
import kotlin.String
import kotlin.Suppress
import kotlin.Unit
import kotlin.collections.List
import kotlin.collections.Map
import kotlin.jvm.JvmName

/**
 * Provides an RDS instance resource.  A DB instance is an isolated database
 * environment in the cloud.  A DB instance can contain multiple user-created
 * databases.
 * Changes to a DB instance can occur when you manually change a parameter, such as
 * `allocated_storage`, and are reflected in the next maintenance window. Because
 * of this, this provider may report a difference in its planning phase because a
 * modification has not yet taken place. You can use the `apply_immediately` flag
 * to instruct the service to apply the change immediately (see documentation
 * below).
 * When upgrading the major version of an engine, `allow_major_version_upgrade` must be set to `true`.
 * > **Note:** using `apply_immediately` can result in a brief downtime as the server reboots.
 * See the AWS Docs on [RDS Instance Maintenance][instance-maintenance] for more information.
 * > **Note:** All arguments including the username and password will be stored in the raw state as plain-text.
 * Read more about sensitive data instate.
 * ## RDS Instance Class Types
 * Amazon RDS supports three types of instance classes: Standard, Memory Optimized, and Burstable Performance.
 * For more information please read the AWS RDS documentation about [DB Instance Class Types](https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/Concepts.DBInstanceClass.html)
 * ## Low-Downtime Updates
 * By default, RDS applies updates to DB Instances in-place, which can lead to service interruptions.
 * Low-downtime updates minimize service interruptions by performing the updates with an [RDS Blue/Green deployment][blue-green] and switching over the instances when complete.
 * Low-downtime updates are only available for DB Instances using MySQL and MariaDB,
 * as other engines are not supported by RDS Blue/Green deployments.
 * Backups must be enabled to use low-downtime updates.
 * Enable low-downtime updates by setting `blue_green_update.enabled` to `true`.
 * ## Example Usage
 * ### Basic Usage
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.rds.Instance;
 * import com.pulumi.aws.rds.InstanceArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var default_ = new Instance("default", InstanceArgs.builder()
 *             .allocatedStorage(10)
 *             .dbName("mydb")
 *             .engine("mysql")
 *             .engineVersion("5.7")
 *             .instanceClass("db.t3.micro")
 *             .parameterGroupName("default.mysql5.7")
 *             .password("foobarbaz")
 *             .skipFinalSnapshot(true)
 *             .username("foo")
 *             .build());
 *     }
 * }
 * ```
 * ### RDS Custom for Oracle Usage with Replica
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.rds.RdsFunctions;
 * import com.pulumi.aws.rds.inputs.GetOrderableDbInstanceArgs;
 * import com.pulumi.aws.kms.KmsFunctions;
 * import com.pulumi.aws.kms.inputs.GetKeyArgs;
 * import com.pulumi.aws.rds.Instance;
 * import com.pulumi.aws.rds.InstanceArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         final var custom-oracle = RdsFunctions.getOrderableDbInstance(GetOrderableDbInstanceArgs.builder()
 *             .engine("custom-oracle-ee")
 *             .engineVersion("19.c.ee.002")
 *             .licenseModel("bring-your-own-license")
 *             .storageType("gp3")
 *             .preferredInstanceClasses(
 *                 "db.r5.xlarge",
 *                 "db.r5.2xlarge",
 *                 "db.r5.4xlarge")
 *             .build());
 *         final var byId = KmsFunctions.getKey(GetKeyArgs.builder()
 *             .keyId("example-ef278353ceba4a5a97de6784565b9f78")
 *             .build());
 *         var default_ = new Instance("default", InstanceArgs.builder()
 *             .allocatedStorage(50)
 *             .autoMinorVersionUpgrade(false)
 *             .customIamInstanceProfile("AWSRDSCustomInstanceProfile")
 *             .backupRetentionPeriod(7)
 *             .dbSubnetGroupName(local.db_subnet_group_name())
 *             .engine(custom_oracle.engine())
 *             .engineVersion(custom_oracle.engineVersion())
 *             .identifier("ee-instance-demo")
 *             .instanceClass(custom_oracle.instanceClass())
 *             .kmsKeyId(byId.applyValue(getKeyResult -> getKeyResult.arn()))
 *             .licenseModel(custom_oracle.licenseModel())
 *             .multiAz(false)
 *             .password("avoid-plaintext-passwords")
 *             .username("test")
 *             .storageEncrypted(true)
 *             .timeouts(%!v(PANIC=Format method: runtime error: invalid memory address or nil pointer dereference))
 *             .build());
 *         var test_replica = new Instance("test-replica", InstanceArgs.builder()
 *             .replicateSourceDb(default_.identifier())
 *             .replicaMode("mounted")
 *             .autoMinorVersionUpgrade(false)
 *             .customIamInstanceProfile("AWSRDSCustomInstanceProfile")
 *             .backupRetentionPeriod(7)
 *             .identifier("ee-instance-replica")
 *             .instanceClass(custom_oracle.instanceClass())
 *             .kmsKeyId(byId.applyValue(getKeyResult -> getKeyResult.arn()))
 *             .multiAz(false)
 *             .skipFinalSnapshot(true)
 *             .storageEncrypted(true)
 *             .timeouts(%!v(PANIC=Format method: runtime error: invalid memory address or nil pointer dereference))
 *             .build());
 *     }
 * }
 * ```
 * ### RDS Custom for SQL Server
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.rds.RdsFunctions;
 * import com.pulumi.aws.rds.inputs.GetOrderableDbInstanceArgs;
 * import com.pulumi.aws.kms.KmsFunctions;
 * import com.pulumi.aws.kms.inputs.GetKeyArgs;
 * import com.pulumi.aws.rds.Instance;
 * import com.pulumi.aws.rds.InstanceArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         final var custom-sqlserver = RdsFunctions.getOrderableDbInstance(GetOrderableDbInstanceArgs.builder()
 *             .engine("custom-sqlserver-se")
 *             .engineVersion("15.00.4249.2.v1")
 *             .storageType("gp3")
 *             .preferredInstanceClasses(
 *                 "db.r5.xlarge",
 *                 "db.r5.2xlarge",
 *                 "db.r5.4xlarge")
 *             .build());
 *         final var byId = KmsFunctions.getKey(GetKeyArgs.builder()
 *             .keyId("example-ef278353ceba4a5a97de6784565b9f78")
 *             .build());
 *         var example = new Instance("example", InstanceArgs.builder()
 *             .allocatedStorage(500)
 *             .autoMinorVersionUpgrade(false)
 *             .customIamInstanceProfile("AWSRDSCustomSQLServerInstanceRole")
 *             .backupRetentionPeriod(7)
 *             .dbSubnetGroupName(local.db_subnet_group_name())
 *             .engine(custom_sqlserver.engine())
 *             .engineVersion(custom_sqlserver.engineVersion())
 *             .identifier("sql-instance-demo")
 *             .instanceClass(custom_sqlserver.instanceClass())
 *             .kmsKeyId(byId.applyValue(getKeyResult -> getKeyResult.arn()))
 *             .multiAz(false)
 *             .password("avoid-plaintext-passwords")
 *             .storageEncrypted(true)
 *             .username("test")
 *             .timeouts(%!v(PANIC=Format method: runtime error: invalid memory address or nil pointer dereference))
 *             .build());
 *     }
 * }
 * ```
 * ### Storage Autoscaling
 * To enable Storage Autoscaling with instances that support the feature, define the `max_allocated_storage` argument higher than the `allocated_storage` argument. This provider will automatically hide differences with the `allocated_storage` argument value if autoscaling occurs.
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.rds.Instance;
 * import com.pulumi.aws.rds.InstanceArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var example = new Instance("example", InstanceArgs.builder()
 *             .allocatedStorage(50)
 *             .maxAllocatedStorage(100)
 *             .build());
 *     }
 * }
 * ```
 * ### Managed Master Passwords via Secrets Manager, default KMS Key
 * > More information about RDS/Aurora Aurora integrates with Secrets Manager to manage master user passwords for your DB clusters can be found in the [RDS User Guide](https://aws.amazon.com/about-aws/whats-new/2022/12/amazon-rds-integration-aws-secrets-manager/) and [Aurora User Guide](https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/rds-secrets-manager.html).
 * You can specify the `manage_master_user_password` attribute to enable managing the master password with Secrets Manager. You can also update an existing cluster to use Secrets Manager by specify the `manage_master_user_password` attribute and removing the `password` attribute (removal is required).
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.rds.Instance;
 * import com.pulumi.aws.rds.InstanceArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var default_ = new Instance("default", InstanceArgs.builder()
 *             .allocatedStorage(10)
 *             .dbName("mydb")
 *             .engine("mysql")
 *             .engineVersion("5.7")
 *             .instanceClass("db.t3.micro")
 *             .manageMasterUserPassword(true)
 *             .parameterGroupName("default.mysql5.7")
 *             .username("foo")
 *             .build());
 *     }
 * }
 * ```
 * ### Managed Master Passwords via Secrets Manager, specific KMS Key
 * > More information about RDS/Aurora Aurora integrates with Secrets Manager to manage master user passwords for your DB clusters can be found in the [RDS User Guide](https://aws.amazon.com/about-aws/whats-new/2022/12/amazon-rds-integration-aws-secrets-manager/) and [Aurora User Guide](https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/rds-secrets-manager.html).
 * You can specify the `master_user_secret_kms_key_id` attribute to specify a specific KMS Key.
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.kms.Key;
 * import com.pulumi.aws.kms.KeyArgs;
 * import com.pulumi.aws.rds.Instance;
 * import com.pulumi.aws.rds.InstanceArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var example = new Key("example", KeyArgs.builder()
 *             .description("Example KMS Key")
 *             .build());
 *         var default_ = new Instance("default", InstanceArgs.builder()
 *             .allocatedStorage(10)
 *             .dbName("mydb")
 *             .engine("mysql")
 *             .engineVersion("5.7")
 *             .instanceClass("db.t3.micro")
 *             .manageMasterUserPassword(true)
 *             .masterUserSecretKmsKeyId(example.keyId())
 *             .username("foo")
 *             .parameterGroupName("default.mysql5.7")
 *             .build());
 *     }
 * }
 * ```
 * ## Import
 * Using `pulumi import`, import DB Instances using the `identifier`. For example:
 * ```sh
 *  $ pulumi import aws:rds/instance:Instance default mydb-rds-instance
 * ```
 * @property allocatedStorage The allocated storage in gibibytes. If `max_allocated_storage` is configured, this argument represents the initial storage allocation and differences from the configuration will be ignored automatically when Storage Autoscaling occurs. If `replicate_source_db` is set, the value is ignored during the creation of the instance.
 * @property allowMajorVersionUpgrade Indicates that major version
 * upgrades are allowed. Changing this parameter does not result in an outage and
 * the change is asynchronously applied as soon as possible.
 * @property applyImmediately Specifies whether any database modifications
 * are applied immediately, or during the next maintenance window. Default is
 * `false`. See [Amazon RDS Documentation for more
 * information.](https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/Overview.DBInstance.Modifying.html)
 * @property autoMinorVersionUpgrade Indicates that minor engine upgrades
 * will be applied automatically to the DB instance during the maintenance window.
 * Defaults to true.
 * @property availabilityZone The AZ for the RDS instance.
 * @property backupRetentionPeriod The days to retain backups for.
 * Must be between `0` and `35`.
 * Default is `0`.
 * Must be greater than `0` if the database is used as a source for a [Read Replica][instance-replication],
 * uses low-downtime updates,
 * or will use [RDS Blue/Green deployments][blue-green].
 * @property backupTarget Specifies where automated backups and manual snapshots are stored. Possible values are `region` (default) and `outposts`. See [Working with Amazon RDS on AWS Outposts](https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/rds-on-outposts.html) for more information.
 * @property backupWindow The daily time range (in UTC) during which automated backups are created if they are enabled.
 * Example: "09:46-10:16". Must not overlap with `maintenance_window`.
 * @property blueGreenUpdate Enables low-downtime updates using [RDS Blue/Green deployments][blue-green].
 * See `blue_green_update` below.
 * @property caCertIdentifier The identifier of the CA certificate for the DB instance.
 * @property characterSetName The character set name to use for DB
 * encoding in Oracle and Microsoft SQL instances (collation). This can't be changed. See [Oracle Character Sets
 * Supported in Amazon RDS](https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/Appendix.OracleCharacterSets.html)
 * or [Server-Level Collation for Microsoft SQL Server](https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/Appendix.SQLServer.CommonDBATasks.Collation.html) for more information.
 * @property copyTagsToSnapshot Copy all Instance `tags` to snapshots. Default is `false`.
 * @property customIamInstanceProfile The instance profile associated with the underlying Amazon EC2 instance of an RDS Custom DB instance.
 * @property customerOwnedIpEnabled Indicates whether to enable a customer-owned IP address (CoIP) for an RDS on Outposts DB instance. See [CoIP for RDS on Outposts](https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/rds-on-outposts.html#rds-on-outposts.coip) for more information.
 * > **NOTE:** Removing the `replicate_source_db` attribute from an existing RDS
 * Replicate database managed by the provider will promote the database to a fully
 * standalone database.
 * @property dbName The name of the database to create when the DB instance is created. If this parameter is not specified, no database is created in the DB instance. Note that this does not apply for Oracle or SQL Server engines. See the [AWS documentation](https://awscli.amazonaws.com/v2/documentation/api/latest/reference/rds/create-db-instance.html) for more details on what applies for those engines. If you are providing an Oracle db name, it needs to be in all upper case. Cannot be specified for a replica.
 * @property dbSubnetGroupName Name of DB subnet group. DB instance will
 * be created in the VPC associated with the DB subnet group. If unspecified, will
 * be created in the `default` VPC, or in EC2 Classic, if available. When working
 * with read replicas, it should be specified only if the source database
 * specifies an instance in another AWS Region. See [DBSubnetGroupName in API
 * action CreateDBInstanceReadReplica](https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_CreateDBInstanceReadReplica.html)
 * for additional read replica constraints.
 * @property deleteAutomatedBackups Specifies whether to remove automated backups immediately after the DB instance is deleted. Default is `true`.
 * @property deletionProtection If the DB instance should have deletion protection enabled. The database can't be deleted when this value is set to `true`. The default is `false`.
 * @property domain The ID of the Directory Service Active Directory domain to create the instance in.
 * @property domainIamRoleName The name of the IAM role to be used when making API calls to the Directory Service.
 * @property enabledCloudwatchLogsExports Set of log types to enable for exporting to CloudWatch logs. If omitted, no logs will be exported. Valid values (depending on `engine`). MySQL and MariaDB: `audit`, `error`, `general`, `slowquery`. PostgreSQL: `postgresql`, `upgrade`. MSSQL: `agent` , `error`. Oracle: `alert`, `audit`, `listener`, `trace`.
 * @property engine The database engine to use. For supported values, see the Engine parameter in [API action CreateDBInstance](https://docs&#46;aws&#46;amazon&#46;com/AmazonRDS/latest/APIReference/API_CreateDBInstance&#46;html)&#46; Note that for Amazon Aurora instances the engine must match the DB cluster's engine'&#46; For information on the difference between the available Aurora MySQL engines see [Comparison between Aurora MySQL 1 and Aurora MySQL 2](https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/AuroraMySQL.Updates.20180206.html) in the Amazon RDS User Guide.
 * @property engineVersion The engine version to use. If `auto_minor_version_upgrade` is enabled, you can provide a prefix of the version such as `5.7` (for `5.7.10`). The actual engine version used is returned in the attribute `engine_version_actual`, see Attribute Reference below. For supported values, see the EngineVersion parameter in [API action CreateDBInstance](https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_CreateDBInstance.html). Note that for Amazon Aurora instances the engine version must match the DB cluster's engine version'.
 * @property finalSnapshotIdentifier The name of your final DB snapshot
 * when this DB instance is deleted. Must be provided if `skip_final_snapshot` is
 * set to `false`. The value must begin with a letter, only contain alphanumeric characters and hyphens, and not end with a hyphen or contain two consecutive hyphens. Must not be provided when deleting a read replica.
 * @property iamDatabaseAuthenticationEnabled Specifies whether mappings of AWS Identity and Access Management (IAM) accounts to database
 * accounts is enabled.
 * @property identifier The name of the RDS instance, if omitted, this provider will assign a random, unique identifier. Required if `restore_to_point_in_time` is specified.
 * @property identifierPrefix Creates a unique identifier beginning with the specified prefix. Conflicts with `identifier`.
 * @property instanceClass The instance type of the RDS instance.
 * @property iops The amount of provisioned IOPS. Setting this implies a
 * storage_type of "io1". Can only be set when `storage_type` is `"io1"` or `"gp3"`.
 * Cannot be specified for gp3 storage if the `allocated_storage` value is below a per-`engine` threshold.
 * See the [RDS User Guide](https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/CHAP_Storage.html#gp3-storage) for details.
 * @property kmsKeyId The ARN for the KMS encryption key. If creating an
 * encrypted replica, set this to the destination KMS ARN.
 * @property licenseModel License model information for this DB instance. Valid values for this field are as follows:
 * * RDS for MariaDB: `general-public-license`
 * * RDS for Microsoft SQL Server: `license-included`
 * * RDS for MySQL: `general-public-license`
 * * RDS for Oracle: `bring-your-own-license | license-included`
 * * RDS for PostgreSQL: `postgresql-license`
 * @property maintenanceWindow The window to perform maintenance in.
 * Syntax: "ddd:hh24:mi-ddd:hh24:mi". Eg: "Mon:00:00-Mon:03:00". See [RDS
 * Maintenance Window
 * docs](http://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_UpgradeDBInstance.Maintenance.html#AdjustingTheMaintenanceWindow)
 * for more information.
 * @property manageMasterUserPassword Set to true to allow RDS to manage the master user password in Secrets Manager. Cannot be set if `password` is provided.
 * @property masterUserSecretKmsKeyId The Amazon Web Services KMS key identifier is the key ARN, key ID, alias ARN, or alias name for the KMS key. To use a KMS key in a different Amazon Web Services account, specify the key ARN or alias ARN. If not specified, the default KMS key for your Amazon Web Services account is used.
 * @property maxAllocatedStorage When configured, the upper limit to which Amazon RDS can automatically scale the storage of the DB instance. Configuring this will automatically ignore differences to `allocated_storage`. Must be greater than or equal to `allocated_storage` or `0` to disable Storage Autoscaling.
 * @property monitoringInterval The interval, in seconds, between points
 * when Enhanced Monitoring metrics are collected for the DB instance. To disable
 * collecting Enhanced Monitoring metrics, specify 0. The default is 0. Valid
 * Values: 0, 1, 5, 10, 15, 30, 60.
 * @property monitoringRoleArn The ARN for the IAM role that permits RDS
 * to send enhanced monitoring metrics to CloudWatch Logs. You can find more
 * information on the [AWS
 * Documentation](https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_Monitoring.html)
 * what IAM permissions are needed to allow Enhanced Monitoring for RDS Instances.
 * @property multiAz Specifies if the RDS instance is multi-AZ
 * @property name
 * @property ncharCharacterSetName The national character set is used in the NCHAR, NVARCHAR2, and NCLOB data types for Oracle instances. This can't be changed. See [Oracle Character Sets
 * Supported in Amazon RDS](https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/Appendix.OracleCharacterSets.html).
 * @property networkType The network type of the DB instance. Valid values: `IPV4`, `DUAL`.
 * @property optionGroupName Name of the DB option group to associate.
 * @property parameterGroupName Name of the DB parameter group to
 * associate.
 * @property password (Required unless `manage_master_user_password` is set to true or unless a `snapshot_identifier` or `replicate_source_db`
 * is provided or `manage_master_user_password` is set.) Password for the master DB user. Note that this may show up in
 * logs, and it will be stored in the state file. Cannot be set if `manage_master_user_password` is set to `true`.
 * @property performanceInsightsEnabled Specifies whether Performance Insights are enabled. Defaults to false.
 * @property performanceInsightsKmsKeyId The ARN for the KMS key to encrypt Performance Insights data. When specifying `performance_insights_kms_key_id`, `performance_insights_enabled` needs to be set to true. Once KMS key is set, it can never be changed.
 * @property performanceInsightsRetentionPeriod Amount of time in days to retain Performance Insights data. Valid values are `7`, `731` (2 years) or a multiple of `31`. When specifying `performance_insights_retention_period`, `performance_insights_enabled` needs to be set to true. Defaults to '7'.
 * @property port The port on which the DB accepts connections.
 * @property publiclyAccessible Bool to control if instance is publicly
 * accessible. Default is `false`.
 * @property replicaMode Specifies whether the replica is in either `mounted` or `open-read-only` mode. This attribute
 * is only supported by Oracle instances. Oracle replicas operate in `open-read-only` mode unless otherwise specified. See [Working with Oracle Read Replicas](https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/oracle-read-replicas.html) for more information.
 * @property replicateSourceDb Specifies that this resource is a Replicate
 * database, and to use this value as the source database. This correlates to the
 * `identifier` of another Amazon RDS Database to replicate (if replicating within
 * a single region) or ARN of the Amazon RDS Database to replicate (if replicating
 * cross-region). Note that if you are
 * creating a cross-region replica of an encrypted database you will also need to
 * specify a `kms_key_id`. See [DB Instance Replication][instance-replication] and [Working with
 * PostgreSQL and MySQL Read Replicas](https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_ReadRepl.html)
 * for more information on using Replication.
 * @property restoreToPointInTime A configuration block for restoring a DB instance to an arbitrary point in time. Requires the `identifier` argument to be set with the name of the new DB instance to be created. See Restore To Point In Time below for details.
 * @property s3Import Restore from a Percona Xtrabackup in S3.  See [Importing Data into an Amazon RDS MySQL DB Instance](http://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/MySQL.Procedural.Importing.html)
 * @property skipFinalSnapshot Determines whether a final DB snapshot is
 * created before the DB instance is deleted. If true is specified, no DBSnapshot
 * is created. If false is specified, a DB snapshot is created before the DB
 * instance is deleted, using the value from `final_snapshot_identifier`. Default
 * is `false`.
 * @property snapshotIdentifier Specifies whether or not to create this
 * database from a snapshot. This correlates to the snapshot ID you'd find in the
 * RDS console, e.g: rds:production-2015-06-26-06-05.
 * @property storageEncrypted Specifies whether the DB instance is
 * encrypted. Note that if you are creating a cross-region read replica this field
 * is ignored and you should instead declare `kms_key_id` with a valid ARN. The
 * default is `false` if not specified.
 * @property storageThroughput The storage throughput value for the DB instance. Can only be set when `storage_type` is `"gp3"`. Cannot be specified if the `allocated_storage` value is below a per-`engine` threshold. See the [RDS User Guide](https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/CHAP_Storage.html#gp3-storage) for details.
 * @property storageType One of "standard" (magnetic), "gp2" (general
 * purpose SSD), "gp3" (general purpose SSD that needs `iops` independently)
 * or "io1" (provisioned IOPS SSD). The default is "io1" if `iops` is specified,
 * "gp2" if not.
 * @property tags A map of tags to assign to the resource. If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
 * @property timezone Time zone of the DB instance. `timezone` is currently
 * only supported by Microsoft SQL Server. The `timezone` can only be set on
 * creation. See [MSSQL User
 * Guide](http://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/CHAP_SQLServer.html#SQLServer.Concepts.General.TimeZone)
 * for more information.
 * @property username (Required unless a `snapshot_identifier` or `replicate_source_db`
 * is provided) Username for the master DB user. Cannot be specified for a replica.
 * @property vpcSecurityGroupIds List of VPC security groups to
 * associate.
 */
public data class InstanceArgs(
    public val allocatedStorage: Output<Int>? = null,
    public val allowMajorVersionUpgrade: Output<Boolean>? = null,
    public val applyImmediately: Output<Boolean>? = null,
    public val autoMinorVersionUpgrade: Output<Boolean>? = null,
    public val availabilityZone: Output<String>? = null,
    public val backupRetentionPeriod: Output<Int>? = null,
    public val backupTarget: Output<String>? = null,
    public val backupWindow: Output<String>? = null,
    public val blueGreenUpdate: Output<InstanceBlueGreenUpdateArgs>? = null,
    public val caCertIdentifier: Output<String>? = null,
    public val characterSetName: Output<String>? = null,
    public val copyTagsToSnapshot: Output<Boolean>? = null,
    public val customIamInstanceProfile: Output<String>? = null,
    public val customerOwnedIpEnabled: Output<Boolean>? = null,
    public val dbName: Output<String>? = null,
    public val dbSubnetGroupName: Output<String>? = null,
    public val deleteAutomatedBackups: Output<Boolean>? = null,
    public val deletionProtection: Output<Boolean>? = null,
    public val domain: Output<String>? = null,
    public val domainIamRoleName: Output<String>? = null,
    public val enabledCloudwatchLogsExports: Output<List<String>>? = null,
    public val engine: Output<String>? = null,
    public val engineVersion: Output<String>? = null,
    public val finalSnapshotIdentifier: Output<String>? = null,
    public val iamDatabaseAuthenticationEnabled: Output<Boolean>? = null,
    public val identifier: Output<String>? = null,
    public val identifierPrefix: Output<String>? = null,
    public val instanceClass: Output<Either<String, InstanceType>>? = null,
    public val iops: Output<Int>? = null,
    public val kmsKeyId: Output<String>? = null,
    public val licenseModel: Output<String>? = null,
    public val maintenanceWindow: Output<String>? = null,
    public val manageMasterUserPassword: Output<Boolean>? = null,
    public val masterUserSecretKmsKeyId: Output<String>? = null,
    public val maxAllocatedStorage: Output<Int>? = null,
    public val monitoringInterval: Output<Int>? = null,
    public val monitoringRoleArn: Output<String>? = null,
    public val multiAz: Output<Boolean>? = null,
    @Deprecated(
        message = """
  This property has been deprecated. Please use 'dbName' instead.
  """,
    )
    public val name: Output<String>? = null,
    public val ncharCharacterSetName: Output<String>? = null,
    public val networkType: Output<String>? = null,
    public val optionGroupName: Output<String>? = null,
    public val parameterGroupName: Output<String>? = null,
    public val password: Output<String>? = null,
    public val performanceInsightsEnabled: Output<Boolean>? = null,
    public val performanceInsightsKmsKeyId: Output<String>? = null,
    public val performanceInsightsRetentionPeriod: Output<Int>? = null,
    public val port: Output<Int>? = null,
    public val publiclyAccessible: Output<Boolean>? = null,
    public val replicaMode: Output<String>? = null,
    public val replicateSourceDb: Output<String>? = null,
    public val restoreToPointInTime: Output<InstanceRestoreToPointInTimeArgs>? = null,
    public val s3Import: Output<InstanceS3ImportArgs>? = null,
    public val skipFinalSnapshot: Output<Boolean>? = null,
    public val snapshotIdentifier: Output<String>? = null,
    public val storageEncrypted: Output<Boolean>? = null,
    public val storageThroughput: Output<Int>? = null,
    public val storageType: Output<Either<String, StorageType>>? = null,
    public val tags: Output<Map<String, String>>? = null,
    public val timezone: Output<String>? = null,
    public val username: Output<String>? = null,
    public val vpcSecurityGroupIds: Output<List<String>>? = null,
) : ConvertibleToJava<com.pulumi.aws.rds.InstanceArgs> {
    override fun toJava(): com.pulumi.aws.rds.InstanceArgs = com.pulumi.aws.rds.InstanceArgs.builder()
        .allocatedStorage(allocatedStorage?.applyValue({ args0 -> args0 }))
        .allowMajorVersionUpgrade(allowMajorVersionUpgrade?.applyValue({ args0 -> args0 }))
        .applyImmediately(applyImmediately?.applyValue({ args0 -> args0 }))
        .autoMinorVersionUpgrade(autoMinorVersionUpgrade?.applyValue({ args0 -> args0 }))
        .availabilityZone(availabilityZone?.applyValue({ args0 -> args0 }))
        .backupRetentionPeriod(backupRetentionPeriod?.applyValue({ args0 -> args0 }))
        .backupTarget(backupTarget?.applyValue({ args0 -> args0 }))
        .backupWindow(backupWindow?.applyValue({ args0 -> args0 }))
        .blueGreenUpdate(blueGreenUpdate?.applyValue({ args0 -> args0.let({ args0 -> args0.toJava() }) }))
        .caCertIdentifier(caCertIdentifier?.applyValue({ args0 -> args0 }))
        .characterSetName(characterSetName?.applyValue({ args0 -> args0 }))
        .copyTagsToSnapshot(copyTagsToSnapshot?.applyValue({ args0 -> args0 }))
        .customIamInstanceProfile(customIamInstanceProfile?.applyValue({ args0 -> args0 }))
        .customerOwnedIpEnabled(customerOwnedIpEnabled?.applyValue({ args0 -> args0 }))
        .dbName(dbName?.applyValue({ args0 -> args0 }))
        .dbSubnetGroupName(dbSubnetGroupName?.applyValue({ args0 -> args0 }))
        .deleteAutomatedBackups(deleteAutomatedBackups?.applyValue({ args0 -> args0 }))
        .deletionProtection(deletionProtection?.applyValue({ args0 -> args0 }))
        .domain(domain?.applyValue({ args0 -> args0 }))
        .domainIamRoleName(domainIamRoleName?.applyValue({ args0 -> args0 }))
        .enabledCloudwatchLogsExports(
            enabledCloudwatchLogsExports?.applyValue({ args0 ->
                args0.map({ args0 -> args0 })
            }),
        )
        .engine(engine?.applyValue({ args0 -> args0 }))
        .engineVersion(engineVersion?.applyValue({ args0 -> args0 }))
        .finalSnapshotIdentifier(finalSnapshotIdentifier?.applyValue({ args0 -> args0 }))
        .iamDatabaseAuthenticationEnabled(iamDatabaseAuthenticationEnabled?.applyValue({ args0 -> args0 }))
        .identifier(identifier?.applyValue({ args0 -> args0 }))
        .identifierPrefix(identifierPrefix?.applyValue({ args0 -> args0 }))
        .instanceClass(
            instanceClass?.applyValue({ args0 ->
                args0.transform({ args0 -> args0 }, { args0 ->
                    args0.let({ args0 -> args0.toJava() })
                })
            }),
        )
        .iops(iops?.applyValue({ args0 -> args0 }))
        .kmsKeyId(kmsKeyId?.applyValue({ args0 -> args0 }))
        .licenseModel(licenseModel?.applyValue({ args0 -> args0 }))
        .maintenanceWindow(maintenanceWindow?.applyValue({ args0 -> args0 }))
        .manageMasterUserPassword(manageMasterUserPassword?.applyValue({ args0 -> args0 }))
        .masterUserSecretKmsKeyId(masterUserSecretKmsKeyId?.applyValue({ args0 -> args0 }))
        .maxAllocatedStorage(maxAllocatedStorage?.applyValue({ args0 -> args0 }))
        .monitoringInterval(monitoringInterval?.applyValue({ args0 -> args0 }))
        .monitoringRoleArn(monitoringRoleArn?.applyValue({ args0 -> args0 }))
        .multiAz(multiAz?.applyValue({ args0 -> args0 }))
        .name(name?.applyValue({ args0 -> args0 }))
        .ncharCharacterSetName(ncharCharacterSetName?.applyValue({ args0 -> args0 }))
        .networkType(networkType?.applyValue({ args0 -> args0 }))
        .optionGroupName(optionGroupName?.applyValue({ args0 -> args0 }))
        .parameterGroupName(parameterGroupName?.applyValue({ args0 -> args0 }))
        .password(password?.applyValue({ args0 -> args0 }))
        .performanceInsightsEnabled(performanceInsightsEnabled?.applyValue({ args0 -> args0 }))
        .performanceInsightsKmsKeyId(performanceInsightsKmsKeyId?.applyValue({ args0 -> args0 }))
        .performanceInsightsRetentionPeriod(
            performanceInsightsRetentionPeriod?.applyValue({ args0 ->
                args0
            }),
        )
        .port(port?.applyValue({ args0 -> args0 }))
        .publiclyAccessible(publiclyAccessible?.applyValue({ args0 -> args0 }))
        .replicaMode(replicaMode?.applyValue({ args0 -> args0 }))
        .replicateSourceDb(replicateSourceDb?.applyValue({ args0 -> args0 }))
        .restoreToPointInTime(
            restoreToPointInTime?.applyValue({ args0 ->
                args0.let({ args0 ->
                    args0.toJava()
                })
            }),
        )
        .s3Import(s3Import?.applyValue({ args0 -> args0.let({ args0 -> args0.toJava() }) }))
        .skipFinalSnapshot(skipFinalSnapshot?.applyValue({ args0 -> args0 }))
        .snapshotIdentifier(snapshotIdentifier?.applyValue({ args0 -> args0 }))
        .storageEncrypted(storageEncrypted?.applyValue({ args0 -> args0 }))
        .storageThroughput(storageThroughput?.applyValue({ args0 -> args0 }))
        .storageType(
            storageType?.applyValue({ args0 ->
                args0.transform({ args0 -> args0 }, { args0 ->
                    args0.let({ args0 -> args0.toJava() })
                })
            }),
        )
        .tags(tags?.applyValue({ args0 -> args0.map({ args0 -> args0.key.to(args0.value) }).toMap() }))
        .timezone(timezone?.applyValue({ args0 -> args0 }))
        .username(username?.applyValue({ args0 -> args0 }))
        .vpcSecurityGroupIds(
            vpcSecurityGroupIds?.applyValue({ args0 ->
                args0.map({ args0 ->
                    args0
                })
            }),
        ).build()
}

/**
 * Builder for [InstanceArgs].
 */
@PulumiTagMarker
public class InstanceArgsBuilder internal constructor() {
    private var allocatedStorage: Output<Int>? = null

    private var allowMajorVersionUpgrade: Output<Boolean>? = null

    private var applyImmediately: Output<Boolean>? = null

    private var autoMinorVersionUpgrade: Output<Boolean>? = null

    private var availabilityZone: Output<String>? = null

    private var backupRetentionPeriod: Output<Int>? = null

    private var backupTarget: Output<String>? = null

    private var backupWindow: Output<String>? = null

    private var blueGreenUpdate: Output<InstanceBlueGreenUpdateArgs>? = null

    private var caCertIdentifier: Output<String>? = null

    private var characterSetName: Output<String>? = null

    private var copyTagsToSnapshot: Output<Boolean>? = null

    private var customIamInstanceProfile: Output<String>? = null

    private var customerOwnedIpEnabled: Output<Boolean>? = null

    private var dbName: Output<String>? = null

    private var dbSubnetGroupName: Output<String>? = null

    private var deleteAutomatedBackups: Output<Boolean>? = null

    private var deletionProtection: Output<Boolean>? = null

    private var domain: Output<String>? = null

    private var domainIamRoleName: Output<String>? = null

    private var enabledCloudwatchLogsExports: Output<List<String>>? = null

    private var engine: Output<String>? = null

    private var engineVersion: Output<String>? = null

    private var finalSnapshotIdentifier: Output<String>? = null

    private var iamDatabaseAuthenticationEnabled: Output<Boolean>? = null

    private var identifier: Output<String>? = null

    private var identifierPrefix: Output<String>? = null

    private var instanceClass: Output<Either<String, InstanceType>>? = null

    private var iops: Output<Int>? = null

    private var kmsKeyId: Output<String>? = null

    private var licenseModel: Output<String>? = null

    private var maintenanceWindow: Output<String>? = null

    private var manageMasterUserPassword: Output<Boolean>? = null

    private var masterUserSecretKmsKeyId: Output<String>? = null

    private var maxAllocatedStorage: Output<Int>? = null

    private var monitoringInterval: Output<Int>? = null

    private var monitoringRoleArn: Output<String>? = null

    private var multiAz: Output<Boolean>? = null

    private var name: Output<String>? = null

    private var ncharCharacterSetName: Output<String>? = null

    private var networkType: Output<String>? = null

    private var optionGroupName: Output<String>? = null

    private var parameterGroupName: Output<String>? = null

    private var password: Output<String>? = null

    private var performanceInsightsEnabled: Output<Boolean>? = null

    private var performanceInsightsKmsKeyId: Output<String>? = null

    private var performanceInsightsRetentionPeriod: Output<Int>? = null

    private var port: Output<Int>? = null

    private var publiclyAccessible: Output<Boolean>? = null

    private var replicaMode: Output<String>? = null

    private var replicateSourceDb: Output<String>? = null

    private var restoreToPointInTime: Output<InstanceRestoreToPointInTimeArgs>? = null

    private var s3Import: Output<InstanceS3ImportArgs>? = null

    private var skipFinalSnapshot: Output<Boolean>? = null

    private var snapshotIdentifier: Output<String>? = null

    private var storageEncrypted: Output<Boolean>? = null

    private var storageThroughput: Output<Int>? = null

    private var storageType: Output<Either<String, StorageType>>? = null

    private var tags: Output<Map<String, String>>? = null

    private var timezone: Output<String>? = null

    private var username: Output<String>? = null

    private var vpcSecurityGroupIds: Output<List<String>>? = null

    /**
     * @param value The allocated storage in gibibytes. If `max_allocated_storage` is configured, this argument represents the initial storage allocation and differences from the configuration will be ignored automatically when Storage Autoscaling occurs. If `replicate_source_db` is set, the value is ignored during the creation of the instance.
     */
    @JvmName("vysjjpykmmbbycap")
    public suspend fun allocatedStorage(`value`: Output<Int>) {
        this.allocatedStorage = value
    }

    /**
     * @param value Indicates that major version
     * upgrades are allowed. Changing this parameter does not result in an outage and
     * the change is asynchronously applied as soon as possible.
     */
    @JvmName("kradqaxmydfqlddx")
    public suspend fun allowMajorVersionUpgrade(`value`: Output<Boolean>) {
        this.allowMajorVersionUpgrade = value
    }

    /**
     * @param value Specifies whether any database modifications
     * are applied immediately, or during the next maintenance window. Default is
     * `false`. See [Amazon RDS Documentation for more
   * information.](https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/Overview.DBInstance.Modifying.html)
     */
    @JvmName("wfnnytflwcnhkqoo")
    public suspend fun applyImmediately(`value`: Output<Boolean>) {
        this.applyImmediately = value
    }

    /**
     * @param value Indicates that minor engine upgrades
     * will be applied automatically to the DB instance during the maintenance window.
     * Defaults to true.
     */
    @JvmName("qjwuxwpuyrroupgb")
    public suspend fun autoMinorVersionUpgrade(`value`: Output<Boolean>) {
        this.autoMinorVersionUpgrade = value
    }

    /**
     * @param value The AZ for the RDS instance.
     */
    @JvmName("kbmxogimyvnmxhvg")
    public suspend fun availabilityZone(`value`: Output<String>) {
        this.availabilityZone = value
    }

    /**
     * @param value The days to retain backups for.
     * Must be between `0` and `35`.
     * Default is `0`.
     * Must be greater than `0` if the database is used as a source for a [Read Replica][instance-replication],
     * uses low-downtime updates,
     * or will use [RDS Blue/Green deployments][blue-green].
     */
    @JvmName("scwmgmwdgsrmvtns")
    public suspend fun backupRetentionPeriod(`value`: Output<Int>) {
        this.backupRetentionPeriod = value
    }

    /**
     * @param value Specifies where automated backups and manual snapshots are stored. Possible values are `region` (default) and `outposts`. See [Working with Amazon RDS on AWS Outposts](https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/rds-on-outposts.html) for more information.
     */
    @JvmName("snoirukoxekgdjqs")
    public suspend fun backupTarget(`value`: Output<String>) {
        this.backupTarget = value
    }

    /**
     * @param value The daily time range (in UTC) during which automated backups are created if they are enabled.
     * Example: "09:46-10:16". Must not overlap with `maintenance_window`.
     */
    @JvmName("qdbupirnuxxgcpqd")
    public suspend fun backupWindow(`value`: Output<String>) {
        this.backupWindow = value
    }

    /**
     * @param value Enables low-downtime updates using [RDS Blue/Green deployments][blue-green].
     * See `blue_green_update` below.
     */
    @JvmName("kevrneyermkuolbl")
    public suspend fun blueGreenUpdate(`value`: Output<InstanceBlueGreenUpdateArgs>) {
        this.blueGreenUpdate = value
    }

    /**
     * @param value The identifier of the CA certificate for the DB instance.
     */
    @JvmName("rwwosxjhpebrcafw")
    public suspend fun caCertIdentifier(`value`: Output<String>) {
        this.caCertIdentifier = value
    }

    /**
     * @param value The character set name to use for DB
     * encoding in Oracle and Microsoft SQL instances (collation). This can't be changed. See [Oracle Character Sets
   * Supported in Amazon RDS](https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/Appendix.OracleCharacterSets.html)
     * or [Server-Level Collation for Microsoft SQL Server](https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/Appendix.SQLServer.CommonDBATasks.Collation.html) for more information.
     */
    @JvmName("mkjwqwtlhwoulvhy")
    public suspend fun characterSetName(`value`: Output<String>) {
        this.characterSetName = value
    }

    /**
     * @param value Copy all Instance `tags` to snapshots. Default is `false`.
     */
    @JvmName("cdlghykojneqwble")
    public suspend fun copyTagsToSnapshot(`value`: Output<Boolean>) {
        this.copyTagsToSnapshot = value
    }

    /**
     * @param value The instance profile associated with the underlying Amazon EC2 instance of an RDS Custom DB instance.
     */
    @JvmName("qidapskymvyljymi")
    public suspend fun customIamInstanceProfile(`value`: Output<String>) {
        this.customIamInstanceProfile = value
    }

    /**
     * @param value Indicates whether to enable a customer-owned IP address (CoIP) for an RDS on Outposts DB instance. See [CoIP for RDS on Outposts](https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/rds-on-outposts.html#rds-on-outposts.coip) for more information.
     * > **NOTE:** Removing the `replicate_source_db` attribute from an existing RDS
     * Replicate database managed by the provider will promote the database to a fully
     * standalone database.
     */
    @JvmName("jacajmaremkynkoc")
    public suspend fun customerOwnedIpEnabled(`value`: Output<Boolean>) {
        this.customerOwnedIpEnabled = value
    }

    /**
     * @param value The name of the database to create when the DB instance is created. If this parameter is not specified, no database is created in the DB instance. Note that this does not apply for Oracle or SQL Server engines. See the [AWS documentation](https://awscli.amazonaws.com/v2/documentation/api/latest/reference/rds/create-db-instance.html) for more details on what applies for those engines. If you are providing an Oracle db name, it needs to be in all upper case. Cannot be specified for a replica.
     */
    @JvmName("rsmfipvkqheeiwmq")
    public suspend fun dbName(`value`: Output<String>) {
        this.dbName = value
    }

    /**
     * @param value Name of DB subnet group. DB instance will
     * be created in the VPC associated with the DB subnet group. If unspecified, will
     * be created in the `default` VPC, or in EC2 Classic, if available. When working
     * with read replicas, it should be specified only if the source database
     * specifies an instance in another AWS Region. See [DBSubnetGroupName in API
   * action CreateDBInstanceReadReplica](https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_CreateDBInstanceReadReplica.html)
     * for additional read replica constraints.
     */
    @JvmName("qqdljijakgehycpq")
    public suspend fun dbSubnetGroupName(`value`: Output<String>) {
        this.dbSubnetGroupName = value
    }

    /**
     * @param value Specifies whether to remove automated backups immediately after the DB instance is deleted. Default is `true`.
     */
    @JvmName("rejoyivgsouodkrn")
    public suspend fun deleteAutomatedBackups(`value`: Output<Boolean>) {
        this.deleteAutomatedBackups = value
    }

    /**
     * @param value If the DB instance should have deletion protection enabled. The database can't be deleted when this value is set to `true`. The default is `false`.
     */
    @JvmName("rxyccffftnaucmuh")
    public suspend fun deletionProtection(`value`: Output<Boolean>) {
        this.deletionProtection = value
    }

    /**
     * @param value The ID of the Directory Service Active Directory domain to create the instance in.
     */
    @JvmName("ypnttenhecrynujv")
    public suspend fun domain(`value`: Output<String>) {
        this.domain = value
    }

    /**
     * @param value The name of the IAM role to be used when making API calls to the Directory Service.
     */
    @JvmName("diejcgeprqtoeqcg")
    public suspend fun domainIamRoleName(`value`: Output<String>) {
        this.domainIamRoleName = value
    }

    /**
     * @param value Set of log types to enable for exporting to CloudWatch logs. If omitted, no logs will be exported. Valid values (depending on `engine`). MySQL and MariaDB: `audit`, `error`, `general`, `slowquery`. PostgreSQL: `postgresql`, `upgrade`. MSSQL: `agent` , `error`. Oracle: `alert`, `audit`, `listener`, `trace`.
     */
    @JvmName("rberfcnwijwsysoq")
    public suspend fun enabledCloudwatchLogsExports(`value`: Output<List<String>>) {
        this.enabledCloudwatchLogsExports = value
    }

    @JvmName("tlltqdgnjbrrqjju")
    public suspend fun enabledCloudwatchLogsExports(vararg values: Output<String>) {
        this.enabledCloudwatchLogsExports = Output.all(values.asList())
    }

    /**
     * @param values Set of log types to enable for exporting to CloudWatch logs. If omitted, no logs will be exported. Valid values (depending on `engine`). MySQL and MariaDB: `audit`, `error`, `general`, `slowquery`. PostgreSQL: `postgresql`, `upgrade`. MSSQL: `agent` , `error`. Oracle: `alert`, `audit`, `listener`, `trace`.
     */
    @JvmName("siqunykwroxhmtve")
    public suspend fun enabledCloudwatchLogsExports(values: List<Output<String>>) {
        this.enabledCloudwatchLogsExports = Output.all(values)
    }

    /**
     * @param value The database engine to use. For supported values, see the Engine parameter in [API action CreateDBInstance](https://docs&#46;aws&#46;amazon&#46;com/AmazonRDS/latest/APIReference/API_CreateDBInstance&#46;html)&#46; Note that for Amazon Aurora instances the engine must match the DB cluster's engine'&#46; For information on the difference between the available Aurora MySQL engines see [Comparison between Aurora MySQL 1 and Aurora MySQL 2](https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/AuroraMySQL.Updates.20180206.html) in the Amazon RDS User Guide.
     */
    @JvmName("rfhyldelbvmauqge")
    public suspend fun engine(`value`: Output<String>) {
        this.engine = value
    }

    /**
     * @param value The engine version to use. If `auto_minor_version_upgrade` is enabled, you can provide a prefix of the version such as `5.7` (for `5.7.10`). The actual engine version used is returned in the attribute `engine_version_actual`, see Attribute Reference below. For supported values, see the EngineVersion parameter in [API action CreateDBInstance](https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_CreateDBInstance.html). Note that for Amazon Aurora instances the engine version must match the DB cluster's engine version'.
     */
    @JvmName("dxivdlyarsmrdnlr")
    public suspend fun engineVersion(`value`: Output<String>) {
        this.engineVersion = value
    }

    /**
     * @param value The name of your final DB snapshot
     * when this DB instance is deleted. Must be provided if `skip_final_snapshot` is
     * set to `false`. The value must begin with a letter, only contain alphanumeric characters and hyphens, and not end with a hyphen or contain two consecutive hyphens. Must not be provided when deleting a read replica.
     */
    @JvmName("begjslbqocfdwjtn")
    public suspend fun finalSnapshotIdentifier(`value`: Output<String>) {
        this.finalSnapshotIdentifier = value
    }

    /**
     * @param value Specifies whether mappings of AWS Identity and Access Management (IAM) accounts to database
     * accounts is enabled.
     */
    @JvmName("ojxqhjvqsyxjanoy")
    public suspend fun iamDatabaseAuthenticationEnabled(`value`: Output<Boolean>) {
        this.iamDatabaseAuthenticationEnabled = value
    }

    /**
     * @param value The name of the RDS instance, if omitted, this provider will assign a random, unique identifier. Required if `restore_to_point_in_time` is specified.
     */
    @JvmName("ewussfokqlqtyqvk")
    public suspend fun identifier(`value`: Output<String>) {
        this.identifier = value
    }

    /**
     * @param value Creates a unique identifier beginning with the specified prefix. Conflicts with `identifier`.
     */
    @JvmName("ptgqcfqpeotmhsbc")
    public suspend fun identifierPrefix(`value`: Output<String>) {
        this.identifierPrefix = value
    }

    /**
     * @param value The instance type of the RDS instance.
     */
    @JvmName("tjskpekxvudxmpnn")
    public suspend fun instanceClass(`value`: Output<Either<String, InstanceType>>) {
        this.instanceClass = value
    }

    /**
     * @param value The amount of provisioned IOPS. Setting this implies a
     * storage_type of "io1". Can only be set when `storage_type` is `"io1"` or `"gp3"`.
     * Cannot be specified for gp3 storage if the `allocated_storage` value is below a per-`engine` threshold.
     * See the [RDS User Guide](https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/CHAP_Storage.html#gp3-storage) for details.
     */
    @JvmName("ehpfxdnmabiookyv")
    public suspend fun iops(`value`: Output<Int>) {
        this.iops = value
    }

    /**
     * @param value The ARN for the KMS encryption key. If creating an
     * encrypted replica, set this to the destination KMS ARN.
     */
    @JvmName("sxpbkucptanpwayb")
    public suspend fun kmsKeyId(`value`: Output<String>) {
        this.kmsKeyId = value
    }

    /**
     * @param value License model information for this DB instance. Valid values for this field are as follows:
     * * RDS for MariaDB: `general-public-license`
     * * RDS for Microsoft SQL Server: `license-included`
     * * RDS for MySQL: `general-public-license`
     * * RDS for Oracle: `bring-your-own-license | license-included`
     * * RDS for PostgreSQL: `postgresql-license`
     */
    @JvmName("lwgkhqmyvwqqvjvi")
    public suspend fun licenseModel(`value`: Output<String>) {
        this.licenseModel = value
    }

    /**
     * @param value The window to perform maintenance in.
     * Syntax: "ddd:hh24:mi-ddd:hh24:mi". Eg: "Mon:00:00-Mon:03:00". See [RDS
   * Maintenance Window
   * docs](http://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_UpgradeDBInstance.Maintenance.html#AdjustingTheMaintenanceWindow)
     * for more information.
     */
    @JvmName("isanmbdquqxppqqp")
    public suspend fun maintenanceWindow(`value`: Output<String>) {
        this.maintenanceWindow = value
    }

    /**
     * @param value Set to true to allow RDS to manage the master user password in Secrets Manager. Cannot be set if `password` is provided.
     */
    @JvmName("maoyqcloyjsdhsxy")
    public suspend fun manageMasterUserPassword(`value`: Output<Boolean>) {
        this.manageMasterUserPassword = value
    }

    /**
     * @param value The Amazon Web Services KMS key identifier is the key ARN, key ID, alias ARN, or alias name for the KMS key. To use a KMS key in a different Amazon Web Services account, specify the key ARN or alias ARN. If not specified, the default KMS key for your Amazon Web Services account is used.
     */
    @JvmName("mrqisivoaifvgirt")
    public suspend fun masterUserSecretKmsKeyId(`value`: Output<String>) {
        this.masterUserSecretKmsKeyId = value
    }

    /**
     * @param value When configured, the upper limit to which Amazon RDS can automatically scale the storage of the DB instance. Configuring this will automatically ignore differences to `allocated_storage`. Must be greater than or equal to `allocated_storage` or `0` to disable Storage Autoscaling.
     */
    @JvmName("xowdbnkgnscoxbrb")
    public suspend fun maxAllocatedStorage(`value`: Output<Int>) {
        this.maxAllocatedStorage = value
    }

    /**
     * @param value The interval, in seconds, between points
     * when Enhanced Monitoring metrics are collected for the DB instance. To disable
     * collecting Enhanced Monitoring metrics, specify 0. The default is 0. Valid
     * Values: 0, 1, 5, 10, 15, 30, 60.
     */
    @JvmName("xqjssuwumafivett")
    public suspend fun monitoringInterval(`value`: Output<Int>) {
        this.monitoringInterval = value
    }

    /**
     * @param value The ARN for the IAM role that permits RDS
     * to send enhanced monitoring metrics to CloudWatch Logs. You can find more
     * information on the [AWS
   * Documentation](https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_Monitoring.html)
     * what IAM permissions are needed to allow Enhanced Monitoring for RDS Instances.
     */
    @JvmName("iscanhmjlmtyadqv")
    public suspend fun monitoringRoleArn(`value`: Output<String>) {
        this.monitoringRoleArn = value
    }

    /**
     * @param value Specifies if the RDS instance is multi-AZ
     */
    @JvmName("fovwygforrghrbgq")
    public suspend fun multiAz(`value`: Output<Boolean>) {
        this.multiAz = value
    }

    /**
     * @param value
     */
    @Deprecated(
        message = """
  This property has been deprecated. Please use 'dbName' instead.
  """,
    )
    @JvmName("ryvdurysveyapmqc")
    public suspend fun name(`value`: Output<String>) {
        this.name = value
    }

    /**
     * @param value The national character set is used in the NCHAR, NVARCHAR2, and NCLOB data types for Oracle instances. This can't be changed. See [Oracle Character Sets
   * Supported in Amazon RDS](https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/Appendix.OracleCharacterSets.html).
     */
    @JvmName("xsobrybeyrfddkie")
    public suspend fun ncharCharacterSetName(`value`: Output<String>) {
        this.ncharCharacterSetName = value
    }

    /**
     * @param value The network type of the DB instance. Valid values: `IPV4`, `DUAL`.
     */
    @JvmName("oihmiskloyyrnese")
    public suspend fun networkType(`value`: Output<String>) {
        this.networkType = value
    }

    /**
     * @param value Name of the DB option group to associate.
     */
    @JvmName("xmdrclalerkbgvgl")
    public suspend fun optionGroupName(`value`: Output<String>) {
        this.optionGroupName = value
    }

    /**
     * @param value Name of the DB parameter group to
     * associate.
     */
    @JvmName("uhqemixlnxbkwqkk")
    public suspend fun parameterGroupName(`value`: Output<String>) {
        this.parameterGroupName = value
    }

    /**
     * @param value (Required unless `manage_master_user_password` is set to true or unless a `snapshot_identifier` or `replicate_source_db`
     * is provided or `manage_master_user_password` is set.) Password for the master DB user. Note that this may show up in
     * logs, and it will be stored in the state file. Cannot be set if `manage_master_user_password` is set to `true`.
     */
    @JvmName("ifrlrdjwelojtpid")
    public suspend fun password(`value`: Output<String>) {
        this.password = value
    }

    /**
     * @param value Specifies whether Performance Insights are enabled. Defaults to false.
     */
    @JvmName("shhlgwjptktoatvi")
    public suspend fun performanceInsightsEnabled(`value`: Output<Boolean>) {
        this.performanceInsightsEnabled = value
    }

    /**
     * @param value The ARN for the KMS key to encrypt Performance Insights data. When specifying `performance_insights_kms_key_id`, `performance_insights_enabled` needs to be set to true. Once KMS key is set, it can never be changed.
     */
    @JvmName("cstdojwhgjmrhhhu")
    public suspend fun performanceInsightsKmsKeyId(`value`: Output<String>) {
        this.performanceInsightsKmsKeyId = value
    }

    /**
     * @param value Amount of time in days to retain Performance Insights data. Valid values are `7`, `731` (2 years) or a multiple of `31`. When specifying `performance_insights_retention_period`, `performance_insights_enabled` needs to be set to true. Defaults to '7'.
     */
    @JvmName("tvqnbnitrsbbpidp")
    public suspend fun performanceInsightsRetentionPeriod(`value`: Output<Int>) {
        this.performanceInsightsRetentionPeriod = value
    }

    /**
     * @param value The port on which the DB accepts connections.
     */
    @JvmName("rmirsplpnptnsavl")
    public suspend fun port(`value`: Output<Int>) {
        this.port = value
    }

    /**
     * @param value Bool to control if instance is publicly
     * accessible. Default is `false`.
     */
    @JvmName("llklwmrmuinjhufq")
    public suspend fun publiclyAccessible(`value`: Output<Boolean>) {
        this.publiclyAccessible = value
    }

    /**
     * @param value Specifies whether the replica is in either `mounted` or `open-read-only` mode. This attribute
     * is only supported by Oracle instances. Oracle replicas operate in `open-read-only` mode unless otherwise specified. See [Working with Oracle Read Replicas](https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/oracle-read-replicas.html) for more information.
     */
    @JvmName("shhcexyofjjnmavo")
    public suspend fun replicaMode(`value`: Output<String>) {
        this.replicaMode = value
    }

    /**
     * @param value Specifies that this resource is a Replicate
     * database, and to use this value as the source database. This correlates to the
     * `identifier` of another Amazon RDS Database to replicate (if replicating within
     * a single region) or ARN of the Amazon RDS Database to replicate (if replicating
     * cross-region). Note that if you are
     * creating a cross-region replica of an encrypted database you will also need to
     * specify a `kms_key_id`. See [DB Instance Replication][instance-replication] and [Working with
   * PostgreSQL and MySQL Read Replicas](https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_ReadRepl.html)
     * for more information on using Replication.
     */
    @JvmName("wbjqsfgjdqflnxdo")
    public suspend fun replicateSourceDb(`value`: Output<String>) {
        this.replicateSourceDb = value
    }

    /**
     * @param value A configuration block for restoring a DB instance to an arbitrary point in time. Requires the `identifier` argument to be set with the name of the new DB instance to be created. See Restore To Point In Time below for details.
     */
    @JvmName("jvfkswdosaeeghxj")
    public suspend fun restoreToPointInTime(`value`: Output<InstanceRestoreToPointInTimeArgs>) {
        this.restoreToPointInTime = value
    }

    /**
     * @param value Restore from a Percona Xtrabackup in S3.  See [Importing Data into an Amazon RDS MySQL DB Instance](http://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/MySQL.Procedural.Importing.html)
     */
    @JvmName("qkqampophdefftaa")
    public suspend fun s3Import(`value`: Output<InstanceS3ImportArgs>) {
        this.s3Import = value
    }

    /**
     * @param value Determines whether a final DB snapshot is
     * created before the DB instance is deleted. If true is specified, no DBSnapshot
     * is created. If false is specified, a DB snapshot is created before the DB
     * instance is deleted, using the value from `final_snapshot_identifier`. Default
     * is `false`.
     */
    @JvmName("vnbbrxraobrdxylq")
    public suspend fun skipFinalSnapshot(`value`: Output<Boolean>) {
        this.skipFinalSnapshot = value
    }

    /**
     * @param value Specifies whether or not to create this
     * database from a snapshot. This correlates to the snapshot ID you'd find in the
     * RDS console, e.g: rds:production-2015-06-26-06-05.
     */
    @JvmName("lfbqprmkhefswgtd")
    public suspend fun snapshotIdentifier(`value`: Output<String>) {
        this.snapshotIdentifier = value
    }

    /**
     * @param value Specifies whether the DB instance is
     * encrypted. Note that if you are creating a cross-region read replica this field
     * is ignored and you should instead declare `kms_key_id` with a valid ARN. The
     * default is `false` if not specified.
     */
    @JvmName("avunkwvxyfiajcan")
    public suspend fun storageEncrypted(`value`: Output<Boolean>) {
        this.storageEncrypted = value
    }

    /**
     * @param value The storage throughput value for the DB instance. Can only be set when `storage_type` is `"gp3"`. Cannot be specified if the `allocated_storage` value is below a per-`engine` threshold. See the [RDS User Guide](https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/CHAP_Storage.html#gp3-storage) for details.
     */
    @JvmName("gqylxcfeknblytgn")
    public suspend fun storageThroughput(`value`: Output<Int>) {
        this.storageThroughput = value
    }

    /**
     * @param value One of "standard" (magnetic), "gp2" (general
     * purpose SSD), "gp3" (general purpose SSD that needs `iops` independently)
     * or "io1" (provisioned IOPS SSD). The default is "io1" if `iops` is specified,
     * "gp2" if not.
     */
    @JvmName("hcjmmfticdnvajvu")
    public suspend fun storageType(`value`: Output<Either<String, StorageType>>) {
        this.storageType = value
    }

    /**
     * @param value A map of tags to assign to the resource. If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
     */
    @JvmName("omkveclstcpiusuu")
    public suspend fun tags(`value`: Output<Map<String, String>>) {
        this.tags = value
    }

    /**
     * @param value Time zone of the DB instance. `timezone` is currently
     * only supported by Microsoft SQL Server. The `timezone` can only be set on
     * creation. See [MSSQL User
   * Guide](http://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/CHAP_SQLServer.html#SQLServer.Concepts.General.TimeZone)
     * for more information.
     */
    @JvmName("luwubukmudkmoeew")
    public suspend fun timezone(`value`: Output<String>) {
        this.timezone = value
    }

    /**
     * @param value (Required unless a `snapshot_identifier` or `replicate_source_db`
     * is provided) Username for the master DB user. Cannot be specified for a replica.
     */
    @JvmName("kebmuhswdgsfbdmd")
    public suspend fun username(`value`: Output<String>) {
        this.username = value
    }

    /**
     * @param value List of VPC security groups to
     * associate.
     */
    @JvmName("xkiktvfnfnkqcpfc")
    public suspend fun vpcSecurityGroupIds(`value`: Output<List<String>>) {
        this.vpcSecurityGroupIds = value
    }

    @JvmName("cvvimbgprnivfgax")
    public suspend fun vpcSecurityGroupIds(vararg values: Output<String>) {
        this.vpcSecurityGroupIds = Output.all(values.asList())
    }

    /**
     * @param values List of VPC security groups to
     * associate.
     */
    @JvmName("xqcswwcpcufcgiwx")
    public suspend fun vpcSecurityGroupIds(values: List<Output<String>>) {
        this.vpcSecurityGroupIds = Output.all(values)
    }

    /**
     * @param value The allocated storage in gibibytes. If `max_allocated_storage` is configured, this argument represents the initial storage allocation and differences from the configuration will be ignored automatically when Storage Autoscaling occurs. If `replicate_source_db` is set, the value is ignored during the creation of the instance.
     */
    @JvmName("qntmiyxcwycwcyfj")
    public suspend fun allocatedStorage(`value`: Int?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.allocatedStorage = mapped
    }

    /**
     * @param value Indicates that major version
     * upgrades are allowed. Changing this parameter does not result in an outage and
     * the change is asynchronously applied as soon as possible.
     */
    @JvmName("qsffyrnikinifdjf")
    public suspend fun allowMajorVersionUpgrade(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.allowMajorVersionUpgrade = mapped
    }

    /**
     * @param value Specifies whether any database modifications
     * are applied immediately, or during the next maintenance window. Default is
     * `false`. See [Amazon RDS Documentation for more
   * information.](https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/Overview.DBInstance.Modifying.html)
     */
    @JvmName("ojscemygwyvtuoha")
    public suspend fun applyImmediately(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.applyImmediately = mapped
    }

    /**
     * @param value Indicates that minor engine upgrades
     * will be applied automatically to the DB instance during the maintenance window.
     * Defaults to true.
     */
    @JvmName("flbstcgqolsteaat")
    public suspend fun autoMinorVersionUpgrade(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.autoMinorVersionUpgrade = mapped
    }

    /**
     * @param value The AZ for the RDS instance.
     */
    @JvmName("fptmknpcicnyslti")
    public suspend fun availabilityZone(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.availabilityZone = mapped
    }

    /**
     * @param value The days to retain backups for.
     * Must be between `0` and `35`.
     * Default is `0`.
     * Must be greater than `0` if the database is used as a source for a [Read Replica][instance-replication],
     * uses low-downtime updates,
     * or will use [RDS Blue/Green deployments][blue-green].
     */
    @JvmName("vlrteeffkrftnpie")
    public suspend fun backupRetentionPeriod(`value`: Int?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.backupRetentionPeriod = mapped
    }

    /**
     * @param value Specifies where automated backups and manual snapshots are stored. Possible values are `region` (default) and `outposts`. See [Working with Amazon RDS on AWS Outposts](https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/rds-on-outposts.html) for more information.
     */
    @JvmName("qkuxltibppiktjou")
    public suspend fun backupTarget(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.backupTarget = mapped
    }

    /**
     * @param value The daily time range (in UTC) during which automated backups are created if they are enabled.
     * Example: "09:46-10:16". Must not overlap with `maintenance_window`.
     */
    @JvmName("kjdaxpahtadvgvlj")
    public suspend fun backupWindow(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.backupWindow = mapped
    }

    /**
     * @param value Enables low-downtime updates using [RDS Blue/Green deployments][blue-green].
     * See `blue_green_update` below.
     */
    @JvmName("fvsaxwlqkmgyvktd")
    public suspend fun blueGreenUpdate(`value`: InstanceBlueGreenUpdateArgs?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.blueGreenUpdate = mapped
    }

    /**
     * @param argument Enables low-downtime updates using [RDS Blue/Green deployments][blue-green].
     * See `blue_green_update` below.
     */
    @JvmName("skuogorxhuntvnqx")
    public suspend
    fun blueGreenUpdate(argument: suspend InstanceBlueGreenUpdateArgsBuilder.() -> Unit) {
        val toBeMapped = InstanceBlueGreenUpdateArgsBuilder().applySuspend { argument() }.build()
        val mapped = of(toBeMapped)
        this.blueGreenUpdate = mapped
    }

    /**
     * @param value The identifier of the CA certificate for the DB instance.
     */
    @JvmName("jauodrrbxwqwdxnn")
    public suspend fun caCertIdentifier(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.caCertIdentifier = mapped
    }

    /**
     * @param value The character set name to use for DB
     * encoding in Oracle and Microsoft SQL instances (collation). This can't be changed. See [Oracle Character Sets
   * Supported in Amazon RDS](https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/Appendix.OracleCharacterSets.html)
     * or [Server-Level Collation for Microsoft SQL Server](https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/Appendix.SQLServer.CommonDBATasks.Collation.html) for more information.
     */
    @JvmName("hjvywwqrfwrqhdix")
    public suspend fun characterSetName(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.characterSetName = mapped
    }

    /**
     * @param value Copy all Instance `tags` to snapshots. Default is `false`.
     */
    @JvmName("ighxlaqtmjwmqqms")
    public suspend fun copyTagsToSnapshot(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.copyTagsToSnapshot = mapped
    }

    /**
     * @param value The instance profile associated with the underlying Amazon EC2 instance of an RDS Custom DB instance.
     */
    @JvmName("lcvwfpgnjdguuedo")
    public suspend fun customIamInstanceProfile(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.customIamInstanceProfile = mapped
    }

    /**
     * @param value Indicates whether to enable a customer-owned IP address (CoIP) for an RDS on Outposts DB instance. See [CoIP for RDS on Outposts](https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/rds-on-outposts.html#rds-on-outposts.coip) for more information.
     * > **NOTE:** Removing the `replicate_source_db` attribute from an existing RDS
     * Replicate database managed by the provider will promote the database to a fully
     * standalone database.
     */
    @JvmName("rlythbjrsklunwft")
    public suspend fun customerOwnedIpEnabled(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.customerOwnedIpEnabled = mapped
    }

    /**
     * @param value The name of the database to create when the DB instance is created. If this parameter is not specified, no database is created in the DB instance. Note that this does not apply for Oracle or SQL Server engines. See the [AWS documentation](https://awscli.amazonaws.com/v2/documentation/api/latest/reference/rds/create-db-instance.html) for more details on what applies for those engines. If you are providing an Oracle db name, it needs to be in all upper case. Cannot be specified for a replica.
     */
    @JvmName("ywsnnporlhjmkqlh")
    public suspend fun dbName(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.dbName = mapped
    }

    /**
     * @param value Name of DB subnet group. DB instance will
     * be created in the VPC associated with the DB subnet group. If unspecified, will
     * be created in the `default` VPC, or in EC2 Classic, if available. When working
     * with read replicas, it should be specified only if the source database
     * specifies an instance in another AWS Region. See [DBSubnetGroupName in API
   * action CreateDBInstanceReadReplica](https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_CreateDBInstanceReadReplica.html)
     * for additional read replica constraints.
     */
    @JvmName("vvkbacgmudupigcc")
    public suspend fun dbSubnetGroupName(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.dbSubnetGroupName = mapped
    }

    /**
     * @param value Specifies whether to remove automated backups immediately after the DB instance is deleted. Default is `true`.
     */
    @JvmName("eixkegtorahwibfo")
    public suspend fun deleteAutomatedBackups(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.deleteAutomatedBackups = mapped
    }

    /**
     * @param value If the DB instance should have deletion protection enabled. The database can't be deleted when this value is set to `true`. The default is `false`.
     */
    @JvmName("aidupqnrrjfwqgyd")
    public suspend fun deletionProtection(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.deletionProtection = mapped
    }

    /**
     * @param value The ID of the Directory Service Active Directory domain to create the instance in.
     */
    @JvmName("nwssvdynojcxruel")
    public suspend fun domain(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.domain = mapped
    }

    /**
     * @param value The name of the IAM role to be used when making API calls to the Directory Service.
     */
    @JvmName("icgifvcdocnldauc")
    public suspend fun domainIamRoleName(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.domainIamRoleName = mapped
    }

    /**
     * @param value Set of log types to enable for exporting to CloudWatch logs. If omitted, no logs will be exported. Valid values (depending on `engine`). MySQL and MariaDB: `audit`, `error`, `general`, `slowquery`. PostgreSQL: `postgresql`, `upgrade`. MSSQL: `agent` , `error`. Oracle: `alert`, `audit`, `listener`, `trace`.
     */
    @JvmName("nmdfspbbfofbmxor")
    public suspend fun enabledCloudwatchLogsExports(`value`: List<String>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.enabledCloudwatchLogsExports = mapped
    }

    /**
     * @param values Set of log types to enable for exporting to CloudWatch logs. If omitted, no logs will be exported. Valid values (depending on `engine`). MySQL and MariaDB: `audit`, `error`, `general`, `slowquery`. PostgreSQL: `postgresql`, `upgrade`. MSSQL: `agent` , `error`. Oracle: `alert`, `audit`, `listener`, `trace`.
     */
    @JvmName("pottjnkfxtvklyqi")
    public suspend fun enabledCloudwatchLogsExports(vararg values: String) {
        val toBeMapped = values.toList()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.enabledCloudwatchLogsExports = mapped
    }

    /**
     * @param value The database engine to use. For supported values, see the Engine parameter in [API action CreateDBInstance](https://docs&#46;aws&#46;amazon&#46;com/AmazonRDS/latest/APIReference/API_CreateDBInstance&#46;html)&#46; Note that for Amazon Aurora instances the engine must match the DB cluster's engine'&#46; For information on the difference between the available Aurora MySQL engines see [Comparison between Aurora MySQL 1 and Aurora MySQL 2](https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/AuroraMySQL.Updates.20180206.html) in the Amazon RDS User Guide.
     */
    @JvmName("xierofnbeqwrjmjg")
    public suspend fun engine(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.engine = mapped
    }

    /**
     * @param value The engine version to use. If `auto_minor_version_upgrade` is enabled, you can provide a prefix of the version such as `5.7` (for `5.7.10`). The actual engine version used is returned in the attribute `engine_version_actual`, see Attribute Reference below. For supported values, see the EngineVersion parameter in [API action CreateDBInstance](https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_CreateDBInstance.html). Note that for Amazon Aurora instances the engine version must match the DB cluster's engine version'.
     */
    @JvmName("jkcqpbnnbyochcqq")
    public suspend fun engineVersion(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.engineVersion = mapped
    }

    /**
     * @param value The name of your final DB snapshot
     * when this DB instance is deleted. Must be provided if `skip_final_snapshot` is
     * set to `false`. The value must begin with a letter, only contain alphanumeric characters and hyphens, and not end with a hyphen or contain two consecutive hyphens. Must not be provided when deleting a read replica.
     */
    @JvmName("wivusbkpmcbhgquv")
    public suspend fun finalSnapshotIdentifier(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.finalSnapshotIdentifier = mapped
    }

    /**
     * @param value Specifies whether mappings of AWS Identity and Access Management (IAM) accounts to database
     * accounts is enabled.
     */
    @JvmName("lyvcomhsfkkbxihi")
    public suspend fun iamDatabaseAuthenticationEnabled(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.iamDatabaseAuthenticationEnabled = mapped
    }

    /**
     * @param value The name of the RDS instance, if omitted, this provider will assign a random, unique identifier. Required if `restore_to_point_in_time` is specified.
     */
    @JvmName("eagcqraxcopbgcid")
    public suspend fun identifier(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.identifier = mapped
    }

    /**
     * @param value Creates a unique identifier beginning with the specified prefix. Conflicts with `identifier`.
     */
    @JvmName("nehjetkmwafnkxss")
    public suspend fun identifierPrefix(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.identifierPrefix = mapped
    }

    /**
     * @param value The instance type of the RDS instance.
     */
    @JvmName("fjclctfltsgimswn")
    public suspend fun instanceClass(`value`: Either<String, InstanceType>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.instanceClass = mapped
    }

    /**
     * @param value The instance type of the RDS instance.
     */
    @JvmName("calpkdrdvmtcxcvx")
    public fun instanceClass(`value`: String) {
        val toBeMapped = Either.ofLeft<String, InstanceType>(value)
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.instanceClass = mapped
    }

    /**
     * @param value The instance type of the RDS instance.
     */
    @JvmName("hfippypidvjvctet")
    public fun instanceClass(`value`: InstanceType) {
        val toBeMapped = Either.ofRight<String, InstanceType>(value)
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.instanceClass = mapped
    }

    /**
     * @param value The amount of provisioned IOPS. Setting this implies a
     * storage_type of "io1". Can only be set when `storage_type` is `"io1"` or `"gp3"`.
     * Cannot be specified for gp3 storage if the `allocated_storage` value is below a per-`engine` threshold.
     * See the [RDS User Guide](https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/CHAP_Storage.html#gp3-storage) for details.
     */
    @JvmName("hcicaehdddgmdraf")
    public suspend fun iops(`value`: Int?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.iops = mapped
    }

    /**
     * @param value The ARN for the KMS encryption key. If creating an
     * encrypted replica, set this to the destination KMS ARN.
     */
    @JvmName("gvxikvawgwqtiyey")
    public suspend fun kmsKeyId(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.kmsKeyId = mapped
    }

    /**
     * @param value License model information for this DB instance. Valid values for this field are as follows:
     * * RDS for MariaDB: `general-public-license`
     * * RDS for Microsoft SQL Server: `license-included`
     * * RDS for MySQL: `general-public-license`
     * * RDS for Oracle: `bring-your-own-license | license-included`
     * * RDS for PostgreSQL: `postgresql-license`
     */
    @JvmName("hfddalqghhnoxyyh")
    public suspend fun licenseModel(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.licenseModel = mapped
    }

    /**
     * @param value The window to perform maintenance in.
     * Syntax: "ddd:hh24:mi-ddd:hh24:mi". Eg: "Mon:00:00-Mon:03:00". See [RDS
   * Maintenance Window
   * docs](http://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_UpgradeDBInstance.Maintenance.html#AdjustingTheMaintenanceWindow)
     * for more information.
     */
    @JvmName("ncgwvkhtkqokbtpk")
    public suspend fun maintenanceWindow(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.maintenanceWindow = mapped
    }

    /**
     * @param value Set to true to allow RDS to manage the master user password in Secrets Manager. Cannot be set if `password` is provided.
     */
    @JvmName("kfbhwfjqureywgoi")
    public suspend fun manageMasterUserPassword(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.manageMasterUserPassword = mapped
    }

    /**
     * @param value The Amazon Web Services KMS key identifier is the key ARN, key ID, alias ARN, or alias name for the KMS key. To use a KMS key in a different Amazon Web Services account, specify the key ARN or alias ARN. If not specified, the default KMS key for your Amazon Web Services account is used.
     */
    @JvmName("aywuwjpipfwiekqv")
    public suspend fun masterUserSecretKmsKeyId(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.masterUserSecretKmsKeyId = mapped
    }

    /**
     * @param value When configured, the upper limit to which Amazon RDS can automatically scale the storage of the DB instance. Configuring this will automatically ignore differences to `allocated_storage`. Must be greater than or equal to `allocated_storage` or `0` to disable Storage Autoscaling.
     */
    @JvmName("dqlatcsvugbqdnwo")
    public suspend fun maxAllocatedStorage(`value`: Int?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.maxAllocatedStorage = mapped
    }

    /**
     * @param value The interval, in seconds, between points
     * when Enhanced Monitoring metrics are collected for the DB instance. To disable
     * collecting Enhanced Monitoring metrics, specify 0. The default is 0. Valid
     * Values: 0, 1, 5, 10, 15, 30, 60.
     */
    @JvmName("jgxvfxjwixjblhoa")
    public suspend fun monitoringInterval(`value`: Int?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.monitoringInterval = mapped
    }

    /**
     * @param value The ARN for the IAM role that permits RDS
     * to send enhanced monitoring metrics to CloudWatch Logs. You can find more
     * information on the [AWS
   * Documentation](https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_Monitoring.html)
     * what IAM permissions are needed to allow Enhanced Monitoring for RDS Instances.
     */
    @JvmName("gbevxkhvbscgpuxh")
    public suspend fun monitoringRoleArn(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.monitoringRoleArn = mapped
    }

    /**
     * @param value Specifies if the RDS instance is multi-AZ
     */
    @JvmName("grkfefnwjvqqevnr")
    public suspend fun multiAz(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.multiAz = mapped
    }

    /**
     * @param value
     */
    @Deprecated(
        message = """
  This property has been deprecated. Please use 'dbName' instead.
  """,
    )
    @JvmName("mbjqtcmrouciidwg")
    public suspend fun name(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.name = mapped
    }

    /**
     * @param value The national character set is used in the NCHAR, NVARCHAR2, and NCLOB data types for Oracle instances. This can't be changed. See [Oracle Character Sets
   * Supported in Amazon RDS](https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/Appendix.OracleCharacterSets.html).
     */
    @JvmName("twerpchkeisujiuk")
    public suspend fun ncharCharacterSetName(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.ncharCharacterSetName = mapped
    }

    /**
     * @param value The network type of the DB instance. Valid values: `IPV4`, `DUAL`.
     */
    @JvmName("dveigpybjuchyapk")
    public suspend fun networkType(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.networkType = mapped
    }

    /**
     * @param value Name of the DB option group to associate.
     */
    @JvmName("imrkavbxdgjmhlyq")
    public suspend fun optionGroupName(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.optionGroupName = mapped
    }

    /**
     * @param value Name of the DB parameter group to
     * associate.
     */
    @JvmName("mwxsuperwyhfgcxa")
    public suspend fun parameterGroupName(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.parameterGroupName = mapped
    }

    /**
     * @param value (Required unless `manage_master_user_password` is set to true or unless a `snapshot_identifier` or `replicate_source_db`
     * is provided or `manage_master_user_password` is set.) Password for the master DB user. Note that this may show up in
     * logs, and it will be stored in the state file. Cannot be set if `manage_master_user_password` is set to `true`.
     */
    @JvmName("klijginqlemnbsae")
    public suspend fun password(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.password = mapped
    }

    /**
     * @param value Specifies whether Performance Insights are enabled. Defaults to false.
     */
    @JvmName("xuvukvrkawpnllfd")
    public suspend fun performanceInsightsEnabled(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.performanceInsightsEnabled = mapped
    }

    /**
     * @param value The ARN for the KMS key to encrypt Performance Insights data. When specifying `performance_insights_kms_key_id`, `performance_insights_enabled` needs to be set to true. Once KMS key is set, it can never be changed.
     */
    @JvmName("hohpnvoitmjjuudg")
    public suspend fun performanceInsightsKmsKeyId(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.performanceInsightsKmsKeyId = mapped
    }

    /**
     * @param value Amount of time in days to retain Performance Insights data. Valid values are `7`, `731` (2 years) or a multiple of `31`. When specifying `performance_insights_retention_period`, `performance_insights_enabled` needs to be set to true. Defaults to '7'.
     */
    @JvmName("btrkalkyocpiukpe")
    public suspend fun performanceInsightsRetentionPeriod(`value`: Int?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.performanceInsightsRetentionPeriod = mapped
    }

    /**
     * @param value The port on which the DB accepts connections.
     */
    @JvmName("jhsnwkhbtapsiadh")
    public suspend fun port(`value`: Int?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.port = mapped
    }

    /**
     * @param value Bool to control if instance is publicly
     * accessible. Default is `false`.
     */
    @JvmName("tgohpjhgmmbclegu")
    public suspend fun publiclyAccessible(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.publiclyAccessible = mapped
    }

    /**
     * @param value Specifies whether the replica is in either `mounted` or `open-read-only` mode. This attribute
     * is only supported by Oracle instances. Oracle replicas operate in `open-read-only` mode unless otherwise specified. See [Working with Oracle Read Replicas](https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/oracle-read-replicas.html) for more information.
     */
    @JvmName("hfhesmikfpqldlcm")
    public suspend fun replicaMode(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.replicaMode = mapped
    }

    /**
     * @param value Specifies that this resource is a Replicate
     * database, and to use this value as the source database. This correlates to the
     * `identifier` of another Amazon RDS Database to replicate (if replicating within
     * a single region) or ARN of the Amazon RDS Database to replicate (if replicating
     * cross-region). Note that if you are
     * creating a cross-region replica of an encrypted database you will also need to
     * specify a `kms_key_id`. See [DB Instance Replication][instance-replication] and [Working with
   * PostgreSQL and MySQL Read Replicas](https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_ReadRepl.html)
     * for more information on using Replication.
     */
    @JvmName("kslufeqrlunawyrf")
    public suspend fun replicateSourceDb(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.replicateSourceDb = mapped
    }

    /**
     * @param value A configuration block for restoring a DB instance to an arbitrary point in time. Requires the `identifier` argument to be set with the name of the new DB instance to be created. See Restore To Point In Time below for details.
     */
    @JvmName("qmsueuongddvpdgg")
    public suspend fun restoreToPointInTime(`value`: InstanceRestoreToPointInTimeArgs?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.restoreToPointInTime = mapped
    }

    /**
     * @param argument A configuration block for restoring a DB instance to an arbitrary point in time. Requires the `identifier` argument to be set with the name of the new DB instance to be created. See Restore To Point In Time below for details.
     */
    @JvmName("jeflbtpsjqvvnuah")
    public suspend
    fun restoreToPointInTime(argument: suspend InstanceRestoreToPointInTimeArgsBuilder.() -> Unit) {
        val toBeMapped = InstanceRestoreToPointInTimeArgsBuilder().applySuspend { argument() }.build()
        val mapped = of(toBeMapped)
        this.restoreToPointInTime = mapped
    }

    /**
     * @param value Restore from a Percona Xtrabackup in S3.  See [Importing Data into an Amazon RDS MySQL DB Instance](http://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/MySQL.Procedural.Importing.html)
     */
    @JvmName("cbjnhgwrcdhggusi")
    public suspend fun s3Import(`value`: InstanceS3ImportArgs?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.s3Import = mapped
    }

    /**
     * @param argument Restore from a Percona Xtrabackup in S3.  See [Importing Data into an Amazon RDS MySQL DB Instance](http://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/MySQL.Procedural.Importing.html)
     */
    @JvmName("mhuvgrxgxjqkayga")
    public suspend fun s3Import(argument: suspend InstanceS3ImportArgsBuilder.() -> Unit) {
        val toBeMapped = InstanceS3ImportArgsBuilder().applySuspend { argument() }.build()
        val mapped = of(toBeMapped)
        this.s3Import = mapped
    }

    /**
     * @param value Determines whether a final DB snapshot is
     * created before the DB instance is deleted. If true is specified, no DBSnapshot
     * is created. If false is specified, a DB snapshot is created before the DB
     * instance is deleted, using the value from `final_snapshot_identifier`. Default
     * is `false`.
     */
    @JvmName("pafnebispmrwbcoy")
    public suspend fun skipFinalSnapshot(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.skipFinalSnapshot = mapped
    }

    /**
     * @param value Specifies whether or not to create this
     * database from a snapshot. This correlates to the snapshot ID you'd find in the
     * RDS console, e.g: rds:production-2015-06-26-06-05.
     */
    @JvmName("kwashyiihhbjquwu")
    public suspend fun snapshotIdentifier(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.snapshotIdentifier = mapped
    }

    /**
     * @param value Specifies whether the DB instance is
     * encrypted. Note that if you are creating a cross-region read replica this field
     * is ignored and you should instead declare `kms_key_id` with a valid ARN. The
     * default is `false` if not specified.
     */
    @JvmName("tunaykrxmmwqjpai")
    public suspend fun storageEncrypted(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.storageEncrypted = mapped
    }

    /**
     * @param value The storage throughput value for the DB instance. Can only be set when `storage_type` is `"gp3"`. Cannot be specified if the `allocated_storage` value is below a per-`engine` threshold. See the [RDS User Guide](https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/CHAP_Storage.html#gp3-storage) for details.
     */
    @JvmName("giolxqkxdikyoubt")
    public suspend fun storageThroughput(`value`: Int?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.storageThroughput = mapped
    }

    /**
     * @param value One of "standard" (magnetic), "gp2" (general
     * purpose SSD), "gp3" (general purpose SSD that needs `iops` independently)
     * or "io1" (provisioned IOPS SSD). The default is "io1" if `iops` is specified,
     * "gp2" if not.
     */
    @JvmName("njjcuttrpvgygmmt")
    public suspend fun storageType(`value`: Either<String, StorageType>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.storageType = mapped
    }

    /**
     * @param value One of "standard" (magnetic), "gp2" (general
     * purpose SSD), "gp3" (general purpose SSD that needs `iops` independently)
     * or "io1" (provisioned IOPS SSD). The default is "io1" if `iops` is specified,
     * "gp2" if not.
     */
    @JvmName("kdfqweegamhablce")
    public fun storageType(`value`: String) {
        val toBeMapped = Either.ofLeft<String, StorageType>(value)
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.storageType = mapped
    }

    /**
     * @param value One of "standard" (magnetic), "gp2" (general
     * purpose SSD), "gp3" (general purpose SSD that needs `iops` independently)
     * or "io1" (provisioned IOPS SSD). The default is "io1" if `iops` is specified,
     * "gp2" if not.
     */
    @JvmName("qtjmptfahsauyujp")
    public fun storageType(`value`: StorageType) {
        val toBeMapped = Either.ofRight<String, StorageType>(value)
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.storageType = mapped
    }

    /**
     * @param value A map of tags to assign to the resource. If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
     */
    @JvmName("vtncfpwtajahpqdt")
    public suspend fun tags(`value`: Map<String, String>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.tags = mapped
    }

    /**
     * @param values A map of tags to assign to the resource. If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
     */
    @JvmName("picftgoafhmeytqy")
    public fun tags(vararg values: Pair<String, String>) {
        val toBeMapped = values.toMap()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.tags = mapped
    }

    /**
     * @param value Time zone of the DB instance. `timezone` is currently
     * only supported by Microsoft SQL Server. The `timezone` can only be set on
     * creation. See [MSSQL User
   * Guide](http://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/CHAP_SQLServer.html#SQLServer.Concepts.General.TimeZone)
     * for more information.
     */
    @JvmName("raieqsvaqfflppcd")
    public suspend fun timezone(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.timezone = mapped
    }

    /**
     * @param value (Required unless a `snapshot_identifier` or `replicate_source_db`
     * is provided) Username for the master DB user. Cannot be specified for a replica.
     */
    @JvmName("yghynauipaethapk")
    public suspend fun username(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.username = mapped
    }

    /**
     * @param value List of VPC security groups to
     * associate.
     */
    @JvmName("amjbtjleygoekmao")
    public suspend fun vpcSecurityGroupIds(`value`: List<String>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.vpcSecurityGroupIds = mapped
    }

    /**
     * @param values List of VPC security groups to
     * associate.
     */
    @JvmName("abtruqqahfybjprw")
    public suspend fun vpcSecurityGroupIds(vararg values: String) {
        val toBeMapped = values.toList()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.vpcSecurityGroupIds = mapped
    }

    internal fun build(): InstanceArgs = InstanceArgs(
        allocatedStorage = allocatedStorage,
        allowMajorVersionUpgrade = allowMajorVersionUpgrade,
        applyImmediately = applyImmediately,
        autoMinorVersionUpgrade = autoMinorVersionUpgrade,
        availabilityZone = availabilityZone,
        backupRetentionPeriod = backupRetentionPeriod,
        backupTarget = backupTarget,
        backupWindow = backupWindow,
        blueGreenUpdate = blueGreenUpdate,
        caCertIdentifier = caCertIdentifier,
        characterSetName = characterSetName,
        copyTagsToSnapshot = copyTagsToSnapshot,
        customIamInstanceProfile = customIamInstanceProfile,
        customerOwnedIpEnabled = customerOwnedIpEnabled,
        dbName = dbName,
        dbSubnetGroupName = dbSubnetGroupName,
        deleteAutomatedBackups = deleteAutomatedBackups,
        deletionProtection = deletionProtection,
        domain = domain,
        domainIamRoleName = domainIamRoleName,
        enabledCloudwatchLogsExports = enabledCloudwatchLogsExports,
        engine = engine,
        engineVersion = engineVersion,
        finalSnapshotIdentifier = finalSnapshotIdentifier,
        iamDatabaseAuthenticationEnabled = iamDatabaseAuthenticationEnabled,
        identifier = identifier,
        identifierPrefix = identifierPrefix,
        instanceClass = instanceClass,
        iops = iops,
        kmsKeyId = kmsKeyId,
        licenseModel = licenseModel,
        maintenanceWindow = maintenanceWindow,
        manageMasterUserPassword = manageMasterUserPassword,
        masterUserSecretKmsKeyId = masterUserSecretKmsKeyId,
        maxAllocatedStorage = maxAllocatedStorage,
        monitoringInterval = monitoringInterval,
        monitoringRoleArn = monitoringRoleArn,
        multiAz = multiAz,
        name = name,
        ncharCharacterSetName = ncharCharacterSetName,
        networkType = networkType,
        optionGroupName = optionGroupName,
        parameterGroupName = parameterGroupName,
        password = password,
        performanceInsightsEnabled = performanceInsightsEnabled,
        performanceInsightsKmsKeyId = performanceInsightsKmsKeyId,
        performanceInsightsRetentionPeriod = performanceInsightsRetentionPeriod,
        port = port,
        publiclyAccessible = publiclyAccessible,
        replicaMode = replicaMode,
        replicateSourceDb = replicateSourceDb,
        restoreToPointInTime = restoreToPointInTime,
        s3Import = s3Import,
        skipFinalSnapshot = skipFinalSnapshot,
        snapshotIdentifier = snapshotIdentifier,
        storageEncrypted = storageEncrypted,
        storageThroughput = storageThroughput,
        storageType = storageType,
        tags = tags,
        timezone = timezone,
        username = username,
        vpcSecurityGroupIds = vpcSecurityGroupIds,
    )
}
