@file:Suppress("NAME_SHADOWING", "DEPRECATION")

package com.pulumi.aws.s3.kotlin

import com.pulumi.aws.s3.BucketArgs.builder
import com.pulumi.aws.s3.kotlin.enums.CannedAcl
import com.pulumi.aws.s3.kotlin.inputs.BucketCorsRuleArgs
import com.pulumi.aws.s3.kotlin.inputs.BucketCorsRuleArgsBuilder
import com.pulumi.aws.s3.kotlin.inputs.BucketGrantArgs
import com.pulumi.aws.s3.kotlin.inputs.BucketGrantArgsBuilder
import com.pulumi.aws.s3.kotlin.inputs.BucketLifecycleRuleArgs
import com.pulumi.aws.s3.kotlin.inputs.BucketLifecycleRuleArgsBuilder
import com.pulumi.aws.s3.kotlin.inputs.BucketLoggingArgs
import com.pulumi.aws.s3.kotlin.inputs.BucketLoggingArgsBuilder
import com.pulumi.aws.s3.kotlin.inputs.BucketObjectLockConfigurationArgs
import com.pulumi.aws.s3.kotlin.inputs.BucketObjectLockConfigurationArgsBuilder
import com.pulumi.aws.s3.kotlin.inputs.BucketReplicationConfigurationArgs
import com.pulumi.aws.s3.kotlin.inputs.BucketReplicationConfigurationArgsBuilder
import com.pulumi.aws.s3.kotlin.inputs.BucketServerSideEncryptionConfigurationArgs
import com.pulumi.aws.s3.kotlin.inputs.BucketServerSideEncryptionConfigurationArgsBuilder
import com.pulumi.aws.s3.kotlin.inputs.BucketVersioningArgs
import com.pulumi.aws.s3.kotlin.inputs.BucketVersioningArgsBuilder
import com.pulumi.aws.s3.kotlin.inputs.BucketWebsiteArgs
import com.pulumi.aws.s3.kotlin.inputs.BucketWebsiteArgsBuilder
import com.pulumi.core.Either
import com.pulumi.core.Output
import com.pulumi.core.Output.of
import com.pulumi.kotlin.ConvertibleToJava
import com.pulumi.kotlin.PulumiTagMarker
import com.pulumi.kotlin.applySuspend
import kotlin.Boolean
import kotlin.Pair
import kotlin.String
import kotlin.Suppress
import kotlin.Unit
import kotlin.collections.List
import kotlin.collections.Map
import kotlin.jvm.JvmName

/**
 * Provides a S3 bucket resource.
 * > This functionality is for managing S3 in an AWS Partition. To manage [S3 on Outposts](https://docs.aws.amazon.com/AmazonS3/latest/dev/S3onOutposts.html), see the `aws.s3control.Bucket` resource.
 * ## Example Usage
 * ### Private Bucket w/ Tags
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.s3.Bucket;
 * import com.pulumi.aws.s3.BucketArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var bucket = new Bucket("bucket", BucketArgs.builder()
 *             .acl("private")
 *             .tags(Map.ofEntries(
 *                 Map.entry("Environment", "Dev"),
 *                 Map.entry("Name", "My bucket")
 *             ))
 *             .build());
 *     }
 * }
 * ```
 * ### Static Website Hosting
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.s3.Bucket;
 * import com.pulumi.aws.s3.BucketArgs;
 * import com.pulumi.aws.s3.inputs.BucketWebsiteArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var bucket = new Bucket("bucket", BucketArgs.builder()
 *             .acl("public-read")
 *             .policy(Files.readString(Paths.get("policy.json")))
 *             .website(BucketWebsiteArgs.builder()
 *                 .indexDocument("index.html")
 *                 .errorDocument("error.html")
 *                 .routingRules("""
 * [{
 *     "Condition": {
 *         "KeyPrefixEquals": "docs/"
 *     },
 *     "Redirect": {
 *         "ReplaceKeyPrefixWith": "documents/"
 *     }
 * }]
 *                 """)
 *                 .build())
 *             .build());
 *     }
 * }
 * ```
 * ### Using CORS
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.s3.Bucket;
 * import com.pulumi.aws.s3.BucketArgs;
 * import com.pulumi.aws.s3.inputs.BucketCorsRuleArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var bucket = new Bucket("bucket", BucketArgs.builder()
 *             .acl("public-read")
 *             .corsRules(BucketCorsRuleArgs.builder()
 *                 .allowedHeaders("*")
 *                 .allowedMethods(
 *                     "PUT",
 *                     "POST")
 *                 .allowedOrigins("https://s3-website-test.mydomain.com")
 *                 .exposeHeaders("ETag")
 *                 .maxAgeSeconds(3000)
 *                 .build())
 *             .build());
 *     }
 * }
 * ```
 * ### Using versioning
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.s3.Bucket;
 * import com.pulumi.aws.s3.BucketArgs;
 * import com.pulumi.aws.s3.inputs.BucketVersioningArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var bucket = new Bucket("bucket", BucketArgs.builder()
 *             .acl("private")
 *             .versioning(BucketVersioningArgs.builder()
 *                 .enabled(true)
 *                 .build())
 *             .build());
 *     }
 * }
 * ```
 * ### Enable Logging
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.s3.Bucket;
 * import com.pulumi.aws.s3.BucketArgs;
 * import com.pulumi.aws.s3.inputs.BucketLoggingArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var logBucket = new Bucket("logBucket", BucketArgs.builder()
 *             .acl("log-delivery-write")
 *             .build());
 *         var bucket = new Bucket("bucket", BucketArgs.builder()
 *             .acl("private")
 *             .loggings(BucketLoggingArgs.builder()
 *                 .targetBucket(logBucket.id())
 *                 .targetPrefix("log/")
 *                 .build())
 *             .build());
 *     }
 * }
 * ```
 * ### Using object lifecycle
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.s3.Bucket;
 * import com.pulumi.aws.s3.BucketArgs;
 * import com.pulumi.aws.s3.inputs.BucketLifecycleRuleArgs;
 * import com.pulumi.aws.s3.inputs.BucketLifecycleRuleExpirationArgs;
 * import com.pulumi.aws.s3.inputs.BucketLifecycleRuleNoncurrentVersionExpirationArgs;
 * import com.pulumi.aws.s3.inputs.BucketVersioningArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var bucket = new Bucket("bucket", BucketArgs.builder()
 *             .acl("private")
 *             .lifecycleRules(
 *                 BucketLifecycleRuleArgs.builder()
 *                     .enabled(true)
 *                     .expiration(BucketLifecycleRuleExpirationArgs.builder()
 *                         .days(90)
 *                         .build())
 *                     .id("log")
 *                     .prefix("log/")
 *                     .tags(Map.ofEntries(
 *                         Map.entry("autoclean", "true"),
 *                         Map.entry("rule", "log")
 *                     ))
 *                     .transitions(
 *                         BucketLifecycleRuleTransitionArgs.builder()
 *                             .days(30)
 *                             .storageClass("STANDARD_IA")
 *                             .build(),
 *                         BucketLifecycleRuleTransitionArgs.builder()
 *                             .days(60)
 *                             .storageClass("GLACIER")
 *                             .build())
 *                     .build(),
 *                 BucketLifecycleRuleArgs.builder()
 *                     .enabled(true)
 *                     .expiration(BucketLifecycleRuleExpirationArgs.builder()
 *                         .date("2016-01-12")
 *                         .build())
 *                     .id("tmp")
 *                     .prefix("tmp/")
 *                     .build())
 *             .build());
 *         var versioningBucket = new Bucket("versioningBucket", BucketArgs.builder()
 *             .acl("private")
 *             .lifecycleRules(BucketLifecycleRuleArgs.builder()
 *                 .enabled(true)
 *                 .noncurrentVersionExpiration(BucketLifecycleRuleNoncurrentVersionExpirationArgs.builder()
 *                     .days(90)
 *                     .build())
 *                 .noncurrentVersionTransitions(
 *                     BucketLifecycleRuleNoncurrentVersionTransitionArgs.builder()
 *                         .days(30)
 *                         .storageClass("STANDARD_IA")
 *                         .build(),
 *                     BucketLifecycleRuleNoncurrentVersionTransitionArgs.builder()
 *                         .days(60)
 *                         .storageClass("GLACIER")
 *                         .build())
 *                 .prefix("config/")
 *                 .build())
 *             .versioning(BucketVersioningArgs.builder()
 *                 .enabled(true)
 *                 .build())
 *             .build());
 *     }
 * }
 * ```
 * ### Using replication configuration
 * > **NOTE:** See the `aws.s3.BucketReplicationConfig` resource to support bi-directional replication configuration and additional features.
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.Provider;
 * import com.pulumi.aws.ProviderArgs;
 * import com.pulumi.aws.iam.Role;
 * import com.pulumi.aws.iam.RoleArgs;
 * import com.pulumi.aws.s3.Bucket;
 * import com.pulumi.aws.s3.BucketArgs;
 * import com.pulumi.aws.s3.inputs.BucketVersioningArgs;
 * import com.pulumi.aws.s3.inputs.BucketReplicationConfigurationArgs;
 * import com.pulumi.aws.iam.Policy;
 * import com.pulumi.aws.iam.PolicyArgs;
 * import com.pulumi.aws.iam.RolePolicyAttachment;
 * import com.pulumi.aws.iam.RolePolicyAttachmentArgs;
 * import com.pulumi.resources.CustomResourceOptions;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var central = new Provider("central", ProviderArgs.builder()
 *             .region("eu-central-1")
 *             .build());
 *         var replicationRole = new Role("replicationRole", RoleArgs.builder()
 *             .assumeRolePolicy("""
 * {
 *   "Version": "2012-10-17",
 *   "Statement": [
 *     {
 *       "Action": "sts:AssumeRole",
 *       "Principal": {
 *         "Service": "s3.amazonaws.com"
 *       },
 *       "Effect": "Allow",
 *       "Sid": ""
 *     }
 *   ]
 * }
 *             """)
 *             .build());
 *         var destination = new Bucket("destination", BucketArgs.builder()
 *             .versioning(BucketVersioningArgs.builder()
 *                 .enabled(true)
 *                 .build())
 *             .build());
 *         var source = new Bucket("source", BucketArgs.builder()
 *             .acl("private")
 *             .versioning(BucketVersioningArgs.builder()
 *                 .enabled(true)
 *                 .build())
 *             .replicationConfiguration(BucketReplicationConfigurationArgs.builder()
 *                 .role(replicationRole.arn())
 *                 .rules(BucketReplicationConfigurationRuleArgs.builder()
 *                     .id("foobar")
 *                     .status("Enabled")
 *                     .filter(BucketReplicationConfigurationRuleFilterArgs.builder()
 *                         .tags()
 *                         .build())
 *                     .destination(BucketReplicationConfigurationRuleDestinationArgs.builder()
 *                         .bucket(destination.arn())
 *                         .storageClass("STANDARD")
 *                         .replicationTime(BucketReplicationConfigurationRuleDestinationReplicationTimeArgs.builder()
 *                             .status("Enabled")
 *                             .minutes(15)
 *                             .build())
 *                         .metrics(BucketReplicationConfigurationRuleDestinationMetricsArgs.builder()
 *                             .status("Enabled")
 *                             .minutes(15)
 *                             .build())
 *                         .build())
 *                     .build())
 *                 .build())
 *             .build(), CustomResourceOptions.builder()
 *                 .provider(aws.central())
 *                 .build());
 *         var replicationPolicy = new Policy("replicationPolicy", PolicyArgs.builder()
 *             .policy(Output.tuple(source.arn(), source.arn(), destination.arn()).applyValue(values -> {
 *                 var sourceArn = values.t1;
 *                 var sourceArn1 = values.t2;
 *                 var destinationArn = values.t3;
 *                 return """
 * {
 *   "Version": "2012-10-17",
 *   "Statement": [
 *     {
 *       "Action": [
 *         "s3:GetReplicationConfiguration",
 *         "s3:ListBucket"
 *       ],
 *       "Effect": "Allow",
 *       "Resource": [
 *         "%s"
 *       ]
 *     },
 *     {
 *       "Action": [
 *         "s3:GetObjectVersionForReplication",
 *         "s3:GetObjectVersionAcl",
 *          "s3:GetObjectVersionTagging"
 *       ],
 *       "Effect": "Allow",
 *       "Resource": [
 *         "%s/*"
 *       ]
 *     },
 *     {
 *       "Action": [
 *         "s3:ReplicateObject",
 *         "s3:ReplicateDelete",
 *         "s3:ReplicateTags"
 *       ],
 *       "Effect": "Allow",
 *       "Resource": "%s/*"
 *     }
 *   ]
 * }
 * ", sourceArn,sourceArn1,destinationArn);
 *             }))
 *             .build());
 *         var replicationRolePolicyAttachment = new RolePolicyAttachment("replicationRolePolicyAttachment", RolePolicyAttachmentArgs.builder()
 *             .role(replicationRole.name())
 *             .policyArn(replicationPolicy.arn())
 *             .build());
 *     }
 * }
 * ```
 * ### Enable Default Server Side Encryption
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.kms.Key;
 * import com.pulumi.aws.kms.KeyArgs;
 * import com.pulumi.aws.s3.Bucket;
 * import com.pulumi.aws.s3.BucketArgs;
 * import com.pulumi.aws.s3.inputs.BucketServerSideEncryptionConfigurationArgs;
 * import com.pulumi.aws.s3.inputs.BucketServerSideEncryptionConfigurationRuleArgs;
 * import com.pulumi.aws.s3.inputs.BucketServerSideEncryptionConfigurationRuleApplyServerSideEncryptionByDefaultArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var mykey = new Key("mykey", KeyArgs.builder()
 *             .description("This key is used to encrypt bucket objects")
 *             .deletionWindowInDays(10)
 *             .build());
 *         var mybucket = new Bucket("mybucket", BucketArgs.builder()
 *             .serverSideEncryptionConfiguration(BucketServerSideEncryptionConfigurationArgs.builder()
 *                 .rule(BucketServerSideEncryptionConfigurationRuleArgs.builder()
 *                     .applyServerSideEncryptionByDefault(BucketServerSideEncryptionConfigurationRuleApplyServerSideEncryptionByDefaultArgs.builder()
 *                         .kmsMasterKeyId(mykey.arn())
 *                         .sseAlgorithm("aws:kms")
 *                         .build())
 *                     .build())
 *                 .build())
 *             .build());
 *     }
 * }
 * ```
 * ### Using ACL policy grants
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.s3.S3Functions;
 * import com.pulumi.aws.s3.Bucket;
 * import com.pulumi.aws.s3.BucketArgs;
 * import com.pulumi.aws.s3.inputs.BucketGrantArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         final var currentUser = S3Functions.getCanonicalUserId();
 *         var bucket = new Bucket("bucket", BucketArgs.builder()
 *             .grants(
 *                 BucketGrantArgs.builder()
 *                     .id(currentUser.applyValue(getCanonicalUserIdResult -> getCanonicalUserIdResult.id()))
 *                     .type("CanonicalUser")
 *                     .permissions("FULL_CONTROL")
 *                     .build(),
 *                 BucketGrantArgs.builder()
 *                     .type("Group")
 *                     .permissions(
 *                         "READ_ACP",
 *                         "WRITE")
 *                     .uri("http://acs.amazonaws.com/groups/s3/LogDelivery")
 *                     .build())
 *             .build());
 *     }
 * }
 * ```
 * ## Import
 * S3 bucket can be imported using the `bucket`, e.g.,
 * ```sh
 *  $ pulumi import aws:s3/bucket:Bucket bucket bucket-name
 * ```
 *  The `policy` argument is not imported and will be deprecated in a future version of the provider. Use the `aws_s3_bucket_policy` resource to manage the S3 Bucket Policy instead.
 * @property accelerationStatus Sets the accelerate configuration of an existing bucket. Can be `Enabled` or `Suspended`.
 * @property acl The [canned ACL](https://docs.aws.amazon.com/AmazonS3/latest/dev/acl-overview.html#canned-acl) to apply. Valid values are `private`, `public-read`, `public-read-write`, `aws-exec-read`, `authenticated-read`, and `log-delivery-write`. Defaults to `private`.  Conflicts with `grant`.
 * @property arn The ARN of the bucket. Will be of format `arn:aws:s3:::bucketname`.
 * @property bucket The name of the bucket. If omitted, this provider will assign a random, unique name. Must be lowercase and less than or equal to 63 characters in length. A full list of bucket naming rules [may be found here](https://docs.aws.amazon.com/AmazonS3/latest/userguide/bucketnamingrules.html).
 * @property bucketPrefix Creates a unique bucket name beginning with the specified prefix. Conflicts with `bucket`. Must be lowercase and less than or equal to 37 characters in length. A full list of bucket naming rules [may be found here](https://docs.aws.amazon.com/AmazonS3/latest/userguide/bucketnamingrules.html).
 * @property corsRules A rule of [Cross-Origin Resource Sharing](https://docs.aws.amazon.com/AmazonS3/latest/dev/cors.html) (documented below).
 * @property forceDestroy A boolean that indicates all objects (including any [locked objects](https://docs.aws.amazon.com/AmazonS3/latest/dev/object-lock-overview.html)) should be deleted from the bucket so that the bucket can be destroyed without error. These objects are *not* recoverable.
 * @property grants An [ACL policy grant](https://docs.aws.amazon.com/AmazonS3/latest/dev/acl-overview.html#sample-acl) (documented below). Conflicts with `acl`.
 * @property hostedZoneId The [Route 53 Hosted Zone ID](https://docs.aws.amazon.com/general/latest/gr/rande.html#s3_website_region_endpoints) for this bucket's region.
 * @property lifecycleRules A configuration of [object lifecycle management](http://docs.aws.amazon.com/AmazonS3/latest/dev/object-lifecycle-mgmt.html) (documented below).
 * @property loggings A settings of [bucket logging](https://docs.aws.amazon.com/AmazonS3/latest/UG/ManagingBucketLogging.html) (documented below).
 * @property objectLockConfiguration A configuration of [S3 object locking](https://docs.aws.amazon.com/AmazonS3/latest/dev/object-lock.html) (documented below)
 * > **NOTE:** You cannot use `acceleration_status` in `cn-north-1` or `us-gov-west-1`
 * @property policy A valid [bucket policy](https://docs.aws.amazon.com/AmazonS3/latest/dev/example-bucket-policies.html) JSON document. Note that if the policy document is not specific enough (but still valid), this provider may view the policy as constantly changing in a `pulumi preview`. In this case, please make sure you use the verbose/specific version of the policy.
 * @property replicationConfiguration A configuration of [replication configuration](http://docs.aws.amazon.com/AmazonS3/latest/dev/crr.html) (documented below).
 * @property requestPayer Specifies who should bear the cost of Amazon S3 data transfer.
 * Can be either `BucketOwner` or `Requester`. By default, the owner of the S3 bucket would incur
 * the costs of any data transfer. See [Requester Pays Buckets](http://docs.aws.amazon.com/AmazonS3/latest/dev/RequesterPaysBuckets.html)
 * developer guide for more information.
 * @property serverSideEncryptionConfiguration A configuration of [server-side encryption configuration](http://docs.aws.amazon.com/AmazonS3/latest/dev/bucket-encryption.html) (documented below)
 * @property tags A map of tags to assign to the bucket. If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
 * @property versioning A state of [versioning](https://docs.aws.amazon.com/AmazonS3/latest/dev/Versioning.html) (documented below)
 * @property website A website object (documented below).
 * @property websiteDomain The domain of the website endpoint, if the bucket is configured with a website. If not, this will be an empty string. This is used to create Route 53 alias records.
 * @property websiteEndpoint The website endpoint, if the bucket is configured with a website. If not, this will be an empty string.
 * */*/
 */
public data class BucketArgs(
    public val accelerationStatus: Output<String>? = null,
    public val acl: Output<Either<String, CannedAcl>>? = null,
    public val arn: Output<String>? = null,
    public val bucket: Output<String>? = null,
    public val bucketPrefix: Output<String>? = null,
    public val corsRules: Output<List<BucketCorsRuleArgs>>? = null,
    public val forceDestroy: Output<Boolean>? = null,
    public val grants: Output<List<BucketGrantArgs>>? = null,
    public val hostedZoneId: Output<String>? = null,
    public val lifecycleRules: Output<List<BucketLifecycleRuleArgs>>? = null,
    public val loggings: Output<List<BucketLoggingArgs>>? = null,
    public val objectLockConfiguration: Output<BucketObjectLockConfigurationArgs>? = null,
    public val policy: Output<String>? = null,
    public val replicationConfiguration: Output<BucketReplicationConfigurationArgs>? = null,
    public val requestPayer: Output<String>? = null,
    public val serverSideEncryptionConfiguration: Output<BucketServerSideEncryptionConfigurationArgs>? =
        null,
    public val tags: Output<Map<String, String>>? = null,
    public val versioning: Output<BucketVersioningArgs>? = null,
    public val website: Output<BucketWebsiteArgs>? = null,
    public val websiteDomain: Output<String>? = null,
    public val websiteEndpoint: Output<String>? = null,
) : ConvertibleToJava<com.pulumi.aws.s3.BucketArgs> {
    override fun toJava(): com.pulumi.aws.s3.BucketArgs = com.pulumi.aws.s3.BucketArgs.builder()
        .accelerationStatus(accelerationStatus?.applyValue({ args0 -> args0 }))
        .acl(
            acl?.applyValue({ args0 ->
                args0.transform({ args0 -> args0 }, { args0 ->
                    args0.let({ args0 ->
                        args0.toJava()
                    })
                })
            }),
        )
        .arn(arn?.applyValue({ args0 -> args0 }))
        .bucket(bucket?.applyValue({ args0 -> args0 }))
        .bucketPrefix(bucketPrefix?.applyValue({ args0 -> args0 }))
        .corsRules(
            corsRules?.applyValue({ args0 ->
                args0.map({ args0 ->
                    args0.let({ args0 ->
                        args0.toJava()
                    })
                })
            }),
        )
        .forceDestroy(forceDestroy?.applyValue({ args0 -> args0 }))
        .grants(grants?.applyValue({ args0 -> args0.map({ args0 -> args0.let({ args0 -> args0.toJava() }) }) }))
        .hostedZoneId(hostedZoneId?.applyValue({ args0 -> args0 }))
        .lifecycleRules(
            lifecycleRules?.applyValue({ args0 ->
                args0.map({ args0 ->
                    args0.let({ args0 ->
                        args0.toJava()
                    })
                })
            }),
        )
        .loggings(
            loggings?.applyValue({ args0 ->
                args0.map({ args0 ->
                    args0.let({ args0 ->
                        args0.toJava()
                    })
                })
            }),
        )
        .objectLockConfiguration(
            objectLockConfiguration?.applyValue({ args0 ->
                args0.let({ args0 ->
                    args0.toJava()
                })
            }),
        )
        .policy(policy?.applyValue({ args0 -> args0 }))
        .replicationConfiguration(
            replicationConfiguration?.applyValue({ args0 ->
                args0.let({ args0 ->
                    args0.toJava()
                })
            }),
        )
        .requestPayer(requestPayer?.applyValue({ args0 -> args0 }))
        .serverSideEncryptionConfiguration(
            serverSideEncryptionConfiguration?.applyValue({ args0 ->
                args0.let({ args0 -> args0.toJava() })
            }),
        )
        .tags(tags?.applyValue({ args0 -> args0.map({ args0 -> args0.key.to(args0.value) }).toMap() }))
        .versioning(versioning?.applyValue({ args0 -> args0.let({ args0 -> args0.toJava() }) }))
        .website(website?.applyValue({ args0 -> args0.let({ args0 -> args0.toJava() }) }))
        .websiteDomain(websiteDomain?.applyValue({ args0 -> args0 }))
        .websiteEndpoint(websiteEndpoint?.applyValue({ args0 -> args0 })).build()
}

/**
 * Builder for [BucketArgs].
 */
@PulumiTagMarker
public class BucketArgsBuilder internal constructor() {
    private var accelerationStatus: Output<String>? = null

    private var acl: Output<Either<String, CannedAcl>>? = null

    private var arn: Output<String>? = null

    private var bucket: Output<String>? = null

    private var bucketPrefix: Output<String>? = null

    private var corsRules: Output<List<BucketCorsRuleArgs>>? = null

    private var forceDestroy: Output<Boolean>? = null

    private var grants: Output<List<BucketGrantArgs>>? = null

    private var hostedZoneId: Output<String>? = null

    private var lifecycleRules: Output<List<BucketLifecycleRuleArgs>>? = null

    private var loggings: Output<List<BucketLoggingArgs>>? = null

    private var objectLockConfiguration: Output<BucketObjectLockConfigurationArgs>? = null

    private var policy: Output<String>? = null

    private var replicationConfiguration: Output<BucketReplicationConfigurationArgs>? = null

    private var requestPayer: Output<String>? = null

    private var serverSideEncryptionConfiguration:
        Output<BucketServerSideEncryptionConfigurationArgs>? = null

    private var tags: Output<Map<String, String>>? = null

    private var versioning: Output<BucketVersioningArgs>? = null

    private var website: Output<BucketWebsiteArgs>? = null

    private var websiteDomain: Output<String>? = null

    private var websiteEndpoint: Output<String>? = null

    /**
     * @param value Sets the accelerate configuration of an existing bucket. Can be `Enabled` or `Suspended`.
     */
    @JvmName("ideqxlxklcowhwgo")
    public suspend fun accelerationStatus(`value`: Output<String>) {
        this.accelerationStatus = value
    }

    /**
     * @param value The [canned ACL](https://docs.aws.amazon.com/AmazonS3/latest/dev/acl-overview.html#canned-acl) to apply. Valid values are `private`, `public-read`, `public-read-write`, `aws-exec-read`, `authenticated-read`, and `log-delivery-write`. Defaults to `private`.  Conflicts with `grant`.
     */
    @JvmName("fqnnvcbhxbvhnfwy")
    public suspend fun acl(`value`: Output<Either<String, CannedAcl>>) {
        this.acl = value
    }

    /**
     * @param value The ARN of the bucket. Will be of format `arn:aws:s3:::bucketname`.
     */
    @JvmName("glvbnoyykvxxiuts")
    public suspend fun arn(`value`: Output<String>) {
        this.arn = value
    }

    /**
     * @param value The name of the bucket. If omitted, this provider will assign a random, unique name. Must be lowercase and less than or equal to 63 characters in length. A full list of bucket naming rules [may be found here](https://docs.aws.amazon.com/AmazonS3/latest/userguide/bucketnamingrules.html).
     */
    @JvmName("qiyqksagqgasgspj")
    public suspend fun bucket(`value`: Output<String>) {
        this.bucket = value
    }

    /**
     * @param value Creates a unique bucket name beginning with the specified prefix. Conflicts with `bucket`. Must be lowercase and less than or equal to 37 characters in length. A full list of bucket naming rules [may be found here](https://docs.aws.amazon.com/AmazonS3/latest/userguide/bucketnamingrules.html).
     */
    @JvmName("ljbnpswxgbipeshy")
    public suspend fun bucketPrefix(`value`: Output<String>) {
        this.bucketPrefix = value
    }

    /**
     * @param value A rule of [Cross-Origin Resource Sharing](https://docs.aws.amazon.com/AmazonS3/latest/dev/cors.html) (documented below).
     */
    @JvmName("qapoypupjjqkgqct")
    public suspend fun corsRules(`value`: Output<List<BucketCorsRuleArgs>>) {
        this.corsRules = value
    }

    @JvmName("euelxerlwqfnropk")
    public suspend fun corsRules(vararg values: Output<BucketCorsRuleArgs>) {
        this.corsRules = Output.all(values.asList())
    }

    /**
     * @param values A rule of [Cross-Origin Resource Sharing](https://docs.aws.amazon.com/AmazonS3/latest/dev/cors.html) (documented below).
     */
    @JvmName("qkebkxkuftonybvp")
    public suspend fun corsRules(values: List<Output<BucketCorsRuleArgs>>) {
        this.corsRules = Output.all(values)
    }

    /**
     * @param value A boolean that indicates all objects (including any [locked objects](https://docs.aws.amazon.com/AmazonS3/latest/dev/object-lock-overview.html)) should be deleted from the bucket so that the bucket can be destroyed without error. These objects are *not* recoverable.
     */
    @JvmName("folomlqcmrfbbhjn")
    public suspend fun forceDestroy(`value`: Output<Boolean>) {
        this.forceDestroy = value
    }

    /**
     * @param value An [ACL policy grant](https://docs.aws.amazon.com/AmazonS3/latest/dev/acl-overview.html#sample-acl) (documented below). Conflicts with `acl`.
     */
    @JvmName("piuphkepymbidxgu")
    public suspend fun grants(`value`: Output<List<BucketGrantArgs>>) {
        this.grants = value
    }

    @JvmName("cuvpqarybflqqdqg")
    public suspend fun grants(vararg values: Output<BucketGrantArgs>) {
        this.grants = Output.all(values.asList())
    }

    /**
     * @param values An [ACL policy grant](https://docs.aws.amazon.com/AmazonS3/latest/dev/acl-overview.html#sample-acl) (documented below). Conflicts with `acl`.
     */
    @JvmName("klkubjwehtryihrb")
    public suspend fun grants(values: List<Output<BucketGrantArgs>>) {
        this.grants = Output.all(values)
    }

    /**
     * @param value The [Route 53 Hosted Zone ID](https://docs.aws.amazon.com/general/latest/gr/rande.html#s3_website_region_endpoints) for this bucket's region.
     */
    @JvmName("myqlifurnfrmcxww")
    public suspend fun hostedZoneId(`value`: Output<String>) {
        this.hostedZoneId = value
    }

    /**
     * @param value A configuration of [object lifecycle management](http://docs.aws.amazon.com/AmazonS3/latest/dev/object-lifecycle-mgmt.html) (documented below).
     */
    @JvmName("jdxjivqplwjojswh")
    public suspend fun lifecycleRules(`value`: Output<List<BucketLifecycleRuleArgs>>) {
        this.lifecycleRules = value
    }

    @JvmName("ucwsmydhfijitjhx")
    public suspend fun lifecycleRules(vararg values: Output<BucketLifecycleRuleArgs>) {
        this.lifecycleRules = Output.all(values.asList())
    }

    /**
     * @param values A configuration of [object lifecycle management](http://docs.aws.amazon.com/AmazonS3/latest/dev/object-lifecycle-mgmt.html) (documented below).
     */
    @JvmName("bwnxgcreoohqtdro")
    public suspend fun lifecycleRules(values: List<Output<BucketLifecycleRuleArgs>>) {
        this.lifecycleRules = Output.all(values)
    }

    /**
     * @param value A settings of [bucket logging](https://docs.aws.amazon.com/AmazonS3/latest/UG/ManagingBucketLogging.html) (documented below).
     */
    @JvmName("ouxjeapjvhjnahwa")
    public suspend fun loggings(`value`: Output<List<BucketLoggingArgs>>) {
        this.loggings = value
    }

    @JvmName("oxenhmbralxihxma")
    public suspend fun loggings(vararg values: Output<BucketLoggingArgs>) {
        this.loggings = Output.all(values.asList())
    }

    /**
     * @param values A settings of [bucket logging](https://docs.aws.amazon.com/AmazonS3/latest/UG/ManagingBucketLogging.html) (documented below).
     */
    @JvmName("djckjhxwscysrrrx")
    public suspend fun loggings(values: List<Output<BucketLoggingArgs>>) {
        this.loggings = Output.all(values)
    }

    /**
     * @param value A configuration of [S3 object locking](https://docs.aws.amazon.com/AmazonS3/latest/dev/object-lock.html) (documented below)
     * > **NOTE:** You cannot use `acceleration_status` in `cn-north-1` or `us-gov-west-1`
     */
    @JvmName("qaeyxolscwxoebqv")
    public suspend fun objectLockConfiguration(`value`: Output<BucketObjectLockConfigurationArgs>) {
        this.objectLockConfiguration = value
    }

    /**
     * @param value A valid [bucket policy](https://docs.aws.amazon.com/AmazonS3/latest/dev/example-bucket-policies.html) JSON document. Note that if the policy document is not specific enough (but still valid), this provider may view the policy as constantly changing in a `pulumi preview`. In this case, please make sure you use the verbose/specific version of the policy.
     */
    @JvmName("sirckeujuknmaffb")
    public suspend fun policy(`value`: Output<String>) {
        this.policy = value
    }

    /**
     * @param value A configuration of [replication configuration](http://docs.aws.amazon.com/AmazonS3/latest/dev/crr.html) (documented below).
     */
    @JvmName("offeadvsebcbtjur")
    public suspend fun replicationConfiguration(`value`: Output<BucketReplicationConfigurationArgs>) {
        this.replicationConfiguration = value
    }

    /**
     * @param value Specifies who should bear the cost of Amazon S3 data transfer.
     * Can be either `BucketOwner` or `Requester`. By default, the owner of the S3 bucket would incur
     * the costs of any data transfer. See [Requester Pays Buckets](http://docs.aws.amazon.com/AmazonS3/latest/dev/RequesterPaysBuckets.html)
     * developer guide for more information.
     */
    @JvmName("kupmqltfupscmrtm")
    public suspend fun requestPayer(`value`: Output<String>) {
        this.requestPayer = value
    }

    /**
     * @param value A configuration of [server-side encryption configuration](http://docs.aws.amazon.com/AmazonS3/latest/dev/bucket-encryption.html) (documented below)
     */
    @JvmName("tkkbolumepocwngs")
    public suspend
    fun serverSideEncryptionConfiguration(`value`: Output<BucketServerSideEncryptionConfigurationArgs>) {
        this.serverSideEncryptionConfiguration = value
    }

    /**
     * @param value A map of tags to assign to the bucket. If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
     */
    @JvmName("gkkcpmldreehomup")
    public suspend fun tags(`value`: Output<Map<String, String>>) {
        this.tags = value
    }

    /**
     * @param value A state of [versioning](https://docs.aws.amazon.com/AmazonS3/latest/dev/Versioning.html) (documented below)
     */
    @JvmName("dfbovitfmbpnpsbi")
    public suspend fun versioning(`value`: Output<BucketVersioningArgs>) {
        this.versioning = value
    }

    /**
     * @param value A website object (documented below).
     */
    @JvmName("yuikfwcijooogixb")
    public suspend fun website(`value`: Output<BucketWebsiteArgs>) {
        this.website = value
    }

    /**
     * @param value The domain of the website endpoint, if the bucket is configured with a website. If not, this will be an empty string. This is used to create Route 53 alias records.
     */
    @JvmName("xuyvwkajvehqtbkw")
    public suspend fun websiteDomain(`value`: Output<String>) {
        this.websiteDomain = value
    }

    /**
     * @param value The website endpoint, if the bucket is configured with a website. If not, this will be an empty string.
     */
    @JvmName("wxdiwrhvgahdoane")
    public suspend fun websiteEndpoint(`value`: Output<String>) {
        this.websiteEndpoint = value
    }

    /**
     * @param value Sets the accelerate configuration of an existing bucket. Can be `Enabled` or `Suspended`.
     */
    @JvmName("bwqrqggjlxmywryl")
    public suspend fun accelerationStatus(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.accelerationStatus = mapped
    }

    /**
     * @param value The [canned ACL](https://docs.aws.amazon.com/AmazonS3/latest/dev/acl-overview.html#canned-acl) to apply. Valid values are `private`, `public-read`, `public-read-write`, `aws-exec-read`, `authenticated-read`, and `log-delivery-write`. Defaults to `private`.  Conflicts with `grant`.
     */
    @JvmName("dqvnodlxfpsprfcd")
    public suspend fun acl(`value`: Either<String, CannedAcl>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.acl = mapped
    }

    /**
     * @param value The [canned ACL](https://docs.aws.amazon.com/AmazonS3/latest/dev/acl-overview.html#canned-acl) to apply. Valid values are `private`, `public-read`, `public-read-write`, `aws-exec-read`, `authenticated-read`, and `log-delivery-write`. Defaults to `private`.  Conflicts with `grant`.
     */
    @JvmName("bofwkuifcgivxfrc")
    public fun acl(`value`: String) {
        val toBeMapped = Either.ofLeft<String, CannedAcl>(value)
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.acl = mapped
    }

    /**
     * @param value The [canned ACL](https://docs.aws.amazon.com/AmazonS3/latest/dev/acl-overview.html#canned-acl) to apply. Valid values are `private`, `public-read`, `public-read-write`, `aws-exec-read`, `authenticated-read`, and `log-delivery-write`. Defaults to `private`.  Conflicts with `grant`.
     */
    @JvmName("txendrxsqiqocqea")
    public fun acl(`value`: CannedAcl) {
        val toBeMapped = Either.ofRight<String, CannedAcl>(value)
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.acl = mapped
    }

    /**
     * @param value The ARN of the bucket. Will be of format `arn:aws:s3:::bucketname`.
     */
    @JvmName("niwywnaksyctoyqk")
    public suspend fun arn(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.arn = mapped
    }

    /**
     * @param value The name of the bucket. If omitted, this provider will assign a random, unique name. Must be lowercase and less than or equal to 63 characters in length. A full list of bucket naming rules [may be found here](https://docs.aws.amazon.com/AmazonS3/latest/userguide/bucketnamingrules.html).
     */
    @JvmName("pnaldgoejrtjmodt")
    public suspend fun bucket(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.bucket = mapped
    }

    /**
     * @param value Creates a unique bucket name beginning with the specified prefix. Conflicts with `bucket`. Must be lowercase and less than or equal to 37 characters in length. A full list of bucket naming rules [may be found here](https://docs.aws.amazon.com/AmazonS3/latest/userguide/bucketnamingrules.html).
     */
    @JvmName("ievmtqhagjtahogi")
    public suspend fun bucketPrefix(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.bucketPrefix = mapped
    }

    /**
     * @param value A rule of [Cross-Origin Resource Sharing](https://docs.aws.amazon.com/AmazonS3/latest/dev/cors.html) (documented below).
     */
    @JvmName("narkoyealecausnm")
    public suspend fun corsRules(`value`: List<BucketCorsRuleArgs>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.corsRules = mapped
    }

    /**
     * @param argument A rule of [Cross-Origin Resource Sharing](https://docs.aws.amazon.com/AmazonS3/latest/dev/cors.html) (documented below).
     */
    @JvmName("lstyksaegrppaohh")
    public suspend fun corsRules(argument: List<suspend BucketCorsRuleArgsBuilder.() -> Unit>) {
        val toBeMapped = argument.toList().map {
            BucketCorsRuleArgsBuilder().applySuspend {
                it()
            }.build()
        }
        val mapped = of(toBeMapped)
        this.corsRules = mapped
    }

    /**
     * @param argument A rule of [Cross-Origin Resource Sharing](https://docs.aws.amazon.com/AmazonS3/latest/dev/cors.html) (documented below).
     */
    @JvmName("ikvydooirvebrrkm")
    public suspend fun corsRules(vararg argument: suspend BucketCorsRuleArgsBuilder.() -> Unit) {
        val toBeMapped = argument.toList().map {
            BucketCorsRuleArgsBuilder().applySuspend {
                it()
            }.build()
        }
        val mapped = of(toBeMapped)
        this.corsRules = mapped
    }

    /**
     * @param argument A rule of [Cross-Origin Resource Sharing](https://docs.aws.amazon.com/AmazonS3/latest/dev/cors.html) (documented below).
     */
    @JvmName("exnkixmqghvsammr")
    public suspend fun corsRules(argument: suspend BucketCorsRuleArgsBuilder.() -> Unit) {
        val toBeMapped = listOf(BucketCorsRuleArgsBuilder().applySuspend { argument() }.build())
        val mapped = of(toBeMapped)
        this.corsRules = mapped
    }

    /**
     * @param values A rule of [Cross-Origin Resource Sharing](https://docs.aws.amazon.com/AmazonS3/latest/dev/cors.html) (documented below).
     */
    @JvmName("ivlmkftnwebgngfk")
    public suspend fun corsRules(vararg values: BucketCorsRuleArgs) {
        val toBeMapped = values.toList()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.corsRules = mapped
    }

    /**
     * @param value A boolean that indicates all objects (including any [locked objects](https://docs.aws.amazon.com/AmazonS3/latest/dev/object-lock-overview.html)) should be deleted from the bucket so that the bucket can be destroyed without error. These objects are *not* recoverable.
     */
    @JvmName("uyqrolhkbeauwrtk")
    public suspend fun forceDestroy(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.forceDestroy = mapped
    }

    /**
     * @param value An [ACL policy grant](https://docs.aws.amazon.com/AmazonS3/latest/dev/acl-overview.html#sample-acl) (documented below). Conflicts with `acl`.
     */
    @JvmName("xducewgovnjdjwli")
    public suspend fun grants(`value`: List<BucketGrantArgs>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.grants = mapped
    }

    /**
     * @param argument An [ACL policy grant](https://docs.aws.amazon.com/AmazonS3/latest/dev/acl-overview.html#sample-acl) (documented below). Conflicts with `acl`.
     */
    @JvmName("tebiuuxaofegxdqj")
    public suspend fun grants(argument: List<suspend BucketGrantArgsBuilder.() -> Unit>) {
        val toBeMapped = argument.toList().map { BucketGrantArgsBuilder().applySuspend { it() }.build() }
        val mapped = of(toBeMapped)
        this.grants = mapped
    }

    /**
     * @param argument An [ACL policy grant](https://docs.aws.amazon.com/AmazonS3/latest/dev/acl-overview.html#sample-acl) (documented below). Conflicts with `acl`.
     */
    @JvmName("hotujmjlfwmtfrvl")
    public suspend fun grants(vararg argument: suspend BucketGrantArgsBuilder.() -> Unit) {
        val toBeMapped = argument.toList().map { BucketGrantArgsBuilder().applySuspend { it() }.build() }
        val mapped = of(toBeMapped)
        this.grants = mapped
    }

    /**
     * @param argument An [ACL policy grant](https://docs.aws.amazon.com/AmazonS3/latest/dev/acl-overview.html#sample-acl) (documented below). Conflicts with `acl`.
     */
    @JvmName("visoiowccqpgelot")
    public suspend fun grants(argument: suspend BucketGrantArgsBuilder.() -> Unit) {
        val toBeMapped = listOf(BucketGrantArgsBuilder().applySuspend { argument() }.build())
        val mapped = of(toBeMapped)
        this.grants = mapped
    }

    /**
     * @param values An [ACL policy grant](https://docs.aws.amazon.com/AmazonS3/latest/dev/acl-overview.html#sample-acl) (documented below). Conflicts with `acl`.
     */
    @JvmName("upxetruuutnegcan")
    public suspend fun grants(vararg values: BucketGrantArgs) {
        val toBeMapped = values.toList()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.grants = mapped
    }

    /**
     * @param value The [Route 53 Hosted Zone ID](https://docs.aws.amazon.com/general/latest/gr/rande.html#s3_website_region_endpoints) for this bucket's region.
     */
    @JvmName("ldcjwpjllhsibfct")
    public suspend fun hostedZoneId(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.hostedZoneId = mapped
    }

    /**
     * @param value A configuration of [object lifecycle management](http://docs.aws.amazon.com/AmazonS3/latest/dev/object-lifecycle-mgmt.html) (documented below).
     */
    @JvmName("uskrqdqvwknwdsef")
    public suspend fun lifecycleRules(`value`: List<BucketLifecycleRuleArgs>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.lifecycleRules = mapped
    }

    /**
     * @param argument A configuration of [object lifecycle management](http://docs.aws.amazon.com/AmazonS3/latest/dev/object-lifecycle-mgmt.html) (documented below).
     */
    @JvmName("neipjukabyhxqhsj")
    public suspend
    fun lifecycleRules(argument: List<suspend BucketLifecycleRuleArgsBuilder.() -> Unit>) {
        val toBeMapped = argument.toList().map {
            BucketLifecycleRuleArgsBuilder().applySuspend {
                it()
            }.build()
        }
        val mapped = of(toBeMapped)
        this.lifecycleRules = mapped
    }

    /**
     * @param argument A configuration of [object lifecycle management](http://docs.aws.amazon.com/AmazonS3/latest/dev/object-lifecycle-mgmt.html) (documented below).
     */
    @JvmName("twlmvorlrbvgbkki")
    public suspend fun lifecycleRules(
        vararg
        argument: suspend BucketLifecycleRuleArgsBuilder.() -> Unit,
    ) {
        val toBeMapped = argument.toList().map {
            BucketLifecycleRuleArgsBuilder().applySuspend {
                it()
            }.build()
        }
        val mapped = of(toBeMapped)
        this.lifecycleRules = mapped
    }

    /**
     * @param argument A configuration of [object lifecycle management](http://docs.aws.amazon.com/AmazonS3/latest/dev/object-lifecycle-mgmt.html) (documented below).
     */
    @JvmName("wxhomiflxjgransl")
    public suspend fun lifecycleRules(argument: suspend BucketLifecycleRuleArgsBuilder.() -> Unit) {
        val toBeMapped = listOf(BucketLifecycleRuleArgsBuilder().applySuspend { argument() }.build())
        val mapped = of(toBeMapped)
        this.lifecycleRules = mapped
    }

    /**
     * @param values A configuration of [object lifecycle management](http://docs.aws.amazon.com/AmazonS3/latest/dev/object-lifecycle-mgmt.html) (documented below).
     */
    @JvmName("jirfjifnlgawbeck")
    public suspend fun lifecycleRules(vararg values: BucketLifecycleRuleArgs) {
        val toBeMapped = values.toList()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.lifecycleRules = mapped
    }

    /**
     * @param value A settings of [bucket logging](https://docs.aws.amazon.com/AmazonS3/latest/UG/ManagingBucketLogging.html) (documented below).
     */
    @JvmName("asicwwevyxakthmb")
    public suspend fun loggings(`value`: List<BucketLoggingArgs>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.loggings = mapped
    }

    /**
     * @param argument A settings of [bucket logging](https://docs.aws.amazon.com/AmazonS3/latest/UG/ManagingBucketLogging.html) (documented below).
     */
    @JvmName("kusfswhcgleoegos")
    public suspend fun loggings(argument: List<suspend BucketLoggingArgsBuilder.() -> Unit>) {
        val toBeMapped = argument.toList().map {
            BucketLoggingArgsBuilder().applySuspend { it() }.build()
        }
        val mapped = of(toBeMapped)
        this.loggings = mapped
    }

    /**
     * @param argument A settings of [bucket logging](https://docs.aws.amazon.com/AmazonS3/latest/UG/ManagingBucketLogging.html) (documented below).
     */
    @JvmName("yfyebbpsmqhamvie")
    public suspend fun loggings(vararg argument: suspend BucketLoggingArgsBuilder.() -> Unit) {
        val toBeMapped = argument.toList().map {
            BucketLoggingArgsBuilder().applySuspend { it() }.build()
        }
        val mapped = of(toBeMapped)
        this.loggings = mapped
    }

    /**
     * @param argument A settings of [bucket logging](https://docs.aws.amazon.com/AmazonS3/latest/UG/ManagingBucketLogging.html) (documented below).
     */
    @JvmName("kfkyphswpokrsgfj")
    public suspend fun loggings(argument: suspend BucketLoggingArgsBuilder.() -> Unit) {
        val toBeMapped = listOf(BucketLoggingArgsBuilder().applySuspend { argument() }.build())
        val mapped = of(toBeMapped)
        this.loggings = mapped
    }

    /**
     * @param values A settings of [bucket logging](https://docs.aws.amazon.com/AmazonS3/latest/UG/ManagingBucketLogging.html) (documented below).
     */
    @JvmName("cwhymqgomyqduvue")
    public suspend fun loggings(vararg values: BucketLoggingArgs) {
        val toBeMapped = values.toList()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.loggings = mapped
    }

    /**
     * @param value A configuration of [S3 object locking](https://docs.aws.amazon.com/AmazonS3/latest/dev/object-lock.html) (documented below)
     * > **NOTE:** You cannot use `acceleration_status` in `cn-north-1` or `us-gov-west-1`
     */
    @JvmName("pwvswcijcvaaatll")
    public suspend fun objectLockConfiguration(`value`: BucketObjectLockConfigurationArgs?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.objectLockConfiguration = mapped
    }

    /**
     * @param argument A configuration of [S3 object locking](https://docs.aws.amazon.com/AmazonS3/latest/dev/object-lock.html) (documented below)
     * > **NOTE:** You cannot use `acceleration_status` in `cn-north-1` or `us-gov-west-1`
     */
    @JvmName("cifewpaqxjnakgne")
    public suspend
    fun objectLockConfiguration(argument: suspend BucketObjectLockConfigurationArgsBuilder.() -> Unit) {
        val toBeMapped = BucketObjectLockConfigurationArgsBuilder().applySuspend { argument() }.build()
        val mapped = of(toBeMapped)
        this.objectLockConfiguration = mapped
    }

    /**
     * @param value A valid [bucket policy](https://docs.aws.amazon.com/AmazonS3/latest/dev/example-bucket-policies.html) JSON document. Note that if the policy document is not specific enough (but still valid), this provider may view the policy as constantly changing in a `pulumi preview`. In this case, please make sure you use the verbose/specific version of the policy.
     */
    @JvmName("jkdsntyovtbrhegd")
    public suspend fun policy(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.policy = mapped
    }

    /**
     * @param value A configuration of [replication configuration](http://docs.aws.amazon.com/AmazonS3/latest/dev/crr.html) (documented below).
     */
    @JvmName("dxbdjmxnkmurvela")
    public suspend fun replicationConfiguration(`value`: BucketReplicationConfigurationArgs?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.replicationConfiguration = mapped
    }

    /**
     * @param argument A configuration of [replication configuration](http://docs.aws.amazon.com/AmazonS3/latest/dev/crr.html) (documented below).
     */
    @JvmName("sjacwikvgiwlhqaw")
    public suspend
    fun replicationConfiguration(argument: suspend BucketReplicationConfigurationArgsBuilder.() -> Unit) {
        val toBeMapped = BucketReplicationConfigurationArgsBuilder().applySuspend { argument() }.build()
        val mapped = of(toBeMapped)
        this.replicationConfiguration = mapped
    }

    /**
     * @param value Specifies who should bear the cost of Amazon S3 data transfer.
     * Can be either `BucketOwner` or `Requester`. By default, the owner of the S3 bucket would incur
     * the costs of any data transfer. See [Requester Pays Buckets](http://docs.aws.amazon.com/AmazonS3/latest/dev/RequesterPaysBuckets.html)
     * developer guide for more information.
     */
    @JvmName("wblkighufvyhnwrb")
    public suspend fun requestPayer(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.requestPayer = mapped
    }

    /**
     * @param value A configuration of [server-side encryption configuration](http://docs.aws.amazon.com/AmazonS3/latest/dev/bucket-encryption.html) (documented below)
     */
    @JvmName("pcrtcdvkxwuluexh")
    public suspend
    fun serverSideEncryptionConfiguration(`value`: BucketServerSideEncryptionConfigurationArgs?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.serverSideEncryptionConfiguration = mapped
    }

    /**
     * @param argument A configuration of [server-side encryption configuration](http://docs.aws.amazon.com/AmazonS3/latest/dev/bucket-encryption.html) (documented below)
     */
    @JvmName("agrqroalypyjiiwp")
    public suspend
    fun serverSideEncryptionConfiguration(argument: suspend BucketServerSideEncryptionConfigurationArgsBuilder.() -> Unit) {
        val toBeMapped = BucketServerSideEncryptionConfigurationArgsBuilder().applySuspend {
            argument()
        }.build()
        val mapped = of(toBeMapped)
        this.serverSideEncryptionConfiguration = mapped
    }

    /**
     * @param value A map of tags to assign to the bucket. If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
     */
    @JvmName("upkwkkpwgiuexpys")
    public suspend fun tags(`value`: Map<String, String>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.tags = mapped
    }

    /**
     * @param values A map of tags to assign to the bucket. If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
     */
    @JvmName("vmfpkvbpdsopmlfj")
    public fun tags(vararg values: Pair<String, String>) {
        val toBeMapped = values.toMap()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.tags = mapped
    }

    /**
     * @param value A state of [versioning](https://docs.aws.amazon.com/AmazonS3/latest/dev/Versioning.html) (documented below)
     */
    @JvmName("vdavqwjqqoahdbwf")
    public suspend fun versioning(`value`: BucketVersioningArgs?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.versioning = mapped
    }

    /**
     * @param argument A state of [versioning](https://docs.aws.amazon.com/AmazonS3/latest/dev/Versioning.html) (documented below)
     */
    @JvmName("nioundpiaautgacj")
    public suspend fun versioning(argument: suspend BucketVersioningArgsBuilder.() -> Unit) {
        val toBeMapped = BucketVersioningArgsBuilder().applySuspend { argument() }.build()
        val mapped = of(toBeMapped)
        this.versioning = mapped
    }

    /**
     * @param value A website object (documented below).
     */
    @JvmName("afwomeylsecdnblq")
    public suspend fun website(`value`: BucketWebsiteArgs?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.website = mapped
    }

    /**
     * @param argument A website object (documented below).
     */
    @JvmName("gutceyptxruqvvbx")
    public suspend fun website(argument: suspend BucketWebsiteArgsBuilder.() -> Unit) {
        val toBeMapped = BucketWebsiteArgsBuilder().applySuspend { argument() }.build()
        val mapped = of(toBeMapped)
        this.website = mapped
    }

    /**
     * @param value The domain of the website endpoint, if the bucket is configured with a website. If not, this will be an empty string. This is used to create Route 53 alias records.
     */
    @JvmName("tntbmiljqldotcub")
    public suspend fun websiteDomain(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.websiteDomain = mapped
    }

    /**
     * @param value The website endpoint, if the bucket is configured with a website. If not, this will be an empty string.
     */
    @JvmName("joofhqfshguwqmep")
    public suspend fun websiteEndpoint(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.websiteEndpoint = mapped
    }

    internal fun build(): BucketArgs = BucketArgs(
        accelerationStatus = accelerationStatus,
        acl = acl,
        arn = arn,
        bucket = bucket,
        bucketPrefix = bucketPrefix,
        corsRules = corsRules,
        forceDestroy = forceDestroy,
        grants = grants,
        hostedZoneId = hostedZoneId,
        lifecycleRules = lifecycleRules,
        loggings = loggings,
        objectLockConfiguration = objectLockConfiguration,
        policy = policy,
        replicationConfiguration = replicationConfiguration,
        requestPayer = requestPayer,
        serverSideEncryptionConfiguration = serverSideEncryptionConfiguration,
        tags = tags,
        versioning = versioning,
        website = website,
        websiteDomain = websiteDomain,
        websiteEndpoint = websiteEndpoint,
    )
}
