@file:Suppress("NAME_SHADOWING", "DEPRECATION")

package com.pulumi.aws.s3.kotlin

import com.pulumi.aws.s3.BucketNotificationArgs.builder
import com.pulumi.aws.s3.kotlin.inputs.BucketNotificationLambdaFunctionArgs
import com.pulumi.aws.s3.kotlin.inputs.BucketNotificationLambdaFunctionArgsBuilder
import com.pulumi.aws.s3.kotlin.inputs.BucketNotificationQueueArgs
import com.pulumi.aws.s3.kotlin.inputs.BucketNotificationQueueArgsBuilder
import com.pulumi.aws.s3.kotlin.inputs.BucketNotificationTopicArgs
import com.pulumi.aws.s3.kotlin.inputs.BucketNotificationTopicArgsBuilder
import com.pulumi.core.Output
import com.pulumi.core.Output.of
import com.pulumi.kotlin.ConvertibleToJava
import com.pulumi.kotlin.PulumiTagMarker
import com.pulumi.kotlin.applySuspend
import kotlin.Boolean
import kotlin.String
import kotlin.Suppress
import kotlin.Unit
import kotlin.collections.List
import kotlin.jvm.JvmName

/**
 * Manages a S3 Bucket Notification Configuration. For additional information, see the [Configuring S3 Event Notifications section in the Amazon S3 Developer Guide](https://docs.aws.amazon.com/AmazonS3/latest/dev/NotificationHowTo.html).
 * > **NOTE:** S3 Buckets only support a single notification configuration. Declaring multiple `aws.s3.BucketNotification` resources to the same S3 Bucket will cause a perpetual difference in configuration. See the example "Trigger multiple Lambda functions" for an option.
 * > This resource cannot be used with S3 directory buckets.
 * ## Example Usage
 * ### Add notification configuration to SNS Topic
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.s3.BucketV2;
 * import com.pulumi.aws.iam.IamFunctions;
 * import com.pulumi.aws.iam.inputs.GetPolicyDocumentArgs;
 * import com.pulumi.aws.sns.Topic;
 * import com.pulumi.aws.sns.TopicArgs;
 * import com.pulumi.aws.s3.BucketNotification;
 * import com.pulumi.aws.s3.BucketNotificationArgs;
 * import com.pulumi.aws.s3.inputs.BucketNotificationTopicArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var bucket = new BucketV2("bucket");
 *         final var topicPolicyDocument = IamFunctions.getPolicyDocument(GetPolicyDocumentArgs.builder()
 *             .statements(GetPolicyDocumentStatementArgs.builder()
 *                 .effect("Allow")
 *                 .principals(GetPolicyDocumentStatementPrincipalArgs.builder()
 *                     .type("Service")
 *                     .identifiers("s3.amazonaws.com")
 *                     .build())
 *                 .actions("SNS:Publish")
 *                 .resources("arn:aws:sns:*:*:s3-event-notification-topic")
 *                 .conditions(GetPolicyDocumentStatementConditionArgs.builder()
 *                     .test("ArnLike")
 *                     .variable("aws:SourceArn")
 *                     .values(bucket.arn())
 *                     .build())
 *                 .build())
 *             .build());
 *         var topicTopic = new Topic("topicTopic", TopicArgs.builder()
 *             .policy(topicPolicyDocument.applyValue(getPolicyDocumentResult -> getPolicyDocumentResult).applyValue(topicPolicyDocument -> topicPolicyDocument.applyValue(getPolicyDocumentResult -> getPolicyDocumentResult.json())))
 *             .build());
 *         var bucketNotification = new BucketNotification("bucketNotification", BucketNotificationArgs.builder()
 *             .bucket(bucket.id())
 *             .topics(BucketNotificationTopicArgs.builder()
 *                 .topicArn(topicTopic.arn())
 *                 .events("s3:ObjectCreated:*")
 *                 .filterSuffix(".log")
 *                 .build())
 *             .build());
 *     }
 * }
 * ```
 * ### Add notification configuration to SQS Queue
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.s3.BucketV2;
 * import com.pulumi.aws.iam.IamFunctions;
 * import com.pulumi.aws.iam.inputs.GetPolicyDocumentArgs;
 * import com.pulumi.aws.sqs.Queue;
 * import com.pulumi.aws.sqs.QueueArgs;
 * import com.pulumi.aws.s3.BucketNotification;
 * import com.pulumi.aws.s3.BucketNotificationArgs;
 * import com.pulumi.aws.s3.inputs.BucketNotificationQueueArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var bucket = new BucketV2("bucket");
 *         final var queuePolicyDocument = IamFunctions.getPolicyDocument(GetPolicyDocumentArgs.builder()
 *             .statements(GetPolicyDocumentStatementArgs.builder()
 *                 .effect("Allow")
 *                 .principals(GetPolicyDocumentStatementPrincipalArgs.builder()
 *                     .type("*")
 *                     .identifiers("*")
 *                     .build())
 *                 .actions("sqs:SendMessage")
 *                 .resources("arn:aws:sqs:*:*:s3-event-notification-queue")
 *                 .conditions(GetPolicyDocumentStatementConditionArgs.builder()
 *                     .test("ArnEquals")
 *                     .variable("aws:SourceArn")
 *                     .values(bucket.arn())
 *                     .build())
 *                 .build())
 *             .build());
 *         var queueQueue = new Queue("queueQueue", QueueArgs.builder()
 *             .policy(queuePolicyDocument.applyValue(getPolicyDocumentResult -> getPolicyDocumentResult).applyValue(queuePolicyDocument -> queuePolicyDocument.applyValue(getPolicyDocumentResult -> getPolicyDocumentResult.json())))
 *             .build());
 *         var bucketNotification = new BucketNotification("bucketNotification", BucketNotificationArgs.builder()
 *             .bucket(bucket.id())
 *             .queues(BucketNotificationQueueArgs.builder()
 *                 .queueArn(queueQueue.arn())
 *                 .events("s3:ObjectCreated:*")
 *                 .filterSuffix(".log")
 *                 .build())
 *             .build());
 *     }
 * }
 * ```
 * ### Add notification configuration to Lambda Function
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.iam.IamFunctions;
 * import com.pulumi.aws.iam.inputs.GetPolicyDocumentArgs;
 * import com.pulumi.aws.iam.Role;
 * import com.pulumi.aws.iam.RoleArgs;
 * import com.pulumi.aws.lambda.Function;
 * import com.pulumi.aws.lambda.FunctionArgs;
 * import com.pulumi.aws.s3.BucketV2;
 * import com.pulumi.aws.lambda.Permission;
 * import com.pulumi.aws.lambda.PermissionArgs;
 * import com.pulumi.aws.s3.BucketNotification;
 * import com.pulumi.aws.s3.BucketNotificationArgs;
 * import com.pulumi.aws.s3.inputs.BucketNotificationLambdaFunctionArgs;
 * import com.pulumi.resources.CustomResourceOptions;
 * import com.pulumi.asset.FileArchive;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         final var assumeRole = IamFunctions.getPolicyDocument(GetPolicyDocumentArgs.builder()
 *             .statements(GetPolicyDocumentStatementArgs.builder()
 *                 .effect("Allow")
 *                 .principals(GetPolicyDocumentStatementPrincipalArgs.builder()
 *                     .type("Service")
 *                     .identifiers("lambda.amazonaws.com")
 *                     .build())
 *                 .actions("sts:AssumeRole")
 *                 .build())
 *             .build());
 *         var iamForLambda = new Role("iamForLambda", RoleArgs.builder()
 *             .assumeRolePolicy(assumeRole.applyValue(getPolicyDocumentResult -> getPolicyDocumentResult.json()))
 *             .build());
 *         var func = new Function("func", FunctionArgs.builder()
 *             .code(new FileArchive("your-function.zip"))
 *             .role(iamForLambda.arn())
 *             .handler("exports.example")
 *             .runtime("go1.x")
 *             .build());
 *         var bucket = new BucketV2("bucket");
 *         var allowBucket = new Permission("allowBucket", PermissionArgs.builder()
 *             .action("lambda:InvokeFunction")
 *             .function(func.arn())
 *             .principal("s3.amazonaws.com")
 *             .sourceArn(bucket.arn())
 *             .build());
 *         var bucketNotification = new BucketNotification("bucketNotification", BucketNotificationArgs.builder()
 *             .bucket(bucket.id())
 *             .lambdaFunctions(BucketNotificationLambdaFunctionArgs.builder()
 *                 .lambdaFunctionArn(func.arn())
 *                 .events("s3:ObjectCreated:*")
 *                 .filterPrefix("AWSLogs/")
 *                 .filterSuffix(".log")
 *                 .build())
 *             .build(), CustomResourceOptions.builder()
 *                 .dependsOn(allowBucket)
 *                 .build());
 *     }
 * }
 * ```
 * ### Trigger multiple Lambda functions
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.iam.IamFunctions;
 * import com.pulumi.aws.iam.inputs.GetPolicyDocumentArgs;
 * import com.pulumi.aws.iam.Role;
 * import com.pulumi.aws.iam.RoleArgs;
 * import com.pulumi.aws.lambda.Function;
 * import com.pulumi.aws.lambda.FunctionArgs;
 * import com.pulumi.aws.s3.BucketV2;
 * import com.pulumi.aws.lambda.Permission;
 * import com.pulumi.aws.lambda.PermissionArgs;
 * import com.pulumi.aws.s3.BucketNotification;
 * import com.pulumi.aws.s3.BucketNotificationArgs;
 * import com.pulumi.aws.s3.inputs.BucketNotificationLambdaFunctionArgs;
 * import com.pulumi.resources.CustomResourceOptions;
 * import com.pulumi.asset.FileArchive;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         final var assumeRole = IamFunctions.getPolicyDocument(GetPolicyDocumentArgs.builder()
 *             .effect("Allow")
 *             .principals(%!v(PANIC=Format method: runtime error: invalid memory address or nil pointer dereference))
 *             .actions("sts:AssumeRole")
 *             .build());
 *         var iamForLambda = new Role("iamForLambda", RoleArgs.builder()
 *             .assumeRolePolicy(assumeRole.applyValue(getPolicyDocumentResult -> getPolicyDocumentResult.json()))
 *             .build());
 *         var func1 = new Function("func1", FunctionArgs.builder()
 *             .code(new FileArchive("your-function1.zip"))
 *             .role(iamForLambda.arn())
 *             .handler("exports.example")
 *             .runtime("go1.x")
 *             .build());
 *         var bucket = new BucketV2("bucket");
 *         var allowBucket1 = new Permission("allowBucket1", PermissionArgs.builder()
 *             .action("lambda:InvokeFunction")
 *             .function(func1.arn())
 *             .principal("s3.amazonaws.com")
 *             .sourceArn(bucket.arn())
 *             .build());
 *         var func2 = new Function("func2", FunctionArgs.builder()
 *             .code(new FileArchive("your-function2.zip"))
 *             .role(iamForLambda.arn())
 *             .handler("exports.example")
 *             .build());
 *         var allowBucket2 = new Permission("allowBucket2", PermissionArgs.builder()
 *             .action("lambda:InvokeFunction")
 *             .function(func2.arn())
 *             .principal("s3.amazonaws.com")
 *             .sourceArn(bucket.arn())
 *             .build());
 *         var bucketNotification = new BucketNotification("bucketNotification", BucketNotificationArgs.builder()
 *             .bucket(bucket.id())
 *             .lambdaFunctions(
 *                 BucketNotificationLambdaFunctionArgs.builder()
 *                     .lambdaFunctionArn(func1.arn())
 *                     .events("s3:ObjectCreated:*")
 *                     .filterPrefix("AWSLogs/")
 *                     .filterSuffix(".log")
 *                     .build(),
 *                 BucketNotificationLambdaFunctionArgs.builder()
 *                     .lambdaFunctionArn(func2.arn())
 *                     .events("s3:ObjectCreated:*")
 *                     .filterPrefix("OtherLogs/")
 *                     .filterSuffix(".log")
 *                     .build())
 *             .build(), CustomResourceOptions.builder()
 *                 .dependsOn(
 *                     allowBucket1,
 *                     allowBucket2)
 *                 .build());
 *     }
 * }
 * ```
 * ### Add multiple notification configurations to SQS Queue
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.s3.BucketV2;
 * import com.pulumi.aws.iam.IamFunctions;
 * import com.pulumi.aws.iam.inputs.GetPolicyDocumentArgs;
 * import com.pulumi.aws.sqs.Queue;
 * import com.pulumi.aws.sqs.QueueArgs;
 * import com.pulumi.aws.s3.BucketNotification;
 * import com.pulumi.aws.s3.BucketNotificationArgs;
 * import com.pulumi.aws.s3.inputs.BucketNotificationQueueArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var bucket = new BucketV2("bucket");
 *         final var queuePolicyDocument = IamFunctions.getPolicyDocument(GetPolicyDocumentArgs.builder()
 *             .statements(GetPolicyDocumentStatementArgs.builder()
 *                 .effect("Allow")
 *                 .principals(GetPolicyDocumentStatementPrincipalArgs.builder()
 *                     .type("*")
 *                     .identifiers("*")
 *                     .build())
 *                 .actions("sqs:SendMessage")
 *                 .resources("arn:aws:sqs:*:*:s3-event-notification-queue")
 *                 .conditions(GetPolicyDocumentStatementConditionArgs.builder()
 *                     .test("ArnEquals")
 *                     .variable("aws:SourceArn")
 *                     .values(bucket.arn())
 *                     .build())
 *                 .build())
 *             .build());
 *         var queueQueue = new Queue("queueQueue", QueueArgs.builder()
 *             .policy(queuePolicyDocument.applyValue(getPolicyDocumentResult -> getPolicyDocumentResult).applyValue(queuePolicyDocument -> queuePolicyDocument.applyValue(getPolicyDocumentResult -> getPolicyDocumentResult.json())))
 *             .build());
 *         var bucketNotification = new BucketNotification("bucketNotification", BucketNotificationArgs.builder()
 *             .bucket(bucket.id())
 *             .queues(
 *                 BucketNotificationQueueArgs.builder()
 *                     .id("image-upload-event")
 *                     .queueArn(queueQueue.arn())
 *                     .events("s3:ObjectCreated:*")
 *                     .filterPrefix("images/")
 *                     .build(),
 *                 BucketNotificationQueueArgs.builder()
 *                     .id("video-upload-event")
 *                     .queueArn(queueQueue.arn())
 *                     .events("s3:ObjectCreated:*")
 *                     .filterPrefix("videos/")
 *                     .build())
 *             .build());
 *     }
 * }
 * ```
 * ### Emit events to EventBridge
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.s3.BucketV2;
 * import com.pulumi.aws.s3.BucketNotification;
 * import com.pulumi.aws.s3.BucketNotificationArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var bucket = new BucketV2("bucket");
 *         var bucketNotification = new BucketNotification("bucketNotification", BucketNotificationArgs.builder()
 *             .bucket(bucket.id())
 *             .eventbridge(true)
 *             .build());
 *     }
 * }
 * ```
 * ## Import
 * Using `pulumi import`, import S3 bucket notification using the `bucket`. For example:
 * ```sh
 *  $ pulumi import aws:s3/bucketNotification:BucketNotification bucket_notification bucket-name
 * ```
 * @property bucket Name of the bucket for notification configuration.
 * The following arguments are optional:
 * @property eventbridge Whether to enable Amazon EventBridge notifications. Defaults to `false`.
 * @property lambdaFunctions Used to configure notifications to a Lambda Function. See below.
 * @property queues Notification configuration to SQS Queue. See below.
 * @property topics Notification configuration to SNS Topic. See below.
 */
public data class BucketNotificationArgs(
    public val bucket: Output<String>? = null,
    public val eventbridge: Output<Boolean>? = null,
    public val lambdaFunctions: Output<List<BucketNotificationLambdaFunctionArgs>>? = null,
    public val queues: Output<List<BucketNotificationQueueArgs>>? = null,
    public val topics: Output<List<BucketNotificationTopicArgs>>? = null,
) : ConvertibleToJava<com.pulumi.aws.s3.BucketNotificationArgs> {
    override fun toJava(): com.pulumi.aws.s3.BucketNotificationArgs =
        com.pulumi.aws.s3.BucketNotificationArgs.builder()
            .bucket(bucket?.applyValue({ args0 -> args0 }))
            .eventbridge(eventbridge?.applyValue({ args0 -> args0 }))
            .lambdaFunctions(
                lambdaFunctions?.applyValue({ args0 ->
                    args0.map({ args0 ->
                        args0.let({ args0 ->
                            args0.toJava()
                        })
                    })
                }),
            )
            .queues(queues?.applyValue({ args0 -> args0.map({ args0 -> args0.let({ args0 -> args0.toJava() }) }) }))
            .topics(
                topics?.applyValue({ args0 ->
                    args0.map({ args0 ->
                        args0.let({ args0 ->
                            args0.toJava()
                        })
                    })
                }),
            ).build()
}

/**
 * Builder for [BucketNotificationArgs].
 */
@PulumiTagMarker
public class BucketNotificationArgsBuilder internal constructor() {
    private var bucket: Output<String>? = null

    private var eventbridge: Output<Boolean>? = null

    private var lambdaFunctions: Output<List<BucketNotificationLambdaFunctionArgs>>? = null

    private var queues: Output<List<BucketNotificationQueueArgs>>? = null

    private var topics: Output<List<BucketNotificationTopicArgs>>? = null

    /**
     * @param value Name of the bucket for notification configuration.
     * The following arguments are optional:
     */
    @JvmName("plpvjpfbnhielfyt")
    public suspend fun bucket(`value`: Output<String>) {
        this.bucket = value
    }

    /**
     * @param value Whether to enable Amazon EventBridge notifications. Defaults to `false`.
     */
    @JvmName("egmhyslnxoamlfol")
    public suspend fun eventbridge(`value`: Output<Boolean>) {
        this.eventbridge = value
    }

    /**
     * @param value Used to configure notifications to a Lambda Function. See below.
     */
    @JvmName("kuhnhaddjudeagtu")
    public suspend fun lambdaFunctions(`value`: Output<List<BucketNotificationLambdaFunctionArgs>>) {
        this.lambdaFunctions = value
    }

    @JvmName("hcxgvfaerxkjtyxr")
    public suspend fun lambdaFunctions(vararg values: Output<BucketNotificationLambdaFunctionArgs>) {
        this.lambdaFunctions = Output.all(values.asList())
    }

    /**
     * @param values Used to configure notifications to a Lambda Function. See below.
     */
    @JvmName("uuslsqqqaefpormj")
    public suspend fun lambdaFunctions(values: List<Output<BucketNotificationLambdaFunctionArgs>>) {
        this.lambdaFunctions = Output.all(values)
    }

    /**
     * @param value Notification configuration to SQS Queue. See below.
     */
    @JvmName("isogaeqnjynkmtnb")
    public suspend fun queues(`value`: Output<List<BucketNotificationQueueArgs>>) {
        this.queues = value
    }

    @JvmName("fprrgtfnbdikbygx")
    public suspend fun queues(vararg values: Output<BucketNotificationQueueArgs>) {
        this.queues = Output.all(values.asList())
    }

    /**
     * @param values Notification configuration to SQS Queue. See below.
     */
    @JvmName("nqqymagabdxsdnun")
    public suspend fun queues(values: List<Output<BucketNotificationQueueArgs>>) {
        this.queues = Output.all(values)
    }

    /**
     * @param value Notification configuration to SNS Topic. See below.
     */
    @JvmName("ofnhokucgfyyacrt")
    public suspend fun topics(`value`: Output<List<BucketNotificationTopicArgs>>) {
        this.topics = value
    }

    @JvmName("bfyolvlmpewvmuvr")
    public suspend fun topics(vararg values: Output<BucketNotificationTopicArgs>) {
        this.topics = Output.all(values.asList())
    }

    /**
     * @param values Notification configuration to SNS Topic. See below.
     */
    @JvmName("nonjieaejlyqcrvo")
    public suspend fun topics(values: List<Output<BucketNotificationTopicArgs>>) {
        this.topics = Output.all(values)
    }

    /**
     * @param value Name of the bucket for notification configuration.
     * The following arguments are optional:
     */
    @JvmName("dwnwrtrnhdrelxre")
    public suspend fun bucket(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.bucket = mapped
    }

    /**
     * @param value Whether to enable Amazon EventBridge notifications. Defaults to `false`.
     */
    @JvmName("wgaqghtngtimxudk")
    public suspend fun eventbridge(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.eventbridge = mapped
    }

    /**
     * @param value Used to configure notifications to a Lambda Function. See below.
     */
    @JvmName("vafexreakballfjf")
    public suspend fun lambdaFunctions(`value`: List<BucketNotificationLambdaFunctionArgs>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.lambdaFunctions = mapped
    }

    /**
     * @param argument Used to configure notifications to a Lambda Function. See below.
     */
    @JvmName("vdhlyslpyjckwqyp")
    public suspend
    fun lambdaFunctions(argument: List<suspend BucketNotificationLambdaFunctionArgsBuilder.() -> Unit>) {
        val toBeMapped = argument.toList().map {
            BucketNotificationLambdaFunctionArgsBuilder().applySuspend { it() }.build()
        }
        val mapped = of(toBeMapped)
        this.lambdaFunctions = mapped
    }

    /**
     * @param argument Used to configure notifications to a Lambda Function. See below.
     */
    @JvmName("xdhmccewjjjymjes")
    public suspend fun lambdaFunctions(
        vararg
        argument: suspend BucketNotificationLambdaFunctionArgsBuilder.() -> Unit,
    ) {
        val toBeMapped = argument.toList().map {
            BucketNotificationLambdaFunctionArgsBuilder().applySuspend { it() }.build()
        }
        val mapped = of(toBeMapped)
        this.lambdaFunctions = mapped
    }

    /**
     * @param argument Used to configure notifications to a Lambda Function. See below.
     */
    @JvmName("vdcsmugfcxldwiwp")
    public suspend
    fun lambdaFunctions(argument: suspend BucketNotificationLambdaFunctionArgsBuilder.() -> Unit) {
        val toBeMapped = listOf(
            BucketNotificationLambdaFunctionArgsBuilder().applySuspend {
                argument()
            }.build(),
        )
        val mapped = of(toBeMapped)
        this.lambdaFunctions = mapped
    }

    /**
     * @param values Used to configure notifications to a Lambda Function. See below.
     */
    @JvmName("byuvbvgxhuuyemlg")
    public suspend fun lambdaFunctions(vararg values: BucketNotificationLambdaFunctionArgs) {
        val toBeMapped = values.toList()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.lambdaFunctions = mapped
    }

    /**
     * @param value Notification configuration to SQS Queue. See below.
     */
    @JvmName("oquunmqacwgfwpdg")
    public suspend fun queues(`value`: List<BucketNotificationQueueArgs>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.queues = mapped
    }

    /**
     * @param argument Notification configuration to SQS Queue. See below.
     */
    @JvmName("pjovogupcqlgxgts")
    public suspend fun queues(argument: List<suspend BucketNotificationQueueArgsBuilder.() -> Unit>) {
        val toBeMapped = argument.toList().map {
            BucketNotificationQueueArgsBuilder().applySuspend {
                it()
            }.build()
        }
        val mapped = of(toBeMapped)
        this.queues = mapped
    }

    /**
     * @param argument Notification configuration to SQS Queue. See below.
     */
    @JvmName("ebwuqpnubxokgpbt")
    public suspend fun queues(
        vararg
        argument: suspend BucketNotificationQueueArgsBuilder.() -> Unit,
    ) {
        val toBeMapped = argument.toList().map {
            BucketNotificationQueueArgsBuilder().applySuspend {
                it()
            }.build()
        }
        val mapped = of(toBeMapped)
        this.queues = mapped
    }

    /**
     * @param argument Notification configuration to SQS Queue. See below.
     */
    @JvmName("keakrolxlyopxkxn")
    public suspend fun queues(argument: suspend BucketNotificationQueueArgsBuilder.() -> Unit) {
        val toBeMapped = listOf(
            BucketNotificationQueueArgsBuilder().applySuspend {
                argument()
            }.build(),
        )
        val mapped = of(toBeMapped)
        this.queues = mapped
    }

    /**
     * @param values Notification configuration to SQS Queue. See below.
     */
    @JvmName("biwwdithpmbvoxhu")
    public suspend fun queues(vararg values: BucketNotificationQueueArgs) {
        val toBeMapped = values.toList()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.queues = mapped
    }

    /**
     * @param value Notification configuration to SNS Topic. See below.
     */
    @JvmName("oyeeslfqicqbahhb")
    public suspend fun topics(`value`: List<BucketNotificationTopicArgs>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.topics = mapped
    }

    /**
     * @param argument Notification configuration to SNS Topic. See below.
     */
    @JvmName("psrfklbdyjolrttv")
    public suspend fun topics(argument: List<suspend BucketNotificationTopicArgsBuilder.() -> Unit>) {
        val toBeMapped = argument.toList().map {
            BucketNotificationTopicArgsBuilder().applySuspend {
                it()
            }.build()
        }
        val mapped = of(toBeMapped)
        this.topics = mapped
    }

    /**
     * @param argument Notification configuration to SNS Topic. See below.
     */
    @JvmName("dfuobwkecpscsouo")
    public suspend fun topics(
        vararg
        argument: suspend BucketNotificationTopicArgsBuilder.() -> Unit,
    ) {
        val toBeMapped = argument.toList().map {
            BucketNotificationTopicArgsBuilder().applySuspend {
                it()
            }.build()
        }
        val mapped = of(toBeMapped)
        this.topics = mapped
    }

    /**
     * @param argument Notification configuration to SNS Topic. See below.
     */
    @JvmName("cmokmpvuoanavcva")
    public suspend fun topics(argument: suspend BucketNotificationTopicArgsBuilder.() -> Unit) {
        val toBeMapped = listOf(
            BucketNotificationTopicArgsBuilder().applySuspend {
                argument()
            }.build(),
        )
        val mapped = of(toBeMapped)
        this.topics = mapped
    }

    /**
     * @param values Notification configuration to SNS Topic. See below.
     */
    @JvmName("sbegbbyqlvngvnap")
    public suspend fun topics(vararg values: BucketNotificationTopicArgs) {
        val toBeMapped = values.toList()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.topics = mapped
    }

    internal fun build(): BucketNotificationArgs = BucketNotificationArgs(
        bucket = bucket,
        eventbridge = eventbridge,
        lambdaFunctions = lambdaFunctions,
        queues = queues,
        topics = topics,
    )
}
