@file:Suppress("NAME_SHADOWING", "DEPRECATION")

package com.pulumi.aws.shield.kotlin

import com.pulumi.aws.shield.ProtectionGroupArgs.builder
import com.pulumi.core.Output
import com.pulumi.core.Output.of
import com.pulumi.kotlin.ConvertibleToJava
import com.pulumi.kotlin.PulumiTagMarker
import kotlin.Pair
import kotlin.String
import kotlin.Suppress
import kotlin.collections.List
import kotlin.collections.Map
import kotlin.jvm.JvmName

/**
 * Creates a grouping of protected resources so they can be handled as a collective.
 * This resource grouping improves the accuracy of detection and reduces false positives. For more information see
 * [Managing AWS Shield Advanced protection groups](https://docs.aws.amazon.com/waf/latest/developerguide/manage-protection-group.html)
 * ## Example Usage
 * ### Create protection group for all resources
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.shield.ProtectionGroup;
 * import com.pulumi.aws.shield.ProtectionGroupArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var example = new ProtectionGroup("example", ProtectionGroupArgs.builder()
 *             .aggregation("MAX")
 *             .pattern("ALL")
 *             .protectionGroupId("example")
 *             .build());
 *     }
 * }
 * ```
 * ### Create protection group for arbitrary number of resources
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.AwsFunctions;
 * import com.pulumi.aws.inputs.GetRegionArgs;
 * import com.pulumi.aws.inputs.GetCallerIdentityArgs;
 * import com.pulumi.aws.ec2.Eip;
 * import com.pulumi.aws.ec2.EipArgs;
 * import com.pulumi.aws.shield.Protection;
 * import com.pulumi.aws.shield.ProtectionArgs;
 * import com.pulumi.aws.shield.ProtectionGroup;
 * import com.pulumi.aws.shield.ProtectionGroupArgs;
 * import com.pulumi.resources.CustomResourceOptions;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         final var currentRegion = AwsFunctions.getRegion();
 *         final var currentCallerIdentity = AwsFunctions.getCallerIdentity();
 *         var exampleEip = new Eip("exampleEip", EipArgs.builder()
 *             .domain("vpc")
 *             .build());
 *         var exampleProtection = new Protection("exampleProtection", ProtectionArgs.builder()
 *             .resourceArn(exampleEip.id().applyValue(id -> String.format("arn:aws:ec2:%s:%s:eip-allocation/%s", currentRegion.applyValue(getRegionResult -> getRegionResult.name()),currentCallerIdentity.applyValue(getCallerIdentityResult -> getCallerIdentityResult.accountId()),id)))
 *             .build());
 *         var exampleProtectionGroup = new ProtectionGroup("exampleProtectionGroup", ProtectionGroupArgs.builder()
 *             .protectionGroupId("example")
 *             .aggregation("MEAN")
 *             .pattern("ARBITRARY")
 *             .members(exampleEip.id().applyValue(id -> String.format("arn:aws:ec2:%s:%s:eip-allocation/%s", currentRegion.applyValue(getRegionResult -> getRegionResult.name()),currentCallerIdentity.applyValue(getCallerIdentityResult -> getCallerIdentityResult.accountId()),id)))
 *             .build(), CustomResourceOptions.builder()
 *                 .dependsOn(exampleProtection)
 *                 .build());
 *     }
 * }
 * ```
 * ### Create protection group for a type of resource
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.shield.ProtectionGroup;
 * import com.pulumi.aws.shield.ProtectionGroupArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var example = new ProtectionGroup("example", ProtectionGroupArgs.builder()
 *             .aggregation("SUM")
 *             .pattern("BY_RESOURCE_TYPE")
 *             .protectionGroupId("example")
 *             .resourceType("ELASTIC_IP_ALLOCATION")
 *             .build());
 *     }
 * }
 * ```
 * ## Import
 * Using `pulumi import`, import Shield protection group resources using their protection group id. For example:
 * ```sh
 *  $ pulumi import aws:shield/protectionGroup:ProtectionGroup example example
 * ```
 * @property aggregation Defines how AWS Shield combines resource data for the group in order to detect, mitigate, and report events.
 * @property members The Amazon Resource Names (ARNs) of the resources to include in the protection group. You must set this when you set `pattern` to ARBITRARY and you must not set it for any other `pattern` setting.
 * @property pattern The criteria to use to choose the protected resources for inclusion in the group.
 * @property protectionGroupId The name of the protection group.
 * @property resourceType The resource type to include in the protection group. You must set this when you set `pattern` to BY_RESOURCE_TYPE and you must not set it for any other `pattern` setting.
 * @property tags Key-value map of resource tags. If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
 */
public data class ProtectionGroupArgs(
    public val aggregation: Output<String>? = null,
    public val members: Output<List<String>>? = null,
    public val pattern: Output<String>? = null,
    public val protectionGroupId: Output<String>? = null,
    public val resourceType: Output<String>? = null,
    public val tags: Output<Map<String, String>>? = null,
) : ConvertibleToJava<com.pulumi.aws.shield.ProtectionGroupArgs> {
    override fun toJava(): com.pulumi.aws.shield.ProtectionGroupArgs =
        com.pulumi.aws.shield.ProtectionGroupArgs.builder()
            .aggregation(aggregation?.applyValue({ args0 -> args0 }))
            .members(members?.applyValue({ args0 -> args0.map({ args0 -> args0 }) }))
            .pattern(pattern?.applyValue({ args0 -> args0 }))
            .protectionGroupId(protectionGroupId?.applyValue({ args0 -> args0 }))
            .resourceType(resourceType?.applyValue({ args0 -> args0 }))
            .tags(
                tags?.applyValue({ args0 ->
                    args0.map({ args0 ->
                        args0.key.to(args0.value)
                    }).toMap()
                }),
            ).build()
}

/**
 * Builder for [ProtectionGroupArgs].
 */
@PulumiTagMarker
public class ProtectionGroupArgsBuilder internal constructor() {
    private var aggregation: Output<String>? = null

    private var members: Output<List<String>>? = null

    private var pattern: Output<String>? = null

    private var protectionGroupId: Output<String>? = null

    private var resourceType: Output<String>? = null

    private var tags: Output<Map<String, String>>? = null

    /**
     * @param value Defines how AWS Shield combines resource data for the group in order to detect, mitigate, and report events.
     */
    @JvmName("wonfmadqkokofufa")
    public suspend fun aggregation(`value`: Output<String>) {
        this.aggregation = value
    }

    /**
     * @param value The Amazon Resource Names (ARNs) of the resources to include in the protection group. You must set this when you set `pattern` to ARBITRARY and you must not set it for any other `pattern` setting.
     */
    @JvmName("cmdpbtpfvnynagpe")
    public suspend fun members(`value`: Output<List<String>>) {
        this.members = value
    }

    @JvmName("unbcempukxaoyhsi")
    public suspend fun members(vararg values: Output<String>) {
        this.members = Output.all(values.asList())
    }

    /**
     * @param values The Amazon Resource Names (ARNs) of the resources to include in the protection group. You must set this when you set `pattern` to ARBITRARY and you must not set it for any other `pattern` setting.
     */
    @JvmName("pvejlijtbyvgvrjw")
    public suspend fun members(values: List<Output<String>>) {
        this.members = Output.all(values)
    }

    /**
     * @param value The criteria to use to choose the protected resources for inclusion in the group.
     */
    @JvmName("egybgxcgbanqasrr")
    public suspend fun pattern(`value`: Output<String>) {
        this.pattern = value
    }

    /**
     * @param value The name of the protection group.
     */
    @JvmName("odceqbgebfxrfqmt")
    public suspend fun protectionGroupId(`value`: Output<String>) {
        this.protectionGroupId = value
    }

    /**
     * @param value The resource type to include in the protection group. You must set this when you set `pattern` to BY_RESOURCE_TYPE and you must not set it for any other `pattern` setting.
     */
    @JvmName("gekxshtaedrggelu")
    public suspend fun resourceType(`value`: Output<String>) {
        this.resourceType = value
    }

    /**
     * @param value Key-value map of resource tags. If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
     */
    @JvmName("xnchipsvrsuuukxa")
    public suspend fun tags(`value`: Output<Map<String, String>>) {
        this.tags = value
    }

    /**
     * @param value Defines how AWS Shield combines resource data for the group in order to detect, mitigate, and report events.
     */
    @JvmName("ugaqhskwymwlcijh")
    public suspend fun aggregation(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.aggregation = mapped
    }

    /**
     * @param value The Amazon Resource Names (ARNs) of the resources to include in the protection group. You must set this when you set `pattern` to ARBITRARY and you must not set it for any other `pattern` setting.
     */
    @JvmName("wydtqyhognfccinf")
    public suspend fun members(`value`: List<String>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.members = mapped
    }

    /**
     * @param values The Amazon Resource Names (ARNs) of the resources to include in the protection group. You must set this when you set `pattern` to ARBITRARY and you must not set it for any other `pattern` setting.
     */
    @JvmName("sfucdlwoenmtktpo")
    public suspend fun members(vararg values: String) {
        val toBeMapped = values.toList()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.members = mapped
    }

    /**
     * @param value The criteria to use to choose the protected resources for inclusion in the group.
     */
    @JvmName("wuqseprjovpoygmv")
    public suspend fun pattern(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.pattern = mapped
    }

    /**
     * @param value The name of the protection group.
     */
    @JvmName("sgqotokywsioteus")
    public suspend fun protectionGroupId(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.protectionGroupId = mapped
    }

    /**
     * @param value The resource type to include in the protection group. You must set this when you set `pattern` to BY_RESOURCE_TYPE and you must not set it for any other `pattern` setting.
     */
    @JvmName("tkudwdjvspmgdpku")
    public suspend fun resourceType(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.resourceType = mapped
    }

    /**
     * @param value Key-value map of resource tags. If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
     */
    @JvmName("touvutxalnqhmwff")
    public suspend fun tags(`value`: Map<String, String>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.tags = mapped
    }

    /**
     * @param values Key-value map of resource tags. If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
     */
    @JvmName("pgwyprmvakmlxcwc")
    public fun tags(vararg values: Pair<String, String>) {
        val toBeMapped = values.toMap()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.tags = mapped
    }

    internal fun build(): ProtectionGroupArgs = ProtectionGroupArgs(
        aggregation = aggregation,
        members = members,
        pattern = pattern,
        protectionGroupId = protectionGroupId,
        resourceType = resourceType,
        tags = tags,
    )
}
