@file:Suppress("NAME_SHADOWING", "DEPRECATION")

package com.pulumi.aws.sqs.kotlin

import com.pulumi.aws.sqs.QueueArgs.builder
import com.pulumi.core.Output
import com.pulumi.core.Output.of
import com.pulumi.kotlin.ConvertibleToJava
import com.pulumi.kotlin.PulumiTagMarker
import kotlin.Boolean
import kotlin.Int
import kotlin.Pair
import kotlin.String
import kotlin.Suppress
import kotlin.collections.Map
import kotlin.jvm.JvmName

/**
 * ## Example Usage
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.sqs.Queue;
 * import com.pulumi.aws.sqs.QueueArgs;
 * import static com.pulumi.codegen.internal.Serialization.*;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var queue = new Queue("queue", QueueArgs.builder()
 *             .delaySeconds(90)
 *             .maxMessageSize(2048)
 *             .messageRetentionSeconds(86400)
 *             .receiveWaitTimeSeconds(10)
 *             .redrivePolicy(serializeJson(
 *                 jsonObject(
 *                     jsonProperty("deadLetterTargetArn", aws_sqs_queue.queue_deadletter().arn()),
 *                     jsonProperty("maxReceiveCount", 4)
 *                 )))
 *             .tags(Map.of("Environment", "production"))
 *             .build());
 *     }
 * }
 * ```
 * ## FIFO queue
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const queue = new aws.sqs.Queue("queue", {
 *     contentBasedDeduplication: true,
 *     fifoQueue: true,
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * queue = aws.sqs.Queue("queue",
 *     content_based_deduplication=True,
 *     fifo_queue=True)
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var queue = new Aws.Sqs.Queue("queue", new()
 *     {
 *         ContentBasedDeduplication = true,
 *         FifoQueue = true,
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/sqs"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		_, err := sqs.NewQueue(ctx, "queue", &sqs.QueueArgs{
 * 			ContentBasedDeduplication: pulumi.Bool(true),
 * 			FifoQueue:                 pulumi.Bool(true),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.sqs.Queue;
 * import com.pulumi.aws.sqs.QueueArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var queue = new Queue("queue", QueueArgs.builder()
 *             .contentBasedDeduplication(true)
 *             .fifoQueue(true)
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   queue:
 *     type: aws:sqs:Queue
 *     properties:
 *       contentBasedDeduplication: true
 *       fifoQueue: true
 * ```
 * ## High-throughput FIFO queue
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const queue = new aws.sqs.Queue("queue", {
 *     deduplicationScope: "messageGroup",
 *     fifoQueue: true,
 *     fifoThroughputLimit: "perMessageGroupId",
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * queue = aws.sqs.Queue("queue",
 *     deduplication_scope="messageGroup",
 *     fifo_queue=True,
 *     fifo_throughput_limit="perMessageGroupId")
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var queue = new Aws.Sqs.Queue("queue", new()
 *     {
 *         DeduplicationScope = "messageGroup",
 *         FifoQueue = true,
 *         FifoThroughputLimit = "perMessageGroupId",
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/sqs"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		_, err := sqs.NewQueue(ctx, "queue", &sqs.QueueArgs{
 * 			DeduplicationScope:  pulumi.String("messageGroup"),
 * 			FifoQueue:           pulumi.Bool(true),
 * 			FifoThroughputLimit: pulumi.String("perMessageGroupId"),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.sqs.Queue;
 * import com.pulumi.aws.sqs.QueueArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var queue = new Queue("queue", QueueArgs.builder()
 *             .deduplicationScope("messageGroup")
 *             .fifoQueue(true)
 *             .fifoThroughputLimit("perMessageGroupId")
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   queue:
 *     type: aws:sqs:Queue
 *     properties:
 *       deduplicationScope: messageGroup
 *       fifoQueue: true
 *       fifoThroughputLimit: perMessageGroupId
 * ```
 * ## Dead-letter queue
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const exampleQueueDeadletter = new aws.sqs.Queue("exampleQueueDeadletter", {redriveAllowPolicy: JSON.stringify({
 *     redrivePermission: "byQueue",
 *     sourceQueueArns: [aws_sqs_queue&#46;example_queue&#46;arn],
 * })});
 * ```
 * ```python
 * import pulumi
 * import json
 * import pulumi_aws as aws
 * example_queue_deadletter = aws.sqs.Queue("exampleQueueDeadletter", redrive_allow_policy=json.dumps({
 *     "redrivePermission": "byQueue",
 *     "sourceQueueArns": [aws_sqs_queue["example_queue"]["arn"]],
 * }))
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using System.Text.Json;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var exampleQueueDeadletter = new Aws.Sqs.Queue("exampleQueueDeadletter", new()
 *     {
 *         RedriveAllowPolicy = JsonSerializer.Serialize(new Dictionary<string, object?>
 *         {
 *             ["redrivePermission"] = "byQueue",
 *             ["sourceQueueArns"] = new[]
 *             {
 *                 aws_sqs_queue.Example_queue.Arn,
 *             },
 *         }),
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"encoding/json"
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/sqs"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		tmpJSON0, err := json.Marshal(map[string]interface{}{
 * 			"redrivePermission": "byQueue",
 * 			"sourceQueueArns": []interface{}{
 * 				aws_sqs_queue.Example_queue.Arn,
 * 			},
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		json0 := string(tmpJSON0)
 * 		_, err = sqs.NewQueue(ctx, "exampleQueueDeadletter", &sqs.QueueArgs{
 * 			RedriveAllowPolicy: pulumi.String(json0),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.sqs.Queue;
 * import com.pulumi.aws.sqs.QueueArgs;
 * import static com.pulumi.codegen.internal.Serialization.*;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var exampleQueueDeadletter = new Queue("exampleQueueDeadletter", QueueArgs.builder()
 *             .redriveAllowPolicy(serializeJson(
 *                 jsonObject(
 *                     jsonProperty("redrivePermission", "byQueue"),
 *                     jsonProperty("sourceQueueArns", jsonArray(aws_sqs_queue.example_queue().arn()))
 *                 )))
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   exampleQueueDeadletter:
 *     type: aws:sqs:Queue
 *     properties:
 *       redriveAllowPolicy:
 *         fn::toJSON:
 *           redrivePermission: byQueue
 *           sourceQueueArns:
 *             - ${aws_sqs_queue.example_queue.arn}
 * ```
 * ## Server-side encryption (SSE)
 * Using [SSE-SQS](https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/sqs-configure-sqs-sse-queue.html):
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const queue = new aws.sqs.Queue("queue", {sqsManagedSseEnabled: true});
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * queue = aws.sqs.Queue("queue", sqs_managed_sse_enabled=True)
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var queue = new Aws.Sqs.Queue("queue", new()
 *     {
 *         SqsManagedSseEnabled = true,
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/sqs"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		_, err := sqs.NewQueue(ctx, "queue", &sqs.QueueArgs{
 * 			SqsManagedSseEnabled: pulumi.Bool(true),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.sqs.Queue;
 * import com.pulumi.aws.sqs.QueueArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var queue = new Queue("queue", QueueArgs.builder()
 *             .sqsManagedSseEnabled(true)
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   queue:
 *     type: aws:sqs:Queue
 *     properties:
 *       sqsManagedSseEnabled: true
 * ```
 * Using [SSE-KMS](https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/sqs-configure-sse-existing-queue.html):
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const queue = new aws.sqs.Queue("queue", {
 *     kmsDataKeyReusePeriodSeconds: 300,
 *     kmsMasterKeyId: "alias/aws/sqs",
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * queue = aws.sqs.Queue("queue",
 *     kms_data_key_reuse_period_seconds=300,
 *     kms_master_key_id="alias/aws/sqs")
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var queue = new Aws.Sqs.Queue("queue", new()
 *     {
 *         KmsDataKeyReusePeriodSeconds = 300,
 *         KmsMasterKeyId = "alias/aws/sqs",
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/sqs"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		_, err := sqs.NewQueue(ctx, "queue", &sqs.QueueArgs{
 * 			KmsDataKeyReusePeriodSeconds: pulumi.Int(300),
 * 			KmsMasterKeyId:               pulumi.String("alias/aws/sqs"),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.sqs.Queue;
 * import com.pulumi.aws.sqs.QueueArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var queue = new Queue("queue", QueueArgs.builder()
 *             .kmsDataKeyReusePeriodSeconds(300)
 *             .kmsMasterKeyId("alias/aws/sqs")
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   queue:
 *     type: aws:sqs:Queue
 *     properties:
 *       kmsDataKeyReusePeriodSeconds: 300
 *       kmsMasterKeyId: alias/aws/sqs
 * ```
 * ## Import
 * Using `pulumi import`, import SQS Queues using the queue `url`. For example:
 * ```sh
 *  $ pulumi import aws:sqs/queue:Queue public_queue https://queue.amazonaws.com/80398EXAMPLE/MyQueue
 * ```
 * @property contentBasedDeduplication Enables content-based deduplication for FIFO queues. For more information, see the [related documentation](http://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/FIFO-queues.html#FIFO-queues-exactly-once-processing)
 * @property deduplicationScope Specifies whether message deduplication occurs at the message group or queue level. Valid values are `messageGroup` and `queue` (default).
 * @property delaySeconds The time in seconds that the delivery of all messages in the queue will be delayed. An integer from 0 to 900 (15 minutes). The default for this attribute is 0 seconds.
 * @property fifoQueue Boolean designating a FIFO queue. If not set, it defaults to `false` making it standard.
 * @property fifoThroughputLimit Specifies whether the FIFO queue throughput quota applies to the entire queue or per message group. Valid values are `perQueue` (default) and `perMessageGroupId`.
 * @property kmsDataKeyReusePeriodSeconds The length of time, in seconds, for which Amazon SQS can reuse a data key to encrypt or decrypt messages before calling AWS KMS again. An integer representing seconds, between 60 seconds (1 minute) and 86,400 seconds (24 hours). The default is 300 (5 minutes).
 * @property kmsMasterKeyId The ID of an AWS-managed customer master key (CMK) for Amazon SQS or a custom CMK. For more information, see [Key Terms](http://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/sqs-server-side-encryption.html#sqs-sse-key-terms).
 * @property maxMessageSize The limit of how many bytes a message can contain before Amazon SQS rejects it. An integer from 1024 bytes (1 KiB) up to 262144 bytes (256 KiB). The default for this attribute is 262144 (256 KiB).
 * @property messageRetentionSeconds The number of seconds Amazon SQS retains a message. Integer representing seconds, from 60 (1 minute) to 1209600 (14 days). The default for this attribute is 345600 (4 days).
 * @property name The name of the queue. Queue names must be made up of only uppercase and lowercase ASCII letters, numbers, underscores, and hyphens, and must be between 1 and 80 characters long. For a FIFO (first-in-first-out) queue, the name must end with the `.fifo` suffix. If omitted, this provider will assign a random, unique name. Conflicts with `name_prefix`
 * @property namePrefix Creates a unique name beginning with the specified prefix. Conflicts with `name`
 * @property policy The JSON policy for the SQS queue.
 * @property receiveWaitTimeSeconds The time for which a ReceiveMessage call will wait for a message to arrive (long polling) before returning. An integer from 0 to 20 (seconds). The default for this attribute is 0, meaning that the call will return immediately.
 * @property redriveAllowPolicy The JSON policy to set up the Dead Letter Queue redrive permission, see [AWS docs](https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/SQSDeadLetterQueue.html).
 * @property redrivePolicy The JSON policy to set up the Dead Letter Queue, see [AWS docs](https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/SQSDeadLetterQueue.html). **Note:** when specifying `maxReceiveCount`, you must specify it as an integer (`5`), and not a string (`"5"`).
 * @property sqsManagedSseEnabled Boolean to enable server-side encryption (SSE) of message content with SQS-owned encryption keys. See [Encryption at rest](https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/sqs-server-side-encryption.html). The provider will only perform drift detection of its value when present in a configuration.
 * @property tags A map of tags to assign to the queue. If configured with a provider `default_tags` configuration block) present, tags with matching keys will overwrite those defined at the provider-level.
 * @property visibilityTimeoutSeconds The visibility timeout for the queue. An integer from 0 to 43200 (12 hours). The default for this attribute is 30. For more information about visibility timeout, see [AWS docs](https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/AboutVT.html).
 */
public data class QueueArgs(
    public val contentBasedDeduplication: Output<Boolean>? = null,
    public val deduplicationScope: Output<String>? = null,
    public val delaySeconds: Output<Int>? = null,
    public val fifoQueue: Output<Boolean>? = null,
    public val fifoThroughputLimit: Output<String>? = null,
    public val kmsDataKeyReusePeriodSeconds: Output<Int>? = null,
    public val kmsMasterKeyId: Output<String>? = null,
    public val maxMessageSize: Output<Int>? = null,
    public val messageRetentionSeconds: Output<Int>? = null,
    public val name: Output<String>? = null,
    public val namePrefix: Output<String>? = null,
    public val policy: Output<String>? = null,
    public val receiveWaitTimeSeconds: Output<Int>? = null,
    public val redriveAllowPolicy: Output<String>? = null,
    public val redrivePolicy: Output<String>? = null,
    public val sqsManagedSseEnabled: Output<Boolean>? = null,
    public val tags: Output<Map<String, String>>? = null,
    public val visibilityTimeoutSeconds: Output<Int>? = null,
) : ConvertibleToJava<com.pulumi.aws.sqs.QueueArgs> {
    override fun toJava(): com.pulumi.aws.sqs.QueueArgs = com.pulumi.aws.sqs.QueueArgs.builder()
        .contentBasedDeduplication(contentBasedDeduplication?.applyValue({ args0 -> args0 }))
        .deduplicationScope(deduplicationScope?.applyValue({ args0 -> args0 }))
        .delaySeconds(delaySeconds?.applyValue({ args0 -> args0 }))
        .fifoQueue(fifoQueue?.applyValue({ args0 -> args0 }))
        .fifoThroughputLimit(fifoThroughputLimit?.applyValue({ args0 -> args0 }))
        .kmsDataKeyReusePeriodSeconds(kmsDataKeyReusePeriodSeconds?.applyValue({ args0 -> args0 }))
        .kmsMasterKeyId(kmsMasterKeyId?.applyValue({ args0 -> args0 }))
        .maxMessageSize(maxMessageSize?.applyValue({ args0 -> args0 }))
        .messageRetentionSeconds(messageRetentionSeconds?.applyValue({ args0 -> args0 }))
        .name(name?.applyValue({ args0 -> args0 }))
        .namePrefix(namePrefix?.applyValue({ args0 -> args0 }))
        .policy(policy?.applyValue({ args0 -> args0 }))
        .receiveWaitTimeSeconds(receiveWaitTimeSeconds?.applyValue({ args0 -> args0 }))
        .redriveAllowPolicy(redriveAllowPolicy?.applyValue({ args0 -> args0 }))
        .redrivePolicy(redrivePolicy?.applyValue({ args0 -> args0 }))
        .sqsManagedSseEnabled(sqsManagedSseEnabled?.applyValue({ args0 -> args0 }))
        .tags(tags?.applyValue({ args0 -> args0.map({ args0 -> args0.key.to(args0.value) }).toMap() }))
        .visibilityTimeoutSeconds(visibilityTimeoutSeconds?.applyValue({ args0 -> args0 })).build()
}

/**
 * Builder for [QueueArgs].
 */
@PulumiTagMarker
public class QueueArgsBuilder internal constructor() {
    private var contentBasedDeduplication: Output<Boolean>? = null

    private var deduplicationScope: Output<String>? = null

    private var delaySeconds: Output<Int>? = null

    private var fifoQueue: Output<Boolean>? = null

    private var fifoThroughputLimit: Output<String>? = null

    private var kmsDataKeyReusePeriodSeconds: Output<Int>? = null

    private var kmsMasterKeyId: Output<String>? = null

    private var maxMessageSize: Output<Int>? = null

    private var messageRetentionSeconds: Output<Int>? = null

    private var name: Output<String>? = null

    private var namePrefix: Output<String>? = null

    private var policy: Output<String>? = null

    private var receiveWaitTimeSeconds: Output<Int>? = null

    private var redriveAllowPolicy: Output<String>? = null

    private var redrivePolicy: Output<String>? = null

    private var sqsManagedSseEnabled: Output<Boolean>? = null

    private var tags: Output<Map<String, String>>? = null

    private var visibilityTimeoutSeconds: Output<Int>? = null

    /**
     * @param value Enables content-based deduplication for FIFO queues. For more information, see the [related documentation](http://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/FIFO-queues.html#FIFO-queues-exactly-once-processing)
     */
    @JvmName("lhtimtcvyxvqnafl")
    public suspend fun contentBasedDeduplication(`value`: Output<Boolean>) {
        this.contentBasedDeduplication = value
    }

    /**
     * @param value Specifies whether message deduplication occurs at the message group or queue level. Valid values are `messageGroup` and `queue` (default).
     */
    @JvmName("wkxisriiufumwujw")
    public suspend fun deduplicationScope(`value`: Output<String>) {
        this.deduplicationScope = value
    }

    /**
     * @param value The time in seconds that the delivery of all messages in the queue will be delayed. An integer from 0 to 900 (15 minutes). The default for this attribute is 0 seconds.
     */
    @JvmName("gxvnmijqjdonxfrf")
    public suspend fun delaySeconds(`value`: Output<Int>) {
        this.delaySeconds = value
    }

    /**
     * @param value Boolean designating a FIFO queue. If not set, it defaults to `false` making it standard.
     */
    @JvmName("dtujlrpomghxmxbc")
    public suspend fun fifoQueue(`value`: Output<Boolean>) {
        this.fifoQueue = value
    }

    /**
     * @param value Specifies whether the FIFO queue throughput quota applies to the entire queue or per message group. Valid values are `perQueue` (default) and `perMessageGroupId`.
     */
    @JvmName("xaxuuebfmvwybkrm")
    public suspend fun fifoThroughputLimit(`value`: Output<String>) {
        this.fifoThroughputLimit = value
    }

    /**
     * @param value The length of time, in seconds, for which Amazon SQS can reuse a data key to encrypt or decrypt messages before calling AWS KMS again. An integer representing seconds, between 60 seconds (1 minute) and 86,400 seconds (24 hours). The default is 300 (5 minutes).
     */
    @JvmName("noinsxoutdquddbd")
    public suspend fun kmsDataKeyReusePeriodSeconds(`value`: Output<Int>) {
        this.kmsDataKeyReusePeriodSeconds = value
    }

    /**
     * @param value The ID of an AWS-managed customer master key (CMK) for Amazon SQS or a custom CMK. For more information, see [Key Terms](http://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/sqs-server-side-encryption.html#sqs-sse-key-terms).
     */
    @JvmName("rhkafaapolxpqlsa")
    public suspend fun kmsMasterKeyId(`value`: Output<String>) {
        this.kmsMasterKeyId = value
    }

    /**
     * @param value The limit of how many bytes a message can contain before Amazon SQS rejects it. An integer from 1024 bytes (1 KiB) up to 262144 bytes (256 KiB). The default for this attribute is 262144 (256 KiB).
     */
    @JvmName("qviublfmlgucgxpy")
    public suspend fun maxMessageSize(`value`: Output<Int>) {
        this.maxMessageSize = value
    }

    /**
     * @param value The number of seconds Amazon SQS retains a message. Integer representing seconds, from 60 (1 minute) to 1209600 (14 days). The default for this attribute is 345600 (4 days).
     */
    @JvmName("tducdvdwkcijxyix")
    public suspend fun messageRetentionSeconds(`value`: Output<Int>) {
        this.messageRetentionSeconds = value
    }

    /**
     * @param value The name of the queue. Queue names must be made up of only uppercase and lowercase ASCII letters, numbers, underscores, and hyphens, and must be between 1 and 80 characters long. For a FIFO (first-in-first-out) queue, the name must end with the `.fifo` suffix. If omitted, this provider will assign a random, unique name. Conflicts with `name_prefix`
     */
    @JvmName("jcideefosngnpkbq")
    public suspend fun name(`value`: Output<String>) {
        this.name = value
    }

    /**
     * @param value Creates a unique name beginning with the specified prefix. Conflicts with `name`
     */
    @JvmName("owxtqbdcbtlowdyx")
    public suspend fun namePrefix(`value`: Output<String>) {
        this.namePrefix = value
    }

    /**
     * @param value The JSON policy for the SQS queue.
     */
    @JvmName("igjqigmhgbcymdif")
    public suspend fun policy(`value`: Output<String>) {
        this.policy = value
    }

    /**
     * @param value The time for which a ReceiveMessage call will wait for a message to arrive (long polling) before returning. An integer from 0 to 20 (seconds). The default for this attribute is 0, meaning that the call will return immediately.
     */
    @JvmName("oicbckbvnuuupier")
    public suspend fun receiveWaitTimeSeconds(`value`: Output<Int>) {
        this.receiveWaitTimeSeconds = value
    }

    /**
     * @param value The JSON policy to set up the Dead Letter Queue redrive permission, see [AWS docs](https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/SQSDeadLetterQueue.html).
     */
    @JvmName("bnpswcbjfhjssoqm")
    public suspend fun redriveAllowPolicy(`value`: Output<String>) {
        this.redriveAllowPolicy = value
    }

    /**
     * @param value The JSON policy to set up the Dead Letter Queue, see [AWS docs](https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/SQSDeadLetterQueue.html). **Note:** when specifying `maxReceiveCount`, you must specify it as an integer (`5`), and not a string (`"5"`).
     */
    @JvmName("pgycnkyhsxilshrr")
    public suspend fun redrivePolicy(`value`: Output<String>) {
        this.redrivePolicy = value
    }

    /**
     * @param value Boolean to enable server-side encryption (SSE) of message content with SQS-owned encryption keys. See [Encryption at rest](https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/sqs-server-side-encryption.html). The provider will only perform drift detection of its value when present in a configuration.
     */
    @JvmName("aifrmmjeqbjxwqtx")
    public suspend fun sqsManagedSseEnabled(`value`: Output<Boolean>) {
        this.sqsManagedSseEnabled = value
    }

    /**
     * @param value A map of tags to assign to the queue. If configured with a provider `default_tags` configuration block) present, tags with matching keys will overwrite those defined at the provider-level.
     */
    @JvmName("lptvnrfoqmifkkff")
    public suspend fun tags(`value`: Output<Map<String, String>>) {
        this.tags = value
    }

    /**
     * @param value The visibility timeout for the queue. An integer from 0 to 43200 (12 hours). The default for this attribute is 30. For more information about visibility timeout, see [AWS docs](https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/AboutVT.html).
     */
    @JvmName("noopibcjorpkljql")
    public suspend fun visibilityTimeoutSeconds(`value`: Output<Int>) {
        this.visibilityTimeoutSeconds = value
    }

    /**
     * @param value Enables content-based deduplication for FIFO queues. For more information, see the [related documentation](http://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/FIFO-queues.html#FIFO-queues-exactly-once-processing)
     */
    @JvmName("jhtnhtqodbdyhuri")
    public suspend fun contentBasedDeduplication(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.contentBasedDeduplication = mapped
    }

    /**
     * @param value Specifies whether message deduplication occurs at the message group or queue level. Valid values are `messageGroup` and `queue` (default).
     */
    @JvmName("akgxntogrimvmquq")
    public suspend fun deduplicationScope(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.deduplicationScope = mapped
    }

    /**
     * @param value The time in seconds that the delivery of all messages in the queue will be delayed. An integer from 0 to 900 (15 minutes). The default for this attribute is 0 seconds.
     */
    @JvmName("qoebqobsgyjknpfb")
    public suspend fun delaySeconds(`value`: Int?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.delaySeconds = mapped
    }

    /**
     * @param value Boolean designating a FIFO queue. If not set, it defaults to `false` making it standard.
     */
    @JvmName("egupiugqppypdgwy")
    public suspend fun fifoQueue(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.fifoQueue = mapped
    }

    /**
     * @param value Specifies whether the FIFO queue throughput quota applies to the entire queue or per message group. Valid values are `perQueue` (default) and `perMessageGroupId`.
     */
    @JvmName("trosjcyxlweqrdmx")
    public suspend fun fifoThroughputLimit(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.fifoThroughputLimit = mapped
    }

    /**
     * @param value The length of time, in seconds, for which Amazon SQS can reuse a data key to encrypt or decrypt messages before calling AWS KMS again. An integer representing seconds, between 60 seconds (1 minute) and 86,400 seconds (24 hours). The default is 300 (5 minutes).
     */
    @JvmName("newxclhnloltkrwu")
    public suspend fun kmsDataKeyReusePeriodSeconds(`value`: Int?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.kmsDataKeyReusePeriodSeconds = mapped
    }

    /**
     * @param value The ID of an AWS-managed customer master key (CMK) for Amazon SQS or a custom CMK. For more information, see [Key Terms](http://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/sqs-server-side-encryption.html#sqs-sse-key-terms).
     */
    @JvmName("dhjuxytqrmidiouj")
    public suspend fun kmsMasterKeyId(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.kmsMasterKeyId = mapped
    }

    /**
     * @param value The limit of how many bytes a message can contain before Amazon SQS rejects it. An integer from 1024 bytes (1 KiB) up to 262144 bytes (256 KiB). The default for this attribute is 262144 (256 KiB).
     */
    @JvmName("iqkcrgilfdginptl")
    public suspend fun maxMessageSize(`value`: Int?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.maxMessageSize = mapped
    }

    /**
     * @param value The number of seconds Amazon SQS retains a message. Integer representing seconds, from 60 (1 minute) to 1209600 (14 days). The default for this attribute is 345600 (4 days).
     */
    @JvmName("dhxiyeevtrxoodfn")
    public suspend fun messageRetentionSeconds(`value`: Int?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.messageRetentionSeconds = mapped
    }

    /**
     * @param value The name of the queue. Queue names must be made up of only uppercase and lowercase ASCII letters, numbers, underscores, and hyphens, and must be between 1 and 80 characters long. For a FIFO (first-in-first-out) queue, the name must end with the `.fifo` suffix. If omitted, this provider will assign a random, unique name. Conflicts with `name_prefix`
     */
    @JvmName("gcelximnthkcletd")
    public suspend fun name(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.name = mapped
    }

    /**
     * @param value Creates a unique name beginning with the specified prefix. Conflicts with `name`
     */
    @JvmName("tlugqwqvjghylrcs")
    public suspend fun namePrefix(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.namePrefix = mapped
    }

    /**
     * @param value The JSON policy for the SQS queue.
     */
    @JvmName("bwxmcwacowtvxnoc")
    public suspend fun policy(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.policy = mapped
    }

    /**
     * @param value The time for which a ReceiveMessage call will wait for a message to arrive (long polling) before returning. An integer from 0 to 20 (seconds). The default for this attribute is 0, meaning that the call will return immediately.
     */
    @JvmName("carepkyyyrabdcmd")
    public suspend fun receiveWaitTimeSeconds(`value`: Int?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.receiveWaitTimeSeconds = mapped
    }

    /**
     * @param value The JSON policy to set up the Dead Letter Queue redrive permission, see [AWS docs](https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/SQSDeadLetterQueue.html).
     */
    @JvmName("busjfybavnhvefox")
    public suspend fun redriveAllowPolicy(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.redriveAllowPolicy = mapped
    }

    /**
     * @param value The JSON policy to set up the Dead Letter Queue, see [AWS docs](https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/SQSDeadLetterQueue.html). **Note:** when specifying `maxReceiveCount`, you must specify it as an integer (`5`), and not a string (`"5"`).
     */
    @JvmName("xuypovvywlmcpgbf")
    public suspend fun redrivePolicy(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.redrivePolicy = mapped
    }

    /**
     * @param value Boolean to enable server-side encryption (SSE) of message content with SQS-owned encryption keys. See [Encryption at rest](https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/sqs-server-side-encryption.html). The provider will only perform drift detection of its value when present in a configuration.
     */
    @JvmName("pihspxshfpcgipkk")
    public suspend fun sqsManagedSseEnabled(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.sqsManagedSseEnabled = mapped
    }

    /**
     * @param value A map of tags to assign to the queue. If configured with a provider `default_tags` configuration block) present, tags with matching keys will overwrite those defined at the provider-level.
     */
    @JvmName("tkgujodsejxmbyci")
    public suspend fun tags(`value`: Map<String, String>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.tags = mapped
    }

    /**
     * @param values A map of tags to assign to the queue. If configured with a provider `default_tags` configuration block) present, tags with matching keys will overwrite those defined at the provider-level.
     */
    @JvmName("giavutrggcdhxjlr")
    public fun tags(vararg values: Pair<String, String>) {
        val toBeMapped = values.toMap()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.tags = mapped
    }

    /**
     * @param value The visibility timeout for the queue. An integer from 0 to 43200 (12 hours). The default for this attribute is 30. For more information about visibility timeout, see [AWS docs](https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/AboutVT.html).
     */
    @JvmName("lhmcbxrxqefvamvi")
    public suspend fun visibilityTimeoutSeconds(`value`: Int?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.visibilityTimeoutSeconds = mapped
    }

    internal fun build(): QueueArgs = QueueArgs(
        contentBasedDeduplication = contentBasedDeduplication,
        deduplicationScope = deduplicationScope,
        delaySeconds = delaySeconds,
        fifoQueue = fifoQueue,
        fifoThroughputLimit = fifoThroughputLimit,
        kmsDataKeyReusePeriodSeconds = kmsDataKeyReusePeriodSeconds,
        kmsMasterKeyId = kmsMasterKeyId,
        maxMessageSize = maxMessageSize,
        messageRetentionSeconds = messageRetentionSeconds,
        name = name,
        namePrefix = namePrefix,
        policy = policy,
        receiveWaitTimeSeconds = receiveWaitTimeSeconds,
        redriveAllowPolicy = redriveAllowPolicy,
        redrivePolicy = redrivePolicy,
        sqsManagedSseEnabled = sqsManagedSseEnabled,
        tags = tags,
        visibilityTimeoutSeconds = visibilityTimeoutSeconds,
    )
}
