@file:Suppress("NAME_SHADOWING", "DEPRECATION")

package com.pulumi.aws.accessanalyzer.kotlin

import com.pulumi.aws.accessanalyzer.AnalyzerArgs.builder
import com.pulumi.aws.accessanalyzer.kotlin.inputs.AnalyzerConfigurationArgs
import com.pulumi.aws.accessanalyzer.kotlin.inputs.AnalyzerConfigurationArgsBuilder
import com.pulumi.core.Output
import com.pulumi.core.Output.of
import com.pulumi.kotlin.ConvertibleToJava
import com.pulumi.kotlin.PulumiTagMarker
import com.pulumi.kotlin.applySuspend
import kotlin.Pair
import kotlin.String
import kotlin.Suppress
import kotlin.Unit
import kotlin.collections.Map
import kotlin.jvm.JvmName

/**
 * Manages an Access Analyzer Analyzer. More information can be found in the [Access Analyzer User Guide](https://docs.aws.amazon.com/IAM/latest/UserGuide/what-is-access-analyzer.html).
 * ## Example Usage
 * ### Account Analyzer
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const example = new aws.accessanalyzer.Analyzer("example", {analyzerName: "example"});
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * example = aws.accessanalyzer.Analyzer("example", analyzer_name="example")
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var example = new Aws.AccessAnalyzer.Analyzer("example", new()
 *     {
 *         AnalyzerName = "example",
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/accessanalyzer"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		_, err := accessanalyzer.NewAnalyzer(ctx, "example", &accessanalyzer.AnalyzerArgs{
 * 			AnalyzerName: pulumi.String("example"),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.accessanalyzer.Analyzer;
 * import com.pulumi.aws.accessanalyzer.AnalyzerArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var example = new Analyzer("example", AnalyzerArgs.builder()
 *             .analyzerName("example")
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   example:
 *     type: aws:accessanalyzer:Analyzer
 *     properties:
 *       analyzerName: example
 * ```
 * <!--End PulumiCodeChooser -->
 * ### Organization Analyzer
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const example = new aws.organizations.Organization("example", {awsServiceAccessPrincipals: ["access-analyzer&#46;amazonaws&#46;com"]});
 * const exampleAnalyzer = new aws.accessanalyzer.Analyzer("example", {
 *     analyzerName: "example",
 *     type: "ORGANIZATION",
 * }, {
 *     dependsOn: [example],
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * example = aws.organizations.Organization("example", aws_service_access_principals=["access-analyzer&#46;amazonaws&#46;com"])
 * example_analyzer = aws.accessanalyzer.Analyzer("example",
 *     analyzer_name="example",
 *     type="ORGANIZATION",
 *     opts = pulumi.ResourceOptions(depends_on=[example]))
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var example = new Aws.Organizations.Organization("example", new()
 *     {
 *         AwsServiceAccessPrincipals = new[]
 *         {
 *             "access-analyzer.amazonaws.com",
 *         },
 *     });
 *     var exampleAnalyzer = new Aws.AccessAnalyzer.Analyzer("example", new()
 *     {
 *         AnalyzerName = "example",
 *         Type = "ORGANIZATION",
 *     }, new CustomResourceOptions
 *     {
 *         DependsOn =
 *         {
 *             example,
 *         },
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/accessanalyzer"
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/organizations"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		example, err := organizations.NewOrganization(ctx, "example", &organizations.OrganizationArgs{
 * 			AwsServiceAccessPrincipals: pulumi.StringArray{
 * 				pulumi.String("access-analyzer.amazonaws.com"),
 * 			},
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		_, err = accessanalyzer.NewAnalyzer(ctx, "example", &accessanalyzer.AnalyzerArgs{
 * 			AnalyzerName: pulumi.String("example"),
 * 			Type:         pulumi.String("ORGANIZATION"),
 * 		}, pulumi.DependsOn([]pulumi.Resource{
 * 			example,
 * 		}))
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.organizations.Organization;
 * import com.pulumi.aws.organizations.OrganizationArgs;
 * import com.pulumi.aws.accessanalyzer.Analyzer;
 * import com.pulumi.aws.accessanalyzer.AnalyzerArgs;
 * import com.pulumi.resources.CustomResourceOptions;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var example = new Organization("example", OrganizationArgs.builder()
 *             .awsServiceAccessPrincipals("access-analyzer.amazonaws.com")
 *             .build());
 *         var exampleAnalyzer = new Analyzer("exampleAnalyzer", AnalyzerArgs.builder()
 *             .analyzerName("example")
 *             .type("ORGANIZATION")
 *             .build(), CustomResourceOptions.builder()
 *                 .dependsOn(example)
 *                 .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   example:
 *     type: aws:organizations:Organization
 *     properties:
 *       awsServiceAccessPrincipals:
 *         - access-analyzer.amazonaws.com
 *   exampleAnalyzer:
 *     type: aws:accessanalyzer:Analyzer
 *     name: example
 *     properties:
 *       analyzerName: example
 *       type: ORGANIZATION
 *     options:
 *       dependson:
 *         - ${example}
 * ```
 * <!--End PulumiCodeChooser -->
 * ## Import
 * Using `pulumi import`, import Access Analyzer Analyzers using the `analyzer_name`. For example:
 * ```sh
 * $ pulumi import aws:accessanalyzer/analyzer:Analyzer example example
 * ```
 * @property analyzerName Name of the Analyzer.
 * The following arguments are optional:
 * @property configuration A block that specifies the configuration of the analyzer. Documented below
 * @property tags Key-value map of resource tags. If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
 * @property type Type of Analyzer. Valid values are `ACCOUNT`, `ORGANIZATION`, `ACCOUNT_UNUSED_ACCESS `, `ORGANIZATION_UNUSED_ACCESS`. Defaults to `ACCOUNT`.
 */
public data class AnalyzerArgs(
    public val analyzerName: Output<String>? = null,
    public val configuration: Output<AnalyzerConfigurationArgs>? = null,
    public val tags: Output<Map<String, String>>? = null,
    public val type: Output<String>? = null,
) : ConvertibleToJava<com.pulumi.aws.accessanalyzer.AnalyzerArgs> {
    override fun toJava(): com.pulumi.aws.accessanalyzer.AnalyzerArgs =
        com.pulumi.aws.accessanalyzer.AnalyzerArgs.builder()
            .analyzerName(analyzerName?.applyValue({ args0 -> args0 }))
            .configuration(configuration?.applyValue({ args0 -> args0.let({ args0 -> args0.toJava() }) }))
            .tags(tags?.applyValue({ args0 -> args0.map({ args0 -> args0.key.to(args0.value) }).toMap() }))
            .type(type?.applyValue({ args0 -> args0 })).build()
}

/**
 * Builder for [AnalyzerArgs].
 */
@PulumiTagMarker
public class AnalyzerArgsBuilder internal constructor() {
    private var analyzerName: Output<String>? = null

    private var configuration: Output<AnalyzerConfigurationArgs>? = null

    private var tags: Output<Map<String, String>>? = null

    private var type: Output<String>? = null

    /**
     * @param value Name of the Analyzer.
     * The following arguments are optional:
     */
    @JvmName("pkhuqjckksldxxxb")
    public suspend fun analyzerName(`value`: Output<String>) {
        this.analyzerName = value
    }

    /**
     * @param value A block that specifies the configuration of the analyzer. Documented below
     */
    @JvmName("dtewlgsoqghhhpli")
    public suspend fun configuration(`value`: Output<AnalyzerConfigurationArgs>) {
        this.configuration = value
    }

    /**
     * @param value Key-value map of resource tags. If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
     */
    @JvmName("mbeuolflulmikhjb")
    public suspend fun tags(`value`: Output<Map<String, String>>) {
        this.tags = value
    }

    /**
     * @param value Type of Analyzer. Valid values are `ACCOUNT`, `ORGANIZATION`, `ACCOUNT_UNUSED_ACCESS `, `ORGANIZATION_UNUSED_ACCESS`. Defaults to `ACCOUNT`.
     */
    @JvmName("bxewquomldhbyxnx")
    public suspend fun type(`value`: Output<String>) {
        this.type = value
    }

    /**
     * @param value Name of the Analyzer.
     * The following arguments are optional:
     */
    @JvmName("kqlgspaigxnmmvpn")
    public suspend fun analyzerName(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.analyzerName = mapped
    }

    /**
     * @param value A block that specifies the configuration of the analyzer. Documented below
     */
    @JvmName("ydyadaxxkqxmqibn")
    public suspend fun configuration(`value`: AnalyzerConfigurationArgs?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.configuration = mapped
    }

    /**
     * @param argument A block that specifies the configuration of the analyzer. Documented below
     */
    @JvmName("ummwpflmqllwslmq")
    public suspend fun configuration(argument: suspend AnalyzerConfigurationArgsBuilder.() -> Unit) {
        val toBeMapped = AnalyzerConfigurationArgsBuilder().applySuspend { argument() }.build()
        val mapped = of(toBeMapped)
        this.configuration = mapped
    }

    /**
     * @param value Key-value map of resource tags. If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
     */
    @JvmName("kbyqcalbcdikeelo")
    public suspend fun tags(`value`: Map<String, String>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.tags = mapped
    }

    /**
     * @param values Key-value map of resource tags. If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
     */
    @JvmName("cvmqjuiysygmypuf")
    public fun tags(vararg values: Pair<String, String>) {
        val toBeMapped = values.toMap()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.tags = mapped
    }

    /**
     * @param value Type of Analyzer. Valid values are `ACCOUNT`, `ORGANIZATION`, `ACCOUNT_UNUSED_ACCESS `, `ORGANIZATION_UNUSED_ACCESS`. Defaults to `ACCOUNT`.
     */
    @JvmName("gpdbwfrdcwfchowg")
    public suspend fun type(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.type = mapped
    }

    internal fun build(): AnalyzerArgs = AnalyzerArgs(
        analyzerName = analyzerName,
        configuration = configuration,
        tags = tags,
        type = type,
    )
}
