@file:Suppress("NAME_SHADOWING", "DEPRECATION")

package com.pulumi.aws.acm.kotlin

import com.pulumi.aws.acm.AcmFunctions.getCertificatePlain
import com.pulumi.aws.acm.kotlin.inputs.GetCertificatePlainArgs
import com.pulumi.aws.acm.kotlin.inputs.GetCertificatePlainArgsBuilder
import com.pulumi.aws.acm.kotlin.outputs.GetCertificateResult
import com.pulumi.aws.acm.kotlin.outputs.GetCertificateResult.Companion.toKotlin
import kotlinx.coroutines.future.await
import kotlin.Boolean
import kotlin.String
import kotlin.Suppress
import kotlin.Unit
import kotlin.collections.List
import kotlin.collections.Map

public object AcmFunctions {
    /**
     * Use this data source to get the ARN of a certificate in AWS Certificate
     * Manager (ACM), you can reference
     * it by domain without having to hard code the ARNs as input.
     * ## Example Usage
     * <!--Start PulumiCodeChooser -->
     * ```typescript
     * import * as pulumi from "@pulumi/pulumi";
     * import * as aws from "@pulumi/aws";
     * // Find a certificate that is issued
     * const issued = aws.acm.getCertificate({
     *     domain: "tf.example.com",
     *     statuses: ["ISSUED"],
     * });
     * // Find a certificate issued by (not imported into) ACM
     * const amazonIssued = aws.acm.getCertificate({
     *     domain: "tf.example.com",
     *     types: ["AMAZON_ISSUED"],
     *     mostRecent: true,
     * });
     * // Find a RSA 4096 bit certificate
     * const rsa4096 = aws.acm.getCertificate({
     *     domain: "tf.example.com",
     *     keyTypes: ["RSA_4096"],
     * });
     * ```
     * ```python
     * import pulumi
     * import pulumi_aws as aws
     * # Find a certificate that is issued
     * issued = aws.acm.get_certificate(domain="tf.example.com",
     *     statuses=["ISSUED"])
     * # Find a certificate issued by (not imported into) ACM
     * amazon_issued = aws.acm.get_certificate(domain="tf.example.com",
     *     types=["AMAZON_ISSUED"],
     *     most_recent=True)
     * # Find a RSA 4096 bit certificate
     * rsa4096 = aws.acm.get_certificate(domain="tf.example.com",
     *     key_types=["RSA_4096"])
     * ```
     * ```csharp
     * using System.Collections.Generic;
     * using System.Linq;
     * using Pulumi;
     * using Aws = Pulumi.Aws;
     * return await Deployment.RunAsync(() =>
     * {
     *     // Find a certificate that is issued
     *     var issued = Aws.Acm.GetCertificate.Invoke(new()
     *     {
     *         Domain = "tf.example.com",
     *         Statuses = new[]
     *         {
     *             "ISSUED",
     *         },
     *     });
     *     // Find a certificate issued by (not imported into) ACM
     *     var amazonIssued = Aws.Acm.GetCertificate.Invoke(new()
     *     {
     *         Domain = "tf.example.com",
     *         Types = new[]
     *         {
     *             "AMAZON_ISSUED",
     *         },
     *         MostRecent = true,
     *     });
     *     // Find a RSA 4096 bit certificate
     *     var rsa4096 = Aws.Acm.GetCertificate.Invoke(new()
     *     {
     *         Domain = "tf.example.com",
     *         KeyTypes = new[]
     *         {
     *             "RSA_4096",
     *         },
     *     });
     * });
     * ```
     * ```go
     * package main
     * import (
     * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/acm"
     * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
     * )
     * func main() {
     * 	pulumi.Run(func(ctx *pulumi.Context) error {
     * 		// Find a certificate that is issued
     * 		_, err := acm.LookupCertificate(ctx, &acm.LookupCertificateArgs{
     * 			Domain: "tf.example.com",
     * 			Statuses: []string{
     * 				"ISSUED",
     * 			},
     * 		}, nil)
     * 		if err != nil {
     * 			return err
     * 		}
     * 		// Find a certificate issued by (not imported into) ACM
     * 		_, err = acm.LookupCertificate(ctx, &acm.LookupCertificateArgs{
     * 			Domain: "tf.example.com",
     * 			Types: []string{
     * 				"AMAZON_ISSUED",
     * 			},
     * 			MostRecent: pulumi.BoolRef(true),
     * 		}, nil)
     * 		if err != nil {
     * 			return err
     * 		}
     * 		// Find a RSA 4096 bit certificate
     * 		_, err = acm.LookupCertificate(ctx, &acm.LookupCertificateArgs{
     * 			Domain: "tf.example.com",
     * 			KeyTypes: []string{
     * 				"RSA_4096",
     * 			},
     * 		}, nil)
     * 		if err != nil {
     * 			return err
     * 		}
     * 		return nil
     * 	})
     * }
     * ```
     * ```java
     * package generated_program;
     * import com.pulumi.Context;
     * import com.pulumi.Pulumi;
     * import com.pulumi.core.Output;
     * import com.pulumi.aws.acm.AcmFunctions;
     * import com.pulumi.aws.acm.inputs.GetCertificateArgs;
     * import java.util.List;
     * import java.util.ArrayList;
     * import java.util.Map;
     * import java.io.File;
     * import java.nio.file.Files;
     * import java.nio.file.Paths;
     * public class App {
     *     public static void main(String[] args) {
     *         Pulumi.run(App::stack);
     *     }
     *     public static void stack(Context ctx) {
     *         // Find a certificate that is issued
     *         final var issued = AcmFunctions.getCertificate(GetCertificateArgs.builder()
     *             .domain("tf.example.com")
     *             .statuses("ISSUED")
     *             .build());
     *         // Find a certificate issued by (not imported into) ACM
     *         final var amazonIssued = AcmFunctions.getCertificate(GetCertificateArgs.builder()
     *             .domain("tf.example.com")
     *             .types("AMAZON_ISSUED")
     *             .mostRecent(true)
     *             .build());
     *         // Find a RSA 4096 bit certificate
     *         final var rsa4096 = AcmFunctions.getCertificate(GetCertificateArgs.builder()
     *             .domain("tf.example.com")
     *             .keyTypes("RSA_4096")
     *             .build());
     *     }
     * }
     * ```
     * ```yaml
     * variables:
     *   # Find a certificate that is issued
     *   issued:
     *     fn::invoke:
     *       Function: aws:acm:getCertificate
     *       Arguments:
     *         domain: tf.example.com
     *         statuses:
     *           - ISSUED
     *   # Find a certificate issued by (not imported into) ACM
     *   amazonIssued:
     *     fn::invoke:
     *       Function: aws:acm:getCertificate
     *       Arguments:
     *         domain: tf.example.com
     *         types:
     *           - AMAZON_ISSUED
     *         mostRecent: true
     *   # Find a RSA 4096 bit certificate
     *   rsa4096:
     *     fn::invoke:
     *       Function: aws:acm:getCertificate
     *       Arguments:
     *         domain: tf.example.com
     *         keyTypes:
     *           - RSA_4096
     * ```
     * <!--End PulumiCodeChooser -->
     * @param argument A collection of arguments for invoking getCertificate.
     * @return A collection of values returned by getCertificate.
     */
    public suspend fun getCertificate(argument: GetCertificatePlainArgs): GetCertificateResult =
        toKotlin(getCertificatePlain(argument.toJava()).await())

    /**
     * @see [getCertificate].
     * @param domain Domain of the certificate to look up. If no certificate is found with this name, an error will be returned.
     * @param keyTypes List of key algorithms to filter certificates. By default, ACM does not return all certificate types when searching. See the [ACM API Reference](https://docs.aws.amazon.com/acm/latest/APIReference/API_CertificateDetail.html#ACM-Type-CertificateDetail-KeyAlgorithm) for supported key algorithms.
     * @param mostRecent If set to true, it sorts the certificates matched by previous criteria by the NotBefore field, returning only the most recent one. If set to false, it returns an error if more than one certificate is found. Defaults to false.
     * @param statuses List of statuses on which to filter the returned list. Valid values are `PENDING_VALIDATION`, `ISSUED`,
     * `INACTIVE`, `EXPIRED`, `VALIDATION_TIMED_OUT`, `REVOKED` and `FAILED`. If no value is specified, only certificates in the `ISSUED` state
     * are returned.
     * @param tags Mapping of tags for the resource.
     * @param types List of types on which to filter the returned list. Valid values are `AMAZON_ISSUED`, `PRIVATE`, and `IMPORTED`.
     * @return A collection of values returned by getCertificate.
     */
    public suspend fun getCertificate(
        domain: String,
        keyTypes: List<String>? = null,
        mostRecent: Boolean? = null,
        statuses: List<String>? = null,
        tags: Map<String, String>? = null,
        types: List<String>? = null,
    ): GetCertificateResult {
        val argument = GetCertificatePlainArgs(
            domain = domain,
            keyTypes = keyTypes,
            mostRecent = mostRecent,
            statuses = statuses,
            tags = tags,
            types = types,
        )
        return toKotlin(getCertificatePlain(argument.toJava()).await())
    }

    /**
     * @see [getCertificate].
     * @param argument Builder for [com&#46;pulumi&#46;aws&#46;acm&#46;kotlin&#46;inputs&#46;GetCertificatePlainArgs].
     * @return A collection of values returned by getCertificate.
     */
    public suspend fun getCertificate(argument: suspend GetCertificatePlainArgsBuilder.() -> Unit): GetCertificateResult {
        val builder = GetCertificatePlainArgsBuilder()
        builder.argument()
        val builtArgument = builder.build()
        return toKotlin(getCertificatePlain(builtArgument.toJava()).await())
    }
}
