@file:Suppress("NAME_SHADOWING", "DEPRECATION")

package com.pulumi.aws.acmpca.kotlin

import com.pulumi.aws.acmpca.CertificateArgs.builder
import com.pulumi.aws.acmpca.kotlin.inputs.CertificateValidityArgs
import com.pulumi.aws.acmpca.kotlin.inputs.CertificateValidityArgsBuilder
import com.pulumi.core.Output
import com.pulumi.core.Output.of
import com.pulumi.kotlin.ConvertibleToJava
import com.pulumi.kotlin.PulumiTagMarker
import com.pulumi.kotlin.applySuspend
import kotlin.String
import kotlin.Suppress
import kotlin.Unit
import kotlin.jvm.JvmName

/**
 * Provides a resource to issue a certificate using AWS Certificate Manager Private Certificate Authority (ACM PCA).
 * Certificates created using `aws.acmpca.Certificate` are not eligible for automatic renewal,
 * and must be replaced instead.
 * To issue a renewable certificate using an ACM PCA, create a `aws.acm.Certificate`
 * with the parameter `certificate_authority_arn`.
 * ## Example Usage
 * ### Basic
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * import * as tls from "@pulumi/tls";
 * const exampleCertificateAuthority = new aws.acmpca.CertificateAuthority("example", {
 *     certificateAuthorityConfiguration: {
 *         keyAlgorithm: "RSA_4096",
 *         signingAlgorithm: "SHA512WITHRSA",
 *         subject: {
 *             commonName: "example.com",
 *         },
 *     },
 *     permanentDeletionTimeInDays: 7,
 * });
 * const key = new tls.PrivateKey("key", {algorithm: "RSA"});
 * const csr = new tls.CertRequest("csr", {
 *     privateKeyPem: key.privateKeyPem,
 *     subject: {
 *         commonName: "example",
 *     },
 * });
 * const example = new aws.acmpca.Certificate("example", {
 *     certificateAuthorityArn: exampleCertificateAuthority.arn,
 *     certificateSigningRequest: csr.certRequestPem,
 *     signingAlgorithm: "SHA256WITHRSA",
 *     validity: {
 *         type: "YEARS",
 *         value: "1",
 *     },
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * import pulumi_tls as tls
 * example_certificate_authority = aws.acmpca.CertificateAuthority("example",
 *     certificate_authority_configuration={
 *         "key_algorithm": "RSA_4096",
 *         "signing_algorithm": "SHA512WITHRSA",
 *         "subject": {
 *             "common_name": "example.com",
 *         },
 *     },
 *     permanent_deletion_time_in_days=7)
 * key = tls.PrivateKey("key", algorithm="RSA")
 * csr = tls.CertRequest("csr",
 *     private_key_pem=key.private_key_pem,
 *     subject={
 *         "common_name": "example",
 *     })
 * example = aws.acmpca.Certificate("example",
 *     certificate_authority_arn=example_certificate_authority.arn,
 *     certificate_signing_request=csr.cert_request_pem,
 *     signing_algorithm="SHA256WITHRSA",
 *     validity={
 *         "type": "YEARS",
 *         "value": "1",
 *     })
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * using Tls = Pulumi.Tls;
 * return await Deployment.RunAsync(() =>
 * {
 *     var exampleCertificateAuthority = new Aws.Acmpca.CertificateAuthority("example", new()
 *     {
 *         CertificateAuthorityConfiguration = new Aws.Acmpca.Inputs.CertificateAuthorityCertificateAuthorityConfigurationArgs
 *         {
 *             KeyAlgorithm = "RSA_4096",
 *             SigningAlgorithm = "SHA512WITHRSA",
 *             Subject = new Aws.Acmpca.Inputs.CertificateAuthorityCertificateAuthorityConfigurationSubjectArgs
 *             {
 *                 CommonName = "example.com",
 *             },
 *         },
 *         PermanentDeletionTimeInDays = 7,
 *     });
 *     var key = new Tls.PrivateKey("key", new()
 *     {
 *         Algorithm = "RSA",
 *     });
 *     var csr = new Tls.CertRequest("csr", new()
 *     {
 *         PrivateKeyPem = key.PrivateKeyPem,
 *         Subject = new Tls.Inputs.CertRequestSubjectArgs
 *         {
 *             CommonName = "example",
 *         },
 *     });
 *     var example = new Aws.Acmpca.Certificate("example", new()
 *     {
 *         CertificateAuthorityArn = exampleCertificateAuthority.Arn,
 *         CertificateSigningRequest = csr.CertRequestPem,
 *         SigningAlgorithm = "SHA256WITHRSA",
 *         Validity = new Aws.Acmpca.Inputs.CertificateValidityArgs
 *         {
 *             Type = "YEARS",
 *             Value = "1",
 *         },
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/acmpca"
 * 	"github.com/pulumi/pulumi-tls/sdk/v4/go/tls"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		exampleCertificateAuthority, err := acmpca.NewCertificateAuthority(ctx, "example", &acmpca.CertificateAuthorityArgs{
 * 			CertificateAuthorityConfiguration: &acmpca.CertificateAuthorityCertificateAuthorityConfigurationArgs{
 * 				KeyAlgorithm:     pulumi.String("RSA_4096"),
 * 				SigningAlgorithm: pulumi.String("SHA512WITHRSA"),
 * 				Subject: &acmpca.CertificateAuthorityCertificateAuthorityConfigurationSubjectArgs{
 * 					CommonName: pulumi.String("example.com"),
 * 				},
 * 			},
 * 			PermanentDeletionTimeInDays: pulumi.Int(7),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		key, err := tls.NewPrivateKey(ctx, "key", &tls.PrivateKeyArgs{
 * 			Algorithm: pulumi.String("RSA"),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		csr, err := tls.NewCertRequest(ctx, "csr", &tls.CertRequestArgs{
 * 			PrivateKeyPem: key.PrivateKeyPem,
 * 			Subject: &tls.CertRequestSubjectArgs{
 * 				CommonName: pulumi.String("example"),
 * 			},
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		_, err = acmpca.NewCertificate(ctx, "example", &acmpca.CertificateArgs{
 * 			CertificateAuthorityArn:   exampleCertificateAuthority.Arn,
 * 			CertificateSigningRequest: csr.CertRequestPem,
 * 			SigningAlgorithm:          pulumi.String("SHA256WITHRSA"),
 * 			Validity: &acmpca.CertificateValidityArgs{
 * 				Type:  pulumi.String("YEARS"),
 * 				Value: pulumi.String("1"),
 * 			},
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.acmpca.CertificateAuthority;
 * import com.pulumi.aws.acmpca.CertificateAuthorityArgs;
 * import com.pulumi.aws.acmpca.inputs.CertificateAuthorityCertificateAuthorityConfigurationArgs;
 * import com.pulumi.aws.acmpca.inputs.CertificateAuthorityCertificateAuthorityConfigurationSubjectArgs;
 * import com.pulumi.tls.PrivateKey;
 * import com.pulumi.tls.PrivateKeyArgs;
 * import com.pulumi.tls.CertRequest;
 * import com.pulumi.tls.CertRequestArgs;
 * import com.pulumi.tls.inputs.CertRequestSubjectArgs;
 * import com.pulumi.aws.acmpca.Certificate;
 * import com.pulumi.aws.acmpca.CertificateArgs;
 * import com.pulumi.aws.acmpca.inputs.CertificateValidityArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var exampleCertificateAuthority = new CertificateAuthority("exampleCertificateAuthority", CertificateAuthorityArgs.builder()
 *             .certificateAuthorityConfiguration(CertificateAuthorityCertificateAuthorityConfigurationArgs.builder()
 *                 .keyAlgorithm("RSA_4096")
 *                 .signingAlgorithm("SHA512WITHRSA")
 *                 .subject(CertificateAuthorityCertificateAuthorityConfigurationSubjectArgs.builder()
 *                     .commonName("example.com")
 *                     .build())
 *                 .build())
 *             .permanentDeletionTimeInDays(7)
 *             .build());
 *         var key = new PrivateKey("key", PrivateKeyArgs.builder()
 *             .algorithm("RSA")
 *             .build());
 *         var csr = new CertRequest("csr", CertRequestArgs.builder()
 *             .privateKeyPem(key.privateKeyPem())
 *             .subject(CertRequestSubjectArgs.builder()
 *                 .commonName("example")
 *                 .build())
 *             .build());
 *         var example = new Certificate("example", CertificateArgs.builder()
 *             .certificateAuthorityArn(exampleCertificateAuthority.arn())
 *             .certificateSigningRequest(csr.certRequestPem())
 *             .signingAlgorithm("SHA256WITHRSA")
 *             .validity(CertificateValidityArgs.builder()
 *                 .type("YEARS")
 *                 .value(1)
 *                 .build())
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   example:
 *     type: aws:acmpca:Certificate
 *     properties:
 *       certificateAuthorityArn: ${exampleCertificateAuthority.arn}
 *       certificateSigningRequest: ${csr.certRequestPem}
 *       signingAlgorithm: SHA256WITHRSA
 *       validity:
 *         type: YEARS
 *         value: 1
 *   exampleCertificateAuthority:
 *     type: aws:acmpca:CertificateAuthority
 *     name: example
 *     properties:
 *       certificateAuthorityConfiguration:
 *         keyAlgorithm: RSA_4096
 *         signingAlgorithm: SHA512WITHRSA
 *         subject:
 *           commonName: example.com
 *       permanentDeletionTimeInDays: 7
 *   key:
 *     type: tls:PrivateKey
 *     properties:
 *       algorithm: RSA
 *   csr:
 *     type: tls:CertRequest
 *     properties:
 *       privateKeyPem: ${key.privateKeyPem}
 *       subject:
 *         commonName: example
 * ```
 * <!--End PulumiCodeChooser -->
 * ## Import
 * Using `pulumi import`, import ACM PCA Certificates using their ARN. For example:
 * ```sh
 * $ pulumi import aws:acmpca/certificate:Certificate cert arn:aws:acm-pca:eu-west-1:675225743824:certificate-authority/08319ede-83g9-1400-8f21-c7d12b2b6edb/certificate/a4e9c2aa4bcfab625g1b9136464cd3a
 * ```
 * @property apiPassthrough Specifies X.509 certificate information to be included in the issued certificate. To use with API Passthrough templates
 * @property certificateAuthorityArn ARN of the certificate authority.
 * @property certificateSigningRequest Certificate Signing Request in PEM format.
 * @property signingAlgorithm Algorithm to use to sign certificate requests. Valid values: `SHA256WITHRSA`, `SHA256WITHECDSA`, `SHA384WITHRSA`, `SHA384WITHECDSA`, `SHA512WITHRSA`, `SHA512WITHECDSA`.
 * @property templateArn Template to use when issuing a certificate.
 * See [ACM PCA Documentation](https://docs.aws.amazon.com/privateca/latest/userguide/UsingTemplates.html) for more information.
 * @property validity Configures end of the validity period for the certificate. See validity block below.
 */
public data class CertificateArgs(
    public val apiPassthrough: Output<String>? = null,
    public val certificateAuthorityArn: Output<String>? = null,
    public val certificateSigningRequest: Output<String>? = null,
    public val signingAlgorithm: Output<String>? = null,
    public val templateArn: Output<String>? = null,
    public val validity: Output<CertificateValidityArgs>? = null,
) : ConvertibleToJava<com.pulumi.aws.acmpca.CertificateArgs> {
    override fun toJava(): com.pulumi.aws.acmpca.CertificateArgs =
        com.pulumi.aws.acmpca.CertificateArgs.builder()
            .apiPassthrough(apiPassthrough?.applyValue({ args0 -> args0 }))
            .certificateAuthorityArn(certificateAuthorityArn?.applyValue({ args0 -> args0 }))
            .certificateSigningRequest(certificateSigningRequest?.applyValue({ args0 -> args0 }))
            .signingAlgorithm(signingAlgorithm?.applyValue({ args0 -> args0 }))
            .templateArn(templateArn?.applyValue({ args0 -> args0 }))
            .validity(validity?.applyValue({ args0 -> args0.let({ args0 -> args0.toJava() }) })).build()
}

/**
 * Builder for [CertificateArgs].
 */
@PulumiTagMarker
public class CertificateArgsBuilder internal constructor() {
    private var apiPassthrough: Output<String>? = null

    private var certificateAuthorityArn: Output<String>? = null

    private var certificateSigningRequest: Output<String>? = null

    private var signingAlgorithm: Output<String>? = null

    private var templateArn: Output<String>? = null

    private var validity: Output<CertificateValidityArgs>? = null

    /**
     * @param value Specifies X.509 certificate information to be included in the issued certificate. To use with API Passthrough templates
     */
    @JvmName("vjvkyenjiblgdnnd")
    public suspend fun apiPassthrough(`value`: Output<String>) {
        this.apiPassthrough = value
    }

    /**
     * @param value ARN of the certificate authority.
     */
    @JvmName("ufyfbkllnyhxsonm")
    public suspend fun certificateAuthorityArn(`value`: Output<String>) {
        this.certificateAuthorityArn = value
    }

    /**
     * @param value Certificate Signing Request in PEM format.
     */
    @JvmName("kobmtalilbcfudgf")
    public suspend fun certificateSigningRequest(`value`: Output<String>) {
        this.certificateSigningRequest = value
    }

    /**
     * @param value Algorithm to use to sign certificate requests. Valid values: `SHA256WITHRSA`, `SHA256WITHECDSA`, `SHA384WITHRSA`, `SHA384WITHECDSA`, `SHA512WITHRSA`, `SHA512WITHECDSA`.
     */
    @JvmName("fuvjsmeanapnwqqx")
    public suspend fun signingAlgorithm(`value`: Output<String>) {
        this.signingAlgorithm = value
    }

    /**
     * @param value Template to use when issuing a certificate.
     * See [ACM PCA Documentation](https://docs.aws.amazon.com/privateca/latest/userguide/UsingTemplates.html) for more information.
     */
    @JvmName("dinraattfihuoojd")
    public suspend fun templateArn(`value`: Output<String>) {
        this.templateArn = value
    }

    /**
     * @param value Configures end of the validity period for the certificate. See validity block below.
     */
    @JvmName("fjsgvtmitncfutjc")
    public suspend fun validity(`value`: Output<CertificateValidityArgs>) {
        this.validity = value
    }

    /**
     * @param value Specifies X.509 certificate information to be included in the issued certificate. To use with API Passthrough templates
     */
    @JvmName("ciolsetyoopnxifu")
    public suspend fun apiPassthrough(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.apiPassthrough = mapped
    }

    /**
     * @param value ARN of the certificate authority.
     */
    @JvmName("dyqmhbjkgffiifub")
    public suspend fun certificateAuthorityArn(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.certificateAuthorityArn = mapped
    }

    /**
     * @param value Certificate Signing Request in PEM format.
     */
    @JvmName("rjkknirhentnmjcu")
    public suspend fun certificateSigningRequest(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.certificateSigningRequest = mapped
    }

    /**
     * @param value Algorithm to use to sign certificate requests. Valid values: `SHA256WITHRSA`, `SHA256WITHECDSA`, `SHA384WITHRSA`, `SHA384WITHECDSA`, `SHA512WITHRSA`, `SHA512WITHECDSA`.
     */
    @JvmName("vloendjcfederrae")
    public suspend fun signingAlgorithm(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.signingAlgorithm = mapped
    }

    /**
     * @param value Template to use when issuing a certificate.
     * See [ACM PCA Documentation](https://docs.aws.amazon.com/privateca/latest/userguide/UsingTemplates.html) for more information.
     */
    @JvmName("vgotbbbsbxhflxuk")
    public suspend fun templateArn(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.templateArn = mapped
    }

    /**
     * @param value Configures end of the validity period for the certificate. See validity block below.
     */
    @JvmName("xmoxfeijtmhmjgcm")
    public suspend fun validity(`value`: CertificateValidityArgs?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.validity = mapped
    }

    /**
     * @param argument Configures end of the validity period for the certificate. See validity block below.
     */
    @JvmName("mktredwruugrspvg")
    public suspend fun validity(argument: suspend CertificateValidityArgsBuilder.() -> Unit) {
        val toBeMapped = CertificateValidityArgsBuilder().applySuspend { argument() }.build()
        val mapped = of(toBeMapped)
        this.validity = mapped
    }

    internal fun build(): CertificateArgs = CertificateArgs(
        apiPassthrough = apiPassthrough,
        certificateAuthorityArn = certificateAuthorityArn,
        certificateSigningRequest = certificateSigningRequest,
        signingAlgorithm = signingAlgorithm,
        templateArn = templateArn,
        validity = validity,
    )
}
