@file:Suppress("NAME_SHADOWING", "DEPRECATION")

package com.pulumi.aws.amplify.kotlin

import com.pulumi.aws.amplify.BranchArgs.builder
import com.pulumi.core.Output
import com.pulumi.core.Output.of
import com.pulumi.kotlin.ConvertibleToJava
import com.pulumi.kotlin.PulumiTagMarker
import kotlin.Boolean
import kotlin.Pair
import kotlin.String
import kotlin.Suppress
import kotlin.collections.Map
import kotlin.jvm.JvmName

/**
 * Provides an Amplify Branch resource.
 * ## Example Usage
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const example = new aws.amplify.App("example", {name: "app"});
 * const master = new aws.amplify.Branch("master", {
 *     appId: example.id,
 *     branchName: "master",
 *     framework: "React",
 *     stage: "PRODUCTION",
 *     environmentVariables: {
 *         REACT_APP_API_SERVER: "https://api.example.com",
 *     },
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * example = aws.amplify.App("example", name="app")
 * master = aws.amplify.Branch("master",
 *     app_id=example.id,
 *     branch_name="master",
 *     framework="React",
 *     stage="PRODUCTION",
 *     environment_variables={
 *         "REACT_APP_API_SERVER": "https://api.example.com",
 *     })
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var example = new Aws.Amplify.App("example", new()
 *     {
 *         Name = "app",
 *     });
 *     var master = new Aws.Amplify.Branch("master", new()
 *     {
 *         AppId = example.Id,
 *         BranchName = "master",
 *         Framework = "React",
 *         Stage = "PRODUCTION",
 *         EnvironmentVariables =
 *         {
 *             { "REACT_APP_API_SERVER", "https://api.example.com" },
 *         },
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/amplify"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		example, err := amplify.NewApp(ctx, "example", &amplify.AppArgs{
 * 			Name: pulumi.String("app"),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		_, err = amplify.NewBranch(ctx, "master", &amplify.BranchArgs{
 * 			AppId:      example.ID(),
 * 			BranchName: pulumi.String("master"),
 * 			Framework:  pulumi.String("React"),
 * 			Stage:      pulumi.String("PRODUCTION"),
 * 			EnvironmentVariables: pulumi.StringMap{
 * 				"REACT_APP_API_SERVER": pulumi.String("https://api.example.com"),
 * 			},
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.amplify.App;
 * import com.pulumi.aws.amplify.AppArgs;
 * import com.pulumi.aws.amplify.Branch;
 * import com.pulumi.aws.amplify.BranchArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var example = new App("example", AppArgs.builder()
 *             .name("app")
 *             .build());
 *         var master = new Branch("master", BranchArgs.builder()
 *             .appId(example.id())
 *             .branchName("master")
 *             .framework("React")
 *             .stage("PRODUCTION")
 *             .environmentVariables(Map.of("REACT_APP_API_SERVER", "https://api.example.com"))
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   example:
 *     type: aws:amplify:App
 *     properties:
 *       name: app
 *   master:
 *     type: aws:amplify:Branch
 *     properties:
 *       appId: ${example.id}
 *       branchName: master
 *       framework: React
 *       stage: PRODUCTION
 *       environmentVariables:
 *         REACT_APP_API_SERVER: https://api.example.com
 * ```
 * <!--End PulumiCodeChooser -->
 * ### Basic Authentication
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * import * as std from "@pulumi/std";
 * const example = new aws.amplify.App("example", {name: "app"});
 * const master = new aws.amplify.Branch("master", {
 *     appId: example.id,
 *     branchName: "master",
 *     enableBasicAuth: true,
 *     basicAuthCredentials: std.base64encode({
 *         input: "username:password",
 *     }).then(invoke => invoke.result),
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * import pulumi_std as std
 * example = aws.amplify.App("example", name="app")
 * master = aws.amplify.Branch("master",
 *     app_id=example.id,
 *     branch_name="master",
 *     enable_basic_auth=True,
 *     basic_auth_credentials=std.base64encode(input="username:password").result)
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * using Std = Pulumi.Std;
 * return await Deployment.RunAsync(() =>
 * {
 *     var example = new Aws.Amplify.App("example", new()
 *     {
 *         Name = "app",
 *     });
 *     var master = new Aws.Amplify.Branch("master", new()
 *     {
 *         AppId = example.Id,
 *         BranchName = "master",
 *         EnableBasicAuth = true,
 *         BasicAuthCredentials = Std.Base64encode.Invoke(new()
 *         {
 *             Input = "username:password",
 *         }).Apply(invoke => invoke.Result),
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/amplify"
 * 	"github.com/pulumi/pulumi-std/sdk/go/std"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		example, err := amplify.NewApp(ctx, "example", &amplify.AppArgs{
 * 			Name: pulumi.String("app"),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		invokeBase64encode, err := std.Base64encode(ctx, &std.Base64encodeArgs{
 * 			Input: "username:password",
 * 		}, nil)
 * 		if err != nil {
 * 			return err
 * 		}
 * 		_, err = amplify.NewBranch(ctx, "master", &amplify.BranchArgs{
 * 			AppId:                example.ID(),
 * 			BranchName:           pulumi.String("master"),
 * 			EnableBasicAuth:      pulumi.Bool(true),
 * 			BasicAuthCredentials: pulumi.String(invokeBase64encode.Result),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.amplify.App;
 * import com.pulumi.aws.amplify.AppArgs;
 * import com.pulumi.aws.amplify.Branch;
 * import com.pulumi.aws.amplify.BranchArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var example = new App("example", AppArgs.builder()
 *             .name("app")
 *             .build());
 *         var master = new Branch("master", BranchArgs.builder()
 *             .appId(example.id())
 *             .branchName("master")
 *             .enableBasicAuth(true)
 *             .basicAuthCredentials(StdFunctions.base64encode(Base64encodeArgs.builder()
 *                 .input("username:password")
 *                 .build()).result())
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   example:
 *     type: aws:amplify:App
 *     properties:
 *       name: app
 *   master:
 *     type: aws:amplify:Branch
 *     properties:
 *       appId: ${example.id}
 *       branchName: master
 *       enableBasicAuth: true
 *       basicAuthCredentials:
 *         fn::invoke:
 *           Function: std:base64encode
 *           Arguments:
 *             input: username:password
 *           Return: result
 * ```
 * <!--End PulumiCodeChooser -->
 * ### Notifications
 * Amplify Console uses EventBridge (formerly known as CloudWatch Events) and SNS for email notifications.  To implement the same functionality, you need to set `enable_notification` in a `aws.amplify.Branch` resource, as well as creating an EventBridge Rule, an SNS topic, and SNS subscriptions.
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const example = new aws.amplify.App("example", {name: "app"});
 * const master = new aws.amplify.Branch("master", {
 *     appId: example.id,
 *     branchName: "master",
 *     enableNotification: true,
 * });
 * // EventBridge Rule for Amplify notifications
 * const amplifyAppMasterEventRule = new aws.cloudwatch.EventRule("amplify_app_master", {
 *     name: pulumi.interpolate`amplify-${app.id}-${master.branchName}-branch-notification`,
 *     description: pulumi.interpolate`AWS Amplify build notifications for :  App: ${app.id} Branch: ${master.branchName}`,
 *     eventPattern: pulumi.jsonStringify({
 *         detail: {
 *             appId: [example&#46;id],
 *             branchName: [master&#46;branchName],
 *             jobStatus: [
 *                 "SUCCEED",
 *                 "FAILED",
 *                 "STARTED",
 *             ],
 *         },
 *         "detail-type": ["Amplify Deployment Status Change"],
 *         source: ["aws&#46;amplify"],
 *     }),
 * });
 * // SNS Topic for Amplify notifications
 * const amplifyAppMasterTopic = new aws.sns.Topic("amplify_app_master", {name: pulumi.interpolate`amplify-${app.id}_${master.branchName}`});
 * const amplifyAppMasterEventTarget = new aws.cloudwatch.EventTarget("amplify_app_master", {
 *     rule: amplifyAppMasterEventRule.name,
 *     targetId: master.branchName,
 *     arn: amplifyAppMasterTopic.arn,
 *     inputTransformer: {
 *         inputPaths: {
 *             jobId: "$.detail.jobId",
 *             appId: "$.detail.appId",
 *             region: "$.region",
 *             branch: "$.detail.branchName",
 *             status: "$.detail.jobStatus",
 *         },
 *         inputTemplate: "\"Build notification from the AWS Amplify Console for app: https://<branch>.<appId>.amplifyapp.com/. Your build status is <status>. Go to https://console.aws.amazon.com/amplify/home?region=<region>#<appId>/<branch>/<jobId> to view details on your build. \"",
 *     },
 * });
 * const amplifyAppMaster = pulumi.all([master&#46;arn, amplifyAppMasterTopic&#46;arn])&#46;apply(([masterArn, amplifyAppMasterTopicArn]) => aws.iam.getPolicyDocumentOutput({
 *     statements: [{
 *         sid: `Allow_Publish_Events ${masterArn}`,
 *         effect: "Allow",
 *         actions: ["SNS:Publish"],
 *         principals: [{
 *             type: "Service",
 *             identifiers: ["events&#46;amazonaws&#46;com"],
 *         }],
 *         resources: [amplifyAppMasterTopicArn],
 *     }],
 * }));
 * const amplifyAppMasterTopicPolicy = new aws.sns.TopicPolicy("amplify_app_master", {
 *     arn: amplifyAppMasterTopic.arn,
 *     policy: amplifyAppMaster.apply(amplifyAppMaster => amplifyAppMaster.json),
 * });
 * const _this = new aws.sns.TopicSubscription("this", {
 *     topic: amplifyAppMasterTopic.arn,
 *     protocol: "email",
 *     endpoint: "user@acme.com",
 * });
 * ```
 * ```python
 * import pulumi
 * import json
 * import pulumi_aws as aws
 * example = aws.amplify.App("example", name="app")
 * master = aws.amplify.Branch("master",
 *     app_id=example.id,
 *     branch_name="master",
 *     enable_notification=True)
 * # EventBridge Rule for Amplify notifications
 * amplify_app_master_event_rule = aws.cloudwatch.EventRule("amplify_app_master",
 *     name=master.branch_name.apply(lambda branch_name: f"amplify-{app['id']}-{branch_name}-branch-notification"),
 *     description=master.branch_name.apply(lambda branch_name: f"AWS Amplify build notifications for :  App: {app['id']} Branch: {branch_name}"),
 *     event_pattern=pulumi.Output.json_dumps({
 *         "detail": {
 *             "appId": [example&#46;id],
 *             "branchName": [master&#46;branch_name],
 *             "jobStatus": [
 *                 "SUCCEED",
 *                 "FAILED",
 *                 "STARTED",
 *             ],
 *         },
 *         "detail-type": ["Amplify Deployment Status Change"],
 *         "source": ["aws&#46;amplify"],
 *     }))
 * # SNS Topic for Amplify notifications
 * amplify_app_master_topic = aws.sns.Topic("amplify_app_master", name=master.branch_name.apply(lambda branch_name: f"amplify-{app['id']}_{branch_name}"))
 * amplify_app_master_event_target = aws.cloudwatch.EventTarget("amplify_app_master",
 *     rule=amplify_app_master_event_rule.name,
 *     target_id=master.branch_name,
 *     arn=amplify_app_master_topic.arn,
 *     input_transformer={
 *         "input_paths": {
 *             "job_id": "$.detail.jobId",
 *             "app_id": "$.detail.appId",
 *             "region": "$.region",
 *             "branch": "$.detail.branchName",
 *             "status": "$.detail.jobStatus",
 *         },
 *         "input_template": "\"Build notification from the AWS Amplify Console for app: https://<branch>.<appId>.amplifyapp.com/. Your build status is <status>. Go to https://console.aws.amazon.com/amplify/home?region=<region>#<appId>/<branch>/<jobId> to view details on your build. \"",
 *     })
 * amplify_app_master = pulumi.Output.all(master.arn, amplify_app_master_topic.arn).apply(lambda masterArn, amplifyAppMasterTopicArn: aws.iam.get_policy_document_output(statements=[{
 *     "sid": f"Allow_Publish_Events {master_arn}",
 *     "effect": "Allow",
 *     "actions": ["SNS:Publish"],
 *     "principals": [{
 *         "type": "Service",
 *         "identifiers": ["events&#46;amazonaws&#46;com"],
 *     }],
 *     "resources": [amplify_app_master_topic_arn],
 * }]))
 * amplify_app_master_topic_policy = aws.sns.TopicPolicy("amplify_app_master",
 *     arn=amplify_app_master_topic.arn,
 *     policy=amplify_app_master.json)
 * this = aws.sns.TopicSubscription("this",
 *     topic=amplify_app_master_topic.arn,
 *     protocol="email",
 *     endpoint="user@acme.com")
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using System.Text.Json;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var example = new Aws.Amplify.App("example", new()
 *     {
 *         Name = "app",
 *     });
 *     var master = new Aws.Amplify.Branch("master", new()
 *     {
 *         AppId = example.Id,
 *         BranchName = "master",
 *         EnableNotification = true,
 *     });
 *     // EventBridge Rule for Amplify notifications
 *     var amplifyAppMasterEventRule = new Aws.CloudWatch.EventRule("amplify_app_master", new()
 *     {
 *         Name = master.BranchName.Apply(branchName => $"amplify-{app.Id}-{branchName}-branch-notification"),
 *         Description = master.BranchName.Apply(branchName => $"AWS Amplify build notifications for :  App: {app.Id} Branch: {branchName}"),
 *         EventPattern = Output.JsonSerialize(Output.Create(new Dictionary<string, object?>
 *         {
 *             ["detail"] = new Dictionary<string, object?>
 *             {
 *                 ["appId"] = new[]
 *                 {
 *                     example.Id,
 *                 },
 *                 ["branchName"] = new[]
 *                 {
 *                     master.BranchName,
 *                 },
 *                 ["jobStatus"] = new[]
 *                 {
 *                     "SUCCEED",
 *                     "FAILED",
 *                     "STARTED",
 *                 },
 *             },
 *             ["detail-type"] = new[]
 *             {
 *                 "Amplify Deployment Status Change",
 *             },
 *             ["source"] = new[]
 *             {
 *                 "aws.amplify",
 *             },
 *         })),
 *     });
 *     // SNS Topic for Amplify notifications
 *     var amplifyAppMasterTopic = new Aws.Sns.Topic("amplify_app_master", new()
 *     {
 *         Name = master.BranchName.Apply(branchName => $"amplify-{app.Id}_{branchName}"),
 *     });
 *     var amplifyAppMasterEventTarget = new Aws.CloudWatch.EventTarget("amplify_app_master", new()
 *     {
 *         Rule = amplifyAppMasterEventRule.Name,
 *         TargetId = master.BranchName,
 *         Arn = amplifyAppMasterTopic.Arn,
 *         InputTransformer = new Aws.CloudWatch.Inputs.EventTargetInputTransformerArgs
 *         {
 *             InputPaths =
 *             {
 *                 { "jobId", "$.detail.jobId" },
 *                 { "appId", "$.detail.appId" },
 *                 { "region", "$.region" },
 *                 { "branch", "$.detail.branchName" },
 *                 { "status", "$.detail.jobStatus" },
 *             },
 *             InputTemplate = "\"Build notification from the AWS Amplify Console for app: https://<branch>.<appId>.amplifyapp.com/. Your build status is <status>. Go to https://console.aws.amazon.com/amplify/home?region=<region>#<appId>/<branch>/<jobId> to view details on your build. \"",
 *         },
 *     });
 *     var amplifyAppMaster = Aws.Iam.GetPolicyDocument.Invoke(new()
 *     {
 *         Statements = new[]
 *         {
 *             new Aws.Iam.Inputs.GetPolicyDocumentStatementInputArgs
 *             {
 *                 Sid = $"Allow_Publish_Events {master.Arn}",
 *                 Effect = "Allow",
 *                 Actions = new[]
 *                 {
 *                     "SNS:Publish",
 *                 },
 *                 Principals = new[]
 *                 {
 *                     new Aws.Iam.Inputs.GetPolicyDocumentStatementPrincipalInputArgs
 *                     {
 *                         Type = "Service",
 *                         Identifiers = new[]
 *                         {
 *                             "events.amazonaws.com",
 *                         },
 *                     },
 *                 },
 *                 Resources = new[]
 *                 {
 *                     amplifyAppMasterTopic.Arn,
 *                 },
 *             },
 *         },
 *     });
 *     var amplifyAppMasterTopicPolicy = new Aws.Sns.TopicPolicy("amplify_app_master", new()
 *     {
 *         Arn = amplifyAppMasterTopic.Arn,
 *         Policy = amplifyAppMaster.Apply(getPolicyDocumentResult => getPolicyDocumentResult.Json),
 *     });
 *     var @this = new Aws.Sns.TopicSubscription("this", new()
 *     {
 *         Topic = amplifyAppMasterTopic.Arn,
 *         Protocol = "email",
 *         Endpoint = "user@acme.com",
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"encoding/json"
 * 	"fmt"
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/amplify"
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/cloudwatch"
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/iam"
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/sns"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * pulumi.Run(func(ctx *pulumi.Context) error {
 * example, err := amplify.NewApp(ctx, "example", &amplify.AppArgs{
 * Name: pulumi.String("app"),
 * })
 * if err != nil {
 * return err
 * }
 * master, err := amplify.NewBranch(ctx, "master", &amplify.BranchArgs{
 * AppId: example.ID(),
 * BranchName: pulumi.String("master"),
 * EnableNotification: pulumi.Bool(true),
 * })
 * if err != nil {
 * return err
 * }
 * // EventBridge Rule for Amplify notifications
 * amplifyAppMasterEventRule, err := cloudwatch.NewEventRule(ctx, "amplify_app_master", &cloudwatch.EventRuleArgs{
 * Name: master.BranchName.ApplyT(func(branchName string) (string, error) {
 * return fmt.Sprintf("amplify-%v-%v-branch-notification", app.Id, branchName), nil
 * }).(pulumi.StringOutput),
 * Description: master.BranchName.ApplyT(func(branchName string) (string, error) {
 * return fmt.Sprintf("AWS Amplify build notifications for :  App: %v Branch: %v", app.Id, branchName), nil
 * }).(pulumi.StringOutput),
 * EventPattern: pulumi.All(example.ID(),master.BranchName).ApplyT(func(_args []interface{}) (string, error) {
 * id := _args[0].(string)
 * branchName := _args[1].(string)
 * var _zero string
 * tmpJSON0, err := json.Marshal(map[string]interface{}{
 * "detail": map[string]interface{}{
 * "appId": []string{
 * id,
 * },
 * "branchName": []string{
 * branchName,
 * },
 * "jobStatus": []string{
 * "SUCCEED",
 * "FAILED",
 * "STARTED",
 * },
 * },
 * "detail-type": []string{
 * "Amplify Deployment Status Change",
 * },
 * "source": []string{
 * "aws.amplify",
 * },
 * })
 * if err != nil {
 * return _zero, err
 * }
 * json0 := string(tmpJSON0)
 * return json0, nil
 * }).(pulumi.StringOutput),
 * })
 * if err != nil {
 * return err
 * }
 * // SNS Topic for Amplify notifications
 * amplifyAppMasterTopic, err := sns.NewTopic(ctx, "amplify_app_master", &sns.TopicArgs{
 * Name: master.BranchName.ApplyT(func(branchName string) (string, error) {
 * return fmt.Sprintf("amplify-%v_%v", app.Id, branchName), nil
 * }).(pulumi.StringOutput),
 * })
 * if err != nil {
 * return err
 * }
 * _, err = cloudwatch.NewEventTarget(ctx, "amplify_app_master", &cloudwatch.EventTargetArgs{
 * Rule: amplifyAppMasterEventRule.Name,
 * TargetId: master.BranchName,
 * Arn: amplifyAppMasterTopic.Arn,
 * InputTransformer: &cloudwatch.EventTargetInputTransformerArgs{
 * InputPaths: pulumi.StringMap{
 * "jobId": pulumi.String("$.detail.jobId"),
 * "appId": pulumi.String("$.detail.appId"),
 * "region": pulumi.String("$.region"),
 * "branch": pulumi.String("$.detail.branchName"),
 * "status": pulumi.String("$.detail.jobStatus"),
 * },
 * InputTemplate: pulumi.String("\"Build notification from the AWS Amplify Console for app: https://<branch>.<appId>.amplifyapp.com/. Your build status is <status>. Go to https://console.aws.amazon.com/amplify/home?region=<region>#<appId>/<branch>/<jobId> to view details on your build. \""),
 * },
 * })
 * if err != nil {
 * return err
 * }
 * amplifyAppMaster := pulumi.All(master.Arn,amplifyAppMasterTopic.Arn).ApplyT(func(_args []interface{}) (iam.GetPolicyDocumentResult, error) {
 * masterArn := _args[0].(string)
 * amplifyAppMasterTopicArn := _args[1].(string)
 * return iam.GetPolicyDocumentResult(interface{}(iam.GetPolicyDocumentOutput(ctx, iam.GetPolicyDocumentOutputArgs{
 * Statements: []iam.GetPolicyDocumentStatement{
 * {
 * Sid: fmt.Sprintf("Allow_Publish_Events %v", masterArn),
 * Effect: "Allow",
 * Actions: []string{
 * "SNS:Publish",
 * },
 * Principals: []iam.GetPolicyDocumentStatementPrincipal{
 * {
 * Type: "Service",
 * Identifiers: []string{
 * "events.amazonaws.com",
 * },
 * },
 * },
 * Resources: interface{}{
 * amplifyAppMasterTopicArn,
 * },
 * },
 * },
 * }, nil))), nil
 * }).(iam.GetPolicyDocumentResultOutput)
 * _, err = sns.NewTopicPolicy(ctx, "amplify_app_master", &sns.TopicPolicyArgs{
 * Arn: amplifyAppMasterTopic.Arn,
 * Policy: pulumi.String(amplifyAppMaster.ApplyT(func(amplifyAppMaster iam.GetPolicyDocumentResult) (*string, error) {
 * return &amplifyAppMaster.Json, nil
 * }).(pulumi.StringPtrOutput)),
 * })
 * if err != nil {
 * return err
 * }
 * _, err = sns.NewTopicSubscription(ctx, "this", &sns.TopicSubscriptionArgs{
 * Topic: amplifyAppMasterTopic.Arn,
 * Protocol: pulumi.String("email"),
 * Endpoint: pulumi.String("user@acme.com"),
 * })
 * if err != nil {
 * return err
 * }
 * return nil
 * })
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.amplify.App;
 * import com.pulumi.aws.amplify.AppArgs;
 * import com.pulumi.aws.amplify.Branch;
 * import com.pulumi.aws.amplify.BranchArgs;
 * import com.pulumi.aws.cloudwatch.EventRule;
 * import com.pulumi.aws.cloudwatch.EventRuleArgs;
 * import com.pulumi.aws.sns.Topic;
 * import com.pulumi.aws.sns.TopicArgs;
 * import com.pulumi.aws.cloudwatch.EventTarget;
 * import com.pulumi.aws.cloudwatch.EventTargetArgs;
 * import com.pulumi.aws.cloudwatch.inputs.EventTargetInputTransformerArgs;
 * import com.pulumi.aws.iam.IamFunctions;
 * import com.pulumi.aws.iam.inputs.GetPolicyDocumentArgs;
 * import com.pulumi.aws.sns.TopicPolicy;
 * import com.pulumi.aws.sns.TopicPolicyArgs;
 * import com.pulumi.aws.sns.TopicSubscription;
 * import com.pulumi.aws.sns.TopicSubscriptionArgs;
 * import static com.pulumi.codegen.internal.Serialization.*;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var example = new App("example", AppArgs.builder()
 *             .name("app")
 *             .build());
 *         var master = new Branch("master", BranchArgs.builder()
 *             .appId(example.id())
 *             .branchName("master")
 *             .enableNotification(true)
 *             .build());
 *         // EventBridge Rule for Amplify notifications
 *         var amplifyAppMasterEventRule = new EventRule("amplifyAppMasterEventRule", EventRuleArgs.builder()
 *             .name(master.branchName().applyValue(branchName -> String.format("amplify-%s-%s-branch-notification", app.id(),branchName)))
 *             .description(master.branchName().applyValue(branchName -> String.format("AWS Amplify build notifications for :  App: %s Branch: %s", app.id(),branchName)))
 *             .eventPattern(Output.tuple(example.id(), master.branchName()).applyValue(values -> {
 *                 var id = values.t1;
 *                 var branchName = values.t2;
 *                 return serializeJson(
 *                     jsonObject(
 *                         jsonProperty("detail", jsonObject(
 *                             jsonProperty("appId", jsonArray(id)),
 *                             jsonProperty("branchName", jsonArray(branchName)),
 *                             jsonProperty("jobStatus", jsonArray(
 *                                 "SUCCEED",
 *                                 "FAILED",
 *                                 "STARTED"
 *                             ))
 *                         )),
 *                         jsonProperty("detail-type", jsonArray("Amplify Deployment Status Change")),
 *                         jsonProperty("source", jsonArray("aws.amplify"))
 *                     ));
 *             }))
 *             .build());
 *         // SNS Topic for Amplify notifications
 *         var amplifyAppMasterTopic = new Topic("amplifyAppMasterTopic", TopicArgs.builder()
 *             .name(master.branchName().applyValue(branchName -> String.format("amplify-%s_%s", app.id(),branchName)))
 *             .build());
 *         var amplifyAppMasterEventTarget = new EventTarget("amplifyAppMasterEventTarget", EventTargetArgs.builder()
 *             .rule(amplifyAppMasterEventRule.name())
 *             .targetId(master.branchName())
 *             .arn(amplifyAppMasterTopic.arn())
 *             .inputTransformer(EventTargetInputTransformerArgs.builder()
 *                 .inputPaths(Map.ofEntries(
 *                     Map.entry("jobId", "$.detail.jobId"),
 *                     Map.entry("appId", "$.detail.appId"),
 *                     Map.entry("region", "$.region"),
 *                     Map.entry("branch", "$.detail.branchName"),
 *                     Map.entry("status", "$.detail.jobStatus")
 *                 ))
 *                 .inputTemplate("\"Build notification from the AWS Amplify Console for app: https://<branch>.<appId>.amplifyapp.com/. Your build status is <status>. Go to https://console.aws.amazon.com/amplify/home?region=<region>#<appId>/<branch>/<jobId> to view details on your build. \"")
 *                 .build())
 *             .build());
 *         final var amplifyAppMaster = IamFunctions.getPolicyDocument(GetPolicyDocumentArgs.builder()
 *             .statements(GetPolicyDocumentStatementArgs.builder()
 *                 .sid(master.arn().applyValue(arn -> String.format("Allow_Publish_Events %s", arn)))
 *                 .effect("Allow")
 *                 .actions("SNS:Publish")
 *                 .principals(GetPolicyDocumentStatementPrincipalArgs.builder()
 *                     .type("Service")
 *                     .identifiers("events.amazonaws.com")
 *                     .build())
 *                 .resources(amplifyAppMasterTopic.arn())
 *                 .build())
 *             .build());
 *         var amplifyAppMasterTopicPolicy = new TopicPolicy("amplifyAppMasterTopicPolicy", TopicPolicyArgs.builder()
 *             .arn(amplifyAppMasterTopic.arn())
 *             .policy(amplifyAppMaster.applyValue(getPolicyDocumentResult -> getPolicyDocumentResult).applyValue(amplifyAppMaster -> amplifyAppMaster.applyValue(getPolicyDocumentResult -> getPolicyDocumentResult.json())))
 *             .build());
 *         var this_ = new TopicSubscription("this", TopicSubscriptionArgs.builder()
 *             .topic(amplifyAppMasterTopic.arn())
 *             .protocol("email")
 *             .endpoint("user@acme.com")
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   example:
 *     type: aws:amplify:App
 *     properties:
 *       name: app
 *   master:
 *     type: aws:amplify:Branch
 *     properties:
 *       appId: ${example.id}
 *       branchName: master
 *       enableNotification: true
 *   # EventBridge Rule for Amplify notifications
 *   amplifyAppMasterEventRule:
 *     type: aws:cloudwatch:EventRule
 *     name: amplify_app_master
 *     properties:
 *       name: amplify-${app.id}-${master.branchName}-branch-notification
 *       description: 'AWS Amplify build notifications for :  App: ${app.id} Branch: ${master.branchName}'
 *       eventPattern:
 *         fn::toJSON:
 *           detail:
 *             appId:
 *               - ${example.id}
 *             branchName:
 *               - ${master.branchName}
 *             jobStatus:
 *               - SUCCEED
 *               - FAILED
 *               - STARTED
 *           detail-type:
 *             - Amplify Deployment Status Change
 *           source:
 *             - aws.amplify
 *   amplifyAppMasterEventTarget:
 *     type: aws:cloudwatch:EventTarget
 *     name: amplify_app_master
 *     properties:
 *       rule: ${amplifyAppMasterEventRule.name}
 *       targetId: ${master.branchName}
 *       arn: ${amplifyAppMasterTopic.arn}
 *       inputTransformer:
 *         inputPaths:
 *           jobId: $.detail.jobId
 *           appId: $.detail.appId
 *           region: $.region
 *           branch: $.detail.branchName
 *           status: $.detail.jobStatus
 *         inputTemplate: '"Build notification from the AWS Amplify Console for app: https://<branch>.<appId>.amplifyapp.com/. Your build status is <status>. Go to https://console.aws.amazon.com/amplify/home?region=<region>#<appId>/<branch>/<jobId> to view details on your build. "'
 *   # SNS Topic for Amplify notifications
 *   amplifyAppMasterTopic:
 *     type: aws:sns:Topic
 *     name: amplify_app_master
 *     properties:
 *       name: amplify-${app.id}_${master.branchName}
 *   amplifyAppMasterTopicPolicy:
 *     type: aws:sns:TopicPolicy
 *     name: amplify_app_master
 *     properties:
 *       arn: ${amplifyAppMasterTopic.arn}
 *       policy: ${amplifyAppMaster.json}
 *   this:
 *     type: aws:sns:TopicSubscription
 *     properties:
 *       topic: ${amplifyAppMasterTopic.arn}
 *       protocol: email
 *       endpoint: user@acme.com
 * variables:
 *   amplifyAppMaster:
 *     fn::invoke:
 *       Function: aws:iam:getPolicyDocument
 *       Arguments:
 *         statements:
 *           - sid: Allow_Publish_Events ${master.arn}
 *             effect: Allow
 *             actions:
 *               - SNS:Publish
 *             principals:
 *               - type: Service
 *                 identifiers:
 *                   - events.amazonaws.com
 *             resources:
 *               - ${amplifyAppMasterTopic.arn}
 * ```
 * <!--End PulumiCodeChooser -->
 * ## Import
 * Using `pulumi import`, import Amplify branch using `app_id` and `branch_name`. For example:
 * ```sh
 * $ pulumi import aws:amplify/branch:Branch master d2ypk4k47z8u6/master
 * ```
 * @property appId Unique ID for an Amplify app.
 * @property backendEnvironmentArn ARN for a backend environment that is part of an Amplify app.
 * @property basicAuthCredentials Basic authorization credentials for the branch.
 * @property branchName Name for the branch.
 * @property description Description for the branch.
 * @property displayName Display name for a branch. This is used as the default domain prefix.
 * @property enableAutoBuild Enables auto building for the branch.
 * @property enableBasicAuth Enables basic authorization for the branch.
 * @property enableNotification Enables notifications for the branch.
 * @property enablePerformanceMode Enables performance mode for the branch.
 * @property enablePullRequestPreview Enables pull request previews for this branch.
 * @property environmentVariables Environment variables for the branch.
 * @property framework Framework for the branch.
 * @property pullRequestEnvironmentName Amplify environment name for the pull request.
 * @property stage Describes the current stage for the branch. Valid values: `PRODUCTION`, `BETA`, `DEVELOPMENT`, `EXPERIMENTAL`, `PULL_REQUEST`.
 * @property tags Key-value mapping of resource tags. If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
 * @property ttl Content Time To Live (TTL) for the website in seconds.
 */
public data class BranchArgs(
    public val appId: Output<String>? = null,
    public val backendEnvironmentArn: Output<String>? = null,
    public val basicAuthCredentials: Output<String>? = null,
    public val branchName: Output<String>? = null,
    public val description: Output<String>? = null,
    public val displayName: Output<String>? = null,
    public val enableAutoBuild: Output<Boolean>? = null,
    public val enableBasicAuth: Output<Boolean>? = null,
    public val enableNotification: Output<Boolean>? = null,
    public val enablePerformanceMode: Output<Boolean>? = null,
    public val enablePullRequestPreview: Output<Boolean>? = null,
    public val environmentVariables: Output<Map<String, String>>? = null,
    public val framework: Output<String>? = null,
    public val pullRequestEnvironmentName: Output<String>? = null,
    public val stage: Output<String>? = null,
    public val tags: Output<Map<String, String>>? = null,
    public val ttl: Output<String>? = null,
) : ConvertibleToJava<com.pulumi.aws.amplify.BranchArgs> {
    override fun toJava(): com.pulumi.aws.amplify.BranchArgs =
        com.pulumi.aws.amplify.BranchArgs.builder()
            .appId(appId?.applyValue({ args0 -> args0 }))
            .backendEnvironmentArn(backendEnvironmentArn?.applyValue({ args0 -> args0 }))
            .basicAuthCredentials(basicAuthCredentials?.applyValue({ args0 -> args0 }))
            .branchName(branchName?.applyValue({ args0 -> args0 }))
            .description(description?.applyValue({ args0 -> args0 }))
            .displayName(displayName?.applyValue({ args0 -> args0 }))
            .enableAutoBuild(enableAutoBuild?.applyValue({ args0 -> args0 }))
            .enableBasicAuth(enableBasicAuth?.applyValue({ args0 -> args0 }))
            .enableNotification(enableNotification?.applyValue({ args0 -> args0 }))
            .enablePerformanceMode(enablePerformanceMode?.applyValue({ args0 -> args0 }))
            .enablePullRequestPreview(enablePullRequestPreview?.applyValue({ args0 -> args0 }))
            .environmentVariables(
                environmentVariables?.applyValue({ args0 ->
                    args0.map({ args0 ->
                        args0.key.to(args0.value)
                    }).toMap()
                }),
            )
            .framework(framework?.applyValue({ args0 -> args0 }))
            .pullRequestEnvironmentName(pullRequestEnvironmentName?.applyValue({ args0 -> args0 }))
            .stage(stage?.applyValue({ args0 -> args0 }))
            .tags(tags?.applyValue({ args0 -> args0.map({ args0 -> args0.key.to(args0.value) }).toMap() }))
            .ttl(ttl?.applyValue({ args0 -> args0 })).build()
}

/**
 * Builder for [BranchArgs].
 */
@PulumiTagMarker
public class BranchArgsBuilder internal constructor() {
    private var appId: Output<String>? = null

    private var backendEnvironmentArn: Output<String>? = null

    private var basicAuthCredentials: Output<String>? = null

    private var branchName: Output<String>? = null

    private var description: Output<String>? = null

    private var displayName: Output<String>? = null

    private var enableAutoBuild: Output<Boolean>? = null

    private var enableBasicAuth: Output<Boolean>? = null

    private var enableNotification: Output<Boolean>? = null

    private var enablePerformanceMode: Output<Boolean>? = null

    private var enablePullRequestPreview: Output<Boolean>? = null

    private var environmentVariables: Output<Map<String, String>>? = null

    private var framework: Output<String>? = null

    private var pullRequestEnvironmentName: Output<String>? = null

    private var stage: Output<String>? = null

    private var tags: Output<Map<String, String>>? = null

    private var ttl: Output<String>? = null

    /**
     * @param value Unique ID for an Amplify app.
     */
    @JvmName("njcuwnailebnddut")
    public suspend fun appId(`value`: Output<String>) {
        this.appId = value
    }

    /**
     * @param value ARN for a backend environment that is part of an Amplify app.
     */
    @JvmName("drdrradtrdtvstng")
    public suspend fun backendEnvironmentArn(`value`: Output<String>) {
        this.backendEnvironmentArn = value
    }

    /**
     * @param value Basic authorization credentials for the branch.
     */
    @JvmName("csbfrweejhubnnqx")
    public suspend fun basicAuthCredentials(`value`: Output<String>) {
        this.basicAuthCredentials = value
    }

    /**
     * @param value Name for the branch.
     */
    @JvmName("vxladwbawxvvsaqs")
    public suspend fun branchName(`value`: Output<String>) {
        this.branchName = value
    }

    /**
     * @param value Description for the branch.
     */
    @JvmName("tklcujhhuqnugxoh")
    public suspend fun description(`value`: Output<String>) {
        this.description = value
    }

    /**
     * @param value Display name for a branch. This is used as the default domain prefix.
     */
    @JvmName("owpmtlvbvhqyjjlc")
    public suspend fun displayName(`value`: Output<String>) {
        this.displayName = value
    }

    /**
     * @param value Enables auto building for the branch.
     */
    @JvmName("hhmoufhseyolosog")
    public suspend fun enableAutoBuild(`value`: Output<Boolean>) {
        this.enableAutoBuild = value
    }

    /**
     * @param value Enables basic authorization for the branch.
     */
    @JvmName("jsfqjdqdinhgvbav")
    public suspend fun enableBasicAuth(`value`: Output<Boolean>) {
        this.enableBasicAuth = value
    }

    /**
     * @param value Enables notifications for the branch.
     */
    @JvmName("kqxkkwbreorfoyig")
    public suspend fun enableNotification(`value`: Output<Boolean>) {
        this.enableNotification = value
    }

    /**
     * @param value Enables performance mode for the branch.
     */
    @JvmName("tcidlvtjjliqmypv")
    public suspend fun enablePerformanceMode(`value`: Output<Boolean>) {
        this.enablePerformanceMode = value
    }

    /**
     * @param value Enables pull request previews for this branch.
     */
    @JvmName("onihjriitpxnisuh")
    public suspend fun enablePullRequestPreview(`value`: Output<Boolean>) {
        this.enablePullRequestPreview = value
    }

    /**
     * @param value Environment variables for the branch.
     */
    @JvmName("rpmrgjpbglfekalk")
    public suspend fun environmentVariables(`value`: Output<Map<String, String>>) {
        this.environmentVariables = value
    }

    /**
     * @param value Framework for the branch.
     */
    @JvmName("bygvbmcowtojmbxh")
    public suspend fun framework(`value`: Output<String>) {
        this.framework = value
    }

    /**
     * @param value Amplify environment name for the pull request.
     */
    @JvmName("pctxljqqphyecxic")
    public suspend fun pullRequestEnvironmentName(`value`: Output<String>) {
        this.pullRequestEnvironmentName = value
    }

    /**
     * @param value Describes the current stage for the branch. Valid values: `PRODUCTION`, `BETA`, `DEVELOPMENT`, `EXPERIMENTAL`, `PULL_REQUEST`.
     */
    @JvmName("rlbepmhcuvjiwisl")
    public suspend fun stage(`value`: Output<String>) {
        this.stage = value
    }

    /**
     * @param value Key-value mapping of resource tags. If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
     */
    @JvmName("cjheuldgeutkjafv")
    public suspend fun tags(`value`: Output<Map<String, String>>) {
        this.tags = value
    }

    /**
     * @param value Content Time To Live (TTL) for the website in seconds.
     */
    @JvmName("sqnprhehosrhmkfx")
    public suspend fun ttl(`value`: Output<String>) {
        this.ttl = value
    }

    /**
     * @param value Unique ID for an Amplify app.
     */
    @JvmName("aqiekcdyrkancvur")
    public suspend fun appId(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.appId = mapped
    }

    /**
     * @param value ARN for a backend environment that is part of an Amplify app.
     */
    @JvmName("katorwyjtaggvyro")
    public suspend fun backendEnvironmentArn(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.backendEnvironmentArn = mapped
    }

    /**
     * @param value Basic authorization credentials for the branch.
     */
    @JvmName("orvklscehgwjyplw")
    public suspend fun basicAuthCredentials(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.basicAuthCredentials = mapped
    }

    /**
     * @param value Name for the branch.
     */
    @JvmName("xrvkxkgnkuectbxf")
    public suspend fun branchName(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.branchName = mapped
    }

    /**
     * @param value Description for the branch.
     */
    @JvmName("rwjpouavwiypgmwg")
    public suspend fun description(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.description = mapped
    }

    /**
     * @param value Display name for a branch. This is used as the default domain prefix.
     */
    @JvmName("odcwmvxfbovpttxe")
    public suspend fun displayName(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.displayName = mapped
    }

    /**
     * @param value Enables auto building for the branch.
     */
    @JvmName("wfchutfjownrbpxb")
    public suspend fun enableAutoBuild(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.enableAutoBuild = mapped
    }

    /**
     * @param value Enables basic authorization for the branch.
     */
    @JvmName("esgnwmyjecfwqjdg")
    public suspend fun enableBasicAuth(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.enableBasicAuth = mapped
    }

    /**
     * @param value Enables notifications for the branch.
     */
    @JvmName("lyfjuwjgcogstkjt")
    public suspend fun enableNotification(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.enableNotification = mapped
    }

    /**
     * @param value Enables performance mode for the branch.
     */
    @JvmName("nppnkcpwplcwgqle")
    public suspend fun enablePerformanceMode(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.enablePerformanceMode = mapped
    }

    /**
     * @param value Enables pull request previews for this branch.
     */
    @JvmName("tgygxgvgfhxddlxg")
    public suspend fun enablePullRequestPreview(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.enablePullRequestPreview = mapped
    }

    /**
     * @param value Environment variables for the branch.
     */
    @JvmName("iaodqspvnjbmdjpv")
    public suspend fun environmentVariables(`value`: Map<String, String>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.environmentVariables = mapped
    }

    /**
     * @param values Environment variables for the branch.
     */
    @JvmName("dfsuilcnyfmgkcop")
    public fun environmentVariables(vararg values: Pair<String, String>) {
        val toBeMapped = values.toMap()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.environmentVariables = mapped
    }

    /**
     * @param value Framework for the branch.
     */
    @JvmName("frkmpagpueipnqkq")
    public suspend fun framework(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.framework = mapped
    }

    /**
     * @param value Amplify environment name for the pull request.
     */
    @JvmName("aeteglaxefunhmbc")
    public suspend fun pullRequestEnvironmentName(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.pullRequestEnvironmentName = mapped
    }

    /**
     * @param value Describes the current stage for the branch. Valid values: `PRODUCTION`, `BETA`, `DEVELOPMENT`, `EXPERIMENTAL`, `PULL_REQUEST`.
     */
    @JvmName("vybxlanrfcrabjkp")
    public suspend fun stage(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.stage = mapped
    }

    /**
     * @param value Key-value mapping of resource tags. If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
     */
    @JvmName("eqlukdjevtkelsdc")
    public suspend fun tags(`value`: Map<String, String>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.tags = mapped
    }

    /**
     * @param values Key-value mapping of resource tags. If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
     */
    @JvmName("bgqjqaldqdfeokhu")
    public fun tags(vararg values: Pair<String, String>) {
        val toBeMapped = values.toMap()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.tags = mapped
    }

    /**
     * @param value Content Time To Live (TTL) for the website in seconds.
     */
    @JvmName("nrvmdutmtilyvrep")
    public suspend fun ttl(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.ttl = mapped
    }

    internal fun build(): BranchArgs = BranchArgs(
        appId = appId,
        backendEnvironmentArn = backendEnvironmentArn,
        basicAuthCredentials = basicAuthCredentials,
        branchName = branchName,
        description = description,
        displayName = displayName,
        enableAutoBuild = enableAutoBuild,
        enableBasicAuth = enableBasicAuth,
        enableNotification = enableNotification,
        enablePerformanceMode = enablePerformanceMode,
        enablePullRequestPreview = enablePullRequestPreview,
        environmentVariables = environmentVariables,
        framework = framework,
        pullRequestEnvironmentName = pullRequestEnvironmentName,
        stage = stage,
        tags = tags,
        ttl = ttl,
    )
}
