@file:Suppress("NAME_SHADOWING", "DEPRECATION")

package com.pulumi.aws.backup.kotlin

import com.pulumi.aws.backup.SelectionArgs.builder
import com.pulumi.aws.backup.kotlin.inputs.SelectionConditionArgs
import com.pulumi.aws.backup.kotlin.inputs.SelectionConditionArgsBuilder
import com.pulumi.aws.backup.kotlin.inputs.SelectionSelectionTagArgs
import com.pulumi.aws.backup.kotlin.inputs.SelectionSelectionTagArgsBuilder
import com.pulumi.core.Output
import com.pulumi.core.Output.of
import com.pulumi.kotlin.ConvertibleToJava
import com.pulumi.kotlin.PulumiTagMarker
import com.pulumi.kotlin.applySuspend
import kotlin.String
import kotlin.Suppress
import kotlin.Unit
import kotlin.collections.List
import kotlin.jvm.JvmName

/**
 * Manages selection conditions for AWS Backup plan resources.
 * ## Example Usage
 * ### IAM Role
 * > For more information about creating and managing IAM Roles for backups and restores, see the [AWS Backup Developer Guide](https://docs.aws.amazon.com/aws-backup/latest/devguide/iam-service-roles.html).
 * The below example creates an IAM role with the default managed IAM Policy for allowing AWS Backup to create backups.
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const assumeRole = aws.iam.getPolicyDocument({
 *     statements: [{
 *         effect: "Allow",
 *         principals: [{
 *             type: "Service",
 *             identifiers: ["backup&#46;amazonaws&#46;com"],
 *         }],
 *         actions: ["sts:AssumeRole"],
 *     }],
 * });
 * const example = new aws.iam.Role("example", {
 *     name: "example",
 *     assumeRolePolicy: assumeRole.then(assumeRole => assumeRole.json),
 * });
 * const exampleRolePolicyAttachment = new aws.iam.RolePolicyAttachment("example", {
 *     policyArn: "arn:aws:iam::aws:policy/service-role/AWSBackupServiceRolePolicyForBackup",
 *     role: example.name,
 * });
 * const exampleSelection = new aws.backup.Selection("example", {iamRoleArn: example.arn});
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * assume_role = aws.iam.get_policy_document(statements=[{
 *     "effect": "Allow",
 *     "principals": [{
 *         "type": "Service",
 *         "identifiers": ["backup&#46;amazonaws&#46;com"],
 *     }],
 *     "actions": ["sts:AssumeRole"],
 * }])
 * example = aws.iam.Role("example",
 *     name="example",
 *     assume_role_policy=assume_role.json)
 * example_role_policy_attachment = aws.iam.RolePolicyAttachment("example",
 *     policy_arn="arn:aws:iam::aws:policy/service-role/AWSBackupServiceRolePolicyForBackup",
 *     role=example.name)
 * example_selection = aws.backup.Selection("example", iam_role_arn=example.arn)
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var assumeRole = Aws.Iam.GetPolicyDocument.Invoke(new()
 *     {
 *         Statements = new[]
 *         {
 *             new Aws.Iam.Inputs.GetPolicyDocumentStatementInputArgs
 *             {
 *                 Effect = "Allow",
 *                 Principals = new[]
 *                 {
 *                     new Aws.Iam.Inputs.GetPolicyDocumentStatementPrincipalInputArgs
 *                     {
 *                         Type = "Service",
 *                         Identifiers = new[]
 *                         {
 *                             "backup.amazonaws.com",
 *                         },
 *                     },
 *                 },
 *                 Actions = new[]
 *                 {
 *                     "sts:AssumeRole",
 *                 },
 *             },
 *         },
 *     });
 *     var example = new Aws.Iam.Role("example", new()
 *     {
 *         Name = "example",
 *         AssumeRolePolicy = assumeRole.Apply(getPolicyDocumentResult => getPolicyDocumentResult.Json),
 *     });
 *     var exampleRolePolicyAttachment = new Aws.Iam.RolePolicyAttachment("example", new()
 *     {
 *         PolicyArn = "arn:aws:iam::aws:policy/service-role/AWSBackupServiceRolePolicyForBackup",
 *         Role = example.Name,
 *     });
 *     var exampleSelection = new Aws.Backup.Selection("example", new()
 *     {
 *         IamRoleArn = example.Arn,
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/backup"
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/iam"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		assumeRole, err := iam.GetPolicyDocument(ctx, &iam.GetPolicyDocumentArgs{
 * 			Statements: []iam.GetPolicyDocumentStatement{
 * 				{
 * 					Effect: pulumi.StringRef("Allow"),
 * 					Principals: []iam.GetPolicyDocumentStatementPrincipal{
 * 						{
 * 							Type: "Service",
 * 							Identifiers: []string{
 * 								"backup.amazonaws.com",
 * 							},
 * 						},
 * 					},
 * 					Actions: []string{
 * 						"sts:AssumeRole",
 * 					},
 * 				},
 * 			},
 * 		}, nil)
 * 		if err != nil {
 * 			return err
 * 		}
 * 		example, err := iam.NewRole(ctx, "example", &iam.RoleArgs{
 * 			Name:             pulumi.String("example"),
 * 			AssumeRolePolicy: pulumi.String(assumeRole.Json),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		_, err = iam.NewRolePolicyAttachment(ctx, "example", &iam.RolePolicyAttachmentArgs{
 * 			PolicyArn: pulumi.String("arn:aws:iam::aws:policy/service-role/AWSBackupServiceRolePolicyForBackup"),
 * 			Role:      example.Name,
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		_, err = backup.NewSelection(ctx, "example", &backup.SelectionArgs{
 * 			IamRoleArn: example.Arn,
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.iam.IamFunctions;
 * import com.pulumi.aws.iam.inputs.GetPolicyDocumentArgs;
 * import com.pulumi.aws.iam.Role;
 * import com.pulumi.aws.iam.RoleArgs;
 * import com.pulumi.aws.iam.RolePolicyAttachment;
 * import com.pulumi.aws.iam.RolePolicyAttachmentArgs;
 * import com.pulumi.aws.backup.Selection;
 * import com.pulumi.aws.backup.SelectionArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         final var assumeRole = IamFunctions.getPolicyDocument(GetPolicyDocumentArgs.builder()
 *             .statements(GetPolicyDocumentStatementArgs.builder()
 *                 .effect("Allow")
 *                 .principals(GetPolicyDocumentStatementPrincipalArgs.builder()
 *                     .type("Service")
 *                     .identifiers("backup.amazonaws.com")
 *                     .build())
 *                 .actions("sts:AssumeRole")
 *                 .build())
 *             .build());
 *         var example = new Role("example", RoleArgs.builder()
 *             .name("example")
 *             .assumeRolePolicy(assumeRole.applyValue(getPolicyDocumentResult -> getPolicyDocumentResult.json()))
 *             .build());
 *         var exampleRolePolicyAttachment = new RolePolicyAttachment("exampleRolePolicyAttachment", RolePolicyAttachmentArgs.builder()
 *             .policyArn("arn:aws:iam::aws:policy/service-role/AWSBackupServiceRolePolicyForBackup")
 *             .role(example.name())
 *             .build());
 *         var exampleSelection = new Selection("exampleSelection", SelectionArgs.builder()
 *             .iamRoleArn(example.arn())
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   example:
 *     type: aws:iam:Role
 *     properties:
 *       name: example
 *       assumeRolePolicy: ${assumeRole.json}
 *   exampleRolePolicyAttachment:
 *     type: aws:iam:RolePolicyAttachment
 *     name: example
 *     properties:
 *       policyArn: arn:aws:iam::aws:policy/service-role/AWSBackupServiceRolePolicyForBackup
 *       role: ${example.name}
 *   exampleSelection:
 *     type: aws:backup:Selection
 *     name: example
 *     properties:
 *       iamRoleArn: ${example.arn}
 * variables:
 *   assumeRole:
 *     fn::invoke:
 *       Function: aws:iam:getPolicyDocument
 *       Arguments:
 *         statements:
 *           - effect: Allow
 *             principals:
 *               - type: Service
 *                 identifiers:
 *                   - backup.amazonaws.com
 *             actions:
 *               - sts:AssumeRole
 * ```
 * <!--End PulumiCodeChooser -->
 * ### Selecting Backups By Tag
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const example = new aws.backup.Selection("example", {
 *     iamRoleArn: exampleAwsIamRole.arn,
 *     name: "my_example_backup_selection",
 *     planId: exampleAwsBackupPlan.id,
 *     selectionTags: [{
 *         type: "STRINGEQUALS",
 *         key: "foo",
 *         value: "bar",
 *     }],
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * example = aws.backup.Selection("example",
 *     iam_role_arn=example_aws_iam_role["arn"],
 *     name="my_example_backup_selection",
 *     plan_id=example_aws_backup_plan["id"],
 *     selection_tags=[{
 *         "type": "STRINGEQUALS",
 *         "key": "foo",
 *         "value": "bar",
 *     }])
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var example = new Aws.Backup.Selection("example", new()
 *     {
 *         IamRoleArn = exampleAwsIamRole.Arn,
 *         Name = "my_example_backup_selection",
 *         PlanId = exampleAwsBackupPlan.Id,
 *         SelectionTags = new[]
 *         {
 *             new Aws.Backup.Inputs.SelectionSelectionTagArgs
 *             {
 *                 Type = "STRINGEQUALS",
 *                 Key = "foo",
 *                 Value = "bar",
 *             },
 *         },
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/backup"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		_, err := backup.NewSelection(ctx, "example", &backup.SelectionArgs{
 * 			IamRoleArn: pulumi.Any(exampleAwsIamRole.Arn),
 * 			Name:       pulumi.String("my_example_backup_selection"),
 * 			PlanId:     pulumi.Any(exampleAwsBackupPlan.Id),
 * 			SelectionTags: backup.SelectionSelectionTagArray{
 * 				&backup.SelectionSelectionTagArgs{
 * 					Type:  pulumi.String("STRINGEQUALS"),
 * 					Key:   pulumi.String("foo"),
 * 					Value: pulumi.String("bar"),
 * 				},
 * 			},
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.backup.Selection;
 * import com.pulumi.aws.backup.SelectionArgs;
 * import com.pulumi.aws.backup.inputs.SelectionSelectionTagArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var example = new Selection("example", SelectionArgs.builder()
 *             .iamRoleArn(exampleAwsIamRole.arn())
 *             .name("my_example_backup_selection")
 *             .planId(exampleAwsBackupPlan.id())
 *             .selectionTags(SelectionSelectionTagArgs.builder()
 *                 .type("STRINGEQUALS")
 *                 .key("foo")
 *                 .value("bar")
 *                 .build())
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   example:
 *     type: aws:backup:Selection
 *     properties:
 *       iamRoleArn: ${exampleAwsIamRole.arn}
 *       name: my_example_backup_selection
 *       planId: ${exampleAwsBackupPlan.id}
 *       selectionTags:
 *         - type: STRINGEQUALS
 *           key: foo
 *           value: bar
 * ```
 * <!--End PulumiCodeChooser -->
 * ### Selecting Backups By Conditions
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const example = new aws.backup.Selection("example", {
 *     iamRoleArn: exampleAwsIamRole.arn,
 *     name: "my_example_backup_selection",
 *     planId: exampleAwsBackupPlan.id,
 *     resources: ["*"],
 *     conditions: [{
 *         stringEquals: [{
 *             key: "aws:ResourceTag/Component",
 *             value: "rds",
 *         }],
 *         stringLikes: [{
 *             key: "aws:ResourceTag/Application",
 *             value: "app*",
 *         }],
 *         stringNotEquals: [{
 *             key: "aws:ResourceTag/Backup",
 *             value: "false",
 *         }],
 *         stringNotLikes: [{
 *             key: "aws:ResourceTag/Environment",
 *             value: "test*",
 *         }],
 *     }],
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * example = aws.backup.Selection("example",
 *     iam_role_arn=example_aws_iam_role["arn"],
 *     name="my_example_backup_selection",
 *     plan_id=example_aws_backup_plan["id"],
 *     resources=["*"],
 *     conditions=[{
 *         "string_equals": [{
 *             "key": "aws:ResourceTag/Component",
 *             "value": "rds",
 *         }],
 *         "string_likes": [{
 *             "key": "aws:ResourceTag/Application",
 *             "value": "app*",
 *         }],
 *         "string_not_equals": [{
 *             "key": "aws:ResourceTag/Backup",
 *             "value": "false",
 *         }],
 *         "string_not_likes": [{
 *             "key": "aws:ResourceTag/Environment",
 *             "value": "test*",
 *         }],
 *     }])
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var example = new Aws.Backup.Selection("example", new()
 *     {
 *         IamRoleArn = exampleAwsIamRole.Arn,
 *         Name = "my_example_backup_selection",
 *         PlanId = exampleAwsBackupPlan.Id,
 *         Resources = new[]
 *         {
 *             "*",
 *         },
 *         Conditions = new[]
 *         {
 *             new Aws.Backup.Inputs.SelectionConditionArgs
 *             {
 *                 StringEquals = new[]
 *                 {
 *                     new Aws.Backup.Inputs.SelectionConditionStringEqualArgs
 *                     {
 *                         Key = "aws:ResourceTag/Component",
 *                         Value = "rds",
 *                     },
 *                 },
 *                 StringLikes = new[]
 *                 {
 *                     new Aws.Backup.Inputs.SelectionConditionStringLikeArgs
 *                     {
 *                         Key = "aws:ResourceTag/Application",
 *                         Value = "app*",
 *                     },
 *                 },
 *                 StringNotEquals = new[]
 *                 {
 *                     new Aws.Backup.Inputs.SelectionConditionStringNotEqualArgs
 *                     {
 *                         Key = "aws:ResourceTag/Backup",
 *                         Value = "false",
 *                     },
 *                 },
 *                 StringNotLikes = new[]
 *                 {
 *                     new Aws.Backup.Inputs.SelectionConditionStringNotLikeArgs
 *                     {
 *                         Key = "aws:ResourceTag/Environment",
 *                         Value = "test*",
 *                     },
 *                 },
 *             },
 *         },
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/backup"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		_, err := backup.NewSelection(ctx, "example", &backup.SelectionArgs{
 * 			IamRoleArn: pulumi.Any(exampleAwsIamRole.Arn),
 * 			Name:       pulumi.String("my_example_backup_selection"),
 * 			PlanId:     pulumi.Any(exampleAwsBackupPlan.Id),
 * 			Resources: pulumi.StringArray{
 * 				pulumi.String("*"),
 * 			},
 * 			Conditions: backup.SelectionConditionArray{
 * 				&backup.SelectionConditionArgs{
 * 					StringEquals: backup.SelectionConditionStringEqualArray{
 * 						&backup.SelectionConditionStringEqualArgs{
 * 							Key:   pulumi.String("aws:ResourceTag/Component"),
 * 							Value: pulumi.String("rds"),
 * 						},
 * 					},
 * 					StringLikes: backup.SelectionConditionStringLikeArray{
 * 						&backup.SelectionConditionStringLikeArgs{
 * 							Key:   pulumi.String("aws:ResourceTag/Application"),
 * 							Value: pulumi.String("app*"),
 * 						},
 * 					},
 * 					StringNotEquals: backup.SelectionConditionStringNotEqualArray{
 * 						&backup.SelectionConditionStringNotEqualArgs{
 * 							Key:   pulumi.String("aws:ResourceTag/Backup"),
 * 							Value: pulumi.String("false"),
 * 						},
 * 					},
 * 					StringNotLikes: backup.SelectionConditionStringNotLikeArray{
 * 						&backup.SelectionConditionStringNotLikeArgs{
 * 							Key:   pulumi.String("aws:ResourceTag/Environment"),
 * 							Value: pulumi.String("test*"),
 * 						},
 * 					},
 * 				},
 * 			},
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.backup.Selection;
 * import com.pulumi.aws.backup.SelectionArgs;
 * import com.pulumi.aws.backup.inputs.SelectionConditionArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var example = new Selection("example", SelectionArgs.builder()
 *             .iamRoleArn(exampleAwsIamRole.arn())
 *             .name("my_example_backup_selection")
 *             .planId(exampleAwsBackupPlan.id())
 *             .resources("*")
 *             .conditions(SelectionConditionArgs.builder()
 *                 .stringEquals(SelectionConditionStringEqualArgs.builder()
 *                     .key("aws:ResourceTag/Component")
 *                     .value("rds")
 *                     .build())
 *                 .stringLikes(SelectionConditionStringLikeArgs.builder()
 *                     .key("aws:ResourceTag/Application")
 *                     .value("app*")
 *                     .build())
 *                 .stringNotEquals(SelectionConditionStringNotEqualArgs.builder()
 *                     .key("aws:ResourceTag/Backup")
 *                     .value("false")
 *                     .build())
 *                 .stringNotLikes(SelectionConditionStringNotLikeArgs.builder()
 *                     .key("aws:ResourceTag/Environment")
 *                     .value("test*")
 *                     .build())
 *                 .build())
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   example:
 *     type: aws:backup:Selection
 *     properties:
 *       iamRoleArn: ${exampleAwsIamRole.arn}
 *       name: my_example_backup_selection
 *       planId: ${exampleAwsBackupPlan.id}
 *       resources:
 *         - '*'
 *       conditions:
 *         - stringEquals:
 *             - key: aws:ResourceTag/Component
 *               value: rds
 *           stringLikes:
 *             - key: aws:ResourceTag/Application
 *               value: app*
 *           stringNotEquals:
 *             - key: aws:ResourceTag/Backup
 *               value: 'false'
 *           stringNotLikes:
 *             - key: aws:ResourceTag/Environment
 *               value: test*
 * ```
 * <!--End PulumiCodeChooser -->
 * ### Selecting Backups By Resource
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const example = new aws.backup.Selection("example", {
 *     iamRoleArn: exampleAwsIamRole.arn,
 *     name: "my_example_backup_selection",
 *     planId: exampleAwsBackupPlan.id,
 *     resources: [
 *         exampleAwsDbInstance.arn,
 *         exampleAwsEbsVolume.arn,
 *         exampleAwsEfsFileSystem.arn,
 *     ],
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * example = aws.backup.Selection("example",
 *     iam_role_arn=example_aws_iam_role["arn"],
 *     name="my_example_backup_selection",
 *     plan_id=example_aws_backup_plan["id"],
 *     resources=[
 *         example_aws_db_instance["arn"],
 *         example_aws_ebs_volume["arn"],
 *         example_aws_efs_file_system["arn"],
 *     ])
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var example = new Aws.Backup.Selection("example", new()
 *     {
 *         IamRoleArn = exampleAwsIamRole.Arn,
 *         Name = "my_example_backup_selection",
 *         PlanId = exampleAwsBackupPlan.Id,
 *         Resources = new[]
 *         {
 *             exampleAwsDbInstance.Arn,
 *             exampleAwsEbsVolume.Arn,
 *             exampleAwsEfsFileSystem.Arn,
 *         },
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/backup"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		_, err := backup.NewSelection(ctx, "example", &backup.SelectionArgs{
 * 			IamRoleArn: pulumi.Any(exampleAwsIamRole.Arn),
 * 			Name:       pulumi.String("my_example_backup_selection"),
 * 			PlanId:     pulumi.Any(exampleAwsBackupPlan.Id),
 * 			Resources: pulumi.StringArray{
 * 				exampleAwsDbInstance.Arn,
 * 				exampleAwsEbsVolume.Arn,
 * 				exampleAwsEfsFileSystem.Arn,
 * 			},
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.backup.Selection;
 * import com.pulumi.aws.backup.SelectionArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var example = new Selection("example", SelectionArgs.builder()
 *             .iamRoleArn(exampleAwsIamRole.arn())
 *             .name("my_example_backup_selection")
 *             .planId(exampleAwsBackupPlan.id())
 *             .resources(
 *                 exampleAwsDbInstance.arn(),
 *                 exampleAwsEbsVolume.arn(),
 *                 exampleAwsEfsFileSystem.arn())
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   example:
 *     type: aws:backup:Selection
 *     properties:
 *       iamRoleArn: ${exampleAwsIamRole.arn}
 *       name: my_example_backup_selection
 *       planId: ${exampleAwsBackupPlan.id}
 *       resources:
 *         - ${exampleAwsDbInstance.arn}
 *         - ${exampleAwsEbsVolume.arn}
 *         - ${exampleAwsEfsFileSystem.arn}
 * ```
 * <!--End PulumiCodeChooser -->
 * ### Selecting Backups By Not Resource
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const example = new aws.backup.Selection("example", {
 *     iamRoleArn: exampleAwsIamRole.arn,
 *     name: "my_example_backup_selection",
 *     planId: exampleAwsBackupPlan.id,
 *     notResources: [
 *         exampleAwsDbInstance.arn,
 *         exampleAwsEbsVolume.arn,
 *         exampleAwsEfsFileSystem.arn,
 *     ],
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * example = aws.backup.Selection("example",
 *     iam_role_arn=example_aws_iam_role["arn"],
 *     name="my_example_backup_selection",
 *     plan_id=example_aws_backup_plan["id"],
 *     not_resources=[
 *         example_aws_db_instance["arn"],
 *         example_aws_ebs_volume["arn"],
 *         example_aws_efs_file_system["arn"],
 *     ])
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var example = new Aws.Backup.Selection("example", new()
 *     {
 *         IamRoleArn = exampleAwsIamRole.Arn,
 *         Name = "my_example_backup_selection",
 *         PlanId = exampleAwsBackupPlan.Id,
 *         NotResources = new[]
 *         {
 *             exampleAwsDbInstance.Arn,
 *             exampleAwsEbsVolume.Arn,
 *             exampleAwsEfsFileSystem.Arn,
 *         },
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/backup"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		_, err := backup.NewSelection(ctx, "example", &backup.SelectionArgs{
 * 			IamRoleArn: pulumi.Any(exampleAwsIamRole.Arn),
 * 			Name:       pulumi.String("my_example_backup_selection"),
 * 			PlanId:     pulumi.Any(exampleAwsBackupPlan.Id),
 * 			NotResources: pulumi.StringArray{
 * 				exampleAwsDbInstance.Arn,
 * 				exampleAwsEbsVolume.Arn,
 * 				exampleAwsEfsFileSystem.Arn,
 * 			},
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.backup.Selection;
 * import com.pulumi.aws.backup.SelectionArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var example = new Selection("example", SelectionArgs.builder()
 *             .iamRoleArn(exampleAwsIamRole.arn())
 *             .name("my_example_backup_selection")
 *             .planId(exampleAwsBackupPlan.id())
 *             .notResources(
 *                 exampleAwsDbInstance.arn(),
 *                 exampleAwsEbsVolume.arn(),
 *                 exampleAwsEfsFileSystem.arn())
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   example:
 *     type: aws:backup:Selection
 *     properties:
 *       iamRoleArn: ${exampleAwsIamRole.arn}
 *       name: my_example_backup_selection
 *       planId: ${exampleAwsBackupPlan.id}
 *       notResources:
 *         - ${exampleAwsDbInstance.arn}
 *         - ${exampleAwsEbsVolume.arn}
 *         - ${exampleAwsEfsFileSystem.arn}
 * ```
 * <!--End PulumiCodeChooser -->
 * ## Import
 * Using `pulumi import`, import Backup selection using the role plan_id and id separated by `|`. For example:
 * ```sh
 * $ pulumi import aws:backup/selection:Selection example plan-id|selection-id
 * ```
 * @property conditions A list of conditions that you define to assign resources to your backup plans using tags.
 * @property iamRoleArn The ARN of the IAM role that AWS Backup uses to authenticate when restoring and backing up the target resource. See the [AWS Backup Developer Guide](https://docs.aws.amazon.com/aws-backup/latest/devguide/access-control.html#managed-policies) for additional information about using AWS managed policies or creating custom policies attached to the IAM role.
 * @property name The display name of a resource selection document.
 * @property notResources An array of strings that either contain Amazon Resource Names (ARNs) or match patterns of resources to exclude from a backup plan.
 * @property planId The backup plan ID to be associated with the selection of resources.
 * @property resources An array of strings that either contain Amazon Resource Names (ARNs) or match patterns of resources to assign to a backup plan.
 * @property selectionTags Tag-based conditions used to specify a set of resources to assign to a backup plan.
 */
public data class SelectionArgs(
    public val conditions: Output<List<SelectionConditionArgs>>? = null,
    public val iamRoleArn: Output<String>? = null,
    public val name: Output<String>? = null,
    public val notResources: Output<List<String>>? = null,
    public val planId: Output<String>? = null,
    public val resources: Output<List<String>>? = null,
    public val selectionTags: Output<List<SelectionSelectionTagArgs>>? = null,
) : ConvertibleToJava<com.pulumi.aws.backup.SelectionArgs> {
    override fun toJava(): com.pulumi.aws.backup.SelectionArgs =
        com.pulumi.aws.backup.SelectionArgs.builder()
            .conditions(
                conditions?.applyValue({ args0 ->
                    args0.map({ args0 ->
                        args0.let({ args0 ->
                            args0.toJava()
                        })
                    })
                }),
            )
            .iamRoleArn(iamRoleArn?.applyValue({ args0 -> args0 }))
            .name(name?.applyValue({ args0 -> args0 }))
            .notResources(notResources?.applyValue({ args0 -> args0.map({ args0 -> args0 }) }))
            .planId(planId?.applyValue({ args0 -> args0 }))
            .resources(resources?.applyValue({ args0 -> args0.map({ args0 -> args0 }) }))
            .selectionTags(
                selectionTags?.applyValue({ args0 ->
                    args0.map({ args0 ->
                        args0.let({ args0 ->
                            args0.toJava()
                        })
                    })
                }),
            ).build()
}

/**
 * Builder for [SelectionArgs].
 */
@PulumiTagMarker
public class SelectionArgsBuilder internal constructor() {
    private var conditions: Output<List<SelectionConditionArgs>>? = null

    private var iamRoleArn: Output<String>? = null

    private var name: Output<String>? = null

    private var notResources: Output<List<String>>? = null

    private var planId: Output<String>? = null

    private var resources: Output<List<String>>? = null

    private var selectionTags: Output<List<SelectionSelectionTagArgs>>? = null

    /**
     * @param value A list of conditions that you define to assign resources to your backup plans using tags.
     */
    @JvmName("hkqredkkrskikhlt")
    public suspend fun conditions(`value`: Output<List<SelectionConditionArgs>>) {
        this.conditions = value
    }

    @JvmName("cqtnjqowdthioodk")
    public suspend fun conditions(vararg values: Output<SelectionConditionArgs>) {
        this.conditions = Output.all(values.asList())
    }

    /**
     * @param values A list of conditions that you define to assign resources to your backup plans using tags.
     */
    @JvmName("uekmeshptluyvdsa")
    public suspend fun conditions(values: List<Output<SelectionConditionArgs>>) {
        this.conditions = Output.all(values)
    }

    /**
     * @param value The ARN of the IAM role that AWS Backup uses to authenticate when restoring and backing up the target resource. See the [AWS Backup Developer Guide](https://docs.aws.amazon.com/aws-backup/latest/devguide/access-control.html#managed-policies) for additional information about using AWS managed policies or creating custom policies attached to the IAM role.
     */
    @JvmName("lbfbcggimgpwvdvo")
    public suspend fun iamRoleArn(`value`: Output<String>) {
        this.iamRoleArn = value
    }

    /**
     * @param value The display name of a resource selection document.
     */
    @JvmName("limyqjmrohhsklnv")
    public suspend fun name(`value`: Output<String>) {
        this.name = value
    }

    /**
     * @param value An array of strings that either contain Amazon Resource Names (ARNs) or match patterns of resources to exclude from a backup plan.
     */
    @JvmName("guyrphttkhmvvvkn")
    public suspend fun notResources(`value`: Output<List<String>>) {
        this.notResources = value
    }

    @JvmName("mwunmcxaoqcmklhi")
    public suspend fun notResources(vararg values: Output<String>) {
        this.notResources = Output.all(values.asList())
    }

    /**
     * @param values An array of strings that either contain Amazon Resource Names (ARNs) or match patterns of resources to exclude from a backup plan.
     */
    @JvmName("oexcdqobunaqysjb")
    public suspend fun notResources(values: List<Output<String>>) {
        this.notResources = Output.all(values)
    }

    /**
     * @param value The backup plan ID to be associated with the selection of resources.
     */
    @JvmName("xdtedxflhstnetrn")
    public suspend fun planId(`value`: Output<String>) {
        this.planId = value
    }

    /**
     * @param value An array of strings that either contain Amazon Resource Names (ARNs) or match patterns of resources to assign to a backup plan.
     */
    @JvmName("pxywwicfguqmnsdi")
    public suspend fun resources(`value`: Output<List<String>>) {
        this.resources = value
    }

    @JvmName("devvuvaqgthvubld")
    public suspend fun resources(vararg values: Output<String>) {
        this.resources = Output.all(values.asList())
    }

    /**
     * @param values An array of strings that either contain Amazon Resource Names (ARNs) or match patterns of resources to assign to a backup plan.
     */
    @JvmName("gacgbqgnoghknseg")
    public suspend fun resources(values: List<Output<String>>) {
        this.resources = Output.all(values)
    }

    /**
     * @param value Tag-based conditions used to specify a set of resources to assign to a backup plan.
     */
    @JvmName("xfinihrwixplxsrn")
    public suspend fun selectionTags(`value`: Output<List<SelectionSelectionTagArgs>>) {
        this.selectionTags = value
    }

    @JvmName("eqjfhbcsglvlmqqu")
    public suspend fun selectionTags(vararg values: Output<SelectionSelectionTagArgs>) {
        this.selectionTags = Output.all(values.asList())
    }

    /**
     * @param values Tag-based conditions used to specify a set of resources to assign to a backup plan.
     */
    @JvmName("tcawqxgdaetsjpin")
    public suspend fun selectionTags(values: List<Output<SelectionSelectionTagArgs>>) {
        this.selectionTags = Output.all(values)
    }

    /**
     * @param value A list of conditions that you define to assign resources to your backup plans using tags.
     */
    @JvmName("ghmtvwxpicurkkpt")
    public suspend fun conditions(`value`: List<SelectionConditionArgs>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.conditions = mapped
    }

    /**
     * @param argument A list of conditions that you define to assign resources to your backup plans using tags.
     */
    @JvmName("oyduabthnvqshpsw")
    public suspend fun conditions(argument: List<suspend SelectionConditionArgsBuilder.() -> Unit>) {
        val toBeMapped = argument.toList().map {
            SelectionConditionArgsBuilder().applySuspend {
                it()
            }.build()
        }
        val mapped = of(toBeMapped)
        this.conditions = mapped
    }

    /**
     * @param argument A list of conditions that you define to assign resources to your backup plans using tags.
     */
    @JvmName("kbulffnivefeqedv")
    public suspend fun conditions(vararg argument: suspend SelectionConditionArgsBuilder.() -> Unit) {
        val toBeMapped = argument.toList().map {
            SelectionConditionArgsBuilder().applySuspend {
                it()
            }.build()
        }
        val mapped = of(toBeMapped)
        this.conditions = mapped
    }

    /**
     * @param argument A list of conditions that you define to assign resources to your backup plans using tags.
     */
    @JvmName("yvejrvjglhavkfac")
    public suspend fun conditions(argument: suspend SelectionConditionArgsBuilder.() -> Unit) {
        val toBeMapped = listOf(SelectionConditionArgsBuilder().applySuspend { argument() }.build())
        val mapped = of(toBeMapped)
        this.conditions = mapped
    }

    /**
     * @param values A list of conditions that you define to assign resources to your backup plans using tags.
     */
    @JvmName("duhuhqxfqamvlmio")
    public suspend fun conditions(vararg values: SelectionConditionArgs) {
        val toBeMapped = values.toList()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.conditions = mapped
    }

    /**
     * @param value The ARN of the IAM role that AWS Backup uses to authenticate when restoring and backing up the target resource. See the [AWS Backup Developer Guide](https://docs.aws.amazon.com/aws-backup/latest/devguide/access-control.html#managed-policies) for additional information about using AWS managed policies or creating custom policies attached to the IAM role.
     */
    @JvmName("ujkkbtagmidplilf")
    public suspend fun iamRoleArn(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.iamRoleArn = mapped
    }

    /**
     * @param value The display name of a resource selection document.
     */
    @JvmName("mwoyxkhgkbeetpms")
    public suspend fun name(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.name = mapped
    }

    /**
     * @param value An array of strings that either contain Amazon Resource Names (ARNs) or match patterns of resources to exclude from a backup plan.
     */
    @JvmName("lndvdrfvqqibrdxv")
    public suspend fun notResources(`value`: List<String>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.notResources = mapped
    }

    /**
     * @param values An array of strings that either contain Amazon Resource Names (ARNs) or match patterns of resources to exclude from a backup plan.
     */
    @JvmName("codiprffnujmkbja")
    public suspend fun notResources(vararg values: String) {
        val toBeMapped = values.toList()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.notResources = mapped
    }

    /**
     * @param value The backup plan ID to be associated with the selection of resources.
     */
    @JvmName("anaxxhlshjuitwer")
    public suspend fun planId(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.planId = mapped
    }

    /**
     * @param value An array of strings that either contain Amazon Resource Names (ARNs) or match patterns of resources to assign to a backup plan.
     */
    @JvmName("unulkkjwiafjnnwt")
    public suspend fun resources(`value`: List<String>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.resources = mapped
    }

    /**
     * @param values An array of strings that either contain Amazon Resource Names (ARNs) or match patterns of resources to assign to a backup plan.
     */
    @JvmName("prfbspsnrsymbjtc")
    public suspend fun resources(vararg values: String) {
        val toBeMapped = values.toList()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.resources = mapped
    }

    /**
     * @param value Tag-based conditions used to specify a set of resources to assign to a backup plan.
     */
    @JvmName("hraephrpvgiyqbjm")
    public suspend fun selectionTags(`value`: List<SelectionSelectionTagArgs>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.selectionTags = mapped
    }

    /**
     * @param argument Tag-based conditions used to specify a set of resources to assign to a backup plan.
     */
    @JvmName("jrdmyujugmumdksk")
    public suspend fun selectionTags(argument: List<suspend SelectionSelectionTagArgsBuilder.() -> Unit>) {
        val toBeMapped = argument.toList().map {
            SelectionSelectionTagArgsBuilder().applySuspend {
                it()
            }.build()
        }
        val mapped = of(toBeMapped)
        this.selectionTags = mapped
    }

    /**
     * @param argument Tag-based conditions used to specify a set of resources to assign to a backup plan.
     */
    @JvmName("hfqghcgnstagfrak")
    public suspend fun selectionTags(vararg argument: suspend SelectionSelectionTagArgsBuilder.() -> Unit) {
        val toBeMapped = argument.toList().map {
            SelectionSelectionTagArgsBuilder().applySuspend {
                it()
            }.build()
        }
        val mapped = of(toBeMapped)
        this.selectionTags = mapped
    }

    /**
     * @param argument Tag-based conditions used to specify a set of resources to assign to a backup plan.
     */
    @JvmName("tqsefeyjoedqhclc")
    public suspend fun selectionTags(argument: suspend SelectionSelectionTagArgsBuilder.() -> Unit) {
        val toBeMapped = listOf(SelectionSelectionTagArgsBuilder().applySuspend { argument() }.build())
        val mapped = of(toBeMapped)
        this.selectionTags = mapped
    }

    /**
     * @param values Tag-based conditions used to specify a set of resources to assign to a backup plan.
     */
    @JvmName("cmdgbvsxodpcawfa")
    public suspend fun selectionTags(vararg values: SelectionSelectionTagArgs) {
        val toBeMapped = values.toList()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.selectionTags = mapped
    }

    internal fun build(): SelectionArgs = SelectionArgs(
        conditions = conditions,
        iamRoleArn = iamRoleArn,
        name = name,
        notResources = notResources,
        planId = planId,
        resources = resources,
        selectionTags = selectionTags,
    )
}
